using System;
using Gtk;
using Cairo;

namespace gnomeguitar_cs
{
	public class ScaleCairoRenderer {
		
		ScaleRendererPrefs prefs;
		ScaleWidget scaleWidget = null;
		Frame scaleContainer = new Frame();
		Scale scale;
		Tuning tuning;
		
		public ScaleCairoRenderer (ScaleRendererPrefs prefs)
		{
			this.prefs = prefs;
			prefs.PreferencesChanged += preferences_changed_cb;
			scaleContainer.Shadow = ShadowType.None;
//			scaleWidget = new ScaleWidget(this);
		}

		public ScaleCairoRenderer(ScaleRendererPrefs prefs, Scale scale, Tuning tuning) : this(prefs)
		{
			Render(scale, tuning);
		}
		
		public void Render (Scale scale, Tuning t)
		{
			this.scale = scale;
			this.tuning = t;
			//System.Console.WriteLine("ScaleCairoRenderer.Render: Tuning in cairoRenderer = {0}", tuning.to_text());
			//FIXME! We should be able to reuse the widget but it kept getting lines
			if(scaleWidget != null){
				scaleContainer.Remove(scaleWidget);
			}				
			scaleWidget = new ScaleWidget(this);
			scaleContainer.Child = scaleWidget;
			scaleWidget.Show();
			scaleWidget.Render(scale, tuning);
		}
		
		public void RenderDefault()
		{
			Render(new Scale(), new Tuning ("E,A,D,G,B,E"));
		}		
		
		public Widget GetWidget ()
		{
			return scaleContainer;
		}
		
		void preferences_changed_cb (object obj, EventArgs args)
		{
			//System.Console.WriteLine("ScaleCairoRenderer.preferences_changed_cb: Hello, gscaleCanvasRenderer preferences changed\n");
			Render(scale, tuning);
		}

/***********************************************************************
****************************SCALE WIDGET STUFF**************************
************************************************************************/
		
		class ScaleWidget : DrawingArea {
						
			ScaleCairoRenderer r;
			Scale scale = null;
			Tuning tuning;
			int NoFrets = 13, noStrings;
			FontInfo TuningInfo, FretNoInfo;
			
			double stringGap, fretGap;
				
			double scaleWidth, scaleHeight, tuningX, tuningY, stringX, stringY, noteX, noteY,
			       noteTextX, noteTextY,
			       fretX, fretY, fretNoX, fretNoY;
			
			public ScaleWidget(ScaleCairoRenderer renderer)
			{
				this.r = renderer;
			}
	
			public ScaleWidget(ScaleCairoRenderer renderer, Scale scale, Tuning t): this(renderer)
			{
				Render(scale, t);
			}
			
			public void Render(Scale scale, Tuning t)
			{
				//System.Console.WriteLine("ScaleCairoRenderer.ScaleWidget.Render: ScaleWidget's render called");
				this.scale = scale;
				set_tuning(t);
				//System.Console.WriteLine("ScaleCairoRenderer.ScaleWidget.Render: Tuning used = {0}", this.tuning.to_text());
				noStrings = tuning.get_no();
				if(IsRealized){
					using (Context context = Gdk.CairoHelper.Create(this.GdkWindow)){
						setupFonts(context);
						setupXYs();
						set_scale(context, Allocation);
						draw_scale(context);
					}
				}
			}
			
			void set_tuning(Tuning t)
			{
				//System.Console.WriteLine("ScaleCairoRenderer.ScaleWidget.set_tuning: String order, highStringFirst == {0}", r.prefs.HighStringFirst);
				this.tuning = (Tuning)t.Clone();
				if(!r.prefs.HighStringFirst){
					//System.Console.WriteLine("ScaleCairoRenderer.ScaleWidget.set_tuning: Reverseing tuning");
					this.tuning.Reverse();
				}
			}
			
			void setupFonts(Context c)
			{
				c.Save();
				//System.Console.WriteLine("ScaleCairoRenderer.ScaleWidget.setupFonts: Setting up fontsInfo");
				string text = "X";
				c.SelectFontFace(r.prefs.TuningFont, FontSlant.Normal, FontWeight.Normal);
				c.SetFontSize(r.prefs.TuningSize);
	
				TextExtents te = c.TextExtents(text);
				TuningInfo.Height = te.Height;
				TuningInfo.YBearing = te.YBearing;
				TuningInfo.Width = te.Width * tuning.get_max_no_chars();
				
				text = "00";
				c.SelectFontFace(r.prefs.FretNoFont, FontSlant.Normal, FontWeight.Normal);
				c.SetFontSize(r.prefs.FretNoSize);
	
				te = c.TextExtents(text);
				FretNoInfo.Height = te.Height;
				FretNoInfo.YBearing = te.YBearing;
				FretNoInfo.Width = te.Width;
				text = scale.get_name();
				c.SelectFontFace(r.prefs.TuningFont, FontSlant.Normal, FontWeight.Bold);
				c.SetFontSize(r.prefs.TuningSize);
	
//				te = c.TextExtents(text);
//				TitleInfo.Height = te.Height;
//				TitleInfo.YBearing = te.YBearing;
//				TitleInfo.Width = te.Width;
//						 
				stringGap = TuningInfo.Height + r.prefs.StringPadding;
				fretGap = FretNoInfo.Width + r.prefs.FretPadding;
				//System.Console.WriteLine("ScaleCairoRenderer.ScaleWidget.setupFonts: String Gap = {0}, Fret Gap = {1}", stringGap, fretGap);
				c.Restore();
			}
			
			void setupXYs()
			{
				double tuningOrNoteHeight = (TuningInfo.Height>(r.prefs.NoteOutlineWidth + r.prefs.NoteDiameter)?TuningInfo.Height/2 - (r.prefs.NoteOutlineWidth + r.prefs.NoteDiameter)/2:(r.prefs.NoteOutlineWidth + r.prefs.NoteDiameter)/2 - TuningInfo.Height/2);
				double rTuningOrNoteHeight = (TuningInfo.Height>(r.prefs.NoteOutlineWidth + r.prefs.NoteDiameter)?TuningInfo.Height/2:(r.prefs.NoteOutlineWidth + r.prefs.NoteDiameter)/2);
				double fretNoOrNoteHeight = (FretNoInfo.Height>(r.prefs.NoteOutlineWidth + r.prefs.NoteDiameter)/2?FretNoInfo.Height:(r.prefs.NoteOutlineWidth + r.prefs.NoteDiameter)/2);
				if(r.prefs.RightHanded == true){
					tuningX = r.prefs.BorderWidth + r.prefs.ScaleLeftPadding;
					tuningY = r.prefs.BorderWidth + r.prefs.ScaleTopPadding + tuningOrNoteHeight;
					stringX = tuningX + TuningInfo.Width + r.prefs.TuningPadding + r.prefs.NeckWidth/2;
					stringY = tuningY + TuningInfo.Height/2;
			
					noteX = stringX - fretGap/2;
					noteY = stringY;
					
					noteTextX = noteX;
					noteTextY = noteY;
					
					fretX = stringX;
					fretY = stringY;
					
					fretNoX = fretX ;
					fretNoY = fretY + stringGap*(noStrings-1) + r.prefs.StringWidth/2 + r.prefs.FretNoPadding;

					scaleHeight = fretNoY + fretNoOrNoteHeight + r.prefs.ScaleBottomPadding + r.prefs.BorderWidth;
					scaleWidth = stringX + (NoFrets - 1)*fretGap + FretNoInfo.Width/2 + r.prefs.ScaleRightPadding + r.prefs.BorderWidth;		

				} else {
					
					stringX = r.prefs.BorderWidth + r.prefs.ScaleLeftPadding + FretNoInfo.Width/2 + (NoFrets - 1)*fretGap;
					stringY = r.prefs.BorderWidth + r.prefs.ScaleTopPadding  + rTuningOrNoteHeight;
				
					noteX = stringX + fretGap/2;
					noteY = stringY;
					
					noteTextX = noteX;
					noteTextY = noteY;
					
					tuningX = stringX + r.prefs.NeckWidth/2 + FretNoInfo.Width/2 + r.prefs.TuningPadding;
					tuningY = stringY - TuningInfo.Height/2;	
					
					fretX = stringX;
					fretY = stringY;
					
					fretNoX = fretX ;
					fretNoY = fretY + stringGap*(noStrings-1) + r.prefs.StringWidth/2 + r.prefs.FretNoPadding;
					
					scaleHeight = fretNoY + fretNoOrNoteHeight + r.prefs.ScaleBottomPadding + r.prefs.BorderWidth;
					scaleWidth = tuningX + TuningInfo.Width + r.prefs.ScaleRightPadding + r.prefs.BorderWidth;
				}
//				titleX = scaleWidth/2;
//				titleY = r.prefs.BorderWidth + r.prefs.TitlePadding;
			}
			
			void set_scale(Context c, Gdk.Rectangle area)
			{
				double ratio = Math.Min(area.Width/scaleWidth, area.Height/scaleHeight);
				c.Translate((area.Width - scaleWidth*ratio)/2, (area.Height - scaleHeight*ratio)/2);
				c.Scale(ratio, ratio);
			}
			
			void draw_scale(Context context)
			{
			//	clear(context);
				draw_background(context);
//				draw_title(context);
				draw_strings(context);
				draw_frets(context);
			}
			
//			void clear(Context c)
//			{
//				c.Save();
//				c.Operator = Operator.Clear;
//				c.Paint ();
//				c.Restore();
//			}
			
			void draw_background(Context c)
			{	
				c.Save();
				PointD p1,p2,p3,p4;
			    p1 = new PointD (r.prefs.BorderWidth/2, r.prefs.BorderWidth/2);
			    p2 = new PointD (scaleWidth-r.prefs.BorderWidth/2, r.prefs.BorderWidth/2);
			    p3 = new PointD (scaleWidth-r.prefs.BorderWidth/2, scaleHeight-r.prefs.BorderWidth/2);
			    p4 = new PointD (r.prefs.BorderWidth/2, scaleHeight-r.prefs.BorderWidth/2);
			               
				c.MoveTo (p1);
				c.LineTo (p2);
			    c.LineTo (p3);
			    c.LineTo (p4);
			    c.LineTo (p1);
			    c.ClosePath ();
			                
			    c.Color = color_from_hex(r.prefs.BackgroundColor);
				c.FillPreserve ();
			    c.Color = color_from_hex(r.prefs.BorderColor);
				c.LineWidth = r.prefs.BorderWidth;
			    c.Stroke ();
				c.Restore();
			}

//			void draw_title(Context c)
//			{
//				c.Save();
//				c.Color = color_from_hex(r.prefs.TitleColor);
//				c.SelectFontFace(r.prefs.TitleFont, FontSlant.Normal, FontWeight.Bold);
//				c.SetFontSize(r.prefs.TitleSize);
//		
//				string text = scale.get_name();
//		
//				TextExtents te = c.TextExtents(text);
////					System.Console.WriteLine("ScaleCairoRenderer.ScaleWidget.draw_title: Title height by chordCarioRenderer = {0}\nTitle height by ChordWidget = {1}\nActual Height = {2}",
////					                         r.titleInfo.height, fe.Ascent + fe.Descent, te.Height);
//				
//				double x = titleX - te.XBearing - te.Width/2;
//				double y = titleY - TitleInfo.YBearing;
//				
//				c.MoveTo(x, y);
//				c.ShowText(text);
//				c.Restore();
//			}
			
			void draw_strings(Context c)
			{
				int i = 0;
				//System.Console.WriteLine("ScaleCairoRenderer.ScaleWidget.draw_strings: Tuning in draw_strings = {0}", tuning.to_text());
				foreach (Note note in tuning){
					//System.Console.WriteLine("ScaleCairoRenderer.ScaleWidget.draw_strings: Tuned = {0}", note.to_text());
					draw_tuning(c, i, note);
					draw_string(c, i);
					draw_notes(c, i, scale, note);
					i++;
				}
			}
			
			void draw_tuning(Context c, int stringNo, Note tuning)
			{
				c.Save();
				c.Color = color_from_hex(r.prefs.TuningColor);					
				c.SelectFontFace(r.prefs.TuningFont, FontSlant.Normal, FontWeight.Normal);
				c.SetFontSize(r.prefs.TuningSize);

				string text = tuning.to_text();
				
				TextExtents te = c.TextExtents(text);
				double x = tuningX -te.XBearing;
				double y = tuningY - TuningInfo.YBearing + stringGap*stringNo;
				
				//System.Console.WriteLine("ScaleCairoRenderer.ScaleWidget.draw_tuning: tuningX = {0}, tuningY = {1}, scaleHeight = {2}", x,y, scaleHeight);
				
				c.MoveTo(x, y);
				c.ShowText(text);
				c.Restore();
			}
			
			void draw_string(Context c, int stringNo)
			{
				c.Save();
				double x,y, x2, y2;
				
				if (r.prefs.RightHanded == true){
					x = stringX;
					y = stringY + stringGap*stringNo;
					x2 = x + (NoFrets - 1)*fretGap;
					y2 = y;
				} else {
					x = stringX;
					y = stringY + stringGap*stringNo;
					x2 = x - ((NoFrets - 1)*fretGap);
					y2 = y;
				}
				
				c.MoveTo(x, y);
				c.LineTo(x2, y2);
				c.Color = color_from_hex(r.prefs.StringColor);
				c.LineWidth = r.prefs.StringWidth;
			    c.Stroke ();
				c.Restore();
			}
			
			void draw_notes(Context c, int stringNo, Scale scale, Note tuned)
			{
				foreach (Note note in scale.get_notes()){
					int fretNo = tuned.get_difference(note);
					if (fretNo == 0){ fretNo = 12;}
					draw_note(c, stringNo, fretNo);
					draw_note_text(c, stringNo, fretNo, note);
				}
			}
			
			void draw_note(Context c, int stringNo, int fretNo)
			{
				c.Save();
				double x, y;
				if (r.prefs.RightHanded == true){
					x = noteX + fretGap*fretNo;
					y = noteY + stringGap*stringNo;
				} else {
					x = noteX - fretGap*fretNo;
					y = noteY + stringGap*stringNo;
				}
				//FIXME I don't know why we have an old path that needs clearing
				c.NewPath();
				c.Arc(x, y, r.prefs.NoteDiameter/2, 0, 2 * Math.PI);
				c.Color = color_from_hex(r.prefs.NoteColor);
				c.FillPreserve();
				c.Color = color_from_hex(r.prefs.NoteOutlineColor);
				c.LineWidth = r.prefs.NoteOutlineWidth;
			    c.Stroke ();
				c.Restore();
			}
			
			void draw_note_text(Context c, int stringNo, int fretNo, Note note)
			{
				c.Save();
				c.Color = color_from_hex(r.prefs.NoteFontColor);
				c.SelectFontFace(r.prefs.NoteFont, FontSlant.Normal, FontWeight.Normal);
				c.SetFontSize(r.prefs.NoteFontSize);
		
				string text = note.to_text();
		
				TextExtents te = c.TextExtents(text);
				double x, y;
				
				if (r.prefs.RightHanded == true){
					x = noteTextX + fretGap*fretNo - te.XBearing - te.Width/2;
					y = noteTextY + stringGap*stringNo - te.Height/2 - te.YBearing;
				} else {
					x = noteTextX - fretGap*fretNo - te.XBearing - te.Width/2;
					y = noteTextY + stringGap*stringNo - te.Height/2 - te.YBearing;
				}
				
				c.MoveTo(x, y);
				c.ShowText(text);
				c.Restore();				
			}
			
			void draw_frets(Context c)
			{
				for (int i = 0; i < NoFrets; i++){
					draw_fret(c, i);
					draw_fretNo(c, i);
				}
			}

			void draw_fret(Context c, int fretNo)
			{
				//we need to rember to add on the string widths else it looks a bit odd
				c.Save();
				double x,y, x2, y2;
				
				if (r.prefs.RightHanded == true){
					x = fretX + fretGap*fretNo;
				} else {
					x = fretX - fretGap*fretNo;
				}
				y = fretY - r.prefs.StringWidth/2;
				x2 = x;
				y2 = y + stringGap*(noStrings-1)+r.prefs.StringWidth;
				
				c.MoveTo(x,y);
				c.LineTo(x2, y2);
				c.Color = color_from_hex(r.prefs.FretWireColor);
				c.LineWidth = fretNo == 0?r.prefs.NeckWidth: r.prefs.FretWireWidth;
				c.Stroke ();
				c.Restore();	
			}
			
			void draw_fretNo(Context c, int fretNo)
			{
				c.Save();
				c.Color = color_from_hex(r.prefs.FretNoColor);
				c.SelectFontFace(r.prefs.FretNoFont, FontSlant.Normal, FontWeight.Normal);
				c.SetFontSize(r.prefs.FretNoSize);
		
				string text = fretNo.ToString();
				if (text.Length == 1){
					text = "0" + text;
				}
				TextExtents te = c.TextExtents(text);
				double x,y;
				
				if(r.prefs.RightHanded == true){
					x = fretNoX + fretGap*fretNo - te.Width/2 - te.XBearing;
				} else {
					x = fretNoX - fretGap*fretNo - te.Width/2 - te.XBearing;
				}
				y = fretNoY - te.YBearing;
				
				c.MoveTo(x, y);
				c.ShowText(text);
				c.Restore();	
			}

			Color color_from_hex(string hex)
			{
				string r = hex.Substring(1, 2);
				string g = hex.Substring(3, 2);
				string b = hex.Substring(5, 2);
				
				double rd = ((double)Int32.Parse(r, System.Globalization.NumberStyles.HexNumber))/256.0;
				double gd = ((double)Int32.Parse(g, System.Globalization.NumberStyles.HexNumber))/256.0;
				double bd = ((double)Int32.Parse(b, System.Globalization.NumberStyles.HexNumber))/256.0;
			//	System.Console.WriteLine("ScaleCairoRenderer.ScaleWidget.color_from_hex: r = {0}, rd = {1}\ng = {2}, gd = {3}\nb = {4}, bd = {5}",
			//	                         r, rd, g, gd, b, bd);
				return new Color(rd,gd,bd);
			}
			
			struct FontInfo 
			{
				public double Height;
				public double Width;
				public double YBearing;
				
				public FontInfo(double height, double width, double yBearing){
					this.Height = height;
					this.Width = width;
					this.YBearing = yBearing;
				}
			}
			
			protected override void OnRealized ()
			{
				//System.Console.WriteLine("ScaleCairoRenderer.ScaleWidget.OnRealized: the scaleWidget is Realised!");
				base.OnRealized();
				using(Context context = Gdk.CairoHelper.Create(this.GdkWindow)){
					if(scale != null){
						setupFonts(context);
						setupXYs();
					}
				}
			}
			
			protected override bool OnExposeEvent (Gdk.EventExpose args)
			{
				//System.Console.WriteLine("ScaleCairoRenderer.ScaleWidget.OnExposeEvent: the scaleWidget is Exposed!");
				if(scale == null){
					System.Console.WriteLine("ScaleCairoRenderer.ScaleWidget.OnExposeEvent: But no scale has been set yet!");
					return true;
				}
				using (Context context = Gdk.CairoHelper.Create(args.Window)){
					set_scale(context, this.Allocation);
					draw_scale(context);
				}
				return true;
			}
			
		}
	}
}
