using System;
using Gtk;
using Cairo;
using System.Collections;
	
namespace gnomeguitar_cs
{
	public class ChordCairoRenderer : ChordRendererUI {
		
		RealRenderer realRenderer;
		ChordGroup chords = null;
		ChordRendererPrefs prefs;
		
		public ChordCairoRenderer(ChordRendererPrefs prefs)
		{
			this.prefs = prefs;
			realRenderer = new RealRenderer(this);
			realRenderer.ShowAll();
		}	
	
		public ChordCairoRenderer(ChordRendererPrefs prefs, ChordGroup chords): this(prefs)
		{
			set_chords(chords);
		}
	
		public Widget get_widget ()
		{		
			return realRenderer;
		}

		public void render_default()
		{
			Chord emptyChord =  new Chord();
			ChordGroup chords = new ChordGroup();
			chords.add(emptyChord);
			render(chords);
		}

		public void render(ChordGroup chords)
		{
			set_chords(chords);
			render();
		}
		
		public void render()
		{
			if (chords == null){
				render_default();
				return;
			}
			realRenderer.render(chords);
		}
		
		public void set_chords(ChordGroup chords)
		{
			this.chords = chords;
			
		}
		
		public void clear()
		{
			realRenderer.clear();
		}
		
/***********************************************************
 *********************REAL RENDERER STUFF*******************
 ***********************************************************/
		
		
		class RealRenderer: Table {
			
			int oldAllocationWidth = 0;
			int oldAllocationHeight = 0;
			ChordCairoRenderer r;
			ChordGroup chords = null;
			bool chordsLaidout = false, fontsSetUp = false; 
			FontInfo frettedInfo, relationInfo, tuningInfo, fretNoInfo, titleInfo;
			double greatestWidth, greatestHeight;
			double chordWidth = 0, chordHeight = 0;
			
			//this is greatestWidth + stringPadding so it can't be set up till greatestWidth is
			double stringGap, fretGap;
				
			int maxNoStrings, maxNoFrets;
			// when the chords change we only need to change the possible layouts if the number of chords has changed
			// nothing else should use it !!!
			int noChords = 0;
			ArrayList possibleLayouts = new ArrayList();
			
			int chordsPerRow = 0, chordsPerColumn = 0;		
				
			enum TextType {All, Notes, Numbers};
					
			public RealRenderer(ChordCairoRenderer renderer):base (1,1,true)
			{	
				this.r = renderer;
				r.prefs.PreferencesChanged += on_preferences_changed;
			}	
				
			public void render(ChordGroup chords)
			{	
				//System.Console.WriteLine("ChordCairoRenderer.RealRenderer.render: Rendering requested");
				//clear();
				this.chords = chords;
				chordsLaidout = false;
				maxNoStrings = chords.get_max_no_strings();
				maxNoFrets = Math.Max(r.prefs.MinNoFrets,chords.get_max_no_frets());
				set_size();
				layout_chords();
				
			}	
			
			void set_up_fonts_info(Context c)
			{
				//System.Console.WriteLine("ChordCairoRenderer.RealRenderer.set_up_fonts_info: Setting up fontsInfo");
				//Title can go below the baseline, none of the others can so their heights reflect this
				fretNoInfo = get_font_info(c, r.prefs.FretNoFont, r.prefs.FretNoSize, TextType.Numbers);
				tuningInfo = get_font_info(c, r.prefs.TuningFont, r.prefs.TuningSize, TextType.Notes);
				relationInfo = get_font_info(c, r.prefs.RelationFont, r.prefs.RelationSize, TextType.Notes);
				frettedInfo = get_font_info(c, r.prefs.FrettedFont, r.prefs.FrettedSize, TextType.Notes);
				//fretNo's are two digits long
				fretNoInfo.width = fretNoInfo.width*2;
				//we have to deal with the possibilty of double flats and sharps
				//we can't get the actual max values as the chords haven't been set yet!
				tuningInfo.width = tuningInfo.width*3;
				relationInfo.width = relationInfo.width*3;
				frettedInfo.width = frettedInfo.width*3;
					
				if(tuningInfo.width > relationInfo.width && tuningInfo.width > frettedInfo.width){
					greatestWidth = tuningInfo.width;
				} else if (relationInfo.width > frettedInfo.width){
					greatestWidth = relationInfo.width;
				} else {
					greatestWidth = frettedInfo.width;
				}	
				if(tuningInfo.height > relationInfo.height && tuningInfo.height > frettedInfo.height){
					greatestHeight = tuningInfo.height;
				} else if (relationInfo.height > frettedInfo.height){
					greatestHeight = relationInfo.height;
				} else {
					greatestHeight = frettedInfo.height;
				}
//				FIX ME! ChordCairoRenderer.RealRenderer.set_up_fonts_info: This isn't right if the chord is rotated, but seems to work!?		 
				stringGap = greatestWidth + r.prefs.StringPadding;
				fretGap = fretNoInfo.height + r.prefs.FretPadding;
				titleInfo = get_font_info(c, r.prefs.TitleFont, FontSlant.Normal, FontWeight.Bold, r.prefs.TitleSize, TextType.All);
			}
			

			
			FontInfo get_font_info(Context c, string font, double size, TextType textType)
			{	
				return get_font_info(c, font, FontSlant.Normal,FontWeight.Normal, size, textType);
			}
		
			FontInfo get_font_info(Context c, string font, FontSlant slant, FontWeight weight, double size, TextType textType)
			{	
				const string abcAll = "abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789#";
				const string notes = "ABCDEFGb#";
				const string numbers = "0123456789";
				
				string text = null;
				switch (textType){
				case TextType.All:text = abcAll; break;
				case TextType.Notes:text = notes; break;
				case TextType.Numbers: text = numbers; break;
				}	

				c.SelectFontFace(font, slant, weight);
				c.SetFontSize(size);
		
				TextExtents te = c.TextExtents(text);
				double height = te.Height;
				double yBearing = te.YBearing;
				te = c.TextExtents("X");
				double width = te.Width;
				return new FontInfo(height, width, yBearing);
			}
			
			void setup_chord_dimensions()
			{
				//do we need to take into account string and fretWire widths?
				//System.Console.WriteLine("ChordCairoRenderer.RealRenderer.setup_chord_dimensions: Setting up chord dimensions");
				if (r.prefs.ChordRotation == ChordRotation.Vertical){
					double tuningX = r.prefs.BorderWidth + r.prefs.VoiceingLeftPadding + greatestWidth/2;
					double stringX = tuningX;
					double fretX = stringX;				
					double fretNoX = fretX + stringGap*(maxNoStrings-1) + r.prefs.FretNoPadding;
					chordWidth = fretNoX + fretNoInfo.width + r.prefs.VoiceingRightPadding + r.prefs.BorderWidth;
					
					double tuningY = r.prefs.BorderWidth + r.prefs.TitlePadding + titleInfo.height + r.prefs.VoiceingTopPadding;
					double stringY = tuningY + tuningInfo.height + r.prefs.TuningPadding + r.prefs.NeckWidth/2;
					double frettedY = stringY + (maxNoFrets - 1)*fretGap + fretGap*r.prefs.StringExtraPercent + r.prefs.FrettedPadding + frettedInfo.height;
					double relationY = frettedY + r.prefs.RelationPadding + relationInfo.height;	
					chordHeight = relationY + r.prefs.VoiceingBottomPadding + r.prefs.BorderWidth;
				} else {
					double tuningX = r.prefs.BorderWidth + r.prefs.VoiceingLeftPadding;
					double stringX = tuningX + tuningInfo.width + r.prefs.TuningPadding + r.prefs.NeckWidth/2;
					double frettedX = stringX +(maxNoFrets - 1)*fretGap + fretGap*r.prefs.StringExtraPercent + r.prefs.FrettedPadding;				
					double relationX = frettedX + frettedInfo.width + r.prefs.RelationPadding;
					chordWidth = relationX + r.prefs.VoiceingBottomPadding + r.prefs.BorderWidth;
					
					double fretY = r.prefs.BorderWidth + r.prefs.TitlePadding + titleInfo.height + r.prefs.VoiceingTopPadding + greatestHeight/2;
					double fretNoY = fretY + stringGap*(maxNoStrings-1) + r.prefs.FretNoPadding + fretNoInfo.height;
					chordHeight = fretNoY + r.prefs.VoiceingBottomPadding;
				}
			}
			
			void layout_chords()
			{
				//System.Console.WriteLine("ChordCairoRenderer.RealRenderer.layout_chords: layingout chords");
			
				//the number of rows is the number of chords in each column and vica verca
				clear();
		
				int row = 0;
				int column = 0;			
				foreach (Chord chord in chords)
				{	
					//System.Console.WriteLine("ChordCairoRenderer.RealRenderer.layout_chords: We added a chord!");
					Attach(new ChordWidget(this, chord),(uint)column, (uint)column+1, (uint)row, (uint)row+1 );
					if (chordsPerRow > column+1){
						column++;
					} else {
						column = 0;
						row++;
					}
				}
				chordsLaidout = true;
				ShowAll();
			}	
			
			public void clear ()
			{
				foreach (Widget w in Children){
					Remove(w);
				}
			}
			
			void calculate_rows_and_columns()
			{
				int fakeNo, smallestWholeNo;
				double windowHeight, windowWidth, windowArea;
				double heightRatio, widthRatio;
				double chordsHeight, chordsWidth;
				int[] bestLayout = null;
				double freeSpace, bestFit = Double.MaxValue;
				//System.Console.WriteLine("ChordCairoRenderer.RealRenderer.calculate_rows_and_columns: Calculateing rows and columns");
				//first we create the possible layouts for this number of chords
				//if the number of chords hasn't changed we can use the old layouts
				if (noChords != chords.get_no())
				{
					possibleLayouts.Clear();
					noChords = chords.get_no();
					fakeNo = noChords;
					//odd numbers of chords have the layouts as the next greatest even number of chords except for 1/noChords and noChords/1
					if((noChords %2) != 0){
						fakeNo++;
					}
					possibleLayouts.Add(new int[2]{1,noChords});
					
					smallestWholeNo = fakeNo;
					for (int i = 2; i <= (fakeNo/2);i++){
						int[] layout = new int[2];
						layout[0] = i;
						layout [1] = (int)Math.Ceiling((double)fakeNo/(double)i);
						//we then check that there isn't already a more efficent packing
						//if not we add it to the possible packings
						if (layout[1] < smallestWholeNo){
							possibleLayouts.Add(layout);
							smallestWholeNo = layout[1];			
						}
					}
					possibleLayouts.Add(new int[2]{noChords,1});
				}
				windowHeight = Allocation.Height;
				windowWidth = Allocation.Width;
				windowArea = windowWidth * windowHeight;
				foreach (int[] layout in possibleLayouts){
					//then we multiply the rows and columns by the size of the chords to get the size of the layouts
					chordsWidth = (layout[0]* chordWidth) + ((layout[0] -1) * r.prefs.ChordXPadding);
					chordsHeight = (layout[1]* chordHeight) + ((layout[1] -1) * r.prefs.ChordYPadding);
					//then we scale the layout so it fills the window
					heightRatio = windowHeight / chordsHeight;
					widthRatio = windowWidth / chordsWidth;
					if (heightRatio < widthRatio){
						chordsHeight = chordsHeight * heightRatio;
						chordsWidth = chordsWidth * heightRatio;
					} else {
						chordsHeight = chordsHeight * widthRatio;
						chordsWidth = chordsWidth * widthRatio;
					}	
					//then we work out how well it fits in the window
					freeSpace = windowArea - (chordsWidth*chordsHeight);
					//if we don't have a better fit we set this layout as the best layout
					if (freeSpace < bestFit){
						bestFit = freeSpace;
						bestLayout = layout;
					}
				}
				//tadaa we can now set the number of columns and rows 
				chordsPerRow = bestLayout[0];
				chordsPerColumn = bestLayout[1];
	//			chordsPerRow = 2;
	//			chordsPerColumn = 1;
			}
			
			new public void Resize(uint rows, uint columns)
			{
				//System.Console.WriteLine("ChordCairoRenderer.RealRenderer.Resize: Table resized to {0} rows and {1} column",rows,columns);
				layout_chords();
				base.Resize(rows,columns);
			}
			
			protected override void OnRealized ()
			{
				//System.Console.WriteLine("ChordCairoRenderer.RealRenderer.OnRealized: the scaleWidget is Realised!");
				//in case the chords were set before we were realised
				base.OnRealized();
				set_size();
			}
			
			
			protected override void OnSizeAllocated (Gdk.Rectangle allocation)
			{
				//System.Console.WriteLine("ChordCairoRenderer.RealRenderer.OnSizeAllocated: Size Allocated");
		
				//check the size has actually changed				
				if(allocation.Width == oldAllocationWidth && allocation.Height == oldAllocationHeight && chordsLaidout){
					//System.Console.WriteLine("ChordCairoRenderer.RealRenderer.OnSizeAllocated: Allocation hasn't chanaged or no chords set so we don't need to do anything");
					base.OnSizeAllocated(allocation);
					return;
				}

				set_size();
				base.OnSizeAllocated(allocation);
			}
			
			void set_size()
			{
				//System.Console.WriteLine("ChordCairoRenderer.RealRenderer.set_size: set_size called");
				//check that we have chords and somewhere to draw them!
				if(chords == null || !IsRealized){
					//System.Console.WriteLine("ChordCairoRenderer.RealRenderer.set_size: not renderering because chords = {0} and IsRealized = {1}", chords != null, IsRealized);
					return;
				}
				
				//fonts only need to be set up once
				if(!fontsSetUp){
					using (Context context = Gdk.CairoHelper.Create(this.GdkWindow)){
						set_up_fonts_info(context);
						fontsSetUp = true;
					}
				}
				
				oldAllocationWidth = Allocation.Width;
				oldAllocationHeight = Allocation.Height;

				int perRow = chordsPerRow, perColumn = chordsPerColumn;
				setup_chord_dimensions();
				calculate_rows_and_columns();
				//layout_chords();
				if(perRow != chordsPerRow || perColumn != chordsPerColumn){
					Resize((uint)chordsPerColumn, (uint)chordsPerRow);
					//System.Console.WriteLine("ChordCairoRenderer.RealRenderer.set_size: This call may be recursive");
				}	
				//FIXME we need to add the amount of padding (which depends on the number of chords) somewhere			
				double ratio = Math.Min((Allocation.Height/NRows)/chordHeight, (Allocation.Width/NColumns)/chordWidth);
				uint rs = (uint)(r.prefs.ChordYPadding*ratio);
				uint cs = (uint)(r.prefs.ChordXPadding*ratio);
				//System.Console.WriteLine("ChordCairoRenderer.RealRenderer.set_size: cs = {0}, rs = {1}", cs, rs);
				if(RowSpacing != rs || ColumnSpacing != cs){
					RowSpacing = rs;
					ColumnSpacing = cs;						
				}
				
			}
			
			void on_preferences_changed (object o, EventArgs args)
			{
				//System.Console.WriteLine("ChordCairoRenderer.RealRenderer.on_preferences_changed: the preferences have changed!");
				using (Context context = Gdk.CairoHelper.Create(this.GdkWindow)){
					context.Save();
					set_up_fonts_info(context);
					context.Restore();
				}
				if(chords != null){
					setup_chord_dimensions();
					layout_chords();
				}
			}
			
			struct FontInfo 
			{
				public double height;
				public double width;
				public double yBearing;
				
				public FontInfo(double height, double width, double yBearing){
					this.height = height;
					this.width = width;
					this.yBearing = yBearing;
				}
			}
			
/***********************************************************
 *****************CHORD WIDGET STUFF************************
 ***********************************************************/	
		
			class ChordWidget: DrawingArea {
				
				Chord chord;
				RealRenderer r;
				//startFret is the fret we start drawing from not the first fret of the chord
				int startFret, noStrings;
				
				double titleX, titleY, tuningX, tuningY, stringX, stringY, fingerX, fingerY,
				       fingerTextX, fingerTextY, frettedX, frettedY, relationX, relationY,
				       fretX, fretY, fretNoX, fretNoY;
				
				public ChordWidget(RealRenderer renderer, Chord chord)
				{
					this.r = renderer;
					this.chord = chord;
					noStrings = chord.get_voiceing(0).get_no_strings();
					//this is to make sure we don't start befor fret 0!
					int minFret = chord.get_voiceing(0).get_minFret();
					int fretPadding = r.maxNoFrets - chord.get_voiceing(0).get_noFrets();
					//if we pad uneqally then it goes on the end
					int startFretPadding = (int)fretPadding/2;
					while (minFret - startFretPadding < 0){
						startFretPadding--;
					}
					startFret = minFret - startFretPadding;
					
					setupXYs();
				}
				
				void setupXYs()
				{
					//do we need to take into account string and fretWire widths?
					if(r.r.prefs.ChordRotation == ChordRotation.Vertical){
						titleX = r.chordWidth/2;
						titleY = r.r.prefs.BorderWidth + r.r.prefs.TitlePadding;
						
						tuningX = r.r.prefs.BorderWidth + r.r.prefs.VoiceingLeftPadding + r.greatestWidth/2;
						tuningY = r.r.prefs.BorderWidth + r.r.prefs.TitlePadding + r.titleInfo.height + r.r.prefs.VoiceingTopPadding;
				
						stringX = tuningX;
						stringY = tuningY + r.tuningInfo.height + r.r.prefs.TuningPadding + r.r.prefs.NeckWidth/2;
						
						fingerX = stringX;
						fingerY = stringY - r.fretGap/2;
						
						fingerTextX = fingerX;
						fingerTextY = fingerY;
						
						frettedX = stringX;
						frettedY = stringY + (r.maxNoFrets - 1)*r.fretGap + r.fretGap*r.r.prefs.StringExtraPercent + r.r.prefs.FrettedPadding - r.frettedInfo.yBearing;		
						
						relationX = stringX;
						relationY = frettedY + r.r.prefs.RelationPadding - r.relationInfo.yBearing;	
						
						fretX = stringX;
						fretY = stringY;
						
						fretNoX = fretX + r.stringGap*(noStrings-1) + r.r.prefs.StringWidth/2 + r.r.prefs.FretNoPadding;
						fretNoY = fretY;	
					} else {
						//We have a lot of extra space because we didn't know how much the various notes were going to take up
						//so we need to centre the chord!!!
						
						double tuningWidth = (r.tuningInfo.width/3) * get_max_tuning_chars();
						double relationWidth = (r.relationInfo.width/3) * get_max_relation_chars();
						double frettedWidth = (r.frettedInfo.width/3) * get_max_fretted_chars();
						double actualWidth = r.r.prefs.BorderWidth + r.r.prefs.VoiceingLeftPadding + tuningWidth + r.r.prefs.TuningPadding + r.r.prefs.NeckWidth/2 + (r.maxNoFrets - 1)*r.fretGap + r.fretGap*r.r.prefs.StringExtraPercent + r.r.prefs.FrettedPadding + frettedWidth + r.r.prefs.RelationPadding + relationWidth + r.r.prefs.VoiceingRightPadding + r.r.prefs.BorderWidth;
						double offset = (r.chordWidth-actualWidth)/2;
						if(r.r.prefs.RightHanded == true){
							
							titleX = r.chordWidth/2;
							titleY = r.r.prefs.BorderWidth + r.r.prefs.TitlePadding;
							
							tuningX = offset + r.r.prefs.BorderWidth + r.r.prefs.VoiceingLeftPadding;
							tuningY = r.r.prefs.BorderWidth + r.r.prefs.TitlePadding + r.titleInfo.height + r.r.prefs.VoiceingTopPadding;
							stringX = tuningX + tuningWidth + r.r.prefs.TuningPadding + r.r.prefs.NeckWidth/2;
							stringY = tuningY + r.greatestHeight/2;
					
							
							fingerX = stringX - r.fretGap/2;
							fingerY = stringY;
							
							fingerTextX = fingerX;
							fingerTextY = fingerY;
							
							frettedX = stringX + (r.maxNoFrets - 1)*r.fretGap + r.fretGap*r.r.prefs.StringExtraPercent + r.r.prefs.FrettedPadding;
							frettedY = stringY;
							
							relationX = frettedX + frettedWidth + r.r.prefs.RelationPadding;
							relationY = frettedY;	
							
							fretX = stringX;
							fretY = stringY;
							
							fretNoX = fretX ;
							fretNoY = fretY + r.stringGap*(noStrings-1) + r.r.prefs.StringWidth/2 + r.r.prefs.FretNoPadding;
							
						} else {
							titleX = r.chordWidth/2;
							titleY = r.r.prefs.BorderWidth + r.r.prefs.TitlePadding;
							
							relationX = offset + r.r.prefs.BorderWidth + r.r.prefs.VoiceingLeftPadding;
							relationY = r.r.prefs.BorderWidth + r.r.prefs.TitlePadding + r.titleInfo.height + r.r.prefs.VoiceingTopPadding;
							frettedX = relationX + relationWidth + r.r.prefs.RelationPadding;
							frettedY = relationY;
							
							stringX = frettedX + frettedWidth + r.r.prefs.FrettedPadding +(r.maxNoFrets - 1)*r.fretGap + r.fretGap*r.r.prefs.StringExtraPercent;
							stringY = frettedY + r.greatestHeight/2;
						
							fingerX = stringX + r.fretGap/2;
							fingerY = stringY;
							
							fingerTextX = fingerX;
							fingerTextY = fingerY;
							
							tuningX = stringX + r.r.prefs.NeckWidth/2 + r.r.prefs.TuningPadding;
							tuningY = frettedY;	
							
							fretX = stringX;
							fretY = stringY;
							
							fretNoX = fretX ;
							fretNoY = fretY + r.stringGap*(noStrings-1) + r.r.prefs.StringWidth/2 + r.r.prefs.FretNoPadding;	
						}
					}
				}

				int get_max_tuning_chars()
				{
					return chord.get_max_tuning_chars();
				}
					
				int get_max_relation_chars()
				{
					return chord.get_max_relation_chars();
				}
			
				int get_max_fretted_chars()
				{
					return chord.get_max_fretted_chars();
				}
				
				protected override bool OnExposeEvent (Gdk.EventExpose args)
				{
					//System.Console.WriteLine("ChordCairoRenderer.ChordWidget.OnExposeEvent: the chord widget is Exposed!");
					using (Context context = Gdk.CairoHelper.Create(args.Window)){
						context.Save();
						set_scale(context, this.Allocation);
						draw_background(context);
						draw_title(context);
						draw_strings(context);
						draw_frets(context);
						context.Restore();
					}
					
					return true;
				}
				
				void set_scale(Context c, Gdk.Rectangle area)
				{
					double ratio = Math.Min(area.Width/r.chordWidth, area.Height/r.chordHeight);
					c.Translate((area.Width - r.chordWidth*ratio)/2, (area.Height -r.chordHeight*ratio)/2);
					c.Scale(ratio, ratio);
				}
				
				void draw_background(Context c)
				{	
					c.Save();
					PointD p1,p2,p3,p4;
				    p1 = new PointD (r.r.prefs.BorderWidth/2, r.r.prefs.BorderWidth/2);
				    p2 = new PointD (r.chordWidth-r.r.prefs.BorderWidth/2, r.r.prefs.BorderWidth/2);
				    p3 = new PointD (r.chordWidth-r.r.prefs.BorderWidth/2, r.chordHeight-r.r.prefs.BorderWidth/2);
				    p4 = new PointD (r.r.prefs.BorderWidth/2, r.chordHeight-r.r.prefs.BorderWidth/2);
				               
					c.MoveTo (p1);
					c.LineTo (p2);
				    c.LineTo (p3);
				    c.LineTo (p4);
				    c.LineTo (p1);
				    c.ClosePath ();
				                
				    c.Color = color_from_hex(r.r.prefs.BackgroundColor);
					c.FillPreserve ();
				    c.Color = color_from_hex(r.r.prefs.BorderColor);
					c.LineWidth = r.r.prefs.BorderWidth;
				    c.Stroke ();
					c.Restore();
				}

				void draw_title(Context c)
				{
					c.Save();
					c.Color = color_from_hex(r.r.prefs.TitleColor);
					c.SelectFontFace(r.r.prefs.TitleFont, FontSlant.Normal, FontWeight.Bold);
					c.SetFontSize(r.r.prefs.TitleSize);
			
					string text = chord.get_name();
			
					TextExtents te = c.TextExtents(text);
//					System.Console.WriteLine("ChordCairoRenderer.ChordWidget.draw_title: Title height by chordCarioRenderer = {0}\nTitle height by ChordWidget = {1}\nActual Height = {2}",
//					                         r.titleInfo.height, fe.Ascent + fe.Descent, te.Height);
					
					double x = titleX - te.XBearing - te.Width/2;
					double y = titleY - r.titleInfo.yBearing;
					
					c.MoveTo(x, y);
					c.ShowText(text);
					c.Restore();
				}
				
				void draw_strings(Context c)
				{
					int i = 0;
					GuitarStringGroup guitarStrings =  chord.get_voiceing(0).get_strings();
					if (r.r.prefs.HighStringFirst){
						guitarStrings = (GuitarStringGroup)guitarStrings.Clone();
						guitarStrings.Reverse();
					}
					foreach (GuitarString guitarString in guitarStrings){
						draw_tuning(c, i, guitarString.get_tuned());
						draw_string(c, i, guitarString);
						Finger finger = guitarString.get_finger();
						if(finger != FingerValue.NOT_USED && finger != FingerValue.NO_FINGER){
							draw_finger(c, i, guitarString.get_fretNo(), finger);
							draw_finger_text(c, i, guitarString.get_fretNo(), finger);
						}
						draw_fretted(c, i, guitarString.get_fretted());
						draw_relation(c, i, guitarString.get_relation());
						i++;
					}
				}
				
				void draw_tuning(Context c, int stringNo, Note tuning)
				{
					c.Save();
					c.Color = color_from_hex(r.r.prefs.TuningColor);					
					c.SelectFontFace(r.r.prefs.TuningFont, FontSlant.Normal, FontWeight.Normal);
					c.SetFontSize(r.r.prefs.TuningSize);

					string text = tuning.to_text();
					
					TextExtents te = c.TextExtents(text);
					double x, y;
					if(r.r.prefs.ChordRotation == ChordRotation.Vertical){
						x = tuningX - te.Width/2-te.XBearing + r.stringGap*stringNo;
						y = tuningY - r.tuningInfo.yBearing;
					} else {
						x = tuningX -te.XBearing;
						y = tuningY - r.tuningInfo.yBearing + r.stringGap*stringNo;
					}
					c.MoveTo(x, y);
					c.ShowText(text);
					c.Restore();
				}
				
				void draw_string(Context c, int stringNo, GuitarString guitarString)
				{
					c.Save();
					double x,y, x2, y2;
					if(r.r.prefs.ChordRotation == ChordRotation.Vertical){
						x = stringX + r.stringGap*stringNo;
						y = stringY;
						x2 = x;
						y2 = y + (r.maxNoFrets - 1)*r.fretGap + r.fretGap*r.r.prefs.StringExtraPercent;
					} else {
						if (r.r.prefs.RightHanded == true){
							x = stringX;
							y = stringY + r.stringGap*stringNo;
							x2 = x + (r.maxNoFrets - 1)*r.fretGap + r.fretGap*r.r.prefs.StringExtraPercent;
							y2 = y;
						} else {
							x = stringX;
							y = stringY + r.stringGap*stringNo;
							x2 = x - ((r.maxNoFrets - 1)*r.fretGap + r.fretGap*r.r.prefs.StringExtraPercent);
							y2 = y;
						}
					}
					c.MoveTo(x, y);
					c.LineTo(x2, y2);
					c.Color = color_from_hex(r.r.prefs.StringColor);
					c.LineWidth = r.r.prefs.StringWidth;
					if(!guitarString.played()){
						c.SetDash(new double[]{r.r.prefs.StringDashLength, r.r.prefs.StringDashLength},0);
					}
				    c.Stroke ();
					c.Restore();
				}
				
				void draw_finger(Context c, int stringNo, int fretNo, Finger finger)
				{
					c.Save();
					double x, y;
					if(r.r.prefs.ChordRotation == ChordRotation.Vertical){
						x = fingerX + r.stringGap*stringNo;
						y = fingerY + r.fretGap*(fretNo - startFret);
					} else {
						if (r.r.prefs.RightHanded == true){
							x = fingerX + r.fretGap*(fretNo - startFret);
							y = fingerY + r.stringGap*stringNo;
						} else {
							x = fingerX - r.fretGap*(fretNo - startFret);
							y = fingerY + r.stringGap*stringNo;
						}
					}
						
					c.Arc(x, y, r.r.prefs.FingerDiameter/2, 0, 2 * Math.PI);
					c.Color = color_from_hex(r.r.prefs.FingerColor);
					c.FillPreserve();
					c.Color = color_from_hex(r.r.prefs.FingerOutlineColor);
					c.LineWidth = r.r.prefs.FingerOutlineWidth;
				    c.Stroke ();
					c.Restore();
				}
				
				void draw_finger_text(Context c, int stringNo, int fretNo, Finger finger)
				{
					c.Save();
					c.Color = color_from_hex(r.r.prefs.FingerFontColor);
					c.SelectFontFace(r.r.prefs.FingerFont, FontSlant.Normal, FontWeight.Normal);
					c.SetFontSize(r.r.prefs.FingerFontSize);
			
					string text = finger.to_text();
			
					TextExtents te = c.TextExtents(text);
					double x, y;
					if(r.r.prefs.ChordRotation == ChordRotation.Vertical){
						x = fingerTextX + r.stringGap*stringNo - te.Width/2 - te.XBearing;
						y = fingerTextY + r.fretGap*(fretNo - startFret) - te.YBearing - te.Height/2;
					} else {
						if (r.r.prefs.RightHanded == true){
							x = fingerTextX + r.fretGap*(fretNo - startFret) - te.XBearing - te.Width/2;
							y = fingerTextY + r.stringGap*stringNo - te.Height/2 - te.YBearing;
						} else {
							x = fingerTextX - r.fretGap*(fretNo - startFret) - te.XBearing - te.Width/2;
							y = fingerTextY + r.stringGap*stringNo - te.Height/2 - te.YBearing;
						}
					}
					c.MoveTo(x, y);
					c.ShowText(text);
					c.Restore();				
				}
				
				void draw_fretted(Context c, int stringNo, Note fretted)
				{
					c.Save();
					c.Color = color_from_hex(r.r.prefs.FrettedColor);
					c.SelectFontFace(r.r.prefs.FrettedFont, FontSlant.Normal, FontWeight.Normal);
					c.SetFontSize(r.r.prefs.FrettedSize);
			
					string text = fretted.to_text();
					TextExtents te = c.TextExtents(text);
					double x,y;
					if(r.r.prefs.ChordRotation == ChordRotation.Vertical){
						x = frettedX - te.Width/2 -te.XBearing + r.stringGap*stringNo;				
						y = frettedY;
					} else {
						x = frettedX - te.XBearing;
						if (r.r.prefs.RightHanded == true){
							y = frettedY - te.Height/2 - te.YBearing + r.stringGap*stringNo;
						} else {
							y = frettedY - r.frettedInfo.yBearing + r.stringGap*stringNo;
						}
					}
					
					c.MoveTo(x, y);
					c.ShowText(text);
					c.Restore();
				}
				
				void draw_relation(Context c, int stringNo, Relation relation)
				{
					c.Save();
					c.Color = color_from_hex(r.r.prefs.RelationColor);
					c.SelectFontFace(r.r.prefs.RelationFont, FontSlant.Normal, FontWeight.Normal);
					c.SetFontSize(r.r.prefs.RelationSize);
			
					string text = relation.to_text();
					TextExtents te = c.TextExtents(text);
					double x,y;
					if(r.r.prefs.ChordRotation == ChordRotation.Vertical){
						x = relationX - te.Width/2 -te.XBearing + r.stringGap*stringNo;
						y = relationY;
					} else {
						x = relationX -te.XBearing;
						if (r.r.prefs.RightHanded == true){
							y = relationY - te.Height/2 - te.YBearing + r.stringGap*stringNo;						
						} else {
							y = relationY - r.relationInfo.yBearing + r.stringGap*stringNo;
						}
					}
					c.MoveTo(x, y);
					c.ShowText(text);
					c.Restore();
				}
				
				void draw_frets(Context c)
				{
					//System.Console.WriteLine("ChordCairoRenderer.ChordWidget.draw_frets: startFret = {0}, maxNoFrets = {1}", startFret, r.maxNoFrets);
					int endFret = startFret + r.maxNoFrets;
					for (int i = startFret; i < endFret; i++){
						draw_fret(c, i);
						draw_fretNo(c, i);
					}
				}

				void draw_fret(Context c, int fretNo)
				{
					//we need to rember to add on the string widths else it looks a bit odd
					c.Save();
					double x,y, x2, y2;
					if(r.r.prefs.ChordRotation == ChordRotation.Vertical){
						x = fretX - r.r.prefs.StringWidth/2;
						y = fretY + r.fretGap*(fretNo - startFret);
						x2 = x + r.stringGap*(noStrings-1)+r.r.prefs.StringWidth;
						y2 = y;
					} else {
						if (r.r.prefs.RightHanded == true){
							x = fretX + r.fretGap*(fretNo - startFret);
						} else {
							x = fretX - r.fretGap*(fretNo - startFret);
						}
						y = fretY - r.r.prefs.StringWidth/2;
						x2 = x;
						y2 = y + r.stringGap*(noStrings-1)+r.r.prefs.StringWidth;
					}
					c.MoveTo(x,y);
					c.LineTo(x2, y2);
					c.Color = color_from_hex(r.r.prefs.FretWireColor);
					c.LineWidth = fretNo == 0?r.r.prefs.NeckWidth: r.r.prefs.FretWireWidth;
					c.Stroke ();
					c.Restore();	
				}
				
				void draw_fretNo(Context c, int fretNo)
				{
					c.Save();
					c.Color = color_from_hex(r.r.prefs.FretNoColor);
					c.SelectFontFace(r.r.prefs.FretNoFont, FontSlant.Normal, FontWeight.Normal);
					c.SetFontSize(r.r.prefs.FretNoSize);
			
					string text = fretNo.ToString();
					if (text.Length == 1){
						text = "0" + text;
					}
					TextExtents te = c.TextExtents(text);
					double x,y;
					if(r.r.prefs.ChordRotation == ChordRotation.Vertical){
						x = fretNoX - te.XBearing;
						y = fretNoY + r.fretGap*(fretNo - startFret) - te.YBearing - te.Height/2 ;
					} else {
						if(r.r.prefs.RightHanded == true){
							x = fretNoX + r.fretGap*(fretNo - startFret) - te.Width/2 - te.XBearing;
						} else {
							x = fretNoX - r.fretGap*(fretNo - startFret) - te.Width/2 - te.XBearing;
						}
						y = fretNoY - te.YBearing;
					}
					c.MoveTo(x, y);
					c.ShowText(text);
					c.Restore();	
				}

				Color color_from_hex(string hex)
				{
					string r = hex.Substring(1, 2);
					string g = hex.Substring(3, 2);
					string b = hex.Substring(5, 2);
					
					double rd = ((double)Int32.Parse(r, System.Globalization.NumberStyles.HexNumber))/256.0;
					double gd = ((double)Int32.Parse(g, System.Globalization.NumberStyles.HexNumber))/256.0;
					double bd = ((double)Int32.Parse(b, System.Globalization.NumberStyles.HexNumber))/256.0;
				//	System.Console.WriteLine("ChordCairoRenderer.ChordWidget.color_from_hex: r = {0}, rd = {1}\ng = {2}, gd = {3}\nb = {4}, bd = {5}",
				//	                         r, rd, g, gd, b, bd);
					return new Color(rd,gd,bd);
				}	      
			}
		}
	}
}