/*
   Copyright (C) 2008-2011 by Stefan Taferner <taferner@kde.org>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 3 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, see <http://www.gnu.org/licenses/>.
*/

#include "textstylewidget.h"

#include "i18n.h"
#include "textstyle.h"
#include "utils.h"

#include <kcolorbutton.h>

#include <QColorDialog>
#include <QFontDialog>
#include <QHBoxLayout>
#include <QLabel>
#include <QPushButton>
#include <QVBoxLayout>


#define KATS_BUTTONBLOCK_SPACING 6


TextStyleWidget::TextStyleWidget(KoverArtist::TextStyle* aStyle,
                                 Qt::Orientations aOrients, QWidget* aParent)
:Inherited(aParent)
,mStyle(aStyle)
,mBtnAlignLeft(0)
,mBtnAlignHCenter(0)
,mBtnAlignRight(0)
,mBtnAlignTop(0)
,mBtnAlignVCenter(0)
,mBtnAlignBottom(0)
,mBtnColor(0)
{
   QWidget *base = this;
   QPushButton *btn;

   QHBoxLayout *box = new QHBoxLayout(base);
   box->setMargin(0);
   box->setSpacing(2);

   btn = new QPushButton(KoverArtist::icon("format-text-bold"), QString::null, base);
   btn->setFixedSize(btn->sizeHint());
   btn->setToolTip(i18n("Change the font"));
   connect(btn, SIGNAL(clicked()), SLOT(changeFont()));
   box->addWidget(btn);

   mBtnColor = new KColorButton(Qt::black, base);
   mBtnColor->setFixedSize(mBtnColor->sizeHint());
   connect(mBtnColor, SIGNAL(changed(const QColor&)), SLOT(setColor(const QColor&)));
   mBtnColor->setToolTip(i18n("Change the text color"));
   box->addWidget(mBtnColor);

   box->addSpacing(KATS_BUTTONBLOCK_SPACING);

   if ((aOrients&Qt::Horizontal)==Qt::Horizontal)
   {
      mBtnAlignLeft = createButton("format-justify-left", base, SLOT(alignLeft()), box);
      mBtnAlignLeft->setToolTip(i18n("Align the text to the left side"));
      mBtnAlignLeft->setCheckable(true);

      mBtnAlignHCenter = createButton("format-justify-center", base, SLOT(alignHCenter()), box);
      mBtnAlignHCenter->setToolTip(i18n("Center the text horizontally"));
      mBtnAlignHCenter->setCheckable(true);

      mBtnAlignRight = createButton("format-justify-right", base, SLOT(alignRight()), box);
      mBtnAlignRight->setToolTip(i18n("Align the text to the right side"));
      mBtnAlignRight->setCheckable(true);

      box->addSpacing(KATS_BUTTONBLOCK_SPACING);
   }

   if ((aOrients&Qt::Vertical)==Qt::Vertical)
   {
      mBtnAlignTop = createButton("align-vertical-top", base, SLOT(alignTop()), box);
      mBtnAlignTop->setToolTip(i18n("Align the text to the top"));
      mBtnAlignTop->setCheckable(true);

      mBtnAlignVCenter = createButton("align-vertical-center", base, SLOT(alignVCenter()), box);
      mBtnAlignVCenter->setToolTip(i18n("Center the text vertically"));
      mBtnAlignVCenter->setCheckable(true);

      mBtnAlignBottom = createButton("align-vertical-bottom", base, SLOT(alignBottom()), box);
      mBtnAlignBottom->setToolTip(i18n("Align the text to the bottom"));
      mBtnAlignBottom->setCheckable(true);
   }

   box->addStretch(100);

   updateColorButton();
}


TextStyleWidget::~TextStyleWidget()
{
}


void TextStyleWidget::updateContents()
{
   mBtnColor->setColor(mStyle->color());

   if (mBtnAlignLeft)
   {
      int hf = (mStyle->alignment() & Qt::AlignHorizontal_Mask);
      mBtnAlignLeft->setChecked(hf==Qt::AlignLeft);
      mBtnAlignHCenter->setChecked(hf==Qt::AlignHCenter);
      mBtnAlignRight->setChecked(hf==Qt::AlignRight);
   }

   if (mBtnAlignTop)
   {
      int vf = (mStyle->alignment() & Qt::AlignVertical_Mask);
      mBtnAlignTop->setChecked(vf==Qt::AlignTop);
      mBtnAlignVCenter->setChecked(vf==Qt::AlignVCenter);
      mBtnAlignBottom->setChecked(vf==Qt::AlignBottom);
   }
}


void TextStyleWidget::updateColorButton()
{
}


QPushButton* TextStyleWidget::createButton(const char* aIcon, QWidget* aParent,
                                           const char* aSlot, QBoxLayout* aBox) const
{
   QPushButton *btn = new QPushButton(KoverArtist::icon(aIcon), QString::null, aParent);
   btn->setFixedSize(btn->sizeHint());
   connect(btn, SIGNAL(clicked()), aSlot);
   if (aBox) aBox->addWidget(btn);
   return btn;
}


void TextStyleWidget::changeFont()
{
   bool ok = false;
   QFont font = QFontDialog::getFont(&ok, mStyle->font(), this,
                                     i18n("KoverArtist - Select the Text Font"));
   if (!ok) return;

   mStyle->setFont(font);
   if (updatesEnabled()) emit changed();
}


void TextStyleWidget::setColor(const QColor& c)
{
   mStyle->setColor(c);
   if (updatesEnabled()) emit changed();
}


void TextStyleWidget::alignLeft()
{
   int f = int(mStyle->alignment() & Qt::AlignVertical_Mask);
   mStyle->setAlignment(Qt::Alignment(f | Qt::AlignLeft));
   mBtnAlignHCenter->setChecked(false);
   mBtnAlignRight->setChecked(false);
   if (updatesEnabled()) emit changed();
}


void TextStyleWidget::alignHCenter()
{
   int f = int(mStyle->alignment() & Qt::AlignVertical_Mask);
   mStyle->setAlignment(Qt::Alignment(f | Qt::AlignHCenter));
   mBtnAlignLeft->setChecked(false);
   mBtnAlignRight->setChecked(false);
   if (updatesEnabled()) emit changed();
}


void TextStyleWidget::alignRight()
{
   int f = int(mStyle->alignment() & Qt::AlignVertical_Mask);
   mStyle->setAlignment(Qt::Alignment(f | Qt::AlignRight));
   mBtnAlignLeft->setChecked(false);
   mBtnAlignHCenter->setChecked(false);
   if (updatesEnabled()) emit changed();
}


void TextStyleWidget::alignTop()
{
   int f = int(mStyle->alignment() & Qt::AlignHorizontal_Mask);
   mStyle->setAlignment(Qt::Alignment(f | Qt::AlignTop));
   mBtnAlignVCenter->setChecked(false);
   mBtnAlignBottom->setChecked(false);
   if (updatesEnabled()) emit changed();
}


void TextStyleWidget::alignVCenter()
{
   int f = int(mStyle->alignment() & Qt::AlignHorizontal_Mask);
   mStyle->setAlignment(Qt::Alignment(f | Qt::AlignVCenter));
   mBtnAlignTop->setChecked(false);
   mBtnAlignBottom->setChecked(false);
   if (updatesEnabled()) emit changed();
}


void TextStyleWidget::alignBottom()
{
   int f = int(mStyle->alignment() & Qt::AlignHorizontal_Mask);
   mStyle->setAlignment(Qt::Alignment(f | Qt::AlignBottom));
   mBtnAlignTop->setChecked(false);
   mBtnAlignVCenter->setChecked(false);
   if (updatesEnabled()) emit changed();
}
