/*
   Copyright (C) 2008-2011 by Stefan Taferner <taferner@kde.org>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 3 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, see <http://www.gnu.org/licenses/>.
*/

#include "taboptions.h"

#include "effectwidget.h"
#include "i18n.h"
#include "imagewidget.h"
#include "project.h"
#include "textstylewidget.h"

#include <kcolorbutton.h>

#include <QCheckBox>
#include <QHBoxLayout>
#include <QLabel>


TabOptions::TabOptions(KoverArtist::Project* aProject, QWidget* aParent)
:Inherited(aProject, aParent)
{
   QHBoxLayout* hbox;
   QLabel* lbl;

   //
   //  General
   //
   createCaption(i18n("General"));

   hbox = new QHBoxLayout;
   hbox->setSpacing(2);
   mMainBox->addLayout(hbox);

   lbl = new QLabel(i18n("Background")+':', this);
   lbl->setFixedSize(lbl->sizeHint());
   hbox->addWidget(lbl);

   mBtnBackground = new KColorButton(Qt::white, this);
   connect(mBtnBackground, SIGNAL(changed(const QColor&)),
           this, SLOT(setBackgroundColor(const QColor&)));
   hbox->addWidget(mBtnBackground);

   hbox->addSpacing(8);

   lbl = new QLabel(i18n("Outline")+':', this);
   lbl->setFixedSize(lbl->sizeHint());
   hbox->addWidget(lbl);

   mBtnOutline = new KColorButton(QColor("#888"), this);
   connect(mBtnOutline, SIGNAL(changed(const QColor&)),
           this, SLOT(setOutlineColor(const QColor&)));
   hbox->addWidget(mBtnOutline);

   hbox->addStretch(10);

   //
   //  Front Picture
   //
   createCaption(i18n("Front Picture"));

   mImwFront = new ImageWidget(&mProject->imgFront(), this);
   connect(mImwFront, SIGNAL(changed()), this, SIGNAL(changed()));
   mMainBox->addWidget(mImwFront);

   mCbxWrapImage = new QCheckBox(i18n("Wrap image around front and back side"), this);
   mCbxWrapImage->setFixedHeight(mCbxWrapImage->sizeHint().height());
   connect(mCbxWrapImage, SIGNAL(toggled(bool)), SLOT(setWrapImage(bool)));
   mMainBox->addWidget(mCbxWrapImage);

   mCbxFrontTitleHide = new QCheckBox(i18n("Hide title text on front side"), this);
   mCbxFrontTitleHide->setFixedHeight(mCbxFrontTitleHide->sizeHint().height());
   connect(mCbxFrontTitleHide, SIGNAL(toggled(bool)), SLOT(setFrontTitleHide(bool)));
   mMainBox->addWidget(mCbxFrontTitleHide);

   //
   //  Back Picture
   //
   mLblCaptionBack = createCaption(i18n("Back Picture"));

   mImwBack = new ImageWidget(&mProject->imgBack(), this);
   connect(mImwBack, SIGNAL(changed()), this, SIGNAL(changed()));
   mMainBox->addWidget(mImwBack);

   //
   //  Disc-contents style and effect options
   //
   createCaption(i18n("Disc Contents Text"));
   mTswContents = new TextStyleWidget(&mProject->contentsStyle(),
                                      Qt::Horizontal, this);
   connect(mTswContents, SIGNAL(changed()), this, SIGNAL(changed()));
   mMainBox->addWidget(mTswContents);

   mEffContents = new EffectWidget(&mProject->contentsEffect(), this);
   connect(mEffContents, SIGNAL(changed()), this, SIGNAL(changed()));
   mMainBox->addWidget(mEffContents);

   //
   //  Side-text style and effect options
   //
   createCaption(i18n("Side Text"));
   mTswSide = new TextStyleWidget(&mProject->sideStyle(),
                                  Qt::Horizontal, this);
   connect(mTswSide, SIGNAL(changed()), this, SIGNAL(changed()));
   mMainBox->addWidget(mTswSide);

   mEffSide = new EffectWidget(&mProject->sideEffect(), this);
   connect(mEffSide, SIGNAL(changed()), this, SIGNAL(changed()));
   mMainBox->addWidget(mEffSide);

   mMainBox->addStretch(100);
   updateContents();
}


TabOptions::~TabOptions()
{
}


void TabOptions::updateContents()
{
   mBtnBackground->setColor(mProject->backgroundColor());
   mBtnOutline->setColor(mProject->outlineColor());

   mImwFront->updateContents();
   mImwBack->updateContents();
   mTswContents->updateContents();
   mEffContents->updateContents();
   mTswSide->updateContents();
   mEffSide->updateContents();

   bool doWrap = mProject->imgFrontWrapAround();
   mCbxWrapImage->setChecked(doWrap);
   mImwBack->setEnabled(!doWrap);
   mLblCaptionBack->setEnabled(!doWrap);

   mCbxFrontTitleHide->setChecked(mProject->frontTitleHide());
}


void TabOptions::setBackgroundColor(const QColor& aColor)
{
   mProject->setBackgroundColor(aColor);
   emit changed();
}


void TabOptions::setOutlineColor(const QColor& aColor)
{
   mProject->setOutlineColor(aColor);
   emit changed();
}


void TabOptions::setWrapImage(bool aEnable)
{
   mProject->setImgFrontWrapAround(aEnable);
   mImwBack->setEnabled(!aEnable);
   mLblCaptionBack->setEnabled(!aEnable);
   emit changed();
}


void TabOptions::setFrontTitleHide(bool aEnable)
{
   mProject->setFrontTitleHide(aEnable);
   emit changed();
}
