/*
   Copyright (C) 2008-2011 by Stefan Taferner <taferner@kde.org>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 3 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, see <http://www.gnu.org/licenses/>.
*/

#include "imagewidget.h"

#include "i18n.h"
#include "image.h"
#include "project.h"
#include "tabbase.h"

#include <QtGui/QCheckBox>
#include <QtGui/QComboBox>
#include <QtGui/QFileDialog>
#include <QtGui/QHBoxLayout>
#include <QtGui/QLabel>
#include <QtGui/QLineEdit>
#include <QtGui/QPushButton>
#include <QtGui/QVBoxLayout>


ImageWidget::ImageWidget(KoverArtist::Image* aImage, TabBase* aParent)
:Inherited(aParent)
,mImage(aImage)
{
   QWidget *base = this;
   QHBoxLayout *hbox = 0;
   QLabel *lbl;

   QVBoxLayout *vbox = new QVBoxLayout(base);
   vbox->setMargin(0);
   vbox->setSpacing(2);

   //
   //  File name line-edit and file selection button
   //
   hbox = new QHBoxLayout;
   hbox->setSpacing(2);
   vbox->addLayout(hbox);

   mEdtFileName = new QLineEdit(base);
   mEdtFileName->setFixedHeight(mEdtFileName->sizeHint().height());
   connect(mEdtFileName, SIGNAL(editingFinished()), SLOT(fileNameEdited()));
   connect(mEdtFileName, SIGNAL(returnPressed()), SLOT(fileNameEdited()));
   hbox->addWidget(mEdtFileName, 100);

   mBtnFileSelect = new QPushButton("...", base);
   int h = mEdtFileName->sizeHint().height();
   mBtnFileSelect->setFixedSize(h, h);
   connect(mBtnFileSelect, SIGNAL(clicked()), SLOT(selectFile()));
   hbox->addWidget(mBtnFileSelect, 0);

   //
   //  Image position
   //
   hbox = new QHBoxLayout;
   hbox->setSpacing(2);
   vbox->addLayout(hbox);

   lbl = new QLabel(i18n("Position")+':', base);
   lbl->setFixedSize(lbl->sizeHint());
   hbox->addWidget(lbl);

   mCboPosition = new QComboBox(base);
   mCboPosition->addItem(i18n("Centered"), int(KoverArtist::Centered));
   mCboPosition->addItem(i18n("Tiled"), int(KoverArtist::Tiled));
   mCboPosition->addItem(i18n("Centered Tiled"), int(KoverArtist::CenteredTiled));
   mCboPosition->addItem(i18n("Centered Maxpect"), int(KoverArtist::CenteredMaxpect));
   mCboPosition->addItem(i18n("Scaled"), int(KoverArtist::Scaled));
   mCboPosition->addItem(i18n("Scale & Crop"), int(KoverArtist::ScaleCrop));
   mCboPosition->setMinimumSize(mCboPosition->sizeHint());
   connect(mCboPosition, SIGNAL(activated(int)), this, SLOT(setPosition(int)));
   hbox->addWidget(mCboPosition, Qt::AlignLeft);

   //
   //  "Image covers sides" checkbox
   //
   mCbxOnSides = new QCheckBox(i18n("Image covers sides"), base);
   mCbxOnSides->setFixedSize(mCbxOnSides->sizeHint());
   connect(mCbxOnSides, SIGNAL(toggled(bool)), this, SLOT(setOnSides(bool)));
   vbox->addWidget(mCbxOnSides, Qt::AlignLeft);


   updateContents();
}


ImageWidget::~ImageWidget()
{
}


void ImageWidget::updateContents()
{
   bool upd = updatesEnabled();
   setUpdatesEnabled(false);

   mEdtFileName->setText(mImage->url());
   mCbxOnSides->setChecked(mImage->onSides());

   int idx = mCboPosition->findData(int(mImage->position()));
   mCboPosition->setCurrentIndex(idx);

   setUpdatesEnabled(upd);
}


void ImageWidget::selectFile()
{
   QString fname = mImage->url();
   if (fname.isEmpty())
   {
      TabBase* par = dynamic_cast<TabBase*>(parent());
      if (par) fname = par->project()->dirPath();
      if (fname.isEmpty()) fname = QDir::currentPath();
   }

   fname = QFileDialog::getOpenFileName(this,
               i18n("Load an image")+" - KoverArtist", fname,
               i18n("Images (*.png *.jpg *.gif *.xpm)\n"
                    "All Files (*)\n"));
   if (fname.isEmpty()) return;

   setFileName(fname);
}


void ImageWidget::fileNameEdited()
{
   QString fname = mEdtFileName->text();
   if (fname==mImage->url()) return;
   mImage->load(fname);
   if (updatesEnabled()) emit changed();
}


void ImageWidget::setFileName(const QString& aFileName)
{
   if (aFileName==mImage->url()) return;
   mImage->load(aFileName);
   mEdtFileName->setText(mImage->url());
   if (updatesEnabled()) emit changed();
}


void ImageWidget::setPosition(int aIdx)
{
   int ip = mCboPosition->itemData(aIdx).toInt();
   mImage->setPosition((KoverArtist::ImagePosition)ip);
   if (updatesEnabled()) emit changed();
}


void ImageWidget::setOnSides(bool aOnSides)
{
   if (aOnSides==mImage->onSides()) return;
   mImage->setOnSides(aOnSides);
   if (updatesEnabled()) emit changed();
}
