/*
   Copyright (C) 2008-2011 by Stefan Taferner <taferner@kde.org>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 3 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, see <http://www.gnu.org/licenses/>.
*/

#include "case.h"

#include "config.h"
#include "i18n.h"

#include <QtCore/QDir>
#include <QtXml/QXmlDefaultHandler>
#include <QtXml/QXmlReader>

#include <QtGui/QMessageBox>


namespace KoverArtist
{

QVector<Case> Case::sCases;


//
//  CLASS  CaseLoader
//

class CaseLoader: public QXmlDefaultHandler
{
public:
   CaseLoader(Case* c, const QString& fn);
   virtual ~CaseLoader();

   virtual bool startDocument();
   virtual bool startElement(const QString&, const QString&, const QString&,
                             const QXmlAttributes&);
   virtual bool endElement(const QString&, const QString&, const QString&);
   virtual bool characters(const QString&);

protected:
   Case* mCase;
   int mVersion;
   QString mPath, mFileName;
};


CaseLoader::CaseLoader(Case* aCase, const QString& aFileName)
   :QXmlDefaultHandler()
   ,mCase(aCase)
   ,mVersion(0)
   ,mPath()
   ,mFileName(aFileName)
{
}


CaseLoader::~CaseLoader()
{
}


bool CaseLoader::startDocument()
{
   mPath = "";
   mVersion = 0;
   mCase->clear();
   mCase->mName = "";
   mCase->mLabel = "";
   return true;
}


bool CaseLoader::startElement(const QString&, const QString&, const QString& aName,
                              const QXmlAttributes& aAtts)
{
   mPath += "/"+aName;
   //std::cout<<mPath<<std::endl;

   if (mPath=="/koverartistcase")
   {
      mVersion = aAtts.value("version").toInt();
      if (mVersion<1) mVersion = 1;
   }
   else if (mPath=="/koverartistcase/Label")
   {
   }
   else if (mPath=="/koverartistcase/FrontBackConnected")
   {
   }
   else if (mPath=="/koverartistcase/Front")
   {
      mCase->mFront.setWidth(aAtts.value("width").toInt());
      mCase->mFront.setHeight(aAtts.value("height").toInt());
   }
   else if (mPath=="/koverartistcase/Back")
   {
      mCase->mBack.setWidth(aAtts.value("width").toInt());
      mCase->mBack.setHeight(aAtts.value("height").toInt());
   }
   else if (mPath=="/koverartistcase/BackLeftSide")
   {
      mCase->mBackLeftSide = aAtts.value("width").toInt();
   }
   else if (mPath=="/koverartistcase/BackRightSide")
   {
      mCase->mBackRightSide = aAtts.value("width").toInt();
   }
   else if (mPath=="/koverartistcase/FrontLeftSide")
   {
      mCase->mFrontLeftSide = aAtts.value("width").toInt();
   }
   else if (mPath=="/koverartistcase/FrontRightSide")
   {
      mCase->mFrontRightSide = aAtts.value("width").toInt();
   }
   else
   {
      QMessageBox::critical(0, i18n("KoverArtist Error"),
         i18n("Error reading %1:\nInvalid xml element: %2").arg(mFileName).arg(mPath));
      return false;
   }

   return true;
}



bool CaseLoader::characters(const QString& aData)
{
   QString data = aData.trimmed();
   if (mPath=="/koverartistcase/Label")
   {
      mCase->mLabel = data;
   }
   else if (mPath=="/koverartistcase/FrontBackConnected")
   {
      if (data=="left") mCase->mConnected = Case::ConnectLeft;
      else if (data=="right") mCase->mConnected = Case::ConnectRight;
      else if (data=="none") mCase->mConnected = Case::NotConnected;
      else
      {
         QMessageBox::critical(0, i18n("KoverArtist Error"),
            i18n("Error reading %1:\nInvalid 'Connected' value: %2").arg(mFileName).arg(data));
         return false;
      }
   }
   return true;
}


bool CaseLoader::endElement(const QString&, const QString&, const QString&)
{
   int idx = mPath.lastIndexOf('/');
   if (idx>0) mPath = mPath.left(idx);
   else mPath = "";
   return true;
}



//
//  CLASS  Case
//

Case::Case()
:mName()
,mLabel()
,mConnected(Case::NotConnected)
,mFront(0, 0)
,mBack(0, 0)
,mBackLeftSide(0)
,mBackRightSide(0)
,mFrontLeftSide(0)
,mFrontRightSide(0)
{
}


Case::Case(const Case& o)
:mName(o.mName)
,mLabel(o.mLabel)
,mConnected(o.mConnected)
,mFront(o.mFront)
,mBack(o.mBack)
,mBackLeftSide(o.mBackLeftSide)
,mBackRightSide(o.mBackRightSide)
,mFrontLeftSide(o.mFrontLeftSide)
,mFrontRightSide(o.mFrontRightSide)
{
}


Case::~Case()
{
}


Case& Case::operator=(const Case& o)
{
   if (&o!=this)
   {
      mName = o.mName;
      mLabel = o.mLabel;
      mConnected = o.mConnected;
      mFront = o.mFront;
      mBack = o.mBack;
      mBackLeftSide = o.mBackLeftSide;
      mBackRightSide = o.mBackRightSide;
      mFrontLeftSide = o.mFrontLeftSide;
      mFrontRightSide = o.mFrontRightSide;
   }
   return *this;
}


void Case::clear()
{
   mName = "";
   mLabel = "";
   mConnected = NotConnected;
   mFront = QSize(0, 0);
   mBack = QSize(0, 0);
   mBackLeftSide = 0;
   mBackRightSide = 0;
   mFrontLeftSide = 0;
   mFrontRightSide = 0;
}


Case& Case::standardCase()
{
   static Case stdCase;
   if (stdCase.mFront.isNull())
   {
      stdCase.mName = "cd-standard";
      stdCase.mLabel = "Default Case";
      stdCase.mConnected = NotConnected;
      stdCase.mFront = QSize(1210, 1210);
      stdCase.mBack = QSize(1370, 1180);
      stdCase.mBackLeftSide = 60;
      stdCase.mBackRightSide = 60;
      stdCase.mFrontLeftSide = 0;
      stdCase.mFrontRightSide = 0;
   }
   return stdCase;
}


bool Case::loadFile(const QString& aFileName)
{
   clear();

   CaseLoader loader(this, aFileName);
   QFile file(aFileName);
   QXmlInputSource source(&file);
   QXmlSimpleReader reader;
   reader.setContentHandler(&loader);
   if (!reader.parse(&source, false)) return false;

   int idx = aFileName.lastIndexOf('.');
   if (idx>0) mName = aFileName.left(idx).toLower();
   else mName = aFileName.toLower();

   idx = mName.lastIndexOf('/');
   if (idx>=0) mName = mName.mid(idx+1);

   return true;
}


bool Case::loadCases()
{
   bool ok = true;
   sCases.clear();

   // Try to load the cases from the local subdirectories "cases" and "../cases".
   // This is mainly for testing KoverArtist without installing it.
   loadCases("../cases");
   loadCases("cases");

   loadCases(QDir::homePath()+ "/.kde/share/apps/koverartist/cases");
   loadCases(APPDATA_PREFIX "/cases");

   if (sCases.isEmpty())
   {
      ok = false;

      QMessageBox::warning(0, i18n("KoverArtist Warning"),
         i18n("KoverArtist did not find any case definitions.\n"
              "This looks like a broken installation.\n"
              "\n"
              "Only the builtin standard cd case will be\n"
              "available."));
      sCases.push_back(standardCase());
   }

   return ok;
}


bool Case::loadCases(const QString& aDirName)
{
   QDir dir(aDirName, "*.koac", QDir::Unsorted);
   bool ok = true;
   QString nm;

   if (!dir.exists()) return true;

   for (int i=dir.count()-1; i>=0; --i)
   {
      nm = dir[i];
      if (findCase(nm)) continue;

      Case c;
      if (c.loadFile(aDirName+'/'+nm)) sCases.push_back(c);
      else ok = false;
   }
   return ok;
}


Case* Case::findCase(const QString& aName)
{
   int i;
   for (i=sCases.count()-1; i>=0; --i)
      if (sCases[i].name()==aName) return &sCases[i];
   return 0;
}


int Case::indexOfCase(const QString& aName)
{
   int i;
   for (i=sCases.count()-1; i>=0; --i)
      if (sCases[i].name()==aName) break;
   return i;
}


bool Case::operator<(const Case& o) const
{
   return mLabel<o.mLabel;
}


} //namespace
