#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <errno.h>
#include <stdarg.h>
#include <dirent.h>
#include <fcntl.h>
#include <inttypes.h>

#include <sys/mman.h>
#include <sys/types.h>

#include <xenctrl.h>

#include "xennerctrl.h"
#include "xenner.h"

/* -------------------------------------------------------------- */

#define DEBUGFS_PATH  "/sys/kernel/debug"
#define KVM_MAX       32

static char *kvm_names[KVM_MAX];
static int kvm_count;

static int kvm_init_stats(void)
{
    struct dirent *ent;
    DIR *dir;

    dir = opendir(DEBUGFS_PATH "/kvm");
    if (NULL == dir) {
	fprintf(stderr, "debugfs not mounted at %s -> no kvm stats, sorry\n",
		DEBUGFS_PATH);
	sleep(3);
	return 0;
    }

    while (kvm_count < KVM_MAX && NULL != (ent = readdir(dir))) {
	if ('.' == ent->d_name[0])
	    continue;
	kvm_names[kvm_count] = strdup(ent->d_name);
	kvm_count++;
    }
    closedir(dir);
    return kvm_count;
}

static int kvm_get_stats(uint64_t *vals)
{
    char path[128], buf[64];
    int fd,i, rc;

    for (i = 0; i < kvm_count; i++) {
	snprintf(path, sizeof(path), "%s/kvm/%s", DEBUGFS_PATH, kvm_names[i]);
	fd = open(path, O_RDONLY);
	if (-1 == fd) {
	    fprintf(stderr, "open %s: %s\n", path, strerror(errno));
	    sleep(3);
	    return -1;
	}
	rc = read(fd, buf, sizeof(buf));
	if (-1 == rc) {
	    fprintf(stderr, "read %s: %s\n", path, strerror(errno));
	    sleep(3);
	    return -1;
	}
	buf[rc] = 0;
	close(fd);
	sscanf(buf, "%" PRId64, vals+i);
    }
    return 0;
}

static void kvm_print_stats(uint64_t *now, uint64_t *last)
{
    int i;

    logprintf(NULL, "%-22s : %10s %8s\n", "kvm stats", "total", "diff");
    for (i = 0; i < kvm_count; i++) {
	if (!now[i])
	    continue;
	logprintf(NULL, "  %-20s : %10" PRId64 " %8" PRId64 "\n",
		  kvm_names[i], now[i], last ? now[i] - last[i] : 0);
    }
}

/* -------------------------------------------------------------- */

static int do_stats(int domid, int interval)
{
    uint64_t           last_faults[XEN_FAULT_MAX];
    uint64_t           last_hcalls[XEN_HCALL_MAX];
    uint64_t           last_events[256];
    uint64_t           last_kvm[KVM_MAX];
    uint64_t           this_kvm[KVM_MAX];
    struct xennerdom   *dom;
    struct stat        st;

    dom = xenner_get_dom(domid);
    if (NULL == dom) {
	fprintf(stderr, "failed to open vmcore file\n");
	goto cleanup;
    }
    if (!dom->vminfo) {
	fprintf(stderr, "failed to parse vmcore file\n");
	goto cleanup;
    }

    fprintf(stderr, "collecting stats ...\n");
    kvm_get_stats(this_kvm);
    for (;;) {
	memcpy(last_hcalls, dom->vminfo->hcalls, sizeof(last_hcalls));
	memcpy(last_faults, dom->vminfo->faults, sizeof(last_faults));
	memcpy(last_events, dom->vminfo->events, sizeof(last_faults));
	memcpy(last_kvm,    this_kvm,            sizeof(last_kvm));
	sleep(interval);
	if (dom->vminfo->dying) {
	    fprintf(stderr, "\nvm dying\n");
	    break;
	}
	if (-1 == stat(dom->filename, &st)) {
	    fprintf(stderr, "\nvm gone\n");
	    break;
	}
	kvm_get_stats(this_kvm);
	fprintf(stderr,
		"\x1b[H" /* cursor home  */
		"\x1b[J" /* clear screen */
		"<%s>\n", dom->filename);
	print_hcall_stats(NULL, dom->vminfo->hcalls, last_hcalls);
	print_fault_stats(NULL, dom->vminfo->faults, last_faults);
	print_event_stats(NULL, dom->vminfo->events, last_events, dom->vminfo->enames);
	kvm_print_stats(this_kvm, last_kvm);
    }

cleanup:
    if (dom)
	xenner_put_dom(dom);
    return 0;
}

/* -------------------------------------------------------------- */

int interval = 1;

static void usage(FILE *fp)
{
    fprintf(fp,
	    "\n"
	    "xenner-stats  --  xenner vm statistics\n"
	    "\n"
	    "usage: xenner-stats [options] [domid]\n"
	    "options:\n"
	    "   -h            print this text\n"
	    "   -i <secs>     stats interval                [%d]\n"
	    "\n"
	    "With a domid specified just that one domain is monitored\n"
	    "until it exits.  Otherwise the tool picks one guest, once\n"
	    "it is gone it picks another until stopped with ^C.\n"
	    "\n"
	    "-- \n"
	    "(c) 2007,08 Gerd Hoffmann <kraxel@redhat.com>\n"
	    "\n",
	    interval);
}

int main(int argc, char *argv[])
{
    xc_dominfo_t info;
    int xc, count = 0;

    int c;

    for (;;) {
        if (-1 == (c = getopt(argc, argv, "hi:")))
            break;
        switch (c) {
	case 'i':
	    interval = atoi(optarg);
	    break;
        case 'h':
            usage(stdout);
            exit(0);
        default:
            usage(stderr);
            exit(1);
        }
    }

    kvm_init_stats();

    if (optind < argc)
	return do_stats(atoi(argv[optind]), interval);

    xc = xc_interface_open();
    if (-1 == xc) {
        fprintf(stderr, "can't open xen interface\n");
        exit(1);
    }

    for (;;) {
	if (1 == xc_domain_getinfo(xc, 1, 1, &info)) {
	    if (count)
		fprintf(stderr, "\n");
	    count = 0;
	    do_stats(info.domid, interval);
	    sleep(interval);
	} else {
	    count++;
	    fprintf(stderr, "no vm running (for %d secs)\r", count*interval);
	    sleep(interval);
	}
    }
    xc_interface_close(xc);
}
