#ifndef __XENNER_EMUDEV_H__
#define __XENNER_EMUDEV_H__ 1

/*
 * I/O ports for the virtual emu device
 *     - CONF_ENTRY (write)
 *       bits 31 .. 16  --  config type
 *       bits 15 ..  0  --  config index
 *     - CONF_VALUE (read/write)
 *       32bit config value
 *     - COMMAND (write)
 *       bits 32 .. 16  --  command
 *       bits 15 ..  0  --  argument
 */
#define EMUDEV_REG_BASE                      (0xe0)
#define EMUDEV_REG_CONF_ENTRY                (EMUDEV_REG_BASE)
#define EMUDEV_REG_CONF_VALUE                (EMUDEV_REG_BASE+4)
#define EMUDEV_REG_COMMAND                   (EMUDEV_REG_BASE+8)
#define EMUDEV_REG_RESERVED                  (EMUDEV_REG_BASE+12)

/* simple entries, read */
#define EMUDEV_CONF_DEBUG_LEVEL              0x00
#define EMUDEV_CONF_EMU_START_PFN            0x01
#define EMUDEV_CONF_EMU_PAGE_COUNT           0x02
#define EMUDEV_CONF_M2P_START_PFN            0x03
#define EMUDEV_CONF_M2P_PAGE_COUNT           0x04
#define EMUDEV_CONF_GUEST_START_PFN          0x05
#define EMUDEV_CONF_GUEST_PAGE_COUNT         0x06
#define EMUDEV_CONF_TOTAL_PAGE_COUNT         0x07
#define EMUDEV_CONF_NR_VCPUS                 0x08
#define EMUDEV_CONF_HVM_XENSTORE_PFN         0x09
#define EMUDEV_CONF_HVM_XENSTORE_EVTCHN      0x0a
/* simple entries, write */
#define EMUDEV_CONF_BOOT_CTXT_PFN            0x40
#define EMUDEV_CONF_SHARED_INFO_PFN          0x41
#define EMUDEV_CONF_NEXT_SECONDARY_VCPU      0x42
#define EMUDEV_CONF_HVM_CALLBACK_IRQ         0x43
#define EMUDEV_CONF_VMINFO_PFN               0x70 /* temporary */
/* simple count */
#define EMUDEV_CONF_SIMPLE_COUNT             0x80

/* indexed config entries */
#define EMUDEV_CONF_GRANT_TABLE_PFNS         0x80
#define EMUDEV_CONF_GRANT_TABLE_COUNT        16
#define EMUDEV_CONF_EVTCHN_TO_PIN            0x81
#define EMUDEV_CONF_EVTCHN_TO_PIN_COUNT      64
#define EMUDEV_CONF_COMMAND_RESULT           0x82
#define EMUDEV_CONF_COMMAND_RESULT_COUNT     64

/* commands */
#define EMUDEV_CMD_NOP                       1
#define EMUDEV_CMD_WRITE_CHAR                2
#define EMUDEV_CMD_CONFIGURATION_DONE        3
#define EMUDEV_CMD_EVTCHN_ALLOC              4
#define EMUDEV_CMD_EVTCHN_SEND               5
#define EMUDEV_CMD_INIT_SECONDARY_VCPU       6
#define EMUDEV_CMD_GUEST_SHUTDOWN            7
#define EMUDEV_CMD_EVTCHN_CLOSE              8

/* --------- host side bits --------- */

struct emudev_state {
    uint32_t   entry;
    uint32_t   config[EMUDEV_CONF_SIMPLE_COUNT];
    uint32_t   gnttab[EMUDEV_CONF_GRANT_TABLE_COUNT];
    uint32_t   evtchn[EMUDEV_CONF_EVTCHN_TO_PIN_COUNT];
    uint32_t   result[EMUDEV_CONF_COMMAND_RESULT_COUNT];
};

void emudev_write_entry(struct emudev_state *e, uint32_t value);
void emudev_write_value(struct emudev_state *e, uint32_t value);
uint32_t emudev_read_value(struct emudev_state *e);

/* --------- guest side bits --------- */

static inline void emudev_set(uint16_t type, uint16_t index, uint32_t value)
{
    uint32_t entry = (uint32_t)type << 16 | index;

    asm volatile("outl %[data],%w[port]\n"
		 : /* no output */
		 : [data] "a" (entry), [port] "Nd" (EMUDEV_REG_CONF_ENTRY)
		 : "memory");
    asm volatile("outl %[data],%w[port]\n"
		 : /* no output */
		 : [data] "a" (value), [port] "Nd" (EMUDEV_REG_CONF_VALUE)
		 : "memory");
}

static inline uint32_t emudev_get(uint16_t type, uint16_t index)
{
    uint32_t entry = (uint32_t)type << 16 | index;
    uint32_t value;

    asm volatile("outl %[data],%w[port]\n"
		 : /* no output */
		 : [data] "a" (entry), [port] "Nd" (EMUDEV_REG_CONF_ENTRY)
		 : "memory");
    asm volatile("inl %w[port],%[data]\n"
		 : [data] "=a" (value)
		 : [port] "Nd" (EMUDEV_REG_CONF_VALUE)
		 : "memory");
    return value;
}

static inline void emudev_cmd(uint16_t cmd, uint16_t arg)
{
    uint32_t command = (uint32_t)cmd << 16 | arg;

    asm volatile("outl %[data],%w[port]\n"
		 : /* no output */
		 : [data] "a" (command), [port] "Nd" (EMUDEV_REG_COMMAND)
		 : "memory");
}
#endif /* __XENNER_EMUDEV_H__ */
