#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <errno.h>
#include <stdarg.h>
#include <dirent.h>
#include <fcntl.h>
#include <inttypes.h>

#include <xs.h>
#include <xenctrl.h>

#include "xennerctrl.h"
#include "xenner.h"

/* -------------------------------------------------------------- */

static void cleanup_xenstore(int domid)
{
    char *backs[] = {
	"blkbackd", "qdisk",
        "netbackd", "qnic",
        "vfb", "vkbd", "console"
    };
    char path[128], *val;
    struct xs_handle *xs;
    unsigned int len;
    int i;

    xs = xs_daemon_open();
    if (NULL == xs)
	return;

    xs_release_domain(xs, domid);

    /* delete /vm/<uuid> */
    snprintf(path, sizeof(path), "/local/domain/%d/vm", domid);
    val = xs_read(xs, 0, path, &len);
    if (val)
	xs_rm(xs, 0, val);

    /* delete domain & frontends */
    snprintf(path, sizeof(path), "/local/domain/%d", domid);
    xs_rm(xs, 0, path);

    /* delete backends */
    for (i = 0; i < sizeof(backs)/sizeof(backs[0]); i++) {
	snprintf(path, sizeof(path), "/local/domain/0/backend/%s/%d",
		 backs[i], domid);
	xs_rm(xs, 0, path);
    }

    xs_daemon_close(xs);
}

static int handle_domain(int domid, int secs)
{
    struct xennerdom *dom;
    uint64_t events;
    struct timespec ts;
    int i;

    fprintf(stderr, "Checking domain %d ... ", domid);

    dom = xenner_get_dom(domid);
    if (NULL == dom) {
	fprintf(stderr, "failed to open vmcore file\n");
	goto cleanup;
    }
    if (!dom->vminfo) {
	fprintf(stderr, "failed to parse vmcore file\n");
	goto cleanup;
    }

    events = dom->vminfo->faults[XEN_FAULT_EVENT_CALLBACK];
    for (i = 0; i < secs * 100; i++) {
	ts.tv_sec = 0;
	ts.tv_nsec = 10000000;
	nanosleep(&ts, NULL);
	if (events != dom->vminfo->faults[XEN_FAULT_EVENT_CALLBACK]) {
	    fprintf(stderr, "alive\n");
	    goto cleanup;
	}
    }

    fprintf(stderr, "stale ... ");
    dom->vminfo->dying = 1;
    unlink(dom->filename);
    cleanup_xenstore(domid);

    fprintf(stderr, "wiped\n");

cleanup:
    if (dom)
	xenner_put_dom(dom);
    return 0;
}

/* -------------------------------------------------------------- */

static void usage(FILE *fp)
{
    fprintf(fp,
	    "\n"
	    "xenner-cleanup  --  cleanup stale domains.\n"
	    "\n"
	    "options:\n"
	    "   -h        print this text\n"
	    "\n"
	    "You might need this when xenner crashes hard and doesn't\n"
	    "manage to cleanup.  Which hopefully never happens with\n"
	    "releases.\n"
	    "\n"
	    "Helps also after libvirtd killing off xenner with SIGKILL.\n"
	    "Unfortunaly some versions out there have no delay between\n"
	    "sending STGTERM and SIGKILL, so this actually can happen.\n"
	    "\n"
	    "-- \n"
	    "(c) 2008 Gerd Hoffmann <kraxel@redhat.com>\n"
	    "\n");
}

int main(int argc, char *argv[])
{
    xc_dominfo_t info;
    int domid;
    int xc, count = 0;
    int c;

    for (;;) {
        if (-1 == (c = getopt(argc, argv, "h")))
            break;
        switch (c) {
        case 'h':
            usage(stdout);
            exit(0);
        default:
            usage(stderr);
            exit(1);
        }
    }

    xc = xc_interface_open();
    if (-1 == xc) {
        fprintf(stderr, "can't open xen interface\n");
        exit(1);
    }

    for (domid = 0; ; domid = info.domid+1) {
	if (0 == xc_domain_getinfo(xc, domid, 1, &info))
	    break;
        if (info.domid == 0 && domid > 0)
            break;
	handle_domain(info.domid, 3);
	count++;
    }
    xc_interface_close(xc);

    fprintf(stderr, "I'm done, %d domains checked.\n", count);
    return 0;
}
