#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <errno.h>
#include <inttypes.h>

#include <xenctrl.h>
#include "xen/names.h"
#include "xenner.h"

/* -------------------------------------------------------------- */

static const char *xen_fault_names[] = {
    [ XEN_FAULT_ILLEGAL_INSTRUCTION ]       = "illegal instruction",

    [ XEN_FAULT_GENERAL_PROTECTION ]        = "general protection",
    [ XEN_FAULT_GENERAL_PROTECTION_GUEST ]  = "  gpf  guest",
    [ XEN_FAULT_GENERAL_PROTECTION_EMUINS ] = "  gpf  emu instr",

    [ XEN_FAULT_PAGE_FAULT ]                = "page fault",
    [ XEN_FAULT_PAGE_FAULT_GUEST ]          = "  pf  guest",
    [ XEN_FAULT_PAGE_FAULT_FIX_RO ]         = "  pf  fixup readonly",
    [ XEN_FAULT_PAGE_FAULT_FIX_USER ]       = "  pf  fixup usermode",
    [ XEN_FAULT_PAGE_FAULT_FIX_EXTAB ]      = "  pf  fixup extable",
    [ XEN_FAULT_UPDATE_VA_FIX_RO ]          = "  uva fixup readonly",
    [ XEN_FAULT_UPDATE_VA_FIX_USER ]        = "  uva fixup usermode",

    [ XEN_FAULT_SYSCALL ]                   = "syscall   (64bit)",
    [ XEN_FAULT_INT_80 ]                    = "int 0x80  (32bit)",
    [ XEN_FAULT_EVENT_CALLBACK ]            = "event callback",
    [ XEN_FAULT_LAZY_FPU ]                  = "lazy fpu",
    [ XEN_FAULT_BOUNCE_TRAP ]               = "bounce trap",

    [ XEN_FAULT_MAPS_MAPIT ]                = "pagemap mapit",
    [ XEN_FAULT_MAPS_REUSE ]                = "pagemap reuse",

    [ XEN_FAULT_OTHER_CR3_LOAD ]            = "cr3 load",
    [ XEN_FAULT_OTHER_SWITCH_MODE ]         = "switch mode",
    [ XEN_FAULT_OTHER_CR3_CACHE_HIT ]       = "cr3 cache hit",
    [ XEN_FAULT_OTHER_FLUSH_TLB_ALL ]       = "remote tlb flush",
    [ XEN_FAULT_OTHER_FLUSH_TLB_PAGE ]      = "remote tlb page",
    [ XEN_FAULT_OTHER_FLUSH_TLB_NONE ]      = "remote tlb nop",

    [ XEN_FAULT_TMP_1 ]                     = "ad-hoc #1",
    [ XEN_FAULT_TMP_2 ]                     = "ad-hoc #2",
    [ XEN_FAULT_TMP_3 ]                     = "ad-hoc #3",
    [ XEN_FAULT_TMP_4 ]                     = "ad-hoc #4",
    [ XEN_FAULT_TMP_5 ]                     = "ad-hoc #5",
    [ XEN_FAULT_TMP_6 ]                     = "ad-hoc #6",
    [ XEN_FAULT_TMP_7 ]                     = "ad-hoc #7",
    [ XEN_FAULT_TMP_8 ]                     = "ad-hoc #8",
};


void print_hcall_stats(struct xenvm *xen, uint64_t *hcalls, uint64_t *last)
{
    int i;

    logprintf(xen, "%-22s : %10s %8s\n", "hypercalls", "total", "diff");
    for (i = 0; i < XEN_HCALL_MAX; i++) {
	if (!hcalls[i])
	    continue;
	logprintf(xen, "  %-20s : %10" PRId64 " %8" PRId64 "\n",
		  __hypervisor_name(i), hcalls[i],
		  last ? hcalls[i] - last[i] : 0);
    }
}

void print_fault_stats(struct xenvm *xen, uint64_t *faults, uint64_t *last)
{
    int i;

    logprintf(xen, "%-22s : %10s %8s\n", "emu faults", "total", "diff");
    for (i = 0; i < XEN_FAULT_MAX; i++) {
	if (!faults[i])
	    continue;
	logprintf(xen, "  %-20s : %10" PRId64 " %8" PRId64 "\n",
		  GETNAME(xen_fault_names, i),
		  faults[i], last ? faults[i] - last[i] : 0);
    }
}

void print_event_stats(struct xenvm *xen, uint64_t *events, uint64_t *last,
		       char *enames)
{
    int i;

    logprintf(xen, "%-22s : %10s %8s\n", "event channels", "total", "diff");
    for (i = 0; i < XEN_EVENT_MAX; i++) {
	if (!events[i])
	    continue;
	logprintf(xen, "  %-20s : %10" PRId64 " %8" PRId64 "\n",
		  enames + i * XEN_ENAME_LEN,
		  events[i], last ? events[i] - last[i] : 0);
    }
}
