#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <errno.h>
#include <ctype.h>
#include <dlfcn.h>
#include <fcntl.h>
#include <stdarg.h>
#include <assert.h>
#include <dirent.h>
#include <sys/mman.h>
#include <sys/socket.h>
#include <sys/types.h>
#include <netinet/in.h>

#include <xenctrl.h>

#include "xennerctrl.h"
#include "shared.h"

/* ------------------------------------------------------------------ */

int libxc_trace = 0;
int libxc_fixme = 1;

static void __attribute__ ((constructor)) init(void)
{
    char *env;

    if (NULL != (env = getenv("LIBXC_TRACE")))
	libxc_trace = atoi(env);
    if (NULL != (env = getenv("LIBXC_FIXME")))
	libxc_fixme = atoi(env);
}

/* ------------------------------------------------------------------ */

int xc_interface_open(void)
{
    int rc = 42;

    if (libxc_trace)
	fprintf(stderr, "libxc: %s -> handle %d\n", __FUNCTION__, rc);
    return rc;
}

int xc_interface_close(int xc_handle)
{
    if (libxc_trace)
	fprintf(stderr, "libxc: %s\n", __FUNCTION__);
    return 0;
}

/* ------------------------------------------------------------------ */

static int fill_dominfo(xc_dominfo_t *info, struct xennerdom *dom)
{
    /* good enougth to make the "domain exists?" style users happy */
    memset(info, 0, sizeof(*info));
    info->domid = dom->domid;
    if (dom->vminfo) {
	info->dying    = dom->vminfo->dying;
	info->running  = dom->vminfo->vcpus_running > 0;
	info->blocked  = dom->vminfo->vcpus_running == 0;
#if 0
	info->nr_pages = dom->vminfo->pg_guest;
#endif
    } else if (dom->vmcore) {
	if (0 == strcmp(dom->vmcore, "dying"))
	    info->dying = 1;
    }
    return 0;
}

static int comp_uint(const void *a, const void *b)
{
    const unsigned int *aa = a;
    const unsigned int *bb = b;

    if (*aa < *bb)
	return -1;
    if (*aa > *bb)
	return 1;
    return 0;
}

int xc_domain_getinfo(int xc_handle,
                      uint32_t first_domid,
                      unsigned int max_doms,
                      xc_dominfo_t *info)
{
    struct xennerdom *xdom;
    unsigned int *list, llen;
    int i, dom, count;
    struct dirent *ent;
    DIR *dir;

    if (1 == max_doms) {
	/* single domain ... */
	xdom = xenner_get_dom(first_domid);
	if (NULL != xdom) {
	    /* ok, exists, just return info */
	    fill_dominfo(info, xdom);
	    xenner_put_dom(xdom);
	    return 1;
	}
    }

    /* get sorted list of domains */
    dir = opendir("/var/run/xenner");
    if (NULL == dir)
	return 0;
    list = NULL, llen = 0;
    while (NULL != (ent = readdir(dir))) {
	if (1 != sscanf(ent->d_name, "vmcore.%d", &dom))
	    continue;
	if (0 == (llen % 16))
	    list = realloc(list, sizeof(*list) * (llen + 16));
	list[llen++] = dom;
    }
    closedir(dir);
    qsort(list, llen, sizeof(*list), comp_uint);

    /* fill informations */
    for (i = 0, count = 0; i < llen && count < max_doms; i++) {
	if (list[i] < first_domid)
	    continue;

	xdom = xenner_get_dom(list[i]);
	if (NULL == xdom)
	    continue;
	fill_dominfo(info, xdom);
	xenner_put_dom(xdom);
	count++;
	info++;
    }
    free(list);
    
    return count;
}

/* ------------------------------------------------------------------ */

void *xc_map_foreign_range(int xc_handle, uint32_t dom,
			   int size, int prot,
			   unsigned long mfn)
{
    char filename[256];
    void *ptr;
    int fd;

    snprintf(filename, sizeof(filename), "/var/run/xenner/vmcore.%d", dom);
    fd = open(filename, O_RDWR);
    if (-1 == fd)
	return NULL;
    ptr = mmap(NULL, size, prot, MAP_SHARED, fd,
	       VMCORE_HEADER + mfn * PAGE_SIZE);
    if ((void*)-1 == ptr) {
	fprintf(stderr, "libxc: mmap(%s): %s\n", filename, strerror(errno));
	close(fd);
	return NULL;
    }
    close(fd);

    if (libxc_trace)
	fprintf(stderr, "libxc: %s(handle %d, dom %d, size %d, prot %d, mfn 0x%lx)"
		" -> %p\n",
		__FUNCTION__, xc_handle, dom, size, prot, mfn, ptr);
    return ptr;
}

void *xc_map_foreign_pages(int xc_handle, uint32_t dom, int prot,
			   const xen_pfn_t *arr, int num)
{
    char filename[256];
    void *ptr;
    int fd, i;

    /* get address space by creating a anonymous mapping ... */
    ptr = mmap(NULL, num * PAGE_SIZE, prot, MAP_ANONYMOUS | MAP_SHARED, -1, 0);
    if ((void*)-1 == ptr) {
	ptr = NULL;
	goto done;
    }

    /* ... then replace it page by page with the content we want */
    snprintf(filename, sizeof(filename), "/var/run/xenner/vmcore.%d", dom);
    fd = open(filename, O_RDWR);
    if (-1 == fd) {
	munmap(ptr, num * PAGE_SIZE);
	ptr = NULL;
	goto done;
    }
    for (i = 0; i < num; i++) {
	if ((void*)-1 == mmap(ptr + i*PAGE_SIZE, PAGE_SIZE, prot,
			      MAP_SHARED | MAP_FIXED, fd,
			      VMCORE_HEADER + arr[i] * PAGE_SIZE)) {
	    fprintf(stderr, "libxc: mmap(#%d, mfn 0x%lx): %s\n",
		    i, arr[i], strerror(errno));
	    munmap(ptr, num * PAGE_SIZE);
	    close(fd);
	    ptr = NULL;
	    goto done;
	}
    }
    close(fd);
    
done:
    if (libxc_trace)
	fprintf(stderr, "libxc: %s(handle %d, dom %d, prot %d, arr %p, num %d)"
		" -> %p\n",
		__FUNCTION__, xc_handle, dom, prot, arr, num, ptr);
    return ptr;
}

void *xc_map_foreign_batch(int xc_handle, uint32_t dom, int prot,
                           xen_pfn_t *arr, int num)
{
    return xc_map_foreign_pages(xc_handle, dom, prot, arr, num);
}
