#include <errno.h>
#include <inttypes.h>

#include <xen/xen.h>
#include <xen/version.h>
#include <xen/memory.h>
#include <xen/grant_table.h>
#include <xen/event_channel.h>
#include <xen/sched.h>
#include <xen/hvm/hvm_op.h>
#include <xen/hvm/params.h>

#include "hvm.h"
#include "xenner-emudev.h"

static void write_string(char *msg)
{
    int i;

    for (i = 0; msg[i]; i++)
	emudev_cmd(EMUDEV_CMD_WRITE_CHAR, msg[i]);
}

static void write_long(unsigned long value)
{
    static const char hexmap[] = "0123456789abcdef";
    static const int max = sizeof(unsigned long)*8;
    int shift;

    for (shift = 4; value >> shift && shift <= max; shift += 4)
	;

    emudev_cmd(EMUDEV_CMD_WRITE_CHAR, '0');
    emudev_cmd(EMUDEV_CMD_WRITE_CHAR, 'x');
    while (shift) {
	shift -= 4;
	emudev_cmd(EMUDEV_CMD_WRITE_CHAR, hexmap[(value >> shift) & 0x0f]);
    }
}

static void write_value(char *desc, unsigned long value)
{
    write_string(desc);
    write_string(": ");
    write_long(value);
    write_string("\n");
}

static void memcpy(char *dst, const char *src, char n)
{
    int i;
    for (i = 0; i < n; i++)
	dst[i] = src[i];
}

static sreg_t xen_version(ureg_t *args)
{
    static const char extra[XEN_EXTRAVERSION_LEN] = "-xenner-" VERSION;

    switch (args[0]) {
    case XENVER_version:
	return (3 << 16) | 1;
    case XENVER_extraversion:
    {
	char *dst = (void*)args[1];
	memcpy(dst, extra, sizeof(extra));
	return 0;
    }
    case XENVER_get_features:
    {
	xen_feature_info_t *fi = (void*)args[1];
	fi->submap = 0;
 	break;
    }
    default:
	return -ENOSYS;
    }
    return 0;
}

static sreg_t memory_op(ureg_t *args)
{
    struct xen_add_to_physmap *atp;
    int cmd = args[0] & 0x0f /* MEMOP_CMD_MASK */;

    switch (cmd) {
    case XENMEM_add_to_physmap:
	atp = (void*)args[1];
	if (atp->domid != DOMID_SELF)
	    return -EPERM;
	if (atp->space == XENMAPSPACE_shared_info)
	    emudev_set(EMUDEV_CONF_SHARED_INFO_PFN, 0, atp->gpfn);
	else if (atp->space == XENMAPSPACE_grant_table)
	    emudev_set(EMUDEV_CONF_GRANT_TABLE_PFNS, atp->idx, atp->gpfn);
	else {
	    write_value("unknown mapspace", atp->space);
	    return -EINVAL;
	}
	return 0;
    default:
	write_value("unknown memory_op", cmd);
	return -ENOSYS;
    }
    return 0;
}

static sreg_t grant_table_op(ureg_t *args)
{
    struct gnttab_query_size  *qs;
#if 0
    struct gnttab_setup_table *st;
    unsigned long *frames;
    int i;
#endif

    switch (args[0]) {
    case GNTTABOP_query_size:
	qs = (void*)args[1];
	qs->nr_frames = 0; /* FIXME */
	qs->max_nr_frames = 4;
	qs->status = GNTST_okay;
	break;
#if 0
    case GNTTABOP_setup_table:
	st = (void*)args[1];
	printk(1, "%s: setup_table %d\n", __FUNCTION__, st->nr_frames);
	if (st->nr_frames > GRANT_FRAMES_MAX) {
	    st->status = GNTST_general_error;
	} else {
	    grant_frames = st->nr_frames;
	    frames = st->frame_list.p;
	    for (i = 0; i < grant_frames; i++)
		frames[i] = EMU_MFN(grant_table) + i;
	    st->status = GNTST_okay;
	}
	break;
#endif
    default:
	write_value("unknown grant_table_op", args[0]);
	return -ENOSYS;
    }
    return 0;
}

static sreg_t hvm_op(ureg_t *args)
{
    struct xen_hvm_param *param;

    switch (args[0]) {
    case HVMOP_set_param:
	param = (void*)args[1];
	if (param->domid != DOMID_SELF)
	    return -EPERM;
	switch (param->index) {
	case HVM_PARAM_CALLBACK_IRQ:
	    emudev_set(EMUDEV_CONF_HVM_CALLBACK_IRQ, 0, param->value);
	    break;
	default:
	    write_value("unknown HVMOP_set_param", param->index);
	    return -ENOSYS;
	}
	break;
    case HVMOP_get_param:
	param = (void*)args[1];
	if (param->domid != DOMID_SELF)
	    return -EPERM;
	switch (param->index) {
	case HVM_PARAM_STORE_PFN:
	    param->value = emudev_get(EMUDEV_CONF_HVM_XENSTORE_PFN, 0);
	    break;
	case HVM_PARAM_STORE_EVTCHN:
	    param->value = emudev_get(EMUDEV_CONF_HVM_XENSTORE_EVTCHN, 0);
	    break;
	default:
	    write_value("unknown HVMOP_get_param", param->index);
	    return -ENOSYS;
	}
	break;
    default:
	write_value("unknown hvm_op", args[0]);
	return -ENOSYS;
    }
    return 0;
}

static sreg_t event_channel_op(ureg_t *args)
{
    struct evtchn_send *send;
    struct evtchn_alloc_unbound *au;
    int vcpu_id = 0; /* FIXME: SMP */

    switch (args[0]) {
    case EVTCHNOP_send:
	send = (void*)args[1];
	emudev_cmd(EMUDEV_CMD_EVTCHN_SEND, send->port);
	break;
    case EVTCHNOP_alloc_unbound:
	au = (void*)args[1];
	emudev_cmd(EMUDEV_CMD_EVTCHN_ALLOC, vcpu_id);
	au->port = emudev_get(EMUDEV_CONF_COMMAND_RESULT, vcpu_id);
	write_value("alloc port", au->port);
	break;
    default:
	write_value("unknown event_channel_op", args[0]);
	return -ENOSYS;
    }
    return 0;
}

static sreg_t sched_op(ureg_t *args)
{
    struct sched_shutdown *sh;

    switch (args[0]) {
    case SCHEDOP_shutdown:
	sh = (void*)args[1];
	emudev_cmd(EMUDEV_CMD_GUEST_SHUTDOWN, sh->reason);
	break;
    default:
	write_value("unknown sched_op", args[0]);
	return -ENOSYS;
    }
    return 0;
}

static sreg_t sched_op_compat(ureg_t *args)
{

    switch (args[0]) {
    case SCHEDOP_shutdown:
	emudev_cmd(EMUDEV_CMD_GUEST_SHUTDOWN, args[1]);
	break;
    default:
	write_value("unknown sched_op_compat", args[0]);
	return -ENOSYS;
    }
    return 0;
}

asmlinkage void do_hypercall(struct regs *regs)
{
    char *hypercall;
    ureg_t hcall;
    ureg_t args[6];
    ureg_t retval;

    hcall   = regs->rax;
#if longmode
    args[0] = regs->rdi;
    args[1] = regs->rsi;
    args[2] = regs->rdx;
    args[3] = regs->r10;
    args[4] = regs->r8;
    args[5] = regs->r9;
#else
    args[0] = regs->rbx;
    args[1] = regs->rcx;
    args[2] = regs->rdx;
    args[3] = regs->rsi;
    args[4] = regs->rdi;
    args[5] = regs->rbp;
#endif

    switch(hcall) {
    case __HYPERVISOR_xen_version:
	hypercall = "xen_version";
	retval = xen_version(args);
	break;
    case __HYPERVISOR_memory_op:
	hypercall = "memory_op";
	retval = memory_op(args);
	break;
    case __HYPERVISOR_grant_table_op:
	hypercall = "grant_table_op";
	retval = grant_table_op(args);
	break;
    case __HYPERVISOR_hvm_op:
	hypercall = "hvm_op";
	retval = hvm_op(args);
	break;
    case __HYPERVISOR_event_channel_op:
	hypercall = "event_channel_op";
	retval = event_channel_op(args);
	break;
    case __HYPERVISOR_sched_op:
	hypercall = "sched_op";
	retval = sched_op(args);
	break;
    case __HYPERVISOR_sched_op_compat:
	hypercall = "sched_op_compat";
	retval = sched_op_compat(args);
	break;
    default:
        write_value("hcall", hcall);
	hypercall = "unknown hypercall";
	retval = -ENOSYS;
	break;
    }

    if (retval == -ENOSYS) {
	write_string(hypercall);
	write_string(" -> -ENOSYS\n");
    }
    if (retval == -EINVAL) {
	write_string(hypercall);
	write_string(" -> -EINVAL\n");
    }
    regs->rax = retval;
}
