/* 
 * Copyright (C) 2005  Network Applied Communication Laboratory Co., Ltd.
 *
 * This file is part of Rast.
 * See the file COPYING for redistribution information.
 *
 */

#include <stdio.h>
#include <stdlib.h>

#include <apr_pools.h>
#include <apr_tables.h>
#include <apr_strings.h>
#include <apr_file_io.h>

#include "rast/db.h"
#include "rast/local_db.h"
#include "rast/property.h"

static int memory_error(int retcode);
static void print_error(rast_error_t *error);

static rast_error_t *
read_number(apr_file_t *file, rast_uint_t *number, int is_native)
{
    rast_uint_t n;
    apr_size_t nbytes = sizeof(rast_uint_t);
    apr_status_t status;

    if ((status = apr_file_read(file, &n, &nbytes)) != APR_SUCCESS) {
        return apr_status_to_rast_error(status);
    }
    *number = rast_fix_byte_order(n, is_native);
    return RAST_OK;
}

static rast_error_t *
open_locked_file(const char *lock_filename, int lock_flag, int open_flag,
                 apr_file_t **lock_file, apr_pool_t *pool)
{
    apr_status_t status = apr_file_open(lock_file, lock_filename,
                                        open_flag, APR_OS_DEFAULT, pool);
    if (status != APR_SUCCESS) {
        return apr_status_to_rast_error(status);
    }

    status = apr_file_lock(*lock_file, lock_flag);
    if (status != APR_SUCCESS) {
        apr_file_close(*lock_file);
        return apr_status_to_rast_error(status);
    }

    return RAST_OK;
}

static rast_error_t *
get_num_register(apr_pool_t *pool, rast_local_db_t *db,
                 rast_size_t *num_register)
{
    apr_file_t *file;
    const char *filename;
    rast_error_t *error;
    rast_uint_t max_doc_id;

    filename = apr_pstrcat(pool, db->path, "/doc_info", NULL);
    error = open_locked_file(filename, APR_FLOCK_SHARED, APR_READ,
                             &file, pool);
    if (error != RAST_OK) {
        return error;
    }
    error = read_number(file, &max_doc_id, db->is_native);
    if (error == RAST_OK) {
        error = read_number(file, num_register, db->is_native);
    }
    apr_file_unlock(file);
    apr_file_close(file);
    return error;
}

static const char *
propertiy_type_value_to_type_name(rast_type_e type)
{
    switch (type) {
    case RAST_TYPE_STRING:
        return "RAST_TYPE_STRING";
    case RAST_TYPE_DATE:
        return "RAST_TYPE_DATE";
    case RAST_TYPE_DATETIME:
        return "RAST_TYPE_DATETIME";
    case RAST_TYPE_UINT:
        return "RAST_TYPE_UINT";
    default:
        return NULL;
    }
} 

static char **
property_flag_value_to_flag_names(apr_pool_t *pool,
                                   rast_uint_t flags, int *size)
{
    apr_array_header_t *ary;
    char **elem;
    struct property_types {
        rast_uint_t flags;
        const char *name;
    };
    struct property_types property_types[] = {
        {RAST_PROPERTY_FLAG_SEARCH, "RAST_PROPERTY_FLAG_SEARCH"},
        {RAST_PROPERTY_FLAG_TEXT_SEARCH, "RAST_PROPERTY_FLAG_TEXT_SEARCH"},
        {RAST_PROPERTY_FLAG_FULL_TEXT_SEARCH,
         "RAST_PROPERTY_FLAG_FULL_TEXT_SEARCH"},
        {RAST_PROPERTY_FLAG_UNIQUE, "RAST_PROPERTY_FLAG_UNIQUE"},
        {RAST_PROPERTY_FLAG_OMIT, "RAST_PROPERTY_FLAG_OMIT"},
    };
    int property_type_size =
        sizeof(property_types) / sizeof(struct property_types);
    int i;

    ary = apr_array_make(pool, 1, sizeof(char **));
    for (i = 0; i < property_type_size; i++) {
        if (flags & property_types[i].flags) {
            elem = (char **) apr_array_push(ary);
            *elem = apr_pstrdup(pool, property_types[i].name);
        }
    }
    *size = ary->nelts;
    return (char **) ary->elts;
}

int
main(int argc, char **argv)
{
    apr_pool_t *pool;
    char *dbpath;
    rast_db_t *base;
    rast_local_db_t *db;
    int num_properties, i;
    rast_size_t num_register;
    rast_error_t *error;

    if (argc < 2) {
        fprintf(stderr, "usage: %s <db>\n", argv[0]);
        return 1;
    }
    dbpath = argv[1];

    apr_initialize();
    atexit(apr_terminate);

    error = rast_initialize();
    if (error != RAST_OK) {
        print_error(error);
        return 1;
    }
    atexit(rast_finalize);

    apr_pool_create_ex(&pool, NULL, memory_error, NULL);
    error = rast_db_open(&base, dbpath, RAST_DB_RDWR, NULL, pool);
    if (error != RAST_OK) {
        print_error(error);
        return 1;
    }
    db = (rast_local_db_t *) base;
    error = get_num_register(pool, db, &num_register);
    if (error != RAST_OK) {
        print_error(error);
        return 1;
    }

    rast_db_properties(base, &num_properties);

    printf("---\n");
    printf("db_name: %s\n", dbpath);
    printf("num_register: %d\n", num_register);
    printf("encoding_module: %s\n", db->encoding);
    printf("encoding: %s\n", rast_db_encoding(&db->base));
    printf("preserve_text: %s\n", db->preserve_text ? "true": "false");
    printf("is native: %s\n", db->is_native ? "true": "false");
    printf("pos block size: %d\n", db->pos_block_size);
    printf("properties:");
    if (num_properties == 0)
    {
        printf(" []\n");
    }
    else
    {
        printf("\n");
    }
        
    for (i = 0; i < num_properties; i++) {
        int flag_size, j;
        char **flag_names;
        rast_property_t *property;

        property = db->properties + i;
        printf("  - name: %s\n", property->name);
        printf("    type: %s\n",
               propertiy_type_value_to_type_name(property->type));
        printf("    flags:");
        flag_names = property_flag_value_to_flag_names(pool,
                                                       property->flags,
                                                       &flag_size);
        if (flag_size == 0)
        {
            printf(" []\n");
        }
        else
        {
            printf("\n");
        }
        for (j = 0; j < flag_size; j++) {
            printf("      - %s\n", flag_names[j]);
        }
    }

    rast_db_close(base);

    return 0;
}

static int
memory_error(int retcode)
{
    abort();
    return -1;  /* prevent compiler warnings */
}

static void
print_error(rast_error_t *error)
{
#ifdef RAST_DEBUG
    fprintf(stderr, "%s:%d: %s\n", error->file, error->line, error->message);
#else
    fprintf(stderr, "error: %s\n", error->message);
#endif
    rast_error_destroy(error);
}
