/* 
 * Copyright (C) 2005  Network Applied Communication Laboratory Co., Ltd.
 *
 * This file is part of Rast.
 * See the file COPYING for redistribution information.
 *
 */

#include <stdio.h>
#include <stdlib.h>

#include <apr_pools.h>
#include <apr_file_info.h>
#include <apr_file_io.h>
#include <apr_strings.h>
#include <apr_time.h>

#include "rast/db.h"

static int memory_error(int retcode);
static void print_error(rast_error_t *error);
static void print_apr_error(apr_status_t status);

int
main(int argc, char **argv)
{
    apr_pool_t *pool;
    char *dbpath, *filename;
    rast_db_t *db;
    rast_error_t *error;
    apr_finfo_t finfo;
    apr_file_t *file;
    apr_status_t status;
    char *s;
    apr_size_t len;
    apr_time_exp_t last_modified;
    rast_value_t properties[4];
    rast_doc_id_t doc_id;

    if (argc < 3) {
        fprintf(stderr, "usage: %s <db> <filename>\n", argv[0]);
        return 1;
    }
    dbpath = argv[1];
    filename = argv[2];

    apr_initialize();
    atexit(apr_terminate);
    error = rast_initialize();
    if (error != RAST_OK) {
        print_error(error);
        return 1;
    }
    atexit(rast_finalize);

    apr_pool_create_ex(&pool, NULL, memory_error, NULL);

    status = apr_stat(&finfo, filename,
                      APR_FINFO_SIZE | APR_FINFO_MTIME, pool);
    if (status != APR_SUCCESS) {
        print_apr_error(status);
        return 1;
    }
    status = apr_file_open(&file, filename, APR_READ, APR_OS_DEFAULT, pool);
    if (status != APR_SUCCESS) {
        print_apr_error(status);
        return 1;
    }
    len = finfo.size;
    s = (char *) apr_palloc(pool, len);
    status = apr_file_read(file, s, &len);
    apr_file_close(file);
    if (status != APR_SUCCESS) {
        print_apr_error(status);
        return 1;
    }

    rast_value_set_string(&properties[0], filename);
    rast_value_set_string(&properties[1], filename);
    apr_time_exp_lt(&last_modified, finfo.mtime);
    rast_value_set_date(&properties[2],
                        apr_psprintf(pool, "%04d-%02d-%02dT%02d:%02d:%02d",
                                     last_modified.tm_year + 1900,
                                     last_modified.tm_mon + 1,
                                     last_modified.tm_mday,
                                     last_modified.tm_hour,
                                     last_modified.tm_min,
                                     last_modified.tm_sec));
    rast_value_set_uint(&properties[3], len);

    error = rast_db_open(&db, dbpath, RAST_DB_RDWR, NULL, pool);
    if (error != RAST_OK) {
        print_error(error);
        return 1;
    }

    error = rast_db_register(db, s, len, properties, &doc_id);
    if (error != RAST_OK) {
        rast_db_close(db);
        print_error(error);
        return 1;
    }
    printf("doc_id: %d\n", doc_id);

    rast_db_close(db);

    return 0;
}

static int
memory_error(int retcode)
{
    abort();
    return -1;  /* prevent compiler warnings */
}

static void
print_error(rast_error_t *error)
{
#ifdef RAST_DEBUG
    fprintf(stderr, "%s:%d: %s\n", error->file, error->line, error->message);
#else
    fprintf(stderr, "error: %s\n", error->message);
#endif
    rast_error_destroy(error);
}

static void
print_apr_error(apr_status_t status)
{
    char buf[1024];

    apr_strerror(status, buf, sizeof(buf));
    fprintf(stderr, "error: %s\n", apr_strerror(status, buf, sizeof(buf)));
}

/* vim: set filetype=c sw=4 expandtab : */
