
/* dict.q: ordered dictionaries implemented by AVL trees
   $Id: dict.q,v 1.11 2008/02/21 19:58:49 agraef Exp $ */

/* This file is part of the Q programming system.

   The Q programming system is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   The Q programming system is distributed in the hope that it will be
   useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA. */

/* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
   Updated: 18 February 2008 by Jiri Spitz

   Purpose: More efficient algorithm for association lists implemented
   as AVL trees.

   The used algorithm has its origin in the SWI-Prolog implementation of
   association lists. The original file was created by R.A.O'Keefe and
   updated for the SWI-Prolog by Jan Wielemaker. For the original file
   see http://www.swi-prolog.org.

   The deletion stuff (rmfirst, rmlast, delete) is new, it was missing
   in the original assoc.pl file.
- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - */

public type Dict = virtual dict XYs | private const nil, bin K V B D1 D2;

/* Construction and type checking: */

public emptydict;		// return the empty dictionary
//public dict XYs;		// create a dictionary from a list of key/value
				// pairs (virtual constructor, see above)
public mkdict Y Xs;		// create a dictionary from a list of keys
				// and an initial value
public isdict X;		// check whether X is a dictionary

/* Overloaded and public operations: */

from stddecl include null, member, list, members, keys, vals, first, last,
  rmfirst, rmlast, insert, delete, update;

// #D				// size of a dictionary
// D!X				// return the value Y associated with X in D

// null D			// tests whether D is the empty dictionary
// member D X			// tests whether D contains X as a key
// list D, members D		// list members (key-value pairs) of D in
				// ascending order
// keys D			// list the keys of D in ascending order
// vals D			// list the corresponding values

// first D, last D		// return first and last member of D
// rmfirst D, rmlast D		// remove first and last member from D
// insert D XY			// associate key X with value Y in D; update
				// an existing entry for X
// delete D X			// remove key X from D
// update D X Y			// same as insert D (X,Y)

/* Implementation: *********************************************************/

/* Default view: */

@-0x80000000
view X:Dict			= '(dict Xs) where Xs:List = list X;
@0

/* Private Types: **********************************************************/

// For better readability of the code
private type Balance		= const islt, iseq, isgt;
private type Side		= const left, right;

/* Private Functions: ******************************************************/

private inserta Tree Key Val;
	// insert a new (or replace an existing) member in the tree

private rmfirsta Tree;
	// remove the first member from the tree

private rmlasta Tree;
	// remove the last member from the tree

private deletea Tree Key;
	// delete member with Key from the tree

private adjusti TreeHasChanged Tree LeftOrRight;
	// decide changes needed in order to make a well
	// shaped tree after an insertion

private rebali ToBeRebalanced Tree NewBalance;
	// if ToBeRabalanced = false then set the balance of the root node
	// to NewBalance else call avl_geq

private adjustd TreeHasChanged Tree LeftOrRight;
	// decide changes needed in order to make a well
	// shaped tree after a deletion

private rebald ToBeRebalanced Tree NewBalance WhatHasChanged;
	// if ToBeRabalanced = false then set the balance of the root node
	// to NewBalance else call avl_geq

private avl_geq Tree;
	// single and double rotations of the tree

private tablei BalanceBefore WhereInserted;
	// insert balance rules

private tabled BalanceBefore WhereDeleted;
	// delete balance rules

private table2 BalanceOfSubSubNode;
	// balance rules for double rotations

/*
Tree is either:

-  nil  (empty tree) or
-  bin Key Value Balance Left Right  (Left, Right: trees)
   Balance: islt, iseq, or isgt denoting |L|-|R| = 1, 0, or -1, respectively
*/

inserta nil Key Val		= ((bin Key Val iseq nil nil), true);
inserta (bin K _ B L R) Key Val = ((bin Key Val B L R), false) if Key = K;

inserta (bin K V B L R) Key Val if Key < K:
		= adjusti LeftHasChanged (bin K V B NewL R) left
		    where (NewL, LeftHasChanged) = inserta L Key Val;

inserta (bin K V B L R) Key Val if Key > K:
		= adjusti RightHasChanged (bin K V B L NewR) right
		    where (NewR, RightHasChanged) = inserta R Key Val;

rmfirst D			= fst (rmfirsta D);

rmfirsta nil			= (nil, false);
rmfirsta (bin _ _ _ nil R)	= (R, true);

rmfirsta (bin K V B L   R)
		= adjustd LeftHasChanged (bin K V B NewL R) left
		    where (NewL, LeftHasChanged) = rmfirsta L;

rmlast D			= fst (rmlasta D);

rmlasta nil			= (nil false);
rmlasta (bin _ _ _ L nil)	= (L, true);

rmlasta (bin K V B L   R)
		= adjustd RightHasChanged (bin K V B L NewR) right
		    where (NewR, RightHasChanged) = rmlasta R;

deletea nil _			= (nil, false);
deletea (bin K _ _ nil R  ) Key	= (R, true) if Key = K;
deletea (bin K _ _ L   nil) Key	= (L, true) if Key = K;

deletea (bin K _ B (bin KL VL BL RL LL) R  ) Key if Key = K:
		= adjustd LeftHasChanged (bin LastK LastV B NewL R) left
		    where
		      (LastK, LastV)		= last (bin KL VL BL RL LL),
		      (NewL, LeftHasChanged)	= rmlasta (bin KL VL BL RL LL);

deletea (bin K V B L R) Key if Key < K:
		= adjustd LeftHasChanged (bin K V B NewL R) left
		    where
		      (NewL, LeftHasChanged) = deletea L Key;

deletea (bin K V B L R) Key if Key > K:
		= adjustd RightHasChanged (bin K V B L NewR) right
		    where
		      (NewR, RightHasChanged) = deletea R Key;

// The insertions and deletions are dealt with separately.
// Insertions
adjusti false OldTree _		= (OldTree, false);

adjusti true (bin Key Val B0 L R) LoR
		= (rebali ToBeRebalanced (bin Key Val B0 L R) B1,
		   WhatHasChanged)
		    where
		      (B1, WhatHasChanged, ToBeRebalanced) = tablei B0 LoR;

rebali false (bin K V _ L R) B	= bin K V B L R;
rebali true  OldTree _		= fst (avl_geq OldTree);

// Balance rules for insertions
//	balance	where		balance	  whole tree	to be
//	before	inserted	after	  increased	rebalanced
tablei	iseq	left		= (islt,  true,		false);
tablei	iseq	right		= (isgt,  true,		false);
tablei	islt	left		= (iseq,  false,	true);
tablei	islt	right		= (iseq,  false,	false);
tablei	isgt	left		= (iseq,  false,	false);
tablei	isgt	right		= (iseq,  false,	true);

// Deletions
adjustd false OldTree _		= (OldTree, false);

adjustd true (bin Key Val B0 L R) LoR
		= rebald ToBeRebalanced (bin Key Val B0 L R) B1 WhatHasChanged
		    where
		      (B1, WhatHasChanged, ToBeRebalanced) = tabled B0 LoR;

// Balance rules for deletions
//	balance	where		balance	  whole tree	to be
//	before	deleted		after	  decreased	rebalanced
tabled	iseq	right		= (islt,  false,	false);
tabled	iseq	left		= (isgt,  false,	false);
tabled	islt	right		= (iseq,  true,		true);
//					  ^^^^
// It depends on the tree pattern in avl_geq whether it really decreases

tabled	islt	left		= (iseq,  true, 	false);
tabled	isgt	right		= (iseq,  true,		false);
tabled	isgt	left		= (iseq,  true,		true);
//					  ^^^^
// It depends on the tree pattern in avl_geq whether it really decreases

/*
   Note that rebali and rebald are not symmetrical. With insertions it is
   sufficient to know the original balance and insertion side in order to
   decide whether the whole tree increases. With deletions it is sometimes not
   sufficient and we need to know which kind of tree rotation took place.
*/
rebald false (bin K V _ L R) B WhatHasChanged
				= (bin K V B L R, WhatHasChanged);
rebald true  OldTree _ _	= avl_geq OldTree;

// Single and double tree rotations - these are common for insert and delete
/*
  The patterns isgt-isgt, isgt-islt, islt-islt and islt-isgt on the LHS always
  change the tree height and these are the only patterns which can happen
  after an insertion. That's the reason why we can use tablei only to decide
  the needed changes.
  The patterns isgt-iseq and islt-iseq do not change the tree height. After a
  deletion any pattern can occur and so we return true or false as a flag of
  a height change.
*/
avl_geq (bin A VA isgt Alpha (bin B VB isgt Beta Gamma))
		= (bin B VB iseq (bin A VA iseq Alpha Beta) Gamma, true);

avl_geq (bin A VA isgt Alpha (bin B VB iseq Beta Gamma))
		= (bin B VB islt (bin A VA isgt Alpha Beta) Gamma, false);
			// the tree doesn't decrease with this pattern

avl_geq (bin A VA isgt Alpha (bin B VB islt (bin X VX B1 Beta Gamma) Delta))
		= (bin X VX iseq (bin A VA B2 Alpha Beta)
		   (bin B VB B3 Gamma Delta), true)
		    where (B2, B3) = table2 B1;

avl_geq (bin B VB islt (bin A VA islt Alpha Beta) Gamma)
		= (bin A VA iseq Alpha (bin B VB iseq Beta  Gamma), true);

avl_geq (bin B VB islt (bin A VA iseq Alpha Beta) Gamma)
		= (bin A VA isgt Alpha (bin B VB islt Beta  Gamma), false);
			// the tree doesn't decrease with this pattern

avl_geq (bin B VB islt (bin A VA isgt Alpha (bin X VX B1 Beta Gamma)) Delta)
		= (bin X VX iseq (bin A VA B2 Alpha Beta)
		   (bin B VB B3 Gamma Delta), true)
		    where (B2, B3) = table2 B1;

table2 islt			= (iseq, isgt);
table2 isgt			= (islt, iseq);
table2 iseq			= (iseq, iseq);

/* Public Functions: *******************************************************/

emptydict			= nil;

dict XYs:List			= foldl insert emptydict XYs;

mkdict Y Xs:List		= dict (zip Xs (mklist Y (#Xs)));

#nil				= 0;
#bin _ _ _ D1 D2		= #D1 + #D2 + 1;

null nil			= true;
				= false otherwise;

isdict D:Dict			= true;
				= false otherwise;

(D1:Dict = D2:Dict)		= (members D1 = members D2);
D1:Dict <> D2:Dict		= members D1 <> members D2;

(bin X Y _ D1 D2)!X1		= D1!X1 if X1 < X;
				= D2!X1 if X1 > X;
				= Y;
member nil _			= false;
member (bin X _ _ D1 D2) X1	= member D1 X1 if X > X1;
				= member D2 X1 if X < X1;
				= true if X = X1;

members	nil			= [];
members (bin X Y _ D1 D2)	= members D1 ++ [(X,Y)|members D2];

keys nil			= [];
keys (bin X _ _ D1 D2)		= keys D1 ++ [X|keys D2];

vals nil			= [];
vals (bin _ Y _ D1 D2)		= vals D1 ++ [Y|vals D2];

last (bin X Y _ _ nil)		= (X,Y);
last (bin _ _ _ _ D2)		= last D2;

first (bin X Y _ nil _)		= (X,Y);
first (bin _ _ _ D1 _)		= first D1;

insert D:Dict (X,Y)		= fst (inserta D X Y);

delete D:Dict X			= fst (deletea D X);

update D:Dict X Y		= insert D (X,Y);
