
/* expect 2 shift/reduce, 40 reduce/reduce */

%{

/* qmparse.y: Q machine expression parser */

/* Special case constructs (unary minus) and dangling else cause a number of
   parsing conflicts which are resolved correctly. */

/*  Q eQuational Programming System
    Copyright (c) 1991-2002 by Albert Graef
    <ag@muwiinfa.geschichte.uni-mainz.de>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 1, or (at your option)
    any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*/

/* make sure we have bison */

#ifndef YYBISON
#error "Sorry, this program requires bison."
#endif

#include "qdefs.h"

extern char *prompt, *prompt2, *prompt3;
extern int pmode;

static char		signon[] = Q_SIGNON;
static char		copying[] = COPYING;

static char *psource = "";

static char  *save_file = NULL;
static char *base;

static THREAD *thr;

/*

This module implements a parser for Q expressions and the special
constructs (def, undef, etc.) permitted in the interactive evaluation
loop. The parser can be run in four different modes signalled by special
leading tokens:

- STRING: read a single expression from a null-terminated string.

- LINE: read an expression terminated with newline from a file.

- INTERACT: read and evaluate a command (including def's,
  undef's and shell escapes) from a null-terminated string.

- SOURCE: read and evaluate commands from a file

*/

static swap(), pushtern(), pushbin(), pushun(),
  pushlist(int n, int m), pushstream(int n, int m), pushtuple(int n, int m),
  pushgroup(int *n, int *m),
  savepos(), clean();
static init_stats(), fini_stats(), print_stats();
static copyright(), shell(), expression(),
  vardecl(), initializer(), assignment(), unassignment(),
  sethistfile(), sethistsize(), help(), edit(), save_vars(), clear_vars(),
  add_import(), del_import(), clear_imports(), breakpoints(), profile(),
  who(), whos(), whois(), ambiguous(), import_list(), modules();
static strparam(), intparam(), boolparam(), boolstrparam(), allparam(),
  strlist(), printstr();

/* The following is set to compile for the libqint target. */
#ifdef QINT_LIBRARY
#undef USE_READLINE
#endif

#ifndef USE_READLINE
static histwarn();
#endif

#define MAXARGC 1000
%}

%union {
  int ival;
  mpz_t zval;
  double fval;
  char *sval;
  EXPR *xval;
  struct {
    int n, m;
  } lval;
  struct {
    char **v;
    int c;
  } aval;
}

/* parse mode indicators: */

%token STRING LINE INTERACT SOURCE

/* keywords and multi-character literals: */

%token AS CONST DEF ELSE EXTERN FROM IF IMPORT INCLUDE OTHERWISE
%token PRIVATE PUBLIC SPECIAL THEN TYPE UNDEF VAR VIRTUAL WHERE
%token DOTDOT EQUIV

/* user-defined operators */

%token <ival> OP0 OP1 OP2 OP3 OP4 OP5 OP6 OP7 OP8 OP9

/* additional command keywords */

%token _DEF_ _UNDEF_ _VAR_ _IMPORT_
%token _COPYING_ _HELP_ _EDIT_ _PATH_ _PROMPT_
%token _HISTFILE_ _HISTSIZE_ _CSTACKSIZE_ _STACKSIZE_ _MEMSIZE_ _STATS_
%token _RUN_ _SAVE_ _LOAD_ _CLEAR_ _PROFILE_
%token _BREAK_ _TBREAK_ _DEBUG_ _UNPARSE_ _ECHO_ _CHDIR_ _PWD_ _LS_ _WHICH_ _SOURCE_
%token _WHO_ _WHOS_ _WHOIS_ _IMPORTS_ _MODULES_ _COMPLETION_MATCHES_
%token _DEC_ _HEX_ _OCT_ _STD_ _SCI_ _FIX_ _FORMAT_

/* constants: */

%token XID

%token <ival> QID NID
%token <sval> STR
%token <zval> INT
%token <fval> FLOAT

%token <sval> ARG

/* unrecognized tokens */

%token ERRTOK ARGERR STRERR

%type <ival> ID
%type <ival> op seqop rappop relop0 relop addop mulop unop scriptop compop quoteop symbol whossymbol
%type <aval> arg_list opt_arg_list
%type <sval> arg
%type <lval> expr_list expr_list1 expr_list2
%type <ival> opt_expr

%start source

%%

source		: STRING expression
		| LINE expression
		| LINE expression '\n'
				{ YYACCEPT; }
		| INTERACT input
		| SOURCE input
		| /* empty */
				{ if (pmode != INTERACT && pmode != SOURCE)
				  YYERROR; }
		;

input		: /* empty */
		| '!'
				{ shell(); }
		| commands
		| commands ';'
		;

commands	: command
		| commands ';' command
		;

command		:

expression
{ if (!expression()) YYABORT; }

| _VAR_ vardecls

| _DEF_ defs

| _UNDEF_ undefs

| _COPYING_
{ if (!copyright()) YYABORT; }

| _COMPLETION_MATCHES_ arg
{
  char *s;
  if (strparam($2, &s)) {
    list_completions(s);
    free(s);
  } else
    YYABORT;
}

| _HELP_
{
  int res = 1;
  if (checkbrk) YYABORT;
  if (gflag)
    gcmd("q-help-cmd");
  else
    res = help(NULL);
  if (!res) YYABORT;
}

| _HELP_ arg
{
  char *s;
  if (strparam($2, &s)) {
    int res = 1;
    if (gflag) gcmd_s("q-help-cmd", s);
    else
      res = help(s);
    free(s);
    if (!res) YYABORT;
  } else
    YYABORT;
}

| _PATH_
{
  if (checkbrk) YYABORT;
  printstr(qpath);
}

| _PATH_ arg
{
  char *s;
  if (strparam($2, &s)) {
    change_qpath(s);
    free(s);
    if (lt_dlsetsearchpath(qpath)) {
      thr->qmstat = MEM_OVF;
      clean();
      YYABORT;
    }
    if (gflag) gcmd_s("q-path-cmd", qpath);
  } else
    YYABORT;
}

| _PROMPT_ arg_list
{
  char **argv = $2.v;
  int argc = $2.c, i;
  if (argc == 0) {
    free(argv);
    if (checkbrk) YYABORT;
    printstr(prompt);
  } else if (argc > 3) {
    for (i = 0; i < argc; i++) free(argv[i]);
    free(argv);
    savepos();
    thr->qmstat = SYNTAX_ERR;
  } else {
    if (prompt) free(prompt);
    prompt = argv[0];
    if (gflag) gcmd_s("q-prompt-cmd", argv[0]);
    new_xprompt();
    if (argc > 1) prompt2 = argv[1];
    if (argc > 2) prompt3 = argv[2];
    free(argv);
  }
  if (!clean()) YYABORT;
}

| _FORMAT_
{
  if (checkbrk) YYABORT;
  switch (imode) {
  case 0: printf("dec; "); break;
  case 1: printf("hex; "); break;
  case 2: printf("oct; "); break;
  }
  switch (fmode) {
  case 0: printf("std %d\n", fprec); break;
  case 1: printf("sci %d\n", fprec); break;
  case 2: printf("fix %d\n", fprec); break;
  } 
}

| _DEC_
{ if (checkbrk) YYABORT; imode = 0; if (gflag) gcmd("q-dec-cmd"); }

| _HEX_
{ if (checkbrk) YYABORT; imode = 1; if (gflag) gcmd("q-hex-cmd"); }

| _OCT_
{ if (checkbrk) YYABORT; imode = 2; if (gflag) gcmd("q-oct-cmd"); }

| _STD_
{ 
  if (checkbrk) YYABORT;
  fmode = 0; fprec = 15; sprintf(fformat, STDFORMAT, fprec);
  if (gflag) gcmd_i("q-std-cmd", fprec);
}

| _STD_ arg
{
  long p;
  if (intparam($2, &p)) {
    if (p >= 0) {
      fmode = 0; fprec = p; sprintf(fformat, STDFORMAT, fprec);
      if (gflag) gcmd_i("q-std-cmd", fprec);
    } else {
      thr->qmstat = BAD_FORMAT;
      clean();
      YYABORT;
    }
  } else
    YYABORT;
}

| _SCI_
{ 
  if (checkbrk) YYABORT;
  fmode = 1; fprec = 15; sprintf(fformat, SCIFORMAT, fprec);
  if (gflag) gcmd_i("q-sci-cmd", fprec);
}

| _SCI_ arg
{
  long p;
  if (intparam($2, &p)) {
    if (p >= 0) {
      fmode = 1; fprec = p; sprintf(fformat, SCIFORMAT, fprec);
      if (gflag) gcmd_i("q-sci-cmd", fprec);
    } else {
      thr->qmstat = BAD_FORMAT;
      clean();
      YYABORT;
    }
  } else
    YYABORT;
}

| _FIX_
{ 
  if (checkbrk) YYABORT;
  fmode = 2; fprec = 2; sprintf(fformat, FIXFORMAT, fprec);
  if (gflag) gcmd_i("q-fix-cmd", fprec);
}

| _FIX_ arg
{
  long p;
  if (intparam($2, &p)) {
    if (p >= 0) {
      fmode = 2; fprec = p; sprintf(fformat, FIXFORMAT, fprec);
      if (gflag) gcmd_i("q-fix-cmd", fprec);
    } else {
      thr->qmstat = BAD_FORMAT;
      clean();
      YYABORT;
    }
  } else
    YYABORT;
}

| _HISTFILE_
{
  if (checkbrk) YYABORT;
#ifndef USE_READLINE
  histwarn();
#endif
  printstr(histfile);
}

| _HISTFILE_ arg
{
  char *s;
  if (strparam($2, &s)) {
    sethistfile(s);
    if (gflag) gcmd_s("q-histfile-cmd", s);
  } else
    YYABORT;
}

| _HISTSIZE_
{
  if (checkbrk) YYABORT;
#ifndef USE_READLINE
  histwarn();
#endif
  printf("%d\n", histmax);
}

| _HISTSIZE_ arg
{
  long i;
  if (intparam($2, &i)) {
    sethistsize(i);
    if (gflag) gcmd_i("q-histsize-cmd", i);
  } else
    YYABORT;
}

| _CSTACKSIZE_
{
  if (checkbrk) YYABORT;
  printf("%d\n", cstackmax/1024);
}

| _CSTACKSIZE_ arg
{
  long i;
  if (intparam($2, &i)) {
    i *= 1024;
    if (i == 0 || i >= CSTACKMIN) {
      cstackmax = i;
    } else {
      printf("bad C stack size %d, using default\n", i/1024);
      cstackmax = CSTACKMAX;
    }
    if (gflag) gcmd_i("q-cstacksize-cmd", cstackmax);
  } else
    YYABORT;
}

| _STACKSIZE_
{
  if (checkbrk) YYABORT;
  printf("%d\n", stackmax);
}

| _STACKSIZE_ arg
{
  long i;
  if (intparam($2, &i)) {
    if (i == 0 || i >= STACKMIN) {
      stackmax = i;
    } else {
      printf("bad stack size %d, using default\n", i);
      stackmax = STACKMAX;
    }
    if (gflag) gcmd_i("q-stacksize-cmd", stackmax);
    /* this is needed to have the stack resized properly */
    clean();
  } else
    YYABORT;
}

| _MEMSIZE_
{
  if (checkbrk) YYABORT;
  printf("%d\n", memmax);
}

| _MEMSIZE_ arg
{
  long i;
  if (intparam($2, &i)) {
    if (i == 0 || i >= MEMMIN) {
      memmax = i;
    } else {
      printf("bad memory size %d, using default\n", i);
      memmax = MEMMAX;
    }
    lastblksz = memmax % XBLKSZ;
    maxnblks = memmax/XBLKSZ+((memmax <= 0||lastblksz==0)?0:1);
    if (lastblksz == 0) lastblksz = XBLKSZ;
    if (gflag) gcmd_i("q-memsize-cmd", memmax);
  } else
    YYABORT;
}

| _STATS_
{ if (checkbrk) YYABORT; print_stats(0); }

| _STATS_ arg
{ if (allparam($2)) print_stats(1); else YYABORT; }

| _PWD_
{
  char wd[MAXSTRLEN];
  if (checkbrk) YYABORT;
  if (getcwd(wd, MAXSTRLEN)) {
    printstr(wd);
  } else {
    thr->qmstat = MEM_OVF;
    clean();
    YYABORT;
  }
}

| _CHDIR_
{
  char buf[MAXSTRLEN];
  if (checkbrk) YYABORT;
  if (chdir(expand(buf, "~"))) {
    thr->qmstat = BAD_DIR;
    clean();
    YYABORT;
  } else {
    if (gflag) gcmd_s("q-cd-cmd", buf);
    new_xprompt();
  }
}

| _CHDIR_ arg
{
  char *s;
  if (strparam($2, &s)) {
    int res;
    char *buf =
      malloc((MAXSTRLEN+strlen(s)+1)*
	     sizeof(char));
    if (buf)
      expand(buf, s);
    else
      thr->qmstat = MEM_OVF;
    free(s);
    res = clean();
    if (res && buf)
      if (chdir(buf)) {
	thr->qmstat = BAD_DIR;
	res = clean();
      } else {
	if (gflag) gcmd_s("q-cd-cmd", buf);
	new_xprompt();
      }
    if (buf) free(buf);
    if (!res) YYABORT;
  } else
    YYABORT;
}

| _LS_
arg_list
{
  static char ls_cmd[] = "ls";
  char **argv = $2.v, *cmd;
  int argc = $2.c, l = 0, i;
  for (i = 0; i < argc; i++)
    l += strlen(argv[i])+1;
  if (!(cmd = (char*)malloc((strlen(ls_cmd)+l+1)*sizeof(char))))
    thr->qmstat = MEM_OVF;
  else {
    strcpy(cmd, ls_cmd);
    for (i = 0; i < argc; i++)
      strcat(strcat(cmd, " "), argv[i]);
    push_sigint(SIG_DFL);
    system(cmd);
    pop_sigint();
    free(cmd);
  }
  for (i = 0; i < argc; i++) free(argv[i]);
  free(argv);
  if (!clean()) YYABORT;
}

| _DEBUG_
arg_list
{
  extern int debug_parse_opts(char*);
  extern char *debug_get_opts(char*);
  char **argv = $2.v, *cmd;
  int argc = $2.c, l = 0, i;
  if (!argc) {
    free(argv);
    if (checkbrk) YYABORT;
    printf("debug is %s\n", debug?"on":"off");
  } else if (argc ==1 && (!strcmp(argv[0],"on") || !strcmp(argv[0],"off"))) {
    int b = !strcmp(argv[0],"on");
    for (i = 0; i < argc; i++) free(argv[i]);
    free(argv);
    debug = thr->debug = b;
    if (gflag) gcmd_b("q-debug-cmd", b);
  } else {
    for (i = 0; i < argc; i++)
      l += strlen(argv[i])+1;
    if (!(cmd = (char*)malloc((l+1)*sizeof(char))))
      thr->qmstat = MEM_OVF;
    else {
      *cmd = 0;
      for (i = 0; i < argc; i++) {
	if (i > 0) strcat(cmd, " ");
	strcat(cmd, argv[i]);
      }
      if (!debug_parse_opts(sys_to_utf8(cmd)))
	thr->qmstat = SYNTAX_ERR;
      else if (gflag) {
	char opts[MAXSTRLEN];
	gcmd_s("q-debug-options-cmd", debug_get_opts(opts));
      }
      free(cmd);
    }
    for (i = 0; i < argc; i++) free(argv[i]);
    free(argv);
    if (!clean()) YYABORT;
  }
}

| _BREAK_
{
  if (checkbrk) YYABORT;
  breakpoints();
}

| _BREAK_ arg
{
  int b;
  if (boolparam($2, &b)) {
    brkdbg = thr->brkdbg = b;
    if (gflag) gcmd_b("q-break-cmd", b);
  } else
    YYABORT;
}

| _BREAK_ breaksyms

| _TBREAK_ tbreaksyms

| _PROFILE_
{
  if (checkbrk) YYABORT;
  profile();
}

| _PROFILE_ profilesyms

| _CLEAR_
{
  if (checkbrk) YYABORT;
  clear_vars();
}

| _CLEAR_ _BREAK_
{
  int i;
  if (checkbrk) YYABORT;
  for (i = RESERVED; i < symtbsz+tmptbsz; i++) {
    if (symtb[i].flags & BRK) {
      symtb[i].flags &= ~BRK;
      nbreak--;
    }
    if (symtb[i].flags & TBRK) {
      symtb[i].flags &= ~TBRK;
      nbreak--;
    }
  }
}

| _CLEAR_ _PROFILE_
{
  int i;
  if (checkbrk) YYABORT;
  for (i = RESERVED; i < symtbsz+tmptbsz; i++) {
    if (symtb[i].flags & PROF) {
      symtb[i].flags &= ~PROF;
      symtb[i].nredns = 0;
      nprof--;
    }
  }
}

| _CLEAR_ clearsyms

| _CLEAR_ _BREAK_ clearbreaksyms

| _CLEAR_ _PROFILE_ clearprofilesyms

| _UNPARSE_
{
  if (checkbrk) YYABORT;
  printf("unparse is %s\n", unparseflag?"on":"off");
}

| _UNPARSE_ arg
{
  int b;
  if (boolparam($2, &b))
    unparseflag = b;
  else
    YYABORT;
}

| _ECHO_
{
  if (checkbrk) YYABORT;
  printf("echo is %s\n", eflag?"on":"off");
}

| _ECHO_ arg
{
  int b; char *s;
  if (boolstrparam($2, &b, &s))
    if (!s) {
      eflag = b;
      if (gflag) gcmd_b("q-echo-cmd", b);
    } else {
      printf("%s\n", s);
      free(s);
    }
  else
    YYABORT;
}

| _WHICH_
{
  if (checkbrk) YYABORT;
  printstr(which);
}

| _WHICH_ arg
{
  char *s;
  if (strparam($2, &s)) {
    int l0 = strlen(s), l = strlen(qpath)+l0;
    char *buf = malloc((l+3)*sizeof(char));
    char *name = malloc((l+MAXSTRLEN+3)*sizeof(char));
    if (buf && name) {
      absname(name, searchlib(buf, s));
      if (!chkfile(name)) {
	char *t = malloc((l0+3)*sizeof(char));
	if (!t) goto errexit;
	strcat(strcpy(t, s), ".q");
	absname(name, searchlib(buf, t));
	free(t);
      }
      if (chkfile(name))
	printstr(name);
      else
	printf("%s not found\n", s);
    } else {
    errexit:
      thr->qmstat = MEM_OVF;
    }
    clean();
    free(s);
    if (buf) free(buf);
    if (name) free(name);
  } else
    YYABORT;
}

| _EDIT_
{
  if (checkbrk) YYABORT;
  if (gflag)
    gcmd_s("q-edit-cmd", which);
  else if (!edit(which))
    YYABORT;
}

| _EDIT_ arg
{
  char *s;
  if (strparam($2, &s)) {
    int res;
    char *buf = malloc((MAXSTRLEN+strlen(s)+1)*sizeof(char));
    if (buf) {
      expand(buf, s);
      if (gflag)
	gcmd_s("q-edit-cmd", buf);
      else
	edit(buf);
    } else
      thr->qmstat = MEM_OVF;
    res = clean();
    free(s); if (buf) free(buf);
    if (!res) YYABORT;
  } else
    YYABORT;
}

| _SOURCE_ arg
{
  char *s;
  if (strparam($2, &s)) {
    int l = strlen(qpath)+strlen(s);
    char *buf = malloc((l+1)*sizeof(char));
    if (buf) {
      int res = parsesrc(searchlib(buf, s), 1);
      if (!res) {
	free(buf);
	free(s);
	YYABORT;
      }
    } else
      thr->qmstat = MEM_OVF;
    free(s);
    if (buf) free(buf);
    if (!clean())
      YYABORT;
  } else
    YYABORT;
}

| _LOAD_
{
  if (checkbrk) YYABORT;
  if (!save_file)
    save_file = strdup(".q_vars");
  if (!save_file) {
    thr->qmstat = MEM_OVF;
    clean();
    YYABORT;
  }
  if (chkfile(save_file)) {
    int _eflag = eflag, res;
    printf("loading ");
    printstr(save_file);
    eflag = 0;
    res = parsesrc(save_file, 0);
    eflag = _eflag;
    if (!res) YYABORT;
  }
}

| _LOAD_ arg
{
  char *s;
  if (strparam($2, &s)) {
    char *buf = malloc((strlen(s)+MAXSTRLEN+1)*sizeof(char));
    if (buf) {
      int _eflag = eflag, res;
      eflag = 0;
      res = parsesrc(expand(buf, s), 0);
      eflag = _eflag;
      if (res) {
	if (save_file) free(save_file);
	save_file = buf;
      } else {
	free(buf);
	free(s);
	YYABORT;
      }
    } else
      thr->qmstat = MEM_OVF;
    free(s);
    if (!clean())
      YYABORT;
  } else
    YYABORT;
}

| _SAVE_
{
  if (checkbrk) YYABORT;
  if (!save_file)
    save_file = strdup(".q_vars");
  if (!save_file) {
    thr->qmstat = MEM_OVF;
    clean();
    YYABORT;
  }
  printf("saving ");
  printstr(save_file);
  if (!save_vars(NULL)) YYABORT;
}

| _SAVE_ arg
{
  char *s;
  if (strparam($2, &s)) {
    int res;
    char *buf = malloc((strlen(s)+MAXSTRLEN+1)*sizeof(char));
    if (buf) {
      res = save_vars(expand(buf, s));
      if (!res)	free(buf);
    } else {
      thr->qmstat = MEM_OVF;
      res = 0;
    }
    free(s);
    if (!res) YYABORT;
  } else
    YYABORT;
}

| _IMPORT_ arg_list
{
  char **argv = $2.v;
  int count = 0, i, argc = $2.c;
  if (argv && argc > 0) {
    for (i = 0; i < argc; i++) {
      int res;
      if (*argv[i] == '-')
	res = del_import(argv[i]+1);
      else if (*argv[i] == '+')
	res = add_import(strdup(argv[i]+1));
      else
	res = add_import(strdup(argv[i]));
      if (res) count++;
      free(argv[i]);
    }
    free(argv);
    if (count)
      rerun(NULL, 0, NULL);
  } else if (clear_imports())
    rerun(NULL, 0, NULL);
}

| _IMPORTS_
{ if (!import_list()) YYABORT; }

| _MODULES_
{ if (!modules()) YYABORT; }

| _WHO_
{ if (!who(0)) YYABORT; }

| _WHO_ arg
{ if (!allparam($2) || !who(1)) YYABORT; }

| _WHOS_ whossyms

| _WHOIS_ whoissyms

| _RUN_
{ if (checkbrk) YYABORT; rerun(NULL, 0, NULL); }

| _RUN_ arg
{
  char *s;
  if (strparam($2, &s)) {
    $<aval>$.v = (char**)malloc((MAXARGC+1)*sizeof(char*));
    if (!$<aval>$.v) {
      thr->qmstat = MEM_OVF;
      free(s);
      clean();
      YYABORT;
    }
    $<aval>$.v[0] = s;
    $<aval>$.c = 1;
  } else
    YYABORT;
}
opt_arg_list
{
  char **argv = $4.v;
  int argc = $4.c, ret;
  argv[argc] = NULL;
  ret = rerun(*argv, argc, argv);
  while (argc > 0)
    if (argv[argc-1]) free(argv[--argc]);
  if (argv) free(argv);
  if (!ret) YYABORT;
}
;

breaksyms: breaksym | breaksyms breaksym ;

breaksym: symbol
{
  if (checkbrk) YYABORT;
  savepos();
  if (!(symtb[$1].flags&VSYM)) {
    if (symtb[$1].flags & TBRK) {
      symtb[$1].flags &= ~TBRK;
      nbreak--;
    }
    if (!(symtb[$1].flags & BRK)) {
      symtb[$1].flags |= BRK;
      nbreak++;
    }
    if (!brkdbg || !thr->brkdbg) {
      brkdbg = thr->brkdbg = 1;
      if (gflag) gcmd_b("q-break-cmd", 1);
    }
  } else {
    thr->qmstat = BAD_SYM;
    clean();
    YYABORT;
  }
}
;

tbreaksyms: tbreaksym | tbreaksyms tbreaksym ;

tbreaksym: symbol
{
  if (checkbrk) YYABORT;
  savepos();
  if (!(symtb[$1].flags&VSYM)) {
    if (symtb[$1].flags & BRK) {
      symtb[$1].flags &= ~BRK;
      nbreak--;
    }
    if (!(symtb[$1].flags & TBRK)) {
      symtb[$1].flags |= TBRK;
      nbreak++;
    }
    if (!brkdbg || !thr->brkdbg) {
      brkdbg = thr->brkdbg = 1;
      if (gflag) gcmd_b("q-break-cmd", 1);
    }
  } else {
    thr->qmstat = BAD_SYM;
    clean();
    YYABORT;
  }
}
;

profilesyms: profilesym | profilesyms profilesym ;

profilesym: symbol
{
  if (checkbrk) YYABORT;
  savepos();
  if (!(symtb[$1].flags&VSYM)) {
    if (!(symtb[$1].flags & PROF)) {
      symtb[$1].flags |= PROF;
      symtb[$1].nredns = 0;
      nprof++;
    }
  } else {
    thr->qmstat = BAD_SYM;
    clean();
    YYABORT;
  }
}
;

clearsyms: clearsym | clearsyms clearsym ;

clearsym: symbol
{
  if (checkbrk) YYABORT;
  savepos();
  if (!(symtb[$1].flags & VSYM)) {
    thr->qmstat = BAD_SYM;
    clean();
    YYABORT;
  } else if (!unassignment($1))
    YYABORT; 
}
;

clearbreaksyms: clearbreaksym | clearbreaksyms clearbreaksym ;

clearbreaksym: symbol
{
  if (checkbrk) YYABORT;
  savepos();
  if (!(symtb[$1].flags & VSYM)) {
    if (symtb[$1].flags & BRK) {
      symtb[$1].flags &= ~BRK;
      nbreak--;
    }
    if (symtb[$1].flags & TBRK) {
      symtb[$1].flags &= ~TBRK;
      nbreak--;
    }
  } else {
    thr->qmstat = BAD_SYM;
    clean();
    YYABORT;
  }
}
;

clearprofilesyms: clearprofilesym | clearprofilesyms clearprofilesym ;

clearprofilesym: symbol
{
  if (checkbrk) YYABORT;
  savepos();
  if (!(symtb[$1].flags & VSYM)) {
    if (symtb[$1].flags & PROF) {
      symtb[$1].flags &= ~PROF;
      symtb[$1].nredns = 0;
      nprof--;
    }
  } else {
    thr->qmstat = BAD_SYM;
    clean();
    YYABORT;
  }
}
;

whossyms: whossym | whossyms whossym ;

whossym: whossymbol { extern char yylastsym[];
                      if (checkbrk) YYABORT;
		      if ($1 == NONE && thr->qmstat == BAD_REF &&
			  *yylastsym) {
			thr->qmstat = OK;
			ambiguous($1, yylastsym);
		      } else if ($1 != NONE) {
			printf("\n");
			whos($1);
		      } else
			YYABORT; } ;

whoissyms: whoissym | whoissyms whoissym ;

whoissym: whossymbol { extern char yylastsym[];
                       if (checkbrk) YYABORT;
		       if ($1 == NONE && thr->qmstat == BAD_REF &&
			   *yylastsym) {
			 thr->qmstat = OK;
			 whois($1, yylastsym);
		       } else if ($1 != NONE && *yylastsym) {
			 whois($1, yylastsym);
		       } else if ($1 != NONE) {
			 printf("\n");
			 whos($1);
		       } else
			 YYABORT; } ;

symbol: ID | '(' op ')' { $$ = $2; } ;

whossymbol: ID | '(' op ')' { $$ = $2; } | ERRTOK { $$ = NONE; } ;

ID: NID | QID

opt_arg_list:

/* empty */
{
  $$ = $<aval>0;
}

|
opt_arg_list arg
{
  char *s;
  if ($1.c >= MAXARGC) {
    thr->qmstat = ARGS_ERR;
    while ($1.c > 0)
      free($1.v[--$1.c]);
    free($1.v);
    clean();
    YYABORT;
  } else if (strparam($2, &s)) {
    $1.v[$1.c++] = s;
  } else {
    while ($1.c > 0)
      free($1.v[--$1.c]);
    free($1.v);
    YYABORT;
  }
  $$ = $1;
}
;

arg_list:

/* empty */
{
  if (checkbrk) YYABORT;
  $<aval>$.v = (char**)malloc((MAXARGC+1)*sizeof(char*));
  if (!$<aval>$.v) {
    thr->qmstat = MEM_OVF;
    clean();
    YYABORT;
  }
  $<aval>$.c = 0;
}

|
arg_list arg
{
  char *s;
  if ($1.c >= MAXARGC) {
    thr->qmstat = ARGS_ERR;
    while ($1.c > 0)
      free($1.v[--$1.c]);
    free($1.v);
    clean();
    YYABORT;
  } else if (strparam($2, &s))
    $1.v[$1.c++] = s;
  else {
    while ($1.c > 0)
      free($1.v[--$1.c]);
    free($1.v);
    YYABORT;
  }
  $$ = $1;
}
;

arg		: ARG
		| ARGERR
				{ YYERROR; }
		;

vardecls	: vardecl
		| vardecls ',' vardecl
		;

vardecl		: NID
				{ if (!vardecl($1)) YYABORT; }
		  initializer
		;

initializer	: /* empty */
		| '=' expression0
				{ if (!initializer()) YYABORT; }
		;

defs		: def
		| defs ',' def
		;

def		: expression0 '=' expression0
				{ if (!assignment()) YYABORT; }
		;

undefs		: undef
		| undefs ',' undef
		;

undef		: ID
				{ if (!unassignment($1)) YYABORT; }
		;

expression0	: sequence0
		| '\\' lambda0
		;

lambda0		: primary '.' expression0
				{ if (!pushbin(LAMBDAOP)) YYABORT; }
		| primary lambda0
				{ if (!pushbin(LAMBDAOP)) YYABORT; }
		;

sequence0	: cond0
		| sequence0 seqop cond0
				{ if (!pushbin($2)) YYABORT; }
		;

cond0		: rightapp0
		| IF rightapp THEN cond0 ELSE cond0
				{ static char sym[20];
				  int fno = mksym(strcpy(sym, "cond::ifelse"));
				  if (fno == NONE) {
				    thr->qmstat = SYNTAX_ERR; YYABORT;
				  } else if (!pushtern(fno))
				    YYABORT; }
		| IF rightapp THEN cond0
				{ static char sym[20];
				  int fno = mksym(strcpy(sym, "cond::when"));
				  if (fno == NONE) {
				    thr->qmstat = SYNTAX_ERR; YYABORT;
				  } else if (!pushbin(fno))
				    YYABORT; }
		;

rightapp0	: relation0
		| relation0 rappop rightapp0
				{ if (!pushbin($2)) YYABORT; }
		;

relation0	: addition
		| addition relop0 addition
				{ if (!pushbin($2)) YYABORT; }
		;

relop0		: EQUIV		{ $$ = IDOP; }
		| OP2
		;

expression	: sequence
		| '\\' lambda
		;

lambda		: primary '.' expression
				{ if (!pushbin(LAMBDAOP)) YYABORT; }
		| primary lambda
				{ if (!pushbin(LAMBDAOP)) YYABORT; }
		;

sequence	: cond
		| sequence seqop cond
				{ if (!pushbin($2)) YYABORT; }
		;

seqop		: OP0
		;

cond		: rightapp
		| IF rightapp THEN cond ELSE cond
				{ static char sym[20];
				  int fno = mksym(strcpy(sym, "cond::ifelse"));
				  if (fno == NONE) {
				    thr->qmstat = SYNTAX_ERR; YYABORT;
				  } else if (!pushtern(fno))
				    YYABORT; }
		| IF rightapp THEN cond
				{ static char sym[20];
				  int fno = mksym(strcpy(sym, "cond::when"));
				  if (fno == NONE) {
				    thr->qmstat = SYNTAX_ERR; YYABORT;
				  } else if (!pushbin(fno))
				    YYABORT; }
		;

rightapp	: relation
		| relation rappop rightapp
				{ if (!pushbin($2)) YYABORT; }
		;

rappop		: OP1
		;

relation	: addition
		| addition relop addition
				{ if (!pushbin($2)) YYABORT; }
		;

relop		: '='		{ $$ = EQOP; }
		| EQUIV		{ $$ = IDOP; }
		| OP2
		;

addition	: multiplication
		| addition addop multiplication
				{ if (!pushbin($2)) YYABORT; }
		| addition '-' multiplication
				{ if (!pushbin(MINOP)) YYABORT; }
		;

addop		: OP3 ELSE	{ if ($1 != OROP) {
				    thr->qmstat = SYNTAX_ERR;
				    YYABORT;
				  } else
				    $$ = ORELSEOP; }
		| OP3
		;

multiplication	: unary
		| multiplication mulop unary
				{ if (!pushbin($2)) YYABORT; }
		;

mulop		: OP4 THEN	{ if ($1 != ANDOP) {
				    thr->qmstat = SYNTAX_ERR;
				    YYABORT;
				  } else
				    $$ = ANDTHENOP; }
		| OP4
		;

/* ! ambiguous rule */

unary		: script
		| '-' INT	{ mpz_neg($2, $2);
		                  if (!pushmpz(thr, $2)) YYABORT; }
		| '-' FLOAT	{ if (!pushfloat(thr, -$2)) YYABORT; }
		| '-' unary     { if (!pushun(UMINOP)) YYABORT; }
		| unop unary    { if (!pushun($1)) YYABORT; }
		;

unop		: OP5
		;

script		: composition
		| composition scriptop script
				{ if (!pushbin($2)) YYABORT; }
		;

scriptop	: OP6
		;

composition	: application
		| composition compop application
				{ if (!pushbin($2)) YYABORT; }
		;

compop		: '.'		{ $$ = COMPOP; }
		| OP7
		;

application	: primary
		| application primary
				{ if (!pushfun(thr, APPOP)) YYABORT; }
		;

quoteop		: '~'		{ $$ = FORCEOP; }
		| OP9
		;

primary
		/* constants: */

		: INT		{ if (!pushmpz(thr, $1)) YYABORT; }
		| FLOAT         { if (!pushfloat(thr, $1)) YYABORT; }
		| STR           { if (!pushstr(thr, $1)) YYABORT; }
		| STRERR	{ YYERROR; }

		/* variable and function symbols: */

		| ID            { if ($1 == NONE) { YYABORT; }
				  else if (!pushfun(thr, $1)) YYABORT; }
		| '(' op ')'	{ if (!pushfun(thr, $2)) YYABORT; }

/* reconstruction rules for external objects to be added here */

		| XID		{ thr->qmstat = BAD_OBJ; YYABORT; }

		/* quoted expressions: */

		| quoteop primary
				{ if (!pushun($1)) YYABORT; }

                /* sections: */

		| '(' sequence seqop ')'
				{ if (!pushun($3)) YYABORT; }

		| '(' seqop rightapp ')'
				{ if (!(pushfun(thr, FLIPOP) && pushfun(thr, $2) &&
					pushfun(thr, APPOP) && swap() &&
					pushfun(thr, APPOP))) YYABORT; }

		| '(' relation rappop ')'
				{ if (!pushun($3)) YYABORT; }

		| '(' rappop rightapp ')'
				{ if (!(pushfun(thr, FLIPOP) && pushfun(thr, $2) &&
					pushfun(thr, APPOP) && swap() &&
					pushfun(thr, APPOP))) YYABORT; }

		| '(' addition relop ')'
				{ if (!pushun($3)) YYABORT; }

		| '(' relop addition ')'
				{ if (!(pushfun(thr, FLIPOP) && pushfun(thr, $2) &&
					pushfun(thr, APPOP) && swap() &&
					pushfun(thr, APPOP))) YYABORT; }

		| '(' addition addop ')'
				{ if (!pushun($3)) YYABORT; }

		| '(' addition '-' ')'
				{ if (!pushun(MINOP)) YYABORT; }

		| '(' addop multiplication ')'
				{ if (!(pushfun(thr, FLIPOP) && pushfun(thr, $2) &&
					pushfun(thr, APPOP) && swap() &&
					pushfun(thr, APPOP))) YYABORT; }

		| '(' multiplication mulop ')'
				{ if (!pushun($3)) YYABORT; }

		| '(' mulop unary ')'
				{ if (!(pushfun(thr, FLIPOP) && pushfun(thr, $2) &&
					pushfun(thr, APPOP) && swap() &&
					pushfun(thr, APPOP))) YYABORT; }

		| '(' composition scriptop ')'
				{ if (!pushun($3)) YYABORT; }

		| '(' scriptop script ')'
				{ if (!(pushfun(thr, FLIPOP) && pushfun(thr, $2) &&
					pushfun(thr, APPOP) && swap() &&
					pushfun(thr, APPOP))) YYABORT; }

		| '(' composition compop ')'
				{ if (!pushun($3)) YYABORT; }

		| '(' compop application ')'
				{ if (!(pushfun(thr, FLIPOP) && pushfun(thr, $2) &&
					pushfun(thr, APPOP) && swap() &&
					pushfun(thr, APPOP))) YYABORT; }

		/* parenthesized expressions and tuples: */

		| '(' ')'
				{ if (!pushfun(thr, VOIDOP)) YYABORT; }
/* special 1-tuple construct removed as of Q 6.2
		| '(' primary ')'
				{ if (!pushfun(thr, VOIDOP) ||
				      !pushfun(thr, PAIROP)) YYABORT; }
*/
		| '(' expression ')'
		| '(' expression ',' ')'
				{ if (!pushfun(thr, VOIDOP) ||
				      !pushfun(thr, PAIROP)) YYABORT; }
		| '(' expression ';' ')'
				{ if (!pushfun(thr, VOIDOP) ||
				      !pushfun(thr, PAIROP) ||
				      !pushfun(thr, VOIDOP) ||
				      !pushfun(thr, PAIROP)) YYABORT; }
		| '(' expression '|' expression ')'
				{ if (!pushfun(thr, PAIROP)) YYABORT; }
		| '(' expression DOTDOT
				{ if (!pushfun(thr, NILOP) ||
				      !pushfun(thr, CONSOP)) YYABORT; }
		  opt_expr ')'
				{ if (($5)?(!pushbin(TENUMOP)):
				      (!pushun(TENUM1OP))) YYABORT; }
		| '(' expression ',' expr_list1 ')'
				{ if (!pushfun(thr, VOIDOP) ||
				      !pushtuple($4.n+1, $4.m)) YYABORT; }
		| '(' expression ',' expr_list1 ',' ')'
				{ if (!pushfun(thr, VOIDOP) ||
				      !pushtuple($4.n+1, $4.m)) YYABORT; }
		| '(' expression ',' expr_list1 ';' ')'
				{ if ($4.m < 0) $4.m = 0;
				  if (!pushfun(thr, VOIDOP) ||
				      !pushtuple($4.n+1, $4.m)) YYABORT; }
		| '(' expression ',' expr_list1 '|' expression ')'
				{ if (!pushtuple($4.n+1, $4.m)) YYABORT; }
		| '(' expression ',' expr_list1 DOTDOT
				{ if (!pushfun(thr, NILOP) ||
				      !pushlist($4.n+1, $4.m)) YYABORT; }
		  opt_expr ')'
				{ if (($7)?(!pushbin(TENUMOP)):
				      (!pushun(TENUM1OP))) YYABORT; }
		| '(' expression ':'
				{ static char sym[20];
				  int fno = mksym(strcpy(sym, "cond::tupleof"));
				  if (fno == NONE) {
				    thr->qmstat = SYNTAX_ERR; YYABORT;
				  } else
				    $<ival>$ = fno; }
		  expr_list ')'
				{ if (!pushfun(thr, VOIDOP) ||
				      !pushtuple($5.n+1, $5.m) ||
				      !pushbin($<ival>4)) YYABORT; }
/* handle the special case of a group of size 1 at the beginning of the
   tuple */
		| '(' expression ';' expr_list2 ')'
				{ if (!pushfun(thr, VOIDOP) ||
				      !pushtuple($4.n+1, $4.m)) YYABORT; }
		| '(' expression ';' expr_list2 ',' ')'
				{ if (!pushfun(thr, VOIDOP) ||
				      !pushtuple($4.n+1, $4.m)) YYABORT; }
		| '(' expression ';' expr_list2 ';' ')'
				{ if (!pushfun(thr, VOIDOP) ||
				      !pushtuple($4.n+1, $4.m)) YYABORT; }
		| '(' expression ';' expr_list2 '|' expression ')'
				{ if (!pushtuple($4.n+1, $4.m)) YYABORT; }

		/* lists: */

		| '[' ']'
				{ if (!pushfun(thr, NILOP)) YYABORT; }
		| '[' expr_list ']'
				{ if (!pushfun(thr, NILOP) ||
				      !pushlist($2.n+1, $2.m)) YYABORT; }
		| '[' expr_list ',' ']'
				{ if (!pushfun(thr, NILOP) ||
				      !pushlist($2.n+1, $2.m)) YYABORT; }
		| '[' expr_list ';' ']'
				{ if ($2.m < 0) $2.m = 0;
				  if (!pushfun(thr, NILOP) ||
				      !pushlist($2.n+1, $2.m)) YYABORT; }
		| '[' expr_list '|' expression ']'
				{ if (!pushlist($2.n+1, $2.m)) YYABORT; }
		| '[' expr_list DOTDOT
				{ if (!pushfun(thr, NILOP) ||
				      !pushlist($2.n+1, $2.m)) YYABORT; }
		  opt_expr ']'
				{ if (($5)?(!pushbin(ENUMOP)):
				      (!pushun(ENUM1OP))) YYABORT; }
		| '[' expression ':'
				{ static char sym[20];
				  int fno = mksym(strcpy(sym, "cond::listof"));
				  if (fno == NONE) {
				    thr->qmstat = SYNTAX_ERR; YYABORT;
				  } else
				    $<ival>$ = fno; }
		  expr_list ']'
				{ if (!pushfun(thr, VOIDOP) ||
				      !pushtuple($5.n+1, $5.m) ||
				      !pushbin($<ival>4)) YYABORT; }

		/* streams: */

		| '{' '}'
				{ if (!pushfun(thr, SNILOP)) YYABORT; }
		| '{' expr_list '}'
				{ if (!pushfun(thr, SNILOP) ||
				      !pushstream($2.n+1, $2.m)) YYABORT; }
		| '{' expr_list ',' '}'
				{ if (!pushfun(thr, SNILOP) ||
				      !pushstream($2.n+1, $2.m)) YYABORT; }
		| '{' expr_list ';' '}'
				{ if ($2.m < 0) $2.m = 0;
				  if (!pushfun(thr, SNILOP) ||
				      !pushstream($2.n+1, $2.m)) YYABORT; }
		| '{' expr_list '|' expression '}'
				{ if (!pushstream($2.n+1, $2.m)) YYABORT; }
		| '{' expr_list DOTDOT
				{ if (!pushfun(thr, NILOP) ||
				      !pushlist($2.n+1, $2.m)) YYABORT; }
		  opt_expr '}'
				{ if (($5)?(!pushbin(SENUMOP)):
				      (!pushun(SENUM1OP))) YYABORT; }
		| '{' expression ':'
				{ static char sym[20];
				  int fno = mksym(strcpy(sym, "cond::streamof"));
				  if (fno == NONE) {
				    thr->qmstat = SYNTAX_ERR; YYABORT;
				  } else
				    $<ival>$ = fno; }
		  expr_list '}'
				{ if (!pushfun(thr, VOIDOP) ||
				      !pushtuple($5.n+1, $5.m) ||
				      !pushbin($<ival>4)) YYABORT; }

		/* catch syntax errors here, so no default redns are taken
		   before an expression has been fully parsed */
		| error { YYABORT; }

		;

opt_expr	: /* empty */ { $$ = 0; }
		| expression { $$ = 1; }
		;

/* we use left-recursive rules here, and collect the lists on the
   interpreter's stack, to prevent stack overflows in the parser */

expr_list1	: expression
				{ $$.n = 2; $$.m = -1; }
		| expr_list1 ',' expression
				{ $$.n = $1.n+1; $$.m = $1.m; }
		| expr_list1 ';' expression
				{ $1.n++; if (!pushgroup(&$1.n, &$1.m)) YYABORT;
				  $$.n = $1.n; $$.m = $1.m; }
		;

expr_list2	: expression
				{ if (!swap() ||
				      !pushfun(thr, VOIDOP) ||
				      !pushfun(thr, PAIROP) ||
				      !swap()) YYABORT;
				  $$.n = 2; $$.m = 1; }
		| expr_list2 ',' expression
				{ $$.n = $1.n+1; $$.m = $1.m; }
		| expr_list2 ';' expression
				{ $1.n++; if (!pushgroup(&$1.n, &$1.m)) YYABORT;
				  $$.n = $1.n; $$.m = $1.m; }
		;

expr_list	: expression
				{ $$.n = 1; $$.m = -1; }
		| expr_list ',' expression
				{ $$.n = $1.n+1; $$.m = $1.m; }
		| expr_list ';' expression
				{ $1.n++; if (!pushgroup(&$1.n, &$1.m)) YYABORT;
				  $$.n = $1.n; $$.m = $1.m; }
		;

op		: '='		{ $$ = EQOP; }
		| EQUIV		{ $$ = IDOP; }
		| '-'		{ $$ = MINOP; }
		| OP3 ELSE	{ if ($1 != OROP) {
				    thr->qmstat = SYNTAX_ERR;
				    YYABORT;
				  } else
				    $$ = ORELSEOP; }
		| OP4 THEN	{ if ($1 != ANDOP) {
				    thr->qmstat = SYNTAX_ERR;
				    YYABORT;
				  } else
				    $$ = ANDTHENOP; }
		| '~'		{ $$ = FORCEOP; }
		| '.'		{ $$ = COMPOP; }
		| OP0
		| OP1
		| OP2
		| OP3
		| OP4
		| OP5
		| OP6
		| OP7
		| OP9
		;

%%

yyerror(s)
	char           *s;
{
  if (thr->qmstat == OK && (thr > thr0 || !checkbrk) &&
      (strcmp(s, "parse error") == 0 || strcmp(s, "syntax error") == 0))
    thr->qmstat = SYNTAX_ERR;
}

/* unicode helpers */

#ifdef HAVE_UNICODE

static inline long
u8decodes(char *s, char **t)
{
  size_t n = 0;
  unsigned p = 0, q = 0;
  unsigned long c = 0;
  if (s[0] == 0)
    return -1;
  else if (s[1] == 0) {
    *t = s+1;
    return (unsigned char)s[0];
  }
  for (; n == 0 && *s; s++) {
    unsigned char uc = (unsigned char)*s;
    if (q == 0) {
      if (((signed char)uc) < 0) {
	switch (uc & 0xf0) {
	case 0xc0: case 0xd0:
	  q = 1;
	  c = uc & 0x1f;
	  break;
	case 0xe0:
	  q = 2;
	  c = uc & 0xf;
	  break;
	case 0xf0:
	  if ((uc & 0x8) == 0) {
	    q = 3;
	    c = uc & 0x7;
	  } else
	    c = uc;
	  break;
	default:
	  c = uc;
	  break;
	}
      } else
	c = uc;
      p = 0; if (q == 0) n++;
    } else if ((uc & 0xc0) == 0x80) {
      /* continuation byte */
      c = c << 6 | (uc & 0x3f);
      if (--q == 0)
	n++;
      else
	p++;
    } else {
      /* malformed char */
      return -1;
    }
  }
  if (n == 1) {
    *t = s;
    return c;
  } else
    return -1;
}

static inline char *
u8encode(char *t, unsigned long c)
{
  unsigned char *uc = (unsigned char*)t;
  if (c < 0x80) {
    uc[1] = 0;
    uc[0] = c;
  } else if (c < 0x800) {
    uc[2] = 0;
    uc[1] = 0x80 | c&0x3f;
    c = c >> 6;
    uc[0] = 0xc0 | c;
  } else if (c < 0x10000) {
    uc[3] = 0;
    uc[2] = 0x80 | c&0x3f;
    c = c >> 6;
    uc[1] = 0x80 | c&0x3f;
    c = c >> 6;
    uc[0] = 0xe0 | c;
  } else {
    uc[4] = 0;
    uc[3] = 0x80 | c&0x3f;
    c = c >> 6;
    uc[2] = 0x80 | c&0x3f;
    c = c >> 6;
    uc[1] = 0x80 | c&0x3f;
    c = c >> 6;
    uc[0] = 0xf0 | c;
  }
  return t;
}

#ifdef HAVE_ICONV

#define CHUNKSZ 128

static inline char *
icfromutf8(iconv_t ic[2], char *s)
{
  if (ic[1] == (iconv_t)-2) {
    char *codeset = default_encoding();
    if (codeset && strcmp(codeset, "UTF-8"))
      ic[1] = iconv_open(codeset, "UTF-8");
    else
      ic[1] = (iconv_t)-1;
  }
  if (ic[1] == (iconv_t)-1)
    return NULL;
  else {
    /* Here the input buffer may be NULL, to emit a terminating shift
       sequence. In this case we initialize an output buffer of size
       CHUNKSZ. */
    size_t l = s?strlen(s):0, m = s?l:CHUNKSZ;
    char *t = malloc(m+1), *t1;
    char *inbuf = s, *outbuf = t;
    size_t inbytes = l, outbytes = m;

    while (iconv(ic[1], &inbuf, &inbytes, &outbuf, &outbytes) ==
	   (size_t)-1)
      if (errno == E2BIG) {
	/* try to enlarge the output buffer */
	size_t k = outbuf-t;
	if ((t1 = realloc(t, m+CHUNKSZ+1))) {
	  t = t1;
	  outbuf = t+k;
	  m += CHUNKSZ;
	  outbytes += CHUNKSZ;
	} else {
	  /* memory overflow */
	  free(t);
	  return NULL;
	}
      } else {
	/* conversion error */
	free(t);
	return NULL;
      }
    /* terminate the output string */
    *outbuf = 0;
    if (!(t1 = realloc(t, strlen(t)+1)))
      /* this shouldn't happen */
      return t;
    else
      return t1;
  }
}

/* conversion between UTF-8 and wchar_t */

static iconv_t myic[2] = { (iconv_t)-1, (iconv_t)-1 };

static inline wchar_t *
ictowcs(wchar_t *t, char *s)
{
  if (myic[1] == (iconv_t)-1)
    myic[1] = iconv_open("WCHAR_T", "UTF-8");
  if (myic[1] == (iconv_t)-1)
    return NULL;
  else {
    size_t l = strlen(s);
    char *inbuf = s; wchar_t *outbuf = t;
    size_t inbytes = l, outbytes = l*sizeof(wchar_t);

    if (iconv(myic[1], &inbuf, &inbytes, (char**)&outbuf, &outbytes) ==
	(size_t)-1)
      return NULL;
    /* terminate the output string */
    *outbuf = 0;
    return t;
  }
}

#endif

#ifdef __STDC_ISO_10646__
#define MY_ISO_10646
#else
#ifndef HAVE_ICONV
#warning "wchar_t encoding unknown and iconv not available, assuming ISO 10646"
#define MY_ISO_10646
#endif
#endif

#ifdef MY_ISO_10646
#define towchar(c) ((wchar_t)(c))
#else
static wchar_t towchar(unsigned long c)
{
  char s[5]; /* single utf-8 char can have at most 4 bytes, plus terminal 0 */
  wchar_t t[5]; /* just to be safe; 2 should actually be enough */
  u8encode(s, c);
  if (ictowcs(t, s))
    return t[0];
  else
    /* Your system is so utterly broken that we can't even convert UTF-8 to
       wchar_t. You should probably configure with --without-unicode. But
       let's just pretend we have an ISO 10646 compatible encoding anyway. */
    return (wchar_t)c;
}
#endif

#endif

/* routines to construct expressions: */

static swap()
	/* swap the topmost two stack elements */
{
  EXPR		*x;

  x = thr->xsp[-2]; thr->xsp[-2] = thr->xsp[-1]; thr->xsp[-1] = x;
  return (1);
}

static rot()
	/* rotate the topmost three stack elements */
{
  EXPR		*x;

  x = thr->xsp[-3]; thr->xsp[-3] = thr->xsp[-2];
  thr->xsp[-2] = thr->xsp[-1]; thr->xsp[-1] = x;
  return (1);
}

static pushtern(fno)
     /* process ternary op */
     int		fno;
{
  return (rot() && pushfun(thr, fno) && swap() && pushfun(thr, APPOP) &&
	  rot() && pushfun(thr, APPOP) && swap() && pushfun(thr, APPOP));
}

static pushbin(fno)
     /* process binary op */
     int		fno;
{
  return (swap() && pushfun(thr, fno) && swap() && pushfun(thr, APPOP) &&
	  swap() && pushfun(thr, APPOP));
}

static pushun(fno)
	/* process unary op */
     int		fno;
{
  return (pushfun(thr, fno) && swap() && pushfun(thr, APPOP));
}

static pushlist(int n, int m)
/* process list elements */
{
  if (m >= 0 && !pushgroup(&n, &m)) return 0;
  while (--n > 0)
    if (!pushfun(thr, CONSOP)) return 0;
  return 1;
}

static pushstream(int n, int m)
/* process stream elements */
{
  if (m >= 0 && !pushgroup(&n, &m)) return 0;
  while (--n > 0)
    if (!pushbin(SCONSOP)) return 0;
  return 1;
}

static pushtuple(int n, int m)
/* process tuple elements */
{
  if (m >= 0 && !pushgroup(&n, &m)) return 0;
  while (--n > 0)
    if (!pushfun(thr, PAIROP)) return 0;
  return 1;
}

static pushgroup(int *n, int *m)
/* process element group */
{
  int k, res;
  EXPR *x;
  if (*m < 0) *m = 0;
  k = *n-*m;
  x = *--thr->xsp;
  res = pushfun(thr, VOIDOP) && pushtuple(k, -1);
  if (thr->xsp) *(thr->xsp++) = x;
  *n -= k-2; *m = *n-1;
  return res;
}

/* eval stats */

static init_stats()
{
  thr0->stats_init = 1;
  thr0->stats_fini = 0;
  thr0->nredns = thr0->nexprs = thr0->maxexprs = 0;
  thr0->starttime = clock();
  return 1;
}

static fini_stats()
{
  thr0->endtime = clock();
  thr0->stats_fini = 1;
  return 1;
}

static print_stats(all)
     int all;
{
  THREAD *thr;
  for (thr = thr0; thr < thr0+(all?nthreads:1); thr++)
    if (thr->used && (thr == thr0 || !thr->active) && thr->stats_init) {
      if (!thr->stats_fini ||
	  thr->starttime == (clock_t)-1 ||
	  thr->endtime == (clock_t)-1)
	if (all || thr > thr0)
	  printf("thread #%d: %lu reduction%s, %lu cell%s\n",
		 thr-thr0,
		 thr->nredns, thr->nredns!=1?"s":"",
		 thr->maxexprs, thr->maxexprs!=1?"s":"");
	else
	  printf("%lu reduction%s, %lu cell%s\n",
		 thr->nredns, thr->nredns!=1?"s":"",
		 thr->maxexprs, thr->maxexprs!=1?"s":"");
      else
	if (all || thr > thr0)
	  printf("thread #%d: %0.3g secs, %lu reduction%s, %lu cell%s\n",
		 thr-thr0,
		 ((double)(thr->endtime-thr->starttime))/(double)CLOCKS_PER_SEC,
		 thr->nredns, thr->nredns!=1?"s":"",
		 thr->maxexprs, thr->maxexprs!=1?"s":"");
	else
	  printf("%0.3g secs, %lu reduction%s, %lu cell%s\n",
		 ((double)(thr->endtime-thr->starttime))/(double)CLOCKS_PER_SEC,
		 thr->nredns, thr->nredns!=1?"s":"",
		 thr->maxexprs, thr->maxexprs!=1?"s":"");
  }
}

/* parse actions: */

#if defined(HAVE_UNICODE) && !defined(HAVE_WCWIDTH)
static inline wcwidth(wchar_t c)
{
  if (iswprint(c))
    return 1;
  else
    return -1;
}
#endif

static int tokoffs = 0;

static error_report(base, p, tok)
     char *base, *p, *tok;
{
  char *q = p, *r = strchr(p, '\n'), *s, *t, c;
  int l = strlen(tok), i, xw = 0;
  while (q > base && q[-1] != '\n') --q;
  if (r) c = *r, *r = 0;
  if (tok && *tok && p-q >= l)
    if (strncmp(p-l, tok, l) == 0)
      p -= l;
    else if (p-q >= l+1 && strncmp(p-l-1, tok, l) == 0)
      /* token may actually be off one to the left because of lookahead */
      p -= l+1;
    else
      /* token not found; assume it's the character to the left of the current
	 position */
      --p;
  if (tok && strcmp(tok, "\"") == 0 && *p == '"') {
    /* it's a string token, try to back up to the beginning of the string */
    while (p > q && (p[-1] != '"' || p > q+1 && p[-2] == '\\')) --p;
    if (p > q && p[-1] == '"') p = p+tokoffs-1;
  }
  if (p < q) p = q;
  s = malloc(p-q+2);
  if (!s) fatal("memory overflow");
  strncpy(s, q, p-q+1);
  s[p-q+1] = s[p-q] = 0;
#ifdef HAVE_UNICODE
  {
    /* Try to guestimate the actual width of the output string. */
    char *s1 = s, *t1, *s2 = malloc(p-q+2), *t2 = s2;
    if (!s2) fatal("memory overflow");
    while (*s1) {
      long c = u8decodes(s1, &t1);
      if (c < 0)
	/* No legal UTF-8, bail out. */
	break;
      else {
	int wd = wcwidth(towchar(c));
	if (c < 128 && isspace(c))
	  *t2++ = c;
	else if (wd >= 1) {
	  *t2++ = ' ';
	  xw += wd-1;
	}
	s1 = t1;
      }
    }
    *t2++ = '^';
    *t2++ = 0;
    free(s); s = s2;
  }
#else
  for (t = s; *t; t++)
    if (!isspace(*t))
      *t = ' ';
  s[p-q] = '^';
#endif
  printf(">>> %s\n", utf8_to_sys(q));
  for (i = 0; i < xw; i++) putc(' ', stdout);
  printf("    %s\n", s);
  if (r) *r = c;
  free(s);
}

DECLARE_YYTEXT

static char *tok = NULL, *p = NULL;;
static int toksz = 0;

extern char *my_yytext;

static savepos()
{
  THREAD *thr = get_thr();
  char *text = (*yytext||!my_yytext)?yytext:my_yytext;
  int l = strlen(text);
  p = actchar();
  if (thr->qmstat == BAD_ESC && actbuf())
    tokoffs = strlen(actbuf())+1;
  else
    tokoffs = 0;
  if (l < 1000) l = 1000;
  if (!tok)
    tok = malloc((toksz = l+1)*sizeof(char));
  else if (toksz < l+1)
    tok = realloc(tok, (toksz = l+1)*sizeof(char));
  if (!tok)
    fatal("memory overflow");
  strcpy(tok, text);
}

extern int actlineno;

static clean()
{
  THREAD *thr = get_thr();
  int res = !(thr->qmstat != OK && thr->qmstat != HALT && thr->qmstat != QUIT);
  if (ferror(stdin))
    clearerr(stdin);
  flush_shift();
  if (!res) {
    if (pmode == SOURCE) {
      char msg[MAXSTRLEN];
      sprintf(msg, "File %s, line %d: %s", psource, actlineno,
	      qmmsg[thr->qmstat]);
      error(msg);
    } else
      error(qmmsg[thr->qmstat]);
    if (thr->qmstat == XCEPT && thr->xsp > thr->xst) {
      printx(thr->xsp[-1]); flush_shift(); printf("\n");
    }
    if ((pmode == INTERACT || pmode == SOURCE) && base && p)
      error_report(base, p, tok);
  }
  if (!quitflag && thr->qmstat != QUIT)
    clear(0);
  thr->debug = debug;
  thr->mode = 1;
  return res && !checkbrk;
}

static expression()
{
#ifdef USE_THREADS
  pthread_mutex_unlock(&parse_mutex);
#endif
  if (!checkbrk && init_stats() && eval(thr, thr->xsp[-1]) && fini_stats() &&
      !checkbrk) {
    flush_shift();
    printx(thr->xsp[-1]);
    flush_shift();
    putchar('\n');
    if (thr->xsp[-1]->fno != DEFVAROP)
      setvar(DEFVAROP, thr->xsp[-1]);
  } else if (!checkbrk)
    fini_stats();
#ifdef USE_THREADS
  pthread_mutex_lock(&parse_mutex);
#endif
  savepos();
  return clean();
}

static assignment()
{
#ifdef USE_THREADS
  pthread_mutex_unlock(&parse_mutex);
#endif
  if (!checkbrk && init_stats() && eval(thr, thr->xsp[-1])) {
    thr->nredns++;
    thr->args[0] = thr->xsp[-3];
    thr->args[1] = thr->xsp[-1];
    (*funtb[DEFOP])(thr);
    fini_stats();
  }
#ifdef USE_THREADS
  pthread_mutex_lock(&parse_mutex);
#endif
  savepos();
  return clean();
}

static unassignment(int id)
{
  if (symtb[id].x)
    setvar(id, NULL);
  savepos();
  return clean();
}

static int declid = NONE;
static vardecl(int id)
{
  if (id != NONE) {
    if (id >= symtbsz)
      symtb[id].flags |= (VSYM|DCL);
    else if (symtb[id].modno == mainno) {
      if (!(symtb[id].flags&VSYM)) {
	thr->qmstat = BAD_DEF;
	id = NONE;
      }
    } else {
      symtb[id].flags &= ~VIS;
      id = newsym(strsp+symtb[id].pname);
      if (id != NONE) symtb[id].flags |= (VSYM|DCL);
    }
  }
  declid = id;
  savepos();
  return clean();
}

static initializer()
{
  if (declid != NONE) {
#ifdef USE_THREADS
    pthread_mutex_unlock(&parse_mutex);
#endif
    if (!checkbrk && init_stats() && eval(thr, thr->xsp[-1])) {
      setvar(declid, thr->xsp[-1]);
      fini_stats();
    }
#ifdef USE_THREADS
    pthread_mutex_lock(&parse_mutex);
#endif
  }
  savepos();
  return clean();
}

/* built-in commands: */

static printstr(s)
     char *s;
{
  printf("%s\n", s);
}

static strparam(a, s)
     char *a, **s;
{
  *s = NULL;
  if (!checkbrk && !(*s = utf8_to_sys_dup(a)))
    thr->qmstat = MEM_OVF;
  savepos();
  if (clean())
    return *s != NULL;
  else {
    if (*s) free(*s);
    *s = NULL;
    return 0;
  }
}

static intparam(a, i)
     char *a; long *i;
{
  int res = !checkbrk;
  if (res) {
    char *s = a;
    while (isdigit(*s)) s++;
    if (!*s) {
      res = 1;
      *i = atol(a);
    } else
      thr->qmstat = SYNTAX_ERR;
  }
  savepos();
  return clean() && res;
}

static boolparam(a, b)
     char *a; int *b;
{
  int res = !checkbrk;
  if (res)
    if (!strcmp(a, "on"))
      *b = 1;
    else if (!strcmp(a, "off"))
      *b = 0;
    else {
      res = 0;
      thr->qmstat = SYNTAX_ERR;
    }
  savepos();
  return clean() && res;
}

static boolstrparam(a, b, s)
     char *a; int *b; char **s;
{
  int res = !checkbrk;
  *s = NULL;
  if (res)
    if (!strcmp(a, "on"))
      *b = 1;
    else if (!strcmp(a, "off"))
      *b = 0;
    else
      return strparam(a, s);
  savepos();
  return clean() && res;
}

static allparam(a)
     char *a;
{
  int res = !checkbrk;
  if (res)
    if (strcmp(a, "all")) {
      res = 0;
      thr->qmstat = SYNTAX_ERR;
    }
  savepos();
  return clean() && res;
}

static copyright()
{
  if (!checkbrk) {
    printf("\n");
    printf(signon, version, sysinfo, year);
    printf(copying);
  }
  savepos();
  return clean();
}

static shell()
{
  if (!checkbrk) {
    char shell_command[MAXSTRLEN];
    getln(shell_command);
    push_sigint(SIG_DFL);
    system(shell_command);
    pop_sigint();
  }
  savepos();
  return clean();
}

#ifdef USE_READLINE
extern int use_readline;
#else
static histwarn()
{
  static done = 0;
  if (done)
    return;
  else if (gflag || !iflag || !isatty(fileno(stdin)))
    done = 1;
  else {
    error("Warning: history not supported");
    done = 1;
  }
}
#endif

static sethistfile(s)
     char *s;
{
#ifdef USE_READLINE
  if (!gflag && use_readline && histmax > 0) {
    char h[MAXSTRLEN];
    write_history(expand(h, histfile));
  }
#else
  histwarn();
#endif
  if (histfile) free(histfile);
  histfile = s;
#ifdef USE_READLINE
  if (!gflag && use_readline) {
    char h[MAXSTRLEN];
    clear_history();
    if (histmax > 0)
      read_history(expand(h, histfile));
    stifle_history(histmax);
  }
#endif
}

static sethistsize(i)
     int i;
{
  histmax = i>=0?i:0;
#ifdef USE_READLINE
  if (!gflag && use_readline)
    stifle_history(histmax);
#else
  histwarn();
#endif
}

static help(s)
     char *s;
{
  static char *info = NULL;
  char *cmd;
  if (!info && !(info = getenv("INFO_PROGRAM")))
    info = "info";
  if ((cmd = malloc(((s?strlen(s):0)*4+3+
		     strlen(info)+MAXSTRLEN)*sizeof(char)))) {
    if (s) {
      char *buf = malloc(4*strlen(s)+1);
      if (buf) {
	sprintf(cmd, "%s -f qdoc --index-search \"%s\"", info, pstr(buf, s));
	free(buf);
      } else {
	free(cmd);
	thr->qmstat = MEM_OVF;
	return clean();
      }
    } else
      sprintf(cmd, "%s -f qdoc", info);
    push_sigint(SIG_DFL);
    system(cmd);
    pop_sigint();
    free(cmd);
  } else
    thr->qmstat = MEM_OVF;
  return clean();
}

static edit(s)
     char *s;
{
  static char *editor = NULL;
  char *cmd;
  if (!editor && !(editor = getenv("EDITOR")))
    editor = "vi";
  if ((cmd = malloc((4*strlen(s)+strlen(editor)+10)*sizeof(char)))) {
    if (s && *s) {
      char *buf = malloc(4*strlen(s)+1);
      if (buf) {
	sprintf(cmd, "%s \"%s\"", editor, pstr(buf, s));
	free(buf);
      } else {
	free(cmd);
	thr->qmstat = MEM_OVF;
	return clean();
      }
    } else
      strcpy(cmd, editor);
    push_sigint(SIG_DFL);
    system(cmd);
    pop_sigint();
    free(cmd);
  } else
    thr->qmstat = MEM_OVF;
  return clean();
}

static save_vars(s)
     char *s;
{
  FILE	       *fp;
  int		res = 0, i;
  char         *s_bak = save_file;

  if (!s) {
    if (!save_file) save_file = strdup(".q_vars");
    if (!save_file) {
      error(qmmsg[MEM_OVF]);
      return 0;
    }
    s = save_file;
  } else
    save_file = s;
  if ((fp = fopen(s, "w"))) {
#if defined(HAVE_UNICODE) && defined(HAVE_ICONV)
    iconv_t ic[2] = { (iconv_t)-2, (iconv_t)-2 };
#endif
    /* store variable definitions: */
    for (i = symtbsz; i<symtbsz+tmptbsz; i++)
      if (symtb[i].x && i != DEFVAROP) {
	char str[MAXSTRLEN];
	fprintf(fp, "var %s = ", utf8_to_sys(pname(str, i)));
#if defined(HAVE_UNICODE) && defined(HAVE_ICONV)
	fprintx(fp, ic, symtb[i].x);
	/* flush out pending shift state */
	if (ic[1] != (iconv_t)-2 &&
	    ic[1] != (iconv_t)-1) {
	  char *s = icfromutf8(ic, NULL), *t = s;
	  if (t) {
	    while (*s) putc(*s++, fp);
	    free(t);
	  }
	}
#else
	fprintx(fp, symtb[i].x);
#endif
	fprintf(fp, ";\n");
      }
    res = !ferror(fp);
#if defined(HAVE_UNICODE) && defined(HAVE_ICONV)
    if (ic[1] != (iconv_t)-2 &&
	ic[1] != (iconv_t)-1)
      iconv_close(ic[1]);
#endif
    fclose(fp);
  }
  if (!res) {
    char msg[MAXSTRLEN];
    sprintf(msg, qmmsg[FILE_ERR], s);
    error(msg);
    if (save_file && s_bak != save_file) free(save_file);
    save_file = s_bak;
  } else if (s_bak && s_bak != save_file)
    free(s_bak);
  return res;
}

static clear_vars()
{
  int		i;

  for (i = symtbsz; i<symtbsz+tmptbsz; i++)
    if (symtb[i].x)
      setvar(i, NULL);
  clear(0);
}

int aximpsz = 0, ximpsz = 0;
char **ximp = NULL;

static
search_import(m)
     char *m;
{
  int i;
  char modname[MAXSTRLEN];
  for (i = 0; i < ximpsz; i++) {
    basename(modname, ximp[i], '.');
    if (!strcmp(modname, m))
      return i;
  }
  return NONE;
}

static
add_import(s)
     char *s;
{
  int mno;
  char modname[MAXSTRLEN], fname[MAXSTRLEN], fname2[MAXSTRLEN],
    aname[MAXSTRLEN];
  if (!s) return 0;
  basename(modname, sys_to_utf8(s), '.');
  if ((mno = getmodno(modname)) != NONE && 
      (mno == 0 || mno == mainno || globs[mno])) {
    printf("module %s already imported\n", utf8_to_sys(modname));
    free(s);
    return 0;
  }
  if (search_import(modname) != NONE) {
    printf("module %s already imported\n", utf8_to_sys(modname));
    free(s);
    return 0;
  }
  absname(aname, searchlib(fname, s));
  if (!chkfile(aname)) {
    strcat(strcpy(fname2, s), ".q");
    absname(aname, searchlib(fname, fname2));
  }
  if (!chkfile(aname)) {
    printf("file %s not found\n", s);
    free(s);
    return 0;
  }
  if (ximpsz >= aximpsz) {
    char **ximp1;
    if (!(ximp1 = arealloc(ximp, aximpsz, 10, sizeof(char*)))) {
      printf("error processing %s -- out of memory\n", s);
      free(s);
      return 0;
    } else {
      ximp = ximp1;
      aximpsz += 10;
    }
  }
  ximp[ximpsz++] = s;
  return 1;
}

extern char *source;

static
del_import(s)
     char *s;
{
  char modname[MAXSTRLEN], fname[MAXSTRLEN], fname2[MAXSTRLEN],
    aname[MAXSTRLEN];
  int i, mno, pos;
  if (!s) return 0;
  basename(modname, sys_to_utf8(s), '.');
  absname(aname, searchlib(fname, s));
  if (!chkfile(aname)) {
    strcat(strcpy(fname2, s), ".q");
    absname(aname, searchlib(fname, fname2));
  }
  pos = search_import(modname);
  mno = getmodno(modname);
  if (mno != NONE && mno != mainno && mno != 0 && !globs[mno])
    mno = NONE;
  if (mno != NONE && strcmp(aname, strsp+fnametb[mno])) {
    printf("module %s not imported\n", utf8_to_sys(modname));
    return 0;
  }
  if (mno == NONE || pos == NONE && mno != mainno) {
    if (mno != NONE)
      printf("module %s cannot be unloaded\n", utf8_to_sys(modname));
    else
      printf("module %s not imported\n", utf8_to_sys(modname));
    return 0;
  }
  if (mno == mainno) {
    if (source) free(source);
    source = NULL;
  }
  if (pos != NONE) {
    free(ximp[pos]);
    for (i = pos+1; i < ximpsz; i++)
      ximp[i-1] = ximp[i];
    ximpsz--;
  }
  return 1;
}

static
clear_imports()
{
  if (ximpsz) {
    int i;
    for (i = 0; i < ximpsz; i++)
      free(ximp[i]);
    ximpsz = 0;
    return 1;
  } else
    return 0;
}

static
modcmp(v1, v2)
	int	       *v1, *v2;
{
	return strcmp(strsp + modtb[*v1], strsp + modtb[*v2]);
}

static
print_modlist()
{
  int i, n, *v, *v0;
  if (modtbsz <= 0)
    return;
  else if (!(v = (int *)calloc(modtbsz, sizeof(int)))) {
    thr->qmstat = MEM_OVF;
    return;
  }
  for (i = 0; i < modtbsz; i++) v[i] = i;
  qsort(v, modtbsz, sizeof(int), (int(*)())modcmp);
  v0 = v; n = modtbsz;
  if (!strsp[modtb[*v]]) {
    v++; n--;
  }
  if (iflag) {
    for (i = 0; i < n; i++) {
      if (i > 0)
	if (i % 4 == 0)
	  printf("\n");
	else
	  printf("\t");
      if (dll_name[v[i]]) {
	char modn[MAXSTRLEN];
	strcat(strcpy(modn, strsp+modtb[v[i]]), "*");
	printf("%-15s", utf8_to_sys(modn));
      } else
	printf("%-15s", utf8_to_sys(strsp+modtb[v[i]]));
    }
    printf("\n");
  } else {
    for (i = 0; i < n; i++)
      printf("%s%s\n", utf8_to_sys(strsp+modtb[v[i]]),
	     dll_name[v[i]]?"*":" ");
  }
  free(v0);
}

static modules()
{
  if (!checkbrk) {
    print_modlist();
  }
  savepos();
  return clean();
}

static
print_implist()
{
  int i, n = 0, *v;
  for (i = 0; i < modtbsz; i++) 
    if (strsp[modtb[i]])
      if (i == 0 || i == mainno || globs[i])
	n++;
  if (n <= 0)
    return;
  else if (!(v = (int *)calloc(n, sizeof(int)))) {
    thr->qmstat = MEM_OVF;
    return;
  }
  n = 0;
  for (i = 0; i < modtbsz; i++) 
    if (strsp[modtb[i]])
      if (i == 0 || i == mainno || globs[i])
	v[n++] = i;
  qsort(v, n, sizeof(int), (int(*)())modcmp);
  if (iflag) {
    for (i = 0; i < n; i++) {
      char modn[MAXSTRLEN];
      if (i > 0)
	if (i % 4 == 0)
	  printf("\n");
	else
	  printf("\t");
      sprintf(modn, "%s%s%s",
	      (v[i] == mainno||search_import(strsp+modtb[v[i]])!=NONE)?"+":"",
	      strsp+modtb[v[i]], dll_name[v[i]]?"*":" ");
      printf("%-15s", utf8_to_sys(modn));
    }
    printf("\n");
  } else {
    for (i = 0; i < n; i++)
      printf("%s%s%s\n",
	     (v[i] == mainno||search_import(strsp+modtb[v[i]])!=NONE)?"+":"",
	     utf8_to_sys(strsp+modtb[v[i]]), dll_name[v[i]]?"*":" ");
  }
  free(v);
}

static import_list()
{
  if (!checkbrk) {
    print_implist();
  }
  savepos();
  return clean();
}

static
idcmp(v1, v2)
     int	       *v1, *v2;
{
  int cmp = strcmp(strsp + symtb[*v1].pname, strsp + symtb[*v2].pname);
  if (cmp)
    return cmp;
  else
    return strcmp(strsp + modtb[symtb[*v1].modno],
		  strsp + modtb[symtb[*v2].modno]);
}

static
print_varlist(int all)
{
  int	       *v, n, i, sz = all?BINARY:symtbsz;
  char		s[MAXSTRLEN];
  for (n = 0, i = sz; i < symtbsz+tmptbsz; i++)
    if (visible(i) && (i < symtbsz || (symtb[i].flags&DCL) || symtb[i].x) &&
	(symtb[i].flags&VSYM))
      n++;
  if (n == 0) {
    /* 		printf("*** no defined symbols ***\n"); */
    return;
  } else if (!(v = (int *)calloc(n, sizeof(int)))) {
    thr->qmstat = MEM_OVF;
    return;
  }
  for (n = 0, i = sz; i < symtbsz+tmptbsz; i++)
    if (visible(i) && (i < symtbsz || (symtb[i].flags&DCL) || symtb[i].x) &&
	(symtb[i].flags&VSYM))
      v[n++] = i;
  qsort(v, n, sizeof(int), (int(*)())idcmp);
  if (iflag) {
    for (i = 0; i < n; i++) {
      if (i > 0)
	if (i % 4 == 0)
	  printf("\n");
	else
	  printf("\t");
      printf("%-15s", utf8_to_sys(pname(s, v[i])));
    }
    printf("\n");
  } else {
    for (i = 0; i < n; i++)
      printf("%s\n", utf8_to_sys(pname(s, v[i])));
  }
  free(v);
}

print_breakpoints()
{
  int	       *v, n, i;
  char		s[MAXSTRLEN];
  printf("break is %s", brkdbg?"on":"off");
  for (n = 0, i = BINARY; i < symtbsz+tmptbsz; i++)
    if (visible(i) && (symtb[i].flags&(BRK|TBRK)))
      n++;
  if (n == 0) {
    /* 		printf(", no breakpoints\n"); */
    printf("\n");
    return;
  } else if (!(v = (int *)calloc(n, sizeof(int)))) {
    thr->qmstat = MEM_OVF;
    return;
  }
  for (n = 0, i = BINARY; i < symtbsz+tmptbsz; i++)
    if (visible(i) && (symtb[i].flags&(BRK|TBRK)))
      v[n++] = i;
  qsort(v, n, sizeof(int), (int(*)())idcmp);
  printf(", %d breakpoint%s\n", n, (n>1)?"s":"");
  if (iflag) {
    for (i = 0; i < n; i++) {
      if (i > 0)
	if (i % 4 == 0)
	  printf("\n");
	else
	  printf("\t");
      pname(s, v[i]);
      if (symtb[v[i]].flags & TBRK)
	strcat(s, "*");
      printf("%-15s", utf8_to_sys(s));
    }
    printf("\n");
  } else {
    for (i = 0; i < n; i++)
      printf("%s\n", utf8_to_sys(pname(s, v[i])));
  }
  free(v);
}

static breakpoints()
{
  if (!checkbrk) print_breakpoints();
  savepos();
  return clean();
}

static
profcmp(v1, v2)
     int	       *v1, *v2;
{
  int cmp = symtb[*v2].nredns - symtb[*v1].nredns;
  if (cmp)
    return cmp;
  else {
    cmp = strcmp(strsp + symtb[*v1].pname, strsp + symtb[*v2].pname);
    if (cmp)
      return cmp;
    else
      return strcmp(strsp + modtb[symtb[*v1].modno],
		    strsp + modtb[symtb[*v2].modno]);
  }
}

print_profile()
{
  int	       *v, n, i;
  char		s[MAXSTRLEN];
  for (n = 0, i = BINARY; i < symtbsz+tmptbsz; i++)
    if (visible(i) && (symtb[i].flags&PROF))
      n++;
  if (n == 0)
    return;
  else if (!(v = (int *)calloc(n, sizeof(int)))) {
    thr->qmstat = MEM_OVF;
    return;
  }
  for (n = 0, i = BINARY; i < symtbsz+tmptbsz; i++)
    if (visible(i) && (symtb[i].flags&PROF))
      v[n++] = i;
  qsort(v, n, sizeof(int), (int(*)())profcmp);
  for (i = 0; i < n; i++) {
    printf("%8d\t%s\n", symtb[v[i]].nredns,
	   utf8_to_sys(pname(s, v[i])));
    symtb[v[i]].nredns = 0;
  }
  free(v);
}

static profile()
{
  if (!checkbrk) print_profile();
  savepos();
  return clean();
}

static who(int all)
{
  if (!checkbrk) print_varlist(all);
  savepos();
  return clean();
}

static whos(int sym)
{
  if (sym >= BINARY) {
    int mno = symtb[sym].modno,
      lineno = symtb[sym].lineno,
      flags = symtb[sym].flags,
      argc = symtb[sym].argc,
      type = symtb[sym].type,
      prec = symtb[sym].prec;
    unsigned long argv = symtb[sym].argv;
    EXPR *x = (EXPR*)symtb[sym].x;
    char s[MAXSTRLEN], ts[MAXSTRLEN];
    
    pname(s, sym);
    if (type) pname(ts, type);
    if (mno == NONE || sym < BUILTIN || sym >= symtbsz || !(flags & DCL)) {
      if ((flags & VSYM) && !x && !(flags & DCL) ||
	  !(flags & VSYM) && !(flags & DCL))
	printf("%-15s\tnot a defined symbol", utf8_to_sys(s));
      else
	printf("%-15s\t%s %s%s%s%s symbol", utf8_to_sys(s),
	       (sym<BUILTIN)?"builtin":"user-defined",
	       argv?"special ":"",
	       (flags&CST)?"const ":"",
	       (flags&VIRT)?"virtual ":"",
	       (flags&VSYM)?"variable":"function");
      if ((flags&EXT) && symtb[sym].xfno!=sym)
	printf("\n%-15s\texternal override in %s, line %d", " ",
	       strsp+fnametb[symtb[symtb[sym].xfno].modno],
	       symtb[symtb[sym].xfno].lineno);
    } else {
      printf("%-15s\t%s%s%s%s%s symbol defined in %s, line %d", utf8_to_sys(s),
	     (flags&PRIV)?"private ":"",
	     argv?"special ":"",
	     (flags&CST)?"const ":"",
	     (flags&VIRT)?"virtual ":"",
	     (flags&VSYM)?"variable":
	     ((flags&EXT)&&symtb[sym].xfno==sym)?"external function":
	     "function",
	     strsp+fnametb[mno], lineno);
      if ((flags&EXT) && symtb[sym].xfno!=sym)
	printf("\n%-15s\texternal override in %s, line %d", " ",
	       strsp+fnametb[symtb[symtb[sym].xfno].modno],
	       symtb[symtb[sym].xfno].lineno);
    }
    if (type)
      printf("\n%-15s\t%s %s", " ", utf8_to_sys(ts), utf8_to_sys(s));
    else if (argc > 0)
      printf("\n%-15s\t%s", " ", utf8_to_sys(s));
    if (argc > 0) {
      int i;
      for (i = 0; i < argc; i++) {
	printf(" ");
	if (argv && !((argv>>i)&1)) printf("~");
	printf("X%d", i+1);
      }
    }
    if (prec != NONE)
      printf(" @%d", prec);
    if (type) {
      int tmno = symtb[type].modno,
	tlineno = symtb[type].lineno;
      if (tmno == NONE || type < BUILTIN)
	printf("\n%-15s\tbuiltin type %s", " ", utf8_to_sys(ts));
      else
	printf("\n%-15s\t%s%stype %s defined in %s, line %d", " ",
	       (symtb[type].flags&PRIV)?"private ":"",
	       (symtb[type].flags&EXT)?"external ":"", ts,
	       strsp+fnametb[tmno], tlineno);
    } else if (x) {
      printf("\n%-15s\t= ", " ");
      printx(x); flush_shift();
    }
    printf("\n");
  } else
    printf("unknown or bad symbol\n");
  savepos();
  return clean();
}

#define matchsym(fno,s) (!(symtb[fno].flags & TSYM)&&\
			 strcmp((s), strsp+symtb[fno].pname)==0)

static int strhash(char *s, int sz)
{
  unsigned h = 0, g;
  while (*s) {
    h = (h<<4)+*(s++);
    if ((g = (h & 0xf0000000)))	{
      h = h^(g>>24);
      h = h^g;
    }
  }
  return h % sz;
}

static whois(int fno, char *s)
{
  if (strstr(s, "::")) {
    if (fno != NONE) {
      printf("\n(%s)\n", s);
      return whos(fno);
    } else
      return 0;
  } else {
    int fno, k = strhash(s, hashtbsz);
    int modno = (mainno>=0)?mainno:0;
    for (fno = hashtb[k]; fno != NONE; fno = symtb[fno].next)
      if (matchsym(fno, s) &&
	  (symtb[fno].modno == NONE || symtb[fno].modno == modno ||
	   globs[symtb[fno].modno]))
	if (symtb[fno].modno == modno || !(symtb[fno].flags & PRIV)) {
	  char name[MAXSTRLEN];
	  int sym = fno;
	  while (symtb[sym].ref)
	    sym = symtb[sym].ref;
	  if (sym < symtbsz || (symtb[sym].flags & DCL)) {
	    if (symtb[fno].modno >= 0)
	      sprintf(name, "%s::%s", strsp + modtb[symtb[fno].modno],
		      strsp + symtb[fno].pname);
	    else
	      sprintf(name, "::%s", strsp + symtb[fno].pname);
	    printf("\n(%s)\n", name);
	  } else
	    printf("\n");
	  whos(sym);
	}
    return 1;
  }
}

static ambiguous(int fno, char *s)
{
  printf("! whos: Ambiguous symbol `%s'. Alternatives are:\n", s);
  whois(fno, s);
}

/* make the parser reentrant */

static struct {
  int yychar, yynerrs;
  YYSTYPE yylval;
  THREAD *thr;
} yystack[MAXSTACK], *yystptr = NULL;

static yypush()
{
  if (yystptr) {
    if (yystptr - yystack >= MAXSTACK)
      fatal("source stack overflow");
    yystptr->yychar = yychar;
    yystptr->yynerrs = yynerrs;
    memcpy(&yystptr->yylval, &yylval, sizeof(YYSTYPE));
    yystptr->thr = thr;
    yystptr++;
  } else
    yystptr = yystack;
}

static yypop()
{
  if (yystptr > yystack) {
    --yystptr;
    yychar = yystptr->yychar;
    yynerrs = yystptr->yynerrs;
    memcpy(&yylval, &yystptr->yylval, sizeof(YYSTYPE));
    thr = yystptr->thr;
  } else
    yystptr = NULL;
}

/* generic parse routine: */

static parse(source, arg, _mode)
     void	       *source, *arg;
     int		_mode;
{
  int c, actmode;
  long xbp;
  THREAD *_thr = get_thr();
  actmode = _thr->mode;
  xbp = _thr->xsp-_thr->xst;
  if (!initlex(source, arg, _mode)) {
    _thr->qmstat = STACK_OVF;
    return 0;
  };
  yypush();
  thr = _thr;
  thr->mode = 1;
  if (_mode == INTERACT || _mode == SOURCE)
    base = (char*)source;
  else
    base = NULL;	
  if (yyparse()) {
    if (_mode == INTERACT || _mode == SOURCE) {
      thr->debug = debug;
      savepos();
    }
    thr->mode = actmode;
    if (_mode == LINE)
      /* skip remainder of input line */
      if (yychar && yychar != 10 && (thr > thr0 || !checkbrk)) skip();
    finilex();
    /* clean up the stack: */
    while (thr->xsp-thr->xst > xbp)
      qmfree(thr, *--thr->xsp);
    yypop();
    return (0);
  } else {
    if (_mode == INTERACT || _mode == SOURCE) {
      thr->debug = debug;
      savepos();
    }
    thr->mode = actmode;
    finilex();
    yypop();
    return (1);
  }
}

/* interface routines: */

int sparsex(char *s)
{
  THREAD *thr = get_thr();
  int ret;
#ifdef USE_THREADS
  pthread_mutex_lock(&parse_mutex);
#endif
  ret = parse(s, NULL, STRING);
  if (!ret && thr->qmstat == BAD_OBJ) thr->qmstat = SYNTAX_ERR;
#ifdef USE_THREADS
  pthread_mutex_unlock(&parse_mutex);
#endif
  return ret;
}

#if defined(HAVE_UNICODE) && defined(HAVE_ICONV)
int fparsex(FILE *fp, iconv_t *ic)
#else
int fparsex(FILE *fp)
#endif
{
  THREAD *thr = get_thr();
  int ret;
#ifdef USE_THREADS
  pthread_mutex_lock(&parse_mutex);
#endif
#if defined(HAVE_UNICODE) && defined(HAVE_ICONV)
  ret = parse(fp, ic, LINE);
#else
  ret = parse(fp, NULL, LINE);
#endif
  if (!ret && thr->qmstat == BAD_OBJ) thr->qmstat = SYNTAX_ERR;
#ifdef USE_THREADS
  pthread_mutex_unlock(&parse_mutex);
#endif
  if (ret)
    if (thr == thr0 && checkbrk) {
      qmfree(thr, *--thr->xsp);
      return 0;
    } else
      return 1;
  else
    return 0;
}

int parsesrc(char *fname, int chk)
{
  char *buf;
  FILE *fp;
  THREAD *thr = get_thr();
  if (!chkfile(fname) || !(fp = fopen(fname, "r"))) {
    char msg[MAXSTRLEN];
    sprintf(msg, qmmsg[FILE_NOT_FOUND], fname);
    error(msg);
    return 0;
  }
#ifdef USE_THREADS
  pthread_mutex_lock(&parse_mutex);
#endif
  psource = fname;
  actlineno = 0;
  while (!quitflag && thr->qmstat != QUIT && (buf = mygetline(fp, "", 0))) {
    if (eflag) echo(utf8_to_sys(buf));
    if (!parse(buf, NULL, SOURCE)) {
      if (thr->qmstat == BAD_OBJ) thr->qmstat = chk?SYNTAX_ERR:OK;
      clean();
      free(buf);
      fclose(fp);
#ifdef USE_THREADS
      pthread_mutex_unlock(&parse_mutex);
#endif
      return 0;
    }
    free(buf);
    fflush(stderr);
    fflush(stdout);
  }
  fclose(fp);
#ifdef USE_THREADS
  pthread_mutex_unlock(&parse_mutex);
#endif
  return 1;
}

int parsex(char *s)
{
  THREAD *thr = get_thr();
  int ret;
#ifdef USE_THREADS
  pthread_mutex_lock(&parse_mutex);
#endif
  ret = parse(s, NULL, INTERACT);
  if (!ret && thr->qmstat == BAD_OBJ) thr->qmstat = SYNTAX_ERR;
#ifdef USE_THREADS
  pthread_mutex_unlock(&parse_mutex);
#endif
  if (!ret)
    return clean();
  else
    return 1;
}
