/**
 * $Id: mod_estraier.c,v 1.60 2006/04/22 19:22:55 shinh Exp $
 *
 * Copyright (C) shinichiro.h <hamaji@nii.ac.jp>
 *  http://shinh.skr.jp/
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */

/**
 * The present copyright holders of this program have given permission,
 * as a special exception, to link this program with Apache Portable
 * Runtime (APR) and to include header files for APR components when
 * those header files are covered by the Apache licenses, as long as
 * the GNU GPL is followed for this program in all other ways. 
 */

#include "estutil.h"
#include "inflate.h"

#include <estraier.h>
#include <estnode.h>
#include <cabin.h>

#include <stdlib.h>
#include <math.h>
#include <assert.h>

/* apache */
#include <http_protocol.h>
#include <http_config.h>
#include <http_log.h>
#include <apr_strings.h>
#include <pcreposix.h>
#include <util_md5.h>

#define DATTRVIEWCNT  "viewcount"

module AP_MODULE_DECLARE_DATA estraier_module;

typedef struct {
  const char *node;
  const char *user;
  const char *pass;

  const char *proxy_host;
  int proxy_port;
  int timeout;

  apr_array_header_t *filter_uris;
  int language;
  int detach_thread;
  apr_array_header_t *filter_req_headers, *filter_res_headers;
  int use_weight;

  apr_array_header_t *filter_cmds;
  const char *filter_tmp_dir;

  int document_size_limit;
} estraier_dir_config;

typedef struct {
  char *body;
  int body_len;
  int error;
} estraier_ctx;

typedef struct {
  server_rec *serv;
  estraier_dir_config *conf;
  char *body;
  int body_len;
  apr_pool_t *pool;
  const char *mdate;
  const char *request_time;
  const char *content_type;
  const char *content_language;
  const char *etag;
  const char *uri;
  int lang;
  int is_enc;
  const char *charset;
} estraier_commit_ctx;

/* for apache 2.1.3 or later. */
#if AP_SERVER_MINORVERSION_NUMBER > 0
# define regex_t ap_regex_t
#endif

typedef struct {
  regex_t *reg;
  int is_deny;
} filter_uri;

typedef struct {
  regex_t *header_reg;
  regex_t *value_reg;
} filter_header;

typedef struct {
  regex_t *content_type_reg;
  const char *xcmd;
} filter_cmd;

enum {
  FF_NONE = 0,
  FF_TEXT,
  FF_HTML,
  FF_XCMD_START,
};

enum {
  STATE_UNKNOWN = 0,
  STATE_SUCCESS,
  STATE_ERROR,
};

enum {
  LANGUNKNOWN = ESTLANGEN - 1,
};

#define PTR_OR(a, b) (((a) != NULL) ? (a) : (b))
#define INT_OR(a, b) (((a) != -1) ? (a) : (b))

/* for debugging */
static void printfinfo(const char *format, ...) {
  FILE* fp = fopen("/tmp/log", "a");
  va_list ap;
  va_start(ap, format);
  fprintf(fp, "%s: INFO: ", "mod_estraier");
  vfprintf(fp, format, ap);
  fputc('\n', fp);
  fflush(fp);
  va_end(ap);
  fclose(fp);
}

static apr_status_t delete_estraier_node(void *vp) {
  ESTNODE *node = (ESTNODE *)vp;
  est_node_delete(node);
  return APR_SUCCESS;
}

static void register_node_cleanup(apr_pool_t *p, ESTNODE *node) {
  apr_pool_cleanup_register(p, node,
                            delete_estraier_node, apr_pool_cleanup_null);
}

static apr_status_t delete_estraier_noderes(void *vp) {
  ESTNODERES *nres = (ESTNODERES *)vp;
  est_noderes_delete(nres);
  return APR_SUCCESS;
}

static void register_noderes_cleanup(apr_pool_t *p, ESTNODERES *nres) {
  apr_pool_cleanup_register(p, nres,
                            delete_estraier_noderes, apr_pool_cleanup_null);
}

static apr_status_t free_estraier_string(void *vp) {
  char *cp = (char *)vp;
  free(cp);
  return APR_SUCCESS;
}

static int get_estraier_language(const char *lang) {
  if (strcmp(lang, "en") == 0) {
    return ESTLANGEN;
  }
  else if (strcmp(lang, "ja") == 0) {
    return ESTLANGJA;
  }
  else if (strcmp(lang, "zh") == 0) {
    return ESTLANGZH;
  }
  else if (strcmp(lang, "ko") == 0) {
    return ESTLANGKO;
  }
  else if (strcmp(lang, "misc") == 0) {
    return ESTLANGMISC;
  }
  else {
    return LANGUNKNOWN;
  }
}

static apr_status_t init_estraier_instance(ap_filter_t *f) {
  estraier_ctx *ctx;

  if (f->ctx) return APR_SUCCESS;

  f->ctx = ctx = (estraier_ctx *)apr_pcalloc(f->r->pool, sizeof(estraier_ctx));

  ctx->body = NULL;
  ctx->body_len = 0;
  ctx->error = STATE_UNKNOWN;

  return APR_SUCCESS;
}

static const char *get_charset(ap_filter_t *f) {
  const char *type;
  const char *charset;
  const char *p;
  type = f->r->content_type;

  if (!type) return NULL;

  if (!(charset = strstr(type, "charset=")) &&
      !(charset = strstr(type, "Charset=")) &&
      !(charset = strstr(type, "CHARSET=")))
  {
    return NULL;
  }
  charset += 8;
  p = strchr(charset, ';');
  if (p) {
    return apr_pstrndup(f->r->pool, charset, p-charset);
  }
  return charset;
}

static double calc_scoreweight(int viewcnt) {
  double weight = log(10+viewcnt);
  weight *= weight;
  return weight;
}

static int get_viewcnt(ESTRESDOC *doc) {
  const char *cnt;
  int c;

  if (doc) {
    cnt = est_resdoc_attr(doc, DATTRVIEWCNT);
    if (cnt) {
      c = atoi(cnt)+1;
      return c;
    }
  }

  return 1;
}

static void add_scoreweight(ESTDOC *doc,
                            estraier_dir_config *conf, ESTRESDOC *odoc)
{
  int viewcnt = 1;
  double weight = 1.0;
  char buf[256];

  if (INT_OR(conf->use_weight, 0)) {
    viewcnt = get_viewcnt(odoc);
    weight = calc_scoreweight(viewcnt);
    sprintf(buf, "%d", viewcnt);
    est_doc_add_attr(doc, DATTRVIEWCNT, buf);
  }

  sprintf(buf, "%f", weight);
  est_doc_add_attr(doc, ESTDATTRWEIGHT, buf);
}

static char *get_full_uri(ap_filter_t *f, apr_pool_t *pool) {
  char *uri;

  uri = f->r->uri;

  /* add hostname */
  if (!strstr(uri, "://")) {
    uri = apr_pstrcat(pool, "http://", f->r->hostname, uri, NULL);
  }

  /* add query string */
  if (f->r->args && !strstr(uri, f->r->args)) {
    uri = apr_pstrcat(pool, uri, "?", f->r->args, NULL);
  }

  return uri;
}

static int check_content_type(apr_array_header_t *filters, const char *type) {
  if (!type) return FF_TEXT;

  if (strncmp(type, "text/html", 9) == 0 ||
      strncmp(type, "application/xhtml+xml", 21) == 0)
  {
    return FF_HTML;
  }
  else if (strncmp(type, "text/plain", 10) == 0) {
    return FF_TEXT;
  }
  else {
    int i;
    filter_cmd *filter_cmds = (filter_cmd *)filters->elts;
    for (i = 0; i < filters->nelts; i++) {
      if (!ap_regexec(filter_cmds[i].content_type_reg, type, 0, NULL, 0)) {
        return FF_XCMD_START + i;
      }
    }
    return FF_NONE;
  }
}

static ESTDOC *est_doc_new_from_any(estraier_commit_ctx *cctx,
                             const char *buf, int size)
{
  ESTDOC *doc;
  int ff = check_content_type(cctx->conf->filter_cmds, cctx->content_type);

  if (ff == FF_HTML) {
    doc = est_doc_new_from_html(buf, size, cctx->charset, cctx->lang, TRUE);
    if (!doc) {
      ap_log_error(APLOG_MARK, APLOG_ERR, 0, cctx->serv,
                   "the html may be binary");
    }
    return doc;
  }
  else if (ff == FF_TEXT) {
    doc = est_doc_new_from_text(buf, size, cctx->charset, cctx->lang, TRUE);
    if (!doc) {
      ap_log_error(APLOG_MARK, APLOG_ERR, 0, cctx->serv,
                   "the text may be binary");
    }
    return doc;
  }
  else if (ff >= FF_XCMD_START) {
    filter_cmd *filter_cmds;
    ff -= FF_XCMD_START;
    filter_cmds = (filter_cmd *)cctx->conf->filter_cmds->elts;
    if (!cctx->conf->filter_tmp_dir) {
      ap_log_error(APLOG_MARK, APLOG_ERR, 0, cctx->serv,
                   "please set EstraierFilterTmpdir");
      return NULL;
    }
    return est_doc_new_with_xcmd(buf, size, cctx->uri, filter_cmds[ff].xcmd,
                                 cctx->conf->filter_tmp_dir,
                                 cctx->charset, cctx->lang);
  }

  ap_log_error(APLOG_MARK, APLOG_ERR, 0, cctx->serv,
               "unknown file format");
  return NULL;
}

static ESTNODE *get_estraier_node(apr_pool_t *p, estraier_dir_config *conf) {
  ESTNODE *node;

  node = est_node_new(conf->node);
  register_node_cleanup(p, node);
  if (conf->proxy_host) {
    est_node_set_proxy(node, conf->proxy_host, INT_OR(conf->proxy_port, 80));
  }
  est_node_set_timeout(node, INT_OR(conf->timeout, 5));
  est_node_set_auth(node, conf->user, conf->pass);

  return node;
}

static const char *check_digest(
  ESTNODE *node, ESTRESDOC *doc, estraier_commit_ctx *cctx)
{
  const char *digest, *digest_old;

  if (cctx->etag) {
    digest = cctx->etag;
  }
  else {
    digest = ap_md5_binary(cctx->pool,
                           (unsigned char*)cctx->body, cctx->body_len);
  }

  if (!doc) return digest;

  digest_old = est_resdoc_attr(doc, ESTDATTRDIGEST);

  if (digest_old && strcmp(digest, digest_old) == 0) {
    return NULL;
  }

  return digest;
}

static ESTDOC *create_estraier_doc(
  estraier_commit_ctx *cctx, ESTRESDOC *odoc, const char *digest)
{
  ESTDOC *doc;
  char *inf = NULL;
  int inf_len;

  if (cctx->is_enc) {
    inf = estraier_uncompress(
      cctx->body, cctx->body_len, &inf_len,
      INT_OR(cctx->conf->document_size_limit, 10000000));
    if (inf == NULL) {
      ap_log_error(APLOG_MARK, APLOG_ERR, 0, cctx->serv,
                   "too big gzip document");
      return NULL;
    }
  }

  if (inf) {
    doc = est_doc_new_from_any(cctx, inf, inf_len);
    free(inf);
  }
  else {
    doc = est_doc_new_from_any(cctx, cctx->body, cctx->body_len);
  }

  if (!doc) return NULL;

  est_doc_add_attr(doc, ESTDATTRURI, cctx->uri);
  est_doc_add_attr(doc, ESTDATTRDIGEST, digest);
  est_doc_add_attr(doc, ESTDATTRADATE, cctx->request_time);
  est_doc_add_attr(doc, ESTDATTRTYPE, cctx->content_type);
  if (cctx->content_language) {
    est_doc_add_attr(doc, ESTDATTRLANG, cctx->content_language);
  }

  add_scoreweight(doc, cctx->conf, odoc);
  if (cctx->mdate) est_doc_add_attr(doc, ESTDATTRMDATE, cctx->mdate);

  return doc;
}

static const char *header_get(ap_filter_t *f, char *key) {
  const char *val = apr_table_get(f->r->headers_out, key);
  if (val) return val;
  return apr_table_get(f->r->err_headers_out, key);
}

static estraier_commit_ctx *create_commit_ctx(
  apr_pool_t *p, ap_filter_t *f, estraier_ctx *ctx, estraier_dir_config *conf)
{
  estraier_commit_ctx *cctx;
  char buf[256];
  const char *tmp;

  cctx = (estraier_commit_ctx *)apr_pcalloc(p, sizeof(estraier_commit_ctx));
  cctx->serv = f->r->server;
  cctx->pool = p;
  cctx->conf = conf;

  cctx->body_len = ctx->body_len;
  cctx->body = ctx->body;
  apr_pool_cleanup_register(p, cctx->body,
                            free_estraier_string, apr_pool_cleanup_null);

  apr_rfc822_date(buf, f->r->request_time);
  cctx->request_time = apr_pstrdup(p, buf);
  cctx->mdate =
    apr_pstrdup(p, header_get(f, "Last-Modified"));
  cctx->content_type = apr_pstrdup(p, f->r->content_type);
  if (!cctx->content_type) header_get(f, "Content-Type");
  cctx->etag = apr_pstrdup(p, header_get(f, "ETag"));
  cctx->uri = get_full_uri(f, p);

  cctx->lang = LANGUNKNOWN;
  tmp = header_get(f, "Content-Language");
  if (tmp) {
    cctx->lang = get_estraier_language(tmp);
  }
  if (cctx->lang == LANGUNKNOWN) {
    cctx->lang = (conf->language != LANGUNKNOWN) ? conf->language : ESTLANGEN;
  }
  cctx->content_language = tmp;

  cctx->is_enc = 0;
  tmp = f->r->content_encoding;
  if (!tmp) tmp = header_get(f, "Content-Encoding");
  if (tmp && strcmp(tmp, "gzip") == 0) cctx->is_enc = 1;
  else {
    tmp = header_get(f, "Transfer-Encoding");
    if (tmp && strcmp(tmp, "gzip") == 0) cctx->is_enc = 1;
  }

  cctx->charset = apr_pstrdup(p, get_charset(f));

  return cctx;
}

static ESTRESDOC *get_estraier_resdoc_by_uri(
  apr_pool_t *pool, ESTNODE *node, const char *uri)
{
  ESTNODERES *nres;
  ESTCOND *cond;
  const char *phrase;

  cond = est_cond_new();

  phrase = apr_pstrcat(pool, "[URI] ", uri, NULL);
  est_cond_set_phrase(cond, phrase);
  est_node_set_snippet_width(node, 0, 0, 0);

  nres = est_node_search(node, cond, 0);
  est_cond_delete(cond);
  if (nres == NULL) return NULL;

  register_noderes_cleanup(pool, nres);

  if (est_noderes_doc_num(nres) != 1) {
    return NULL;
  }

  return est_noderes_get_doc(nres, 0);

}

/* return non NULL if errored */
static char *commit_estraier_db_p2p_impl(estraier_commit_ctx *cctx) {
  ESTNODE *node;
  ESTDOC *doc;
  const char *digest;
  ESTRESDOC *odoc;

  node = get_estraier_node(cctx->pool, cctx->conf);

  odoc = get_estraier_resdoc_by_uri(cctx->pool, node, cctx->uri);

  digest = check_digest(node, odoc, cctx);

  if (!digest) {
    if (INT_OR(cctx->conf->use_weight, 0)) {
      doc = est_doc_new();
      CBLIST *attrs = est_resdoc_attr_names(odoc);
      int i;

      for (i = 0; i < cblistnum(attrs); i++) {
        const char *attr = cblistval(attrs, i, NULL);
        est_doc_add_attr(doc, attr, est_resdoc_attr(odoc, attr));
      }

      add_scoreweight(doc, cctx->conf, odoc);
      est_doc_add_attr(doc, ESTDATTRADATE, cctx->request_time);

      if (!est_node_edit_doc(node, doc)) {
        est_doc_delete(doc);
        return "est_node_edit_doc failed";
      }
      est_doc_delete(doc);
    }
    return NULL;
  }

  doc = create_estraier_doc(cctx, odoc, digest);

  if (!doc) {
    return "cannot create estdoc";
  }

  if (!est_node_put_doc(node, doc)) {
    return "est_node_put_doc failed";
  }

  return NULL;
}

static void *commit_estraier_db_p2p_thread(
  apr_thread_t *thread, void *thread_arg)
{
  estraier_commit_ctx *cctx = (estraier_commit_ctx *)thread_arg;
  char *errmsg;

  errmsg = commit_estraier_db_p2p_impl(cctx);
  if (errmsg) {
    ap_log_error(APLOG_MARK, APLOG_ERR, 0, cctx->serv, errmsg);
  }

  return NULL;
}

static int commit_estraier_db_p2p_async(
  ap_filter_t *f, estraier_ctx *ctx, estraier_dir_config *conf)
{
  apr_pool_t *pool;
  estraier_commit_ctx *cctx;
  apr_thread_t *thread;

  apr_pool_create(&pool, NULL);

  cctx = create_commit_ctx(pool, f, ctx, conf);

  /* @@@ ignoring thread handle */
  apr_thread_create(&thread, NULL, commit_estraier_db_p2p_thread, cctx,
                    f->r->server->process->pool);

  return TRUE;
}

static int commit_estraier_db_p2p(
  ap_filter_t *f, estraier_ctx *ctx, estraier_dir_config *conf)
{
  estraier_commit_ctx *cctx;
  char *errmsg;

  cctx = create_commit_ctx(f->r->pool, f, ctx, conf);

  errmsg = commit_estraier_db_p2p_impl(cctx);

  if (errmsg) {
    ap_log_rerror(APLOG_MARK, APLOG_ERR, 0, f->r, errmsg);
    return FALSE;
  }

  return TRUE;
}

static int delete_estraier_db_p2p(ap_filter_t *f, estraier_dir_config *conf) {
  ESTNODE *node;

  node = get_estraier_node(f->r->pool, conf);

  if (!est_node_out_doc_by_uri(node, f->r->uri)) {
    return FALSE;
  }

  return TRUE;
}

static int check_estraier_config(ap_filter_t *f, estraier_dir_config *conf) {
  if (conf->node == NULL) {
    ap_log_rerror(APLOG_MARK, APLOG_ERR, 0, f->r, "EstraierNode is not set");
    return TRUE;
  }
  if (conf->user == NULL) {
    ap_log_rerror(APLOG_MARK, APLOG_ERR, 0, f->r, "EstraierUser is not set");
    return TRUE;
  }
  if (conf->pass == NULL) {
    ap_log_rerror(APLOG_MARK, APLOG_ERR, 0, f->r, "EstraierPass is not set");
    return TRUE;
  }

  return FALSE;
}

static int check_estraier_deny_header(apr_table_t *headers,
                                      apr_array_header_t *filters)
{
  int i, j;
  const apr_array_header_t *ha = apr_table_elts(headers);
  apr_table_entry_t *helts = (apr_table_entry_t *)ha->elts;
  filter_header *felts = (filter_header *)filters->elts;

  for (i = 0; i < ha->nelts; i++) {
    for (j = 0; j < filters->nelts; j++) {
      if (!ap_regexec(felts[j].header_reg, helts[i].key, 0, NULL, 0) &&
          !ap_regexec(felts[j].value_reg, helts[i].val, 0, NULL, 0))
      {
        return TRUE;
      }
    }
  }

  return FALSE;
}

static int estraier_filter(ap_filter_t *f, apr_bucket_brigade* bb) {
  const char *buf;
  apr_size_t buf_len;
  char *next_body;

  apr_status_t rv;
  apr_bucket *b;

  estraier_ctx *ctx;
  estraier_dir_config *conf;

  filter_uri *filter_uris;
  int i;

  /* initialize f->ctx */
  if ((rv = init_estraier_instance(f)) != APR_SUCCESS) {
    ap_log_rerror(APLOG_MARK, APLOG_ERR, rv, f->r,
                  "init_estraier_instance() failed");
    return rv;
  }
  ctx = (estraier_ctx *)f->ctx;

  /* get directory config */
  conf = (estraier_dir_config *)ap_get_module_config(
    f->r->per_dir_config, &estraier_module);

  if (ctx->error == STATE_SUCCESS) {
    goto check_end;
  }
  else if (ctx->error == STATE_ERROR) {
    goto pass_brigade;
  }

  /* check conf */
  if ((rv = check_estraier_config(f, conf))) {
    ctx->error = STATE_ERROR;
    goto pass_brigade;
  }

  /* check content-type */
  if (check_content_type(conf->filter_cmds, f->r->content_type) == FF_NONE) {
    ctx->error = STATE_ERROR;
    goto pass_brigade;
  }

  /* deny post? */
  if (f->r->method_number != M_GET) {
    goto pass_brigade;
  }

  /* deny uri? */
  filter_uris = (filter_uri *)conf->filter_uris->elts;
  for (i = conf->filter_uris->nelts-1; i >= 0 ; i--) {
    if (!ap_regexec(filter_uris[i].reg, f->r->uri, 0, NULL, 0)) {
      if (filter_uris[i].is_deny) {
        ctx->error = STATE_ERROR;
        goto pass_brigade;
      }
      else {
        break;
      }
    }
  }

  /* deny header? */
  if (check_estraier_deny_header(f->r->headers_in, conf->filter_req_headers) ||
      check_estraier_deny_header(f->r->headers_out, conf->filter_res_headers))
  {
    ctx->error = STATE_ERROR;
    goto pass_brigade;
  }

  /* check status */
  if (f->r->status == HTTP_NOT_MODIFIED ||
      f->r->status == HTTP_PARTIAL_CONTENT ||
      f->r->status == HTTP_UNAUTHORIZED)
  {
    ctx->error = STATE_ERROR;
    goto pass_brigade;
  }
  else if (f->r->status != HTTP_OK &&
           f->r->status != HTTP_NON_AUTHORITATIVE)
  {
    delete_estraier_db_p2p(f, conf);
    ctx->error = STATE_ERROR;
    goto pass_brigade;
  }

  ctx->error = STATE_SUCCESS;

check_end:

  for (b = APR_BRIGADE_FIRST(bb);
       b != APR_BRIGADE_SENTINEL(bb);
       b = APR_BUCKET_NEXT(b))
  {
    if (APR_BUCKET_IS_EOS(b)) {
      if (ctx->body == NULL) {
        ap_log_rerror(APLOG_MARK, APLOG_ERR, 0, f->r,
                      "body is empty");
        goto pass_brigade;
      }
      if (INT_OR(conf->detach_thread, 0)) {
        rv = commit_estraier_db_p2p_async(f, ctx, conf);
      }
      else {
        rv = commit_estraier_db_p2p(f, ctx, conf);
      }
      if (!rv) {
        ap_log_rerror(APLOG_MARK, APLOG_ERR, 0, f->r,
                      "commit_estraier_db_p2p failed");
        goto pass_brigade;
      }
    }
    else {
      rv = apr_bucket_read(b, &buf, &buf_len, APR_BLOCK_READ);
      if (rv != APR_SUCCESS) {
        ap_log_rerror(APLOG_MARK, APLOG_ERR, rv, f->r,
                      "apr_bucket_read() failed");
        return rv;
      }

      if (buf_len+ctx->body_len >
          INT_OR(conf->document_size_limit, 10000000)) {
        ap_log_rerror(APLOG_MARK, APLOG_ERR, rv, f->r, "too big document");
        free(ctx->body);
        ctx->error = 1;
        goto pass_brigade;
      }

      next_body =
        (char *)realloc(ctx->body, sizeof(char)*(buf_len+ctx->body_len+1));
      if (next_body == NULL) {
        free(ctx->body);
        ap_log_rerror(APLOG_MARK, APLOG_ERR, rv, f->r, "realloc() failed");
        ctx->error = 1;
        goto pass_brigade;
      }
      ctx->body = next_body;
      memcpy(ctx->body+ctx->body_len, buf, buf_len);
      ctx->body_len += buf_len;
      ctx->body[ctx->body_len] = '\0';
    }
  }

pass_brigade:
  if ((rv = ap_pass_brigade(f->next, bb)) != APR_SUCCESS) {
    ap_log_rerror(APLOG_MARK, APLOG_ERR, rv, f->r,
                  "ap_pass_brigade(filtered buffer) failed");
    return rv;
  }

  return APR_SUCCESS;
}

static apr_status_t cleanup_estraier(void *p) {
  est_free_net_env();
  return OK;
}

static int estraier_post_config(apr_pool_t* p, apr_pool_t* p1, apr_pool_t* p2,
                                server_rec* s) {
  if (!est_init_net_env()) {
    ap_log_error(APLOG_MARK, APLOG_ERR, 0, s, "est_init_net_env failed");
    return !OK;
  }
  apr_pool_cleanup_register(p, NULL, cleanup_estraier, apr_pool_cleanup_null);

  return OK;
}

static void register_estraier_hooks(apr_pool_t* p) {
  ap_register_output_filter("estraier", estraier_filter,
                            NULL, AP_FTYPE_RESOURCE);
  ap_hook_post_config(estraier_post_config, NULL, NULL, APR_HOOK_MIDDLE);
}

static const char *set_estraier_node(cmd_parms *cmd,
                                     void *tmp, const char *arg)
{
  estraier_dir_config *conf = (estraier_dir_config *)tmp;
  conf->node = apr_pstrdup(cmd->pool, arg);

  return NULL;
}

static const char *set_estraier_user(cmd_parms *cmd,
                                     void *tmp, const char *arg)
{
  estraier_dir_config *conf = (estraier_dir_config *)tmp;
  conf->user = apr_pstrdup(cmd->pool, arg);

  return NULL;
}

static const char *set_estraier_pass(cmd_parms *cmd,
                                     void *tmp, const char *arg)
{
  estraier_dir_config *conf = (estraier_dir_config *)tmp;
  conf->pass = apr_pstrdup(cmd->pool, arg);

  return NULL;
}

static const char *set_estraier_proxy_host(cmd_parms *cmd,
                                           void *tmp, const char *arg)
{
  estraier_dir_config *conf = (estraier_dir_config *)tmp;
  conf->proxy_host = apr_pstrdup(cmd->pool, arg);

  return NULL;
}

static const char *set_estraier_proxy_port(cmd_parms *cmd,
                                           void *tmp, const char *arg)
{
  int port = atoi(arg);
  if (port < 1 || port > 65535) {
    return "EstraierPort must be valid port";
  }

  estraier_dir_config *conf = (estraier_dir_config *)tmp;
  conf->proxy_port = port;

  return NULL;
}

static const char *set_estraier_timeout(cmd_parms *cmd,
                                        void *tmp, const char *arg)
{
  int timeout = atoi(arg);
  if (timeout < 1 || timeout > 30) {
    return "EstraierTimeout must be between 1 and 30";
  }

  estraier_dir_config *conf = (estraier_dir_config *)tmp;
  conf->timeout = timeout;

  return NULL;
}

static const char *set_estraier_deny_uri(cmd_parms *cmd,
                                            void *tmp, const char *arg)
{
  filter_uri *filter;

  estraier_dir_config *conf = (estraier_dir_config *)tmp;
  filter = apr_array_push(conf->filter_uris);
  filter->reg = ap_pregcomp(cmd->pool, arg, REG_EXTENDED | REG_NOSUB);
  filter->is_deny = TRUE;

  return NULL;
}

static const char *set_estraier_allow_uri(cmd_parms *cmd,
                                            void *tmp, const char *arg)
{
  filter_uri *filter;

  estraier_dir_config *conf = (estraier_dir_config *)tmp;
  filter = apr_array_push(conf->filter_uris);
  filter->reg = ap_pregcomp(cmd->pool, arg, REG_EXTENDED | REG_NOSUB);
  filter->is_deny = FALSE;

  return NULL;
}

static const char *set_estraier_language(cmd_parms *cmd,
                                         void *tmp, const char *arg)
{
  estraier_dir_config *conf = (estraier_dir_config *)tmp;
  if ((conf->language = get_estraier_language(arg)) == -1) {
    return "EstraierLanguage must be en, ja, zh, ko, misc";
  }

  return NULL;
}

static const char *set_estraier_detach_thread(cmd_parms *cmd,
                                              void *tmp, int flag)
{
  estraier_dir_config *conf = (estraier_dir_config *)tmp;
  conf->detach_thread = flag;

  return NULL;
}

static const char *set_estraier_deny_request_header(
  cmd_parms *cmd, void *tmp, const char *arg1, const char *arg2)
{
  filter_header *filter;

  estraier_dir_config *conf = (estraier_dir_config *)tmp;
  filter = apr_array_push(conf->filter_req_headers);
  filter->header_reg = ap_pregcomp(cmd->pool, arg1,
                                   REG_EXTENDED | REG_NOSUB | REG_ICASE);
  filter->value_reg = ap_pregcomp(cmd->pool, arg2, REG_EXTENDED | REG_NOSUB);

  return NULL;
}

static const char *set_estraier_deny_response_header(
  cmd_parms *cmd, void *tmp, const char *arg1, const char *arg2)
{
  filter_header *filter;

  estraier_dir_config *conf = (estraier_dir_config *)tmp;
  filter = apr_array_push(conf->filter_res_headers);
  filter->header_reg = ap_pregcomp(cmd->pool, arg1,
                                   REG_EXTENDED | REG_NOSUB | REG_ICASE);
  filter->value_reg = ap_pregcomp(cmd->pool, arg2, REG_EXTENDED | REG_NOSUB);

  return NULL;
}

static const char *set_estraier_use_weight(cmd_parms *cmd,
                                                 void *tmp, int flag)
{
  estraier_dir_config *conf = (estraier_dir_config *)tmp;
  conf->use_weight = flag;

  return NULL;
}

static const char *set_estraier_filter_command(
  cmd_parms *cmd, void *tmp, const char *arg1, const char *arg2)
{
  filter_cmd *filter;

  estraier_dir_config *conf = (estraier_dir_config *)tmp;
  filter = apr_array_push(conf->filter_cmds);
  filter->content_type_reg = ap_pregcomp(cmd->pool, arg1,
                                         REG_EXTENDED | REG_NOSUB);
  filter->xcmd = apr_pstrdup(cmd->pool, arg2);

  return NULL;
}

static const char *set_estraier_filter_tmp_dir(cmd_parms *cmd,
                                               void *tmp, const char *arg)
{
  estraier_dir_config *conf = (estraier_dir_config *)tmp;
  conf->filter_tmp_dir = apr_pstrdup(cmd->pool, arg);

  return NULL;
}

static const char *set_estraier_document_size_limit(cmd_parms *cmd,
                                                    void *tmp, const char *arg)
{
  int document_size_limit = atoi(arg);
  if (document_size_limit < 0) {
    return "EstraierTimeout must be positive";
  }

  estraier_dir_config *conf = (estraier_dir_config *)tmp;
  conf->document_size_limit = document_size_limit;

  return NULL;
}

static void *create_estraier_dir_config(apr_pool_t *p, char *path) {
  estraier_dir_config *conf;

  conf = (estraier_dir_config *)apr_pcalloc(p, sizeof(estraier_dir_config));

  conf->node = NULL;
  conf->user = NULL;
  conf->pass = NULL;

  conf->proxy_host = NULL;
  conf->proxy_port = -1;
  conf->timeout = -1;

  conf->filter_uris = apr_array_make(p, 0, sizeof(filter_uri));
  conf->language = LANGUNKNOWN;
  conf->detach_thread = -1;
  conf->filter_req_headers = apr_array_make(p, 0, sizeof(filter_header));
  conf->filter_res_headers = apr_array_make(p, 0, sizeof(filter_header));
  conf->use_weight = -1;

  conf->filter_cmds = apr_array_make(p, 0, sizeof(filter_cmd));
  conf->filter_tmp_dir = NULL;

  conf->document_size_limit = -1;

  return (void *)conf;
}

static void *merge_estraier_dir_configs(apr_pool_t *p, void *bp, void *ap) {
  estraier_dir_config *base = (estraier_dir_config *)bp;
  estraier_dir_config *add = (estraier_dir_config *)ap;
  estraier_dir_config *conf;

  conf = (estraier_dir_config *)apr_pcalloc(p, sizeof(estraier_dir_config));

  conf->node = PTR_OR(add->node, base->node);
  conf->user = PTR_OR(add->user, base->user);
  conf->pass = PTR_OR(add->pass, base->pass);

  conf->proxy_host = PTR_OR(add->proxy_host, base->proxy_host);
  conf->proxy_port = INT_OR(add->proxy_port, base->proxy_port);
  conf->timeout = INT_OR(add->timeout, base->timeout);

  conf->filter_uris = apr_array_append(p, add->filter_uris, base->filter_uris);
  conf->language =
    (add->language != LANGUNKNOWN) ? add->language : base->language;
  conf->detach_thread = INT_OR(add->detach_thread, base->detach_thread);
  conf->filter_req_headers =
    apr_array_append(p, add->filter_req_headers, base->filter_req_headers);
  conf->filter_res_headers =
    apr_array_append(p, add->filter_res_headers, base->filter_res_headers);
  conf->use_weight = INT_OR(add->use_weight, base->use_weight);

  conf->filter_cmds = apr_array_append(p, add->filter_cmds, base->filter_cmds);
  conf->filter_tmp_dir = PTR_OR(add->filter_tmp_dir, base->filter_tmp_dir);

  conf->document_size_limit =
    INT_OR(add->document_size_limit, base->document_size_limit);

  return conf;
}

static const command_rec estraier_cmds[] = {
  AP_INIT_TAKE1("EstraierNode", set_estraier_node, NULL, OR_OPTIONS,
                "url of node"),
  AP_INIT_TAKE1("EstraierUser", set_estraier_user, NULL, OR_OPTIONS,
                "user of node"),
  AP_INIT_TAKE1("EstraierPass", set_estraier_pass, NULL, OR_OPTIONS,
                "password of node"),
  AP_INIT_TAKE1("EstraierProxyHost", set_estraier_proxy_host, NULL, OR_OPTIONS,
                "proxy host for node"),
  AP_INIT_TAKE1("EstraierProxyPort", set_estraier_proxy_port, NULL, OR_OPTIONS,
                "proxy port for node"),
  AP_INIT_TAKE1("EstraierTimeout", set_estraier_timeout, NULL, OR_OPTIONS,
                "timeout seconds, integer number"),
  AP_INIT_TAKE1("EstraierDenyURI", set_estraier_deny_uri, NULL,
                OR_OPTIONS, "regular expression of deny uri"),
  AP_INIT_TAKE1("EstraierAllowURI", set_estraier_allow_uri, NULL,
                OR_OPTIONS, "regular expression of allow uri"),
  AP_INIT_TAKE1("EstraierLanguage", set_estraier_language, NULL, OR_OPTIONS,
                "en, ja, zh, ko, misc"),
  AP_INIT_FLAG("EstraierDetachThread", set_estraier_detach_thread, NULL,
               OR_OPTIONS, "on or off"),
  AP_INIT_TAKE2("EstraierDenyRequestHeader", set_estraier_deny_request_header,
                NULL, OR_OPTIONS, "header regexp and value regexp"),
  AP_INIT_TAKE2("EstraierDenyResponseHeader",
                set_estraier_deny_response_header,
                NULL, OR_OPTIONS, "header regexp and value regexp"),
  AP_INIT_FLAG("EstraierUseWeight", set_estraier_use_weight, NULL,
               OR_OPTIONS, "on or off"),
  AP_INIT_TAKE1("EstraierFilterTmpdir", set_estraier_filter_tmp_dir,
                NULL, OR_OPTIONS, "tmpdir used by filter command"),
  AP_INIT_TAKE2("EstraierFilterCommand", set_estraier_filter_command,
                NULL, OR_OPTIONS, "content-type and filter command"),
  AP_INIT_TAKE1("EstraierDocumentSizeLimit", set_estraier_document_size_limit,
                NULL, OR_OPTIONS, "document size limit"),
  {NULL}
};

module AP_MODULE_DECLARE_DATA estraier_module = {
  STANDARD20_MODULE_STUFF,
  create_estraier_dir_config,
  merge_estraier_dir_configs,
  NULL,
  NULL,
  estraier_cmds,
  register_estraier_hooks
};
