#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <string.h>
#include <stdio.h>

#include <bonobo.h>
#include <gnome.h>

#include "mgwtypes.h"
#include "interface.h"
#include "callbacks.h"
#include "arrange.h"
#include "filters.h"
#include "process.h"
#include "sort.h"
#include "image.h"
#include "textize.h"
#include "bat.h"
#include "utils.h"
#include "support.h"

#include "pixmaps/window_icon.xpm"

enum {
  TARGET_URI_LIST,
  TARGET_URL
};

static GtkWidget *app;
static GtkWidget *appbar;
static GtkWidget *drawing_area;
static GdkGC     *gc_invert_dash = NULL;

static GnomeUIInfo file_menu_uiinfo[] =
{
  GNOMEUIINFO_MENU_OPEN_ITEM (on_open_activate, NULL),
  GNOMEUIINFO_MENU_SAVE_ITEM (on_save_activate, NULL),
  GNOMEUIINFO_MENU_SAVE_AS_ITEM (on_save_as_activate, NULL),
  GNOMEUIINFO_MENU_REVERT_ITEM (on_revert_activate, NULL),
  GNOMEUIINFO_SEPARATOR,
  GNOMEUIINFO_MENU_EXIT_ITEM (on_quit_activate, NULL),
  GNOMEUIINFO_END
};

#ifdef USE_EDIT_MENU
static GnomeUIInfo edit_menu_uiinfo[] =
{
  GNOMEUIINFO_MENU_CUT_ITEM (on_cut_activate, NULL),
  GNOMEUIINFO_MENU_COPY_ITEM (on_copy_activate, NULL),
  GNOMEUIINFO_MENU_PASTE_ITEM (on_paste_activate, NULL),
  GNOMEUIINFO_MENU_CLEAR_ITEM (on_clear_activate, NULL),
  GNOMEUIINFO_SEPARATOR,
  GNOMEUIINFO_MENU_PROPERTIES_ITEM (on_properties_activate, NULL),
  GNOMEUIINFO_SEPARATOR,
  GNOMEUIINFO_MENU_PREFERENCES_ITEM (on_preferences_activate, NULL),
  GNOMEUIINFO_END
};
#endif

static GnomeUIInfo zoom_uiinfo[] =
{
  { GNOME_APP_UI_ITEM, N_("100%"),
    NULL, (gpointer) view_zoom, "100%", NULL,
    GNOME_APP_PIXMAP_NONE, NULL, '1', (GdkModifierType) 0, NULL },
  GNOMEUIINFO_SEPARATOR,
  { GNOME_APP_UI_ITEM, N_("90%"),
    NULL, (gpointer) view_zoom, "90%", NULL,
    GNOME_APP_PIXMAP_NONE, NULL, 0, (GdkModifierType) 0, NULL },
  { GNOME_APP_UI_ITEM, N_("75%"),
    NULL, (gpointer) view_zoom, "75%", NULL,
    GNOME_APP_PIXMAP_NONE, NULL, 0, (GdkModifierType) 0, NULL },
  { GNOME_APP_UI_ITEM, N_("50%"),
    NULL, (gpointer) view_zoom, "50%", NULL,
    GNOME_APP_PIXMAP_NONE, NULL, '2', (GdkModifierType) 0, NULL },
  { GNOME_APP_UI_ITEM, N_("25%"),
    NULL, (gpointer) view_zoom, "25%", NULL,
    GNOME_APP_PIXMAP_NONE, NULL, 0, (GdkModifierType) 0, NULL },
  { GNOME_APP_UI_ITEM, N_("10%"),
    NULL, (gpointer) view_zoom, "10%", NULL,
    GNOME_APP_PIXMAP_NONE, NULL, 0, (GdkModifierType) 0, NULL },
  GNOMEUIINFO_SEPARATOR,
  { GNOME_APP_UI_ITEM, N_("110%"),
    NULL, (gpointer) view_zoom, "110%", NULL,
    GNOME_APP_PIXMAP_NONE, NULL, 0, (GdkModifierType) 0, NULL },
  { GNOME_APP_UI_ITEM, N_("125%"),
    NULL, (gpointer) view_zoom, "125%", NULL,
    GNOME_APP_PIXMAP_NONE, NULL, 0, (GdkModifierType) 0, NULL },
  { GNOME_APP_UI_ITEM, N_("150%"),
    NULL, (gpointer) view_zoom, "150%", NULL,
    GNOME_APP_PIXMAP_NONE, NULL, 0, (GdkModifierType) 0, NULL },
  { GNOME_APP_UI_ITEM, N_("200%"),
    NULL, (gpointer) view_zoom, "200%", NULL,
    GNOME_APP_PIXMAP_NONE, NULL, 0, (GdkModifierType) 0, NULL },
  GNOMEUIINFO_END
};

static GnomeUIInfo sort_uiinfo[] =
{
  { GNOME_APP_UI_ITEM, N_("Fast Quick Sort"),
    NULL, (gpointer) sort_exec, "Fast Quick Sort", NULL,
    GNOME_APP_PIXMAP_NONE, NULL, 0, (GdkModifierType) 0, NULL },
  { GNOME_APP_UI_ITEM, N_("Quick Sort"),
    NULL, (gpointer) sort_exec, "Quick Sort", NULL,
    GNOME_APP_PIXMAP_NONE, NULL, 0, (GdkModifierType) 0, NULL },
  { GNOME_APP_UI_ITEM, N_("Comb Sort"),
    NULL, (gpointer) sort_exec, "Comb Sort", NULL,
    GNOME_APP_PIXMAP_NONE, NULL, 0, (GdkModifierType) 0, NULL },
  { GNOME_APP_UI_ITEM, N_("Shell Sort"),
    NULL, (gpointer) sort_exec, "Shell Sort", NULL,
    GNOME_APP_PIXMAP_NONE, NULL, 0, (GdkModifierType) 0, NULL },
  { GNOME_APP_UI_ITEM, N_("Heap Sort"),
    NULL, (gpointer) sort_exec, "Heap Sort", NULL,
    GNOME_APP_PIXMAP_NONE, NULL, 0, (GdkModifierType) 0, NULL },
  { GNOME_APP_UI_ITEM, N_("K & R"),
    NULL, (gpointer) sort_exec, "K & R", NULL,
    GNOME_APP_PIXMAP_NONE, NULL, 0, (GdkModifierType) 0, NULL },
  GNOMEUIINFO_END
};

static GnomeUIInfo view_menu_uiinfo[] =
{
  { GNOME_APP_UI_ITEM, N_("Shrink Wrap"),
    NULL, (gpointer) shrink_wrap, NULL, NULL,
    GNOME_APP_PIXMAP_NONE, NULL, 'E',
    (GdkModifierType) GDK_CONTROL_MASK, NULL },
  { GNOME_APP_UI_ITEM, N_("Screen Height"),
    NULL, (gpointer) adjust_window_to_screen_size, NULL, NULL,
    GNOME_APP_PIXMAP_NONE, NULL, 'F',
    (GdkModifierType) GDK_CONTROL_MASK, NULL },
  GNOMEUIINFO_SEPARATOR,
  GNOMEUIINFO_SUBTREE (N_("Zoom"), zoom_uiinfo),
  GNOMEUIINFO_SEPARATOR,
  GNOMEUIINFO_SUBTREE (N_("Sort"), sort_uiinfo),
  GNOMEUIINFO_END
};

static GnomeUIInfo resize_uiinfo[] =
{
  { GNOME_APP_UI_ITEM, N_("Screen Size"),
    NULL, (gpointer) process_exec, "Screen Size", NULL,
    GNOME_APP_PIXMAP_NONE, NULL, 'A',
    (GdkModifierType) GDK_CONTROL_MASK, NULL },
  GNOMEUIINFO_SEPARATOR,
  { GNOME_APP_UI_ITEM, N_("x0.90"),
    NULL, (gpointer) process_exec, "Resize 90%", NULL,
    GNOME_APP_PIXMAP_NONE, NULL, 0, (GdkModifierType) 0, NULL },
  { GNOME_APP_UI_ITEM, N_("x0.71"),
    NULL, (gpointer) process_exec, "Resize 71%", NULL,
    GNOME_APP_PIXMAP_NONE, NULL, 0, (GdkModifierType) 0, NULL },
  { GNOME_APP_UI_ITEM, N_("x0.50"),
    NULL, (gpointer) process_exec, "Resize 50%", NULL,
    GNOME_APP_PIXMAP_NONE, NULL, 0, (GdkModifierType) 0, NULL },
  { GNOME_APP_UI_ITEM, N_("x0.33"),
    NULL, (gpointer) process_exec, "Resize 33%", NULL,
    GNOME_APP_PIXMAP_NONE, NULL, 0, (GdkModifierType) 0, NULL },
  GNOMEUIINFO_SEPARATOR,
  { GNOME_APP_UI_ITEM, N_("x1.10"),
    NULL, (gpointer) process_exec, "Resize 110%", NULL,
    GNOME_APP_PIXMAP_NONE, NULL, 0, (GdkModifierType) 0, NULL },
  { GNOME_APP_UI_ITEM, N_("x1.41"),
    NULL, (gpointer) process_exec, "Resize 141%", NULL,
    GNOME_APP_PIXMAP_NONE, NULL, 0, (GdkModifierType) 0, NULL },
  { GNOME_APP_UI_ITEM, N_("x2.00"),
    NULL, (gpointer) process_exec, "Resize 200%", NULL,
    GNOME_APP_PIXMAP_NONE, NULL, 0, (GdkModifierType) 0, NULL },
  { GNOME_APP_UI_ITEM, N_("x3.00"),
    NULL, (gpointer) process_exec, "Resize 300%", NULL,
    GNOME_APP_PIXMAP_NONE, NULL, 0, (GdkModifierType) 0, NULL },
  GNOMEUIINFO_SEPARATOR,
  { GNOME_APP_UI_ITEM, N_("Factor..."),
    NULL, (gpointer) on_scale_image_factor_activate, NULL, NULL,
    GNOME_APP_PIXMAP_NONE, NULL, 0, (GdkModifierType) 0, NULL},
  { GNOME_APP_UI_ITEM, N_("W x H..."),
    NULL, (gpointer) on_scale_image_size_activate, NULL, NULL,
    GNOME_APP_PIXMAP_NONE, NULL, 0, (GdkModifierType) 0, NULL},
  GNOMEUIINFO_END
};

static GnomeUIInfo rotate_uiinfo[] =
{
  { GNOME_APP_UI_ITEM, N_("Rotate 90 deg."),
    NULL, (gpointer) process_exec, "Rotate 90deg.", NULL,
    GNOME_APP_PIXMAP_NONE, NULL, 0, (GdkModifierType) 0, NULL },
  { GNOME_APP_UI_ITEM, N_("Rotate 180 deg."),
    NULL, (gpointer) process_exec, "Rotate 180deg.", NULL,
    GNOME_APP_PIXMAP_NONE, NULL, 0, (GdkModifierType) 0, NULL },
  { GNOME_APP_UI_ITEM, N_("Rotate 270 deg."),
    NULL, (gpointer) process_exec, "Rotate 270deg.", NULL,
    GNOME_APP_PIXMAP_NONE, NULL, 0, (GdkModifierType) 0, NULL },
  GNOMEUIINFO_END
};

static GnomeUIInfo flip_uiinfo[] =
{
  { GNOME_APP_UI_ITEM, N_("Flip Horizontal"),
    NULL, (gpointer) process_exec, "Flip Horizontal", NULL,
    GNOME_APP_PIXMAP_NONE, NULL, 0, (GdkModifierType) 0, NULL },
  { GNOME_APP_UI_ITEM, N_("Flip Vertical"),
    NULL, (gpointer) process_exec, "Flip Vertical", NULL,
    GNOME_APP_PIXMAP_NONE, NULL, 0, (GdkModifierType) 0, NULL },
  GNOMEUIINFO_END
};

static GnomeUIInfo textize_uiinfo[] =
{
  { GNOME_APP_UI_ITEM, N_("Textize"),
    NULL, (gpointer) on_textize_activate, NULL, NULL,
    GNOME_APP_PIXMAP_NONE, NULL, 0, (GdkModifierType) 0, NULL },
  { GNOME_APP_UI_ITEM, N_("HTMLize..."),
    NULL, (gpointer) save_another_type, "html", NULL,
    GNOME_APP_PIXMAP_NONE, NULL, 0, (GdkModifierType) 0, NULL },
  { GNOME_APP_UI_ITEM, N_("PostScriptize..."),
    NULL, (gpointer) save_another_type, "ps", NULL,
    GNOME_APP_PIXMAP_NONE, NULL, 0, (GdkModifierType) 0, NULL },
  GNOMEUIINFO_END
};

static GnomeUIInfo process_menu_uiinfo[] =
{
  GNOMEUIINFO_SUBTREE (N_("Resize"), resize_uiinfo),
  GNOMEUIINFO_SUBTREE (N_("Rotate"), rotate_uiinfo),
  GNOMEUIINFO_SUBTREE (N_("Flip"), flip_uiinfo),
  GNOMEUIINFO_SEPARATOR,
  GNOMEUIINFO_SUBTREE (N_("Textize"), textize_uiinfo),
  GNOMEUIINFO_END
};

static GnomeUIInfo discolor_uiinfo[] =
{
  { GNOME_APP_UI_ITEM, N_("Stella by Starlight"),
    NULL, (gpointer) filter_exec, "Stella by Starlight", NULL,
    GNOME_APP_PIXMAP_NONE, NULL, 0, (GdkModifierType) 0, NULL },
  { GNOME_APP_UI_ITEM, N_("Autumn Leaves"),
    NULL, (gpointer) filter_exec, "Autumn Leaves", NULL,
    GNOME_APP_PIXMAP_NONE, NULL, 0, (GdkModifierType) 0, NULL },
  GNOMEUIINFO_SEPARATOR,
  { GNOME_APP_UI_ITEM, N_("Caricature"),
    NULL, (gpointer) filter_exec, "Caricature", NULL,
    GNOME_APP_PIXMAP_NONE, NULL, 0, (GdkModifierType) 0, NULL },
  { GNOME_APP_UI_ITEM, N_("Pale"),
    NULL, (gpointer) filter_exec, "Pale", NULL,
    GNOME_APP_PIXMAP_NONE, NULL, 0, (GdkModifierType) 0, NULL },
  { GNOME_APP_UI_ITEM, N_("Monotonic"),
    NULL, (gpointer) filter_exec, "Monotonic", NULL,
    GNOME_APP_PIXMAP_NONE, NULL, 0, (GdkModifierType) 0, NULL },
  { GNOME_APP_UI_ITEM, N_("Nature"),
    NULL, (gpointer) filter_exec, "Nature", NULL,
    GNOME_APP_PIXMAP_NONE, NULL, 0, (GdkModifierType) 0, NULL },
  { GNOME_APP_UI_ITEM, N_("Mesh Auto"),
    NULL, (gpointer) filter_exec, "Mesh Auto-level", NULL,
    GNOME_APP_PIXMAP_NONE, NULL, 0, (GdkModifierType) 0, NULL },
  GNOMEUIINFO_SEPARATOR,
  { GNOME_APP_UI_ITEM, N_("Impulse"),
    NULL, (gpointer) filter_exec, "Impulse", NULL,
    GNOME_APP_PIXMAP_NONE, NULL, 0, (GdkModifierType) 0, NULL },
  { GNOME_APP_UI_ITEM, N_("Infinity"),
    NULL, (gpointer) filter_exec, "Infinity", NULL,
    GNOME_APP_PIXMAP_NONE, NULL, 0, (GdkModifierType) 0, NULL },
  { GNOME_APP_UI_ITEM, N_("Impressions"),
    NULL, (gpointer) filter_exec, "Impressions", NULL,
    GNOME_APP_PIXMAP_NONE, NULL, 0, (GdkModifierType) 0, NULL },
  GNOMEUIINFO_SEPARATOR,
  { GNOME_APP_UI_ITEM, N_("Black Expansion"),
    NULL, (gpointer) filter_exec, "Black Expansion", NULL,
    GNOME_APP_PIXMAP_NONE, NULL, 0, (GdkModifierType) 0, NULL },
  { GNOME_APP_UI_ITEM, N_("White Expansion"),
    NULL, (gpointer) filter_exec, "White Expansion", NULL,
    GNOME_APP_PIXMAP_NONE, NULL, 0, (GdkModifierType) 0, NULL },
  GNOMEUIINFO_END
};

static GnomeUIInfo color_uiinfo[] =
{
  { GNOME_APP_UI_ITEM, N_("Reverse"),
    NULL, (gpointer) filter_exec, "Reverse", NULL,
    GNOME_APP_PIXMAP_NONE, NULL, 0, (GdkModifierType) 0, NULL },
  GNOMEUIINFO_SEPARATOR,
  { GNOME_APP_UI_ITEM, N_("Gray Scale"),
    NULL, (gpointer) filter_exec, "Gray Scale", NULL,
    GNOME_APP_PIXMAP_NONE, NULL, 0, (GdkModifierType) 0, NULL },
  { GNOME_APP_UI_ITEM, N_("Sepia"),
    NULL, (gpointer) filter_exec, "Sepia", NULL,
    GNOME_APP_PIXMAP_NONE, NULL, 0, (GdkModifierType) 0, NULL },
  { GNOME_APP_UI_ITEM, N_("Anti Sepia"),
    NULL, (gpointer) filter_exec, "Anti Sepia", NULL,
    GNOME_APP_PIXMAP_NONE, NULL, 0, (GdkModifierType) 0, NULL },
  GNOMEUIINFO_SEPARATOR,
  { GNOME_APP_UI_ITEM, N_("Black/White"),
    NULL, (gpointer) filter_exec, "Black/White", NULL,
    GNOME_APP_PIXMAP_NONE, NULL, 0, (GdkModifierType) 0, NULL },
  { GNOME_APP_UI_ITEM, N_("Enhanced B/W"),
    NULL, (gpointer) filter_exec, "Enhanced Black/White", NULL,
    GNOME_APP_PIXMAP_NONE, NULL, 0, (GdkModifierType) 0, NULL },
  GNOMEUIINFO_SEPARATOR,
  GNOMEUIINFO_SUBTREE (N_("Discolor"), discolor_uiinfo),
  GNOMEUIINFO_END
};

static GnomeUIInfo power_uiinfo[] =
{
  { GNOME_APP_UI_ITEM, N_("Pow-Brightness-On"),
    NULL, (gpointer) filter_exec, "Pow-Brightness-On", NULL,
    GNOME_APP_PIXMAP_NONE, NULL, 0, (GdkModifierType) 0, NULL },
  { GNOME_APP_UI_ITEM, N_("Pow-Contrast-On"),
    NULL, (gpointer) filter_exec, "Pow-Contrast-On", NULL,
    GNOME_APP_PIXMAP_NONE, NULL, 0, (GdkModifierType) 0, NULL },
  { GNOME_APP_UI_ITEM, N_("Pow-Contrast-Off"),
    NULL, (gpointer) filter_exec, "Pow-Contrast-Off", NULL,
    GNOME_APP_PIXMAP_NONE, NULL, 0, (GdkModifierType) 0, NULL },
  { GNOME_APP_UI_ITEM, N_("Pow-Brightness-Off"),
    NULL, (gpointer) filter_exec, "Pow-Brightness-Off", NULL,
    GNOME_APP_PIXMAP_NONE, NULL, 0, (GdkModifierType) 0, NULL },
  GNOMEUIINFO_END
};

static GnomeUIInfo level_uiinfo[] =
{
  { GNOME_APP_UI_ITEM, N_("Brightness-On"),
    NULL, (gpointer) filter_exec, "Brightness-On", NULL,
    GNOME_APP_PIXMAP_NONE, NULL, 0, (GdkModifierType) 0, NULL },
  { GNOME_APP_UI_ITEM, N_("Contrast-On"),
    NULL, (gpointer) filter_exec, "Contrast-On", NULL,
    GNOME_APP_PIXMAP_NONE, NULL, 0, (GdkModifierType) 0, NULL },
  { GNOME_APP_UI_ITEM, N_("Contrast-Off"),
    NULL, (gpointer) filter_exec, "Contrast-Off", NULL,
    GNOME_APP_PIXMAP_NONE, NULL, 0, (GdkModifierType) 0, NULL },
  { GNOME_APP_UI_ITEM, N_("Brightness-Off"),
    NULL, (gpointer) filter_exec, "Brightness-Off", NULL,
    GNOME_APP_PIXMAP_NONE, NULL, 0, (GdkModifierType) 0, NULL },
  GNOMEUIINFO_SEPARATOR,
  { GNOME_APP_UI_ITEM, N_("Sunshine"),
    NULL, (gpointer) filter_exec, "Sunshine", NULL,
    GNOME_APP_PIXMAP_NONE, NULL, 0, (GdkModifierType) 0, NULL },
  { GNOME_APP_UI_ITEM, N_("Super Nova"),
    NULL, (gpointer) filter_exec, "Super Nova", NULL,
    GNOME_APP_PIXMAP_NONE, NULL, 0, (GdkModifierType) 0, NULL },
  GNOMEUIINFO_SEPARATOR,
  GNOMEUIINFO_SUBTREE (N_("Power Function"), power_uiinfo),
  GNOMEUIINFO_END
};

static GnomeUIInfo tone_uiinfo[] =
{
  { GNOME_APP_UI_ITEM, N_("Blur"),
    NULL, (gpointer) filter_exec, "Blur", NULL,
    GNOME_APP_PIXMAP_NONE, NULL, 0, (GdkModifierType) 0, NULL },
  { GNOME_APP_UI_ITEM, N_("Softning"),
    NULL, (gpointer) filter_exec, "Softning", NULL,
    GNOME_APP_PIXMAP_NONE, NULL, 0, (GdkModifierType) 0, NULL },
  { GNOME_APP_UI_ITEM, N_("Enhance"),
    NULL, (gpointer) filter_exec, "Enhance", NULL,
    GNOME_APP_PIXMAP_NONE, NULL, 0, (GdkModifierType) 0, NULL },
  GNOMEUIINFO_SEPARATOR,
  { GNOME_APP_UI_ITEM, N_("Noise Reduction"),
    NULL, (gpointer) filter_exec, "Noise Reduction", NULL,
    GNOME_APP_PIXMAP_NONE, NULL, 0, (GdkModifierType) 0, NULL },
  GNOMEUIINFO_END
};

static GnomeUIInfo border_uiinfo[] =
{
  { GNOME_APP_UI_ITEM, N_("Wooden-1"),
    NULL, (gpointer) arrange_exec, "Border Wooden-1", NULL,
    GNOME_APP_PIXMAP_NONE, NULL, 0, (GdkModifierType) 0, NULL },
  { GNOME_APP_UI_ITEM, N_("Wooden-2"),
    NULL, (gpointer) arrange_exec, "Border Wooden-2", NULL,
    GNOME_APP_PIXMAP_NONE, NULL, 0, (GdkModifierType) 0, NULL },
  GNOMEUIINFO_SEPARATOR,
  {
   GNOME_APP_UI_ITEM, N_("Wooden color..."),
   NULL, (gpointer) on_border_wooden_color_set_activate, NULL, NULL,
   GNOME_APP_PIXMAP_NONE, NULL, 0, (GdkModifierType) 0, NULL},
  GNOMEUIINFO_SEPARATOR,
  { GNOME_APP_UI_ITEM, N_("Transparent"),
    NULL, (gpointer) arrange_exec, "Border Transparent", NULL,
    GNOME_APP_PIXMAP_NONE, NULL, 0, (GdkModifierType) 0, NULL },
  { GNOME_APP_UI_ITEM, N_("Gradation"),
    NULL, (gpointer) arrange_exec, "Border Gradation", NULL,
    GNOME_APP_PIXMAP_NONE, NULL, 0, (GdkModifierType) 0, NULL },
  { GNOME_APP_UI_ITEM, N_("Shade"),
    NULL, (gpointer) arrange_exec, "Border Shade", NULL,
    GNOME_APP_PIXMAP_NONE, NULL, 0, (GdkModifierType) 0, NULL },
  GNOMEUIINFO_SEPARATOR,
  { GNOME_APP_UI_ITEM, N_("Ripple-1"),
    NULL, (gpointer) arrange_exec, "Border Ripple-1", NULL,
    GNOME_APP_PIXMAP_NONE, NULL, 0, (GdkModifierType) 0, NULL },
  { GNOME_APP_UI_ITEM, N_("Ripple-2"),
    NULL, (gpointer) arrange_exec, "Border Ripple-2", NULL,
    GNOME_APP_PIXMAP_NONE, NULL, 0, (GdkModifierType) 0, NULL },
  GNOMEUIINFO_SEPARATOR,
  { GNOME_APP_UI_ITEM, N_("Random-1"),
    NULL, (gpointer) arrange_exec, "Border Random-1", NULL,
    GNOME_APP_PIXMAP_NONE, NULL, 0, (GdkModifierType) 0, NULL },
  { GNOME_APP_UI_ITEM, N_("Random-2"),
    NULL, (gpointer) arrange_exec, "Border Random-2", NULL,
    GNOME_APP_PIXMAP_NONE, NULL, 0, (GdkModifierType) 0, NULL },
  GNOMEUIINFO_SEPARATOR,
  { GNOME_APP_UI_ITEM, N_("Jagged-1"),
    NULL, (gpointer) arrange_exec, "Border Jagged-1", NULL,
    GNOME_APP_PIXMAP_NONE, NULL, 0, (GdkModifierType) 0, NULL },
  { GNOME_APP_UI_ITEM, N_("Jagged-2"),
    NULL, (gpointer) arrange_exec, "Border Jagged-2", NULL,
    GNOME_APP_PIXMAP_NONE, NULL, 0, (GdkModifierType) 0, NULL },
  GNOMEUIINFO_SEPARATOR,
  {
   GNOME_APP_UI_ITEM, N_("Border color..."),
   NULL, (gpointer) on_border_background_color_set_activate, NULL, NULL,
   GNOME_APP_PIXMAP_NONE, NULL, 0, (GdkModifierType) 0, NULL},
  GNOMEUIINFO_END
};

static GnomeUIInfo glassblock_uiinfo[] =
{
  { GNOME_APP_UI_ITEM, N_("Both Direction"),
    NULL, (gpointer) arrange_exec, "Glassblock", NULL,
    GNOME_APP_PIXMAP_NONE, NULL, 0, (GdkModifierType) 0, NULL },
  GNOMEUIINFO_SEPARATOR,
  { GNOME_APP_UI_ITEM, N_("Horizontal"),
    NULL, (gpointer) arrange_exec, "Glassblock_horizontal", NULL,
    GNOME_APP_PIXMAP_NONE, NULL, 0, (GdkModifierType) 0, NULL },
  { GNOME_APP_UI_ITEM, N_("Vertical"),
    NULL, (gpointer) arrange_exec, "Glassblock_vertical", NULL,
    GNOME_APP_PIXMAP_NONE, NULL, 0, (GdkModifierType) 0, NULL },
  GNOMEUIINFO_END
};

static GnomeUIInfo corrugated_uiinfo[] =
{
  { GNOME_APP_UI_ITEM, N_("Both Direction"),
    NULL, (gpointer) arrange_exec, "Corrugated", NULL,
    GNOME_APP_PIXMAP_NONE, NULL, 0, (GdkModifierType) 0, NULL },
  GNOMEUIINFO_SEPARATOR,
  { GNOME_APP_UI_ITEM, N_("Horizontal"),
    NULL, (gpointer) arrange_exec, "Corrugated_horizontal", NULL,
    GNOME_APP_PIXMAP_NONE, NULL, 0, (GdkModifierType) 0, NULL },
  { GNOME_APP_UI_ITEM, N_("Vertical"),
    NULL, (gpointer) arrange_exec, "Corrugated_vertical", NULL,
    GNOME_APP_PIXMAP_NONE, NULL, 0, (GdkModifierType) 0, NULL },
  GNOMEUIINFO_END
};

static GnomeUIInfo ambiguous_uiinfo[] =
{
  { GNOME_APP_UI_ITEM, N_("Mosaic"),
    NULL, (gpointer) arrange_exec, "Mosaic", NULL,
    GNOME_APP_PIXMAP_NONE, NULL, 0, (GdkModifierType) 0, NULL },
  { GNOME_APP_UI_ITEM, N_("Pixel Vibration"),
    NULL, (gpointer) filter_exec, "Pixel Vibration", NULL,
    GNOME_APP_PIXMAP_NONE, NULL, 0, (GdkModifierType) 0, NULL },
  { GNOME_APP_UI_ITEM, N_("Puzzle"),
    NULL, (gpointer) arrange_exec, "Puzzle", NULL,
    GNOME_APP_PIXMAP_NONE, NULL, 0, (GdkModifierType) 0, NULL },
  GNOMEUIINFO_END
};

static GnomeUIInfo arrange_uiinfo[] =
{
  GNOMEUIINFO_SUBTREE (N_("Glassblock"), glassblock_uiinfo),
  GNOMEUIINFO_SUBTREE (N_("Corrugated"), corrugated_uiinfo),
  GNOMEUIINFO_SUBTREE (N_("Ambiguous"), ambiguous_uiinfo),
  GNOMEUIINFO_SUBTREE (N_("Border"), border_uiinfo),
  GNOMEUIINFO_END
};

static GnomeUIInfo auto_uiinfo[] =
{
  { GNOME_APP_UI_ITEM, N_("Outline"),
    NULL, (gpointer) filter_exec, "Outline", NULL,
    GNOME_APP_PIXMAP_NONE, NULL, 0, (GdkModifierType) 0, NULL },
  GNOMEUIINFO_SEPARATOR,
  { GNOME_APP_UI_ITEM, N_("Auto Draw"),
    NULL, (gpointer) filter_exec, "Auto Draw", NULL,
    GNOME_APP_PIXMAP_NONE, NULL, 0, (GdkModifierType) 0, NULL },
  { GNOME_APP_UI_ITEM, N_("Auto Paint"),
    NULL, (gpointer) filter_exec, "Auto Paint", NULL,
    GNOME_APP_PIXMAP_NONE, NULL, 0, (GdkModifierType) 0, NULL },
  { GNOME_APP_UI_ITEM, N_("b & #"),
    NULL, (gpointer) filter_exec, "b & #", NULL,
    GNOME_APP_PIXMAP_NONE, NULL, 0, (GdkModifierType) 0, NULL },
  GNOMEUIINFO_SEPARATOR,
  { GNOME_APP_UI_ITEM, N_("R-Paint"),
    NULL, (gpointer) filter_exec, "Recursive Paint", NULL,
    GNOME_APP_PIXMAP_NONE, NULL, 0, (GdkModifierType) 0, NULL },
  GNOMEUIINFO_END
};

static GnomeUIInfo cocktail_uiinfo[] =
{
  { GNOME_APP_UI_ITEM, N_("Alice"),
    NULL, (gpointer) bat_exec, "Alice", NULL,
    GNOME_APP_PIXMAP_NONE, NULL, 0, (GdkModifierType) 0, NULL },
  { GNOME_APP_UI_ITEM, N_("Becky"),
    NULL, (gpointer) bat_exec, "Becky", NULL,
    GNOME_APP_PIXMAP_NONE, NULL, 0, (GdkModifierType) 0, NULL },
  { GNOME_APP_UI_ITEM, N_("Chris"),
    NULL, (gpointer) filter_exec, "Chris", NULL,
    GNOME_APP_PIXMAP_NONE, NULL, 0, (GdkModifierType) 0, NULL },
  { GNOME_APP_UI_ITEM, N_("Daisy"),
    NULL, (gpointer) filter_exec, "Daisy", NULL,
    GNOME_APP_PIXMAP_NONE, NULL, 0, (GdkModifierType) 0, NULL },
  { GNOME_APP_UI_ITEM, N_("Elly"),
    NULL, (gpointer) bat_exec, "Elly", NULL,
    GNOME_APP_PIXMAP_NONE, NULL, 0, (GdkModifierType) 0, NULL },
  { GNOME_APP_UI_ITEM, N_("Flora"),
    NULL, (gpointer) bat_exec, "Flora", NULL,
    GNOME_APP_PIXMAP_NONE, NULL, 0, (GdkModifierType) 0, NULL },
  { GNOME_APP_UI_ITEM, N_("Susie"),
    NULL, (gpointer) filter_exec, "Susie", NULL,
    GNOME_APP_PIXMAP_NONE, NULL, 0, (GdkModifierType) 0, NULL },
  GNOMEUIINFO_END
};

static GnomeUIInfo special_uiinfo[] =
{
  { GNOME_APP_UI_ITEM, N_("Sketch"),
    NULL, (gpointer) filter_exec, "Sketch", NULL,
    GNOME_APP_PIXMAP_NONE, NULL, 0, (GdkModifierType) 0, NULL },
  { GNOME_APP_UI_ITEM, N_("Crayon"),
    NULL, (gpointer) filter_exec, "Crayon", NULL,
    GNOME_APP_PIXMAP_NONE, NULL, 0, (GdkModifierType) 0, NULL },
  { GNOME_APP_UI_ITEM, N_("Dessin"),
    NULL, (gpointer) filter_exec, "Dessin", NULL,
    GNOME_APP_PIXMAP_NONE, NULL, 0, (GdkModifierType) 0, NULL },
  { GNOME_APP_UI_ITEM, N_("Cartoon"),
    NULL, (gpointer) filter_exec, "Cartoon", NULL,
    GNOME_APP_PIXMAP_NONE, NULL, 0, (GdkModifierType) 0, NULL },
  { GNOME_APP_UI_ITEM, N_("Watercolor"),
    NULL, (gpointer) filter_exec, "Watercolor", NULL,
    GNOME_APP_PIXMAP_NONE, NULL, 0, (GdkModifierType) 0, NULL },
  { GNOME_APP_UI_ITEM, N_("Oilify"),
    NULL, (gpointer) filter_exec, "Oilify", NULL,
    GNOME_APP_PIXMAP_NONE, NULL, 0, (GdkModifierType) 0, NULL },
  { GNOME_APP_UI_ITEM, N_("Poster"),
    NULL, (gpointer) filter_exec, "Poster", NULL,
    GNOME_APP_PIXMAP_NONE, NULL, 0, (GdkModifierType) 0, NULL },
  GNOMEUIINFO_SEPARATOR,
  GNOMEUIINFO_SUBTREE (N_("Cocktail"), cocktail_uiinfo),
  GNOMEUIINFO_END
};

static GnomeUIInfo filter_menu_uiinfo[] =
{
  GNOMEUIINFO_SUBTREE (N_("Color"), color_uiinfo),
  GNOMEUIINFO_SUBTREE (N_("Level"), level_uiinfo),
  GNOMEUIINFO_SUBTREE (N_("Tone"), tone_uiinfo),
  GNOMEUIINFO_SUBTREE (N_("Arrange"), arrange_uiinfo),
  GNOMEUIINFO_SUBTREE (N_("Auto"), auto_uiinfo),
  GNOMEUIINFO_SEPARATOR,
  GNOMEUIINFO_SUBTREE (N_("Special"), special_uiinfo),
  GNOMEUIINFO_END
};

static GnomeUIInfo help_menu_uiinfo[] =
{
  GNOMEUIINFO_MENU_ABOUT_ITEM (on_about_activate, NULL),
  { GNOME_APP_UI_ITEM, N_("Image Information"),
    NULL, (gpointer) image_info, NULL, NULL,
    GNOME_APP_PIXMAP_NONE, NULL, 0,
    (GdkModifierType) 0, NULL },
  GNOMEUIINFO_END
};

static GnomeUIInfo menubar_uiinfo[] =
{
  GNOMEUIINFO_MENU_FILE_TREE (file_menu_uiinfo),
#ifdef USE_EDIT_MENU
  GNOMEUIINFO_MENU_EDIT_TREE (edit_menu_uiinfo),
#endif
  GNOMEUIINFO_MENU_VIEW_TREE (view_menu_uiinfo),
  GNOMEUIINFO_SUBTREE (N_("Process"), process_menu_uiinfo),
  GNOMEUIINFO_SUBTREE (N_("Filter"), filter_menu_uiinfo),
  GNOMEUIINFO_MENU_HELP_TREE (help_menu_uiinfo),
  GNOMEUIINFO_END
};

static void
da_drag_data_received (GtkWidget *widget, GdkDragContext *context, gint x0, gint y0, GtkSelectionData *selection_data, guint info, guint time)
{
  GList  *it, *uri_list;;
  gchar *filename;
  /*GSList *str_list = NULL;*/

  if (info != TARGET_URI_LIST)
    return;

  uri_list = gnome_vfs_uri_list_parse ((const gchar *) selection_data->data);

  for (it = uri_list; it != NULL; it = it->next) {
    filename = gnome_vfs_uri_to_string (it->data, GNOME_VFS_URI_HIDE_NONE);
    /*str_list = g_slist_prepend (str_list, filename);*/

    if (!strncmp (filename, "file://", 7)) {
      display_image_file (filename + 7);
      break; /* Only 1st one */
    }
  }

  gnome_vfs_uri_list_free (uri_list);
  /*str_list = g_slist_reverse (str_list);*/
}

static void
configure_drop_on (GtkWidget *widget)
{
  static GtkTargetEntry drag_types[] = {
    {"text/uri-list", 0, TARGET_URI_LIST},
    {"_NETSCAPE_URL", 0, TARGET_URL}
  };
  static gint n_drag_types = sizeof (drag_types) / sizeof (drag_types[0]);

  gtk_drag_dest_set (widget,
		     GTK_DEST_DEFAULT_MOTION |
		     GTK_DEST_DEFAULT_HIGHLIGHT |
		     GTK_DEST_DEFAULT_DROP,
		     drag_types, n_drag_types, GDK_ACTION_COPY);

  gtk_signal_connect (GTK_OBJECT (widget), "drag_data_received",
		      GTK_SIGNAL_FUNC (da_drag_data_received), NULL);
}

static void
set_window_icon (GtkWidget *w)
{
  GtkStyle *style = gtk_widget_get_style (w);
  GdkPixmap *icon = gdk_pixmap_create_from_xpm_d (w->window,
						  NULL,
						  &style->bg[GTK_STATE_NORMAL],
						  (gchar **) window_icon_xpm);

  gdk_window_set_icon (w->window, NULL, icon, NULL);
}

/* Entry point */
GtkWidget*
create_app (void)
{
  GtkWidget *bonobodock1;

  app = gnome_app_new ("Mgw", NULL);

  bonobodock1 = GNOME_APP (app)->dock;
  gtk_widget_show (bonobodock1);

  gnome_app_create_menus (GNOME_APP (app), menubar_uiinfo);

  drawing_area = gtk_drawing_area_new ();
  gtk_widget_show (drawing_area);

  gnome_app_set_contents (GNOME_APP (app), drawing_area);

  appbar = gnome_appbar_new (TRUE, TRUE, GNOME_PREFERENCES_NEVER);
  gtk_widget_show (appbar);
  gnome_app_set_statusbar (GNOME_APP (app), appbar);

  gnome_app_install_menu_hints (GNOME_APP (app), menubar_uiinfo);

  menubar_hilite_control (INITIAL);

  /* SIGNAL CONNECTION */
  gtk_signal_connect (GTK_OBJECT (app), "realize",
		      GTK_SIGNAL_FUNC (set_window_icon), NULL);
  gtk_signal_connect (GTK_OBJECT (app), "delete_event",
		      GTK_SIGNAL_FUNC (on_quit_activate), NULL);
  gtk_signal_connect (GTK_OBJECT (app), "key_press_event",
		      GTK_SIGNAL_FUNC (key_press_event), NULL);
  gtk_signal_connect (GTK_OBJECT (app), "key_release_event",
		      GTK_SIGNAL_FUNC (key_release_event), NULL);
  gtk_widget_set_events (app, GDK_KEY_PRESS_MASK | GDK_KEY_RELEASE_MASK);

  gtk_signal_connect (GTK_OBJECT (drawing_area), "expose_event",
		      GTK_SIGNAL_FUNC (on_da_expose_event), NULL);
  gtk_signal_connect (GTK_OBJECT (drawing_area), "configure_event",
		      GTK_SIGNAL_FUNC (on_da_configure_event), NULL);
  gtk_signal_connect (GTK_OBJECT (drawing_area), "button_press_event",
		      GTK_SIGNAL_FUNC (on_da_button_press_event), NULL);
  gtk_signal_connect (GTK_OBJECT (drawing_area), "motion_notify_event",
		      GTK_SIGNAL_FUNC (on_da_motion_notify_event), NULL);
  gtk_signal_connect (GTK_OBJECT (drawing_area), "button_release_event",
		      GTK_SIGNAL_FUNC (on_da_button_release_event), NULL);
  gtk_signal_connect (GTK_OBJECT (drawing_area), "leave_notify_event",
		      GTK_SIGNAL_FUNC (on_da_leave_event), NULL);
  gtk_signal_connect (GTK_OBJECT (drawing_area), "leave_notify_event",
		      GTK_SIGNAL_FUNC (on_da_leave_event), NULL);
  gtk_signal_connect (GTK_OBJECT (drawing_area), "scroll_event",
		      GTK_SIGNAL_FUNC (on_da_scroll_event), NULL);
  gtk_widget_set_events (drawing_area,
			 GDK_EXPOSURE_MASK            |
			 GDK_BUTTON_PRESS_MASK        |
			 GDK_BUTTON_RELEASE_MASK      |
			 GDK_BUTTON_MOTION_MASK       |
			 GDK_POINTER_MOTION_HINT_MASK |
			 GDK_LEAVE_NOTIFY_MASK        |
			 GDK_SCROLL_MASK);

  configure_drop_on (drawing_area);
  gtk_window_resize (GTK_WINDOW (app), 1, 300);

  return app;
}

GtkWidget*
create_about (void)
{
  const gchar *authors[]     = { _("Maegawa Toshio"), NULL };
  const gchar *documenters[] = { NULL };
  gchar *translators         = _("Maegawa Toshio");
  GtkWidget *about;
  GdkPixbuf *icon;

  icon = gdk_pixbuf_new_from_xpm_data ((const char **) window_icon_xpm);
  about = gnome_about_new ("Mgw", VERSION,
			   _("maegawa@jcom.home.ne.jp"),
			   _("Plesae mail me about this program."),
			   authors,
			   documenters,
			   translators,
			   icon);
  gtk_container_set_border_width (GTK_CONTAINER (about), 5);
  gtk_signal_connect (GTK_OBJECT (about), "realize",
		      GTK_SIGNAL_FUNC (set_window_icon), NULL);
  return about;
}

void
change_drawing_area_size (gint width, gint height)
{
  static gboolean first_time = TRUE;

  if (first_time) {
    if (gc_invert_dash == NULL) {
      gc_invert_dash = gdk_gc_new (drawing_area->window);
      gdk_gc_set_function (gc_invert_dash, GDK_INVERT);
      gdk_gc_set_line_attributes (gc_invert_dash, 1,
				  GDK_LINE_ON_OFF_DASH,
				  GDK_CAP_ROUND, GDK_JOIN_ROUND);
    }
    set_cursor (GDK_TOP_LEFT_ARROW);
    first_time = FALSE;
  }

  /* 48 is MAGIC NUMBER */
  gtk_window_resize (GTK_WINDOW (app), width, height + 48);
}

void
set_window_title (gchar *filename, gint w, gint h, gboolean changed)
{
  gchar buff[128];

  if (changed)
    sprintf (buff, "** %s(%dx%d)", filename, w, h);
  else
    sprintf (buff, "%s(%dx%d)", filename, w, h);

  gtk_window_set_title (GTK_WINDOW (app), buff);
}

void
drift_image (GdkPixbuf *pbuf, gint dx, gint dy, gint w, gint h)
{
  gint w0 = drawing_area->allocation.width;
  gint h0 = drawing_area->allocation.height;

  gdk_draw_pixbuf (drawing_area->window,
		   drawing_area->style->fg_gc[GTK_WIDGET_STATE (drawing_area)],
		   pbuf,
		    0,  0,
		   dx, dy,
	            w,  h,
		   GDK_RGB_DITHER_NORMAL, 0, 0);

  if (dy > 0)
    gdk_draw_rectangle (drawing_area->window, drawing_area->style ->bg_gc[GTK_WIDGET_STATE (drawing_area)], 1, 0, 0, w0, dy);
  gdk_draw_rectangle (drawing_area->window, drawing_area->style ->bg_gc[GTK_WIDGET_STATE (drawing_area)], 1, 0, h + dy, w0, h0 - h - dy);
  if (dx > 0)
    gdk_draw_rectangle (drawing_area->window, drawing_area->style ->bg_gc[GTK_WIDGET_STATE (drawing_area)], 1, 0, 0, dx, h0);
  gdk_draw_rectangle (drawing_area->window, drawing_area->style ->bg_gc[GTK_WIDGET_STATE (drawing_area)], 1, w + dx, 0, w0 - w - dx, h0);

}

void
scroll_image_horizontal (GdkPixbuf *pbuf, gint scroll, gint w, gint h)
{
  gint w0 = drawing_area->allocation.width;
  gint h0 = drawing_area->allocation.height;

  scroll = scroll % w;
  if (scroll < 0) scroll = w + scroll;

  gdk_draw_pixbuf (drawing_area->window,
		   drawing_area->style->fg_gc[GTK_WIDGET_STATE (drawing_area)],
		   pbuf,
		                  scroll,            0,
		            (w0 - w) / 2, (h0 - h) / 2,
	                      w - scroll,            h,
		   GDK_RGB_DITHER_NORMAL, 0, 0);
  gdk_draw_pixbuf (drawing_area->window,
		   drawing_area->style->fg_gc[GTK_WIDGET_STATE (drawing_area)],
		   pbuf,
		                       0,            0,
		   (w0 + w) / 2 - scroll, (h0 - h) / 2,
		                  scroll,            h,
		   GDK_RGB_DITHER_NORMAL, 0, 0);
}

void
scroll_image_vertical (GdkPixbuf *pbuf, gint scroll, gint w, gint h)
{
  gint w0 = drawing_area->allocation.width;
  gint h0 = drawing_area->allocation.height;

  scroll = scroll % h;
  if (scroll < 0) scroll = h + scroll;

  gdk_draw_pixbuf (drawing_area->window,
		   drawing_area->style->fg_gc[GTK_WIDGET_STATE (drawing_area)],
		   pbuf,
		              0,                scroll,
		   (w0 - w) / 2,          (h0 - h) / 2,
		              w,            h - scroll,
		   GDK_RGB_DITHER_NORMAL, 0, 0);
  gdk_draw_pixbuf (drawing_area->window,
		   drawing_area->style->fg_gc[GTK_WIDGET_STATE (drawing_area)],
		   pbuf,
		              0,                     0,
		   (w0 - w) / 2, (h0 + h) / 2 - scroll,
		              w,                scroll,
		   GDK_RGB_DITHER_NORMAL, 0, 0);
}

void
redraw_image (GdkPixbuf *pbuf)
{
  gdk_draw_pixbuf (drawing_area->window,
		   drawing_area->style->fg_gc[GTK_WIDGET_STATE (drawing_area)],
		   pbuf, 0, 0,
		   (drawing_area->allocation.width
		    - gdk_pixbuf_get_width (pbuf)) / 2,
		   (drawing_area->allocation.height
		    - gdk_pixbuf_get_height (pbuf)) / 2,
		   gdk_pixbuf_get_width (pbuf),
		   gdk_pixbuf_get_height (pbuf),
		   GDK_RGB_DITHER_NORMAL, 0, 0);
}

void
menubar_hilite_control (MenuStatus status)
{
  switch (status) {
  case INITIAL:
    gtk_widget_set_sensitive (file_menu_uiinfo[1].widget, FALSE); /*save     */
    gtk_widget_set_sensitive (file_menu_uiinfo[2].widget, FALSE); /*save as  */
    gtk_widget_set_sensitive (file_menu_uiinfo[3].widget, FALSE); /*revert   */
    gtk_widget_set_sensitive (menubar_uiinfo[1].widget,   FALSE); /*view     */
    gtk_widget_set_sensitive (menubar_uiinfo[2].widget,   FALSE); /*process  */
    gtk_widget_set_sensitive (menubar_uiinfo[3].widget,   FALSE); /*filters  */
    gtk_widget_set_sensitive (help_menu_uiinfo[1].widget, FALSE); /*ImageInfo*/
#ifdef USE_EDIT_MENU
    gtk_widget_set_sensitive (edit_menu_uiinfo[0].widget, FALSE); /*cut      */
    gtk_widget_set_sensitive (edit_menu_uiinfo[1].widget, FALSE); /*copy     */
    gtk_widget_set_sensitive (edit_menu_uiinfo[2].widget, FALSE); /*paste    */
    gtk_widget_set_sensitive (edit_menu_uiinfo[3].widget, FALSE); /*clear    */
    gtk_widget_set_sensitive (edit_menu_uiinfo[5].widget, FALSE); /*property */
    gtk_widget_set_sensitive (edit_menu_uiinfo[7].widget, FALSE); /*pref.    */
#endif
    break;

  case IMAGE_ON:
    gtk_widget_set_sensitive (file_menu_uiinfo[0].widget, TRUE);  /*open     */
    gtk_widget_set_sensitive (file_menu_uiinfo[1].widget, TRUE);  /*save     */
    gtk_widget_set_sensitive (file_menu_uiinfo[2].widget, TRUE);  /*save as  */
    gtk_widget_set_sensitive (file_menu_uiinfo[3].widget, TRUE);  /*revert   */
    gtk_widget_set_sensitive (file_menu_uiinfo[6].widget, TRUE);  /*exit     */
    gtk_widget_set_sensitive (menubar_uiinfo[1].widget,   TRUE);  /*view     */
    gtk_widget_set_sensitive (menubar_uiinfo[2].widget,   TRUE);  /*process  */
    gtk_widget_set_sensitive (menubar_uiinfo[3].widget,   TRUE);  /*filters  */
    gtk_widget_set_sensitive (help_menu_uiinfo[1].widget, TRUE);  /*ImageInfo*/
    break;

  case LOCK:
    gtk_widget_set_sensitive (file_menu_uiinfo[1].widget, FALSE); /*save     */
    gtk_widget_set_sensitive (file_menu_uiinfo[2].widget, FALSE); /*save as  */
    gtk_widget_set_sensitive (file_menu_uiinfo[3].widget, FALSE); /*revert   */
    gtk_widget_set_sensitive (file_menu_uiinfo[0].widget, FALSE); /*open     */
    gtk_widget_set_sensitive (file_menu_uiinfo[6].widget, FALSE); /*exit     */

    gtk_widget_set_sensitive (view_menu_uiinfo[3].widget, FALSE); /*zoom     */
    gtk_widget_set_sensitive (view_menu_uiinfo[5].widget, FALSE); /*sort     */
    gtk_widget_set_sensitive (menubar_uiinfo[2].widget,   FALSE); /*process  */
    gtk_widget_set_sensitive (menubar_uiinfo[3].widget,   FALSE); /*filters  */
    gtk_widget_set_sensitive (help_menu_uiinfo[1].widget, FALSE); /*ImageInfo*/
    break;

  case UNLOCK:
    gtk_widget_set_sensitive (file_menu_uiinfo[1].widget, TRUE); /*save     */
    gtk_widget_set_sensitive (file_menu_uiinfo[2].widget, TRUE); /*save as  */
    gtk_widget_set_sensitive (file_menu_uiinfo[3].widget, TRUE); /*revert   */
    gtk_widget_set_sensitive (file_menu_uiinfo[0].widget, TRUE); /*open     */
    gtk_widget_set_sensitive (file_menu_uiinfo[6].widget, TRUE); /*exit     */

    gtk_widget_set_sensitive (view_menu_uiinfo[3].widget, TRUE); /*zoom     */
    gtk_widget_set_sensitive (view_menu_uiinfo[5].widget, TRUE); /*sort     */
    gtk_widget_set_sensitive (menubar_uiinfo[2].widget,   TRUE); /*process  */
    gtk_widget_set_sensitive (menubar_uiinfo[3].widget,   TRUE); /*filters  */
    gtk_widget_set_sensitive (help_menu_uiinfo[1].widget, TRUE); /*ImageInfo*/
    break;

  default:
    break;
  }
}

void
put_string_to_appbar (gchar *buff)
{
  /*
  gnome_appbar_clear_stack (GNOME_APPBAR (appbar));
  */
  gnome_appbar_push (GNOME_APPBAR (appbar), buff);
  while (gtk_events_pending ())
    gtk_main_iteration ();
}

void
show_progress (gdouble progress, GdkPixbuf *pbuf)
{
  if (pbuf)
    redraw_image (pbuf);
  gnome_appbar_set_progress_percentage (GNOME_APPBAR (appbar), progress);

  while (gtk_events_pending ())
    gtk_main_iteration ();
}

void
draw_clip_rect (gint x0, gint y0, gint x1, gint y1)
{
  gint x_end, y_end;

  x_end = x0 + x1;
  if (x_end < x0)
    swap (&x_end, &x0);
  x1 = x_end - x0;

  y_end = y0 + y1;
  if (y_end < y0)
    swap (&y_end, &y0);
  y1 = y_end - y0;

  /*if (gc_invert_dash)*/
  gdk_draw_rectangle (drawing_area->window, gc_invert_dash, 0, x0, y0, x1, y1);
}

void
draw_clip_rect_area (void)
{
  MgwImageInfo *i;

  i = get_image_info ();
  if (i->has_clip)
    draw_clip_rect (i->clip.x0, i->clip.y0, i->clip.x1, i->clip.y1);
}

void
set_cursor (gint type)
{
  GdkCursor *cursor;

  cursor = gdk_cursor_new (type);
  gdk_window_set_cursor (drawing_area->window, cursor);
  gdk_cursor_destroy (cursor);
}

void
adjust_window_to_screen_size (void)
{
  GdkPixbuf *p;
  gint sy;

  if (!(p = get_pbuf ()))
    return;
  sy = gdk_screen_height ();

  /* 48 is MAGIC NUMBER */
  gtk_window_resize (GTK_WINDOW (app),
		     gdk_pixbuf_get_width (p),
		     sy - SCREENSIZE_MARGE_Y + 48);
}

GtkWidget *
get_app (void)
{
  return app;
}
