#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <gnome.h>

#include <math.h>

#include "mgwtypes.h"
#include "filters.h"
#include "interface.h"
#include "image.h"
#include "utils.h"

/*
  FUNCTION NAME CHANGE: 0.2.06 --->0.3.xx
  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
  brightness    (x)       -> power_f1                  (x, pbuf);
  contrast      (x)       -> power_f2                  (x, pbuf);
  image_set_gbc (g, b, c) -> contrast_brightness_gamma (c / 256,
                                                        b / 256,
							g / 256,
							pbuf);
  table_filter  (table)   -> color_level_change_by     (table, pbuf);
  autumn_leaves (x)       -> stella_by_starlight       (x, pbuf);
*/

static gboolean break_execution = FALSE;

static void
blur (GdkPixbuf *pbuf)
{
  MgwSizeInfo si;
  gint i, j, k;
  guchar *p, *p0 = gdk_pixbuf_get_pixels (pbuf);
  guchar *tmp;

  get_size_info (&si, pbuf);
  tmp = stock_rgb_data (pbuf, si);

  for (i = 2; i < si.h - 2; i++) {
    for (j = 2; j < si.w - 2; j++) {
      p = p0 + i * si.rs + j * si.nc;
      for (k = 0; k < 3; k++) {
	p[k] =
	  (  tmp[3 * ((i - 2) * si.w + (j - 2)) + k]
	   + tmp[3 * ((i - 2) * si.w + (j - 1)) + k]
	   + tmp[3 * ((i - 2) * si.w +  j     ) + k]
	   + tmp[3 * ((i - 2) * si.w + (j + 1)) + k]
	   + tmp[3 * ((i - 2) * si.w + (j + 2)) + k]

	   + tmp[3 * ((i - 1) * si.w + (j - 2)) + k]
	   + tmp[3 * ((i - 1) * si.w + (j - 1)) + k]
	   + tmp[3 * ((i - 1) * si.w +  j     ) + k]
	   + tmp[3 * ((i - 1) * si.w + (j + 1)) + k]
	   + tmp[3 * ((i - 1) * si.w + (j + 2)) + k]

	   + tmp[3 * ( i      * si.w + (j - 2)) + k]
	   + tmp[3 * ( i      * si.w + (j - 1)) + k]
	   + tmp[3 * ( i      * si.w +  j     ) + k]
	   + tmp[3 * ( i      * si.w + (j + 1)) + k]
	   + tmp[3 * ( i      * si.w + (j + 2)) + k]

	   + tmp[3 * ((i + 1) * si.w + (j - 2)) + k]
	   + tmp[3 * ((i + 1) * si.w + (j - 1)) + k]
	   + tmp[3 * ((i + 1) * si.w +  j     ) + k]
	   + tmp[3 * ((i + 1) * si.w + (j + 1)) + k]
	   + tmp[3 * ((i + 1) * si.w + (j + 2)) + k]

	   + tmp[3 * ((i + 2) * si.w + (j - 2)) + k]
	   + tmp[3 * ((i + 2) * si.w + (j - 1)) + k]
	   + tmp[3 * ((i + 2) * si.w +  j     ) + k]
	   + tmp[3 * ((i + 2) * si.w + (j + 1)) + k]
	   + tmp[3 * ((i + 2) * si.w + (j + 2)) + k]) / 25;
      }
    }
  }
  g_free (tmp);
}

static void
poster_color (GdkPixbuf *pbuf)
{
  MgwSizeInfo si;
  gdouble v;
  gint i, j, k;
  guchar *p, *p0 = gdk_pixbuf_get_pixels (pbuf);

  get_size_info (&si, pbuf);

  for (i = 0; i < si.h; i++) {
    for (j = 0; j < si.w; j++) {
      p = p0 + i * si.rs + j * si.nc;
      v = sqrt (p[0] * p[0] + p[1] * p[1] + p[2] * p[2]);
      p[0] *= 255 / v;
      p[1] *= 255 / v;
      p[2] *= 255 / v;
    }
  }
}

static void
save_black_area (guchar threshold, guchar lev, GdkPixbuf *pbuf, guchar *org)
{
  MgwSizeInfo si;
  gint i, j, k;
  gint op;
  guchar *p, *p0 = gdk_pixbuf_get_pixels (pbuf);

  get_size_info (&si, pbuf);

  for (i = 0; i < si.h; i++) {
    for (j = 0; j < si.w; j++) {
      p = p0 + i * si.rs + j * si.nc;
      op = 3 * (i * si.w + j);
      if (org[op    ] < threshold &&
	  org[op + 1] < threshold &&
	  org[op + 2] < threshold) {
	p[0] = p[1] = p[2] = lev;
      }
    }
  }
}

static void
save_white_area (guchar threshold, guchar lev, GdkPixbuf *pbuf, guchar *org)
{
  MgwSizeInfo si;
  gint i, j, k;
  gint op;
  guchar *p, *p0 = gdk_pixbuf_get_pixels (pbuf);

  get_size_info (&si, pbuf);

  for (i = 0; i < si.h; i++) {
    for (j = 0; j < si.w; j++) {
      p = p0 + i * si.rs + j * si.nc;
      op = 3 * (i * si.w + j);
      if (org[op    ] > threshold &&
	  org[op + 1] > threshold &&
	  org[op + 2] > threshold) {

	p[0] = p[1] = p[2] = lev;
      }
    }
  }
}

static guchar
ca_converter (guchar a, guchar b, guchar c, guchar d, guchar e, guchar f, guchar g, guchar h, guchar i)
{
  gint nbr = 0;

  if (a > e) nbr++;
  if (b > e) nbr++;
  if (c > e) nbr++;
  if (d > e) nbr++;
  if (f > e) nbr++;
  if (g > e) nbr++;
  if (h > e) nbr++;
  if (i > e) nbr++;

  if (nbr < 3)
    if (e > 0)
      return e - 1;
    else
      return e;
  else if (nbr > 3)
    if (e < 255)
      return e + 1;
    else
      return e;
  else
    return e;
}

static void
noise_reduction (gint iteration, GdkPixbuf *pbuf)
{
  MgwSizeInfo si;
  gint i, j, k;
  gint generation;
  gint m1, m2, m3, m4, m5, m6, m7, m8, m9;
  guchar *p, *p0 = gdk_pixbuf_get_pixels (pbuf);
  guchar *tmp = NULL;

  get_size_info (&si, pbuf);

  for (generation = 0; generation < iteration; generation++) {
    if (!(tmp = stock_rgb_data (pbuf, si))) {
      g_warning ("noise_reduction(filters.c): memory allocation failed");
      return;
    }

    for (i = 1; i < si.h - 1; i++) {
      for (j = 1; j < si.w - 1; j++) {
	m5 = 3 * (i * si.w + j);

	m1 = m5 - 3 * si.w - 3;
	m2 = m5 - 3 * si.w;
	m3 = m5 - 3 * si.w + 3;
	m4 = m5 - 3;
	m6 = m5 + 3;
	m7 = m5 + 3 * si.w - 3;
	m8 = m5 + 3 * si.w;
	m9 = m5 + 3 * si.w + 3;

	p = p0 + i * si.rs + j * si.nc;
	for (k = 0; k < 3; k++)
	  p[k] = ca_converter (tmp[m1 + k], tmp[m2 + k], tmp[m3 + k],
			       tmp[m4 + k], tmp[m5 + k], tmp[m6 + k],
			       tmp[m7 + k], tmp[m8 + k], tmp[m9 + k]);
	if ((i * (si.w - 1) + j) % 100000 == 0) {
	  if (break_execution) {
	    g_free (tmp);
	    break_execution = FALSE;
	    return;
	  }
	  show_progress (((gdouble) i / si.h + generation) / iteration, pbuf);
	}
      }
    }
    g_free (tmp);
  }
}

static void
enhance (gdouble ratio, GdkPixbuf *pbuf)
{
  MgwSizeInfo si;
  gdouble laplacian;
  gint i, j, k, p;
  guchar *org = NULL, *p0 = gdk_pixbuf_get_pixels (pbuf);

  get_size_info (&si, pbuf);

  if (!(org = stock_rgb_data (pbuf, si))) {
    g_warning ("enhance(filters.c): memory allocation failed");
    goto FUNC_END;
  }

  for (i = 1; i < si.h - 1; i++) {
    for (j = 1; j < si.w - 1; j++) {
      for (k = 0; k < 3; k++) {
	p = 3 * (i * si.w + j) + k;
	laplacian
	  = org[p - 3 * si.w]
	  + org[p + 3 * si.w]
	  + org[p - 3]
	  + org[p + 3]
	  - 4 * org[p];

	*(p0 + i * si.rs + j * si.nc + k)
	  = limit_byte_range (org[p] - ratio * laplacian);
      }
    }
  }

 FUNC_END:
  g_free (org);
}

static void
edge_detect (gboolean reverse, GdkPixbuf *pbuf)
{
  MgwSizeInfo si;
  gint edge;
  gint i, j, k;
  guchar *org = NULL, *p, *p0 = gdk_pixbuf_get_pixels (pbuf);

  get_size_info (&si, pbuf);

  if (!(org = stock_rgb_data (pbuf, si))) {
    g_warning ("enhance(filters.c): memory allocation failed");
    goto FUNC_END;
  }

  if (reverse) {
    for (i = 1; i < si.h - 1; i++) {
      for (j = 1; j < si.w - 1; j++) {
	p = p0 + i * si.rs + j * si.nc;
	for (k = 0; k < 3; k++){
	  edge =
	    - *(org + ((i - 1) * si.w + (j - 1)) * 3 + k)
	    - *(org + ((i - 1) * si.w +  j     ) * 3 + k)
	    - *(org + ((i - 1) * si.w + (j + 1)) * 3 + k)
	    - *(org + ( i      * si.w + (j - 1)) * 3 + k)
	    + *(org + ( i      * si.w +  j     ) * 3 + k) * 8
	    - *(org + ( i      * si.w + (j + 1)) * 3 + k)
	    - *(org + ((i + 1) * si.w + (j - 1)) * 3 + k)
	    - *(org + ((i + 1) * si.w +  j     ) * 3 + k)
	    - *(org + ((i + 1) * si.w + (j + 1)) * 3 + k);
	  p[k] = ~limit_byte_range (edge);
	}
      }
    }
  } else {
    for (i = 1; i < si.h - 1; i++) {
      for (j = 1; j < si.w - 1; j++) {
	p = p0 + i * si.rs + j * si.nc;
	for (k = 0; k < 3; k++){
	  edge =
	    - *(org + ((i - 1) * si.w + (j - 1)) * 3 + k)
	    - *(org + ((i - 1) * si.w +  j     ) * 3 + k)
	    - *(org + ((i - 1) * si.w + (j + 1)) * 3 + k)
	    - *(org + ( i      * si.w + (j - 1)) * 3 + k)
	    + *(org + ( i      * si.w +  j     ) * 3 + k) * 8
	    - *(org + ( i      * si.w + (j + 1)) * 3 + k)
	    - *(org + ((i + 1) * si.w + (j - 1)) * 3 + k)
	    - *(org + ((i + 1) * si.w +  j     ) * 3 + k)
	    - *(org + ((i + 1) * si.w + (j + 1)) * 3 + k);
	  p[k] = limit_byte_range (edge);
	}
      }
    }
  }

 FUNC_END:
  g_free (org);
}

static void
color_level_change_by (guchar * table, GdkPixbuf *pbuf)
{
  MgwSizeInfo si;
  gint i, j, k;
  guchar *p, *p0 = gdk_pixbuf_get_pixels (pbuf);

  get_size_info (&si, pbuf);

  for (i = 0; i < si.h; i++) {
    for (j = 0; j < si.w; j++) {
      p = p0 + i * si.rs + j * si.nc;
      for (k = 0; k < 3; k++) {
	p[k] = table [p[k]];
      }
    }
  }
}

static void
high_cut (GdkPixbuf *pbuf)
{
  gint i;
  guchar table[256];

  for (i = 0; i < 128; i++)
    table[i] = i;
  for (i = 128; i < 256; i++)
    table[i] = 128 + (i - 128) / 2;

  color_level_change_by (table, pbuf);
}

static void
power_f1 (gdouble p, GdkPixbuf *pbuf)
{
  gint i;
  guchar table[256];

  /* perfect circle (x**p + y**p = 1)
  for (i = 0; i < 256; i++) {
    x = (gdouble) i / 255;
    table[i] = 255 * (1 - pow (1 - pow (x, p), (1.0 / p)));
  }
  */
  for (i = 0; i < 256; i++)
    table[i] = 255 * pow ((gdouble) i / 255, p);

  color_level_change_by (table, pbuf);
}

static void
power_f2 (gdouble p, GdkPixbuf *pbuf)
{
  gint i;
  guchar table[256];

  for (i = 0; i < 128; i++)
    table[i] = 127 * pow ((gdouble) i / 128, p);
  for (i = 128; i < 256; i++)
    table[i] = 255 - table[255 - i];

  color_level_change_by (table, pbuf);
}

static void
contrast_brightness_gamma (gdouble c, gdouble b, gdouble g, GdkPixbuf *pbuf)
{
  gdouble v;
  gint i;
  guchar table[256];

  for (i = 0; i < 256; i++) {
    v = pow ((((gdouble) i / 256.0 - 0.5) * c + 0.5 + b - 1), 1 / g) * 256.0;
    table[i] = limit_byte_range (v);
  }

  color_level_change_by (table, pbuf);
}

static void
reverse (GdkPixbuf *pbuf)
{
  MgwSizeInfo si;
  gint i, j, k;
  guchar *p, *p0 = gdk_pixbuf_get_pixels (pbuf);

  get_size_info (&si, pbuf);

  for (i = 0; i < si.h; i++) {
    for (j = 0; j < si.w; j++) {
      p = p0 + i * si.rs + j * si.nc;
      for (k = 0; k < 3; k++) {
	p[k] = ~p[k];
      }
    }
  }
}

static void
gray_scale (GdkPixbuf *pbuf)
{
  MgwSizeInfo si;
  gdouble ave;
  gint i, j, k;
  guchar *p, *p0 = gdk_pixbuf_get_pixels (pbuf);

  get_size_info (&si, pbuf);

  for (i = 0; i < si.h; i++) {
    for (j = 0; j < si.w; j++) {
      p = p0 + i * si.rs + j * si.nc;
      ave = 0.299 * p[0] + 0.587 * p[1] + 0.114 * p[2];
      for (k = 0; k < 3; k++) {
	p[k] = ave;
      }
    }
  }
}

static void
flat_paint (guchar *p0, guchar *tmp, gint x, gint y, guchar *c, gint w, gint h, gint rs, gint nc)
{
  gboolean xdone = FALSE, ydone = FALSE;
  gint i, j, k, p;
  gint xiteration, yiteration;

  /* You are not expected to understand this */
  yiteration = 0;
  for (j = y; j && !ydone; j--) {
    xiteration = 0;
    for (i = x; i && !xdone; i--) {
      p = 3 * (j * w + i);
      for (k = 0; k < 3; k++) {
	if (abs (tmp[p + k] - c[k]) > AUTO_PAINT_COLOR_RANGE) {
	  if (i == x)
	    ydone = xdone = TRUE;
	  else
	    xdone = TRUE;
	}
      }

      if (++xiteration > AUTO_PAINT_ITERATION)
	xdone = TRUE;

      if (!xdone)
	for (k = 0; k < 3; k++)
	  p0[j * rs + i * nc + k] = c[k];
    }
    xdone = FALSE;

    xiteration = 0;
    for (i = x + 1; i < w && !xdone; i++) {
      p = 3 * (j * w + i);
      for (k = 0; k < 3; k++) {
	if (abs (tmp[p + k] - c[k]) > AUTO_PAINT_COLOR_RANGE) {
	  if (i == x + 1)
	    ydone = xdone = TRUE;
	  else
	    xdone = TRUE;
	}
      }

      if (++xiteration > AUTO_PAINT_ITERATION)
	xdone = TRUE;

      if (!xdone)
	for (k = 0; k < 3; k++)
	  p0[j * rs + i * nc + k] = c[k];
    }
    xdone = FALSE;

    if (++yiteration > AUTO_PAINT_ITERATION)
      ydone = TRUE;
  }

  ydone = xdone = FALSE;

  yiteration = 0;
  for (j = y + 1; j < h && !ydone; j++) {
    xiteration = 0;
    for (i = x; i && !xdone; i--) {
      p = 3 * (j * w + i);
      for (k = 0; k < 3; k++) {
	if (abs (tmp[p + k] - c[k]) > AUTO_PAINT_COLOR_RANGE) {
	  if (i == x)
	    ydone = xdone = TRUE;
	  else
	    xdone = TRUE;
	}
      }

      if (++xiteration > AUTO_PAINT_ITERATION)
	xdone = TRUE;

      if (!xdone)
	for (k = 0; k < 3; k++)
	  p0[j * rs + i * nc + k] = c[k];
    }
    xdone = FALSE;

    xiteration = 0;
    for (i = x + 1; i < w && !xdone; i++) {
      p = 3 * (j * w + i);
      for (k = 0; k < 3; k++) {
	if (abs (tmp[p + k] - c[k]) > AUTO_PAINT_COLOR_RANGE) {
	  if (i == x + 1)
	    ydone = xdone = TRUE;
	  else
	    xdone = TRUE;
	}
      }

      if (++xiteration > AUTO_PAINT_ITERATION)
	xdone = TRUE;

      if (!xdone)
	for (k = 0; k < 3; k++)
	  p0[j * rs + i * nc + k] = c[k];
    }
    xdone = FALSE;

    if (++yiteration > AUTO_PAINT_ITERATION)
      ydone = TRUE;
  }
}

edge_append_bump (GdkPixbuf *pbuf, guchar *draw, MgwSizeInfo si)
{
  gint i, j, k, tmp;
  guchar *p, *p0 = gdk_pixbuf_get_pixels (pbuf);

  for (i = 1; i < si.h; i++) {
    for (j = 1; j < si.w; j++) {
      p = p0 + (i - 1) * si.rs + (j - 1)* si.nc;
      for (k = 0; k < 3; k++) {
	if (draw[3 * (i * si.w + j) + k] < 100) {
	  p[k] = MIN (255, p[k] + 55);
	}
      }
    }
  }
    
  for (i = 0; i < si.h - 1; i++) {
    for (j = 0; j < si.w - 1; j++) {
      p = p0 + (i + 1) * si.rs + (j + 1)* si.nc;
      for (k = 0; k < 3; k++) {
	p[k] *= draw[3 * (i * si.w + j) + k] / 255.0;
      }
    }
  }

  for (i = 0; i < si.h; i++) {
    for (k = 0; k < 3; k++) {
      *(p0 + i * si.rs                      + k) =   0;
      *(p0 + i * si.rs +              si.nc + k) = 128;
      *(p0 + i * si.rs + (si.w - 2) * si.nc + k) = 128;
      *(p0 + i * si.rs + (si.w - 1) * si.nc + k) =   0;
    }
  }

  for (j = 0; j < si.w; j++) {
    for (k = 0; k < 3; k++) {
      *(p0 +                      j * si.nc + k) =   0;
      *(p0 +              si.rs + j * si.nc + k) = 128;
      *(p0 + (si.h - 2) * si.rs + j * si.nc + k) = 128;
      *(p0 + (si.h - 1) * si.rs + j * si.nc + k) =   0;
    }
  }
}

static void
edge_append (GdkPixbuf *pbuf, guchar *draw, MgwSizeInfo si)
{
  gint i, j, k;
  guchar *p0 = gdk_pixbuf_get_pixels (pbuf);

  for (i = 0; i < si.h; i++) {
    for (j = 0; j < si.w; j++) {
      for (k = 0; k < 3; k++) {
	*(p0 + i * si.rs + j * si.nc + k)
	  *= draw[3 * (i * si.w + j) + k] / 255.0;
      }
    }
  }
}

static void
pixel_vibration (gint range, GdkPixbuf *pbuf)
{
  MgwSizeInfo si;
  gint i, j, k, i2, j2;
  guchar *org = NULL, *p, *p0 = gdk_pixbuf_get_pixels (pbuf);

  get_size_info (&si, pbuf);
  if (!(org = stock_rgb_data (pbuf, si))) {
    g_warning ("pixel_vibration(filters.c): memory allocation failed");
    goto FUNC_END;
  }

  for (i = 0; i < si.h; i++) {
    for (j = 0; j < si.w; j++) {
      i2 = i + (1 + range * 2) * (gdouble) random () / RAND_MAX - range;
      j2 = j + (1 + range * 2) * (gdouble) random () / RAND_MAX - range;

      if (i2 < 0) i2 = 0;
      if (si.h - 1 < i2) i2 = si.h - 1;
      if (j2 < 0) j2 = 0;
      if (si.w - 1 < j2) j2 = si.w - 1;

      p = p0 + i * si.rs + j * si.nc;

      for (k = 0; k < 3; k++)
	p[k] = org[(i2 * si.w + j2) * 3 + k];
    }
  }

 FUNC_END:
  g_free (org);
}

static void
auto_draw (gboolean vibrated, gboolean colored, gint nr, GdkPixbuf *pbuf)
{
  noise_reduction (nr, pbuf);
  reverse (pbuf);
  if (vibrated)
    pixel_vibration (1, pbuf);
  edge_detect (TRUE, pbuf);
  power_f1 (2.5, pbuf);
  contrast_brightness_gamma (2.343, 0.391, 1, pbuf);
  gray_scale (pbuf);
  if (!colored)
    gray_scale (pbuf);
}

static void
auto_paint (GdkPixbuf *pbuf)
{
  MgwSizeInfo si;
  gint i, k, ix, iy, rpx, rpy;
  guchar *org = NULL, *p0 = gdk_pixbuf_get_pixels (pbuf);
  guchar rpc[3]; /* Reference Point Color */

  get_size_info (&si, pbuf);
  if (!(org = stock_rgb_data (pbuf, si))) {
    g_warning ("auto_paint(filters.c): memory allocation failed");
    goto BREAK_POINT;
  }

  edge_detect (FALSE, pbuf);

  for (iy = 0; iy < AUTO_PAINT_DIV; iy++) {
    for (ix = 0; ix < AUTO_PAINT_DIV; ix++) {
      rpx = (ix + 1) * si.w / (AUTO_PAINT_DIV + 1);
      rpy = (iy + 1) * si.h / (AUTO_PAINT_DIV + 1);

      for (k = 0; k < 3; k++)
	rpc[k] = org[3 * (rpy * si.w + rpx) + k];

      flat_paint (p0, org, rpx, rpy, rpc, si.w, si.h, si.rs, si.nc);
    }

    if (iy % (AUTO_PAINT_DIV / 25) == 0) {
      if (break_execution) {
	restore_rgb_data (pbuf, org, si);
	goto BREAK_POINT;
      }
      show_progress ((gdouble) (iy + 1) / AUTO_PAINT_DIV, pbuf);
    }
  }
  
 BREAK_POINT:
  g_free (org);
  break_execution = FALSE;
}

static void
stella_by_starlight (gdouble parameter, GdkPixbuf *pbuf)
{
  MgwSizeInfo si;
  gint i, j, k;
  guchar *p0 = gdk_pixbuf_get_pixels (pbuf);
  guchar *p, ave;

  get_size_info (&si, pbuf);

  for (i = 0; i < si.h; i++) {
    for (j = 0; j < si.w; j++) {
      p = p0 + i * si.rs + j * si.nc;
      ave = (p[0] + p[1] + p[2]) / 3;
      for (k = 0; k < 3; k++) {
	p[k] = limit_byte_range (ave + parameter * (p[k] - ave));
      }
    }
  }
}

static void
convert_rgb_to_cmyk (guchar r, guchar g, guchar b, guchar *c, guchar *m, guchar *y, guchar *k)
{
  *c = ~r;
  *m = ~g;
  *y = ~b;

  *k = MIN (*c, MIN (*m, *y));

  *c -= *k;
  *m -= *k;
  *y -= *k;
}

static void
cmy_k0 (gdouble klev, GdkPixbuf *pbuf)
{
  MgwSizeInfo si;
  gint i, j;
  guchar *p, *p0 = gdk_pixbuf_get_pixels (pbuf);
  guchar c, m, y, k;

  get_size_info (&si, pbuf);

  for (i = 0; i < si.h; i++) {
    for (j = 0; j < si.w; j++) {
      p = p0 + i * si.rs + j * si.nc;
      convert_rgb_to_cmyk (p[0], p[1], p[2], &c, &m, &y, &k);
      p[0] = 255 - c - k * klev;
      p[1] = 255 - m - k * klev;
      p[2] = 255 - y - k * klev;
    }
  }
}

static void
impulse (GdkPixbuf *pbuf)
{
  cmy_k0 (0.5, pbuf);
  stella_by_starlight (1.5, pbuf);
  power_f2 (1.5, pbuf);
  power_f1 (2, pbuf);
}

static void
impressions (GdkPixbuf *pbuf)
{
  cmy_k0 (0, pbuf);
  power_f1 (6, pbuf);
}

static void
monotonic (gint range, GdkPixbuf *pbuf)
{
  MgwSizeInfo si;
  gdouble mean, target;
  gint i, j;
  guchar *p, *p0 = gdk_pixbuf_get_pixels (pbuf);
  guchar r, g, b;

  get_size_info (&si, pbuf);

  for (i = 0; i < si.h; i++) {
    for (j = 0; j < si.w; j++) {
      p = p0 + i * si.rs + j * si.nc;

      r = ~p[0];
      g = ~p[1];
      b = ~p[2];

      mean = (r + g + b) / 3.0;
      target = range * (guchar) (mean / range);

      r *= target / mean;
      g *= target / mean;
      b *= target / mean;

      r = MIN (r, 255);
      g = MIN (g, 255);
      b = MIN (b, 255);

      p[0] = ~r;
      p[1] = ~g;
      p[2] = ~b;
    }
  }
}

static void
monotonic_flat (GdkPixbuf *pbuf)
{
  contrast_brightness_gamma (0.5, 1, 1, pbuf);
  stella_by_starlight (2, pbuf);
  monotonic (64, pbuf);
  contrast_brightness_gamma (1.9531, 0.8789, 1, pbuf);
}

static void
sepia (GdkPixbuf *pbuf)
{
  MgwSizeInfo si;
  gdouble ave;
  gint i, j;
  guchar *p, *p0 = gdk_pixbuf_get_pixels (pbuf);

  get_size_info (&si, pbuf);

  for (i = 0; i < si.h; i++) {
    for (j = 0; j < si.w; j++) {
      p = p0 + i * si.rs + j * si.nc;
      ave = 0.299 * p[0] + 0.587 * p[1] + 0.114 * p[2];
      p[0] = ave * 0.94;
      p[1] = ave * 0.78;
      p[2] = ave * 0.54;
    }
  }
}

static void
super_nova (GdkPixbuf *pbuf)
{
  power_f1 (0.5, pbuf);
  stella_by_starlight (1.5, pbuf);
}

static void
flat_and_sharp (GdkPixbuf *pbuf)
{
  MgwSizeInfo si;
  gint i, j, k;
  guchar *current = NULL, *p0 = gdk_pixbuf_get_pixels (pbuf);

  get_size_info (&si, pbuf);
  if (!(current = stock_rgb_data (pbuf, si))) {
    g_warning ("flat_and_sharp(filters.c): memory allocation failed");
    goto FUNC_END;
  }

  auto_draw (FALSE, TRUE, 10, pbuf);

  for (i = 0; i < si.h; i++) {
    for (j = 0; j < si.w; j++) {
      for (k = 0; k < 3; k++) {
	*(p0 + i * si.rs + j * si.nc + k) = *(p0 + i * si.rs + j * si.nc + k) / 2 + current[(i * si.w + j) * 3 + k] / 2;
      }
    }
  }
  contrast_brightness_gamma (2, 0.586, 1, pbuf); redraw_image (pbuf);
  monotonic (25, pbuf);                          redraw_image (pbuf);
  noise_reduction (20, pbuf);                    redraw_image (pbuf);

 FUNC_END:
  g_free (current);
}

static void
mesh_auto_level (gint shift_level, GdkPixbuf *pbuf)
{
  MgwSizeInfo si;
  gdouble ave;
  gint x, y, dx, dy, xs, xe, ys, ye;
  gint i, j, k, count;
  guchar *p, *p0 = gdk_pixbuf_get_pixels (pbuf);

  get_size_info (&si, pbuf);

  dx = dy = MESH_AUTO_LEVEL_DIV;

  y = 0;
  while (y + dy - 1 < si.h) {
    ys =  y;
    ye =  y + dy - 1;

    x = 0;
    while (x + dx - 1 < si.w) {
      xs =  x;
      xe =  x + dx - 1;

      ave = 0.0;
      count = 0;

      for (i = ys; i <= ye; i++)
	for (j = xs; j <= xe; j++)
	  for (k = 0; k < 3; k++) {
	    ave += (gdouble) *(p0 + i * si.rs + j * si.nc + k);
	    count++;
	  }

      ave /= count;
      ave -= shift_level;

      for (i = ys; i <= ye; i++)
	for (j = xs; j <= xe; j++)
	  for (k = 0; k < 3; k++) {
	    p = p0 + i * si.rs + j * si.nc + k;
	    *p = limit_byte_range (*p - ave);
	  }
      x += dx;
    }
    y += dy;
  }
}

static void
black_expansion (gint threshold, gint center, gint udlr, gint x, GdkPixbuf *pbuf)
{
  /*
    If pixel RGB-level is less than 100, set Moore Neighborhood value as...
     ------------------------
    |   X   |   Up   |   X   | center = value of CENTER
    |------------------------|   udlr = value of neighbor-Up, Down, Left, Right
    |  Left | CENTER | Right |      x = value of neighbor-X
    |------------------------|
    |   X   |  Down  |   X   |
     ------------------------
  */
  MgwSizeInfo si;
  gint i, j, k, op;
  guchar *org = NULL, *p, *p0 = gdk_pixbuf_get_pixels (pbuf);

  get_size_info (&si, pbuf);
  if (!(org = stock_rgb_data (pbuf, si))) {
    g_warning ("black_expansion(filters.c): memory allocation failed");
    goto FUNC_END;
  }

  for (i = 1; i < si.h - 1; i++) {
    for (j = 1; j < si.w - 1; j++) {
      p = p0 + i * si.rs + j * si.nc;
      op = (i * si.w + j) * 3;
      if (org[op    ] < threshold &&
	  org[op + 1] < threshold &&
	  org[op + 2] < threshold) {
	for (k = 0; k < 3; k++) {
	  if (x      != -1) p[k - si.rs - si.nc] = x;
	  if (udlr   != -1) p[k - si.rs        ] = udlr;
	  if (x      != -1) p[k - si.rs + si.nc] = x;
	  if (udlr   != -1) p[k         - si.nc] = udlr;
	  if (center != -1) p[k                ] = center;
	  if (udlr   != -1) p[k         + si.nc] = udlr;
	  if (x      != -1) p[k + si.rs - si.nc] = x;
	  if (udlr   != -1) p[k + si.rs        ] = udlr;
	  if (x      != -1) p[k + si.rs + si.nc] = x;
	}
      }
    }
  }

 FUNC_END:
  g_free (org);
}

static void
white_expansion (gint threshold, gint center, gint udlr, gint x, GdkPixbuf *pbuf)
{
  MgwSizeInfo si;
  gint i, j, k, op;
  guchar *org = NULL, *p, *p0 = gdk_pixbuf_get_pixels (pbuf);

  get_size_info (&si, pbuf);
  if (!(org = stock_rgb_data (pbuf, si))) {
    g_warning ("white_expansion(filters.c): memory allocation failed");
    goto FUNC_END;
  }

  for (i = 1; i < si.h - 1; i++) {
    for (j = 1; j < si.w - 1; j++) {
      p = p0 + i * si.rs + j * si.nc;
      op = (i * si.w + j) * 3;
      if (org[op    ] > threshold &&
	  org[op + 1] > threshold &&
	  org[op + 2] > threshold) {
	for (k = 0; k < 3; k++) {
	  if (x      != -1) p[k - si.rs - si.nc] = x;
	  if (udlr   != -1) p[k - si.rs        ] = udlr;
	  if (x      != -1) p[k - si.rs + si.nc] = x;
	  if (udlr   != -1) p[k         - si.nc] = udlr;
	  if (center != -1) p[k                ] = center;
	  if (udlr   != -1) p[k         + si.nc] = udlr;
	  if (x      != -1) p[k + si.rs - si.nc] = x;
	  if (udlr   != -1) p[k + si.rs        ] = udlr;
	  if (x      != -1) p[k + si.rs + si.nc] = x;
	}
      }
    }
  }

 FUNC_END:
  g_free (org);
}

static guchar
median (guchar a, guchar b, guchar c, guchar d, guchar e, guchar f, guchar g, guchar h, guchar i)
{
  gint m, n;
  gint p[9];

  p[0] = a;
  p[1] = b;
  p[2] = c;
  p[3] = d;
  p[4] = e;
  p[5] = f;
  p[6] = g;
  p[7] = h;
  p[8] = i;

  for (m = 0; m < 5; m++)
    for (n = m + 1; n < 9; n++)
      if (p[m] > p[n])
	swap (p + m, p + n);

  return p[4];
}

static void
softning (GdkPixbuf *pbuf)
{
  MgwSizeInfo si;
  gint i, j, k;
  guint m1, m2, m3, m4, m5, m6, m7, m8, m9;
  guchar *org = NULL, *p0 = gdk_pixbuf_get_pixels (pbuf);

  get_size_info (&si, pbuf);
  if (!(org = stock_rgb_data (pbuf, si))) {
    g_warning ("softning(filters.c): memory allocation failed");
    goto FUNC_END;
  }

  for (i = 1; i < si.h - 1; i++) {
    for (j = 1; j < si.w - 1; j++) {
      m5 = 3 * (i * si.w + j);
      m1 = m5 - 3 * si.w - 3;
      m2 = m5 - 3 * si.w;
      m3 = m5 - 3 * si.w + 3;
      m4 = m5 - 3;
      m6 = m5 + 3;
      m7 = m5 + 3 * si.w - 3;
      m8 = m5 + 3 * si.w;
      m9 = m5 + 3 * si.w + 3;
      for (k = 0; k < 3; k++) {
	p0[i * si.rs + j * si.nc + k] =
	  median (*(org + m1 + k), *(org + m2 + k), *(org + m3 + k),
		  *(org + m4 + k), *(org + m5 + k), *(org + m6 + k),
		  *(org + m7 + k), *(org + m8 + k), *(org + m9 + k));
      }
    }

    if (!(i % 100))
      show_progress ((gdouble) i / si.h, pbuf);
  }

 FUNC_END:
  g_free (org);
}

static void
black_white (GdkPixbuf *pbuf)
{
  MgwSizeInfo si;
  gint i, j, n, count;
  gint freq[256];
  gint threshold = 128, sum = 0;
  guchar *gs = NULL;
  guchar *p, *p0 = gdk_pixbuf_get_pixels (pbuf);

  get_size_info (&si, pbuf);
  n = si.w * si.h;
  if (!(gs = (guchar *) g_malloc (sizeof (guchar) * n))) {
    g_warning ("black_white(filters.c): memory allocation failed");
    goto FUNC_END;
  }

  count = 0;
  for (i = 0; i < si.h; i++) {
    for (j = 0; j < si.w; j++) {
      p = p0 + i * si.rs + j * si.nc;
      gs[count] =
	0.299 * (gdouble) p[0] + 0.587 * (gdouble) p[1] + 0.114 * (gdouble) p[2];
      count++;
    }
  }

  for (i = 0; i < 256; i++)
    freq[i] = 0;

  for (i = 0; i < 256; i++) {
    sum += freq[i];
    if (sum > n / 2) {
      threshold = i;
      break;
    }
  }

  count = 0;
  for (i = 0; i < si.h; i++) {
    for (j = 0; j < si.w; j++) {
      p = p0 + i * si.rs + j * si.nc;
      if (gs[count] < threshold)
	p[0] = p[1] = p[2] = 0;
      else
	p[0] = p[1] = p[2] = 255;
      count++;
    }
  }

 FUNC_END:
  g_free (gs);
}

sketch (GdkPixbuf *pbuf)
{
  MgwSizeInfo si;
  guchar *current, *draw;

  get_size_info (&si, pbuf);
  power_f1 (0.667, pbuf);
  current = stock_rgb_data (pbuf, si);
  auto_draw (FALSE, TRUE, 10, pbuf);                show_progress (0.10, pbuf);
  draw = stock_rgb_data (pbuf, si);                 show_progress (0.20, pbuf);
  restore_rgb_data (pbuf, current, si);             show_progress (0.40, pbuf);
  monotonic_flat (pbuf);                            show_progress (0.50, pbuf);
  power_f1 (1.5, pbuf);                             show_progress (0.60, pbuf);
  edge_append (pbuf, draw, si);                     show_progress (0.70, pbuf);
  edge_append_bump (pbuf, draw, si);                show_progress (0.80, pbuf);
  contrast_brightness_gamma (1, 1.05, 1, pbuf);     show_progress (0.90, pbuf);
  stella_by_starlight (1.5, pbuf);                  show_progress (1.00, pbuf);

  g_free (current);
  g_free (draw);
}

static void
set_core_white (gint radius, guchar *p, gint rs, gint nc)
{
  gint i, k;

  for (i = 0; i < 2 * radius + 1; i++) {
    for (k = 0; k < 3; k++) {
      *(p - radius * (nc + rs) + i * nc + k) = 255;
      *(p - radius * (nc - rs) + i * nc + k) = 255;
    }
  }

  for (i = 0; i < 2 * radius - 1; i++) {
    for (k = 0; k < 3; k++) {
      *(p - radius * (rs + nc) + (i + 1) * rs + k) = 255;
      *(p - radius * (rs - nc) + (i + 1) * rs + k) = 255;
    }
  }
}

static gboolean
check_border_white (gint radius, guchar *p, gint rs, gint nc)
{
  gint i;

  for (i = 0; i < 2 * radius + 1; i++) {
    if (*(p - radius * (nc + rs) + i * nc) != 255)
      return FALSE;
    if (*(p - radius * (nc - rs) + i * nc) != 255)
      return FALSE;
  }

  for (i = 0; i < 2 * radius - 1; i++) {
    if (*(p - radius * (rs + nc) + (i + 1) * rs) != 255)
      return FALSE;
    if (*(p - radius * (rs - nc) + (i + 1) * rs) != 255)
      return FALSE;
  }

  return TRUE;
}

static void
remove_alone_pixel (GdkPixbuf *pbuf)
{
  MgwSizeInfo si;
  gint i, j, k;
  guchar *p, *p0 = gdk_pixbuf_get_pixels (pbuf);

  get_size_info (&si, pbuf);

  for (i = 3; i < si.h - 3; i++) {
    for (j = 3; j < si.w - 3; j++) {
      p = p0 + i * si.rs + j * si.nc;
      if (check_border_white (1, p, si.rs, si.nc)) {
	p[0] = p[1] = p[2] = 255;
      }
      if (check_border_white (2, p, si.rs, si.nc)) {
	set_core_white (1, p, si.rs, si.nc);
      }
      if (check_border_white (3, p, si.rs, si.nc)) {
	set_core_white (2, p, si.rs, si.nc);
	set_core_white (1, p, si.rs, si.nc);
	p[0] = p[1] = p[2] = 255;
      }
      /*
      if (check_border_white (4, p, si.rs, si.nc)) {
	set_core_white (3, p, si.rs, si.nc);
	set_core_white (2, p, si.rs, si.nc);
	set_core_white (1, p, si.rs, si.nc);
	p[0] = p[1] = p[2] = 255;
      }
      */
    }
  }
}

static void
outline (GdkPixbuf *pbuf, gboolean expand, gint nr)
{
  auto_draw (FALSE, TRUE, nr, pbuf);
  enhance (1.0, pbuf);
  black_white (pbuf);
  remove_alone_pixel (pbuf);
  if (expand)
    black_expansion (100, 0, 0, -1, pbuf);
}

static void
caricature (gdouble x, gdouble y, GdkPixbuf *pbuf)
{
  impulse (pbuf);                                   show_progress (0.50, pbuf);
  stella_by_starlight (x, pbuf);                    show_progress (0.75, pbuf);
  power_f2 (y, pbuf);                               show_progress (1.00, pbuf);
}

static void
cartoon (GdkPixbuf *pbuf)
{
  flat_and_sharp (pbuf);                            show_progress (0.20, pbuf);
  high_cut (pbuf);                                  show_progress (0.40, pbuf);
  monotonic (64, pbuf);                             show_progress (0.60, pbuf);
  contrast_brightness_gamma (2, 0.70, 2, pbuf);     show_progress (0.80, pbuf);
  stella_by_starlight (2.25, pbuf);                 show_progress (1.00, pbuf);
}

static void
poster (GdkPixbuf *pbuf)
{
  MgwSizeInfo si;
  guchar *current, *draw;

  get_size_info (&si, pbuf);
  current = stock_rgb_data (pbuf, si);
  auto_draw (FALSE, TRUE, 10, pbuf);
  draw = stock_rgb_data (pbuf, si);
  restore_rgb_data (pbuf, current, si);
  poster_color (pbuf);                              show_progress (0.20, pbuf);
  power_f2 (3, pbuf);                               show_progress (0.40, pbuf);
  monotonic_flat (pbuf);                            show_progress (0.60, pbuf);
  edge_append (pbuf, draw, si);                     show_progress (0.80, pbuf);
  save_black_area (64, 64, pbuf, current);          show_progress (1.00, pbuf);

  g_free (current);
  g_free (draw);
}

static void
dessin_restore_color (GdkPixbuf *pbuf, guchar *org, MgwSizeInfo si)
{
  gint i, j, k;
  guchar *p0 = gdk_pixbuf_get_pixels (pbuf);

  for (i = 0; i < si.h; i++) {
    for (j = 0; j < si.w; j++) {
      for (k = 0; k < 3; k++) {
	*(p0 + i * si.rs + j * si.nc + k)
	  = (*(p0 + i * si.rs + j * si.nc + k) + org[(i * si.w + j) * 3 + k]) / 2;
      }
    }
  }
}

static void
dessin (GdkPixbuf *pbuf)
{
  MgwSizeInfo si;
  guchar *flat, *org;

  get_size_info (&si, pbuf);
  org = stock_rgb_data (pbuf, si);
  monotonic_flat (pbuf);
  flat = stock_rgb_data (pbuf, si);
  restore_rgb_data (pbuf, org, si);

  auto_draw (TRUE, FALSE, 10, pbuf);
  dessin_restore_color (pbuf, flat, si);
  power_f2 (1.5, pbuf);
  power_f1 (2, pbuf);

  g_free (flat);
  g_free (org);
}

static void
watercolor (GdkPixbuf *pbuf)
{
  flat_and_sharp (pbuf);
  auto_paint (pbuf);
  softning (pbuf);
  enhance (0.5, pbuf);
  stella_by_starlight (1.5, pbuf);
}

static void
crayon (GdkPixbuf *pbuf)
{
  MgwSizeInfo si;
  guchar *current, *draw;

  get_size_info (&si, pbuf);
  current = stock_rgb_data (pbuf, si);
  auto_draw (FALSE, TRUE, 10, pbuf);

  draw = stock_rgb_data (pbuf, si);                 show_progress (0.20, pbuf);
  restore_rgb_data (pbuf, current, si);             show_progress (0.40, pbuf);
  mesh_auto_level (127, pbuf);                      show_progress (0.50, pbuf);
  monotonic (64, pbuf);                             show_progress (0.60, pbuf);
  power_f2 (4, pbuf);                               show_progress (0.65, pbuf);
  pixel_vibration (5, pbuf);                        show_progress (0.70, pbuf);
  edge_append (pbuf, draw, si);                     show_progress (0.75, pbuf);
  edge_append_bump (pbuf, draw, si);                show_progress (0.80, pbuf);
  stella_by_starlight (1.5, pbuf);                  show_progress (0.85, pbuf);
  contrast_brightness_gamma (1, 1.1, 1, pbuf);      show_progress (0.90, pbuf);
  save_black_area (64, 64, pbuf, current);          show_progress (0.95, pbuf);
  save_white_area (240, 240, pbuf, current);        show_progress (1.00, pbuf);
  
  g_free (current);
  g_free (draw);
}

static void
daisy (GdkPixbuf *pbuf)
{
  MgwSizeInfo si;
  guchar *current, *edge;

  get_size_info (&si, pbuf);
  power_f1 (0.667, pbuf);
  current = stock_rgb_data (pbuf, si);
  auto_draw (FALSE, TRUE, 10, pbuf);
  black_expansion (100, 0, 0, -1, pbuf);            show_progress (0.10, pbuf);
  edge = stock_rgb_data (pbuf, si);                 show_progress (0.20, pbuf);
  restore_rgb_data (pbuf, current, si);             show_progress (0.30, pbuf);
  pixel_vibration (10, pbuf);                       show_progress (0.40, pbuf);
  edge_append (pbuf, edge, si);                     show_progress (0.50, pbuf);
  edge_append_bump (pbuf, edge, si);                show_progress (0.60, pbuf);
  monotonic_flat (pbuf);                            show_progress (0.70, pbuf);
  power_f1 (1.5, pbuf);                             show_progress (0.90, pbuf);
  power_f2 (1.5, pbuf);                             show_progress (1.00, pbuf);

  g_free (current);
  g_free (edge);
}

static void
susie (GdkPixbuf *pbuf)
{
  impulse (pbuf);
  noise_reduction (30, pbuf);
  softning (pbuf);
  stella_by_starlight (0.667, pbuf);
  power_f2 (1.5, pbuf);
}

static void
infinity (GdkPixbuf *pbuf)
{
  power_f1 (3.0, pbuf);
  reverse (pbuf);
  impressions (pbuf);
  reverse (pbuf);
  super_nova (pbuf);
}

static void
recursive (guchar *p0, gint i, gint j, MgwSizeInfo si, gboolean *checked, GdkColor c0, gint count)
{
  if (i < 0 || si.h - 1 < i || j < 0 || si.w - 1 < j || checked[i * si.w + j])
    return;

  if ((abs (p0[i * si.rs + j * si.nc    ] - c0.red  ) > RECURSIVE_THRESHOLD) ||
      (abs (p0[i * si.rs + j * si.nc + 1] - c0.green) > RECURSIVE_THRESHOLD) ||
      (abs (p0[i * si.rs + j * si.nc + 2] - c0.blue ) > RECURSIVE_THRESHOLD))
    return;

  count++;
  if (RECURSIVE_MAX_ITERATION < count)
    return;

  checked[i * si.w + j] = TRUE; 

  recursive (p0, i - 1, j - 1, si, checked, c0, count);
  recursive (p0, i - 1, j,     si, checked, c0, count);
  recursive (p0, i - 1, j + 1, si, checked, c0, count);

  recursive (p0, i,     j - 1, si, checked, c0, count);
  recursive (p0, i,     j + 1, si, checked, c0, count);

  recursive (p0, i + 1, j - 1, si, checked, c0, count);
  recursive (p0, i + 1, j,     si, checked, c0, count);
  recursive (p0, i + 1, j + 1, si, checked, c0, count);

  p0[i * si.rs + j * si.nc    ] = c0.red;
  p0[i * si.rs + j * si.nc + 1] = c0.green;
  p0[i * si.rs + j * si.nc + 2] = c0.blue;
}

static void
recursive_paint (GdkPixbuf *pbuf)
{
  MgwSizeInfo si;
  GdkColor c;
  gint i, j;
  guchar *p0 = gdk_pixbuf_get_pixels (pbuf);
  gboolean *checked = NULL;

  get_size_info (&si, pbuf);

  if (!(checked = (gboolean *) g_malloc0 (sizeof (gboolean) * si.h * si.w))) {
    g_warning ("recursive_paint(filters.c): memory allocation failed");
    goto FUNC_END;
  }

  for (i = 0; i < si.h; i++) {
    for (j = 0; j < si.w; j++) {
      c.red   = p0[i * si.rs + j * si.nc    ];
      c.green = p0[i * si.rs + j * si.nc + 1];
      c.blue  = p0[i * si.rs + j * si.nc + 2];
      recursive (p0, i, j, si, checked, c, 0);
    }
  }

 FUNC_END:
  g_free (checked);
}

static void
chris (GdkPixbuf *pbuf)
{
  crayon (pbuf);
  blur (pbuf);
  recursive_paint (pbuf);
  softning (pbuf);
  enhance (1.0, pbuf);
}

static void
oilify (GdkPixbuf *pbuf)
{
  recursive_paint (pbuf);                           show_progress (0.33, pbuf);
  pixel_vibration (1, pbuf);                        show_progress (0.66, pbuf);
  softning (pbuf);                                  show_progress (0.99, pbuf);
  auto_paint (pbuf);
  noise_reduction (10, pbuf);
  impulse (pbuf);                                   show_progress (0.50, pbuf);
  stella_by_starlight (0.667, pbuf);                show_progress (1.00, pbuf);
}

static void
nature (GdkPixbuf *pbuf)
{
  monotonic_flat (pbuf);
  power_f1 (1.5, pbuf);
  power_f2 (2, pbuf);
}

static void
pale (GdkPixbuf *pbuf)
{
  power_f1 (0.667, pbuf);
  stella_by_starlight (0.3, pbuf);
  power_f2 (2, pbuf);
  power_f1 (2, pbuf);
}

/* Entry point */
void
filter_break_execution (void)
{
  break_execution = TRUE;
}

void
filter_exec (GtkWidget *dummy, gchar *name)
{
  MgwImageInfo *mii = get_image_info ();
  MgwClipRect c;
  GdkPixbuf *pbuf, *org;
  gdouble zoom_factor;
  gchar buff[64];
  gboolean has_clip, now_executing = FALSE, fail = FALSE;

  if (mii->handling)
    return;
  else
    mii->handling = TRUE;

  if (!(org = get_pbuf ()))
    return;

  set_cursor (GDK_WATCH);
  menubar_hilite_control (LOCK);
  sprintf (buff, "%s: EXECUTING...", name);
  put_string_to_appbar (buff);

  if (has_clip = get_clip_rect_area (&c)) {
    pbuf = gdk_pixbuf_new (GDK_COLORSPACE_RGB, FALSE, 8, c.x1, c.y1);
    gdk_pixbuf_copy_area (org, c.x0, c.y0, c.x1, c.y1, pbuf, 0, 0);
  } else {
    pbuf = org;
  }

  zoom_factor = mii->zoom;
  view_zoom (NULL, "100%");

  switch (name[0]) {
  case 'b':
    if (!strcmp (name, "b & #"))
      flat_and_sharp (pbuf);
    else fail = TRUE;
    break;

  case 'A':
    if (!strcmp (name, "Autumn Leaves"))
      stella_by_starlight (0.667, pbuf);
    else if (!strcmp (name, "Auto Draw"))
      auto_draw (FALSE, TRUE, 10, pbuf);
    else if (!strcmp (name, "Auto Paint"))
      auto_paint (pbuf);
    else if (!strcmp (name, "Anti Sepia")) {
      reverse (pbuf);
      sepia (pbuf);
      reverse (pbuf);
    }
    else fail = TRUE;
    break;

  case 'B':
    if (!strcmp (name, "Black/White"))
      black_white (pbuf);
    else if (!strcmp (name, "Brightness-On"))
      contrast_brightness_gamma (1, 1.1, 1, pbuf);
    else if (!strcmp (name, "Brightness-Off"))
      contrast_brightness_gamma (1, 0.9, 1, pbuf);
    else if (!strcmp (name, "Black Expansion"))
      black_expansion (100, 0, 0, -1, pbuf);
    else if (!strcmp (name, "Brightness-Open"))
      contrast_brightness_gamma (1, 1.1, 1, pbuf);
    else if (!strcmp (name, "Blur"))
      blur (pbuf);
    else fail = TRUE;
    break;

  case 'C':
    if (!strcmp (name, "Contrast-On"))
      contrast_brightness_gamma (1.25, 1, 1, pbuf);
    else if (!strcmp (name, "Contrast-Off"))
      contrast_brightness_gamma (0.75, 1, 1, pbuf);
    else if (!strcmp (name, "Caricature"))
      caricature (0.5, 2, pbuf);
    else if (!strcmp (name, "Cartoon"))
      cartoon (pbuf);
    else if (!strcmp (name, "Crayon"))
      crayon (pbuf);
    else if (!strcmp (name, "Chris"))
      chris (pbuf);
    else fail = TRUE;
    break;

  case 'D':
    if (!strcmp (name, "Dessin"))
      dessin (pbuf);
    else if (!strcmp (name, "Daisy"))
      daisy (pbuf);
    else fail = TRUE;
    break;

  case 'E':
    if (!strcmp (name, "Enhance"))
      enhance (0.5, pbuf);
    else if (!strcmp (name, "Enhanced Black/White")) {
      enhance (1.0, pbuf);
      black_white (pbuf);
    }
    else fail = TRUE;
    break;

  case 'G':
    if (!strcmp (name, "Gray Scale"))
      gray_scale (pbuf);
    else fail = TRUE;
    break;

  case 'I':
    if (!strcmp (name, "Impulse"))
      impulse (pbuf);
    else if (!strcmp (name, "Impressions"))
      impressions (pbuf);
    else if (!strcmp (name, "Infinity"))
      infinity (pbuf);
    else fail = TRUE;
    break;

  case 'M':
    if (!strcmp (name, "Monotonic"))
      monotonic_flat (pbuf);
    else if (!strcmp (name, "Mesh Auto-level")) {
      mesh_auto_level (255, pbuf);
      power_f1 (20, pbuf);
    }
    else fail = TRUE;
    break;

  case 'N':
    if (!strcmp (name, "Noise Reduction"))
      noise_reduction (10, pbuf);
    else if (!strcmp (name, "Nature"))
      nature (pbuf);
    else fail = TRUE;
    break;

  case 'O':
    if (!strcmp (name, "Oilify"))
      oilify (pbuf);
    else if (!strcmp (name, "Outline"))
      outline (pbuf, TRUE, 10);
    else if (!strcmp (name, "Outline(for PostScriptize)"))
      outline (pbuf, FALSE, 10);
    else fail = TRUE;
    break;

  case 'P':
    if (!strcmp (name, "Pixel Vibration"))
      pixel_vibration (5, pbuf);
    else if (!strcmp (name, "Poster"))
      poster (pbuf);
    else if (!strcmp (name, "Pow-Brightness-On"))
      power_f1 (0.667, pbuf);
    else if (!strcmp (name, "Pow-Brightness-Off"))
      power_f1 (1.5, pbuf);
    else if (!strcmp (name, "Pow-Contrast-On"))
      power_f2 (1.5, pbuf);
    else if (!strcmp (name, "Pow-Contrast-Off"))
      power_f2 (0.667, pbuf);
    else if (!strcmp (name, "Pale"))
      pale (pbuf);
    else fail = TRUE;
    break;

  case 'R':
    if (!strcmp (name, "Reverse"))
      reverse (pbuf);
    else if (!strcmp (name, "Recursive Paint"))
      recursive_paint (pbuf);
    else fail = TRUE;
    break;

  case 'S':
    if (!strcmp (name, "Sepia"))
      sepia (pbuf);
    else if (!strcmp (name, "Stella by Starlight"))
      stella_by_starlight (1.5, pbuf);
    else if (!strcmp (name, "Sunshine"))
      contrast_brightness_gamma (1.25, 1.25, 1, pbuf);
    else if (!strcmp (name, "Super Nova"))
      super_nova (pbuf);
    else if (!strcmp (name, "Softning"))
      softning (pbuf);
    else if (!strcmp (name, "Sketch"))
      sketch (pbuf);
    else if (!strcmp (name, "Susie"))
      susie (pbuf);
    else fail = TRUE;
    break;

  case 'W':
    if (!strcmp (name, "Watercolor"))
      watercolor (pbuf);
    else if (!strcmp (name, "White Expansion"))
      white_expansion (200, 255, 255, -1, pbuf);
    else fail = TRUE;
    break;

  default:
    fail = TRUE;
    break;
  }

  if (!fail) {
    if (has_clip) {
      gdk_pixbuf_copy_area (pbuf, 0, 0, c.x1, c.y1, org, c.x0, c.y0);
      g_object_unref (pbuf);
    }
    sprintf (buff, "%d", (gint) (zoom_factor * 100. + 0.5));
    show_progress (0, org);
    view_zoom (NULL, buff);
    draw_clip_rect_area ();

    sprintf (buff, "%s: DONE", name);
    put_string_to_appbar (buff);
    image_changed ();
  } else { 
    g_warning ("filter_exec: FALIED(Internal error): %s", name);
    sprintf (buff, "%s: FAILED(Internal error)", name);
    put_string_to_appbar (buff);
  }

  set_cursor (GDK_TOP_LEFT_ARROW);
  menubar_hilite_control (UNLOCK);
  mii->handling = FALSE;
}
