/***************************************************************************
 *
 *   KYum - a KDE GUI for yum
 *
 *   Copyright (C) 2005 by Steffen Offermann
 *   steffen_ac@yahoo.com
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *   General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the
 *   Free Software Foundation, Inc.,
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 ***************************************************************************/

#include <qlayout.h>
#include <qlabel.h>

#include <klocale.h>

#include "CfgFile.h"
#include "OptionsWidget.h"

/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

OptionsWidget::OptionsWidget(const Option::List & options,
                             QWidget            * pParent,
                             const char         * pName)
  : QWidget(pParent, pName)

{
    m_pTopLayout = new QGridLayout(this,   // parent
                                   1,      // number of rows
                                   2,      // number of columns
                                   5);     // margin

    m_pTopLayout->setAutoAdd(true);
    m_pTopLayout->setColStretch(1, 1);   // only second column may grow

    Option::List::const_iterator itOpt;

    for ( itOpt = options.begin(); itOpt != options.end(); itOpt++ )
    {
        addOption(*itOpt);
    }

/*
    Option::Ptr ptrOption = new TextOption("other options");
    addOption(ptrOption);
*/
}


/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

OptionsWidget::~OptionsWidget()
{
}


/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

void OptionsWidget::addOption(Option::Ptr ptrOption)
{
    QLabel        * pLabel  = new QLabel(this);
    OptionEditor  * pEditor = ptrOption->createEditor(this);

    m_pTopLayout->expand(1, 0);
    m_pTopLayout->setRowStretch(m_optInfos.size(), 0);
    m_pTopLayout->setRowStretch(m_optInfos.size() + 1, 1);


    pLabel->setText(ptrOption->getLabel() + ':');
    pEditor->setOptionValue(ptrOption->getValue());

    OptionInfo::Ptr ptrInfo = new OptionInfo(ptrOption, pEditor);
    m_editorMap[pEditor] = ptrInfo;
    m_optInfos.push_back(ptrInfo);

    connect(pEditor,  SIGNAL(optionChanged(const OptionEditor *, const QString &)),
            this,     SLOT(slotOptionChanged(const OptionEditor *, const QString &)));

    connect(pEditor,  SIGNAL(activated(const OptionEditor *)),
            this,     SLOT(slotEditorFocus(const OptionEditor *)));
}


/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

void OptionsWidget::slotEditorFocus(const OptionEditor * pEditor)
{
    EditorMap::iterator it = m_editorMap.find(pEditor);

    if ( it != m_editorMap.end() )
    {
        OptionInfo::Ptr ptrInfo   = it->second;
        Option::Ptr     ptrOption = ptrInfo->m_ptrOption;
        QString         strHelp   = ptrOption->getComment();

        if ( strHelp.isEmpty() )
            strHelp = i18n("No help available for this option.");

        emit optionHelp(strHelp);
    }
}


/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

void OptionsWidget::slotOptionChanged(const OptionEditor * pEditor, const QString & strValue)
{
    EditorMap::iterator it = m_editorMap.find(pEditor);

    if ( it != m_editorMap.end() )
    {
        OptionInfo::Ptr ptrInfo   = it->second;
        Option::Ptr     ptrOption = ptrInfo->m_ptrOption;
        QString         strNewValue;

        strNewValue = ptrOption->convertValueFromEditor(strValue);

        if ( ptrOption->getValue() != strNewValue )
        {
            ptrOption->setValue(strNewValue);
            emit optionChanged(ptrOption->getLabel(), strNewValue);
        }
    }
}


/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

void OptionsWidget::setOption(QString strKey, QString strValue)
{
    OptionInfo::List::const_iterator  itInfo;
    bool                              bFound = false;

    strKey = strKey.lower();

    for ( itInfo = m_optInfos.begin(); itInfo != m_optInfos.end(); itInfo++ )
    {
        OptionInfo::Ptr ptrInfo   = *itInfo;
        Option::Ptr     ptrOption = ptrInfo->m_ptrOption;
        QString         strLabel  = ptrOption->getLabel().lower();

        if ( strLabel == strKey )
        {
            ptrOption->setValue(strValue);

            ptrInfo->m_ptrEditor->setOptionValue(
                          ptrOption->convertValueForEditor(strValue));

            bFound = true;
            break;
        }
    }

    if ( !bFound  )
    {
        Option::Ptr ptrOption = new StringOption(strKey);

        ptrOption->setValue(strValue);
        addOption(ptrOption);
    }
}


/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

void OptionsWidget::setOptions(const CfgGroup * pGroup)
{
    if ( pGroup )
    {
        QStringList                 items;
        QStringList::const_iterator it;

        pGroup->getItemNames(items);

        for ( it = items.begin(); it != items.end(); it++ )
        {
            setOption(*it, pGroup->getValue(*it));
        }
    }
}


/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

void OptionsWidget::setDefaultOptions()
{
    OptionInfo::List::const_iterator itInfo;

    for ( itInfo = m_optInfos.begin(); itInfo != m_optInfos.end(); itInfo++ )
    {
        OptionInfo::Ptr ptrInfo = *itInfo;

        ptrInfo->m_ptrEditor->setOptionValue(
                    ptrInfo->m_ptrOption->getDefault());
    }
}

#include "OptionsWidget.moc"
