/***************************************************************************
 *
 *   KYum - a KDE GUI for yum
 *
 *   Copyright (C) 2005 by Steffen Offermann
 *   steffen_ac@yahoo.com
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *   General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the
 *   Free Software Foundation, Inc.,
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 ***************************************************************************/

#include "StrTok.h"
#include "KYumPreferences.h"
#include "KYumConf.h"

#include <qdir.h>
#include <qmutex.h>
#include <qregexp.h>
#include <qtextstream.h>
#include <qmessagebox.h>

#include <klocale.h>


#define ArraySize(a)  (sizeof(a) / sizeof(a)[0])


/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

KYumConf::KYumConf()
  : CfgFile("KYumConf")
{
}


/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

KYumConf::~KYumConf()
{
}


/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

QString KYumConf::getRepoDir() const
{
    QString       strRepoDir;
    CfgGroup::Ptr ptrGroup = getGroup("main");

    if ( ptrGroup.isValid() )
        strRepoDir = ptrGroup->getValue("reposdir");

    if ( strRepoDir.isEmpty() )
        strRepoDir = "/etc/yum.repos.d";

    if ( !strRepoDir.endsWith("/") )
        strRepoDir += '/';

    return strRepoDir;
}


/***************************************************************************/
/**
 *  Moves repo sections defined in yum.conf to kyum.repo.
 *
 ***************************************************************************/

void KYumConf::moveReposToRepoFile()
{
    QStringList           groups;
    QStringList::Iterator itGroups;
    CfgGroupPtr           ptrMainGroup;
    QString               strRepoFile(getRepoDir() + "kyum.repo");
    int                   numReposAdded = 0;

    getGroupList(groups);

    for ( itGroups = groups.begin(); itGroups != groups.end(); itGroups++ )
    {
        CfgGroupPtr ptrGroup = getGroup(*itGroups);

        //
        // Save main entry before clearing...
        //
        if ( ptrGroup->getTag().lower() == "main" )
        {
            ptrMainGroup = ptrGroup;
        }
        //
        // Add repository definitions to kyum.repo
        //
        else
        {
            Repo::Ptr ptrRepo = new Repo(*ptrGroup.getPtr(), strRepoFile);
            if ( addRepo(ptrRepo, strRepoFile) )
                numReposAdded++;
        }
    }

    //
    // Erase current contents of yum.conf...
    //
    clear();

    //
    // ... and restore [main] group.
    //
    addGroup(ptrMainGroup);

    if ( 0 < numReposAdded )
        emit reposLoaded();
}


/***************************************************************************/
/**
 * Saves the current yum repository settings as a profile
 *
 ***************************************************************************/

bool KYumConf::saveProfile(QString strFileName)
{
    bool    bRetry    = false,
            bSuccess  = false;
    QFile   file(strFileName);

    do
    {
        if ( file.open(IO_WriteOnly) )
        {
            QTextStream os(&file);

            os << "<yum.conf>\n";
            os << CfgFile::toString() << "\n";

            Repo::Ptr                 ptrRepo;
            RepoFile::List::iterator  itFiles;
            bool                      bSuccess = true;

            for ( itFiles = m_repoFiles.begin(); itFiles != m_repoFiles.end(); itFiles++ )
            {
                RepoFile::Ptr ptrRepoFile = *itFiles;

                //
                // "repoFileName" is not a yum configuration item. We only need it
                // for kyum to be able to restore the original repo file, when this
                // profile will be reloaded.
                //
                os << "<repos>\n";
                os << "repoFileName="<< ptrRepoFile->getFileName() << "\n";

                os << ptrRepoFile->toString();
                os << "\n";
            }

            file.close();
            bRetry = false;
            bSuccess = true;
        }
        else
        {
            QString strMsg = i18n("Could not create or write to file ");

            strMsg += strFileName;

            bRetry = 0 == QMessageBox::warning(
                                0,
                                "KYum",
                                strMsg,
                                i18n("Retry"),
                                i18n("Quit"),
                                0,
                                0,
                                1);
        }
    }
    while (bRetry);

    return bSuccess;
}

/***************************************************************************/
/**
 *  Loads the current configuration from a profile.
 *
 ***************************************************************************/

bool KYumConf::loadProfile(QString strFileName)
{
    bool        bSuccess = false;
    QFile       file(strFileName);
    CfgGroupPtr ptrGroup;

    //
    // Try to read the configuration file yum.conf, if it exists...
    //
    if ( file.exists() )
    {
        bSuccess = file.open(IO_ReadOnly);

        if ( bSuccess )
        {
            bSuccess = loadProfile(file);
            file.close();
        }
    }
    else
    {
        QString strMsg;

        strMsg.sprintf(i18n("The file\n\n  %s\n\ndoes not seem to exist."),
                       strFileName.ascii());

        QMessageBox::warning(0, "KYum", strMsg);
    }

    return bSuccess;
}


/***************************************************************************/
/**
 *  Loads the current configuration from a profile.
 *
 ***************************************************************************/

bool KYumConf::loadProfile(QFile & file)
{
    bool            bSuccess = true;
    QString         strProfile;
    ProfileSection  currentSection = c_SectionNone;

    //
    // Delete current repo files...
    //
    RepoFile::Ptr             ptrRepoFile;
    RepoFile::List::iterator  itFiles;

    for ( itFiles = m_repoFiles.begin(); itFiles != m_repoFiles.end(); itFiles++ )
    {
        QFile repoFile((*itFiles)->getFileName());

        if ( repoFile.exists() )
            repoFile.remove();
    }

    //
    // Clear configuration...
    //
    m_repoFiles.clear();
    clear();

    //
    // Load configuration from profile...
    //
    int numReposLoaded = 0;

    while ( bSuccess && !file.atEnd() )
    {
        QString strLine;

        file.readLine(strLine, Q_ULONG(1024));
        strLine = strLine.stripWhiteSpace();

        if ( !strLine.isEmpty() )
        {
            if ( '<' == strLine[0] )
            {
                int pos2 = strLine.find('>', 1);

                if ( 1 < pos2 )
                {
                    QString strSection = strLine.mid(1, pos2 - 1).lower();

                    bSuccess   = parseProfileSection(currentSection, strProfile);
                    strProfile = "";

                    if ( bSuccess )
                    {
                        if ( c_SectionRepos == currentSection )
                            numReposLoaded++;

                        if ( strSection == "yum.conf" )
                        {
                            currentSection = c_SectionYumConf;
                        }
                        else if ( strSection == "repos" )
                        {
                            currentSection = c_SectionRepos;
                        }
                    }
                }
            }
            else
                strProfile += strLine + '\n';
        }
    }

    if ( bSuccess )
    {
        bSuccess = parseProfileSection(currentSection, strProfile);

        if ( bSuccess && c_SectionRepos == currentSection )
            numReposLoaded++;
     }

    //
    // Save new configuration, so yum will use the correct files...
    //
    if ( !saveAll() )
        bSuccess = false;

    emit mainCfgLoaded();

//    if ( 0 < numReposLoaded )
        emit reposLoaded();

    return bSuccess;
}


/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

bool KYumConf::parseProfileSection(ProfileSection section, QString strProfile)
{
    bool bSuccess = true;

    if ( !strProfile.isEmpty() )
    {
        switch ( section )
        {
            case c_SectionYumConf:
                    bSuccess = parse(strProfile);
                    break;

            case c_SectionRepos:
                    bSuccess = parseAndAddRepo(strProfile);
                    break;

            default: /**/
                    break;
        }
    }

    return bSuccess;
}


/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

bool KYumConf::loadMainCfg()
{
    QFile         file(KYumPreferences::getYumConfPath());
    bool          bSuccess = true;
    CfgGroupPtr   ptrGroup = getGroup("main");

    //
    // Set default values...
    //
    if ( ptrGroup.isNull() )
    {
        ptrGroup = new CfgGroup("main");

        ptrGroup->setValue("reposdir",        "/etc/yum.repos.d");
        ptrGroup->setValue("cachedir",        "/var/cache/yum");
        ptrGroup->setValue("debuglevel",      "2");
        ptrGroup->setValue("errorlevel",      "2");
        ptrGroup->setValue("logfile",         "/var/log/yum.log");
        ptrGroup->setValue("pkgpolicy",       "newest");
        ptrGroup->setValue("distroverpkg",    "redhat-release");
        ptrGroup->setValue("diskspacecheck",  "1");
        ptrGroup->setValue("tolerant",        "1");
        ptrGroup->setValue("exactarch",       "1");
        ptrGroup->setValue("retries",         "20");
        ptrGroup->setValue("obsoletes",       "1");
        ptrGroup->setValue("gpgcheck",        "1");

        addGroup(ptrGroup);
    }

    //
    // Try to read the configuration file yum.conf, if it exists...
    //
    if ( file.exists() )
    {
        bSuccess = file.open(IO_ReadOnly);

        if ( bSuccess )
        {
            bSuccess = parse(file);
            file.close();
        }
    }
    else
        saveMainCfg();

    if ( bSuccess )
        emit mainCfgLoaded();

    return bSuccess;
}


/***************************************************************************/
/**
 *  Loads the yum repositories
 *
 ***************************************************************************/

bool KYumConf::loadRepos()
{
    QDir                    dir(getRepoDir(), "*.repo");
    const QFileInfoList   * pFileList = dir.entryInfoList();
    int                     numLoaded = 0;
    bool                    bSuccess  = false;

    //
    // Clear the current list of repository files from memory...
    //
    m_repoFiles.clear();

    //
    // Is the configured directory for *.repo files valid?
    //
    if ( pFileList )
    {
        bSuccess = true;

        //
        // Then run through the list of *.repo files found in the directory...
        //
        QFileInfoListIterator  it(*pFileList);

        while ( it.current() )
        {
            QFileInfo   * pFileInfo   = it.current();
            RepoFile::Ptr ptrRepoFile = new RepoFile(pFileInfo->absFilePath());

            //
            // Try to load the repository file...
            //
            if ( ptrRepoFile.isValid() && ptrRepoFile->load() )
            {
                m_repoFiles.push_back(ptrRepoFile);
                numLoaded++;
            }
            else
                bSuccess = false;

            ++it;
        }
    }

    if ( 0 == numLoaded )
    {
    }
//    else
        emit reposLoaded();

    return bSuccess;
}


/***************************************************************************/
/**
 *  Loads the yum configuration files
 *
 ***************************************************************************/

bool KYumConf::loadAll()
{
    bool  bSuccess = loadMainCfg();

    if ( bSuccess )
        loadRepos();

    return bSuccess;
}


/***************************************************************************/
/**
 *  Saves the repository files
 *
 *  @returns true, if all repos could be saved.
 *
 ***************************************************************************/

bool KYumConf::saveMainCfg()
{
    QString strFile   = KYumPreferences::getYumConfPath();
    bool    bRetry    = false,
            bSuccess  = false;
    QFile   file(strFile);

    do
    {
        if ( file.open(IO_WriteOnly) )
        {
            QTextStream(&file) << toString();

            file.close();
            bRetry = false;
            bSuccess = true;
        }
        else
        {
            QString strMsg = i18n("Could not create or write to file ");

            strMsg += strFile;

            bRetry = 0 == QMessageBox::warning(
                                0,
                                "KYum",
                                strMsg,
                                i18n("Retry"),
                                i18n("Quit"),
                                0,
                                0,
                                1);
        }
    }
    while (bRetry);

    if ( bSuccess )
        emit mainCfgSaved();

    return bSuccess;
}


/***************************************************************************/
/**
 *  Saves yum.conf and the repository files
 *
 *  @returns true, if all repos could be saved.
 *
 ***************************************************************************/

bool KYumConf::saveAll()
{
    bool bSuccess = saveMainCfg();
    return saveRepos() && bSuccess;
}


/***************************************************************************/
/**
 *  Saves the repository files
 *
 *  @returns true, if all repos could be saved.
 *
 ***************************************************************************/

bool KYumConf::saveRepos()
{
    Repo::Ptr                 ptrRepo;
    RepoFile::List::iterator  itFiles;
    bool                      bSuccess = true;

    for ( itFiles = m_repoFiles.begin(); itFiles != m_repoFiles.end(); itFiles++ )
    {
        if ( !(*itFiles)->save() )
            bSuccess = false;
    }

    return bSuccess;
}


/***************************************************************************/
/**
 *  Saves the repository file containing the specified repository.
 *
 *  @returns true, if the file could be saved.
 *
 ***************************************************************************/

bool KYumConf::saveRepo(Repo::Ptr ptrRepo)
{
    RepoFile::List::iterator  itFiles;
    bool                      bSuccess = false;

    //
    // Try to find a repository file in our configuration that contains
    // the repository in question...
    //
    for ( itFiles = m_repoFiles.begin(); itFiles != m_repoFiles.end(); itFiles++ )
    {
        if ( (*itFiles)->findRepo(ptrRepo->getTag()) == ptrRepo )
        {
            bSuccess = (*itFiles)->save();
            break;
        }
    }

    //
    // If the file was found, but could not be saved, an error message has
    // already been displayed within RepoFile::save(). So we only deal with
    // the (impossible ?) situation that the repo file was not found - which
    // would be an internal error.
    //
    if ( itFiles == m_repoFiles.end() )
    {
        QString strMsg = i18n("Peculiar... could not find the file containing this repository.");

        QMessageBox::warning(0,
                             "KYum",
                             strMsg,
                             i18n("Retry"),
                             i18n("Quit"),
                             0, 0, 1);
    }

    return bSuccess;
}

/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

Repo::Ptr KYumConf::pvtFindRepo(QString strTag, QString * pStrFileName) const
{
    Repo::Ptr                       ptrRepo;
    RepoFile::List::const_iterator  itFiles;

    if ( pStrFileName )
        *pStrFileName = "";

    for ( itFiles = m_repoFiles.begin(); itFiles != m_repoFiles.end(); itFiles++ )
    {
        ptrRepo = (*itFiles)->findRepo(strTag);

        if ( ptrRepo.isValid() )
        {
            if ( pStrFileName )
                *pStrFileName = (*itFiles)->getFileName();

            break;
        }
    }

    return ptrRepo;
}

/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

Repo::Ptr KYumConf::findRepo(QString strTag, QString * pStrFileName) const
{
    return pvtFindRepo(strTag, pStrFileName);
}


/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

RepoFile::Ptr KYumConf::getFileWithRepo(Repo::Ptr ptrRepo) const
{
    RepoFile::Ptr                   ptrFile;
    RepoFile::List::const_iterator  itFiles;

    for ( itFiles = m_repoFiles.begin(); itFiles != m_repoFiles.end(); itFiles++ )
    {
        Repo::Ptr ptrFound = (*itFiles)->findRepo(ptrRepo->getTag());

        if ( ptrFound.isValid() )
        {
            ptrFile = *itFiles;
            break;
        }
    }

    return ptrFile;
}


/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

void KYumConf::deleteRepo(Repo::Ptr ptrRepo)
{
    RepoFile::List::iterator  itFiles;

    for ( itFiles = m_repoFiles.begin(); itFiles != m_repoFiles.end(); itFiles++ )
    {
        Repo::Ptr ptrFound = (*itFiles)->findRepo(ptrRepo->getTag());

        if ( ptrFound.isValid() )
        {
            RepoFile::Ptr ptrFile = *itFiles;

            if ( 1 == ptrFile->countRepos() )
            {
                QFile repoFile((*itFiles)->getFileName());

                if ( repoFile.exists() )
                {
                    bool bRetry = false;

                    do
                    {
                        if ( !repoFile.remove() )
                        {
                            QString strMsg;

                            strMsg.sprintf(i18n("Could not remove the file\n\n%s\n\nfrom your disk!"),
                                          repoFile.name().ascii());

                            bRetry = 0 == QMessageBox::warning(0,
                                                               "KYum",
                                                               strMsg,
                                                               i18n("Retry"),
                                                               i18n("Quit"),
                                                               0, 0, 1);
                        }
                    }
                    while ( bRetry );
                }

                m_repoFiles.erase(itFiles);
            }
            else
                ptrFile->deleteRepo(ptrRepo->getTag());

            break;
        }
    }
}


/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

bool KYumConf::parseAndAddRepo(QString strProfile)
{
    bool        bSuccess = true;
    Repo::Ptr   ptrRepo  = 0;
    StrTok      strTok(strProfile);
    QString     strLine,
                strRepoFile;
    ConfigItem  cfgItem;

    while ( strTok.nextToken('\n', strLine) )
    {
        strLine = strLine.stripWhiteSpace();

        if ( !strLine.isEmpty() )
        {
            cfgItem.readFrom(strLine);

            if ( cfgItem.isComment() )
            {
                // ignore
            }
            else if ( cfgItem.isGroup() )
            {
                if ( ptrRepo.isValid() && !ptrRepo->getTag().isEmpty() )
                    addRepo(ptrRepo, strRepoFile);

                ptrRepo = new Repo;
                ptrRepo->setTag(cfgItem.getName());
            }
            else if ( ptrRepo.isValid() )
            {
                ptrRepo->setValue(cfgItem.getName(), cfgItem.getValue());
            }
            else
            {
                QString strName = cfgItem.getName();

                //
                // "repoFileName" is not a valid yum configuration item -
                // we use it before the first repo tag of each repo file section
                // to be able to restore the original filenames from a profile.
                //
                if ( strName == "repofilename")
                    strRepoFile = cfgItem.getValue();
            }
        }
    }

    if ( ptrRepo.isValid() && !ptrRepo->getTag().isEmpty() )
        addRepo(ptrRepo, strRepoFile);

    return bSuccess;
}


/***************************************************************************/
/**
 * Adds a new repository.
 *
 * @param ptrRepo     The new repository
 *
 * @param strRepoFile The name of the repo file (incl. path). Empty string
 *                    or null means: Use kyum.repo
 *
 * @returns true, if the repo has been added. false, if it exists already.
 *
 ***************************************************************************/

bool KYumConf::addRepo(Repo::Ptr ptrRepo, QString & strRepoFile)
{
    bool bSuccess = false;

    //
    // If a repository with the same tag exists, we won't do anything...
    //
    if ( pvtFindRepo(ptrRepo->getTag()).isNull() )
    {
        if ( strRepoFile.isNull() || strRepoFile.isEmpty() )
            strRepoFile = getRepoDir() + "kyum.repo";

        //
        // Try to find the specified repository file in memory...
        //
        RepoFile::Ptr                   ptrRepoFile;
        RepoFile::List::const_iterator  itFiles;
        bool                            bFound = false;

        for ( itFiles = m_repoFiles.begin();
              itFiles != m_repoFiles.end() && !bFound;
              itFiles++ )
        {
            ptrRepoFile = *itFiles;

            if ( ptrRepoFile->getFileName() == strRepoFile )
                bFound = true;
        }

        //
        // If a file with this name does not exist, a new repository
        // file item will be created...
        //
        if ( !bFound )
        {
            ptrRepoFile = new RepoFile(strRepoFile);
            m_repoFiles.push_back(ptrRepoFile);
        }
        //
        // If it does exist, we correct the file name stored in the
        // repository (although this is not really needed, since
        // ptrRepoFile->addRepo() will take care of that, too).
        //
        else
            ptrRepo->setFileName(strRepoFile);

        //
        // Now add the repository to the file...
        //
        ptrRepoFile->addRepo(ptrRepo);

        bSuccess = true;
    }

    return bSuccess;
}


/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

QString KYumConf::getLogFile() const
{
    QString strLogFile = getValue("main", "logfile");

    if ( strLogFile.isEmpty() )
        strLogFile = "/var/log/yum.log";

    return strLogFile;
}


/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

QString KYumConf::getDistroVerPkg() const
{
    QString strPkg = getValue("main", "distroverpkg");

    if ( strPkg.isEmpty() )
        strPkg = "redhat-release";

    return strPkg;
}


/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

Repo::Repo()
{
}


/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

Repo::Repo(const CfgGroup & group, QString strFileName)

  : CfgGroup(group)

{
    setFileName(strFileName);
}


/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

Repo::Repo(const Repo & repo)

  : CfgGroup(repo)

{
    setFileName(repo.getFileName());
}


/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

void Repo::copyFrom(const Repo & repo)
{
    CfgGroup::operator =(repo);
    setFileName(repo.getFileName());
}


/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

void Repo::clear()
{
    CfgGroup::clear();
    setTag("");
}


/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

QString Repo::toString() const
{
    QString strResult;

    strResult += CfgGroup::toString();

    return strResult;
}


/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

QString Repo::getOtherOptions() const
{
    //
    // Exclude these
    //
    const char * pKnownOptions[] = { "baseurl", "mirrorlist", "enabled",
                                     "gpgcheck", "name", 0 };

    return getOptions(pKnownOptions);
}


/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

RepoFile::RepoFile(QString strFileName)
{
    setFileName(strFileName);
}


/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

bool RepoFile::parse(QFile & file)
{
    bool      bSuccess = true;
    Repo::Ptr ptrRepo  = 0;

    while ( !file.atEnd() )
    {
        QString     strLine;
        ConfigItem  cfgItem;

        file.readLine(strLine, Q_ULONG(1024));
        cfgItem.readFrom(strLine);

        if ( cfgItem.isComment() )
        {
            // ignore
        }
        else if ( cfgItem.isGroup() )
        {
            if ( ptrRepo.isValid() && !ptrRepo->getTag().isEmpty() )
                addRepo(ptrRepo);

            ptrRepo = new Repo;
            ptrRepo->setTag(cfgItem.getName());
        }
        else if ( ptrRepo.isValid() )
            ptrRepo->setValue(cfgItem.getName(), cfgItem.getValue());
    }

    if ( ptrRepo.isValid() && !ptrRepo->getTag().isEmpty() )
        addRepo(ptrRepo);

    return bSuccess;
}


/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

bool RepoFile::load()
{
    bool  bSuccess = false;
    QFile file(getFileName());

    if ( file.exists() && file.open(IO_ReadOnly) )
    {
        bSuccess = parse(file);
        file.close();
    }

    return bSuccess;
}


/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

bool RepoFile::save()
{
    bool bSuccess = false;

    if ( 0 == m_repos.size() )
    {
        bool bRetry = false;

        do
        {
            if ( !QFile(getFileName()).remove() )
            {
                QString strMsg = i18n("Could not remove file ");

                strMsg += getFileName();

                bRetry = 0 == QMessageBox::warning(
                                    0,
                                    "KYum",
                                    strMsg,
                                    i18n("Retry"),
                                    i18n("Quit"),
                                    0,
                                    0,
                                    1);
            }
        }
        while (bRetry);
    }
    else
    {
        QFile repoFile(getFileName());
        bool  bRetry = false;

        do
        {
            if ( repoFile.open(IO_WriteOnly) )
            {
                QTextStream(&repoFile) << toString();
                repoFile.close();
                bRetry = false;
                bSuccess = true;
            }
            else
            {
                QString strMsg = i18n("Could not create or write to file ");

                strMsg += getFileName();

                bRetry = 0 == QMessageBox::warning(
                                    0,
                                    "KYum",
                                    strMsg,
                                    i18n("Retry"),
                                    i18n("Quit"),
                                    0,
                                    0,
                                    1);
            }
        }
        while (bRetry);
    }

    return bSuccess;
}


/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

void RepoFile::addRepo(Repo::Ptr ptrRepo)
{
    ptrRepo->setFileName(getFileName());
    m_repos.push_back(ptrRepo);
}


/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

QString RepoFile::toString() const
{
    QString                    strResult;
    Repo::List::const_iterator it;

    for ( it = m_repos.begin(); it != m_repos.end(); it++ )
    {
        strResult += (*it)->toString();
        strResult += '\n';
    }

    return strResult;
}


/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

Repo::Ptr RepoFile::findRepo(QString strTag) const
{
    Repo::Ptr                  ptrRepo;
    Repo::List::const_iterator itRepos;

    for ( itRepos = m_repos.begin(); itRepos != m_repos.end(); itRepos++ )
    {
        if ( (*itRepos)->getTag() == strTag )
        {
            ptrRepo = *itRepos;
            break;
        }
    }

    return ptrRepo;
}


/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

void RepoFile::deleteRepo(QString strTag)
{
    Repo::List::iterator  itRepos;

    for ( itRepos = m_repos.begin(); itRepos != m_repos.end(); itRepos++ )
    {
        if ( (*itRepos)->getTag() == strTag )
        {
            m_repos.erase(itRepos);
            break;
        }
    }
}


#include "KYumConf.moc"

