/***************************************************************************
 *
 *   KYum - a KDE GUI for yum
 *
 *   Copyright (C) 2005 by Steffen Offermann
 *   steffen_ac@yahoo.com
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *   General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the
 *   Free Software Foundation, Inc.,
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 ***************************************************************************/

#include <qapp.h>
#include <qlayout.h>
#include <qpushbutton.h>
#include <qwaitcondition.h>
#include <qlabel.h>
#include <qsplitter.h>

#include <klocale.h>
#include <ktextedit.h>

#include "KYumPreferences.h"
#include "UserEvents.h"
#include "OutputView.h"
#include "Process.h"
#include "DlgRun.h"


class TextView : public QWidget
  {
    private:
      KTextEdit * m_pTextEdit;

    public:
                  TextView(QString strLabel, QWidget * pParent, const char * pName = 0);
      virtual   ~ TextView() {}

      void        setText(QString strText)
                  {
                      m_pTextEdit->setText(QString::fromUtf8(strText));
                  }

      void        appendText(QString strText)
                  {
                      m_pTextEdit->append(QString::fromUtf8(strText));
                  }

      void        setTextColor(const QColor & clr)
                  {
                      m_pTextEdit->setColor(clr);
                  }
  };


/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

TextView::TextView(QString strLabel, QWidget * pParent, const char * pName)
  : QWidget(pParent, pName)
{
    QVBoxLayout * pTopLayout = new QVBoxLayout(this, 0, 5);
    QLabel      * pLabel     = new QLabel(strLabel, this);

    m_pTextEdit = new KTextEdit(this, QString(pName) + "_TextEdit");
    m_pTextEdit->setReadOnly(true);

    QFont monoFont(m_pTextEdit->font());
    monoFont.setFamily("Courier New");
//    monoFont.setPointSize(monoFont.pointSize() + 1);
    m_pTextEdit->setFont(monoFont);

    pTopLayout->addWidget(pLabel);
    pTopLayout->addWidget(m_pTextEdit, 1);
}


/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

DlgRun::DlgRun(QWidget * pParent, const char * pName)

  : ProcessDlg(pParent, pName)


{
    QWidget     * pMainView   = new QWidget(this);
    QVBoxLayout * pTopLayout  = new QVBoxLayout(this, 5, 5),
                * pMainLayout = new QVBoxLayout(pMainView);

    pMainLayout->setAutoAdd(true);
    pTopLayout->addWidget(pMainView, 1);

    QSplitter * pVSplitter = new QSplitter(pMainView);
    pVSplitter->setOrientation(QSplitter::Vertical);

    m_pOutputView = new TextView(i18n("Output"), pVSplitter, "OutputView"),
    m_pStatusView = new TextView(i18n("Status"), pVSplitter, "StatusView");

    QWidget     * pButtonPanel  = new QWidget(this);
    QHBoxLayout * pButtonLayout = new QHBoxLayout(pButtonPanel, 6);
    QSpacerItem * pHSpacer      = new QSpacerItem(60,
                                                  60,
                                                  QSizePolicy::Expanding,
                                                  QSizePolicy::Minimum);

    pButtonLayout->addItem(pHSpacer);

    m_pButtonOk = new QPushButton(pButtonPanel, "m_pButtonStart");
    m_pButtonOk->setText(i18n("&Start"));
    pButtonLayout->addWidget(m_pButtonOk);

    m_pButtonCancel = new QPushButton(pButtonPanel, "m_pButtonCancel");
    m_pButtonCancel->setText(i18n("&Exit"));
    pButtonLayout->addWidget(m_pButtonCancel);

    pTopLayout->addWidget(pButtonPanel);

    resize(400, 600);

    connect(m_pButtonOk,     SIGNAL(clicked()), this, SLOT(accept()));
    connect(m_pButtonCancel, SIGNAL(clicked()), this, SLOT(reject()));
}


/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

DlgRun::~DlgRun()
{
}


/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

void DlgRun::startProcess()
{
    m_pButtonOk->setEnabled(false);
    m_pButtonCancel->setText(i18n("&Abort"));

    TextView * pStatus = dynamic_cast<TextView *>(m_pStatusView);
    pStatus->setText("");

    QString strCmd = m_pProcess->getCmdLine();

    pStatus->setTextColor(KYumPreferences::c_clrInfo);
    pStatus->setText(i18n("Executing command: "));

    pStatus->setTextColor(KYumPreferences::c_clrAddInfo);
    pStatus->appendText(strCmd + '\n');

    ProcessDlg::startProcess();
}


/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

void DlgRun::accept()
{
    startProcess();
}


/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

void DlgRun::reject()
{
    if ( !QThread::finished() )
    {
        TextView * pStatus = dynamic_cast<TextView *>(m_pStatusView);

        pStatus->setTextColor(KYumPreferences::c_clrInfo);
        pStatus->appendText("Aborting...\n");
    }

    ProcessDlg::reject();
}


/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

void DlgRun::slotProcessStarted()
{
    TextView * pStatus = dynamic_cast<TextView *>(m_pStatusView);

    pStatus->setTextColor(KYumPreferences::c_clrInfo);

    QString strMsg;

    strMsg.sprintf(i18n("Process %d (%s) has been started.\n"),
                   m_pProcess->processIdentifier(),
                   m_pProcess->getCmdName().ascii());

    pStatus->appendText(strMsg);
}


/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

void DlgRun::slotProcessExited()
{
    {
        QMutexLocker  locker(m_pMutex);
        QString       strMsg;
        TextView    * pStatus = dynamic_cast<TextView *>(m_pStatusView);

        if ( m_pProcess->normalExit() )
        {
            pStatus->setTextColor(KYumPreferences::c_clrInfo);

            strMsg.sprintf(i18n("Process %d exited normally with code %d"),
                          m_pProcess->processIdentifier(),
                          m_pProcess->exitStatus());
        }
        else
        {
            pStatus->setTextColor(KYumPreferences::c_clrError);

            strMsg.sprintf(i18n("Process %d exited abnormally with code %d"),
                          m_pProcess->processIdentifier(),
                          m_pProcess->exitStatus());
        }

        pStatus->appendText(strMsg);
    }

    m_pButtonOk->setEnabled(true);
    m_pButtonCancel->setText(i18n("&Exit"));
}


/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

void DlgRun::slotReadyStdout()
{
    QMutexLocker  locker(m_pMutex);
    TextView    * pOutput = dynamic_cast<TextView *>(m_pOutputView);

    pOutput->setTextColor(KYumPreferences::c_clrInfo);
    pOutput->appendText(m_pProcess->readStdout());
}


/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

void DlgRun::slotReadyStderr()
{
    QMutexLocker  locker(m_pMutex);
    TextView    * pOutput = dynamic_cast<TextView *>(m_pOutputView);

    pOutput->setTextColor(KYumPreferences::c_clrError);
    pOutput->appendText(m_pProcess->readStderr());
}


#include "DlgRun.moc"
