/***************************************************************************
 *
 *   KYum - a KDE GUI for yum
 *
 *   Copyright (C) 2005 by Steffen Offermann
 *   steffen_ac@yahoo.com
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *   General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the
 *   Free Software Foundation, Inc.,
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 ***************************************************************************/

#include <kmessagebox.h>
#include <kfiledialog.h>
#include <klocale.h>

#include <qlineedit.h>
#include <qcheckbox.h>
#include <qtextedit.h>
#include <qpushbutton.h>
#include <qfile.h>
#include <qtextstream.h>

#include "KYumConf.h"
#include "KYumView.h"
#include "KYumPreferences.h"
#include "DlgSelectRepository.h"
#include "DlgEditRepo.h"



/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

DlgEditRepo::DlgEditRepo(QWidget * pParent, QString strName, Repo & repo)

  : BaseDlgEditRepo(pParent, strName, true),
    m_repo(repo)

{
    setFields(repo, c_SetAll);
    checkFields();

    connect(m_pEdTag,  SIGNAL(textChanged(const QString &)),
            this,      SLOT(slotTextChanged(const QString &)));

    connect(m_pEdName, SIGNAL(textChanged(const QString &)),
            this,      SLOT(slotTextChanged(const QString &)));

    connect(m_pEdBaseURL, SIGNAL(textChanged(const QString &)),
            this,         SLOT(slotTextChanged(const QString &)));

    connect(m_pEdMirrorList, SIGNAL(textChanged(const QString &)),
            this,            SLOT(slotTextChanged(const QString &)));

    connect(m_pButtonRepoFile, SIGNAL(clicked()),
            this,              SLOT(selectRepoFile()));

    connect(m_pButtonMirrorList, SIGNAL(clicked()),
            this,                SLOT(selectMirrorList()));
}


/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

DlgEditRepo::~DlgEditRepo()
{
}


/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

void DlgEditRepo::setFields(const Repo & repo, E_SetFieldsMode which)
{
    if ( c_SetAll == which || m_pEdRepoFile->text().isEmpty() )
    {
        QString strRepoFile = repo.getFileName(),
                strRepoPath = KYumView::getKYumView()->getYumConf().getRepoDir();

        if ( strRepoFile.startsWith(strRepoPath) )
            strRepoFile = strRepoFile.mid(strRepoPath.length());

        m_pEdRepoFile->setText(strRepoFile);
    }

    if ( c_SetAll == which || m_pEdTag->text().isEmpty() )
    {
        m_pEdTag->setText(repo.getTag());
    }

    if ( c_SetAll == which || m_pEdName->text().isEmpty() )
    {
        m_pEdName->setText(repo.getName());
    }

    if ( c_SetAll == which || m_pEdMirrorList->text().isEmpty() )
    {
        QString strMirrorList = repo.getValue("MirrorList");

        m_pEdMirrorList->setText(strMirrorList);

        if ( strMirrorList.startsWith("file://") )
            loadMirrorList(strMirrorList.mid(7));
    }


    if ( c_SetAll == which || m_pEdBaseURL->text().isEmpty() )
    {
        m_pEdBaseURL->setText(repo.getBaseURL().url());
    }

    if ( c_SetAll == which || m_pEdMore->text().isEmpty() )
    {
        m_pEdMore->setText(repo.getOtherOptions());
    }

    if ( c_SetAll == which )
    {
        m_pChkEnabled->setChecked(repo.isEnabled());
        m_pChkGPG->setChecked(repo.isGPGCheckOn());
    }
}


/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

void DlgEditRepo::accept()
{
    QString strMirrorList = m_pEdMirrorList->text(),
            strRepoFile   = m_pEdRepoFile->text();

    m_repo.clear();
    m_repo.setOptions(m_pEdMore->text());

    m_repo.setTag(m_pEdTag->text());
    m_repo.setName(m_pEdName->text());
    m_repo.setBaseURL(m_pEdBaseURL->text());
    m_repo.setValue("MirrorList", strMirrorList);

    m_repo.setEnabled(m_pChkEnabled->isChecked());
    m_repo.setGPGCheck(m_pChkGPG->isChecked());

    if ( strMirrorList.startsWith("file://") )
    {
        QFile file(strMirrorList.mid(7));

        if ( file.open(IO_WriteOnly) )
        {
            QTextStream(&file) << m_pEdMirrors->text();
            file.close();
        }
    }

    if ( strRepoFile.isEmpty() )
        strRepoFile = KYumView::getKYumView()->getYumConf().getRepoDir() + "kyum.repo";
    else
    {
        int     idxSlash    = strRepoFile.findRev('/');
        QString strRepoPath = KYumView::getKYumView()->getYumConf().getRepoDir();

        if ( idxSlash < 0 )
        {
            strRepoFile = strRepoPath + strRepoFile;
        }
        else if ( !strRepoFile.startsWith(strRepoPath) )
        {
            warnRepoPath();
            strRepoFile = strRepoPath + strRepoFile.mid(idxSlash + 1);
        }
    }

    m_repo.setFileName(strRepoFile);

    BaseDlgEditRepo::accept();
}


/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

void DlgEditRepo::warnRepoPath()
{
    QString strMsg;

    strMsg.sprintf(i18n("You selected a *.repo file that is not in %s.\n"
                        "Be aware that yum only recognizes repositories stored\n"
                        "in the repository directory.\n"
                        "\n"
                        "KYum will ignore the path you entered and put the\n"
                        "repository where it belongs."),
                        KYumView::getKYumView()->getYumConf().getRepoDir().ascii());

    KMessageBox::information(this, strMsg);
}


/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

void DlgEditRepo::loadMirrorList(QString strFile)
{
    QFile file(strFile);

    if ( file.exists() && file.open(IO_ReadOnly) )
    {
        QString strLine,
                strMirrors;

        while ( !file.atEnd() )
        {
            file.readLine(strLine, 1024);
            strMirrors += strLine;
        }

        m_pEdMirrors->setText(strMirrors);

        file.close();
    }
}


/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

void DlgEditRepo::selectMirrorList()
{
    static QString  strDir,
                    strFile;
    QString         strFilter("*.mirror|Mirror Files (*.mirror)\n*|All Files (*)"),
                    strSelection;

    if ( strDir.isEmpty() )
    {
        QString strMirrorList = m_pEdMirrorList->text();

        if ( strMirrorList.startsWith("file://") )
        {
            strDir = strMirrorList.mid(7);

            int idxSlash = strDir.findRev('/');

            if ( 0 < idxSlash )
                strDir = strDir.left(idxSlash);
        }
        else
            strDir = KYumView::getKYumView()->getYumConf().getRepoDir();
    }

    strSelection = KFileDialog::getOpenFileName(strDir,
                                                strFilter,
                                                this,
                                                i18n("Select mirror list"));

    if ( !strSelection.isEmpty() )
    {
        QString strMirrorList("file://");

        strFile         = strSelection;
        strDir          = QFileInfo(strFile).filePath();
        strMirrorList  += strFile;

        loadMirrorList(strFile);

        m_pEdMirrorList->setText(strMirrorList);
        slotTextChanged(strMirrorList);
    }
}


/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

void DlgEditRepo::selectRepoFile()
{
    static QString  strDir,
                    strFile;
    QString         strFilter("*.repo|Repository Files (*.repo)\n*|All Files (*)"),
                    strSelection;

    if ( strDir.isEmpty() )
    {
        QString strRepoPath = m_pEdRepoFile->text();

        int idxSlash = strDir.findRev('/');

        if ( 0 < idxSlash )
            strDir = strDir.left(idxSlash);
        else
            strDir = KYumView::getKYumView()->getYumConf().getRepoDir();
    }

    strSelection = KFileDialog::getOpenFileName(strDir,
                                                strFilter,
                                                this,
                                                i18n("Select repository file"));

    if ( !strSelection.isEmpty() )
    {
        strFile = strSelection;
        strDir  = QFileInfo(strFile).filePath();

        //
        // If the file exists, try to parse it and fill the empty
        // fields. Don't touch fields with existing contents.
        //
        if ( QFile(strFile).exists() )
        {
            RepoFile repoFile(strFile);

            if ( repoFile.load() )
            {
                Repo::Ptr           ptrRepo;
                DlgSelectRepository dlg(this, m_pEdTag->text(), repoFile.getRepos());

                if ( DlgSelectRepository::Accepted == dlg.exec() )
                {
                    ptrRepo = dlg.getSelectedRepo();

                    if ( ptrRepo.isValid() )
                    {
                        setFields(*ptrRepo.getPtr(), c_SetEmpty);
                        checkFields();
                    }
                }
            }
        }

        //
        // No matter which path the user selected, we will always store *.repo
        // files in the one and only repository directory /etc/yum.repos.d, so
        // strip off the directory part before displaying the file name.
        //
        QString strRepoPath = KYumView::getKYumView()->getYumConf().getRepoDir();

        if ( strFile.startsWith(strRepoPath) )
            m_pEdRepoFile->setText(strFile.mid(strRepoPath.length()));
        else
        {
            int idxSlash = strFile.findRev('/');

            if ( 0 <= idxSlash )
            {
                warnRepoPath();
                m_pEdRepoFile->setText(strFile.mid(idxSlash + 1));
            }
            else
                m_pEdRepoFile->setText(strFile);
        }
    }
}


/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

void DlgEditRepo::checkFields()
{
    bool    bValid;
    QString strMirrorList = m_pEdMirrorList->text();

    bValid = !m_pEdTag->text().isEmpty()   &&
             !m_pEdName->text().isEmpty()  &&
             (!m_pEdBaseURL->text().isEmpty() || !strMirrorList.isEmpty());

    m_pButtonOk->setEnabled(bValid);
    m_pEdMirrors->setEnabled(strMirrorList.startsWith("file://"));
}


/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

void DlgEditRepo::slotTextChanged(const QString &)
{
    checkFields();
}


#include "DlgEditRepo.moc"

