/***************************************************************************
 *
 *   KYum - a KDE GUI for yum
 *
 *   Copyright (C) 2005 by Steffen Offermann
 *   steffen_ac@yahoo.com
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *   General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the
 *   Free Software Foundation, Inc.,
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 ***************************************************************************/

#include <qapp.h>
#include <qsplitter.h>
#include <qlayout.h>
#include <qtabwidget.h>
#include <qtextbrowser.h>
#include <qcolor.h>

#include <klocale.h>

#include "UserEvents.h"
#include "Process.h"
#include "KYumPreferences.h"
#include "ActionsView.h"
#include "OutputView.h"
#include "YumLogView.h"
#include "ActionsTab.h"


/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

ActionsTab::ActionsTab(QWidget * pParent)

  : QWidget(pParent)

{
    QHBoxLayout * pTopLayout = new QHBoxLayout(this);
    pTopLayout->setAutoAdd(true);

    //
    // Vertical splitter between list view and tabs/info view
    //
    QSplitter * pVSplitter = new QSplitter(this);
    pVSplitter->setOrientation(QSplitter::Vertical);

    m_pActionsView = new ActionsView(pVSplitter);

    //
    // Now we prepare the bottom part...
    //
    QSplitter * pHSplitter = new QSplitter(pVSplitter);
    pHSplitter->setOrientation(QSplitter::Horizontal);
    pHSplitter->setSizePolicy(QSizePolicy::Preferred, QSizePolicy::Maximum);

    //
    // 1. the tab view, which contains the yum.log and the output tab.
    //
    QTabWidget  * pTabView        = new QTabWidget(pHSplitter);
    QHBoxLayout * pTabViewLayout  = new QHBoxLayout(pTabView);

    (void) pTabViewLayout;
//    pTabViewLayout->setAutoAdd(true);

    pTabView->addTab(m_pOutputView = new OutputView(this), i18n("Output"));
    pTabView->addTab(m_pYumLogView = new YumLogView(this), i18n("yum.log"));

    //
    // 2. the info view that will display the description for the
    //    focused package in the list.
    //
    m_pInfoView = new QTextBrowser(pHSplitter);
    QVBoxLayout * pInfoViewLayout = new QVBoxLayout(m_pInfoView);

    (void) pInfoViewLayout;

/*
    m_pInfoView->setBackgroundMode(Qt::FixedColor);
    m_pInfoView->setBackgroundColor(QColor(255,255,255));
    m_pInfoView->setTextFormat(QTextEdit::RichText);
*/
    QFont infoFont(m_pInfoView->font());
//    infoFont.setFamily("System");
//    infoFont.setBold(true);
    infoFont.setPointSize(infoFont.pointSize() + 2);
    m_pInfoView->setFont(infoFont);

    connect(m_pActionsView, SIGNAL(yumInfo(QString)),
            this,           SLOT(slotYumInfo(QString)));

    connect(m_pActionsView, SIGNAL(yumPkgInfo(QString)),
            this,           SLOT(slotYumPkgInfo(QString)));

    connect(m_pActionsView, SIGNAL(yumError(QString)),
            this,           SLOT(slotYumError(QString)));

    connect(m_pActionsView, SIGNAL(actionStarted(Process *)),
            this,           SLOT(slotActionStarted(Process *)));

    connect(m_pActionsView, SIGNAL(actionTerminated(Process *)),
            this,           SLOT(slotActionTerminated(Process *)));

    connect(m_pActionsView, SIGNAL(showPackageInfo(const PackageInfo &)),
            this,           SLOT(slotShowPackageInfo(const PackageInfo &)));
}


/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

void ActionsTab::slotYumInfo(QString strLine)
{
    m_pOutputView->setColor(KYumPreferences::c_clrInfo);
    m_pOutputView->append(strLine);
}


/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

void ActionsTab::slotYumPkgInfo(QString strLine)
{
    m_pOutputView->setColor(KYumPreferences::c_clrPkgInfo);
    m_pOutputView->append(strLine);
}


/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

void ActionsTab::slotYumError(QString strLine)
{
    m_pOutputView->setColor(KYumPreferences::c_clrError);
    m_pOutputView->append(strLine);
}


/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

void ActionsTab::slotActionStarted(Process * pProcess)
{
    QString strMsg,
            strCmd;

    strMsg.sprintf(i18n("Process %u (%s) has been started.\n"),
                   pProcess->processIdentifier(),
                   pProcess->getCmdName().ascii());

    strCmd.sprintf(i18n("(Command being executed: %s).\n\n"), pProcess->getCmdLine().ascii());

    m_pOutputView->setColor(KYumPreferences::c_clrInfo);
    m_pOutputView->setText(strMsg +
                           i18n("Please be patient, this may be a lengthy operation...\n\n"));

    m_pOutputView->setColor(KYumPreferences::c_clrAddInfo);
    m_pOutputView->append(strCmd);

    m_pInfoView->setText("");
}


/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

void ActionsTab::slotActionTerminated(Process * pProcess)
{
    QString strMsg;

    if ( pProcess->normalExit() )
    {
        m_pOutputView->setColor(KYumPreferences::c_clrInfo);
        strMsg = i18n("Process exited normally with code ");
    }
    else
    {
        m_pOutputView->setColor(KYumPreferences::c_clrError);
        strMsg = i18n("Process exited abnormally with code ");
    }

    strMsg += QString::number(pProcess->exitStatus(), 10);
    strMsg += ".\n";

    m_pOutputView->append(strMsg);
}


/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

bool ActionsTab::event(QEvent * pEvent)
{
    bool bProcessed = false;

    if ( pEvent->type() == QEvent::Type(c_evt_ShowPkgInfo) )
    {
        ShowPkgInfoEvent * pEvt = dynamic_cast<ShowPkgInfoEvent *>(pEvent);

        m_pInfoView->setUpdatesEnabled(false);
        m_pInfoView->setText(QString::fromUtf8(pEvt->getDescription()));
        m_pInfoView->setUpdatesEnabled(true);

        bProcessed = true;
    }
    else
        bProcessed = QWidget::event(pEvent);

    return bProcessed;
}




/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

void ActionsTab::slotShowPackageInfo(const PackageInfo & info)
{
    //
    // Posting this as an event is probably not necessary - we might as well
    // do the processing here. But I was trying to isolate the bug that
    // makes the info window keep repainting forever, when the text does not
    // fit its visible size, and now I'm too lazy - err, too busy - to change
    // it back.
    //
    QApplication::postEvent(this, new ShowPkgInfoEvent(info.m_strDescription));
}

#include "ActionsTab.moc"

