/*
 * This file is a part of KleanSweep.
 *
 * Copyright (C) 2005 Pawel Stolowski <pawel.stolowski@wp.pl>
 *
 * KleanSweep is free software; you can redestribute it and/or modify it
 * under terms of GNU General Public License by Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY. See GPL for more details.
 */

#include "filelistview.h"
#include "fileviewitem.h"
#include <kfileitem.h>
#include <kpopupmenu.h>
#include <klocale.h>
#include <kglobal.h>
#include <kiconloader.h>
#include <qmap.h>
#include "sweepfileinfo.h"

FileListView::FileListView(QWidget *parent, FileListView *prev): KListView(parent), prevview(prev), cols(0)
{
	if (prevview)
		prevview->nextview = this;
	nextview = NULL;

	diricon = new QPixmap(KGlobal::iconLoader()->loadIcon("folder", KIcon::Small));
	menu = new KPopupMenu(this);
	menu->insertItem(*diricon, i18n("Open directory"), this, SLOT(openDirSelected()));

	connect(this, SIGNAL(contextMenuRequested(QListViewItem *, const QPoint&, int)), this, SLOT(showContextMenu(QListViewItem *, const QPoint&, int)));
}

FileListView::~FileListView()
{
	if (prevview)
		prevview->nextview = nextview;
	if (nextview)
		nextview->prevview = prevview;
	delete diricon;
}

void FileListView::setupColumns(int cols)
{
	this->cols = cols;
	coltypes.clear();
	//for (int i=columns()-1; i>=0; i--)
	//	removeColumn(i);
	const static struct { Column t; QString hdr; } hdrs[] = {
		{FileName,    i18n("Name")},
		{FileDir,     i18n("Directory")},
		{Score,       i18n("Score")},
		{Target,      i18n("Target")},
		{Size,        i18n("Size")},
		{ModTime,     i18n("Date")},
		{AccessTime,  i18n("Last access")},
		{Permissions, i18n("Permissions")},
		{Owner,       i18n("Owner")},
		{Group,       i18n("Group")},
		{Group,      QString::null}};

	for (int i=0; hdrs[i].hdr != QString::null; i++)
	{
		if (cols & hdrs[i].t)
		{
			addColumn(hdrs[i].hdr);
			coltypes.append(hdrs[i].t);
		}
	}
}

void FileListView::stateChanged(SweepFileInfo *f, bool state)
{
	emit selectionChanged(state ? 1 : -1, f->size());
	//
	// propagate change to other views
	for (FileListView *v = prevview; v; v = v->prevview)
	{
		QListViewItemIterator it(v);
		while (it.current())
		{
			FileViewItem *item = dynamic_cast<FileViewItem *>(*it);
			if (item && item->fileItem() == f)
				item->setOn(state);
			++it;
    		}
	}
	for (FileListView *v = nextview; v; v = v->nextview)
	{
		QListViewItemIterator it(v);
		while (it.current())
		{
			FileViewItem *item = dynamic_cast<FileViewItem *>(*it);
			if (item && item->fileItem() == f)
				item->setOn(state);
			++it;
    		}
	}
}

FileViewItem* FileListView::find(SweepFileInfo *f)
{
	QListViewItemIterator it(this);
	while (it.current())
	{
		FileViewItem *item = dynamic_cast<FileViewItem *>(*it);
		if (item && item->fileItem() == f)
			return item;
		++it;
	}
	return NULL;
}

KPopupMenu* FileListView::contextMenu() const
{
	return menu;
}
		
void FileListView::showContextMenu(QListViewItem *item, const QPoint &point, int col)
{
	if (dynamic_cast<FileViewItem*>(item))
	{
		menu->popup(point);
	}
}

void FileListView::openDirSelected()
{
	if (FileViewItem *f = dynamic_cast<FileViewItem*>(selectedItem()))
		emit openDirectory(f->fileItem()->url().directory());
}
		
void FileListView::setMatching(const Matcher &matcher, bool state)
{
	QMap<SweepFileInfo *, bool> marked; //quick-lookup hash; the bool is NOT really used
	QListViewItemIterator it(this);
	int size = 0;
	int count = 0;
	//
	// mark all files in the current view;
	// mark all checkbox items in the current view
	// put all items of the current view in the lookup hash
	while (it.current())
	{
		if (FileViewItem *item = dynamic_cast<FileViewItem *>(*it))
		{
			if (SweepFileInfo *finfo = item->fileItem())
			{
				//
				// this is a bit tricky:
				// first mark the file item for removal, then turn the checkbox;
				// this causes the FileViewItem to NOT propagate the change to other views
				// as it thinks the item has not changed...
				if (finfo->isMarkedForRemoval() != state && finfo->getScore()>0 && matcher.matches(finfo))
				{
					size += finfo->size();
					++count;
					finfo->markForRemoval(state);
					item->setOn(state);
					marked.insert(finfo, true);
				}
			}
		}
		++it;
	}

	//
	// mark all checkboxes in remaining views for items in lookup cache
	for (FileListView *v = prevview; v; v = v->prevview)
	{
		QListViewItemIterator it(v);
		while (it.current())
		{
			FileViewItem *item = dynamic_cast<FileViewItem *>(*it);
			if (item && item->fileItem() && marked.contains(item->fileItem()))
				item->setOn(state);
			++it;
    		}
	}
	for (FileListView *v = nextview; v; v = v->nextview)
	{
		QListViewItemIterator it(v);
		while (it.current())
		{
			FileViewItem *item = dynamic_cast<FileViewItem *>(*it);
			if (item && item->fileItem() && marked.contains(item->fileItem()))
				item->setOn(state);
			++it;
    		}
	}
	if (!state)
		count = -count;

	emit selectionChanged(count, size);
}

void FileListView::selectAll()
{
	AllMatcher m;
	setMatching(m, true);
}

void FileListView::selectNone()
{
	AllMatcher m;
	setMatching(m, false);
}

void FileListView::set(const QRegExp &rx, bool state)
{
	RegExpMatcher m(rx);
	setMatching(m, state);
}

void FileListView::select(const QRegExp &rx)
{
	set(rx, true);
}

void FileListView::unselect(const QRegExp &rx)
{
	set(rx, false);
}

#include "filelistview.moc"
