#include "config.h"

#ifdef __LINUX__
#define __KERNEL__   // 無いとS_ISSOCKがlinux/stat.hで定義されない
#include <linux/stat.h>
#undef __KERNEL__
#endif

#if defined(HAVE_CURSES_H)
#include <curses.h>
#elif defined(HAVE_NCURSES_H)
#include <ncurses.h>
#elif defined(HAVE_NCURSES_NCURSES_H)
#include <ncurses/ncurses.h>
#endif

#if defined(__LINUX__)

#define __KERNEL__      //  無いとS_ISSOCKとかlinux/stat.hで定義されない

#include <sys/stat.h>     // ↑のマクロを使ってコンパイルしないといけない
#include <linux/stat.h>

#undef __KERNEL__
#endif

#if defined(__CYGWIN7__)
#include <sys/stat.h>
#endif

#ifdef __FREEBSD__
#include <limits.h>
#endif


#if !defined(__FREEBSD__)
extern int lstat (__const char *__restrict __file,
    struct stat *__restrict __buf) __attribute__ ((__nothrow__)) __attribute__ ((__nonnull__ (1, 2)));
#endif


#include "kitutuki.h"

#ifdef __LINUX__

#include <linux/limits.h>
#define __USE_POSIX
#define __USE_XOPEN_EXTENDED
#define __USE_POSIX199309
#define __USE_UNIX98
#include <signal.h>
#undef __USE_POSIX
#undef __USE_XOPEN_EXTENDED
#undef __USE_POSIX199309
#undef __USE_UNIX98
#include <sys/wait.h>

#else

#include <signal.h>
#include <sys/stat.h>
#include <sys/wait.h>

#endif

#include "kitutuki.h"

#include <errno.h>
#include <stdlib.h>
#include <fcntl.h>
#include <glob.h>
#include <libgen.h>
#include <stdio.h>
#include <string.h>
#include <sys/ioctl.h>
#include <sys/select.h>
#include <sys/types.h>
#include <sys/uio.h>
#include <termios.h>
#include <unistd.h>
#include <dirent.h>
#include <math.h>
#include <oniguruma.h>

#include <sys/types.h>
#define __USE_XOPEN
#include <sys/ipc.h>
#include <sys/msg.h>

#define FNAME_MAX                   128     // ファイル名の大きさに使う

extern int setenv(const char *name, const char *value, int overwrite);
extern int unsetenv(const char *name);
#if !defined(__FREEBSD__)
extern int kill (__pid_t __pid, int __sig) __THROW;
extern int killpg (__pid_t __pgrp, int __sig) __THROW;
#endif
#if defined(__LINUX__)
extern int lstat (__const char *__restrict __file,
		  struct stat *__restrict __buf) __THROW __nonnull ((1, 2));
extern int usleep (__useconds_t __useconds);
#endif

//////////////////////////////////////////////////////////////////////
// 宣言
//////////////////////////////////////////////////////////////////////
enum eLineField gScriptLineField = kLF;      // スクリプトの改行コード
enum eLineField gLineField = kLF;            // kitutukiが使う改行コード
enum eKanjiCode gKanjiCode = kUtf8;          // kitutukiが使う漢字コード

vector_obj* gPrograms;               // 補完候補
BOOL gKitutukiExit;                  // exitコマンドが実行された

/// 内部コマンド
enum eCommandKind { 
    kSubshell, kMSleep, kTrue, kFalse, kTest,
    kIndex, kRIndex, kLength, kLc, kUc,
    kChomp, kSubstr, kEval, kFg, kBg, kCpg,
    kJobs, kRehash, kKanjiCode, kLineField, kVar, kGlobal,
    kCalc, kExport, kPrint,  kPuts, kLoad, 
    kWhile, kBreak, kExit, kIf, kChoise, kSplit, kJoin, 
    kAryNew, 
    kAryAdd,  kAryErase, kAryClear, kMatch, kSub, kScan, kRead, kClose,
    kCd, 
    kAdd, kDel, kSelector, kMax, kMin, kExtName, kParentName, kNoExtName, 
    kCompile, kRaise,
    kPopd, kPushd,
    kCommand 
};

static char* gStatmentKindStrs[kCommand+1] = {
    "subshell", "msleep", "true",  "false", "[",  "index", "rindex", 
    "length",
    "lc",  "uc", "chomp", "substr", "eval", "fg", "bg", "cpg", "jobs", 
    "rehash", 
    "kanjicode", "linefield", "var", "global", "calc", "export", "print",  "puts",
    "load",
    "while", "break", "exit", "if", "choise", "split", "join", 
    "ary_new", 
    "ary_add", "ary_erase", "ary_clear", "match", "sub",  "scan", "read", "close",
    "cd", 
    "add", "del",  "selector", "max", "min", "extname", "parentname", "noextname", 
    "compile", "raise",
    "popd", "pushd",
    "command"
};

static hash_obj* gFuncs;                // ユーザー関数 
static hash_obj* gGlobals;              // グローバル変数
static hash_obj* gArrays;               // グローバル変数の配列

static vector_obj* gStackFrame;         // ローカル変数のスタック
static enum eAppType gAppType;          // アプリケーションの種類
static BOOL gJobControl;                // ジョブのコントロールをするかどうか
static char gFName[FNAME_MAX];          // 読み込んだスクリプトのファイル名
static int gLine;                          // 読み込んだスクリプトの行数
static vector_obj* gJobs;                  // ジョブ
static list_obj* gSigChldRecieved;  
                                // SIGCHLDを受け取ったPID,STATUSのリスト
static BOOL forground_job(int num);
static void background_job(int num);

static char* gMagicNumber = "KITUTUKI_OBJ";  // オブジェクトファイル
                                            // のマジックナンバー

static hash_obj* gFileHandles;              // ファイル記述子

static int gPipeFDs[2] = { -1, -1 };      // グローバルパイプ

static hash_obj* gRegexs;                   // 正規表現のキャッシュ
static hash_obj* gFBuffers;                 // 読み込みバッファ

static int gMsgId;                          // メッセージキュー

static hash_obj* gInnerCommands;          // 外部登録の内部コマンド

static vector_obj* gDirStack;

//////////////////////////////////////////////////////////////////////
// 内部関数
//////////////////////////////////////////////////////////////////////
static BOOL is_line_field(char* p)
{
    if(gScriptLineField == kLF && *p == '\n') {
        return TRUE;
    }
    else if(gScriptLineField == kCR && *p == '\r') {
        return TRUE;
    }
    else if(gScriptLineField == kCRLF && *p == '\r' && *(p+1) == '\n') {
        return TRUE;
    }

    return FALSE;
}

static BOOL is_line_field2(enum eLineField lf, char* p)
{
    if(lf == kLF && *p == '\n') {
        return TRUE;
    }
    else if(lf == kCR && *p == '\r') {
        return TRUE;
    }
    else if(lf == kCRLF && *p == '\r' && *(p+1) == '\n') {
        return TRUE;
    }

    return FALSE;
}

static void string_chomp(string_obj* str)
{
    char* s = string_c_str(str);
    const int len = strlen(s);

    if(len > 2 && s[len-2] == '\r' && s[len-1] == '\n')
    {
        string_trunc(str, len-2);
    }
    else if(len > 1 && s[len-1] == '\n') {
        string_trunc(str, len-1);
    }
    else if(len > 1 && s[len-1] == '\r') {
        string_trunc(str, len-1);
    }
}

static void xclose(int fd) 
{
    char buf[32];
    sprintf(buf, "%d", fd);

    string_obj* str = hash_item(gFBuffers, buf);
    if(str) {
        string_delete(str);
        hash_erase(gFBuffers, buf);
    }

    char* key = hash_key(gFileHandles, (char*)fd);
    if(key) {
        hash_erase(gFileHandles, key);
    }

    close(fd);
}

static void update_ary_env(char* key)
{
    vector_obj* v = hash_item(gArrays, key);

    if(v) {
        char name[128];

        sprintf(name, "%s_size", key);
        char number[128];
        sprintf(number, "%d", vector_size(v));

        string_obj* str = hash_item(gGlobals, name);
        if(str) string_delete(str);
        hash_put(gGlobals, name, STRING_NEW(number));
    }
}

static void clear_ary_env(char* key)
{
    char name[128];

    sprintf(name, "%s_size", key);

    string_obj* str = hash_item(gGlobals, name);
    if(str) string_delete(str);
    hash_erase(gGlobals, name);
}

char gErrMsg[512];
static void err_msg(char* msg)
{
    sprintf(gErrMsg, "%s %d: %s\n", gFName, gLine, msg);
}

static int get_command_kind(char* arg0)
{
    int i;
    for(i=0; i<kCommand; i++) {
        if(strcmp(arg0, gStatmentKindStrs[i]) == 0) {
            return i;
        }
    }

    return kCommand;
}

static void null_fun()
{
}

static BOOL parse_is_redirect(char* p)
{
    return (*p == '1' && *(p+1) == '>' && *(p+2) == '>') 
            || (*p == '1' && *(p+1) == '>') 
            || (*p == '2' && *(p+1) == '>' && *(p+2) == '>') 
            || (*p == '2' && *(p+1) == '>') 
            || (*p == '>' && *(p+1) == '>')
            || *p =='>' 
            || *p =='<';
}

static BOOL parse_is_redirect2(char* p)
{
    return *p == '2' && *(p+1) == '>' && *(p+2) == '&' && *(p+3) == '1'
        || *p == '&' && *(p+1) == '>';
}


static BOOL is_env_name_char(char c)
{
    return c >= 'a' && c <= 'z' || c >= 'A' && c <= 'Z' 
            || c >= '0' && c <= '9'
            || c == '_';
}

static BOOL is_quoted_char(char c)
{
    return c==' ' || c=='*' || c=='>' || c=='&' || c=='~' || c=='#' || c =='$'
    || c=='(' || c==')'
    || c=='\\' || c=='|' || c=='[' || c==']' || c=='{' || c=='}' || c==';'
    || c=='\'' || c=='"' || c=='<' || c=='>' || c=='?' || c=='\n' || c=='\t'
    || c=='\r' || c==':';
}

static BOOL is_glob_terminated_char(char* p)
{
    return *p=='\n' || *p=='\r' || *p==' ' || *p=='\t' || *p=='>' 
    || *p=='&' || *p=='~' 
    || *p=='#' || *p =='$' || *p=='(' || *p==')'
    || *p=='\\' || *p=='|' || *p=='{' || *p=='}' 
    || *p==';'
    || *p=='\'' || *p=='"' || *p=='<' || *p=='>' || *p==':' 
    || parse_is_redirect(p) || parse_is_redirect2(p);
}

static BOOL is_terminated_char(char* p)
{
    return *p=='\'' || *p=='"' || *p == '$' || *p == '%' || *p== '~' || *p=='*'
    || *p=='?' || *p=='[' || *p=='<' || *p=='>' || *p==' ' || *p=='\t' 
    || *p==':'
    || *p=='{' || *p=='|' || *p=='&' || *p==';' || *p=='\n' || *p=='\r' 
    || *p=='#' || *p==0
    || *p=='(' || parse_is_redirect(p) || parse_is_redirect2(p);
}

void kitutuki_get_quoted_fname(char* fname, string_obj* quoted_fname)
{
    char* p = fname;
    while(*p) {
        if(*p == '\n') {
            string_push_back(quoted_fname, "\\n");
            p++;
        }
        else if(*p == '\r') {
            string_push_back(quoted_fname, "\\r");
            p++;
        }
        else if(is_quoted_char(*p)) {
            string_push_back2(quoted_fname, '\\');
            string_push_back2(quoted_fname, *p++);
        }
        else {
            string_push_back2(quoted_fname, *p++);
        }
    }
}

static void get_quoted_linefield(char* str, string_obj* out)
{
    char* p = str;
    while(*p) {
        if(*p == '\n') {
            string_push_back(out, "\\n");
            p++;
        }
        else if(*p == '\r') {
            string_push_back(out, "\\r");
            p++;
        }
        else {
            string_push_back2(out, *p++);
        }
    }
}

void kitutuki_set_local_var(char* name, char* value)
{
    hash_obj* top_stack 
        = vector_item(gStackFrame, vector_size(gStackFrame)-1);

    string_obj* var = hash_item(top_stack, name);

    if(var) {
        string_put(var, value);
    }
    else {
        hash_put(top_stack , name, STRING_NEW(value));
    }
}

char* kitutuki_get_global_var(char* name)
{
    string_obj* global = hash_item(gGlobals, name);

    if(global) {
        return string_c_str(global);
    }
    else {
        return NULL;
    }
}

vector_obj* kitutuki_get_array(char* name)
{
    return hash_item(gArrays, name);
}

char* kitutuki_get_local_var(char* name)
{
    hash_obj* top_stack 
        = vector_item(gStackFrame, vector_size(gStackFrame)-1);

    string_obj* var = hash_item(top_stack, name);

    if(var) {
        return string_c_str(var);
    }
    else {
        return NULL;
    }
}

static void delete_local_var(char* name)
{
    hash_obj* top_stack 
        = vector_item(gStackFrame, vector_size(gStackFrame)-1);

    string_obj* var = hash_item(top_stack, name);

    if(var) {
        string_delete(var);
        hash_erase(top_stack, name);
    }
}

//////////////////////////////////////////////////////////////////////
// シグナル処理
//////////////////////////////////////////////////////////////////////
volatile BOOL gSaphireSigInt = FALSE;
volatile BOOL gSigUser = FALSE;

void sig_int()
{
    gSaphireSigInt = TRUE; 
#ifdef MDEBUG
fprintf(stderr, "SIGINT!!\n");
#endif
}

void sig_user()
{
    gSigUser = TRUE;
}


void sigchld_action(int signum, siginfo_t* info, void* ctx)
{
    list_push_back(gSigChldRecieved, (void*)info->si_pid);
}

void sigchld_block(int block)
{
    sigset_t sigset;

    sigemptyset(&sigset);
    sigaddset(&sigset, SIGCHLD);

    if(sigprocmask(block?SIG_BLOCK:SIG_UNBLOCK, &sigset, NULL) != 0)
    {
        fprintf(stderr, "error\n");
        exit(1);
    }
}

void sigttou_block(int block)
{
    sigset_t sigset;

    sigemptyset(&sigset);
    sigaddset(&sigset, SIGTTOU);

    if(sigprocmask(block?SIG_BLOCK:SIG_UNBLOCK, &sigset, NULL) != 0)
    {
        fprintf(stderr, "error\n");
        exit(1);
    }
}

void kitutuki_restore_signal_default()
{
    struct sigaction sa;

    memset(&sa, 0, sizeof(sa));
    sa.sa_handler = SIG_DFL;
    if(sigaction(SIGPIPE, &sa, NULL) < 0) {
        perror("sigaction1");
        exit(1);
    }
    if(sigaction(SIGCHLD, &sa, NULL) < 0) {
        perror("sigaction1");
        exit(1);
    }
    if(sigaction(SIGINT, &sa, NULL) < 0) {
        perror("sigaction2");
        exit(1);
    }
    if(sigaction(SIGCONT, &sa, NULL) < 0) {
        perror("sigaction3");
        exit(1);
    }
    if(sigaction(SIGWINCH, &sa, NULL) < 0) {
        perror("sigaction4");
        exit(1);
    }

    if(sigaction(SIGPROF, &sa, NULL) < 0) {
        perror("sigaction5");
        exit(1);
    }

    if(sigaction(SIGTTIN, &sa, NULL) < 0) {
        perror("sigaction6");
        exit(1);
    }
    if(sigaction(SIGTTOU, &sa, NULL) < 0) {
        perror("sigaction7");
        exit(1);
    }
    if(sigaction(SIGTSTP, &sa, NULL) < 0) {
        perror("sigaction8");
        exit(1);
    }
    if(sigaction(SIGQUIT, &sa, NULL) < 0) {
        perror("sigaction9");
        exit(1);
    }
    if(sigaction(SIGUSR1, &sa, NULL) < 0) {
        perror("sigaction10");
        exit(1);
    }
}

void (*kitutuki_set_signal_other)() = NULL;

void kitutuki_set_signal()
{
    struct sigaction sa;
    memset(&sa, 0, sizeof(sa));
    sa.sa_sigaction = sigchld_action;
    sa.sa_flags = SA_SIGINFO;
    //sa.sa_flags = SA_RESTART | SA_SIGINFO;
    if(sigaction(SIGCHLD, &sa, NULL) < 0) {
        perror("sigaction1");
        exit(1);
    }

    struct sigaction sa2;
    memset(&sa2, 0, sizeof(sa2));
    sa2.sa_handler = sig_int;
    //sa2.sa_flags |= SA_RESTART;
    if(sigaction(SIGINT, &sa2, NULL) < 0) {
        perror("sigaction2");
        exit(1);
    }

    struct sigaction sa4;

    memset(&sa4, 0, sizeof(sa4));
    sa4.sa_handler = SIG_IGN;
    //sa4.sa_flags |= SA_RESTART;
    if(sigaction(SIGCONT, &sa4, NULL) < 0) {
        perror("sigaction4");
        exit(1);
    }

    struct sigaction sa5;

    memset(&sa5, 0, sizeof(sa5));
    sa5.sa_handler = SIG_IGN;
    //sa5.sa_flags |= SA_RESTART;
    if(sigaction(SIGWINCH, &sa5, NULL) < 0) {
        perror("sigaction5");
        exit(1);
    }

    struct sigaction sa10;

    memset(&sa10, 0, sizeof(sa10));
    sa10.sa_handler = SIG_IGN;
    sa10.sa_flags = 0;
    //sa10.sa_flags |= SA_RESTART;
    if(sigaction(SIGTTOU, &sa10, NULL) < 0) {
        perror("sigaction10");
        exit(1);
    }

    struct sigaction sa11;

    memset(&sa11, 0, sizeof(sa11));
    sa11.sa_handler = SIG_IGN;
    sa11.sa_flags = 0;
    //sa11.sa_flags |= SA_RESTART;
    if(sigaction(SIGTTIN, &sa11, NULL) < 0) {
        perror("sigaction11");
        exit(1);
    }

    struct sigaction sa12;
    memset(&sa12, 0, sizeof(sa12));
    sa12.sa_handler = SIG_IGN;
    sa12.sa_flags = 0;
    //sa12.sa_flags |= SA_RESTART;
    if(sigaction(SIGTSTP, &sa12, NULL) < 0) {
        perror("sigaction12");
        exit(1);
    }

    struct sigaction sa13;
    memset(&sa13, 0, sizeof(sa13));
    sa13.sa_handler = SIG_IGN;
    sa13.sa_flags = 0;
    //sa13.sa_flags |= SA_RESTART;
    if(sigaction(SIGQUIT, &sa13, NULL) < 0) {
        perror("sigaction13");
        exit(1);
    }

    struct sigaction sa14;
    memset(&sa14, 0, sizeof(sa14));
    sa14.sa_handler = sig_user;
    sa14.sa_flags = 0;
    //sa14.sa_flags |= SA_RESTART;
    if(sigaction(SIGUSR1, &sa14, NULL) < 0) {
        perror("sigaction14");
        exit(1);
    }
    memset(&sa2, 0, sizeof(sa2));
    sa2.sa_handler = SIG_IGN;
    sa2.sa_flags = 0;
    //sa2.sa_flags = SA_RESTART;
    if(sigaction(SIGPIPE, &sa2, NULL) < 0) {
        perror("sigaction1");
        exit(1);
    }

    if(kitutuki_set_signal_other) kitutuki_set_signal_other();
}

//////////////////////////////////////////////////////////////////////
// ジョブ
//////////////////////////////////////////////////////////////////////
typedef struct _sJob {
    string_obj* mName;              // ジョブの名前
    pid_t mPGroup;                  // プロセスグループID
    vector_obj* mPIDs;              // 子プロセスのPIDのリスト
    struct termios* mTty;
} sJob;

sJob* sJob_new() 
{
    sJob* self = (sJob*)MALLOC(sizeof(sJob));
    
    self->mName = STRING_NEW("");
    self->mPGroup = 0;
    self->mPIDs = VECTOR_NEW(5);
    self->mTty = NULL;

    return self;
}

void sJob_delete(sJob* self)
{
    string_delete(self->mName);
    vector_delete(self->mPIDs);
    if(self->mTty) FREE(self->mTty);

    FREE(self);
}

//////////////////////////////////////////////////////////////////////
// リダイレクト
//////////////////////////////////////////////////////////////////////
enum eRedirect { kRedirectInput, kRedirectOverwrite, kRedirectAppend
          , kRedirectErrAndOutput, kRedirectGPipeOut, kRedirectGPipeIn};

typedef struct {
    enum eRedirect mType;
    int mFd;
    string_obj* mFName;
} sRedirect;

#ifdef MDEBUG

sRedirect* sRedirect_new(const char* fname, int line, const char* func_name)
{
    sRedirect* self = (sRedirect*)CheckMemLeak_Malloc(sizeof(sRedirect), fname, line, func_name);
    self->mFName = STRING_NEW("");
    return self;
}

#define REDIRECT_NEW() sRedirect_new(__FILE__, __LINE__, __FUNCTION__)

#else

sRedirect* sRedirect_new()
{
    sRedirect* self = (sRedirect*)MALLOC(sizeof(sRedirect));
    self->mFName = STRING_NEW("");
    return self;
}

#define REDIRECT_NEW() sRedirect_new()

#endif

void sRedirect_save(sRedirect* self, int fd)
{
    unsigned char uc;
    int n;
    
    uc = self->mType;
    if(write(fd, &uc, sizeof(unsigned char)) < 0) {
        perror("write");
        exit(1);
    }
    
    n = self->mFd;
    if(write(fd, &n, sizeof(int)) < 0) {
        perror("write");
        exit(1);
    }
    
    n = string_size(self->mFName);
    if(write(fd, &n, sizeof(int)) < 0) {
        perror("write");
        exit(1);
    }
    
    if(write(fd, string_c_str(self->mFName)
                , string_size(self->mFName)) < 0) 
    {
        perror("write");
        exit(1);
    }
}

sRedirect* sRedirect_load(int fd)
{
    unsigned char uc;
    int n;
    
    sRedirect* self = (sRedirect*)MALLOC(sizeof(sRedirect));
    
    if(read(fd, &uc, sizeof(unsigned char)) < 0) {
        perror("read");
        exit(1);
    }
    self->mType = uc;
    
    if(read(fd, &n, sizeof(int)) < 0) {
        perror("read");
        exit(1);
    }
    self->mFd = n;
    
    if(read(fd, &n, sizeof(int)) < 0) {
        perror("read");
        exit(1);
    }
    
    char* buf = MALLOC(n+1);
    if(read(fd, buf, n) < 0) {
        perror("read");
        exit(1);
    }
    buf[n] = 0;
    self->mFName = STRING_NEW(buf);
    FREE(buf);
    
    return self;
}

void sRedirect_delete(sRedirect* self)
{
    string_delete(self->mFName);
    FREE(self);
}

///////////////////////////////////////////////////////////////////////
// sWhile
////////////////////////////////////////////////////////////////////////
enum eStatmentTerminated { kTNormal, kTOrOr, kTAndAnd };

void sStatments_delete(sStatments* self);
void sStatments_save(sStatments* self, int fd);
sStatments* sStatments_load(int fd);

typedef struct
{
    sStatments* joukensiki;
    sStatments* contents;
} sWhile;

sWhile* sWhile_new(MANAGED sStatments* joukensiki, MANAGED sStatments* contents)
{
    sWhile* self = MALLOC(sizeof(sWhile));

    self->joukensiki = joukensiki;
    self->contents = contents;

    return self;
}

void sWhile_delete(sWhile* self)
{
    sStatments_delete(self->joukensiki);
    sStatments_delete(self->contents);

    FREE(self);
}

void sWhile_save(sWhile* self, int fd)
{
    sStatments_save(self->joukensiki, fd);
    sStatments_save(self->contents, fd);
}

sWhile* sWhile_load(int fd)
{
    sWhile* self = MALLOC(sizeof(sWhile));
    
    self->joukensiki = sStatments_load(fd);
    self->contents = sStatments_load(fd);
    
    return self;
}

////////////////////////////////////////////////////////////////////////
// sIf
////////////////////////////////////////////////////////////////////////
typedef struct
{
    vector_obj* joukensiki_list;
    vector_obj* contents_list;
} sIf;

sIf* sIf_new(MANAGED vector_obj* joukensiki_list, MANAGED vector_obj* contents_list)
{
    sIf* self = MALLOC(sizeof(sIf));
    
    self->joukensiki_list = joukensiki_list;
    self->contents_list = contents_list;

    return self;
}

void sIf_delete(sIf* self)
{
    int i;
    for(i=0; i<vector_size(self->joukensiki_list); i++) {
        sStatments_delete(vector_item(self->joukensiki_list, i));
    }
    vector_delete(self->joukensiki_list);

    for(i=0; i<vector_size(self->contents_list); i++) {
        sStatments_delete(vector_item(self->contents_list, i));
    }
    vector_delete(self->contents_list);

    FREE(self);
}

void sIf_save(sIf* self, int fd)
{
    int n;
    unsigned char c;
    int i;
    
    n = vector_size(self->joukensiki_list);
    if(write(fd, &n, sizeof(int)) < 0) {
        perror("write");
        exit(1);
    }
    for(i=0; i<n; i++) {
        sStatments_save(vector_item(self->joukensiki_list, i), fd);
    }
    
    n = vector_size(self->contents_list);
    if(write(fd, &n, sizeof(int)) < 0) {
        perror("write");
        exit(1);
    }
    for(i=0; i<n; i++) {
        sStatments_save(vector_item(self->contents_list, i), fd);
    }
}

sIf* sIf_load(int fd)
{
    int n;
    int i;

    sIf* self = MALLOC(sizeof(sIf));
    
    self->joukensiki_list = VECTOR_NEW(3);
    if(read(fd, &n, sizeof(int)) < 0) {
        perror("read");
        exit(1);
    }
    for(i=0; i<n; i++) {
        vector_add(self->joukensiki_list, sStatments_load(fd));
    }

    self->contents_list = VECTOR_NEW(3);
    if(read(fd, &n, sizeof(int)) < 0) {
        perror("read");
        exit(1);
    }
    for(i=0; i<n; i++) {
        vector_add(self->contents_list, sStatments_load(fd));
    }
    
    return self;
}


////////////////////////////////////////////////////////////////////////
// コマンド ls -al
////////////////////////////////////////////////////////////////////////
typedef struct
{
    enum eCommandKind mKind;
    vector_obj* mArgs;      // コンパイル済みの文 vector of string_obj*

    vector_obj* mRedirects;     // リダイレクションのリスト

    void* mExtra;
} sCommand;

#ifdef MDEBUG 

sCommand* sCommand_new(const char* file, int line, const char* function)
{
    sCommand* self = CheckMemLeak_Malloc(sizeof(sCommand), file, line, "COMMAND_NEW");

    self->mKind = kCommand;
    self->mArgs = VECTOR_NEW(5);

    self->mRedirects = VECTOR_NEW(1);

    self->mExtra = NULL;

    return self;
}

#define COMMAND_NEW() sCommand_new(__FILE__, __LINE__, __FUNCTION__)
void sCommand_view(sCommand* self)
{
    printf("self->mKind %s\n", gStatmentKindStrs[self->mKind]);

    int i;
    for(i=0; i<vector_size(self->mArgs); i++) {
        printf("self->mArgs[%d] %s\n"
                , i, string_c_str(vector_item(self->mArgs, i)));
    }
}

#else

sCommand* sCommand_new()
{
    sCommand* self = MALLOC(sizeof(sCommand));

    self->mKind = kCommand;
    self->mArgs = VECTOR_NEW(5);

    self->mRedirects = VECTOR_NEW(1);

    self->mExtra = NULL;

    return self;
}

#define COMMAND_NEW() sCommand_new()

#endif

void sCommand_save(sCommand* self, int fd)
{
    int i;
    unsigned char uc;
    int n;
    
    /// mKind ///
    uc = self->mKind;
    if(write(fd, &uc, sizeof(unsigned char)) < 0) {
        perror("write");
        exit(1);
    }
    
    /// mArgs ///
    n = vector_size(self->mArgs);
    if(write(fd, &n, sizeof(int)) < 0) {
        perror("write");
        exit(1);
    }
    
    for(i=0; i<vector_size(self->mArgs); i++) {
        string_obj* item = vector_item(self->mArgs, i);
        
        n = string_size(item);
        if(write(fd, &n, sizeof(int)) < 0) {
            perror("write");
            exit(1);
        }
        
        if(write(fd, string_c_str(item), n) < 0) {
            perror("write");
            exit(1);
        }
    }
    
    /// mRedirects ///
    n = vector_size(self->mRedirects);
    if(write(fd, &n, sizeof(int)) < 0) {
        perror("write");
        exit(1);
    }
    
    for(i=0; i<vector_size(self->mRedirects); i++) {
        sRedirect* r = vector_item(self->mRedirects, i);
        
        sRedirect_save(r, fd);
    }
    
    /// mExsta ///
    switch(self->mKind) {
    case kIf:
        sIf_save(self->mExtra, fd);
        break;
        
    case kWhile:
        sWhile_save(self->mExtra, fd);
        break;
    }
}

sCommand* sCommand_load(int fd)
{
    int n, m;
    unsigned char uc;
    int i;
    
    sCommand* self = MALLOC(sizeof(sCommand));
    
    if(read(fd, &uc, sizeof(unsigned char)) < 0) {
        perror("read");
        exit(1);
    }
    self->mKind = uc;
    
    self->mArgs = VECTOR_NEW(5);
    
    if(read(fd, &n, sizeof(int)) < 0) {
        perror("read");
        exit(1);
    }
    for(i=0; i<n; i++) {
        if(read(fd, &m, sizeof(int)) < 0) {
            perror("read");
            exit(1);
        }
        
        char* buf = MALLOC(m+1);
        if(read(fd, buf, m) < 0) {
            perror("read");
            exit(1);
        }
        buf[m] = 0;
        
        vector_add(self->mArgs, STRING_NEW(buf));
        
        FREE(buf);
    }
    
    self->mRedirects = VECTOR_NEW(1);
    
    if(read(fd, &n, sizeof(int)) < 0) {
        perror("read");
        exit(1);
    }
    
    for(i=0; i<n; i++) {
        vector_add(self->mRedirects, sRedirect_load(fd));
    }

    switch(self->mKind) {
    case kIf:
        self->mExtra = sIf_load(fd);
        break;
        
    case kWhile:
        self->mExtra = sWhile_load(fd);
        break;
        
    default:
        self->mExtra = NULL;
    }
    
    return self;
}

void sCommand_delete(sCommand* self)
{
    int i;

    for(i=0; i<vector_size(self->mArgs); i++) {
        string_delete(vector_item(self->mArgs, i));
    }
    vector_delete(self->mArgs);

    for(i=0; i<vector_size(self->mRedirects); i++) {
        sRedirect_delete(vector_item(self->mRedirects, i));
    }
    vector_delete(self->mRedirects);

    switch(self->mKind) {
    case kIf:
        sIf_delete(self->mExtra);
        break;

    case kWhile:
        sWhile_delete(self->mExtra);
        break;
    }

    FREE(self);
}

///////////////////////////////////////////////////////////////////////
// 文 ls -al | grep aaa
///////////////////////////////////////////////////////////////////////
typedef struct
{
    vector_obj* mCommands;             // コマンド

    string_obj* mFName;                 // 複文があるファイル名
    int mLine;                          // 複文がある行

    string_obj* mNotEvaled;             // run時にパースされる文
                                        // 環境変数とマクロを含む文
    string_obj* mTitle;                 // タイトル

    enum eStatmentTerminated mTerminated;   // 複文の最後が何で区切られているか

    BOOL mBackground;                   // バックグラウンドで実行
} sStatment;

#ifdef MDEBUG

sStatment* sStatment_new(const char* file, int line, const char* function)
{
    sStatment* self = CheckMemLeak_Malloc(sizeof(sStatment), file, line, "STATMENT_NEW");

    self->mCommands = VECTOR_NEW(5);
    self->mTerminated = kTNormal;
    self->mFName = STRING_NEW("");
    self->mLine = 0;
    self->mTitle = STRING_NEW("");
    self->mNotEvaled = NULL;
    self->mBackground = FALSE;

    return self;
}

#define STATMENT_NEW() sStatment_new(__FILE__, __LINE__, __FUNCTION__)

void sStatment_view(sStatment* self)
{
    printf("command number %d\n", vector_size(self->mCommands));
    int i;
    for(i=0; i<vector_size(self->mCommands); i++) {
        printf("command %d ---\n", i);
        sCommand_view(vector_item(self->mCommands, i));
    }

    if(self->mTerminated == kTNormal)
        printf("self->mTerminated kTNormal\n");
    else if(self->mTerminated == kTOrOr)
        printf("self->mTerminated kTOrOr\n");
    else if(self->mTerminated == kTAndAnd)
        printf("self->mTerminated kTAndAnd\n");

    printf("self->mFName %s\n", string_c_str(self->mFName));
    printf("self->mLine %d\n", self->mLine);
    printf("self->mTitle %s\n", string_c_str(self->mTitle));

    if(self->mNotEvaled) 
        printf("self->mNotEvaled %s\n", string_c_str(self->mNotEvaled));
    else
        printf("self->mNotEvaled NULL\n");

    if(self->mBackground) 
        printf("self->mBackground true\n");
    else
        printf("self->mBackground false\n");
}

#else

sStatment* sStatment_new()
{
    sStatment* self = MALLOC(sizeof(sStatment));

    self->mCommands = VECTOR_NEW(5);
    self->mTerminated = kTNormal;
    self->mFName = STRING_NEW("");
    self->mLine = 0;
    self->mTitle = STRING_NEW("");
    self->mNotEvaled = NULL;
    self->mBackground = FALSE;

    return self;
}

#define STATMENT_NEW() sStatment_new()

#endif

void sStatment_delete(sStatment* self)
{
    int i;
    for(i=0; i<vector_size(self->mCommands); i++) {
        sCommand_delete(vector_item(self->mCommands, i));
    }
    vector_delete(self->mCommands);

    string_delete(self->mFName);
    string_delete(self->mTitle);

    if(self->mNotEvaled) { string_delete(self->mNotEvaled); }

    FREE(self);
}

void sStatment_save(sStatment* self, int fd)
{
    int n;
    unsigned char c;
    int i;
    
    n = vector_size(self->mCommands);
    if(write(fd, &n, sizeof(int)) < 0) {
        perror("write");
        exit(1);
    }
    
    for(i=0; i<n; i++) {
        sCommand* item = vector_item(self->mCommands, i);
        sCommand_save(item, fd);
    }
    
    c = self->mTerminated;
    if(write(fd, &c, sizeof(unsigned char)) < 0) {
        perror("write");
        exit(1);
    }
    
    n = string_size(self->mFName);
    if(write(fd, &n, sizeof(int)) < 0) {
        perror("write");
        exit(1);
    }
    int wret = write(fd, string_c_str(self->mFName), n);
    
    n = self->mLine;
    if(write(fd, &n, sizeof(int)) < 0) {
        perror("write");
        exit(1);
    }
    
    n = string_size(self->mTitle);
    if(write(fd, &n, sizeof(int)) < 0) {
        perror("write");
        exit(1);
    }
    if(write(fd, string_c_str(self->mTitle), n) < 0) {
        perror("write");
        exit(1);
    }
    
    if(self->mNotEvaled) {
        n = 1;
        if(write(fd, &n, sizeof(int)) < 0) {
            perror("write");
            exit(1);
        }
        
        n = string_size(self->mNotEvaled);
        if(write(fd, &n, sizeof(int)) < 0) {
            perror("write");
            exit(1);
        }
        if(write(fd, string_c_str(self->mNotEvaled), n) < 0) {
            perror("write");
            exit(1);
        }
    }
    else {
        n = 0;
        if(write(fd, &n, sizeof(int)) < 0) {
            perror("write");
            exit(1);
        }
    }
    
    n = self->mBackground;
    if(write(fd, &n, sizeof(int)) < 0) {
        perror("write");
        exit(1);
    }
}

sStatment* sStatment_load(int fd)
{
    int n;
    unsigned char c;
    int i;
    
    sStatment* self = MALLOC(sizeof(sStatment));
    
    self->mCommands = VECTOR_NEW(5);
    
    if(read(fd, &n, sizeof(int)) < 0) {
        perror("read");
        exit(1);
    }
    for(i=0; i<n; i++) {
        vector_add(self->mCommands, sCommand_load(fd));
    }
    
    if(read(fd, &c, sizeof(unsigned char)) < 0) {
        perror("read");
        exit(1);
    }
    self->mTerminated = c;
    
    if(read(fd, &n, sizeof(int)) < 0) {
        perror("read");
        exit(1);
    }
    char* buf = MALLOC(n+1);
    if(read(fd, buf, n) < 0) {
        perror("read");
        exit(1);
    }
    buf[n] = 0;
    self->mFName = STRING_NEW(buf);
    FREE(buf);
    
    if(read(fd, &n, sizeof(int)) < 0) {
        perror("read");
        exit(1);
    }
    self->mLine = n;

    if(read(fd, &n, sizeof(int)) < 0) {
        perror("read");
        exit(1);
    }
    buf = MALLOC(n+1);
    if(read(fd, buf, n) < 0) {
        perror("read");
        exit(1);
    }
    buf[n] = 0;
    self->mTitle = STRING_NEW(buf);
    FREE(buf);
    
    if(read(fd, &n, sizeof(int)) < 0) {
        perror("read");
        exit(1);
    }
    if(n == 0) {
        self->mNotEvaled = NULL;
    }
    else {
        if(read(fd, &n, sizeof(int)) < 0) {
            perror("read");
            exit(1);
        }
        buf = MALLOC(n+1);
        if(read(fd, buf, n) < 0) {
            perror("read");
            exit(1);
        }
        buf[n] = 0;
        self->mNotEvaled = STRING_NEW(buf);
        FREE(buf);
    }
    
    if(read(fd, &n, sizeof(int)) < 0) {
        perror("read");
        exit(1);
    }
    self->mBackground = n;
    
    return self;
}

//////////////////////////////////////////////////////////////////////
// 複文 ls -al | grep aaa; pwd; cd aaa
//////////////////////////////////////////////////////////////////////
#ifdef MDEBUG
sStatments* sStatments_new(const char* file, int line, const char* function)
{
    sStatments* self = CheckMemLeak_Malloc(sizeof(sStatments)
                            , file, line, "STATMENTSLIST_NEW");

    self->mStatments = VECTOR_NEW(30);

    return self;
}

#define STATMENTSLIST_NEW() sStatments_new(__FILE__, __LINE__, __FUNCTION__)

void sStatments_view(sStatments* self)
{
    int i;
    for(i=0; i<vector_size(self->mStatments); i++)
        sStatment_view(vector_item(self->mStatments, i));
}
#else

sStatments* sStatments_new()
{
    sStatments* self = MALLOC(sizeof(sStatments));

    self->mStatments = VECTOR_NEW(30);

    return self;
}

#define STATMENTSLIST_NEW() sStatments_new()

#endif

void sStatments_save(sStatments* self, int fd)
{
    int i;
    int n;
    
    n = vector_size(self->mStatments);
    
    if(write(fd, &n, sizeof(int)) < 0) {
        perror("write");
        exit(1);
    }
    
    for(i=0; i<vector_size(self->mStatments); i++) {
        sStatment_save(vector_item(self->mStatments, i), fd);
    }
}

sStatments* sStatments_load(int fd)
{
    int n;
    int i;
    sStatments* self = MALLOC(sizeof(sStatments));
    
    self->mStatments = VECTOR_NEW(30);
    
    if(read(fd, &n, sizeof(int)) < 0) {
        perror("read");
        exit(1);
    }
    
    for(i=0; i<n; i++) {
        vector_add(self->mStatments, sStatment_load(fd));
    }
    
    return self;
}

void sStatments_delete(sStatments* self)
{
    int i;
    for(i=0; i<vector_size(self->mStatments); i++) {
        sStatment_delete(vector_item(self->mStatments, i));
    }
    vector_delete(self->mStatments);

    FREE(self);
}

//////////////////////////////////////////////////////////////////////
// 関数
//////////////////////////////////////////////////////////////////////
typedef struct {
    string_obj* name;       // コマンド名
    string_obj* arg_name;
    sStatments* statments; // 命令表
} sFunction;

static sFunction* sFunction_new(char* name, MANAGED sStatments* statments, char* arg_name)
{
    sFunction* self = MALLOC(sizeof(sFunction));
    self->name = STRING_NEW(name);
    self->arg_name = STRING_NEW(arg_name);
    
    self->statments = statments;

    return self;
}

static void sFunction_delete(sFunction* self)
{
    string_delete(self->name);
    string_delete(self->arg_name);

    sStatments_delete(self->statments);

    FREE(self);
}

static void sFunction_save(sFunction* self, int fd)
{
    int n = string_length(self->name);
    if(write(fd, &n, sizeof(int)) < 0) {
        perror("write");
        exit(1);
    }
    if(write(fd, string_c_str(self->name), sizeof(char)*n) < 0) {
        perror("write");
        exit(1);
    }
    n = string_length(self->arg_name);
    if(write(fd, &n, sizeof(int)) < 0) {
        perror("write");
        exit(1);
    }
    if(write(fd, string_c_str(self->arg_name), sizeof(char)*n) < 0) {
        perror("write");
        exit(1);
    }

    sStatments_save(self->statments, fd);
}

static sFunction* sFunction_load(int fd)
{
    sFunction* self = MALLOC(sizeof(sFunction));
    
    int n;
    if(read(fd, &n, sizeof(int)) < 0) {
        perror("read");
        exit(1);
    }

    char* buf = MALLOC(sizeof(char)*(n+1));
    if(read(fd, buf, sizeof(char)*n) < 0) {
        perror("read");
        exit(1);
    }
    buf[n] = 0;

    self->name = STRING_NEW(buf);
    FREE(buf);

    if(read(fd, &n, sizeof(int)) < 0) {
        perror("read");
        exit(1);
    }

    buf = MALLOC(sizeof(char)*(n+1));
    if(read(fd, buf, sizeof(char)*n) < 0) {
        perror("read");
        exit(1);
    }
    buf[n] = 0;

    self->arg_name = STRING_NEW(buf);
    FREE(buf);

    self->statments = sStatments_load(fd);

    return self;
}


//////////////////////////////////////////////////////////////////////
// run_wait_cprog で使う
//////////////////////////////////////////////////////////////////////
typedef struct
{
    int mPid;
    int mStatus;
    int mPGroup;
} sOtherCProg;

static vector_obj* gOtherCProgs;

static sOtherCProg* sOtherCprog_new(int pid, int status, int pgroup)
{
    sOtherCProg* self = MALLOC(sizeof(sOtherCProg));
    
    self->mPid = pid;
    self->mStatus = status;
    self->mPGroup = pgroup;
    
    return self;
}

static void sOtherCProg_delete(sOtherCProg* self)
{
    FREE(self);
}

//////////////////////////////////////////////////////////////////////
// 子プロセス
//////////////////////////////////////////////////////////////////////
typedef struct {
    pid_t mPid;
    string_obj* mTitle;
} sChildProgram;

static sChildProgram* sChildProgram_new(pid_t pid, char* title)
{
    sChildProgram* self = (sChildProgram*)MALLOC(sizeof(sChildProgram));

    self->mPid = pid;
    self->mTitle = STRING_NEW(title);

    return self;
}

static void sChildProgram_delete(sChildProgram* self)
{
    string_delete(self->mTitle);

    FREE(self);
}

////////////////////////////////////////////////////////////
// 内部コマンド
////////////////////////////////////////////////////////////
typedef struct {
    string_obj* mName;
    fInnerCommand mFun;
} sInnerCommand;

static sInnerCommand* sInnerCommand_new(char* name, fInnerCommand fun)
{
    sInnerCommand* self = MALLOC(sizeof(sInnerCommand));

    self->mName = STRING_NEW(name);
    self->mFun = fun;

    return self;
}

static void sInnerCommand_delete(sInnerCommand* self)
{
    string_delete(self->mName);
    FREE(self);
}

//////////////////////////////////////////////////////////////////////
// プログラム名補完候補を再読み込み
//////////////////////////////////////////////////////////////////////
static BOOL name_sort(void* left, void* right)
{
    string_obj* l = left;
    string_obj* r = right;

    return strcmp(string_c_str(l), string_c_str(r)) < 0;
}

static void kitutuki_rehash_without_user_fun()
{
    hash_obj* check = HASH_NEW(10);

    int i;
    for(i=0; i<vector_size(gPrograms); i++) {
        string_delete(vector_item(gPrograms, i));
    }
    vector_clear(gPrograms);

    /// 内部コマンド ///
    for(i=0; i<kCommand; i++) {
        vector_add(gPrograms, STRING_NEW(gStatmentKindStrs[i]));
    }

    /// 外部登録の内部コマンド ///
    hash_it* it = hash_loop_begin(gInnerCommands);
    while(it != NULL) {
        sInnerCommand* command = hash_loop_item(it);
        vector_add(gPrograms, STRING_NEW(string_c_str(command->mName)));
        it = hash_loop_next(it);
    }

    /// $PATHにあるファイルを全てgProgramsに入れていく ///
    char* path = getenv("PATH");
    if(path == NULL) {
        fprintf(stderr, "$PATH is NULL\n");
        exit(1);
    }

    char* p = path;
    char buf[4096];
    char* p2 = buf;

    while(*p) {
        if(*p != ':') {
            *p2++ = *p;
        }
        else {
            *p2 = 0;

            DIR* dir = opendir(buf);
            if(dir) {
                struct dirent* entry;
                while(entry = readdir(dir)) {
                    char path2[PATH_MAX];
                    sprintf(path2, "%s/%s", buf, entry->d_name);
                    
                    struct stat stat_;
                    memset(&stat_, 0, sizeof(struct stat));
                    stat(path2, &stat_);

                    if(strcmp(entry->d_name, ".") != 0
                            && strcmp(entry->d_name, "..") != 0
                            &&
                            (stat_.st_mode & S_IXUSR
                             ||stat_.st_mode & S_IXGRP
                             ||stat_.st_mode & S_IXOTH))
                    {
                        if(hash_item(check, entry->d_name) == NULL) {
                            hash_put(check, entry->d_name, (void*)1);
#if defined(__CYGWIN7__)
                            if(strstr(entry->d_name, ".exe") 
                                == entry->d_name + strlen(entry->d_name) 
                                    -4) 
                            {
                                char buf[128];
                                memcpy(buf, entry->d_name
                                        , strlen(entry->d_name) -4);
                                buf[strlen(entry->d_name) -4] = 0;
                                vector_add(gPrograms, STRING_NEW(buf));
                            }
                            else {
                                vector_add(gPrograms
                                        , STRING_NEW(entry->d_name));
                            }
                            
#else
                            vector_add(gPrograms
                                        , STRING_NEW(entry->d_name));
#endif
                        }
                    }
                }

                closedir(dir);
            }

            p2 = buf;
        }

        p++;
    }
    *p2 = 0;

    DIR* dir = opendir(buf);
    if(dir) {
        struct dirent* entry;
        while(entry = readdir(dir)) {
            if(strcmp(entry->d_name, ".") != 0
                    && strcmp(entry->d_name, "..") != 0)
            {
                vector_add(gPrograms, STRING_NEW(entry->d_name));
            }
        }

        closedir(dir);
    }

    vector_sort(gPrograms, name_sort);

    hash_delete(check);
}

static void kitutuki_rehash_user_fun()
{
    /// ユーザーオリジナル関数 ///
    hash_it* it = hash_loop_begin(gFuncs);
    while(it != NULL) {
        sFunction* fun = hash_loop_item(it);
        vector_add(gPrograms, STRING_NEW(string_c_str(fun->name)));
        it = hash_loop_next(it);
    }
}

void kitutuki_rehash()
{
    kitutuki_rehash_without_user_fun();
    kitutuki_rehash_user_fun();
}

//////////////////////////////////////////////////////////////////////
// ランタイムスクリプト実行
//////////////////////////////////////////////////////////////////////
static void read_rc_file()
{
    char rc_fname[PATH_MAX];
    sprintf(rc_fname,"%s/kitutuki.ksh", SYSCONFDIR);

    if(access(rc_fname, R_OK) == 0) {
        if(kitutuki_load(rc_fname
           , STDOUT_FILENO, STDIN_FILENO, STDERR_FILENO) < 0)
        {
            fprintf(stderr, "%s", gErrMsg);
            exit(1);
        }
    }
    else {
        fprintf(stderr, "can't find %s file\n", rc_fname);
        exit(1);
    }

    /// ユーザーのランタイムスクリプト ////
    char* user_rc_fname = getenv("KITUTUKI_USERRC");
    if(user_rc_fname) {
        if(access(user_rc_fname, R_OK) == 0) {
            if(kitutuki_load(user_rc_fname
                , STDOUT_FILENO, STDIN_FILENO, STDERR_FILENO) < 0)
            {
                fprintf(stderr, "%s", gErrMsg);
                exit(1);
            }
        }
    }
}

static void read_rc_obj_file()
{
    char rc_fname[PATH_MAX];
    sprintf(rc_fname,"%s/kitutuki.kio", SYSCONFDIR);

    if(access(rc_fname, R_OK) == 0) {
        if(kitutuki_load_obj(rc_fname
           , STDOUT_FILENO, STDIN_FILENO, STDERR_FILENO) < 0)
        {
            fprintf(stderr, "%s", gErrMsg);
            exit(1);
        }
    }
    else {
        fprintf(stderr, "can't find %s file\n", rc_fname);
        exit(1);
    }

    /// ユーザーのランタイムスクリプト ////
    char* user_rc_fname = getenv("KITUTUKI_USERRC");
    if(user_rc_fname) {
        if(access(user_rc_fname, R_OK) == 0) {
            if(kitutuki_load_obj(user_rc_fname
                , STDOUT_FILENO, STDIN_FILENO, STDERR_FILENO) < 0)
            {
                fprintf(stderr, "%s", gErrMsg);
                exit(1);
            }
        }
    }
}

//////////////////////////////////////////////////////////////////////
// 初期化
//////////////////////////////////////////////////////////////////////
void kitutuki_init(enum eAppType app_type, BOOL job_control, enum eRuntimeScript runtime_script)
{
    gAppType = app_type;
    gJobControl = job_control;

    gArrays = HASH_NEW(5);
    gFuncs = HASH_NEW(100);

    gJobs = VECTOR_NEW(10);
    gSigChldRecieved = LIST_NEW();

    gGlobals = HASH_NEW(50);

    gFName[0] = 0;
    gLine = 0;

    gFileHandles = HASH_NEW(10);

    setenv("KITUTUKI_VERSION", "0.9.9c", 1);
    setenv("KITUTUKI_PROMPT", "> ", 1);
    setenv("KITUTUKI_DOCDIR", DOCDIR, 1);
    char* home = getenv("HOME");
    if(home) {
        char path[PATH_MAX];
        sprintf(path, "%s/.kitutuki_history", home);
        setenv("KITUTUKI_HISTFILE", path, 1);
    }
    setenv("KITUTUKI_HISTSIZE", "1000", 1);
    char* user_rc_fname = getenv("KITUTUKI_USERRC");
    if(home) {
        char path[PATH_MAX];
        sprintf(path, "%s/.kitutuki.kio", home);
        setenv("KITUTUKI_USERRC", path, 1);
    }
    
    gPrograms = VECTOR_NEW(100);

    gRegexs = HASH_NEW(30);
    gFBuffers = HASH_NEW(30);
    
    gOtherCProgs = VECTOR_NEW(10);

    gStackFrame = VECTOR_NEW(5);
    vector_add(gStackFrame, HASH_NEW(10));

    gInnerCommands = HASH_NEW(10);
    gKitutukiExit = -1;

    gMsgId = msgget((key_t)1234, 0666 | IPC_CREAT);
    if(gMsgId < 0) {
        fprintf(stderr, "msgget failed\n");
        exit(EXIT_FAILURE);
    }

//    kitutuki_rehash();

    switch(runtime_script) {
        case kRSNoRead:
            break;

        case kRSSource:
            read_rc_file();
            break;

        case kRSObject:
            read_rc_obj_file();
            break;
    }
    mreset_tty();
    gDirStack = VECTOR_NEW(10);
}

void kitutuki_final()
{
    int i;
    hash_it* it = hash_loop_begin(gFuncs);
    while(it != NULL) {
        sFunction_delete(hash_loop_item(it));
        it = hash_loop_next(it);
    }
    hash_delete(gFuncs);

    for(i=0; i<vector_size(gStackFrame); i++) {
        hash_obj* hash = vector_item(gStackFrame, i);

        it = hash_loop_begin(hash);
        while(it != NULL) {
            string_delete(hash_loop_item(it));
            it = hash_loop_next(it);
        }
        hash_delete(hash);
    }
    vector_delete(gStackFrame);

    it = hash_loop_begin(gGlobals);
    while(it != NULL) {
        string_delete(hash_loop_item(it));
        it = hash_loop_next(it);
    }
    hash_delete(gGlobals);

    it = hash_loop_begin(gArrays);
    while(it != NULL) {
        vector_obj* v = hash_loop_item(it);

        int i;
        for(i=0; i<vector_size(v); i++) {
            string_delete(vector_item(v, i));
        }
        vector_delete(v);

        it = hash_loop_next(it);
    }
    hash_delete(gArrays);

    for(i=0; i<vector_size(gJobs); i++) {
        sJob_delete(vector_item(gJobs, i));
    }
    vector_delete(gJobs);

    list_delete(gSigChldRecieved);

    it = hash_loop_begin(gFileHandles);
    while(it) {
        int fd = (int)hash_loop_item(it);
        if(fd != 0 && fd != 1 && fd != 2) (void)close(fd);
        it = hash_loop_next(it);
    }
    hash_delete(gFileHandles);
    
    for(i=0; i<vector_size(gPrograms); i++) {
        string_delete(vector_item(gPrograms, i));
    }
    vector_delete(gPrograms);

    it = hash_loop_begin(gRegexs);
    while(it != NULL) {
        onig_free(hash_loop_item(it));
        it = hash_loop_next(it);
    }
    hash_delete(gRegexs);
    
    it = hash_loop_begin(gFBuffers);
    while(it != NULL) {
        string_delete(hash_loop_item(it));
        it = hash_loop_next(it);
    }
    hash_delete(gFBuffers);
    
    for(i=0; i<vector_size(gOtherCProgs); i++) {
        sOtherCProg_delete(vector_item(gOtherCProgs, i));
    }
    vector_delete(gOtherCProgs);
    
    if(gPipeFDs[0] != -1) close(gPipeFDs[0]);
    if(gPipeFDs[1] != -1) close(gPipeFDs[1]);

    if(msgctl(gMsgId, IPC_RMID, 0) == -1) {
        fprintf(stderr, "msgctl(IPC_RMID) failed\n");
    }

    it = hash_loop_begin(gInnerCommands);
    while(it != NULL) {
        sInnerCommand_delete(hash_loop_item(it));
        it = hash_loop_next(it);
    }
    hash_delete(gInnerCommands);

    for(i=0; i<vector_size(gDirStack); i++) {
        string_delete(vector_item(gDirStack, i));
    }
    vector_delete(gDirStack);
}

//////////////////////////////////////////////////////////////////////
// ブレース展開
//////////////////////////////////////////////////////////////////////
static BOOL read_get_block(char** p, string_obj* contents, string_obj* not_evaled);

// 読み込みだけ。この後パースする
static BOOL read_skip_next_double_brace_end(char** p, string_obj* not_evaled)
{
    int nest = 0;
    BOOL squote = FALSE;
    BOOL dquote = FALSE;
    while(1) {
        /// クォート ///
        if(**p == '\\') {
            string_push_back2(not_evaled, **p);
            (*p)++;
            string_push_back2(not_evaled, **p);
            (*p)++;
        }
        /// シングルクォート ///
        else if(!dquote && **p == '\'') {
            squote = !squote;
            string_push_back2(not_evaled, **p);
            (*p)++;
        }
        /// ダブルクォート ///
        else if(!squote && **p == '"') {
            dquote = !dquote;
            string_push_back2(not_evaled, **p);
            (*p)++;
        }
        /// シングルクォート、ダブルクォート中 ///
        else if(squote || dquote) {
            if(**p == 0) {
                err_msg("require \" or \'");
                return FALSE;
            }
            else {
                string_push_back2(not_evaled, **p);
                (*p)++;
            }
        }
        else if(**p == '{' && *(*p+1) == '{') {
            nest++;
            string_push_back2(not_evaled, **p);
            (*p)++;
            string_push_back2(not_evaled, **p);
            (*p)++;
        }
        else if(**p == '}' && *(*p+1) == '}') {
            if(nest == 0) {
                (*p)++;
                (*p)++;
                break;
            }
            else {
                string_push_back2(not_evaled, **p);
                (*p)++;
                string_push_back2(not_evaled, **p);
                (*p)++;
            }
            nest--;
        }
        else if(**p == 0) {
            err_msg("parser: unexpected end(0). need }}");
            return FALSE;
        }
        else {
            string_push_back2(not_evaled, **p);
            (*p)++;
        }
    }
    
    return TRUE;
}

static BOOL read_expand_brace_expansion(char* p, string_obj* cmdline2, char* separator);

static BOOL expand_brace_expasion_expand_brace_expasion(
    char* head, char* tail, char* block, string_obj* out, char* separator)
{
//printf("head %s tail %s block %s separator %s\n", head, tail, block, separator);

    string_obj* block2 = STRING_NEW("");
    if(strstr(block, "{{")) {
        if(!read_expand_brace_expansion(block, block2, ",")) {
            string_delete(block2);
            return FALSE;
        }
    }
    else {
        string_put(block2, block);
    }

    char* p = string_c_str(block2);
    string_obj* buf = STRING_NEW("");
    int first;
    BOOL flg_first = FALSE;
    BOOL flg_first_alpha = FALSE;
    char first_alpha;
    int zero_padding;

    string_obj* kurikaesi_moji = NULL;

    while(*p) {
        if(*p == '\\') {
            p++;
            string_push_back2(buf, *p++);
        }
        else if(*p == '.' && *(p+1)=='.') {
            if(kurikaesi_moji || flg_first || flg_first_alpha) {
                err_msg("invalid brace expasion");
                string_delete(block2);
                string_delete(buf);
                if(kurikaesi_moji) string_delete(kurikaesi_moji);
                return FALSE;
            }

            p+=2;

            if(string_length(buf) == 1 
                && string_c_str(buf)[0]>='a'
                && string_c_str(buf)[0]<='z') 
            {
                flg_first_alpha = TRUE;
                first_alpha = string_c_str(buf)[0];
            }
            else {
                flg_first = TRUE;

                first = atoi(string_c_str(buf));
                if(first >= 0) {
                    if(string_c_str(buf)[0] == '0') {
                        zero_padding = strlen(string_c_str(buf));
                    }
                    else {
                        zero_padding = 0;
                    }
                }
                else {
                    zero_padding = 0;
                }
            }

            string_put(buf, "");
        }
        else if(*p == '*') {
            if(kurikaesi_moji || flg_first || flg_first_alpha) {
                err_msg("invalid brace expasion");
                string_delete(block2);
                string_delete(buf);
                if(kurikaesi_moji) string_delete(kurikaesi_moji);
                return FALSE;
            }

            p++;

            kurikaesi_moji = STRING_NEW(string_c_str(buf));

            string_put(buf, "");
        }
        else if(*p == ',') {
            if(kurikaesi_moji || flg_first || flg_first_alpha) {
                err_msg("invalid brace expasion");
                string_delete(block2);
                string_delete(buf);
                if(kurikaesi_moji) string_delete(kurikaesi_moji);
                return FALSE;
            }

            p++;

            string_obj* str = STRING_NEW("");

            string_push_back(str, head);
            string_push_back(str, string_c_str(buf));
            string_push_back(str, tail);
            string_push_back(str, separator);

            if(strstr(string_c_str(str), "{{")) {
                string_obj* out2 = STRING_NEW("");
                if(!read_expand_brace_expansion(string_c_str(str), out2, " ")) {
                    string_delete(block2);
                    string_delete(buf);
                    string_delete(str);
                    string_delete(out2);
                    return FALSE;
                }
                string_push_back(out, string_c_str(out2));
                string_delete(out2);
            }
            else {
                string_push_back(out, string_c_str(str));
            }

            string_delete(str);

            string_put(buf,"");
        }
        else {
            string_push_back2(buf, *p++);
        }
    }

    if(flg_first) {
        if(strcmp(string_c_str(buf), "") == 0) {
            err_msg("invalid brace expasion");
            string_delete(block2);
            string_delete(buf);
            return FALSE;
        }

        int last = atoi(string_c_str(buf));

        int i;
        for(i=first; i<=last; i++) {
            string_obj* str = STRING_NEW("");

            char format[32];
            if(zero_padding > 0)
                sprintf(format, "%%0%dd", zero_padding);
            else
                sprintf(format, "%%d");

            char num[32];
            sprintf(num, format, i);

            string_push_back(str, head);
            string_push_back(str, num);
            string_push_back(str, tail);
            if(i!=last) string_push_back(str, separator);

            if(strstr(string_c_str(str), "{{")) {
                string_obj* out2 = STRING_NEW("");
                if(!read_expand_brace_expansion(string_c_str(str)
                        , out2, " ")) 
                {
                    string_delete(block2);
                    string_delete(buf);
                    string_delete(str);
                    string_delete(out2);
                    return FALSE;
                }
                string_push_back(out, string_c_str(out2));
                string_delete(out2);
            }
            else {
                string_push_back(out, string_c_str(str));
            }

            string_delete(str);
        }
    }
    else if(flg_first_alpha) {
        if(strcmp(string_c_str(buf), "") == 0 
            || string_length(buf) != 1) 
        {
            err_msg("invalid brace expasion");
            string_delete(block2);
            string_delete(buf);
            return FALSE;
        }

        char last_alpha = string_c_str(buf)[0];

        char c;
        for(c=first_alpha; c<=last_alpha; c++) {
            string_obj* str = STRING_NEW("");

            char alpha[32];
            sprintf(alpha, "%c", c);

            string_push_back(str, head);
            string_push_back(str, alpha);
            string_push_back(str, tail);
            if(c!=last_alpha) string_push_back(str, separator);

            if(strstr(string_c_str(str), "{{")) {
                string_obj* out2 = STRING_NEW("");
                if(!read_expand_brace_expansion(string_c_str(str)
                        , out2, " ")) 
                {
                    string_delete(block2);
                    string_delete(buf);
                    string_delete(str);
                    string_delete(out2);
                    return FALSE;
                }
                string_push_back(out, string_c_str(out2));
                string_delete(out2);
            }
            else {
                string_push_back(out, string_c_str(str));
            }

            string_delete(str);
        }
    }
    else if(kurikaesi_moji) {
        if(strcmp(string_c_str(buf), "") == 0) {
            err_msg("invalid brace expasion");
            string_delete(block2);
            string_delete(buf);
            string_delete(kurikaesi_moji);
            return FALSE;
        }

        string_obj* str = STRING_NEW("");

        int n = atoi(string_c_str(buf));
        int j;
        string_push_back(str, head);
        for(j=0; j<n; j++) {
            string_push_back(str, string_c_str(kurikaesi_moji));
        }
        string_push_back(str, tail);

        if(strstr(string_c_str(str), "{{")) {
            string_obj* out2 = STRING_NEW("");
            if(!read_expand_brace_expansion
                (string_c_str(str), out2, " ")) 
            {
                string_delete(block2);
                string_delete(buf);
                string_delete(str);
                return FALSE;
            }
            string_push_back(out, string_c_str(out2));
            string_delete(out2);
        }
        else {
            string_push_back(out, string_c_str(str));
        }

        string_delete(str);
        string_delete(kurikaesi_moji);
    }
    else if(strcmp(string_c_str(buf), "") != 0) {
        string_obj* str = STRING_NEW("");

        string_push_back(str, head);
        string_push_back(str, string_c_str(buf));
        string_push_back(str, tail);

        if(strstr(string_c_str(str), "{{")) {
            string_obj* out2 = STRING_NEW("");
            if(!read_expand_brace_expansion
                (string_c_str(str), out2, " ")) 
            {
                string_delete(block2);
                string_delete(buf);
                string_delete(str);
                return FALSE;
            }
            string_push_back(out, string_c_str(out2));
            string_delete(out2);
        }
        else {
            string_push_back(out, string_c_str(str));
        }
        string_delete(str);
    }

    string_delete(block2);
    string_delete(buf);

    return TRUE;
}

// コメントも除去する
// 読み込みだけ。この後パースする
static BOOL read_expand_brace_expansion(char* p, string_obj* cmdline2, char* separator)
{
    BOOL squote = FALSE;
    BOOL dquote = FALSE;

    while(*p == ' ' || *p == '\t') {
        string_push_back2(cmdline2, *p++);
    }

    string_obj* head2 = STRING_NEW("");
    int head_count = 0;

    while(*p) {
        if(*p == '\\') {
            string_push_back2(head2, *p);
            string_push_back2(cmdline2, *p++);
            string_push_back2(head2, *p);
            string_push_back2(cmdline2, *p++);
            head_count+=2;
        }
        else if(!dquote && *p == '\'') {
            string_push_back2(head2, *p);
            string_push_back2(cmdline2, *p++);
            squote = !squote;
            head_count++;
        }
        else if(!squote && *p == '"') {
            string_push_back2(head2, *p);
            string_push_back2(cmdline2, *p++);
            head_count++;
            dquote = !dquote;
        }
        else if(squote || dquote) {
            if(*p == 0) {
                err_msg("require \" or \'");
                return FALSE;
            }
            else {
                string_push_back2(head2, *p);
                string_push_back2(cmdline2, *p++);
                head_count++;
            }
        }
        /// コメント ///
        else if(*p == '#') {
            p++;

            while(*p) {
                if(is_line_field(p)) {
                    if(gScriptLineField == kCRLF) {
                        string_push_back2(cmdline2, *p++);
                        string_push_back2(cmdline2, *p++);
                    }
                    else {
                        string_push_back2(cmdline2, *p++);
                    }

                    break;
                }
                else {
                    p++; 
                }
            }
            head_count = 0;
        }
        else if(*p == '{' && *(p+1) == '{') {
            p+=2;

            string_trunc(cmdline2
                    , string_length(cmdline2)-head_count);

            string_obj* block = STRING_NEW("");
            if(!read_skip_next_double_brace_end(&p, block)) {
                string_delete(head2);
                string_delete(block);
                return FALSE;
            }

            string_obj* tail = STRING_NEW("");
            if(strcmp(separator, ",") == 0) {
                BOOL squote = FALSE;
                BOOL dquote = FALSE;
                while(*p) {
                    if(*p == '\\') {
                        string_push_back2(tail, *p++);
                        string_push_back2(tail, *p++);
                    }
                    else if(!dquote && *p == '\'') {
                        string_push_back2(tail, *p++);
                        squote = !squote;
                    }
                    else if(!squote && *p == '"') {
                        string_push_back2(tail, *p++);
                        dquote = !dquote;
                    }
                    else if(squote || dquote) {
                        if(*p == 0) {
                            err_msg("require \" or \'");

                            string_delete(head2);
                            string_delete(tail);
                            string_delete(block);

                            return FALSE;
                        }
                        else {
                            string_push_back2(tail, *p++);
                        }
                    }
                    else if(*p == '{' && *(p+1) =='{') {
                        string_push_back2(tail, *p++);
                        string_push_back2(tail, *p++);

                        string_obj* block2 = STRING_NEW("");
                        if(!read_skip_next_double_brace_end(&p
                                , block2)) 
                        {
                            string_delete(head2);
                            string_delete(tail);
                            string_delete(block);
                            string_delete(block2);

                            return FALSE;
                        }

                        string_push_back(tail, string_c_str(block2));
                        string_push_back(tail, "}}");

                        string_delete(block2);
                    }
                    else if(*p == ' ' || *p == '\t' || is_line_field(p) 
                            || *p == ',') 
                    {
                        break;
                    }
                    else {
                        string_push_back2(tail, *p++);
                    }
                }
            }
            else {
                BOOL squote = FALSE;
                BOOL dquote = FALSE;
                while(*p) {
                    if(squote && *p == '\\' && *(p+1) == '\'') {
                        string_push_back2(tail, *p++);
                        string_push_back2(tail, *p++);
                    }
                    else if(dquote && *p == '\\' && *(p+1) == '"') {
                        string_push_back2(tail, *p++);
                        string_push_back2(tail, *p++);
                    }
                    else if(*p == '\\') {
                        string_push_back2(tail, *p++);
                        string_push_back2(tail, *p++);
                    }
                    else if(!dquote && *p == '\'') {
                        string_push_back2(tail, *p++);
                        squote = !squote;
                    }
                    else if(!squote && *p == '"') {
                        string_push_back2(tail, *p++);
                        dquote = !dquote;
                    }
                    else if(squote || dquote) {
                        if(*p == 0) {
                            err_msg("require \" or \'");

                            string_delete(head2);
                            string_delete(tail);
                            string_delete(block);

                            return FALSE;
                        }
                        else {
                            string_push_back2(tail, *p++);
                        }
                    }
                    else if(*p == '{' && *(p+1) =='{') {
                        string_push_back2(tail, *p++);
                        string_push_back2(tail, *p++);

                        string_obj* block2 = STRING_NEW("");
                        if(!read_skip_next_double_brace_end(&p
                                , block2)) 
                        {
                            string_delete(head2);
                            string_delete(tail);
                            string_delete(block);
                            string_delete(block2);

                            return FALSE;
                        }

                        string_push_back(tail, string_c_str(block2));
                        string_push_back(tail, "}}");

                        string_delete(block2);
                    }
                    else if(*p == ' ' || *p == '\t' || is_line_field(p)) 
                    {
                        break;
                    }
                    else {
                        string_push_back2(tail, *p++);
                    }
                }
            }

            string_obj* out = STRING_NEW("");
            if(!expand_brace_expasion_expand_brace_expasion(
                    string_c_str(head2), string_c_str(tail)
                    , string_c_str(block), out, separator))
            {
                string_delete(out);
                string_delete(tail);
                string_delete(block);
                string_delete(head2);

                return FALSE;
            }

            string_push_back(cmdline2, string_c_str(out));

            string_delete(out);
            string_delete(tail);
            string_delete(block);

            string_put(head2, "");
            head_count = 0;
        }
        else if(*p == ',' && strcmp(separator, ",") == 0) {
            string_push_back2(cmdline2, *p++);

            string_put(head2, "");
            while(*p == ' ' || *p == '\t' || is_line_field(p)) {
                if(gScriptLineField == kCRLF && is_line_field(p)) {
                    string_push_back2(cmdline2, *p++);
                    string_push_back2(cmdline2, *p++);
                }
                else {
                    string_push_back2(cmdline2, *p++);
                }
            }
            head_count = 0;
        }
        else if(*p == ' ' || *p == '\t' || is_line_field(p)) {
            string_put(head2, "");
            while(*p == ' ' || *p == '\t' || is_line_field(p)) {
                if(gScriptLineField == kCRLF && is_line_field(p)) {
                    string_push_back2(cmdline2, *p++);
                    string_push_back2(cmdline2, *p++);
                }
                else {
                    string_push_back2(cmdline2, *p++);
                }
            }
            head_count = 0;
        }
        else {
            head_count++;
            string_push_back2(head2, *p);
            string_push_back2(cmdline2, *p++);
        }
    }

    string_delete(head2);

    return TRUE;
}

//////////////////////////////////////////////////////////////////////
// マクロ展開
//////////////////////////////////////////////////////////////////////
BOOL (*expand_macro)(char* p, string_obj* cmdline) = NULL;

//////////////////////////////////////////////////////////////////////////////
// 環境変数展開
//////////////////////////////////////////////////////////////////////////////
static BOOL expand_env(char* p, string_obj* cmdline2);
static BOOL read_skip_next_paren_end(char** p, string_obj* not_evaled);
static BOOL read_skip_next_bracket_end(char** p, string_obj* not_evaled);
static BOOL read_skip_next_cbrace_end(char** p, string_obj* not_evaled);

static BOOL expand_env_parse_number(char** p, char* number1, char* number2)
{
    if(**p == '[') {
        (*p)++;

        string_obj* buf = STRING_NEW("");
        while(1) {
            if(**p == 0) {
                err_msg("parse expand env number: require ]. close it");
                return FALSE;
            }
            else if(**p == ']') {
                (*p)++;
                break;
            }
            else {
                string_push_back2(buf, **p);
                (*p)++;
            }
        }

        string_obj* str = STRING_NEW("");
        if(!expand_env(string_c_str(buf), str)) {
            string_delete(str);
            string_delete(buf);
            return FALSE;
        }
        string_delete(buf);

        char* p5;
        string_obj* str2 = STRING_NEW("");
        if(expand_macro) {
            if(!expand_macro(string_c_str(str), str2)) {
                string_delete(str);
                string_delete(str2);
                return FALSE;
            }
            p5 = string_c_str(str2);
        }
        else {
            p5 = string_c_str(str);
        }

        char* p4 = number1;
        while(*p5 >= '0' && *p5 <='9' || *p5 == '-') {
            *p4++ = *p5++;
        }
        *p4 = 0;

        if(*p5 == '.' && *(p5+1) == '.') {
            p5+=2;

            char* p4 = number2;
            while(*p5 >= '0' && *p5 <='9' || *p5 == '-') {
                *p4++ = *p5++;
            }
            *p4 = 0;
        }

        string_delete(str);
        string_delete(str2);
    }

    return TRUE;
}

static BOOL expand_env_expand_env_quoted(string_obj* cmdline2, char* env, char* number1, char* number2, BOOL dquote)
{
    if(strcmp(number1, "") == 0 && strcmp(number2, "") == 0) {
        string_obj* env2 = STRING_NEW("");
        kitutuki_get_quoted_fname(env, env2);
        if(dquote) {
            string_obj* env3 = STRING_NEW("");
            kitutuki_get_quoted_fname(string_c_str(env2), env3);
            string_push_back(cmdline2, string_c_str(env3));
            string_delete(env2);
            string_delete(env3);
        }
        else {
            string_push_back(cmdline2, string_c_str(env2));
            string_delete(env2);
        }
    }
    else if(strcmp(number1, "") != 0 && strcmp(number2, "") == 0) 
    {
        int n = atoi(number1);

        if(n < 0) {
            n = str_kanjilen(gKanjiCode, env) + n;
        }

        if(n< 0) { n = 0; }
        if(n>=str_kanjilen(gKanjiCode, env)) {
            n = str_kanjilen(gKanjiCode, env) -1;
        }

        char str[MB_CUR_MAX+1];
        char* p1 = str_kanjipos2pointer(gKanjiCode, env, n);
        char* p2 = str_kanjipos2pointer(gKanjiCode, env, n + 1);

        memcpy(str, p1, p2-p1);
        str[p2-p1] = 0;

        string_obj* str2 = STRING_NEW("");
        kitutuki_get_quoted_fname(str, str2);
        if(dquote) {
            string_obj* str3 = STRING_NEW("");
            kitutuki_get_quoted_fname(string_c_str(str2), str3);
            string_push_back(cmdline2, string_c_str(str3));
            string_delete(str2);
            string_delete(str3);
        }
        else {
            string_push_back(cmdline2, string_c_str(str2));
            string_delete(str2);
        }
    }
    else {
        int n = atoi(number1);
        int n2 = atoi(number2);

        if(n < 0) {
            n = str_kanjilen(gKanjiCode, env) + n;
        }

        if(n2 < 0) {
            n2 = str_kanjilen(gKanjiCode, env) + n2;
        }

        if(n< 0) { n = 0; }
        if(n>=str_kanjilen(gKanjiCode, env)) {
            n = str_kanjilen(gKanjiCode, env) -1;
        }

        if(n2< 0) { n2 = 0; }
        if(n2>=str_kanjilen(gKanjiCode, env)) {
            n2 = str_kanjilen(gKanjiCode, env) -1;
        }

        if(n <= n2)
        {
            char* p1 = str_kanjipos2pointer(gKanjiCode, env, n);
            char* p2 = str_kanjipos2pointer(gKanjiCode, env, n2+1);

            char* buf = MALLOC(p2-p1+1);
            memcpy(buf, p1, p2-p1);
            buf[p2-p1] = 0;
   
            string_obj* str2 = STRING_NEW("");
            kitutuki_get_quoted_fname(buf, str2);
            if(dquote) {
                string_obj* str3 = STRING_NEW("");
                kitutuki_get_quoted_fname(string_c_str(str2), str3);
                string_push_back(cmdline2, string_c_str(str3));
                string_delete(str2);
                string_delete(str3);
            }
            else {
                string_push_back(cmdline2, string_c_str(str2));
                string_delete(str2);
            }

            FREE(buf);
        }
        else {
            err_msg("expand env: invalid [] range");
            return FALSE;
        }
    }

    return TRUE;
}

static BOOL expand_env_expand_env(string_obj* cmdline2, char* env, char* number1, char* number2)
{
    if(strcmp(number1, "") == 0 && strcmp(number2, "") == 0) {
        string_obj* env3 = STRING_NEW("");
        get_quoted_linefield(env, env3);
        string_push_back(cmdline2, string_c_str(env3));
        string_delete(env3);
    }
    else if(strcmp(number1, "") != 0 && strcmp(number2, "") == 0) 
    {
        int n = atoi(number1);

        if(n < 0) {
            n = str_kanjilen(gKanjiCode, env) + n;
        }

        if(n< 0) { n = 0; }
        if(n>=str_kanjilen(gKanjiCode, env)) {
            n = str_kanjilen(gKanjiCode, env) -1;
        }

        char str[MB_CUR_MAX+1];
        char* p1 = str_kanjipos2pointer(gKanjiCode, env, n);
        char* p2 = str_kanjipos2pointer(gKanjiCode, env, n + 1);

        memcpy(str, p1, p2-p1);
        str[p2-p1] = 0;

        string_obj* env3 = STRING_NEW("");
        get_quoted_linefield(str, env3);
        string_push_back(cmdline2, string_c_str(env3));
        string_delete(env3);
    }
    else {
        int n = atoi(number1);
        int n2 = atoi(number2);

        if(n < 0) {
            n = str_kanjilen(gKanjiCode, env) + n;
        }

        if(n2 < 0) {
            n2 = str_kanjilen(gKanjiCode, env) + n2;
        }

        if(n< 0) { n = 0; }
        if(n>=str_kanjilen(gKanjiCode, env)) {
            n = str_kanjilen(gKanjiCode, env) -1;
        }
        if(n2< 0) { n2 = 0; }
        if(n2>=str_kanjilen(gKanjiCode, env)) {
            n2 = str_kanjilen(gKanjiCode, env) -1;
        }

        if(n <= n2) {
            char* p1 = str_kanjipos2pointer(gKanjiCode, env, n);
            char* p2 = str_kanjipos2pointer(gKanjiCode, env, n2+1);

            char* buf = MALLOC(p2-p1+1);
            memcpy(buf, p1, p2-p1);
            buf[p2-p1] = 0;
    
            string_obj* env3 = STRING_NEW("");
            get_quoted_linefield(buf, env3);
            string_push_back(cmdline2, string_c_str(env3));
            string_delete(env3);

            FREE(buf);
        }
        else {
            err_msg("expand_env: invalid [] range");
            return FALSE;
        }
    }

    return TRUE;
}

static BOOL expand_env_parse_paren(char** p, string_obj* delimiter
                        , BOOL* delimiter_exist) 
{
    if(**p == '(') {
        *delimiter_exist = TRUE;
        (*p)++;

        while(**p != ')') {
            string_push_back2(delimiter, **p);
            (*p)++;
        }
        (*p)++;
    }

    return TRUE;
}

/// 読み込みだけ。その後パースする
static BOOL expand_env(char* p, string_obj* cmdline2)
{
    BOOL squote = FALSE;
    BOOL dquote = FALSE;
    string_obj* buf = STRING_NEW("");

    while(*p == ' ' || *p == '\t') {
        string_push_back2(cmdline2, *p++);
    }
    while(*p) {
        if(*p == '\\') {
            string_push_back2(cmdline2, *p++);
            string_push_back2(cmdline2, *p++);
        }
        else if(!dquote && *p == '\'') {
            string_push_back2(cmdline2, *p++);
            squote = !squote;
        }
        else if(!squote && *p == '"') {
            string_push_back2(cmdline2, *p++);
            dquote = !dquote;
        }

        /// ブロックはスキップする ///
        else if(!squote && !dquote && *p == ':') {
            string_obj* dummy = STRING_NEW("");
            if(!read_get_block(&p, dummy, cmdline2)) {
                string_delete(dummy);
                string_delete(buf);
                return FALSE;
            }
            string_delete(dummy);
        }

        /// ブロックはスキップする ///
        else if(!squote && !dquote && *p == '{')
        {
            string_obj* dummy = STRING_NEW("");
            if(!read_get_block(&p, dummy, cmdline2)) {
                string_delete(dummy);
                string_delete(buf);
                return FALSE;
            }
            string_delete(dummy);
        }

        /// 算術演算
        else if(!squote && *p == '$' && *(p+1) =='(' && *(p+2) == '(')
        {
            p+=3;

            string_obj* str = STRING_NEW("");
            if(!read_skip_next_paren_end(&p, str)) {
                string_delete(str);
                string_delete(buf);
                
                return FALSE;
            }

            if(*p != ')') {
                err_msg("expand_env: need ) in $(())");
                string_delete(str);
                string_delete(buf);
                
                return FALSE;
            }

            p++;

            string_trunc(str, string_length(str)-1);
            
            string_insert(str, 0, " calc \"");
            string_push_back(str, "\"");

            string_obj* ret = STRING_NEW("");
            char fname[128];
            sprintf(fname, "%s %d: expand command", gFName, gLine);
            if(kitutuki_shell3(ret, string_c_str(str), fname) == -1) {
                string_delete(ret);
                string_delete(str);
                string_delete(buf);
                return FALSE;
            }

            string_push_back(cmdline2, string_c_str(ret));

            string_delete(ret);
            string_delete(str);
        }

        /// コマンド置換 $( ///
        else if(!squote 
            && (*p == '$' && *(p+1) == '$' && *(p+2) == '('
                        || *p == '$' && *(p+1) == '(')) 
        {
            BOOL quote;
            if(*(p+1) == '$') {
                p+=3;
                quote = FALSE;
            }
            else {
                p+=2;
                quote = TRUE;
            }

            string_obj* str = STRING_NEW("");
            if(!read_skip_next_paren_end(&p, str)) {
                string_delete(str);
                string_delete(buf);
                
                return FALSE;
            }

            string_trunc(str, string_length(str)-1); // )をとる

            string_obj* ret = STRING_NEW("");
            char fname[128];
            sprintf(fname, "%s %d: expand command", gFName, gLine);
            int rcode = kitutuki_shell3(ret, string_c_str(str), fname);

            if(rcode < 0) {
                string_delete(str);
                string_delete(ret);
                string_delete(buf);
                return FALSE;
            }

            /// [ 演算子 ///
            char number1[1024];
            strcpy(number1, "");
            char number2[1024];
            strcpy(number2, "");

            if(!expand_env_parse_number(&p, number1, number2)) {
                string_delete(str);
                string_delete(ret);
                string_delete(buf);
                return FALSE;
            }

            if(quote) {
                if(!expand_env_expand_env_quoted(cmdline2
                            , string_c_str(ret)
                            , number1, number2, dquote))
                {
                    string_delete(ret);
                    string_delete(str);
                    string_delete(buf);
                    return FALSE;
                }
            }
            else {
                if(!expand_env_expand_env(cmdline2, string_c_str(ret)
                            , number1, number2))
                {
                    string_delete(ret);
                    string_delete(str);
                    string_delete(buf);
                    return FALSE;
                }
            }

            string_delete(str);
            string_delete(ret);
        }

        /// 環境変数 $ ///
        else if(!squote && 
                (*p == '$' && *(p+1) == '$' 
                    && (*(p+2) == '{' || is_env_name_char(*(p+2)))
                    || *p == '$' && 
                        (*(p+1) == '{' || is_env_name_char(*(p+1)))
                ))
        {
            BOOL quote;
            if(*(p+1) == '$') {
                p+=2;
                quote = FALSE;
            }
            else {
                p++;
                quote = TRUE;
            }

            string_obj* name = STRING_NEW("");
            if(*p == '{') {
                p++;

                if(!read_skip_next_cbrace_end(&p, name)) {
                    string_delete(name);
                    string_delete(buf);
                    return FALSE;
                }

                string_trunc(name, string_length(name)-1);
            }
            else {
                while(is_env_name_char(*p)) {
                    string_push_back2(name, *p++);
                }
            }

            string_obj* name2 = STRING_NEW("");
            if(!expand_env(string_c_str(name), name2)) {
                string_delete(name);
                string_delete(name2);
                string_delete(buf);
                return FALSE;
            }

            string_delete(name);

            /// [ 演算子 ///
            char number1[1024];
            strcpy(number1, "");
            char number2[1024];
            strcpy(number2, "");

            if(!expand_env_parse_number(&p, number1, number2)) {
                string_delete(name2);
                string_delete(buf);
                return FALSE;
            }
            
            vector_obj* array = hash_item(gArrays, string_c_str(name2));
            if(array) {
                string_obj* delimiter = STRING_NEW("");
                BOOL delimiter_exist = FALSE;
                if(!expand_env_parse_paren(&p, delimiter
                                                , &delimiter_exist)) 
                {
                    string_delete(delimiter);
                    string_delete(name2);
                    string_delete(buf);
                    return FALSE;
                }
                
                if(!delimiter_exist 
                    && strcmp(string_c_str(delimiter), "") == 0) 
                {
                    string_put(delimiter, " ");
                }

                if(strcmp(number1, "") == 0 && strcmp(number2, "") == 0) 
                {
                    if(quote) {
                        int k;
                        for(k=0; k<vector_size(array); k++) {
                            string_obj* str = vector_item(array, k);
                            string_obj* str2 = STRING_NEW("");
                            kitutuki_get_quoted_fname(string_c_str(str), str2);
                            string_push_back(cmdline2, string_c_str(str2));
                            if(k<vector_size(array)-1)
                                string_push_back(cmdline2
                                    , string_c_str(delimiter));

                            string_delete(str2);
                        }
                    }
                    else {
                        int k;
                        for(k=0; k<vector_size(array); k++) {
                            string_obj* str = vector_item(array, k);
                            if(str == NULL) {
                                err_msg("expand_env: invalid array range");
                                string_delete(delimiter);
                                string_delete(name2);
                                string_delete(buf);
                                return FALSE;
                            }

                            string_obj* str2 = STRING_NEW("");
                            get_quoted_linefield(string_c_str(str)
                                    , str2);
                            string_push_back(cmdline2, string_c_str(str2));
                            string_delete(str2);

                            if(k<vector_size(array)-1)
                                string_push_back(cmdline2
                                        , string_c_str(delimiter));
                        }
                    }
                }
                else if(strcmp(number1, "") != 0 && strcmp(number2, "") == 0) {
                    /// [ 演算子 ///
                    char number3[1024];
                    strcpy(number3, "");
                    char number4[1024];
                    strcpy(number4, "");

                    if(!expand_env_parse_number(&p, number3, number4)) {
                        string_delete(delimiter);
                        string_delete(name2);
                        string_delete(buf);
                        return FALSE;
                    }

                    int n = atoi(number1);

                    if(n < 0) {
                        n = vector_size(array) + n;
                    }

                    if(quote) {
                        if(n >= 0 && n < vector_size(array)) {
                            string_obj* str = vector_item(array, n);
                            if(!expand_env_expand_env_quoted(cmdline2
                                        , string_c_str(str)
                                        , number3, number4, dquote))
                            {
                                string_delete(delimiter);
                                string_delete(name2);
                                string_delete(buf);
                                return FALSE;
                            }
                        }
                    }
                    else {
                        if(n >= 0 && n < vector_size(array)) {
                            string_obj* str = vector_item(array, n);
                            if(!expand_env_expand_env(cmdline2
                                , string_c_str(str), number3, number4))
                            {
                                string_delete(delimiter);
                                string_delete(name2);
                                string_delete(buf);
                                return FALSE;
                            }
                        }
                    }
                }
                else {
                    int n = atoi(number1);
                    int m = atoi(number2);

                    if(n < 0) n = vector_size(array) + n;
                    if(m < 0) m = vector_size(array) + m;

                    if(quote) {
                        if(n < 0) n = 0;
                        if(m >= vector_size(array)) m = vector_size(array)-1;

                        int k;
                        for(k=n; k<=m; k++) {
                            string_obj* str = vector_item(array, k);

                            string_obj* str2 = STRING_NEW("");
                            kitutuki_get_quoted_fname(string_c_str(str), str2);
                            string_push_back(cmdline2, string_c_str(str2));
                            if(k<=m-1) string_push_back(cmdline2
                                        , string_c_str(delimiter));

                            string_delete(str2);
                        }
                    }
                    else {
                        if(n < 0) n = 0;
                        if(m >= vector_size(array)) m = vector_size(array)-1;

                        int k;
                        for(k=n; k<=m; k++) {
                            string_obj* str = vector_item(array, k);

                            string_obj* str2 = STRING_NEW("");
                            get_quoted_linefield(string_c_str(str), str2);

                            string_push_back(cmdline2, string_c_str(str2));
                            string_delete(str2);

                            if(k<=m-1) 
                              string_push_back(cmdline2
                                            , string_c_str(delimiter));
                        }
                    }
                }
                
                string_delete(delimiter);
            }
            else {
                /// 環境変数を参照 ///
                char* env = getenv(string_c_str(name2));

                if(env) {
                    if(quote) {
                        if(!expand_env_expand_env_quoted(
                            cmdline2, env, number1, number2, dquote))
                        {
                            string_delete(name2);
                            string_delete(buf);
                            return FALSE;
                        }
                    }
                    else {
                        expand_env_expand_env(cmdline2, env
                                            , number1, number2);
                    }
                }
                else {
                    /// グローバル変数を参照 ///
                    string_obj* var = hash_item(gGlobals, string_c_str(name2));

                    /// ローカル変数を参照
                    if(var == NULL) {
                        hash_obj* top_stack
                         = vector_item(gStackFrame, vector_size(gStackFrame)-1);
                        var = hash_item(top_stack, string_c_str(name2));
                    }

                    if(var) {
                        if(quote) {
                            if(!expand_env_expand_env_quoted(cmdline2
                                        , string_c_str(var)
                                        , number1, number2, dquote))
                            {
                                string_delete(name2);
                                string_delete(buf);
                                return FALSE;
                            }
                        }
                        else {
                            expand_env_expand_env(cmdline2, string_c_str(var)
                                        , number1, number2);
                        }
                    }
                }
            }
            string_delete(name2);
        }
        else if(squote || dquote) {
            if(*p == 0) {
                err_msg("require ' or \"");
                string_delete(buf);
                return FALSE;
            }
            else {
                string_push_back2(cmdline2, *p++);
            }
        }
        /// グロブ ///
        else if(*p == '*' || *p == '?' 
                    || *p =='[' && (*(p+1) != ' ' 
                    && *(p+1) != '\t' && is_line_field(p+1))) 
        {
            string_obj* pattern = STRING_NEW("");

            if(strcmp(string_c_str(buf), "") != 0) {
                /// patternに入れるから削る
                string_trunc(cmdline2, string_length(cmdline2) - string_length(buf));
                /// patternに入れる
                string_push_back(pattern, string_c_str(buf));
                string_put(buf, "");
            }

            while(*p != 0 && !is_glob_terminated_char(p)) {
                string_push_back2(pattern, *p++);
            }

            glob_t result;
            int rc = glob(string_c_str(pattern), 0, NULL, &result);
            if(rc == GLOB_NOSPACE) {
                err_msg("expand_env: out of space during glob operation");
                string_delete(buf);
                string_delete(pattern);
                    
                return FALSE;
            }
            else if(rc == GLOB_NOMATCH) {
                /*
                char buf2[512];
                sprintf(buf2, "expand_env: no match is %s", string_c_str(pattern));
                err_msg(buf2);
                string_delete(buf);
                string_delete(pattern);
                return FALSE;
                */
                string_delete(pattern);
            }
            else {
                int i;
                for(i=0; i<result.gl_pathc; i++) {
                    char* file = result.gl_pathv[i];
                    if(strcmp(file, ".") != 0 && strcmp(file, "..") != 0)
                    {
                        string_obj* quoted_fname = STRING_NEW("");
                        kitutuki_get_quoted_fname(file, quoted_fname);
                        string_push_back(cmdline2, string_c_str(quoted_fname));
                        string_push_back(cmdline2, " ");
                        string_delete(quoted_fname);
                    }
                }
                string_delete(pattern);
            }
        }
        else if(!is_glob_terminated_char(p)) {
            string_push_back2(buf, *p);
            string_push_back2(cmdline2, *p++);
        }
        else {
            string_push_back2(cmdline2, *p++);
            string_put(buf, "");
        }
    }

    string_delete(buf);
    return TRUE;
}

//////////////////////////////////////////////////////////////////////
// パース
//////////////////////////////////////////////////////////////////////
// ファイルハンドかどうかの判定用。パーサーではない
static BOOL parse_is_file_handle(char* p) 
{
    if(*p == '<') {
        p++;

        BOOL squote = FALSE;
        BOOL dquote = FALSE;
        while(1) {
            /// クォート
            if(*p == '\\') {
                p+=2;
            }
            /// シングルクォート
            else if(!dquote && *p == '\'') {
                p++;
            }
            /// ダブルクォート
            else if(!squote && *p == '"') {
                p++;
                dquote = !dquote;
            }
            else if(squote || dquote) {
                if(*p == 0) {
                    return FALSE;
                }
                p++;
            }
            else if(*p == 0) {
                return FALSE;
            }
            else if(is_line_field(p)) {
                err_msg(
                 "parse file handle: detect line field in file handle");
                return FALSE;
            }
            else if(*p == '>') {
                break;
            }
            else {
                p++;
            }
        }

        if(*p == '>' && is_terminated_char(p+1)) {
            return TRUE;
        }
        else {
            return FALSE;
        }
    }
    else {
        return FALSE;
    }
}

// リダイレクトをパースする
static int parse_redirect(char** p, sRedirect* redirection, string_obj* not_evaled)
{
    int fd;
    enum eRedirect type;
    
    if(**p == '1' && *(*p+1) == '>' && *(*p+2) == '>') {
        string_push_back2(not_evaled, **p);
        string_push_back2(not_evaled, *(*p+1));
        string_push_back2(not_evaled, *(*p+2));
        (*p)+=3;
        
        fd = 1;
        type = kRedirectAppend;
    }
    else if(**p == '1' && *(*p+1) == '>') {
        string_push_back2(not_evaled, **p);
        string_push_back2(not_evaled, *(*p+1));
        (*p)+=2;
        
        fd = 1;
        type = kRedirectOverwrite;
    }
    else if(**p == '2' && *(*p+1) == '>' && *(*p+2) == '>') {
        string_push_back2(not_evaled, **p);
        string_push_back2(not_evaled, *(*p+1));
        string_push_back2(not_evaled, *(*p+2));
        (*p)+=3;
        
        fd = 2;
        type = kRedirectAppend;
    }
    else if(**p == '2' && *(*p+1) == '>') {
        string_push_back2(not_evaled, **p);
        string_push_back2(not_evaled, *(*p+1));
        (*p)+=2;
        
        fd = 2;
        type = kRedirectOverwrite;
    }
    else if(**p == '>' && *(*p+1) == '>') {
        string_push_back2(not_evaled, **p);
        string_push_back2(not_evaled, *(*p+1));
        (*p)+=2;
        
        fd = 1;
        type = kRedirectAppend;
    }
    else if(**p =='>') {
        string_push_back2(not_evaled, **p);
        (*p)++;
        
        fd = 1;
        type = kRedirectOverwrite;
    }
    else {
        string_push_back2(not_evaled, **p);
        (*p)++;
        
        fd = 0;
        type = kRedirectInput;
    }

    redirection->mType = type;
    redirection->mFd = fd;
}

// 中身はもう一回パースされる
static BOOL read_get_block(char** p, string_obj* contents, string_obj* not_evaled)
{
    /// カッコブロック ///
    if(**p == '{') {
        string_push_back2(not_evaled, **p);
        (*p)++;
        
        while(**p == ' ' || **p == '\t' || is_line_field(*p) && gLine++) 
        {
            if(gScriptLineField == kCRLF && is_line_field(*p)) {
                string_push_back2(not_evaled, **p);
                string_push_back2(contents, *((*p)++));
                string_push_back2(not_evaled, **p);
                string_push_back2(contents, *((*p)++));
            }
            else {
                string_push_back2(not_evaled, **p);
                string_push_back2(contents, *((*p)++));
            }
        }

        BOOL squote = FALSE;
        BOOL dquote = FALSE;
        int nest = 0;
        while(1) {
            /// クォート
            if(**p == '\\') {
                string_push_back2(not_evaled, **p);
                string_push_back2(contents, *((*p)++));
                if(is_line_field(*p)) {
                    gLine++;
                    if(gScriptLineField == kCRLF) {
                        string_push_back2(not_evaled, **p);
                        string_push_back2(contents, *((*p)++));
                        string_push_back2(not_evaled, **p);
                        string_push_back2(contents, *((*p)++));
                    }
                    else {
                        string_push_back2(not_evaled, **p);
                        string_push_back2(contents, *((*p)++));
                    }
                }
                else {
                    string_push_back2(not_evaled, **p);
                    string_push_back2(contents, *((*p)++));
                }
            }
            /// シングルクォート
            else if(!dquote && **p == '\'') {
                string_push_back2(not_evaled, **p);
                string_push_back2(contents, *((*p)++));
                squote = !squote;
            }
            /// ダブルクォート
            else if(!squote && **p == '"') {
                string_push_back2(not_evaled, **p);
                string_push_back2(contents, *((*p)++));
                dquote = !dquote;
            }
            /// シングルクォート、ダブルクォート中 ///
            else if(squote || dquote) {
                if(**p == 0) {
                    err_msg("get block: require \" or \'");
                    return FALSE;
                }
                if(is_line_field(*p)) {
                    gLine++;
                    if(gScriptLineField == kCRLF) {
                        string_push_back2(not_evaled, **p);
                        string_push_back2(contents, *((*p)++));
                        string_push_back2(not_evaled, **p);
                        string_push_back2(contents, *((*p)++));
                    }
                    else {
                        string_push_back2(not_evaled, **p);
                        string_push_back2(contents, *((*p)++));
                    }
                }
                else {
                    string_push_back2(not_evaled, **p);
                    string_push_back2(contents, *((*p)++));
                }
            }
            else if(**p == '{') {
                string_push_back2(not_evaled, **p);
                string_push_back2(contents, *((*p)++));
                nest++;
            }
            else if(**p == '}') {
                string_push_back2(not_evaled, **p);
                (*p)++;
                nest--;

                if(nest == -1) {
                    break;
                }
                else {
                    string_push_back2(contents, '}');
                }
            }
            else if(**p == 0) {
                err_msg("get block: found 0. require }");
                return FALSE;
            }
            else if(is_line_field(*p)) {
                gLine++;
                if(gScriptLineField == kCRLF) {
                    string_push_back2(not_evaled, **p);
                    string_push_back2(contents, **p);
                    (*p)++;
                    string_push_back2(not_evaled, **p);
                    string_push_back2(contents, **p);
                    (*p)++;
                }
                else {
                    string_push_back2(not_evaled, **p);
                    string_push_back2(contents, **p);
                    (*p)++;
                }
            }
            else {
                string_push_back2(not_evaled, **p);
                string_push_back2(contents, **p);
                (*p)++;
            }
        }
    }
    /// インデントブロック ///
    else if(**p == ':') {
        string_push_back2(not_evaled, **p);
        (*p)++;
        
        while(**p == ' ' || **p == '\t') {
            string_push_back2(not_evaled, **p);
            (*p)++;
        }

        /// ブロック ///
        if(!is_line_field(*p)) {
            err_msg("get block: invalid block. require line field");
            return FALSE;
        }
        else {
            gLine++;
            if(gScriptLineField == kCRLF) {
                string_push_back2(not_evaled, **p);
                string_push_back2(contents, **p);
                (*p)++;
                string_push_back2(not_evaled, **p);
                string_push_back2(contents, **p);
                (*p)++;
            }
            else {
                string_push_back2(not_evaled, **p);
                string_push_back2(contents, **p);
                (*p)++;
            }
        }

        int indent_before = -1;
        while(1) {
            int indent = 0;
            while((**p == ' ' || **p == '\t') 
                && (indent_before == -1
                    || (indent_before != -1 && indent < indent_before)))
            {
                string_push_back2(not_evaled, **p);
                (*p)++;
                indent++;
            }

            if(**p == ';' && *(*p+1) == ';') break;

            if(!is_line_field(*p) && indent_before != -1 && indent < indent_before) 
            {
                break;
            }

            if(!is_line_field(*p) && indent_before == -1) indent_before = indent;
            
            while(!is_line_field(*p) && **p != 0) {
                string_push_back2(not_evaled, **p);
                string_push_back2(contents, *((*p)++));
            }

            if(**p == 0) {
                break;
            }

            if(is_line_field(*p)) {
                gLine++;
                if(gScriptLineField == kCRLF) {
                    string_push_back2(not_evaled, **p);
                    string_push_back2(contents, *((*p)++));
                    string_push_back2(not_evaled, **p);
                    string_push_back2(contents, *((*p)++));
                }
                else {
                    string_push_back2(not_evaled, **p);
                    string_push_back2(contents, *((*p)++));
                }
            }
            else {
                string_push_back2(not_evaled, **p);
                string_push_back2(contents, *((*p)++));
            }
        }
    }
    
    else {
        err_msg("get block: invalid block. need : or { at block head");
        return FALSE;
    }

    return TRUE;
}

// 中身はもう一回パースされる
static int read_get_joukensiki(char** p, string_obj* joukensiki, string_obj* not_evaled)
{
    BOOL squote = FALSE;
    BOOL dquote = FALSE;

    while(1) {
        if(**p == 0) {
            err_msg("parse condition: require { or :");
            return FALSE;
        }
        /// クォート
        else if(**p == '\\') {
            string_push_back2(not_evaled, **p);
            string_push_back2(joukensiki, *((*p)++));
            string_push_back2(not_evaled, **p);
            string_push_back2(joukensiki, *((*p)++));
        }
        // シングルクォート
        else if(!dquote && **p == '\'') {
            string_push_back2(not_evaled, **p);
            string_push_back2(joukensiki, *((*p)++));
            squote = !squote;
        }
        // ダブルクォート
        else if(!squote && **p == '"') {
            string_push_back2(not_evaled, **p);
            string_push_back2(joukensiki, *((*p)++));
            dquote = !dquote;
        }
        // シングルクォート、ダブルクォート中
        else if(squote || dquote) {
            if(**p == 0) {
                err_msg("get block: require \" or \'");
                return FALSE;
            }
            else {
                string_push_back2(not_evaled, **p);
                string_push_back2(joukensiki, *((*p)++));
            }
        }
        else if(is_line_field(*p)) {
            err_msg( "parse condition: require : or { before line field and block");

            return FALSE;
        }
        else if(**p == ':' || *(*p-1) != '$' && **p=='{') {
            break;
        }
        else {
            string_push_back2(not_evaled, **p);
            string_push_back2(joukensiki, *((*p)++));
        }
    }

    return TRUE;
}

static void parse_add_command(string_obj** buf, sCommand* command
                            , sStatment* statment, string_obj* not_evaled)
{
    /// コマンドに追加 ///
    if(strcmp(string_c_str(*buf), "") != 0) {
        vector_add(command->mArgs, *buf);
        string_push_back(statment->mTitle, string_c_str(*buf));
        string_push_back(statment->mTitle, " ");
        *buf = STRING_NEW("");
    }
}

static void parse_add_statment(sCommand** command, sStatment* statment)
{
    /// 文に追加 ///
    if(vector_size((*command)->mArgs) > 0) {
        char* arg0 = string_c_str(vector_item((*command)->mArgs, 0));
        (*command)->mKind = get_command_kind(arg0);
        vector_add(statment->mCommands, *command);
        *command = COMMAND_NEW();
    }
}

static void parse_add_statments(enum eStatmentTerminated terminated
                                , sStatment** statment
                                , sStatments* statments
                                ,int line, char* fname)
{
    /// 複文に追加 ///
    if(vector_size((*statment)->mCommands) != 0) {
        (*statment)->mTerminated = terminated;
        string_put((*statment)->mFName, fname);
        (*statment)->mLine = line;

        sCommand* last_command 
            = vector_item((*statment)->mCommands
                    , vector_size((*statment)->mCommands)-1);

        vector_add(statments->mStatments, *statment);


        *statment = STATMENT_NEW();
    }
}

static void parse_add_statments2(enum eStatmentTerminated terminated
            , sStatment** statment
            , sStatments* statments, char* fname, int line
            , string_obj* not_evaled)
{
    /// 複文に追加 ///
    (*statment)->mTerminated = terminated;
    string_put((*statment)->mFName, fname);
    (*statment)->mLine = line;
    (*statment)->mNotEvaled = not_evaled;

    vector_add(statments->mStatments, *statment);

    *statment = STATMENT_NEW();
}

static void parse_skip_space(char** p)
{
    while(**p == ' ' || **p == '\t') (*p)++;
}

/// 読み込みだけ。中身はもう一度パースされる
static BOOL read_skip_next_paren_end(char** p, string_obj* not_evaled)
{
    int nest = 0;
    BOOL squote = FALSE;
    BOOL dquote = FALSE;
    while(1) {
        /// クォート ///
        if(**p == '\\') {
            string_push_back2(not_evaled, **p);
            (*p)++;
            string_push_back2(not_evaled, **p);
            (*p)++;
        }
        /// シングルクォート ///
        else if(!dquote && **p == '\'') {
            squote = !squote;
            string_push_back2(not_evaled, **p);
            (*p)++;
        }
        /// ダブルクォート ///
        else if(!squote && **p == '"') {
            dquote = !dquote;
            string_push_back2(not_evaled, **p);
            (*p)++;
        }
        /// シングルクォート、ダブルクォート中 ///
        else if(squote || dquote) {
            if(**p == 0) {
                err_msg("require ' or \"");
                return FALSE;
            }
            else {
                string_push_back2(not_evaled, **p);
                (*p)++;
            }
        }
        else if(**p == '(') {
            nest++;
            string_push_back2(not_evaled, **p);
            (*p)++;
        }
        else if(**p == ')') {
            string_push_back2(not_evaled, **p);
            (*p)++;

            if(nest == 0) {
                break;
            }
            nest--;
        }
        else if(**p == 0) {
            err_msg("parse skip next paren: unexpected end(0). need )");
            return FALSE;
        }
        else {
            string_push_back2(not_evaled, **p);
            (*p)++;
        }
    }
    
    return TRUE;
}

// 中身はもう一度parseされる
static BOOL read_skip_next_bracket_end(char** p, string_obj* not_evaled)
{
    int nest = 0;
    BOOL squote = FALSE;
    BOOL dquote = FALSE;
    while(1) {
        /// クォート ///
        if(**p == '\\') {
            string_push_back2(not_evaled, **p);
            (*p)++;
            string_push_back2(not_evaled, **p);
            (*p)++;
        }
        /// シングルクォート ///
        else if(!dquote && **p == '\'') {
            squote = !squote;
            string_push_back2(not_evaled, **p);
            (*p)++;
        }
        /// ダブルクォート ///
        else if(!squote && **p == '"') {
            dquote = !dquote;
            string_push_back2(not_evaled, **p);
            (*p)++;
        }
        else if(squote || dquote) {
            if(**p == 0) {
                err_msg("require ' or \"");
                return FALSE;
            }
            else {
                string_push_back2(not_evaled, **p);
                (*p)++;
            }
        }
        else if(**p == '[') {
            nest++;
            string_push_back2(not_evaled, **p);
            (*p)++;
        }
        else if(**p == ']') {
            string_push_back2(not_evaled, **p);
            (*p)++;

            if(nest == 0) {
                break;
            }
            nest--;
        }
        else if(**p == 0) {
            err_msg("parse skip next bracket: unexpected end(0). need ]");
            return FALSE;
        }
        else {
            string_push_back2(not_evaled, **p);
            (*p)++;
        }
    }
    
    return TRUE;
}

// 中身はパースされない。読み込みだけ
static int read_get_subshell(char** p, sStatment* statment, sCommand* command, string_obj* buf, string_obj* not_evaled)
{
    string_push_back2(not_evaled, **p);
    (*p)++;

    if(strcmp(string_c_str(buf), "") != 0 
            || vector_size(command->mArgs) != 0) 
    {
        err_msg(
            "parse get subshell: syntax err near unexpected token '('");
        string_delete(buf);
        sCommand_delete(command);
        sStatment_delete(statment);
        string_delete(not_evaled);
            
        return FALSE;
    }

    string_obj* buf2 = STRING_NEW(" ");
    int nest = 0;
    BOOL squote = FALSE;
    BOOL dquote = FALSE;
    while(1) {
        /// クォート ///
        if(**p == '\\') {
            string_push_back2(not_evaled, **p);
            string_push_back2(buf2, **p);
            (*p)++;
            string_push_back2(not_evaled, **p);
            string_push_back2(buf2, **p);
            (*p)++;
        }
        /// シングルクォート ///
        else if(!dquote && **p == '\'') {
            squote = !squote;
            string_push_back2(not_evaled, **p);
            string_push_back2(buf2, **p);
            (*p)++;
        }
        /// ダブルクォート ///
        else if(!squote && **p == '"') {
            dquote = !dquote;
            string_push_back2(not_evaled, **p);
            string_push_back2(buf2, **p);
            (*p)++;
        }
        else if(squote || dquote) {
            if(**p == 0) {
                err_msg("require ' or \"");
                string_delete(buf);
                sCommand_delete(command);
                sStatment_delete(statment);
                string_delete(not_evaled);
                string_delete(buf2);
                
                return FALSE;
            }
            else {
                string_push_back2(not_evaled, **p);
                string_push_back2(buf2, **p);
                (*p)++;
            }
        }
        else if(**p == '(') {
            nest++;
            string_push_back2(not_evaled, **p);
            string_push_back2(buf2, **p);
            (*p)++;
        }
        else if(**p == ')') {
            if(nest == 0) {
                (*p)++;
                break;
            }
            nest--;
            string_push_back2(not_evaled, **p);
            string_push_back2(buf2, **p);
            (*p)++;
        }
        else if(**p == 0) {
            err_msg("parse get subshell: unexpected end(0) in subshell. need )");
            string_delete(buf);
            sCommand_delete(command);
            sStatment_delete(statment);
            string_delete(not_evaled);
            string_delete(buf2);
            
            return FALSE;
        }
        else {
            string_push_back2(not_evaled, **p);
            string_push_back2(buf2, **p);
            (*p)++;
        }
    }

    vector_add(command->mArgs, STRING_NEW("subshell"));
    vector_add(command->mArgs, buf2);

    string_push_back(not_evaled, string_c_str(buf2));
    string_push_back2(not_evaled, ')');
    
    return TRUE;
}

// 中身はもう一度parseされる
static BOOL read_skip_next_cbrace_end(char** p, string_obj* not_evaled)
{
    int nest = 0;
    BOOL squote = FALSE;
    BOOL dquote = FALSE;
    while(1) {
        /// クォート ///
        if(**p == '\\') {
            string_push_back2(not_evaled, **p);
            (*p)++;
            string_push_back2(not_evaled, **p);
            (*p)++;
        }
        /// シングルクォート ///
        else if(!dquote && **p == '\'') {
            squote = !squote;
            string_push_back2(not_evaled, **p);
            (*p)++;
        }
        /// ダブルクォート ///
        else if(!squote && **p == '"') {
            dquote = !dquote;
            string_push_back2(not_evaled, **p);
            (*p)++;
        }
        else if(squote || dquote) {
            if(**p == 0) {
                err_msg("require ' or \"");
                return FALSE;
            }
            else {
                string_push_back2(not_evaled, **p);
                (*p)++;
            }
        }
        else if(**p == '{') {
            nest++;
            string_push_back2(not_evaled, **p);
            (*p)++;
        }
        else if(**p == '}') {
            string_push_back2(not_evaled, **p);
            (*p)++;

            if(nest == 0) {
                break;
            }
            nest--;
        }
        else if(**p == 0) {
            err_msg("parse skip next bracket: unexpected end(0). need }");
            return FALSE;
        }
        else {
            string_push_back2(not_evaled, **p);
            (*p)++;
        }
    }
    
    return TRUE;
}

/// 次のトークンがif, while文での続きかどうか ///
static int parse_check_special_statment_continue(char** p)
{
    if(**p == '&' && *(*p+1) == '&' 
            || **p == '|' && *(*p+1) == '|'
            || **p == '&' 
            || **p == '|'
            || **p == ';' 
            || is_line_field(*p)
            || **p == '#' 
            || **p == '$'
            || **p == 0
            || parse_is_redirect2(*p)
            || parse_is_redirect(*p))
    {
        return TRUE;
    }
    else {
        return FALSE;
    }
}

// 読み込みじゃなくてパースする
static int parse_get_word(char** p, string_obj* word
                , string_obj* not_evaled , int* read_end_of_statment)
{
    parse_skip_space(p);

    BOOL squote = FALSE;
    BOOL dquote = FALSE;
    while(1) {
        /// 特殊文字列 ///
        if(**p == '\\' && *(*p+1) == 't') {
            string_push_back2(not_evaled, *((*p)++));
            string_push_back2(not_evaled, *((*p)++));

            string_push_back2(word, '\t');
        }
        else if(**p == '\\' && *(*p+1) == 'n') {
            string_push_back2(not_evaled, *((*p)++));
            string_push_back2(not_evaled, *((*p)++));

            string_push_back2(word, '\n');
        }
        else if(**p == '\\' && *(*p+1) == 'r') {
            string_push_back2(not_evaled, *((*p)++));
            string_push_back2(not_evaled, *((*p)++));

            string_push_back2(word, '\r');
        }
        /// クォート ///
        else if(**p == '\\') {
            string_push_back2(not_evaled, **p);
            (*p)++;
            if(**p == 0) {
                err_msg("require ' or \". close it");
                return FALSE;
            }
            else if(is_line_field(*p)) {
                gLine++;
                if(gScriptLineField == kCRLF) {
                    string_push_back2(not_evaled, **p);
                    string_push_back2(word, **p);
                    (*p)++;
                    string_push_back2(not_evaled, **p);
                    string_push_back2(word, **p);
                    (*p)++;
                }
                else {
                    string_push_back2(not_evaled, **p);
                    string_push_back2(word, **p);
                    (*p)++;
                }
            }
            else {
                string_push_back2(not_evaled, **p);
                string_push_back2(word, **p);
                (*p)++;
            }
        }
        // シングルクォート
        else if(!dquote && **p == '\'') {
            string_push_back2(not_evaled, **p);
            (*p)++;
            squote = !squote;
        }
        // ダブルクォート
        else if(!squote && **p == '"') {
            string_push_back2(not_evaled, **p);
            (*p)++;
            dquote = !dquote;
        }
        // 環境変数 $()
        else if(!squote 
            && (**p == '$' && *(*p+1) == '$' && *(*p+2) == '('
                || **p == '$' && *(*p+1) == '(')) 
        {
            if(*(*p+1) == '$') {
                string_push_back2(not_evaled, **p);
                (*p)++;
            }
            string_push_back2(not_evaled, **p);
            (*p)++;
            string_push_back2(not_evaled, **p);
            (*p)++;

            if(!read_skip_next_paren_end(p, not_evaled)) {
                return FALSE;
            }

            if(**p == '[') {
                string_push_back2(not_evaled, **p);
                (*p)++;
                read_skip_next_bracket_end(p, not_evaled);
            };
            
            *read_end_of_statment = gLine;
        }
        // 環境変数 $ $$
        else if(!squote && 
                (**p == '$' && *(*p+1) == '$' 
                        && (*(*p+2) == '{' || is_env_name_char(*(*p+2)))
                || **p == '$' && (*(*p+1) == '{' || is_env_name_char(*(*p+1))) 
                ))

        {
            if(*(*p+1) == '$') {
                string_push_back2(not_evaled, **p);
                (*p)++;
            }
            string_push_back2(not_evaled, **p);
            (*p)++;

            if(**p == '{') {
                string_push_back2(not_evaled, **p);
                (*p)++;
            }

            char name[1024];
            char* p3 = name;
            while(is_env_name_char(**p)) {
                *p3++ = **p;
                (*p)++;
            }
            *p3 = 0;
            string_push_back(not_evaled, name);

            if(**p == '}') {
                string_push_back2(not_evaled, **p);
                (*p)++;
            }

            if(**p == '[') {
                string_push_back2(not_evaled, **p);
                (*p)++;
                read_skip_next_bracket_end(p, not_evaled);
            };
            if(**p == '(') {
                string_push_back2(not_evaled, **p);
                (*p)++;
                read_skip_next_paren_end(p, not_evaled);
            }
            
            *read_end_of_statment = gLine;
        }
        
        /// シングルクォート、ダブルクォート中 ///
        else if(squote || dquote) {
            if(**p == 0) {
                err_msg("parse word: need ' or \"\n");
                return FALSE;
            }
            else {
                if(is_line_field(*p)) {
                    gLine++;
                    if(gScriptLineField == kCRLF) {
                        string_push_back2(not_evaled, **p);
                        string_push_back2(word, **p);
                        (*p)++;
                        string_push_back2(not_evaled, **p);
                        string_push_back2(word, **p);
                        (*p)++;
                    }
                    else {
                        string_push_back2(not_evaled, **p);
                        string_push_back2(word, **p);
                        (*p)++;
                    }
                }
                else {
                    string_push_back2(not_evaled, **p);
                    string_push_back2(word, **p);
                    (*p)++;
                }
            }
        }
        else if(**p == '~') {
            string_push_back2(not_evaled, **p);
            (*p)++;
            
            char* home = getenv("HOME");
            if(home == NULL) {
                fprintf(stderr, "HOME is null\n");
                exit(1);
            }
            string_push_back(not_evaled, home);
            string_push_back(word, home);
        }
        /// グロブ ///
        else if(**p == '*' || **p == '?' 
                || **p =='[' && *(*p+1) != ' ')
        {
            err_msg("parse word: invalid brace expansion");
            return FALSE;
        }
        /// ブレース展開 ///
        else if(**p == '{' && *(*p+1) == '{') {
            err_msg("parse word: invalid brace expansion");
            return FALSE;
        }
        /// ファイルハンドル ///
        else if(strcmp(string_c_str(word), "") ==0 
            && parse_is_file_handle(*p)) 
        {
            err_msg("parse word: invalid file handle");
            return FALSE;
        }

        /// 複文(statment)の区切り系 ///
    
        /// アンドアンド ///
        else if(**p == '&' && *(*p+1) == '&') {
            break;
        }
        /// オアオア ///
        else if(**p == '|' && *(*p+1) == '|') {
            break;
        }
        /// バックグラウンド ///
        else if(**p == '&' && *(*p+1) != '&') {
            break;
        }
        /// セミコロン ///
        else if(**p == ';') {
            break;
        }
        /// 改行 ///
        else if(is_line_field(*p)) {
            break;
        }
        /// 終了 ///
        else if(**p == 0) {
            break;
        }
        
        /// 単語の区切り ////////////////////////////////////////////
    
        /// 空白 ///
        else if(**p == ' ' || **p == '\t') {
            parse_skip_space(p);
            break;
        }

        /// リダイレクション - エラー出力を標準出力にする ///
        else if(parse_is_redirect2(*p)) {
            break;
        }

        /// リダイレクション ///
        else if(parse_is_redirect(*p)) {
            break;
        }

        /// ブロック ///
        else if(**p == ':' || **p == '{') {
            break;
        }
        
        /// 文の区切り /////////////////////////////////////////////////
        
        /// パイプ ///
        else if(**p == '|' && *(*p+1) != '|') {
            break;
        }
        
        /// 構文 /////////////////////////////////////////////////////

        /// サブシェル ///
        else if(**p == '(') {
            break;
        }
        else {
            string_push_back2(not_evaled, **p);
            string_push_back2(word, **p);
            (*p)++;
        }
    }

    return TRUE;
}

/// parse()で定義された関数をgFuncsに登録する関数
static void parse_update_func(vector_obj* defined_funcs)
{
    int h;
    for(h=0; h<vector_size(defined_funcs); h++) {
        sFunction* func = vector_item(defined_funcs, h);
        
        sFunction* func_before = hash_item(gFuncs, string_c_str(func->name));
        if(func_before) {
            sFunction_delete(func_before);
            hash_put(gFuncs, string_c_str(func->name)
                        , func);
        }
        else {
            hash_put(gFuncs, string_c_str(func->name)
                        , func);
        }
    }
}

static int parse(char* cmdline, char* fname, sStatments* statments, vector_obj* defined_funcs)
{
    char* p = cmdline;

    /// 空白を取る ///
    while(1) {
        if(*p == ' ') p++;
        else if(*p =='\t') p++;
        else if(is_line_field(p)) {
            gLine++;
            if(gScriptLineField == kCRLF) {
                p+=2;
            }
            else {
                p++;
            }
        }
        else if(*p == 0) return TRUE;
        else
            break;
    }

    string_obj* buf = STRING_NEW("");
        // 単語
    sCommand* command = COMMAND_NEW();
        // コマンド
    sStatment* statment = STATMENT_NEW();
        // 文
    string_obj* not_evaled = STRING_NEW(""); 
        // マクロか環境変数があったら、ここに一行入れる
    BOOL read_end_of_statment = -1;
        // このフラグに行番号が入っていたら環境変数かマクロが
        // あったので行末まで読み込む

    /// グローバルパイプ ///
    if(*p == '|' && *(p+1) != '|') {
        string_push_back2(not_evaled, *p++);

        sRedirect* redirection = REDIRECT_NEW();
        redirection->mType = kRedirectGPipeIn;
        vector_add(command->mRedirects, redirection);

        while(1) {
            if(*p == ' ') p++;
            else if(*p =='\t') p++;
            else if(is_line_field(p)) {
                gLine++;
                if(gScriptLineField == kCRLF) {
                    p+=2;
                }
                else {
                    p++;
                }
            }
            else if(*p == 0) {
                string_delete(buf);
                sCommand_delete(command);
                sStatment_delete(statment);
                string_delete(not_evaled);
                
                return TRUE;
            }
            else
                break;
        }
    }

    /// 開始 ///
    BOOL squote = FALSE;
    BOOL dquote = FALSE;
    while(1) {
        /// 特殊文字 ///
        if(*p == '\\' && *(p+1) == 't') {
            string_push_back2(not_evaled, *p++);
            string_push_back2(not_evaled, *p++);

            string_push_back2(buf, '\t');
        }
        else if(*p == '\\' && *(p+1) == 'n') {
            string_push_back2(not_evaled, *p++);
            string_push_back2(not_evaled, *p++);

            string_push_back2(buf, '\n');
        }
        else if(*p == '\\' && *(p+1) == 'r') {
            string_push_back2(not_evaled, *p++);
            string_push_back2(not_evaled, *p++);

            string_push_back2(buf, '\r');
        }
        /// クォート + 改行 ///
        else if(*p == '\\' && is_line_field(p+1)) {
            p++;

            gLine++;
            if(gScriptLineField == kCRLF) {
                p+=2;
            }
            else {
                p++;
            }
        }
        /// クォート ///
        else if(*p == '\\') {
            string_push_back2(not_evaled, *p++);
            if(is_line_field(p)) {
                gLine++;
                if(gScriptLineField == kCRLF) {
                    string_push_back2(not_evaled, *p);
                    string_push_back2(buf, *p++);
                    string_push_back2(not_evaled, *p);
                    string_push_back2(buf, *p++);
                }
                else {
                    string_push_back2(not_evaled, *p);
                    string_push_back2(buf, *p++);
                }
            }
            else {
                string_push_back2(not_evaled, *p);
                string_push_back2(buf, *p++);
            }
        }
        // 空文字列
        else if(!dquote && *p == '\'' && *(p+1) == '\'' 
                || (!squote && *p == '"' && *(p+1) == '"')) 
        {
            string_push_back2(not_evaled, *p);
            string_push_back2(not_evaled, *(p+1));
            p+=2;

            if(strcmp(string_c_str(buf), "") == 0) {
                vector_add(command->mArgs, buf);       // 空文字列を追加
                string_push_back(statment->mTitle, string_c_str(buf));
                string_push_back(statment->mTitle, " ");
                buf = STRING_NEW("");
            }
        }
        // シングルクォート
        else if(!dquote && *p == '\'') {
            string_push_back2(not_evaled, *p);
            p++;
            squote = !squote;
        }
        // ダブルクォート
        else if(!squote && *p == '"') {
            string_push_back2(not_evaled, *p);
            p++;
            dquote = !dquote;
        }

        /// 算術演算 $(())
        else if(!squote && *p == '$' && *(p+1) =='(' && *(p+2) == '(')
        {
            string_push_back2(not_evaled, *p++);
            string_push_back2(not_evaled, *p++);
            string_push_back2(not_evaled, *p++);

            string_obj* str = STRING_NEW("");
            if(!read_skip_next_paren_end(&p, str)) {
                string_delete(buf);
                sCommand_delete(command);
                sStatment_delete(statment);
                string_delete(not_evaled);
                string_delete(str);
                
                return FALSE;
            }

            if(*p != ')') {
                err_msg("parse: need ) in $(())");

                string_delete(buf);
                sCommand_delete(command);
                sStatment_delete(statment);
                string_delete(not_evaled);
                string_delete(str);
                
                return FALSE;
            }

            string_push_back(not_evaled, string_c_str(str));
            string_push_back2(not_evaled, *p++);  // )

            string_delete(str);

            read_end_of_statment = gLine;
        }

        // コマンド展開 $()
        else if(!squote && (*p == '$' && *(p+1) == '$' && *(p+2) == '('
                || *p == '$' && *(p+1) == '('))
        {
            if(*(p+1) == '$') {
                string_push_back2(not_evaled, *p++);
            }
            string_push_back2(not_evaled, *p++);
            string_push_back2(not_evaled, *p++);

            if(!read_skip_next_paren_end(&p, not_evaled)) {
                string_delete(buf);
                sCommand_delete(command);
                sStatment_delete(statment);
                string_delete(not_evaled);
                
                return FALSE;
            }
            
            if(*p == '[') {
                string_push_back2(not_evaled, *p++);
                read_skip_next_bracket_end(&p, not_evaled);
            };

            read_end_of_statment = gLine;
        }
        // 環境変数 $
        else if(!squote 
                && (
                *p == '$' && *(p+1) == '$' 
                    && (*(p+2) == '{' || is_env_name_char(*(p+2)))
                 || *p == '$' && (*(p+1) == '{' 
                 || is_env_name_char(*(p+1))) 
                ))
        {
            if(*(p+1) == '$') string_push_back2(not_evaled, *p++);
            string_push_back2(not_evaled, *p++);

            if(*p == 0) {
                err_msg("parse: invalid env name");
                string_delete(buf);
                sCommand_delete(command);
                sStatment_delete(statment);
                string_delete(not_evaled);
                return FALSE;
            }

            if(*p == '{') {
                string_push_back2(not_evaled, *p++);
                read_skip_next_cbrace_end(&p, not_evaled);
            }
            else {
                char name[1024];
                char* p3 = name;
                while(is_env_name_char(*p)) {
                    *p3++ = *p++;
                }
                *p3 = 0;

                string_push_back(not_evaled, name);
            }

            if(*p == '[') {
                string_push_back2(not_evaled, *p++);
                read_skip_next_bracket_end(&p, not_evaled);
            };
            if(*p == '(') {
                string_push_back2(not_evaled, *p++);
                read_skip_next_paren_end(&p, not_evaled);
            }

            read_end_of_statment = gLine;
        }
            
        /// シングルクォート、ダブルクォート中 ///
        else if(squote || dquote) {
            if(*p == 0) {
                err_msg("parse: need ' or \"\n");
                
                string_delete(buf);
                sCommand_delete(command);
                sStatment_delete(statment);
                string_delete(not_evaled);

                return FALSE;
            }
            else if(is_line_field(p)) {
                gLine++;
                if(gScriptLineField == kCRLF) {
                    string_push_back2(not_evaled, *p);
                    string_push_back2(buf, *p++);
                    string_push_back2(not_evaled, *p);
                    string_push_back2(buf, *p++);
                }
                else {
                    string_push_back2(not_evaled, *p);
                    string_push_back2(buf, *p++);
                }
            }
            else {
                string_push_back2(not_evaled, *p);
                string_push_back2(buf, *p++);
            }
        }
        /// チルダ ///
        else if(*p == '~')
        {
            string_push_back2(not_evaled, *p);
            p++;
            
            char* home = getenv("HOME");
            if(home == NULL) {
                fprintf(stderr, "HOME is null\n");
                exit(1);
            }
            string_push_back(buf, home);
        }
        /// グロブ ///
        else if(*p == '*' || *p == '?' 
                        || *p =='[' && (*(p+1) != ' ' 
                    && *(p+1) != '\t' && is_line_field(p+1))) 
        {
            if(strcmp(string_c_str(buf), "") != 0) {
                string_push_back(statment->mTitle, string_c_str(buf));
                string_push_back(statment->mTitle, " ");
                string_delete(buf);
                buf = STRING_NEW("");
            }

            while(*p != 0 && !is_glob_terminated_char(p)) 
            {
                string_push_back2(not_evaled, *p++);
            }

            read_end_of_statment = gLine;
        }

        /// ファイルハンドル1 ///
        else if(strcmp(string_c_str(buf), "") ==0 
                && parse_is_file_handle(p)) 
        {
            p++;

            string_obj* fname = STRING_NEW("");

            BOOL squote = FALSE;
            BOOL dquote = FALSE;
            while(1) {
                // クォート
                if(*p == '\\') {
                    string_push_back2(fname, *p++);
                    if(is_line_field(p)) {
                        gLine++;
                        if(gScriptLineField == kCRLF) {
                            string_push_back2(fname, *p++);
                            string_push_back2(fname, *p++);
                        }
                        else {
                            string_push_back2(fname, *p++);
                        }
                    }
                    else {
                        string_push_back2(fname, *p++);
                    }
                }
                // シングルクォート
                else if(!dquote && *p == '\'') {
                    p++;
                    squote = !squote;
                }
                // ダブルクォート
                else if(!squote && *p == '"') {
                    p++;
                    dquote = !dquote;
                }
                else if(!squote && *p == '$') {
                    string_push_back2(fname, *p++);
                    read_end_of_statment = gLine;
                }
                else if(squote || dquote) {
                    if(*p == 0) {
                        err_msg("parse: require ' or \". close it");
                        string_delete(buf);
                        sCommand_delete(command);
                        sStatment_delete(statment);
                        string_delete(not_evaled);
                        return FALSE;
                    }
                    else {
                        string_push_back2(fname, *p++);
                    }
                }
                else if(*p == '>') {
                    p++;
                    break;
                }
                else if(is_line_field(p)) {
                    err_msg("invalid line field in file handle");
                    string_delete(buf);
                    sCommand_delete(command);
                    sStatment_delete(statment);
                    string_delete(not_evaled);
                    return FALSE;
                }
                else {
                    string_push_back2(fname, *p++);
                }
            }

            if(!is_terminated_char(p)) {
                err_msg("parse: invalid file handle");
                string_delete(buf);
                sCommand_delete(command);
                sStatment_delete(statment);
                string_delete(not_evaled);
                return FALSE;
            }
            
            string_push_back(not_evaled, "read ");
            string_push_back(not_evaled, string_c_str(fname));

            string_push_back(buf, "read");
            parse_add_command(&buf, command, statment, not_evaled);

            string_push_back(buf, string_c_str(fname));
            parse_add_command(&buf, command, statment, not_evaled);

            string_delete(fname);
        }

        /// 単語の区切り ////////////////////////////////////////////
    
        /// 空白 ///
        else if(*p == ' ' || *p == '\t') {
            while(*p == ' ' || *p == '\t') {
                string_push_back2(not_evaled, *p++);
            }

            parse_add_command(&buf, command, statment, not_evaled);
        }

        /// リダイレクション - エラー出力を標準出力にする ///
        else if(parse_is_redirect2(p)) {
            parse_add_command(&buf, command, statment, not_evaled);

            if(*p == '&') {
                string_push_back2(not_evaled, *p++);
                string_push_back2(not_evaled, *p++);
            }
            else {
                string_push_back2(not_evaled, *p++);
                string_push_back2(not_evaled, *p++);
                string_push_back2(not_evaled, *p++);
                string_push_back2(not_evaled, *p++);
            }
            
            sRedirect* redirection = REDIRECT_NEW();
            redirection->mType = kRedirectErrAndOutput;

            vector_add(command->mRedirects, redirection);
        }

        /// リダイレクション ///
        else if(parse_is_redirect(p)) 
        {
            parse_add_command(&buf, command, statment, not_evaled);

            sRedirect* redirection = REDIRECT_NEW();
            parse_redirect(&p, redirection, not_evaled);

            string_obj* word = STRING_NEW("");
            if(!parse_get_word(&p, word, not_evaled, &read_end_of_statment)) {
                string_delete(buf);
                sCommand_delete(command);
                sStatment_delete(statment);
                string_delete(not_evaled);
                    
                return FALSE;
            }

            string_put(redirection->mFName, string_c_str(word));

            vector_add(command->mRedirects, redirection);

            string_delete(word);
        }

        /// ブロック
        else if(*p == ':' || *p == '{')
        {
            parse_add_command(&buf, command, statment, not_evaled);

            BOOL brace_block = *p == '{';

            string_obj* str = STRING_NEW("");
            if(!read_get_block(&p, str, not_evaled)) {
                string_delete(buf);
                sCommand_delete(command);
                sStatment_delete(statment);
                string_delete(not_evaled);

                string_delete(str);

                return FALSE;
            }

            string_put(buf, string_c_str(str));
            string_delete(str);

            /// カッコのブロック ///
            if(brace_block) {
                while(*p == ' ' || *p == '\t') {
                    string_push_back2(not_evaled, *p++);
                }

                parse_add_command(&buf, command, statment, not_evaled);
            }
            /// インデントなら、文末の処理をする ///
            else {
                if(read_end_of_statment == -1) {
                    parse_add_command(&buf, command, statment, not_evaled);
                    parse_add_statment(&command, statment);
                    parse_add_statments(kTNormal, &statment, statments
                                        , gLine, fname);

                    string_delete(not_evaled);
                    not_evaled = STRING_NEW("");
                }
                else {
                    parse_add_command(&buf, command, statment, not_evaled);
                    parse_add_statment(&command, statment);
                    parse_add_statments2(kTNormal, &statment, statments
                                      , fname, read_end_of_statment, not_evaled);
        
                    not_evaled = STRING_NEW("");

                    read_end_of_statment = -1;
                }
            }
        }
        
        /// 文の区切り /////////////////////////////////////////////////
        
        /// パイプ ///
        else if(*p == '|' && *(p+1) != '|') {
            string_push_back2(not_evaled, *p);
            p++;

            while(*p == ' ' || *p == '\t') {
                string_push_back2(not_evaled, *p++);
            }
            
            /// 文の区切り ///
            if(*p == ';' || is_line_field(p) || *p==0 || *p=='#'
                || *p == '&' && *(p+1) == '&'
                || *p == '|' && *(p+1) == '|' 
                || *p == '&') 
            {
                sRedirect* redirection = REDIRECT_NEW();
                redirection->mType = kRedirectGPipeOut;
                vector_add(command->mRedirects, redirection);
            }
            
            parse_add_command(&buf, command, statment, not_evaled);
            parse_add_statment(&command, statment);
        }
        
        /// 複文(statment)の区切り系 //////////////////////////////////
    
        /// アンドアンド ///
        else if(*p == '&' && *(p+1) == '&') {
            p+=2;

            int line = gLine;

            while(*p == ' ' || *p == '\t' 
                    || (is_line_field(p) && gLine++)) 
            {
                if(gScriptLineField == kCRLF && is_line_field(p)) {
                    string_push_back2(not_evaled, *p++);
                    string_push_back2(not_evaled, *p++);
                }
                else {
                    string_push_back2(not_evaled, *p++);
                }
            }
            
            if(read_end_of_statment == -1) {
                parse_add_command(&buf, command, statment, not_evaled);
                parse_add_statment(&command, statment);
                parse_add_statments(kTAndAnd, &statment, statments
                                        , line, fname);

                string_delete(not_evaled);
                not_evaled = STRING_NEW("");
            }
            else {
                parse_add_command(&buf, command, statment, not_evaled);
                parse_add_statment(&command, statment);
                parse_add_statments2(kTAndAnd, &statment, statments
                                    , fname
                                    , read_end_of_statment, not_evaled);
                
                not_evaled = STRING_NEW("");
            }
            
            if(*p == '|' && *(p+1) != '|') {
                string_push_back2(not_evaled, *p++);
                
                sRedirect* redirection = REDIRECT_NEW();
                redirection->mType = kRedirectGPipeIn;
                vector_add(command->mRedirects, redirection);
            }
        }
        /// オアオア ///
        else if(*p == '|' && *(p+1) == '|') {
            p+=2;

            int line = gLine;

            while(*p == ' ' || *p == '\t' || (is_line_field(p) && gLine++)) {
                if(gScriptLineField == kCRLF && is_line_field(p)) {
                    string_push_back2(not_evaled, *p++);
                    string_push_back2(not_evaled, *p++);
                }
                else {
                    string_push_back2(not_evaled, *p++);
                }
            }
            
            if(read_end_of_statment == -1) {
                parse_add_command(&buf, command, statment, not_evaled);
                parse_add_statment(&command, statment);
                parse_add_statments(kTOrOr, &statment, statments
                                    , line, fname);
    
                string_delete(not_evaled);
                not_evaled = STRING_NEW("");
            }
            else {
                parse_add_command(&buf, command, statment, not_evaled);
                parse_add_statment(&command, statment);
                parse_add_statments2(kTOrOr, &statment, statments
                                  , fname, read_end_of_statment, not_evaled);
    
                not_evaled = STRING_NEW("");

                read_end_of_statment = -1;
            }
            
            if(*p == '|' && *(p+1) != '|') {
                string_push_back2(not_evaled, *p++);
                
                sRedirect* redirection = REDIRECT_NEW();
                redirection->mType = kRedirectGPipeIn;
                vector_add(command->mRedirects, redirection);
            }
        }
        /// バックグラウンド ///
        else if(*p == '&' && *(p+1) != '&') {
            string_push_back2(not_evaled, *p);
            p++;

            if(read_end_of_statment == -1) {
                statment->mBackground = TRUE;

                parse_add_command(&buf, command, statment, not_evaled);
                parse_add_statment(&command, statment);
                parse_add_statments(kTNormal, &statment, statments
                                , gLine, fname);

                string_delete(not_evaled);
                not_evaled = STRING_NEW("");
            }
            else {
                parse_add_command(&buf, command, statment, not_evaled);
                parse_add_statment(&command, statment);
                parse_add_statments2(kTNormal, &statment, statments
                                   , fname, read_end_of_statment, not_evaled);
    
                not_evaled = STRING_NEW("");

                read_end_of_statment = -1;
            }
            
            if(*p == '|' && *(p+1) != '|') {
                string_push_back2(not_evaled, *p++);
                
                sRedirect* redirection = REDIRECT_NEW();
                redirection->mType = kRedirectGPipeIn;
                vector_add(command->mRedirects, redirection);
            }
        }
        /// セミコロン ///
        else if(*p == ';') {
            p++;
            int line = gLine;

            while(*p == ' ' || *p == '\t' || (is_line_field(p) && gLine++)) {
                if(gScriptLineField == kCRLF && is_line_field(p)) {
                    string_push_back2(not_evaled, *p++);
                    string_push_back2(not_evaled, *p++);
                }
                else {
                    string_push_back2(not_evaled, *p++);
                }
            }
            
            if(read_end_of_statment == -1) {
                parse_add_command(&buf, command, statment, not_evaled);
                parse_add_statment(&command, statment);
                parse_add_statments(kTNormal, &statment, statments
                                , line,  fname);

                string_delete(not_evaled);
                not_evaled = STRING_NEW("");
            }
            else {
                parse_add_command(&buf, command, statment, not_evaled);
                parse_add_statment(&command, statment);
                parse_add_statments2(kTNormal, &statment, statments
                                  , fname, read_end_of_statment, not_evaled);
    
                not_evaled = STRING_NEW("");

                read_end_of_statment = -1;
            }
            
            if(*p == '|' && *(p+1) != '|') {
                string_push_back2(not_evaled, *p++);
                
                sRedirect* redirection = REDIRECT_NEW();
                redirection->mType = kRedirectGPipeIn;
                vector_add(command->mRedirects, redirection);
            }
        }
        /// 改行 ///
        else if(is_line_field(p)) {
            if(gScriptLineField == kCRLF) {
                p+=2;
            }
            else {
                p++;
            }
            int line = gLine;
            gLine++;

            while(*p == ' ' || *p == '\t' || (is_line_field(p) && gLine++)) {
                if(gScriptLineField == kCRLF && is_line_field(p)) {
                    p+=2;
                    /*
                    string_push_back2(not_evaled, *p++);
                    string_push_back2(not_evaled, *p++);
                    */
                }
                else {
                    p++;
                    //string_push_back2(not_evaled, *p++);
                }
            }
            
            if(read_end_of_statment == -1) {
                parse_add_command(&buf, command, statment, not_evaled);
                parse_add_statment(&command, statment);
                parse_add_statments(kTNormal,  &statment
                                    , statments, line, fname);

                string_delete(not_evaled);
                not_evaled = STRING_NEW("");
            }
            else {
                parse_add_command(&buf, command, statment, not_evaled);
                parse_add_statment(&command, statment);
                parse_add_statments2(kTNormal, &statment, statments
                                  , fname, read_end_of_statment, not_evaled);
    
                not_evaled = STRING_NEW("");

                read_end_of_statment = -1;
            }
            
            if(*p == '|' && *(p+1) != '|') {
                string_push_back2(not_evaled, *p++);
                
                sRedirect* redirection = REDIRECT_NEW();
                redirection->mType = kRedirectGPipeIn;
                vector_add(command->mRedirects, redirection);
            }
        }
        /// 終了 ///
        else if(*p == 0) {
            if(read_end_of_statment == -1) {
                parse_add_command(&buf, command, statment, not_evaled);
                parse_add_statment(&command, statment);
                parse_add_statments(kTNormal, &statment, statments
                                    , gLine, fname);

                string_delete(not_evaled);
                string_delete(buf);
                sCommand_delete(command);
                sStatment_delete(statment);
            }
            else {
                parse_add_command(&buf, command, statment, not_evaled);
                parse_add_statment(&command, statment);
                parse_add_statments2(kTNormal,&statment, statments
                             , fname, read_end_of_statment, not_evaled);
    
                string_delete(buf);
                sCommand_delete(command);
                sStatment_delete(statment);
            }

            break;
        }
        
        /// 構文 ///////////////////////////////////////////////////////

        /// サブシェル ///
        else if(*p == '(') {
            if(!read_get_subshell(&p, statment, command, buf, not_evaled)) {
                return FALSE;
            }
        }

        /// if文 ///
        else if(vector_size(command->mArgs) == 0 
                && strcmp(string_c_str(buf), "") ==0 
                && *p == 'i' && *(p+1) == 'f' 
                && is_terminated_char(p+2))
        {
            string_push_back2(not_evaled, *p++);
            string_push_back2(not_evaled, *p++);

            int first_line = gLine;

            /// 条件式 ///
            string_obj* str = STRING_NEW("");
            if(!read_get_joukensiki(&p, str, not_evaled)) {
                string_delete(buf);
                sCommand_delete(command);
                sStatment_delete(statment);
                string_delete(not_evaled);
                
                string_delete(str);

                return FALSE;
            }

            sStatments* joukensiki = STATMENTSLIST_NEW();
            if(!parse(string_c_str(str), gFName, joukensiki
                    , defined_funcs))
            {
                string_delete(buf);
                sCommand_delete(command);
                sStatment_delete(statment);
                string_delete(not_evaled);

                string_delete(str);
                sStatments_delete(joukensiki);

                return FALSE;
            }

            string_delete(str);

            /// 中身 ///
            BOOL brace_block = *p == '{';

            int line = gLine;
            str = STRING_NEW("");
            BOOL ret = read_get_block(&p, str, not_evaled);
            if(!ret) {
                string_delete(buf);
                sCommand_delete(command);
                sStatment_delete(statment);
                string_delete(not_evaled);

                string_delete(str);
                sStatments_delete(joukensiki);

                return FALSE;
            }
            gLine = line;

            sStatments* contents = STATMENTSLIST_NEW();
            if(!parse(string_c_str(str), gFName, contents, defined_funcs)) {
                string_delete(buf);
                sCommand_delete(command);
                sStatment_delete(statment);
                string_delete(not_evaled);

                string_delete(str);
                sStatments_delete(joukensiki);
                sStatments_delete(contents);

                return FALSE;
            }

            string_delete(str);

            while(*p == ' ' || *p == '\t' 
                    || is_line_field(p) && gLine++) 
            {
                if(gScriptLineField == kCRLF && is_line_field(p)) {
                    string_push_back2(not_evaled, *p++);
                    string_push_back2(not_evaled, *p++);
                }
                else {
                    string_push_back2(not_evaled, *p++);
                }
            }
            
            vector_obj* joukensiki_list = VECTOR_NEW(3);
            vector_obj* contents_list = VECTOR_NEW(3);
            
            vector_add(joukensiki_list, joukensiki);
            vector_add(contents_list, contents);

            /// elif 開始 ///
            if(*p == 'e' && *(p+1) == 'l' 
                && *(p+2) == 'i' && *(p+3) == 'f' 
                && is_terminated_char(p+4)) 
            {
                while(*p == 'e' && *(p+1) == 'l' && *(p+2) == 'i' 
                    && *(p+3) == 'f' && is_terminated_char(p+4)) 
                {
                    string_push_back2(not_evaled, *p++);
                    string_push_back2(not_evaled, *p++);
                    string_push_back2(not_evaled, *p++);
                    string_push_back2(not_evaled, *p++);

                    /// 条件式 ///
                    str = STRING_NEW("");

                    if(!read_get_joukensiki(&p, str, not_evaled)) {
                        string_delete(buf);
                        sCommand_delete(command);
                        sStatment_delete(statment);
                        string_delete(not_evaled);

                        string_delete(str);
                        
                        int k;
                        for(k=0; k<vector_size(joukensiki_list); k++) {
                            sStatments_delete(vector_item(joukensiki_list, k));
                        }
                        vector_delete(joukensiki_list);
                        for(k=0; k<vector_size(contents_list); k++) {
                            sStatments_delete(vector_item(contents_list, k));
                        }
                        vector_delete(contents_list);

                        return FALSE;
                    }

                    joukensiki = STATMENTSLIST_NEW();
                    if(!parse(string_c_str(str), gFName, joukensiki, defined_funcs)) {
                        string_delete(buf);
                        sCommand_delete(command);
                        sStatment_delete(statment);
                        string_delete(not_evaled);

                        string_delete(str);
                        sStatments_delete(joukensiki);
                        
                        int k;
                        for(k=0; k<vector_size(joukensiki_list); k++) {
                            sStatments_delete(vector_item(joukensiki_list, k));
                        }
                        vector_delete(joukensiki_list);
                        for(k=0; k<vector_size(contents_list); k++) {
                            sStatments_delete(vector_item(contents_list, k));
                        }
                        vector_delete(contents_list);

                        return FALSE;
                    }

                    string_delete(str);
                    vector_add(joukensiki_list, joukensiki);

                    /// 中身 ///
                    brace_block = *p == '{';

                    str = STRING_NEW("");
                    line = gLine;
                    BOOL ret = read_get_block(&p, str, not_evaled);
                    if(!ret) {
                        string_delete(buf);
                        sCommand_delete(command);
                        sStatment_delete(statment);
                        string_delete(not_evaled);
                        
                        string_delete(str);
                        
                        int k;
                        for(k=0; k<vector_size(joukensiki_list); k++) {
                            string_delete(vector_item(joukensiki_list, k));
                        }
                        vector_delete(joukensiki_list);
                        for(k=0; k<vector_size(contents_list); k++) {
                            string_delete(vector_item(contents_list, k));
                        }
                        vector_delete(contents_list);

                        return FALSE;
                    }
                    gLine = line;

                    contents = STATMENTSLIST_NEW();
                    if(!parse(string_c_str(str), gFName, contents, defined_funcs)) {
                        string_delete(buf);
                        sCommand_delete(command);
                        sStatment_delete(statment);
                        string_delete(not_evaled);
                        
                        string_delete(str);
                        sStatments_delete(contents);
                        
                        int k;
                        for(k=0; k<vector_size(joukensiki_list); k++) {
                            string_delete(vector_item(joukensiki_list, k));
                        }
                        vector_delete(joukensiki_list);
                        for(k=0; k<vector_size(contents_list); k++) {
                            string_delete(vector_item(contents_list, k));
                        }
                        vector_delete(contents_list);

                        return FALSE;
                    }

                    string_delete(str);
                    vector_add(contents_list, contents);

                    while(*p == ' '||*p == '\t'|| is_line_field(p) && gLine++) 
                    {
                        if(gScriptLineField == kCRLF && is_line_field(p)) {
                            p+=2;
                        }
                        else {
                            p++;
                        }
                    }
                }
            }

            /// else 開始 ///
            if(*p == 'e' && *(p+1) == 'l' && *(p+2) == 's' && *(p+3) == 'e'
                && is_terminated_char(p+4)) 
            {
                string_push_back2(not_evaled, *p++);
                string_push_back2(not_evaled, *p++);
                string_push_back2(not_evaled, *p++);
                string_push_back2(not_evaled, *p++);

                /// 条件式がないので:, {まで移動
                while(*p != ':' && *p != '{') {
                    if(*p == ' ' || *p == '\t') {
                        string_push_back2(not_evaled, *p++);
                    }
                    else {
                        err_msg("parse: require : or { after else");

                        string_delete(buf);
                        sCommand_delete(command);
                        sStatment_delete(statment);
                        string_delete(not_evaled);
                        
                        string_delete(str);
                            
                        int k;
                        for(k=0; k<vector_size(joukensiki_list); k++) {
                            string_delete(vector_item(joukensiki_list, k));
                        }
                        vector_delete(joukensiki_list);
                        for(k=0; k<vector_size(contents_list); k++) {
                            string_delete(vector_item(contents_list, k));
                        }
                        vector_delete(contents_list);

                        return FALSE;
                    }
                }

                /// 中身 ///
                brace_block = *p == '{';

                str = STRING_NEW("");
                int line = gLine;
                BOOL ret = read_get_block(&p, str, not_evaled);
                if(!ret) {
                    string_delete(buf);
                    sCommand_delete(command);
                    sStatment_delete(statment);
                    string_delete(not_evaled);
                    
                    string_delete(str);
                        
                    int k;
                    for(k=0; k<vector_size(joukensiki_list); k++) {
                        string_delete(vector_item(joukensiki_list, k));
                    }
                    vector_delete(joukensiki_list);
                    for(k=0; k<vector_size(contents_list); k++) {
                        string_delete(vector_item(contents_list, k));
                    }
                    vector_delete(contents_list);

                    return FALSE;
                }
                gLine = line;

                contents = STATMENTSLIST_NEW();
                if(!parse(string_c_str(str), gFName, contents, defined_funcs)) {
                    string_delete(buf);
                    sCommand_delete(command);
                    sStatment_delete(statment);
                    string_delete(not_evaled);
                    
                    string_delete(str);
                    sStatments_delete(contents);
                        
                    int k;
                    for(k=0; k<vector_size(joukensiki_list); k++) {
                        string_delete(vector_item(joukensiki_list, k));
                    }
                    vector_delete(joukensiki_list);
                    for(k=0; k<vector_size(contents_list); k++) {
                        string_delete(vector_item(contents_list, k));
                    }
                    vector_delete(contents_list);

                    return FALSE;
                }

                string_delete(str);
                vector_add(contents_list, contents);
            }
            
            command->mExtra = sIf_new(joukensiki_list, contents_list);
            string_put(buf, "if");

            /// 文末の処理 ////
            if(brace_block) {
                while(*p == ' ' || *p == '\t') {
                    string_push_back2(not_evaled, *p++);
                }

                if(!parse_check_special_statment_continue(&p)) {
                    err_msg("parse: need to terminate statment after brace end(}), for example ; or && or || etc A");

                    sIf_delete(command->mExtra);

                    string_delete(buf);
                    sCommand_delete(command);
                    sStatment_delete(statment);
                    string_delete(not_evaled);

                    return FALSE;
                }
                parse_add_command(&buf, command, statment, not_evaled);
            }
            else {
                if(read_end_of_statment == -1) {
                    parse_add_command(&buf, command
                                            , statment, not_evaled);
                    parse_add_statment(&command, statment);
                    parse_add_statments(kTNormal, &statment, statments
                                    , first_line, fname);

                    string_delete(not_evaled);
                    not_evaled = STRING_NEW("");
                }
                else {
                    parse_add_command(&buf, command, statment, not_evaled);
                    parse_add_statment(&command, statment);
                    parse_add_statments2(kTNormal, &statment, statments
                                  , fname, read_end_of_statment, not_evaled);
        
                    not_evaled = STRING_NEW("");

                    read_end_of_statment = -1;
                }
            }
        }

        /// break文 ///
        else if(strcmp(string_c_str(buf), "") == 0 
            && vector_size(command->mArgs) == 0
            && *p == 'b' && *(p+1) == 'r' && *(p+2) == 'e' 
            && *(p+3) == 'a' && *(p+4) == 'k' 
            && is_terminated_char(p+5))
        {
            string_push_back2(not_evaled, *p++);
            string_push_back2(not_evaled, *p++);
            string_push_back2(not_evaled, *p++);
            string_push_back2(not_evaled, *p++);
            string_push_back2(not_evaled, *p++);

            /// 追加 ///
            command->mKind = kBreak;
            vector_add(command->mArgs, STRING_NEW("break"));
            vector_add(statment->mCommands, command);
            command = COMMAND_NEW();

            statment->mTerminated = kTNormal;
            string_put(statment->mFName, gFName);
            statment->mLine = gLine;

            vector_add(statments->mStatments, statment);
            statment = STATMENT_NEW();
        }

        /// while 文 ///
        else if(vector_size(command->mArgs) == 0 
                && strcmp(string_c_str(buf), "") == 0 
                && *p == 'w' && *(p+1) == 'h' && *(p+2) == 'i' 
                && *(p+3) == 'l' && *(p+4) == 'e' 
            && is_terminated_char(p+5))
        {
            string_push_back2(not_evaled, *p++);
            string_push_back2(not_evaled, *p++);
            string_push_back2(not_evaled, *p++);
            string_push_back2(not_evaled, *p++);
            string_push_back2(not_evaled, *p++);

            int first_line = gLine;

            /// 条件式 ///
            string_obj* joukensiki = STRING_NEW("");
            if(!read_get_joukensiki(&p, joukensiki, not_evaled)) {
                string_delete(buf);
                sCommand_delete(command);
                sStatment_delete(statment);
                string_delete(not_evaled);

                string_delete(joukensiki);

                return FALSE;
            }

            sStatments* wstatments = STATMENTSLIST_NEW();
            if(!parse(string_c_str(joukensiki), gFName, wstatments, defined_funcs)) {
                string_delete(buf);
                sCommand_delete(command);
                sStatment_delete(statment);
                string_delete(not_evaled);

                string_delete(joukensiki);
                sStatments_delete(wstatments);

                return FALSE;
            }

            string_delete(joukensiki);

            /// 中身 ///
            BOOL brace_block = *p == '{';

            string_obj* contents = STRING_NEW("");
            int line = gLine;
            BOOL ret = read_get_block(&p, contents, not_evaled);
            if(!ret) {
                string_delete(buf);
                sCommand_delete(command);
                sStatment_delete(statment);
                string_delete(not_evaled);

                sStatments_delete(wstatments);
                string_delete(contents);

                return FALSE;
            }
            gLine = line;

            sStatments* wstatments_list2 = STATMENTSLIST_NEW();
            if(!parse(string_c_str(contents), gFName, wstatments_list2, defined_funcs)) 
            {
                string_delete(buf);
                sCommand_delete(command);
                sStatment_delete(statment);
                string_delete(not_evaled);

                sStatments_delete(wstatments);
                string_delete(contents);
                sStatments_delete(wstatments_list2);

                return FALSE;
            }

            string_delete(contents);

            /// 文末の処理 ///
            command->mExtra = sWhile_new(wstatments, wstatments_list2);
            string_put(buf, "while");

            if(brace_block) {
                while(*p == ' ' || *p == '\t') {
                    string_push_back2(not_evaled, *p++);
                }

                if(!parse_check_special_statment_continue(&p)) {
                    err_msg("parse: need to terminate statment after brace end(}), for example ; or && or || etc B");
                    sWhile_delete(command->mExtra);

                    string_delete(buf);
                    sCommand_delete(command);
                    sStatment_delete(statment);
                    string_delete(not_evaled);

                    return FALSE;
                }
                parse_add_command(&buf, command, statment, not_evaled);
            }
            else {
                if(read_end_of_statment == -1) {
                    parse_add_command(&buf, command, statment, not_evaled);
                    parse_add_statment(&command, statment);
                    parse_add_statments(kTNormal, &statment, statments
                                    , first_line, fname);

                    string_delete(not_evaled);
                    not_evaled = STRING_NEW("");
                }
                else {
                    parse_add_command(&buf, command, statment, not_evaled);
                    parse_add_statment(&command, statment);
                    parse_add_statments2(kTNormal, &statment, statments
                                  , fname, read_end_of_statment, not_evaled);
        
                    not_evaled = STRING_NEW("");

                    read_end_of_statment = -1;
                }
            }
        }
        /// def 文 ///
        else if(vector_size(command->mArgs) == 0 
                && strcmp(string_c_str(buf), "") == 0 
                && *p == 'd' && *(p+1) == 'e' && *(p+2) == 'f' 
                && is_terminated_char(p+3))
        {
            string_push_back2(not_evaled, *p++);
            string_push_back2(not_evaled, *p++);
            string_push_back2(not_evaled, *p++);

            int first_line = gLine;

            while(*p == ' ' || *p == '\t') {
                string_push_back2(not_evaled, *p++);
            }

            /// 関数名を得る ///
            string_obj* name = STRING_NEW("");
            while(*p >= 'a' && *p <= 'z' || *p >= 'A' && *p <= 'Z'
                    || *p == '_' || *p >= '0' && *p <= '9')
            {
                string_push_back2(not_evaled, *p);
                string_push_back2(name, *p++);
            }

            /// 引数名指定 ///
            string_obj* arg_name = STRING_NEW("");
            if(*p == '(') {
                string_push_back2(not_evaled, *p);
                p++;

                while(*p == ' ' || *p == '\t') {
                    string_push_back2(not_evaled, *p);
                    p++;
                }

                while(*p >= 'a' && *p <= 'z' || *p >= 'A' && *p <= 'Z'
                        || *p == '_' || *p >= '0' && *p <= '9')
                {
                    string_push_back2(not_evaled, *p);
                    string_push_back2(arg_name, *p++);
                }

                while(*p == ' ' || *p == '\t') {
                    string_push_back2(not_evaled, *p);
                    p++;
                }

                if(*p != ')') {
                    err_msg("require )");

                    string_delete(buf);
                    sCommand_delete(command);
                    sStatment_delete(statment);
                    string_delete(not_evaled);
                    string_delete(name);

                    string_delete(arg_name);

                    return FALSE;
                }

                string_push_back2(not_evaled, *p);
                p++;
            }

            while(*p == ' ' || *p == '\t') {
                string_push_back2(not_evaled, *p);
                p++;
            }
            
            BOOL brace_block = *p == '{';

            string_obj* contents = STRING_NEW("");
            int line = gLine;
            BOOL ret = read_get_block(&p, contents, not_evaled);
            if(!ret) {
                string_delete(buf);
                sCommand_delete(command);
                sStatment_delete(statment);
                string_delete(not_evaled);
                string_delete(name);
                string_delete(arg_name);

                string_delete(contents);

                return FALSE;
            }
            gLine = line;

            sStatments* dstatments = STATMENTSLIST_NEW();

            if(!parse(string_c_str(contents), gFName, dstatments
                    , defined_funcs)) 
            {
                string_delete(buf);
                sCommand_delete(command);
                sStatment_delete(statment);
                string_delete(not_evaled);
                string_delete(name);

                string_delete(contents);
                sStatments_delete(dstatments);
                string_delete(arg_name);

                return FALSE;
            }

            string_delete(contents);

            vector_add(defined_funcs, sFunction_new(string_c_str(name)
                            , dstatments, string_c_str(arg_name)));

            string_delete(name);
            string_delete(arg_name);

            /// 文末の処理 ///
            if(brace_block) {
                while(*p == ' ' || *p == '\t') {
                    string_push_back2(not_evaled, *p++);
                }

                if(!parse_check_special_statment_continue(&p)) {
                    err_msg("parse: need to terminate statment after brace end(}), for example ; or && or || etc C");
                    string_delete(buf);
                    sCommand_delete(command);
                    sStatment_delete(statment);
                    string_delete(not_evaled);

                    return FALSE;
                }
                parse_add_command(&buf, command, statment, not_evaled);
            }
            else {
                if(read_end_of_statment == -1) {
                    parse_add_command(&buf, command, statment, not_evaled);
                    parse_add_statment(&command, statment);
                    parse_add_statments(kTNormal, &statment, statments
                                    , first_line, fname);

                    string_delete(not_evaled);
                    not_evaled = STRING_NEW("");
                }
                else {
                    parse_add_command(&buf, command, statment, not_evaled);
                    parse_add_statment(&command, statment);
                    parse_add_statments2(kTNormal, &statment, statments
                                  , fname, read_end_of_statment, not_evaled);
        
                    not_evaled = STRING_NEW("");

                    read_end_of_statment = -1;
                }
            }
        }
/*
        /// __END__ 文 ///
        else if(vector_sizse(command->mArgs) == 0
                && strcmp(string_c_str(buf), "") == 0
                && *p == '_' && *(p+1) == '_'
                && *(p+2) == 'E' && *(p+3) == 'N'
                && *(p+4) == 'D' && *(p+5) == '_'
                && *(p+6) == '_'
        {
        }
*/
        // その他
        else {
            string_push_back2(not_evaled, *p);
            string_push_back2(buf, *p++);
        }
    }

    return TRUE;
}

//////////////////////////////////////////////////////////////////////
// コンパイル
//////////////////////////////////////////////////////////////////////
BOOL kitutuki_compile(char* fname, char* out_fname)
{
    /// 読み込み ///
    string_obj* str = STRING_NEW("");

    int f = open(fname, O_RDONLY);
    if(f < 0) {
        string_delete(str);
        return FALSE;
    }

    char buf[BUFSIZ];
    while(1) {
        int size = read(f, buf, BUFSIZ-1);
        if(size == 0) {
            break;
        }
        if(size < 0) {
            string_delete(str);
            close(f);
            return FALSE;
        }

        buf[size] = 0;

        string_push_back(str, buf);
    }
    close(f);
    
    /// パース ///
    sStatments* statments = STATMENTSLIST_NEW();

    strcpy(gFName, fname);
    gLine = 1;

    string_obj* str2 = STRING_NEW("");
    if(!read_expand_brace_expansion(string_c_str(str), str2, " ")) {
        sStatments_delete(statments);
        string_delete(str2);
        string_delete(str);
        return FALSE;
    }

    vector_obj* defined_funcs = VECTOR_NEW(20);
    if(!parse(string_c_str(str2), gFName, statments, defined_funcs)) {
        int i;
        for(i=0; i<vector_size(defined_funcs); i++) {
            sFunction_delete(vector_item(defined_funcs, i));
        }
        vector_delete(defined_funcs);
        sStatments_delete(statments);
        
        string_delete(str);
        string_delete(str2);
        return FALSE;
    }

    
    string_delete(str);
    string_delete(str2);

    /// コンパイル結果を保存 ///
    int fd = open(out_fname, O_WRONLY|O_TRUNC|O_CREAT, 0644);
    if(write(fd, gMagicNumber, strlen(gMagicNumber)) < 0) {
        perror("write");
        exit(1);
    }

    sStatments_save(statments, fd);

    /// ユーザー関数を保存 ///
    int n = vector_size(defined_funcs);
    if(write(fd, &n, sizeof(int)) < 0) {
        perror("write");
        exit(1);
    }

    int i;
    for(i=0; i<vector_size(defined_funcs); i++) {
        sFunction* fun = vector_item(defined_funcs, i);
        sFunction_save(fun, fd);
        sFunction_delete(fun);
    }
    vector_delete(defined_funcs);

    close(fd);

    sStatments_delete(statments);
    
    return TRUE;
}

static int run(sStatments* statments, char* title, int pipeout, int pipein, int pipeerr, BOOL finaly_exec, BOOL in_loop);

// -1ならエラーメッセージがgErrMsgに入っているので出力してください

int kitutuki_load_obj(char* fname, int pipeout, int pipein, int pipeerr)
{
    gSaphireSigInt = FALSE;
    int fd = open(fname, O_RDONLY);
    if(fd < 0) {
        return -1;
    }
    
    char magic_number[64];
    if(read(fd, magic_number, strlen(gMagicNumber)) < 0) {
        perror("read");
        exit(1);
    }
    magic_number[strlen(gMagicNumber)] = 0;
    
    if(strcmp(magic_number, gMagicNumber) != 0) {
        return -1;
    }
    
    /// オブジェクトファイルをロード ///
    sStatments* statments = sStatments_load(fd);

    /// 関数定義をロード ///
    int n;

    if(read(fd, &n, sizeof(int)) < 0) {
        perror("read");
        exit(1);
    }

    int i;
    for(i=0; i<n; i++) {
        sFunction* func = sFunction_load(fd);
        sFunction* func_before = hash_item(gFuncs, string_c_str(func->name));
        if(func_before) {
            sFunction_delete(func_before);
            hash_put(gFuncs, string_c_str(func->name), func);
        }
        else {
            hash_put(gFuncs, string_c_str(func->name), func);
        }
    }
    
    /// 実行 ///
    strcpy(gFName, fname);
    gLine = 1;

    int rcode = run(statments, gFName, pipeout, pipein, pipeerr
                    , gAppType == kATOptC, FALSE);

    /// 解放 ///
    close(fd);

    sStatments_delete(statments);

    return rcode;
}

//////////////////////////////////////////////////////////////////////
// 実行
//////////////////////////////////////////////////////////////////////


//////////////////////////////////////////////////////////////////////
/// statment_tree_internal_commandsの子関数
//////////////////////////////////////////////////////////////////////
static int statment_tree_internal_commands_choise(char* msg, char* str[], int len, BOOL list_view)
{
    while(1) {
        /// view ///
        if(list_view)
            printf("%s\n", msg);
        else
            printf("%s ", msg);

        int i;
        for(i=0; i<len; i++) {
            if(list_view) {
                printf("(%d)%s \n", i,  str[i]);
            }
            else {
                printf("(%d)%s ", i,  str[i]);
            }
        }

        printf("--> ");
        fflush(stdout);

        /// input ///
        char str2[32];
        if(scanf("%s", str2) == 1) {
            fflush(stdin);

            char* p = str2;
            while(*p >= '0' && *p <= '9') {
                p++;
            }

            if(*p == 0) {
                int ret = atoi(str2);

                if(ret >= 0 && ret < len) {
                    return ret;
                }
            }
        }
        else {
            fflush(stdin);
        }
    }
}

static void statment_tree_internal_commands_selector(string_obj* output, string_obj* result, int* scrolltop, int* cursor, enum eKanjiCode code, BOOL multiple, enum eLineField lf)
{
    if(tcsetpgrp(0, getpid()) < 0) {
        switch(errno) {
            case EBADF:
                puts("EBADF");
                break;

            case EINVAL:
                puts("EINVAL");
                break;

            case ENOTTY:
                puts("ENOTTY");
                break;

            case EPERM:
                puts("EPERM");
                break;

            default:
                puts("default");
                break;
        }
        perror("tcsetpgrp inner command");
        exit(1);
    }

#if !defined(__CYGWIN7__) && !defined(__FREEBSD__)
    msave_screen();
#endif
    msave_ttysettings();
    minitscr();

    const int maxx = mgetmaxx();
    const int maxy = mgetmaxy();

    vector_obj* lines = VECTOR_NEW(10);

    /// 入力ごとにlinesに格納 ///
    string_obj* tmp = STRING_NEW("");
    char* p = string_c_str(output);
    while(*p) {
        if(is_line_field2(lf, p)) {
            if(lf == kCRLF) {
                p+=2;
            }
            else {
                p++;
            }
            vector_add(lines, tmp);
            tmp = STRING_NEW("");
        }
        else {
            string_push_back2(tmp, *p++);
        }
    }
    vector_add(lines, tmp);

    static int line_num = -1;
    if(line_num != -1 && vector_size(lines) != line_num) {
        *scrolltop = 0;
        *cursor = 0;
    }
    line_num = vector_size(lines);

    int* markfiles = MALLOC(sizeof(int)*vector_size(lines));
    memset(markfiles, 0, sizeof(int)*vector_size(lines));

    if(vector_size(lines) > 0) {
        while(1) {
            /// 描写 ///
            mclear();
            int n = *scrolltop;
            int y = 0;
            while(y < maxy && n < vector_size(lines)) {
                int attrs = 0;
                if(n == *cursor) {
                    attrs |= kCAReverse;
                }
                else if(markfiles[n]) {
                    attrs |= kCACyan;
                    attrs |= kCABold;
                }
                if(attrs) mattron(attrs);

                char* line = string_c_str(vector_item(lines, n));

                if(str_termlen(code, line) <= maxx) {
                    mmvprintw(y, 0, "%s", line);
                    y++;
                }
                else {
                    char* p = line;
                    while(p < line + strlen(line)) {
                        char one_line[BUFSIZ];
                        str_cut(code, p, maxx, one_line, BUFSIZ);
                        mmvprintw(y, 0, "%s", one_line);
                        y++;

                        p+=strlen(one_line);
                    }
                }

                if(attrs) {
                    mattroff();
                }

                n++;
            }
            mrefresh();

            /// インプット ///
/*
if(tcgetpgrp(0) != getpid()) {
    fprintf(stderr, "ERROR\n");
    exit(1);
}
*/
            int meta;
            int key = mgetch(&meta);

            if(key == 14 || key == KEY_DOWN) {
                (*cursor)++;
            }
            else if(key == 16 || key == KEY_UP) {
                (*cursor)--;
            }
            else if(key == 4 || key == KEY_NPAGE) {
                (*cursor) += (maxy / 2);
            }
            else if(key == 21 || key == KEY_PPAGE) {
                (*cursor) -= (maxy /2);
            }
            else if(key == 12) {
                mclear();
                mrefresh();
            }
            else if(key == 'q' || key == 3 | key == 27 || key == 7) {
                break;
            }
            else if(key == ' ' && multiple) {
                markfiles[*cursor] = !markfiles[*cursor];
                (*cursor)++;
            }
            else if(key == 10 || key == 13) {
                if(multiple) {
                    BOOL flg_mark = FALSE;
                    int k;
                    for(k=0; k<vector_size(lines); k++) {
                        if(markfiles[k]) {
                            string_push_back(result
                                , string_c_str(vector_item(lines, k)));

                            string_push_back(result, "\n");

                            flg_mark = TRUE;
                        }
                    }

                    if(flg_mark) {
                        string_chomp(result);
                    }
                    else {
                        string_put(result 
                            , string_c_str(vector_item(lines, *cursor)));
                    }
                }
                else {
                    string_put(result 
                        , string_c_str(vector_item(lines, *cursor)));
                }
                break;
            }

            /// 修正 ///
            if(*cursor < 0) {
                *cursor = 0;
            }
            if(*cursor >= vector_size(lines)) {
                *cursor = vector_size(lines)-1;
            }
            if(*cursor >= n) {
                *scrolltop = n;
            }
            if(*cursor < *scrolltop) {
                int i = *cursor;
                int width_sum = 0;
                while(width_sum < maxy) {
                    char* line = string_c_str(vector_item(lines, i));
                    int width = str_termlen(code, line) / maxx + 1;
                    width_sum += width;
                    i--;
                    if(i < 0) {
                        i = -2;
                        break;
                    }
                }
                
                *scrolltop = i +2;
            }
        }
    }

//mclear();
//mrefresh();
    mendwin();
#if !defined(__CYGWIN7__) && !defined(__FREEBSD__)
    mrestore_screen();
#endif
    mrestore_ttysettings();

    FREE(markfiles);

    int i;
    for(i=0; i<vector_size(lines); i++) {
        string_delete(vector_item(lines, i));
    }
    vector_delete(lines);
}

static double calculate_number(char** p) 
{
    double value = 0;

    BOOL minus;
    if(**p == '-') {
        (*p)++;
        minus = TRUE;

        while(**p == ' ' | **p == '\t') (*p)++;
    }
    else {
        minus = FALSE;
    }

    double keta = 1;
    char* tmp = *p;
    while(*tmp >= '0' && *tmp <= '9') {
        keta = keta * 10;
        tmp++;
    }

    keta = keta / 10;

    while(**p >= '0' && **p <= '9') {
        value = value + keta * (**p-'0');
        keta = keta / 10;
        (*p)++;
    }

    if(**p == '.') {
        (*p)++;

        while(**p >= '0' && **p <= '9') {
            value = value + keta * (**p-'0');
            keta = keta / 10;
            (*p)++;
        }
    }

    if(minus)
        return -value;
    else
        return value;
}

static double calculate_expr_add_sub(char** p, BOOL* result);

static double calculate_node(char** p, BOOL* result)
{
    while(**p == ' ' || **p == '\t') (*p)++;

    if(**p == '(') {
        (*p)++;
        
        double value = calculate_expr_add_sub(p, result);
        if(!*result) return 0;

        while(**p == ' ' || **p == '\t') (*p)++;
        
        if(**p == ')') {
            (*p)++;
            *result = TRUE;
        }
        else {
            err_msg("calc: need )");
            *result = FALSE;
        }

        return value;
    }
    else if(**p == '-' || **p >= '0' && **p <= '9') {
        *result = TRUE;
        return calculate_number(p);
    }
    else if(**p == 'p' && *(*p+1) == 'a' && *(*p+2) == 'i') {
        (*p)+=3;

        *result = TRUE;
        return 16*atan((double)1/5) - 4 * atan((double)1/239);
    }
    else if(**p == 'l' && *(*p+1) == 'o' && *(*p+2) == 'g') {
        (*p)+=3;

        while(**p == ' ' || **p == '\t') (*p)++;

        double value = calculate_expr_add_sub(p, result);
        if(!*result) return 0;

        return log(value);
    }
    else if(**p == 'i' && *(*p+1) == 'n' && *(*p+2) == 't') {
        (*p)+=3;

        while(**p == ' ' || **p == '\t') (*p)++;

        double value = calculate_expr_add_sub(p, result);
        if(!*result) return 0;

        return (int)(value);
    }
    else if(**p == 'c' && *(*p+1) == 'e' && *(*p+2) == 'i' && *(*p+3) == 'l') {
        (*p)+=4;

        while(**p == ' ' || **p == '\t') (*p)++;

        double value = calculate_expr_add_sub(p, result);
        if(!*result) return 0;

        return ceil(value);
    }
    else if(**p == 'f' && *(*p+1) == 'l' && *(*p+2) == 'o' && *(*p+3) == 'o'
            && *(*p+4) == 'r') 
    {
        (*p)+=5;

        while(**p == ' ' || **p == '\t') (*p)++;

        double value = calculate_expr_add_sub(p, result);
        if(!*result) return 0;

        return floor(value);
    }
    else if(**p == 'l' && *(*p+1) == 'o' && *(*p+2) == 'g' && *(*p+3) == '1'
            && *(*p+4) == '0') 
    {
        (*p)+=5;

        while(**p == ' ' || **p == '\t') (*p)++;

        double value = calculate_expr_add_sub(p, result);
        if(!*result) return 0;

        return log10(value);
    }
    else if(**p == 'a' && *(*p+1) == 'b' && *(*p+2) == 's') {
        (*p)+=3;

        while(**p == ' ' || **p == '\t') (*p)++;

        double value = calculate_expr_add_sub(p, result);
        if(!*result) return 0;

        return fabs(value);
    }
    else if(**p == 'e' && *(*p+1) == 'x' && *(*p+2) == 'p') {
        (*p)+=3;

        while(**p == ' ' || **p == '\t') (*p)++;

        double value = calculate_expr_add_sub(p, result);
        if(!*result) return 0;

        return exp(value);
    }
    else if(**p == 's' && *(*p+1) == 'i' && *(*p+2) == 'n') {
        (*p)+=3;

        while(**p == ' ' || **p == '\t') (*p)++;

        double value = calculate_expr_add_sub(p, result);
        if(!*result) return 0;

        return sin(value);
    }
    else if(**p == 'c' && *(*p+1) == 'o' && *(*p+2) == 's') {
        (*p)+=3;

        while(**p == ' ' || **p == '\t') (*p)++;

        double value = calculate_expr_add_sub(p, result);
        if(!*result) return 0;

        return cos(value);
    }
    else if(**p == 't' && *(*p+1) == 'a' && *(*p+2) == 'n') {
        (*p)+=3;

        while(**p == ' ' || **p == '\t') (*p)++;

        double value = calculate_expr_add_sub(p, result);
        if(!*result) return 0;

        return tan(value);
    }
    else if(**p == 'a' && *(*p+1) == 's' && *(*p+2) == 'i' && *(*p+3) == 'n') {
        (*p)+=4;

        while(**p == ' ' || **p == '\t') (*p)++;

        double value = calculate_expr_add_sub(p, result);
        if(!*result) return 0;

        return asin(value);
    }
    else if(**p == 'a' && *(*p+1) == 'c' && *(*p+2) == 'o' && *(*p+3) == 's') {
        (*p)+=4;

        while(**p == ' ' || **p == '\t') (*p)++;

        double value = calculate_expr_add_sub(p, result);
        if(!*result) return 0;

        return acos(value);
    }
    else if(**p == 'a' && *(*p+1) == 't' && *(*p+2) == 'a' && *(*p+3) == 'n') {
        (*p)+=4;

        while(**p == ' ' || **p == '\t') (*p)++;

        double value = calculate_expr_add_sub(p, result);
        if(!*result) return 0;

        return atan(value);
    }
    else if(**p == 's' && *(*p+1) == 'i' && *(*p+2) == 'n' && *(*p+3) == 'h') {
        (*p)+=4;

        while(**p == ' ' || **p == '\t') (*p)++;

        double value = calculate_expr_add_sub(p, result);
        if(!*result) return 0;

        return sinh(value);
    }
    else if(**p == 'c' && *(*p+1) == 'o' && *(*p+2) == 's' && *(*p+3) == 'h')
    {
        (*p)+=4;

        while(**p == ' ' || **p == '\t') (*p)++;

        double value = calculate_expr_add_sub(p, result);
        if(!*result) return 0;

        return cosh(value);
    }
    else if(**p == 't' && *(*p+1) == 'a' && *(*p+2) == 'n' && *(*p+3) == 'h')
    {
        (*p)+=4;

        while(**p == ' ' || **p == '\t') (*p)++;

        double value = calculate_expr_add_sub(p, result);
        if(!*result) return 0;

        return tanh(value);
    }
    else if(**p == 's' && *(*p+1) == 'q' && *(*p+2) == 'r' && *(*p+3) == 't') {
        (*p)+=4;

        while(**p == ' ' || **p == '\t') (*p)++;

        double value = calculate_expr_add_sub(p, result);
        if(!*result) return 0;

        return sqrt(value);
    }
    else {
        char buf[64];
        sprintf(buf, "calc: invalid character %c", **p);
        err_msg(buf);
        *result = FALSE;
        return 0.0;
    }
}

static double calculate_expr_pow(char** p, BOOL* result)
{
    double value = calculate_node(p, result);
    if(!*result) return 0;

    while(1) {
        if(**p == '*' && *(*p+1) == '*') {
            (*p)+=2;
            value = pow(value, calculate_node(p, result));
            if(!*result) {
                return 0;
            }
        }
        else if(**p == ' ' || **p == '\t') {
            (*p)++;
        }
        else {
            break;
        }
    }
    
    return value;
}

static double calculate_expr_mul_div(char** p, BOOL* result)
{
    double value = calculate_expr_pow(p, result);
    if(!*result) return 0;

    while(1) {
        if(**p == '*') {
            (*p)++;
            value *= calculate_expr_pow(p, result);
            if(!*result) {
                return 0;
            }
        }
        else if(**p == '/') {
            (*p)++;
            double waruhou = calculate_expr_pow(p, result);
            if(waruhou == 0.0) {
                err_msg("calc: divide 0");
                *result = FALSE;
                return 0;
            }
            value /= waruhou;
            if(!*result) {
                return 0;
            }
        }
        else if(**p == '%') {
            (*p)++;
            double waruhou = calculate_expr_pow(p, result);
            if(waruhou == 0.0) {
                err_msg("calc: divide 0");
                *result = FALSE;
                return 0;
            }
            value = fmod(value, waruhou);
            if(!*result) {
                return 0;
            }
        }
        else if(**p == ' ' || **p == '\t') {
            (*p)++;
        }
        else {
            break;
        }
    }
    
    return value;
}

static double calculate_expr_add_sub(char** p, BOOL* result)
{
    double value = calculate_expr_mul_div(p, result);
    if(!*result) return 0;

    while(1) {
        if(**p == '+') {
            (*p)++;
            value += calculate_expr_mul_div(p, result);
            if(!*result) {
                return 0;
            }
        }
        else if(**p == '-') {
            (*p)++;
            value -= calculate_expr_mul_div(p, result);
            if(!*result) {
                return 0;
            }
        }
        else if(**p == ' ' || **p == '\t') {
            (*p)++;
        }
        else {
            break;
        }
    }
    
    return value;
}

static BOOL calculate(double* result, char** p) 
{
    BOOL err;
    *result = calculate_expr_add_sub(p, &err);

    if(**p != 0) {
        char buf[64];
        sprintf(buf, "calc: invalid character(%c)\n", **p);
        err_msg(buf);
        return FALSE;
    }

    return err;
}

// 出力をCTRL-Cで止めたいので、この関数を経由して書き込む
// 戻り値 FALSE --> CTRL-Cで止められた
// 戻り値 TRUE --> 正常終了
static BOOL statment_tree_internal_commands_write_nextout(int nextout, char* str)
{
    char* p = str;
    char* str_size = str + strlen(str);

    while(*p) {
        int size;
        if(str_size - p < 1024) {
            size = strlen(p);
        }
        else {
            size = 1024;
        }

        if(write(nextout, p, size) < 0) {
            if(errno == EINTR) {
                return FALSE;
            }
            else {
                perror("write");
                exit(1);
            }
        }

        if(gSaphireSigInt) {
            gSaphireSigInt = FALSE;
            return FALSE;
        }

        p+=size;
    }

    return TRUE;
}

// 出力をCTRL-Cで止めたいので、この関数を経由して読み込む
// 戻り値 FALSE --> CTRL-Cで止められた
// 戻り値 TRUE --> 正常終了
static BOOL statment_tree_internal_commands_read_nextin(int nextin, string_obj* str)
{
    while(1) {
        char buf[BUFSIZ+1];
        int r = read(nextin, buf, BUFSIZ);

        if(r < 0) {
            if(errno == EINTR) {
                return FALSE;
            }
            break;
        }
        if(r == 0) {
            break;
        }
        buf[r] = 0;

        if(gSaphireSigInt) {
            gSaphireSigInt = FALSE;
            return FALSE;
        }

        string_push_back(str, buf);
    }

    return TRUE;
}

// 出力をCTRL-Cで止めたいので、この関数を経由して読み込む
// 戻り値 -1 --> CTRL-Cで止められた
// 戻り値 0 --> 正常終了
// 戻り値 1 --> EOF
static int statment_tree_internal_commands_read_nextin_oneline(int nextin, string_obj* str, enum eLineField lf)
{
    char nextin_str[32];
    sprintf(nextin_str, "%d", nextin);
    string_obj* buffer = hash_item(gFBuffers, nextin_str);

    if(buffer == NULL) {
        buffer = STRING_NEW("");
        hash_put(gFBuffers, nextin_str, buffer);

        while(1) {
            char buf[BUFSIZ+1];

            int r = read(nextin, buf, BUFSIZ);

            if(r < 0) {
                return -1;
            }
            if(gSaphireSigInt) {
                gSaphireSigInt = FALSE;
                return -1;
            }
            buf[r] = 0;

            /// EOF ///
            if(r == 0) {
                break;
            }
            else {
                string_push_back(buffer, buf);
            }
        }
    }

    char *p;
    if(lf == kLF) {
        p = strstr(string_c_str(buffer), "\n"); 
    }
    else if(lf == kCRLF) {
        p = strstr(string_c_str(buffer), "\r\n"); 
    }
    else {
        p = strstr(string_c_str(buffer), "\r"); 
    }

    if(p) {
        int n;
        if(gScriptLineField == kCRLF) {
            n = (p - string_c_str(buffer)) + 2;
        }
        else {
            n = (p - string_c_str(buffer)) + 1;
        }
        string_push_back3(str, string_c_str(buffer), n);
        string_erase(buffer, 0, n);

        return 0;
    }
    else if(strcmp(string_c_str(buffer), "") == 0) {
        return 1;
    }
    else {
        string_push_back(str, string_c_str(buffer));

        string_put(buffer,"");

        return 0;
    }
}

static BOOL statment_tree_internal_commands_test(int* rcode, vector_obj* argv, string_obj* input)
{
    char* arg_last = string_c_str(
            vector_item(argv, vector_size(argv)-1));

    if(strcmp(arg_last, "]") != 0) {
        err_msg("test: missing ]");
        return FALSE;
    }
    else {
        BOOL reverse = FALSE;

        char* arg1;
        char* arg2;
        char* arg3;
        char* arg4;
        char* arg5;

        if(vector_size(argv) > 1) {
            arg1 = string_c_str(vector_item(argv, 1));
            if(strcmp(arg1, "!") == 0) {
                reverse = TRUE;
            }
        }

        if(reverse) {
            if(input) {
                if(vector_size(argv) == 4) {
                    arg1 = string_c_str(vector_item(argv, 2));
                    arg2 = string_c_str(input);
                }
                else if(vector_size(argv) == 5) {
                    arg1 = string_c_str(input);
                    arg2 = string_c_str(vector_item(argv, 2));
                    arg3 = string_c_str(vector_item(argv, 3));
                }
                else {
                    return TRUE;
                    /*
                    err_msg("test: invalid [ arguments 1");
                    return FALSE;
                    */
                }
            }
            else {
                if(vector_size(argv) == 5) {
                    arg1 = string_c_str(vector_item(argv, 2));
                    arg2 = string_c_str(vector_item(argv, 3));
                }
                else if(vector_size(argv) == 6) {
                    arg1 = string_c_str(vector_item(argv, 2));
                    arg2 = string_c_str(vector_item(argv, 3));
                    arg3 = string_c_str(vector_item(argv, 4));
                }
                else {
                    return TRUE;
                    /*
                    err_msg("test: invalid [ arguments 2");
                    return FALSE;
                    */
                }
            }
        }
        else {
            if(input) {
                if(vector_size(argv) == 3) {
                    arg2 = string_c_str(input);
                }
                else if(vector_size(argv) == 4) {
                    arg1 = string_c_str(input);
                    arg2 = string_c_str(vector_item(argv, 1));
                    arg3 = string_c_str(vector_item(argv, 2));
                }
                else {
                    return TRUE;
                    /*
                    char buf[32];
                    sprintf(buf, "test: invalid [ arguments 3");
                    err_msg(buf);
                    return FALSE;
                    */
                }
            }
            else {
                if(vector_size(argv) == 4) {
                    arg2 = string_c_str(vector_item(argv, 2));
                }
                else if(vector_size(argv) == 5) {
                    arg2 = string_c_str(vector_item(argv, 2));
                    arg3 = string_c_str(vector_item(argv, 3));
                }
                else {
                    return TRUE;
                    /*
                    char buf[32];
                    sprintf(buf, "test: invalid [ arguments 4");
                    err_msg(buf);
                    return FALSE;
                    */
                }
            }
        }
        if(!reverse && (vector_size(argv) + (input ? 1: 0)) == 4 
            || reverse && (vector_size(argv) + (input ? 1:0)) == 5) 
        {
            /// string ///
            if(strcmp(arg1, "-n") == 0) {
                if(!reverse && strcmp(arg2, "") != 0) {
                    *rcode = 0;
                }
                else if(reverse && ! strcmp(arg2, "") != 0) {
                    *rcode = 0;
                }
            }
            else if(strcmp(arg1, "-z") == 0) {
                if(!reverse && strcmp(arg2, "") == 0) {
                    *rcode = 0;
                }
                else if(reverse && ! strcmp(arg2, "") == 0) {
                    *rcode = 0;
                }
            }
            
            else {

                /// file type ///
                struct stat stat_;
                if(stat(arg2, &stat_) == 0) {
                    if(strcmp(arg1, "-b") == 0) {
                        if(!reverse && S_ISBLK(stat_.st_mode)) {
                            *rcode = 0;
                        }
                        else if(reverse && ! S_ISBLK(stat_.st_mode)) 
                        {
                            *rcode = 0;
                        }
                    }
                    else if(strcmp(arg1, "-c") == 0) {
                        if(!reverse && S_ISCHR(stat_.st_mode)) {
                            *rcode = 0;
                        }
                        else if(reverse 
                            && ! S_ISCHR(stat_.st_mode)) 
                        {
                            *rcode = 0;
                        }
                    }
                    else if(strcmp(arg1, "-d") == 0) {
                        if(!reverse && S_ISDIR(stat_.st_mode)) {
                            *rcode = 0;
                        }
                        else if(reverse && ! S_ISDIR(stat_.st_mode)) 
                        {
                            *rcode = 0;
                        }
                    }
                    else if(strcmp(arg1, "-f") == 0) {
                        if(!reverse && S_ISREG(stat_.st_mode)) {
                            *rcode = 0;
                        }
                        else if(reverse 
                            && ! S_ISREG(stat_.st_mode)) 
                        {
                            *rcode = 0;
                        }
                    }
                    else if(strcmp(arg1, "-h") == 0 
                             || strcmp(arg1, "-L") == 0) 
                    {
                        if(!reverse && S_ISLNK(stat_.st_mode)) {
                            *rcode = 0;
                        }
                        else if(reverse 
                            && ! S_ISLNK(stat_.st_mode)) 
                        {
                            *rcode = 0;
                        }
                    }
                    else if(strcmp(arg1, "-p") == 0) {
                        if(!reverse && S_ISFIFO(stat_.st_mode)) {
                            *rcode = 0;
                        }
                        else if(reverse 
                            && ! S_ISFIFO(stat_.st_mode)) 
                        {
                            *rcode = 0;
                        }
                    }
                    else if(strcmp(arg1, "-S") == 0) {
                        if(!reverse && S_ISSOCK(stat_.st_mode)) {
                            *rcode = 0;
                        }
                        else if(reverse 
                            && S_ISSOCK(stat_.st_mode)) 
                        {
                            *rcode = 0;
                        }
                    }
                    else if(strcmp(arg1, "-t") == 0) {
                    }
                    /// permission ///
                    else if(strcmp(arg1, "-g") == 0) {
                        if(!reverse && (stat_.st_mode & S_ISGID)) {
                            *rcode = 0;
                        }
                        else if(reverse 
                            && !(stat_.st_mode & S_ISGID)) 
                        {
                            *rcode = 0;
                        }
                    }
                    else if(strcmp(arg1, "-k") == 0) {
#if defined(S_ISTXT)
                        if(!reverse && (stat_.st_mode & S_ISTXT)) {
                            *rcode = 0;
                        }
#endif
#if defined(S_ISVTX)
                        if(!reverse && (stat_.st_mode & S_ISVTX)) {
                            *rcode = 0;
                        }
#endif
#if defined(S_ISTXT)
                        else if(reverse && !(stat_.st_mode & S_ISTXT)) {
                            *rcode = 0;
                        }
#endif
#if defined(S_ISVTX)
                        else if(reverse && !(stat_.st_mode & S_ISVTX)) {
                            *rcode = 0;
                        }
#endif
                    }
                    else if(strcmp(arg1, "-u") == 0) {
                        if(!reverse 
                            && (stat_.st_mode & S_ISUID)) 
                        {
                            *rcode = 0;
                        }
                        else if(reverse 
                          && ! (stat_.st_mode & S_ISUID)) 
                        {
                            *rcode = 0;
                        }
                    }
                    else if(strcmp(arg1, "-r") == 0) {
                        if(!reverse 
                            && access(arg2, R_OK) == 0) 
                        {
                            *rcode = 0;
                        }
                        else if(reverse 
                            && !(access(arg2, R_OK) == 0)) 
                        {
                            *rcode = 0;
                        }
                    }
                    else if(strcmp(arg1, "-w") == 0) {
                        if(!reverse 
                            && access(arg2, W_OK) == 0) 
                        {
                            *rcode = 0;
                        }
                        else if(reverse 
                          && ! ( access(arg2, W_OK) == 0)) 
                        {
                            *rcode = 0;
                        }
                    }
                    else if(strcmp(arg1, "-x") == 0) {
                        if(!reverse && access(arg2, X_OK) == 0) {
                            *rcode = 0;
                        }
                        else if(reverse 
                            && !(access(arg2, X_OK) == 0)) 
                        {
                            *rcode = 0;
                        }
                    }
                    else if(strcmp(arg1, "-O") == 0) {
                        if(!reverse && stat_.st_uid == getuid()) {
                            *rcode = 0;
                        }
                        else if(reverse 
                            && !stat_.st_uid == getuid()) 
                        {
                            *rcode = 0;
                        }
                    }
                    else if(strcmp(arg1, "-G") == 0) {
                        if(!reverse && stat_.st_gid == getgid()) {
                            *rcode = 0;
                        }
                        else if(reverse 
                            && ! stat_.st_gid == getgid()) 
                        {
                            *rcode = 0;
                        }
                    }
                    else if(strcmp(arg1, "-e") == 0) {
                        if(!reverse) *rcode = 0;
                    }
                    else if(strcmp(arg1, "-s") == 0) {
                        if(!reverse && stat_.st_size > 0) {
                            *rcode = 0;
                        }
                        else if(reverse && !(stat_.st_size > 0)) {
                            *rcode = 0;
                        }
                    }
                    else {
                        return TRUE;
                        /*
                        err_msg("test: invalid [ arguments 5");
                        return FALSE;
                        */
                    }
                }
                else {
                    if(reverse) *rcode = 0;
                }
            }
        }
        else if(!reverse && (vector_size(argv) + (input ? 1:0)) == 5 
            || reverse && (vector_size(argv) + (input ? 1:0)) == 6) 
        {
            /// string ///
            if(strcmp(arg2, "=") == 0) {
                if(!reverse && strcmp(arg1, arg3) == 0) {
                    *rcode = 0;
                }
                else if(reverse && !(strcmp(arg1, arg3) == 0)) {
                    *rcode = 0;
                }
            }
            else if(strcmp(arg2, "!=") == 0) {
                if(!reverse && strcmp(arg1, arg3) != 0) {
                    *rcode = 0;
                }
                else if(reverse && !(strcmp(arg1, arg3) != 0)) {
                    *rcode = 0;
                }
            }
            else if(strcmp(arg2, "-re") == 0) {
                char* target = arg1;
                char* regex = arg3;

                int k;
                for(k=1; k<10; k++) {
                    char name[3];
                    sprintf(name, "%d", k);
                    delete_local_var(name);
                }
                delete_local_var("PREMATCH");
                delete_local_var("MATCH");
                delete_local_var("POSTMATCH");

                char* regex2 = regex;
                
                regex_t* reg = hash_item(gRegexs, regex);
                int r;

                if(reg == NULL) {
                    OnigErrorInfo err_info;
                    r = onig_new(&reg, regex2
                            , regex2 + strlen((char*)regex2)
                            , ONIG_OPTION_DEFAULT
                            , ONIG_ENCODING_UTF8
                            //, ONIG_SYNTAX_RUBY
                            //, ONIG_SYNTAX_PERL
                            , ONIG_SYNTAX_DEFAULT
                            , &err_info);

                    if(r == ONIG_NORMAL) {
                        hash_put(gRegexs, regex, reg);
                    }
                    else {
                        onig_free(reg);
                    }
                }
                else {
                    r = ONIG_NORMAL;
                }

                if(r == ONIG_NORMAL) {
                    OnigRegion* region = onig_region_new();
                    int r2 = onig_search(reg, target
                       , target + strlen(target)
                       , target, target + strlen(target)
                       , region, ONIG_OPTION_NONE);

                    if(region->num_regs > 0) {
                        if(region->beg[0] > 0) {
                            /// マッチした文字列の前 ///
                            char* tmp = MALLOC(region->beg[0] + 1);

                            memcpy(tmp, target, region->beg[0]);
                            tmp[region->beg[0]] = 0;

                            kitutuki_set_local_var("PREMATCH", tmp);

                            FREE(tmp);
                        }
                        else {
                            kitutuki_set_local_var("PREMATCH", "");
                        }

                        /// マッチした文字列 ///
                        char* tmp = MALLOC(
                                region->end[0]-region->beg[0] + 1);

                        memcpy(tmp, target + region->beg[0]
                         , region->end[0]-region->beg[0]);

                        tmp[region->end[0]
                            - region->beg[0]] = 0;

                        kitutuki_set_local_var("MATCH", tmp);

                        FREE(tmp);

                        /// マッチした文字列の後 ///
                        const int n = strlen(target)-region->end[0];
                        if(n > 0) {
                            char* tmp = MALLOC(n + 1);

                            memcpy(tmp, target + region->end[0], n);

                            tmp[n] = 0;

                            kitutuki_set_local_var("POSTMATCH", tmp);

                            FREE(tmp);
                        }
                        else {
                            kitutuki_set_local_var("POSTMATCH", "");
                        }
                    }
                    if(r2 >= 0) {
                        int i;
                        for (i=1; i<region->num_regs; i++) {
                            char* tmp =
                                MALLOC(region->end[i]-region->beg[i]+1);

                            memcpy(tmp, target + region->beg[i]
                             , region->end[i]-region->beg[i]);

                            tmp[region->end[i]
                                - region->beg[i]] = 0;

                            char name[16];
                            sprintf(name, "%d", i);

                            kitutuki_set_local_var(name, tmp);

                            FREE(tmp);
                        }
                    }

                    if(r2 >= 0 && !reverse || r2 < 0 && reverse) {
                        *rcode = 0;
                    }

                    onig_region_free(region, 1);
                }
                else {
                    err_msg("test: invalid regex");

                    return FALSE;
                }
            }

            /// number ///
            else if(strcmp(arg2, "-eq") == 0) {
                int l = atoi(arg1);
                int r = atoi(arg3);

                if(!reverse && l == r) {
                    *rcode = 0;
                }
                else if(reverse && !(l == r)) {
                    *rcode = 0;
                }
            }
            else if(strcmp(arg2, "-ne") == 0) {
                int l = atoi(arg1);
                int r = atoi(arg3);

                if(!reverse && l != r) {
                    *rcode = 0;
                }
                else if(reverse && !(l != r)) {
                    *rcode = 0;
                }
            }
            else if(strcmp(arg2, "-lt") == 0) {
                int l = atoi(arg1);
                int r = atoi(arg3);

                if(!reverse && l < r) {
                    *rcode = 0;
                }
                else if(reverse && !(l < r)) {
                    *rcode = 0;
                }
            }
            else if(strcmp(arg2, "-le") == 0) {
                int l = atoi(arg1);
                int r = atoi(arg3);

                if(!reverse && l <= r) {
                    *rcode = 0;
                }
                else if(reverse && !(l <= r)) {
                    *rcode = 0;
                }
            }
            else if(strcmp(arg2, "-gt") == 0) {
                int l = atoi(arg1);
                int r = atoi(arg3);

                if(!reverse && l > r) {
                    *rcode = 0;
                }
                else if(reverse && !(l > r)) {
                    *rcode = 0;
                }
            }
            else if(strcmp(arg2, "-ge") == 0) {
                int l = atoi(arg1);
                int r = atoi(arg3);

                if(!reverse && l >= r) {
                    *rcode = 0;
                }
                else if(reverse && !(l >= r)) {
                    *rcode = 0;
                }
            }

            /// file ///
            else if(strcmp(arg2, "-nt") == 0) {
                struct stat lstat_;
                struct stat rstat_;

                if(lstat(arg1, &lstat_) == 0) {
                    if(lstat(arg3, &rstat_) == 0) {
                        if(!reverse 
                            && lstat_.st_mtime > rstat_.st_mtime) 
                        {
                            *rcode = 0;
                        }
                        else if(reverse 
                          && !(lstat_.st_mtime > rstat_.st_mtime)) 
                        {
                            *rcode = 0;
                        }
                    }
                }
            }
            else if(strcmp(arg2, "-ot") == 0) {
                struct stat lstat_;
                struct stat rstat_;

                if(lstat(arg1, &lstat_) == 0) {
                    if(lstat(arg3, &rstat_) == 0) {
                        if(!reverse 
                            && lstat_.st_mtime < rstat_.st_mtime) 
                        {
                            *rcode = 0;
                        }
                        else if(reverse 
                          && !(lstat_.st_mtime < rstat_.st_mtime)) 
                        {
                            *rcode = 0;
                        }
                    }
                }
            }
            else if(strcmp(arg2, "-ef") == 0) {
            }
            else {
                return TRUE;
                //err_msg("test: invalid [ arguments 6");
                //return FALSE;
            }
        }
    }

    return TRUE;
}

// p: 対象文字列文字列内のポインタ
// start:対象文字列の最初の位置 
// word: 検索文字列
static char* strstr_back(char* p, char* start, char* word)
{
//printf("p %s start %s word %s\n", p, start, word);
    int n = strlen(word);

    while(p >= start) {
        BOOL flg = TRUE;
        int i;
        for(i=-1; i>=-n; i--) {
            if(p[i] != word[n+i]) {
                flg = FALSE;
                break;
            }
        }

        if(flg) {
            return p -n;
        }
        else {
            p--;
        }
    }

    return NULL;
}

static BOOL statment_tree_internal_commands_match(char* str2
  , UChar* regex2, int* rcode, int nextout, BOOL print, BOOL line_field
  , BOOL line_num, int n, string_obj* field)
{
    int k;
    for(k=1; k<10; k++) {
        char name[3];
        sprintf(name, "%d", k);
        delete_local_var(name);
    }
    delete_local_var("PREMATCH");
    delete_local_var("MATCH");
    delete_local_var("POSTMATCH");

    regex_t* reg = hash_item(gRegexs, (char*)regex2);
    int r;

    if(reg == NULL) {
        OnigErrorInfo err_info;
        r = onig_new(&reg, regex2
                , regex2 + strlen((char*)regex2)
                , ONIG_OPTION_DEFAULT
                , ONIG_ENCODING_UTF8
                //, ONIG_SYNTAX_RUBY
                //, ONIG_SYNTAX_PERL_NG
                , ONIG_SYNTAX_DEFAULT
                , &err_info);

        if(r == ONIG_NORMAL) {
            hash_put(gRegexs, (char*)regex2, reg);
        }
        else {
            onig_free(reg);
        }
    }
    else {
        r = ONIG_NORMAL;
    }

    if(r == ONIG_NORMAL) {
        OnigRegion* region = onig_region_new();
        int r2 = onig_search(reg, str2
           , str2 + strlen(str2)
           , str2, str2 + strlen(str2)
           //, region, ONIG_OPTION_CAPTURE_GROUP);
           , region, ONIG_OPTION_NONE);

        if(r2 >= 0) {
            *rcode = 0;

            if(region->num_regs > 0) {
                /// マッチした文字列の前 ///
                if(region->beg[0] > 0) {
                    char* tmp = MALLOC(region->beg[0] + 1);

                    memcpy(tmp, str2, region->beg[0]);
                    tmp[region->beg[0]] = 0;

                    kitutuki_set_local_var("PREMATCH", tmp);

                    FREE(tmp);
                }
                else {
                    kitutuki_set_local_var("PREMATCH", "");
                }

                /// マッチした文字列 ///
                char* tmp = MALLOC(
                        region->end[0]-region->beg[0] + 1);

                memcpy(tmp, str2 + region->beg[0]
                 , region->end[0]-region->beg[0]);

                tmp[region->end[0]
                    - region->beg[0]] = 0;

                kitutuki_set_local_var("MATCH", tmp);

                FREE(tmp);

                /// マッチした文字列の後 ///
                const int n = strlen(str2)-region->end[0];
                if(n > 0) {
                    char* tmp = MALLOC(n + 1);

                    memcpy(tmp, str2 + region->end[0], n);

                    tmp[n] = 0;

                    kitutuki_set_local_var("POSTMATCH", tmp);

                    FREE(tmp);
                }
                else {
                    kitutuki_set_local_var("POSTMATCH", tmp);
                }
            }

            /// マッチしたグループ化文字列 ///
            int i;
            for (i=1; i<region->num_regs; i++) {
                char* tmp = MALLOC(
                    region->end[i]-region->beg[i] + 1);

                memcpy(tmp, str2 + region->beg[i]
                 , region->end[i]-region->beg[i]);

                tmp[region->end[i]
                    - region->beg[i]] = 0;

                char name[16];
                sprintf(name, "%d", i);

                kitutuki_set_local_var(name, tmp);

                FREE(tmp);
            }

            if(print) {
                if(region->num_regs == 1) {
                    if(line_num) {
                        char num[128];
                        sprintf(num, "%d:", n);
                        if(write(nextout, num, strlen(num)) < 0) {
                            perror("write");
                            exit(1);
                        }
                    }
                    
                    char* match = kitutuki_get_local_var("MATCH");

                    if(match) {
                        if(write(nextout, match, strlen(match)) < 0) {
                            perror("write");
                            exit(1);
                        }
                    }
                    if(line_field) {
                        if(write(nextout, "\n", 1) < 0) {
                            perror("write");
                            exit(1);
                        }
                    }
                }
                else if(region->num_regs > 1) {
                    int i;
                    for(i=1; i<region->num_regs; i++) {
                        char name[16];
                        sprintf(name, "%d", i);

                        char* match = kitutuki_get_local_var(name);

                        if(match) {
                            if(line_num) {
                                char num[128];
                                sprintf(num, "%d:", n);
                                if(write(nextout, num, strlen(num)) < 0) {
                                    perror("write");
                                    exit(1);
                                }
                            }

                            if(write(nextout, match, strlen(match)) < 0) {
                                perror("write");
                                exit(1);
                            }
                        }
                        if(i==region->num_regs-1) {
                            if(line_field) {
                                if(write(nextout, "\n", 1) < 0) {
                                    perror("write");
                                    exit(1);
                                }
                            }
                        }
                        else {
                            if(write(nextout, string_c_str(field)
                                    , string_length(field)) 
                                < 0) 
                            {
                                perror("write");
                                exit(1);
                            }
                        }
                    }
                }
            }
        }

        onig_region_free(region, 1);
    }
    else {
        err_msg("match: invalid regex");

        return FALSE;
    }

    return TRUE;
}

static BOOL statment_tree_internal_commands(sCommand* command, int* rcode, vector_obj* argv, int nextout, int nextin, int nexterr, int j, vector_obj* cprogs, char* title, BOOL input)
{
    /// go ///
    switch(command->mKind) {
        case kSubshell:
            if(vector_size(argv) == 2) {
                char* fname = MALLOC(sizeof(gFName) + 32);
                sprintf(fname, "%s %d: subshell", gFName, gLine);
                *rcode = kitutuki_shell(string_c_str(vector_item(argv, 1))
                    , fname, nextout, nextin, nexterr);
                FREE(fname);
                if(*rcode == -1) {
                    return FALSE;
                }
            }
            break;

        case kMSleep: {
            *rcode = 0;

            char c [] = {'.', 'o', 'O'};
            int n = atoi(string_c_str(vector_item(argv, 1)));
            int i;
            for(i=0; i < n*5; i++) {
                char buf[32];
                sprintf(buf, "%c\033[D", c[i % 3]);

                if(!statment_tree_internal_commands_write_nextout(nextout, buf)) 
                {
                    err_msg("signal intrrupt");
                    return FALSE;
                }

                usleep(200000);
            }

            }
            break;

        case kTrue: 
            *rcode = 0;
            break;

        case kFalse:
            break;

        case kTest: {
            string_obj* str = NULL;

            /// 標準入力で受け取る場合は第1引数に代入 ///
            if(input) {
                str = STRING_NEW("");
                if(!statment_tree_internal_commands_read_nextin(
                                    nextin, str))
                {
                    err_msg("intrrupt");
                    string_delete(str);
                    return FALSE;
                }
            }

            if(!statment_tree_internal_commands_test(rcode, argv, str)) {
                if(str) string_delete(str);
                return FALSE;
            }

            if(str) string_delete(str);
            }
            break;

        case kIndex: {
            BOOL quiet = FALSE;
            BOOL new_line = FALSE;
            BOOL flg_n = FALSE;
            int n;
            BOOL flg_c = FALSE;
            int c;
            enum eKanjiCode code = gKanjiCode;

            int l;
            for(l=0; l<vector_size(argv); l++) {
                char* arg = string_c_str(vector_item(argv, l));

                if(strcmp(arg, "-q") == 0) {
                    quiet = TRUE;
                    string_delete(vector_item(argv, l));
                    vector_erase(argv, l);
                    l--;
                    continue;
                }
                else if(strcmp(arg, "-l") == 0) {
                    new_line = TRUE;
                    string_delete(vector_item(argv, l));
                    vector_erase(argv, l);
                    l--;
                    continue;
                }
                else if(strcmp(arg, "-i") == 0) {
                    input = TRUE;
                    string_delete(vector_item(argv, l));
                    vector_erase(argv, l);
                    l--;
                    continue;
                }
                else if(strcmp(arg, "-s") == 0) {
                    code = kSjis;
                    string_delete(vector_item(argv, l));
                    vector_erase(argv, l);
                    l--;
                    continue;
                }
                else if(strcmp(arg, "-e") == 0) {
                    code = kEucjp;
                    string_delete(vector_item(argv, l));
                    vector_erase(argv, l);
                    l--;
                    continue;
                }
                else if(strcmp(arg, "-w") == 0) {
                    code = kUtf8;
                    string_delete(vector_item(argv, l));
                    vector_erase(argv, l);
                    l--;
                    continue;
                }
                else if(strcmp(arg, "-c") == 0 && l+1 < vector_size(argv)) {
                    flg_c = TRUE;
                    c = atoi(string_c_str(vector_item(argv, l+1)));
                    string_delete(vector_item(argv, l));
                    vector_erase(argv, l);
                    string_delete(vector_item(argv, l));
                    vector_erase(argv, l);
                    l--;
                    continue;
                }
                else if(strcmp(arg, "-n") == 0 && l+1 < vector_size(argv)) {
                    flg_n = TRUE;
                    n = atoi(string_c_str(vector_item(argv, l+1)));
                    string_delete(vector_item(argv, l));
                    vector_erase(argv, l);
                    string_delete(vector_item(argv, l));
                    vector_erase(argv, l);
                    l--;
                    continue;
                }
            }

            char* word;
            char* target;
            int start;
            int match_count;
            string_obj* str = STRING_NEW("");

            if(input) {
                if(vector_size(argv) != 2) {
                    err_msg("index -i string");
                    string_delete(str);
                    return FALSE;
                }

                /// 文字列 ///
                if(!statment_tree_internal_commands_read_nextin(nextin
                        , str))
                {
                    err_msg("interrupt");
                    string_delete(str);
                    return FALSE;
                }
                target = string_c_str(str);

                /// 検索文字列 ///
                word = string_c_str(vector_item(argv, 1));

                /// 検索文字列開始位置 ///
                if(flg_n) {
                    start = n;

                    if(start < 0) {
                        start = str_kanjilen(code, target) + start;
                    }

                    if(start < 0 || start >= str_kanjilen(
                        code, target)) 
                    {
                        err_msg("index -> invalid -n range");
                        string_delete(str);
                        return FALSE;
                    }
                }
                else {
                    start = 0;
                }

                /// 検索文字列の検索回数 ///
                if(flg_c) {
                    match_count = c;
                    if(match_count <= 0) {
                        err_msg("index -> invalid -c range");
                        string_delete(str);
                        return FALSE;
                    }
                }
                else {
                    match_count = 1;
                }
            }
            else {
                if(vector_size(argv) != 3) {
                    err_msg("index -> invalid option number");
                    string_delete(str);
                    return FALSE;
                }

                /// 検索文字列 ///
                word = string_c_str(vector_item(argv, 2));

                /// 文字列 ///
                target = string_c_str(vector_item(argv, 1));

                /// 検索文字列開始位置 ///
                if(flg_n) {
                    start = n;

                    if(start < 0) {
                        start = str_kanjilen(code, target) + start;
                    }

                    if(start < 0 || start >= str_kanjilen(
                            code, target)) 
                    {
                        err_msg("index -> invalid -n range");
                        string_delete(str);
                        return FALSE;
                    }
                }
                else {
                    start = 0;
                }

                /// 検索文字列の検索回数 ///
                if(flg_c) {
                    match_count = c;
                    if(match_count <= 0) {
                        err_msg("index -> invalid -c range");
                        string_delete(str);
                        return FALSE;
                    }
                }
                else {
                    match_count = 1;
                }
            }

            char* start_byte = str_kanjipos2pointer(code, target, start);
            int count = match_count;
            char* p = start_byte;
            char* result = NULL;
            while(p < start_byte + strlen(start_byte)) {
                result = strstr(p, word);
                if(result) {
                    count--;
                    if(count == 0) {
                        break;
                    }
                    p = result+strlen(word);
                }
                else {
                    break;
                }
            }

            char msg[64];
            if(result == NULL || count !=0) {
                sprintf(msg, "-1");
            }
            else {
                int c = str_pointer2kanjipos(code, target, result);
                sprintf(msg, "%d", c);
                *rcode = 0;
            }

            if(quiet == FALSE) {
                if(!statment_tree_internal_commands_write_nextout(nextout, msg))
                {
                    err_msg("signal intrrupt");
                    string_delete(str);
                    return FALSE;
                }

                if(new_line) {
                    if(!statment_tree_internal_commands_write_nextout(nextout, "\n"))
                    {
                        err_msg("signal intrrupt");
                        string_delete(str);
                        return FALSE;
                    }
                }
            }

            string_delete(str);
            }
            break;

        case kRIndex: {
            BOOL quiet = FALSE;
            BOOL new_line = FALSE;
            BOOL flg_n = FALSE;
            int n;
            BOOL flg_c = FALSE;
            int c;
            enum eKanjiCode code = gKanjiCode;

            int l;
            for(l=0; l<vector_size(argv); l++) {
                char* arg = string_c_str(vector_item(argv, l));

                if(strcmp(arg, "-q") == 0) {
                    quiet = TRUE;
                    string_delete(vector_item(argv, l));
                    vector_erase(argv, l);
                    l--;
                    continue;
                }
                else if(strcmp(arg, "-l") == 0) {
                    new_line = TRUE;
                    string_delete(vector_item(argv, l));
                    vector_erase(argv, l);
                    l--;
                    continue;
                }
                else if(strcmp(arg, "-i") == 0) {
                    input = TRUE;
                    string_delete(vector_item(argv, l));
                    vector_erase(argv, l);
                    l--;
                    continue;
                }
                else if(strcmp(arg, "-s") == 0) {
                    code = kSjis;
                    string_delete(vector_item(argv, l));
                    vector_erase(argv, l);
                    l--;
                    continue;
                }
                else if(strcmp(arg, "-e") == 0) {
                    code = kEucjp;
                    string_delete(vector_item(argv, l));
                    vector_erase(argv, l);
                    l--;
                    continue;
                }
                else if(strcmp(arg, "-w") == 0) {
                    code = kUtf8;
                    string_delete(vector_item(argv, l));
                    vector_erase(argv, l);
                    l--;
                    continue;
                }
                else if(strcmp(arg, "-c") == 0 && l+1 < vector_size(argv)) {
                    flg_c = TRUE;
                    c = atoi(string_c_str(vector_item(argv, l+1)));
                    string_delete(vector_item(argv, l));
                    vector_erase(argv, l);
                    string_delete(vector_item(argv, l));
                    vector_erase(argv, l);
                    l--;
                    continue;
                }
                else if(strcmp(arg, "-n") == 0 && l+1 < vector_size(argv)) {
                    flg_n = TRUE;
                    n = atoi(string_c_str(vector_item(argv, l+1)));
                    string_delete(vector_item(argv, l));
                    vector_erase(argv, l);
                    string_delete(vector_item(argv, l));
                    vector_erase(argv, l);
                    l--;
                    continue;
                }
            }

            char* word;
            char* target;
            int start;
            int match_count;
            string_obj* str = STRING_NEW("");

            if(input) {
                if(vector_size(argv) != 2) {
                    err_msg("index -i string");
                    string_delete(str);
                    return FALSE;
                }

                /// 文字列 ///
                if(!statment_tree_internal_commands_read_nextin(nextin
                    , str))
                {
                    err_msg("intterrupt");
                    string_delete(str);
                    return FALSE;
                }
                word = string_c_str(str);

                /// 検索文字列 ///
                target = string_c_str(vector_item(argv, 1));

                /// 検索文字列開始位置 ///
                if(flg_n) {
                    start = n;

                    if(start < 0) {
                        start = str_kanjilen(code, target) + start;
                    }

                    if(start < 0 || start >= str_kanjilen(code, target)) 
                    {
                        err_msg("index -> invalid -n range");
                        string_delete(str);
                        return FALSE;
                    }
                }
                else {
                    start = str_kanjilen(code, target) -1;
                }

                /// 検索文字列の検索回数 ///
                if(flg_c) {
                    match_count = c;
                    if(match_count <= 0) {
                        err_msg("index -> invalid -c range");
                        string_delete(str);
                        return FALSE;
                    }
                }
                else {
                    match_count = 1;
                }
            }
            else {
                if(vector_size(argv) != 3) {
                    err_msg("index -> invalid option number");
                    string_delete(str);
                    return FALSE;
                }

                /// 文字列 ///
                word = string_c_str(vector_item(argv, 2));

                /// 検索文字列 ///
                target = string_c_str(vector_item(argv, 1));

                /// 検索文字列開始位置 ///
                if(flg_n) {
                    start = n;

                    if(start < 0) {
                        start = str_kanjilen(code, target) + start;
                    }

                    if(start < 0 || start >= str_kanjilen(
                                code, target)) 
                    {
                        err_msg("index -> invalid -n range");
                        string_delete(str);
                        return FALSE;
                    }
                }
                else {
                    start = str_kanjilen(code, target) -1;
                }

                /// 検索文字列の検索回数 ///
                if(flg_c) {
                    match_count = c;
                    if(match_count <= 0) {
                        err_msg("index -> invalid -c range");
                        string_delete(str);
                        return FALSE;
                    }
                }
                else {
                    match_count = 1;
                }
            }
            
            int count = match_count;

            char* start_byte = str_kanjipos2pointer(code
                    , target, start+1);
            char* p = start_byte;
            char* result = NULL;
            while(p>=target) {
                result = strstr_back(p, target, word);

                if(result != NULL) {
                    count--;
                    if(count == 0) {
                        break;
                    }
                    p = result - 1;
                }
                else {
                    break;
                }
            }

            char msg[64];
            if(result == NULL || count !=0) {
                sprintf(msg, "-1");
            }
            else {
                int c = str_pointer2kanjipos(code, target, result);
                sprintf(msg, "%d", c);
                *rcode = 0;
            }

            if(quiet == FALSE) {
                if(!statment_tree_internal_commands_write_nextout(nextout, msg))
                {
                    err_msg("signal intrrupt");
                    string_delete(str);
                    return FALSE;
                }
                if(new_line) {
                   if(!statment_tree_internal_commands_write_nextout(nextout, "\n"))
                   {
                       err_msg("signal intrrupt");
                       string_delete(str);
                       return FALSE;
                   }
                }
            }

            string_delete(str);
            }
            break;

        case kLength: {
            int i;
            BOOL line_field = FALSE;
            enum eKanjiCode code = gKanjiCode;
            BOOL terminal = FALSE;
            for(i=0; i<vector_size(argv); i++) {
                char* arg = string_c_str(vector_item(argv,i));
                if(strcmp(arg, "-i") == 0) {
                    input = TRUE;
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
                else if(strcmp(arg, "-l") == 0) {
                    line_field = TRUE;
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
                else if(strcmp(arg, "-s") == 0) {
                    code = kSjis;
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
                else if(strcmp(arg, "-e") == 0) {
                    code = kEucjp;
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
                else if(strcmp(arg, "-w") == 0) {
                    code = kUtf8;
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
                else if(strcmp(arg, "-t") == 0) {
                    terminal = TRUE;
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
            }

            if(input) {
                string_obj* str = STRING_NEW("");
                if(!statment_tree_internal_commands_read_nextin(nextin
                        , str))
                {
                    err_msg("interrupt");
                    string_delete(str);
                    return FALSE;
                }
                else {
                    int len;
                    if(terminal)
                        len = str_termlen(code, string_c_str(str));
                    else
                        len = str_kanjilen(code, string_c_str(str));

                    char buf[128];
                    if(line_field)
                        sprintf(buf, "%d\n", len);
                    else
                        sprintf(buf, "%d", len);

                    if(!statment_tree_internal_commands_write_nextout(nextout,  buf))
                    {
                        err_msg("signal intrrupt");
                        string_delete(str);
                        return FALSE;
                    }
                    string_delete(str);

                    *rcode = 0;
                }
            }
            else {
                if(vector_size(argv) == 2) {
                    int len;
                    if(terminal)
                        len = str_termlen(code, string_c_str(vector_item(argv, 1)));
                    else
                        len = str_kanjilen(code, string_c_str(vector_item(argv, 1)));
                    char buf[128];

                    if(line_field)
                        sprintf(buf, "%d\n", len);
                    else
                        sprintf(buf, "%d", len);

                    if(!statment_tree_internal_commands_write_nextout(nextout, buf)) 
                    {
                        err_msg("signal intrrupt");
                        return FALSE;
                    }
                    *rcode = 0;
                }
            }
            }
            break;

        case kUc: {
            int i;
            BOOL line_field = FALSE;
            for(i=0; i<vector_size(argv); i++) {
                char* arg = string_c_str(vector_item(argv,i));
                if(strcmp(arg, "-i") == 0) {
                    input = TRUE;
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
                else if(strcmp(arg, "-l") == 0) {
                    line_field = TRUE;
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
            }

            if(input) {
                string_obj* str = STRING_NEW("");
                if(!statment_tree_internal_commands_read_nextin(nextin
                                    , str))
                {
                    err_msg("interrupt");
                    string_delete(str);
                    return FALSE;
                }
                else {
                    string_toupper(str);

                    if(!statment_tree_internal_commands_write_nextout(nextout, string_c_str(str))) 
                    {
                        err_msg("signal intrrupt");
                        string_delete(str);
                        return FALSE;
                    }

                    if(line_field) {
                        if(!statment_tree_internal_commands_write_nextout(nextout, "\n"))
                        {
                            err_msg("signal intrrupt");
                            string_delete(str);
                            return FALSE;
                        }
                    }

                    *rcode = 0;

                    string_delete(str);
                }
            }
            else {
                if(vector_size(argv) == 2) {
                    string_obj* str = STRING_NEW(
                        string_c_str(vector_item(argv, 1)));

                    string_toupper(str);

                    if(!statment_tree_internal_commands_write_nextout(nextout, string_c_str(str))) 
                    {
                        err_msg("signal intrrupt");
                        string_delete(str);
                        return FALSE;
                    }

                    if(line_field) {
                        if(!statment_tree_internal_commands_write_nextout(nextout, "\n")) 
                        {
                            err_msg("signal intrrupt");
                            string_delete(str);
                            return FALSE;
                        }
                    }

                    *rcode = 0;

                    string_delete(str);
                }
            }
            }
            break;

        case kLc: {
            int i;
            BOOL line_field = FALSE;
            for(i=0; i<vector_size(argv); i++) {
                char* arg = string_c_str(vector_item(argv,i));
                if(strcmp(arg, "-i") == 0) {
                    input = TRUE;
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
                else if(strcmp(arg, "-l") == 0) {
                    line_field = TRUE;
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
            }

            if(input) {
                string_obj* str = STRING_NEW("");
                if(!statment_tree_internal_commands_read_nextin(nextin
                                    , str))
                {
                    err_msg("interrupt");
                    string_delete(str);
                    return FALSE;
                }
                else {
                    string_tolower(str);

                    if(!statment_tree_internal_commands_write_nextout(nextout, string_c_str(str)))
                    {
                        err_msg("signal intrrupt");
                        string_delete(str);
                        return FALSE;
                    }

                    if(line_field) {
                        if(!statment_tree_internal_commands_write_nextout(nextout, "\n"))
                        {
                            err_msg("signal intrrupt");
                            string_delete(str);
                            return FALSE;
                        }
                    }

                    *rcode = 0;

                    string_delete(str);
                }
            }
            else {
                if(vector_size(argv) == 2) {
                    string_obj* str = STRING_NEW(
                        string_c_str(vector_item(argv, 1)));

                    string_tolower(str);

                    if(!statment_tree_internal_commands_write_nextout(nextout, string_c_str(str)))
                    {
                        string_delete(str);
                        err_msg("signal intrrupt");
                        return FALSE;
                    }

                    if(line_field) {
                        if(!statment_tree_internal_commands_write_nextout(nextout, "\n"))
                        {
                            err_msg("signal intrrupt");
                            return FALSE;
                        }
                    }

                    *rcode = 0;

                    string_delete(str);
                }
            }
            }
            break;

        case kChomp: {
            int i;
            for(i=0; i<vector_size(argv); i++) {
                char* arg = string_c_str(vector_item(argv,i));
                if(strcmp(arg, "-i") == 0) {
                    input = TRUE;
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
            }

            if(input) {
                string_obj* str = STRING_NEW("");
                if(!statment_tree_internal_commands_read_nextin(
                        nextin, str)) 
                {
                    err_msg("interrupt");
                    string_delete(str);
                    return FALSE;
                }
                else {
                    string_chomp(str);

                    if(!statment_tree_internal_commands_write_nextout(nextout, string_c_str(str)))
                    {
                        err_msg("signal intrrupt");
                        string_delete(str);
                        return FALSE;
                    }

                    *rcode = 0;

                    string_delete(str);
                }
            }
            }
            break;

        case kSubstr: {
            int i;
            BOOL line_field = FALSE;
            BOOL chomp = FALSE;
            enum eKanjiCode code = gKanjiCode;

            for(i=0; i<vector_size(argv); i++) {
                char* arg = string_c_str(vector_item(argv,i));
                if(strcmp(arg, "-l") == 0) {
                    line_field = TRUE;
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
                else if(strcmp(arg, "-c") == 0) {
                    chomp = TRUE;
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
                else if(strcmp(arg, "-s") == 0) {
                    code = kSjis;
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
                else if(strcmp(arg, "-e") == 0) {
                    code = kEucjp;
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
                else if(strcmp(arg, "-w") == 0) {
                    code = kUtf8;
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
            }

            string_obj* str = NULL;
            char* p;
            int index;
            int count = 1;
            if(input) {
                str = STRING_NEW("");
                if(!statment_tree_internal_commands_read_nextin(
                        nextin, str)) 
                {
                    err_msg("interrupt");
                    string_delete(str);
                    return FALSE;
                }

                if(chomp) string_chomp(str);
                p = string_c_str(str);

                if(vector_size(argv) < 2) {
                    err_msg("substr: invalid arguments");
                    string_delete(str);
                    return FALSE;
                }
                index = atoi(string_c_str(vector_item(argv, 1)));

                if(vector_size(argv) >= 3) {
                    count = atoi(string_c_str(vector_item(argv, 2)));
                }
            }
            else {
                if(vector_size(argv) < 3) {
                    err_msg("substr: invalid arguments");
                    return FALSE;
                }

                p = string_c_str(vector_item(argv, 1));
                index = atoi(string_c_str(vector_item(argv, 2)));

                if(vector_size(argv) >= 4) {
                    count = atoi(string_c_str(vector_item(argv, 3)));
                }
            }

            if(index < 0) index += str_kanjilen(code, p);
            if(index < 0) index = 0;
            if(index >= str_kanjilen(code, p)) 
                index = str_kanjilen(code, p)-1;

            if(count <= 0) {
                count = 1;
            }
            if(index + count > strlen(p)) {
                count = strlen(p)-index;
            }

            char* p1 = str_kanjipos2pointer(code, p, index);
            char* p2 = str_kanjipos2pointer(code, p, index + count);

            char* buf = MALLOC(p2 -p1 + 1);
            memcpy(buf, p1, p2-p1);
            buf[p2-p1] = 0;

            if(!statment_tree_internal_commands_write_nextout(nextout, buf))
            {
                err_msg("signal intrrupt");
                if(str)string_delete(str);
                FREE(buf);
                return FALSE;
            }
            if(line_field) {
                if(!statment_tree_internal_commands_write_nextout(nextout, "\n"))
                {
                    err_msg("signal intrrupt");
                    FREE(buf);
                    if(str)string_delete(str);
                    return FALSE;
                }
            }

            FREE(buf);

            if(str) string_delete(str);
            }
            break;

        case kEval: {
            if(vector_size(argv) == 2) {
                string_obj* cmdline = (string_obj*)vector_item(argv, 1);
/*
                /// 前の引数を保存 ///
                vector_obj* fun_argv_before = hash_item(gArrays, "ARGV");

                /// 引数を渡す ///
                int k;
                vector_obj* fun_argv = VECTOR_NEW(5);
                for(k=2; k<vector_size(argv); k++) {
                    char* value = string_c_str(vector_item(argv, k));
                    vector_add(fun_argv, STRING_NEW(value));
                }
                
                hash_put(gArrays, "ARGV", fun_argv);
                update_ary_env("ARGV");
*/
                char* fname = MALLOC(sizeof(gFName) + 32);
                sprintf(fname, "%s %d: eval", gFName, gLine);
                *rcode = kitutuki_shell(string_c_str(cmdline), fname
                                , nextout, nextin, nexterr);
                FREE(fname);
                if(*rcode == -1) {
/*
                    for(k=0; k<vector_size(fun_argv); k++) {
                        string_delete(vector_item(fun_argv, k));
                    }
                    vector_delete(fun_argv);
                    hash_erase(gArrays, "ARGV");
*/
                    return FALSE;
                }
/*
                /// 引数を消す ///
                for(k=0; k<vector_size(fun_argv); k++) {
                    string_delete(vector_item(fun_argv, k));
                }
                vector_delete(fun_argv);

                hash_erase(gArrays, "ARGV");

                /// 前の引数を元に戻す ///
                if(fun_argv_before) {
                    hash_put(gArrays, "ARGV", fun_argv_before);
                    update_ary_env("ARGV");
                }
*/
                if(*rcode < 0) {
                    return FALSE;
                }
            }
            }
            break;

        case kFg:
            if(gAppType != kATOptC && gJobControl) {
                if(vector_size(argv) >= 2) {
                    char* argv2 
                       = string_c_str((string_obj*)vector_item(argv, 1));
                    if(argv2[0] >= '1' && argv2[0] <= '9') {
                        int num = argv2[0] - '0';
                        if(!forground_job(num-1)) {
                            return FALSE;
                        }
                    }
                    else {
                        err_msg("invalid job number");
                        return FALSE;
                    }

                    *rcode = 0;
                }
                else {
                    if(!forground_job(0)) {
                        return FALSE;
                    }
                    *rcode = 0;
                }
            }
            break;

        case kBg:
            if(gAppType != kATOptC && gJobControl) {
                if(vector_size(argv) >= 2) {
                    char* argv2 = string_c_str((string_obj*)vector_item(argv, 1));
                    if(argv2[0] >= '1' && argv2[0] <= '9') {
                        int num = argv2[0] - '0';
                        background_job(num-1);
                    }

                    *rcode = 0;
                }
                else {
                    background_job(0);
                    *rcode = 0;
                }
            }
            break;

        case kCpg:
            if(vector_size(argv) == 3) {
                int job_id = atoi(string_c_str(vector_item(argv, 1))) -1;
                pid_t pgid = atoi(string_c_str(vector_item(argv, 2)));

                if(job_id >= 0 && job_id < vector_size(gJobs)) {
                    sJob* job = vector_item(gJobs, job_id);

                    job->mPGroup = pgid;
                }
            }
            break;

        case kJobs:
            {
            int i;
            if(!statment_tree_internal_commands_write_nextout(nextout, "number name pgroup\n")) 
            {
                err_msg("signal intrrupt");
                return FALSE;
            }
            for(i=0; i<vector_size(gJobs); i++) {
                sJob* job = vector_item(gJobs, i);
                char* str = MALLOC(string_length(job->mName) + 16);
                sprintf(str, "%d. %s %d\n", i+1, string_c_str(job->mName), job->mPGroup);
                if(!statment_tree_internal_commands_write_nextout(nextout, str)) 
                {
                    FREE(str);
                    return FALSE;
                }

                FREE(str);
            }
            }
            break;

        case kRehash:
            kitutuki_rehash();
            break;

        case kKanjiCode: {
            int l;
            BOOL quiet = FALSE;
            for(l=0; l<vector_size(argv); l++) {
                char* arg = string_c_str(vector_item(argv, l));

                if(strcmp(arg, "-s") == 0) {
                    *rcode = 0;
                    gKanjiCode = kSjis;
                    string_delete(vector_item(argv, l));
                    vector_erase(argv, l);
                    l--;
                    continue;
                }
                else if(strcmp(arg, "-e") == 0) {
                    *rcode = 0;
                    gKanjiCode = kEucjp;
                    string_delete(vector_item(argv, l));
                    vector_erase(argv, l);
                    l--;
                    continue;
                }
                else if(strcmp(arg, "-w") == 0) {
                    *rcode = 0;
                    gKanjiCode = kUtf8;
                    string_delete(vector_item(argv, l));
                    vector_erase(argv, l);
                    l--;
                    continue;
                }
                else if(strcmp(arg, "-q") == 0) {
                    quiet = TRUE;
                    string_delete(vector_item(argv, l));
                    vector_erase(argv, l);
                    l--;
                    continue;
                }
            }

            if(!quiet) {
                if(gKanjiCode == kSjis) {
                    if(!statment_tree_internal_commands_write_nextout(nextout, "sjis\n"))
                    {
                        err_msg("signal intrrupt");
                        return FALSE;
                    }
                }
                else if(gKanjiCode == kEucjp) {
                    if(!statment_tree_internal_commands_write_nextout(nextout, "eucjp\n")) 
                    {
                        err_msg("signal intrrupt");
                        return FALSE;
                    }
                }
                else {
                    if(!statment_tree_internal_commands_write_nextout(nextout, "utf8\n"))
                    {
                        err_msg("signal intrrupt");
                        return FALSE;
                    }
                }
                *rcode = 0;
            }

            }
            break;

        case kLineField: {
            int l;
            BOOL quiet = FALSE;
            for(l=0; l<vector_size(argv); l++) {
                char* arg = string_c_str(vector_item(argv, l));

                if(strcmp(arg, "-Lw") == 0) {
                    *rcode = 0;
                    gLineField = kCRLF;
                    string_delete(vector_item(argv, l));
                    vector_erase(argv, l);
                    l--;
                    continue;
                }
                else if(strcmp(arg, "-Lm") == 0) {
                    *rcode = 0;
                    gLineField = kCR;
                    string_delete(vector_item(argv, l));
                    vector_erase(argv, l);
                    l--;
                    continue;
                }
                else if(strcmp(arg, "-Lu") == 0) {
                    *rcode = 0;
                    gLineField = kLF;
                    string_delete(vector_item(argv, l));
                    vector_erase(argv, l);
                    l--;
                    continue;
                }
                else if(strcmp(arg, "-q") == 0) {
                    quiet = TRUE;
                    string_delete(vector_item(argv, l));
                    vector_erase(argv, l);
                    l--;
                    continue;
                }
            }

            if(!quiet) {
                if(gLineField == kLF) {
                    if(!statment_tree_internal_commands_write_nextout(nextout, "LF\n"))
                    {
                        err_msg("signal intrrupt");
                        return FALSE;
                    }
                }
                else if(gLineField == kCRLF) {
                    if(!statment_tree_internal_commands_write_nextout(nextout, "CRLF\n"))
                    {
                        err_msg("signal intrrupt");
                        return FALSE;
                    }
                }
                else {
                    if(!statment_tree_internal_commands_write_nextout(nextout, "CR\n"))
                    {
                        err_msg("signal intrrupt");
                        return FALSE;
                    }
                }
                *rcode = 0;
            }
            }
            break;

        case kVar:
            {
            int i;
            BOOL all = FALSE;
            BOOL chomp = TRUE;
            BOOL print = FALSE;
            BOOL line_field = FALSE;
            BOOL field = FALSE;
            string_obj* field_str = NULL;
            for(i=0; i<vector_size(argv); i++) {
                char* arg = string_c_str(vector_item(argv,i));
                if(strcmp(arg, "-a") == 0) {
                    all = TRUE;
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
                else if(strcmp(arg, "-f") == 0 && i+1<vector_size(argv))
                {
                    field = TRUE;
                    field_str 
                      = STRING_NEW(string_c_str(vector_item(argv, i+1)));
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
                else if(strcmp(arg, "-nc") == 0) {
                    chomp = FALSE;
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
                else if(strcmp(arg, "-i") == 0) {
                    input = TRUE;
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
                else if(strcmp(arg, "-p") == 0) {
                    print = TRUE;
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
                else if(strcmp(arg, "-l") == 0) {
                    line_field = TRUE;
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
            }

            if(all) {
                if(vector_size(argv) == 2) {
                    char* p = string_c_str((string_obj*)vector_item(argv, 1));
                    string_obj* str = STRING_NEW("");
                    if(!statment_tree_internal_commands_read_nextin(
                                        nextin, str))
                    {
                        err_msg("interrupt");
                        string_delete(str);
                        return FALSE;
                    }

                    hash_obj* stack = vector_item(gStackFrame
                        , vector_size(gStackFrame)-1);

                    string_obj* var = hash_item(stack, p);

                    if(var) {
                        string_put(var, string_c_str(str));
                        string_delete(str);
                    }
                    else {
                        hash_put(stack, p, str);
                    }

                    if(print) {
                        if(!statment_tree_internal_commands_write_nextout(nextout, string_c_str(str)))
                        {
                            err_msg("signal intrrupt");
                            return FALSE;
                        }
                        if(line_field) {
                            if(!statment_tree_internal_commands_write_nextout(nextout, "\n"))
                            {
                                err_msg("signal intrrupt");
                                return FALSE;
                            }
                        }
                    }

                    *rcode = 0;
                }
                else {
                    err_msg("var -a must have one var_name");
                    return FALSE;
                }
            }
            else if(input) {
                int k;
                for(k=1; k<vector_size(argv); k++) {
                    char* item = string_c_str(vector_item(argv, k));

                    string_obj* str = STRING_NEW("");
                    int result = statment_tree_internal_commands_read_nextin_oneline(nextin, str, gLineField);
                    if(result == -1) {
                        err_msg("interrupt");
                        string_delete(str);
                        return FALSE;
                    }
                    
                    if(chomp) string_chomp(str);

                    hash_obj* stack = vector_item(gStackFrame
                        , vector_size(gStackFrame)-1);

                    string_obj* var = hash_item(stack, item);

                    if(var) {
                        string_put(var, string_c_str(str));
                        string_delete(str);
                    }
                    else {
                        hash_put(stack, item, str);
                    }
                            
                    if(print) {
                        if(!statment_tree_internal_commands_write_nextout(nextout, string_c_str(str)))
                        {
                            err_msg("signal intrrupt");
                            return FALSE;
                        }
                        if(line_field) {
                            if(!statment_tree_internal_commands_write_nextout(nextout, "\n")) 
                            {
                                err_msg("signal intrrupt");
                                return FALSE;
                            }
                        }
                    }

                    *rcode = 0;
                }
            }
            else {
                int i;
                for(i=1; i<vector_size(argv); i++) {
                    char* p = string_c_str((string_obj*)vector_item(argv, i));
                    char* p2 = strstr(p, "=");

                    if(p2) {
                        char* var_name = MALLOC(p2 - p + 1);
                        string_obj* value = STRING_NEW(p2 + 1);

                        memcpy(var_name, p, p2 - p);
                        var_name[p2 - p] = 0;

                        hash_obj* stack = vector_item(gStackFrame
                            , vector_size(gStackFrame)-1);

                        string_obj* var = hash_item(stack, var_name);

                        if(var) {
                            string_put(var, string_c_str(value));
                            string_delete(value);
                        }
                        else {
                            hash_put(stack, var_name, value);
                        }

                        *rcode = 0;

                        if(print) {
                            if(!statment_tree_internal_commands_write_nextout(nextout, string_c_str(value))) 
                            {
                                err_msg("signal intrrupt");
                                FREE(var_name);
                                return FALSE;
                            }
                            if(line_field) {
                                if(!statment_tree_internal_commands_write_nextout(nextout, "\n"))
                                {
                                    err_msg("signal intrrupt");
                                    FREE(var_name);
                                    return FALSE;
                                }
                            }
                        }

                        FREE(var_name);
                    }
                    else {
                        hash_obj* stack = vector_item(gStackFrame
                            , vector_size(gStackFrame)-1);

                        string_obj* var = hash_item(stack, p);

                        if(var == NULL) {
                            hash_put(stack, p, STRING_NEW(""));
                            *rcode = 0;
                        }

                        if(print) {
                            if(line_field) {
                                if(!statment_tree_internal_commands_write_nextout(nextout, "\n"))
                                {
                                    err_msg("signal intrrupt");
                                    return FALSE;
                                }
                            }
                        }
                    }
                }
            }
            }
            break;

        case kGlobal:
            {
            int i;
            BOOL all = FALSE;
            BOOL chomp = TRUE;
            BOOL print = FALSE;
            BOOL line_field = FALSE;
            BOOL field = FALSE;
            string_obj* field_str = NULL;
            for(i=0; i<vector_size(argv); i++) {
                char* arg = string_c_str(vector_item(argv,i));
                if(strcmp(arg, "-a") == 0) {
                    all = TRUE;
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
                else if(strcmp(arg, "-f") == 0 && i+1<vector_size(argv))
                {
                    field = TRUE;
                    field_str 
                      = STRING_NEW(string_c_str(vector_item(argv, i+1)));
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
                else if(strcmp(arg, "-nc") == 0) {
                    chomp = FALSE;
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
                else if(strcmp(arg, "-i") == 0) {
                    input = TRUE;
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
                else if(strcmp(arg, "-p") == 0) {
                    print = TRUE;
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
                else if(strcmp(arg, "-l") == 0) {
                    line_field = TRUE;
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
            }

            if(all) {
                if(vector_size(argv) == 2) {
                    char* p = string_c_str((string_obj*)vector_item(argv, 1));
                    string_obj* str = STRING_NEW("");
                    if(!statment_tree_internal_commands_read_nextin(
                                        nextin, str))
                    {
                        err_msg("interrupt");
                        string_delete(str);
                        return FALSE;
                    }

                    string_obj* var = hash_item(gGlobals, p);
                    if(var) {
                        string_put(var, string_c_str(str));
                        string_delete(str);
                    }
                    else {
                        hash_put(gGlobals, p, str);
                    }

                    if(print) {
                        if(!statment_tree_internal_commands_write_nextout(nextout, string_c_str(str)))
                        {
                            err_msg("signal intrrupt");
                            return FALSE;
                        }
                        if(line_field) {
                            if(!statment_tree_internal_commands_write_nextout(nextout, "\n"))
                            {
                                err_msg("signal intrrupt");
                                return FALSE;
                            }
                        }
                    }

                    *rcode = 0;
                }
                else {
                    err_msg("var -a must have one var_name");
                    return FALSE;
                }
            }
            else if(input) {
                int k;
                for(k=1; k<vector_size(argv); k++) {
                    char* item = string_c_str(vector_item(argv, k));

                    string_obj* str = STRING_NEW("");
                    int result = statment_tree_internal_commands_read_nextin_oneline(nextin, str, gLineField);
                    if(result == -1) {
                        err_msg("interrupt");
                        string_delete(str);
                        return FALSE;
                    }
                    
                    if(chomp) string_chomp(str);

                    string_obj* var = hash_item(gGlobals, item);
                    if(var) {
                        string_put(var, string_c_str(str));
                        string_delete(str);
                    }
                    else {
                        hash_put(gGlobals, item, str);
                    }

                    if(print) {
                        if(!statment_tree_internal_commands_write_nextout(nextout, string_c_str(str)))
                        {
                            err_msg("signal intrrupt");
                            return FALSE;
                        }
                        if(line_field) {
                            if(!statment_tree_internal_commands_write_nextout(nextout, "\n")) 
                            {
                                err_msg("signal intrrupt");
                                return FALSE;
                            }
                        }
                    }

                    *rcode = 0;
                }
            }
            else {
                int i;
                for(i=1; i<vector_size(argv); i++) {
                    char* p = string_c_str((string_obj*)vector_item(argv, i));
                    char* p2 = strstr(p, "=");

                    if(p2) {
                        char* var_name = MALLOC(p2 - p + 1);
                        string_obj* value = STRING_NEW(p2 + 1);

                        memcpy(var_name, p, p2 - p);
                        var_name[p2 - p] = 0;

                        /// グローバル変数に変数があるなら ///
                        string_obj* var= hash_item(gGlobals, var_name);

                        if(var) {
                            string_put(var, string_c_str(value));
                            string_delete(value);
                        }
                        else {
                            hash_put(gGlobals, var_name, value);
                        }

                        *rcode = 0;

                        if(print) {
                            if(!statment_tree_internal_commands_write_nextout(nextout, string_c_str(value))) 
                            {
                                err_msg("signal intrrupt");
                                FREE(var_name);
                                return FALSE;
                            }
                            if(line_field) {
                                if(!statment_tree_internal_commands_write_nextout(nextout, "\n"))
                                {
                                    err_msg("signal intrrupt");
                                    FREE(var_name);
                                    return FALSE;
                                }
                            }
                        }

                        FREE(var_name);
                    }
                    else {
                        string_obj* var = hash_item(gGlobals, p);
                        if(var == NULL) {
                            hash_put(gGlobals, p, STRING_NEW(""));
                            *rcode = 0;
                        }

                        if(print) {
                            if(line_field) {
                                if(!statment_tree_internal_commands_write_nextout(nextout, "\n"))
                                {
                                    err_msg("signal intrrupt");
                                    return FALSE;
                                }
                            }
                        }
                    }
                }
            }
            }
            break;

        case kExport:
            {
            int i;
            BOOL all = FALSE;
            BOOL chomp = TRUE;
            BOOL print = FALSE;
            BOOL line_field = FALSE;
            BOOL field = FALSE;
            string_obj* field_str = NULL;
            for(i=0; i<vector_size(argv); i++) {
                char* arg = string_c_str(vector_item(argv,i));
                if(strcmp(arg, "-a") == 0) {
                    all = TRUE;
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
                else if(strcmp(arg, "-f") == 0 && i+1<vector_size(argv))
                {
                    field = TRUE;
                    field_str 
                      = STRING_NEW(string_c_str(vector_item(argv, i+1)));
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
                else if(strcmp(arg, "-nc") == 0) {
                    chomp = FALSE;
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
                else if(strcmp(arg, "-i") == 0) {
                    input = TRUE;
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
                else if(strcmp(arg, "-p") == 0) {
                    print = TRUE;
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
                else if(strcmp(arg, "-l") == 0) {
                    line_field = TRUE;
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
            }

            if(all) {
                if(vector_size(argv) == 2) {
                    char* p = string_c_str((string_obj*)vector_item(argv, 1));
                    string_obj* str = STRING_NEW("");
                    if(!statment_tree_internal_commands_read_nextin(
                                        nextin, str))
                    {
                        err_msg("interrupt");
                        string_delete(str);
                        return FALSE;
                    }

                    setenv(p, string_c_str(str), 1);

                    if(print) {
                        if(!statment_tree_internal_commands_write_nextout(nextout, string_c_str(str)))
                        {
                            err_msg("signal intrrupt");
                            return FALSE;
                        }
                        if(line_field) {
                            if(!statment_tree_internal_commands_write_nextout(nextout, "\n"))
                            {
                                err_msg("signal intrrupt");
                                return FALSE;
                            }
                        }
                    }

                    *rcode = 0;
                }
                else {
                    err_msg("export -a must have one var_name");
                    return FALSE;
                }
            }
            else if(input) {
                int k;
                for(k=1; k<vector_size(argv); k++) {
                    char* item = string_c_str(vector_item(argv, k));

                    string_obj* str = STRING_NEW("");
                    int result = statment_tree_internal_commands_read_nextin_oneline(nextin, str, gLineField);
                    if(result == -1) {
                        err_msg("interrupt");
                        string_delete(str);
                        return FALSE;
                    }
                    
                    if(chomp) string_chomp(str);

                    setenv(item, string_c_str(str), 1);

                    if(print) {
                        if(!statment_tree_internal_commands_write_nextout(nextout, string_c_str(str)))
                        {
                            err_msg("signal intrrupt");
                            return FALSE;
                        }
                        if(line_field) {
                            if(!statment_tree_internal_commands_write_nextout(nextout, "\n")) 
                            {
                                err_msg("signal intrrupt");
                                return FALSE;
                            }
                        }
                    }

                    *rcode = 0;

                    string_delete(str);
                }
            }
            else {
                int i;
                for(i=1; i<vector_size(argv); i++) {
                    char* p = string_c_str((string_obj*)vector_item(argv, i));
                    char* p2 = strstr(p, "=");

                    if(p2) {
                        char* var_name = MALLOC(p2 - p + 1);
                        string_obj* value = STRING_NEW(p2 + 1);

                        memcpy(var_name, p, p2 - p);
                        var_name[p2 - p] = 0;
    
                        setenv(var_name, string_c_str(value), 1);

                        *rcode = 0;

                        if(print) {
                            if(!statment_tree_internal_commands_write_nextout(nextout, string_c_str(value))) 
                            {
                                err_msg("signal intrrupt");
                                FREE(var_name);
                                return FALSE;
                            }
                            if(line_field) {
                                if(!statment_tree_internal_commands_write_nextout(nextout, "\n"))
                                {
                                    err_msg("signal intrrupt");
                                    FREE(var_name);
                                    return FALSE;
                                }
                            }
                        }

                        string_delete(value);

                        FREE(var_name);
                    }
                    else {
                        setenv(p, "", 1);

                        if(print) {
                            if(line_field) {
                                if(!statment_tree_internal_commands_write_nextout(nextout, "\n"))
                                {
                                    err_msg("signal intrrupt");
                                    return FALSE;
                                }
                            }
                        }
                    }
                }
            }
            }
            break;


        case kCalc: {
            int i;
            BOOL new_line=FALSE;
            for(i=0; i<vector_size(argv); i++) {
                if(strcmp(string_c_str(vector_item(argv, i)), "-l") == 0)
                {
                    new_line = TRUE;
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
                else if(strcmp(string_c_str(vector_item(argv, i)), "-i")
                        == 0)
                {
                    input = TRUE;
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
            }

            string_obj* str = NULL;
            char* p;
            if(input) {
                str = STRING_NEW("");
                if(!statment_tree_internal_commands_read_nextin(
                                    nextin, str))
                {
                    err_msg("calc: interrupt");
                    string_delete(str);
                    return FALSE;
                }

                p = string_c_str(str);
            }
            else {
                if(vector_size(argv) == 1) {
                    err_msg("calc expression");
                    return FALSE;
                }

                p = string_c_str(vector_item(argv, 1));
            }

            double n;
            BOOL result = calculate(&n, &p);
            if(!result) {
                if(str) string_delete(str);
                return FALSE;
            }
            
            char buf2[128];
            sprintf(buf2, "%lf", n);

            /// 小数点以下の末尾の０を除外 ///
            if(strstr(buf2, ".")) {
                char* p3 = buf2 + strlen(buf2) -1;
                while(*p3 == '0') {
                    p3--;
                }
                if(*p3 == '.') p3--;
                *(p3+1) = 0;
            }

            if(strcmp(buf2, "-0") == 0) {
                strcpy(buf2, "0");
            }

            if(new_line) {
                strcat(buf2, "\n");
            }

            if(!statment_tree_internal_commands_write_nextout(nextout, buf2))
            {
                if(str) string_delete(str);
                err_msg("signal intrrupt");
                return FALSE;
            }

            *rcode = 0;

            if(str) string_delete(str);
            }
            break;

        case kPrint: {
            int i;
            BOOL line_field = FALSE;
            BOOL err = FALSE;
            for(i=0; i<vector_size(argv); i++) {
                char* arg = string_c_str(vector_item(argv,i));
                if(strcmp(arg, "-l") == 0) {
                    line_field = TRUE;
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
                else if(strcmp(arg, "-i") == 0) {
                    input = TRUE;
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
                else if(strcmp(arg, "-e") == 0) {
                    err = TRUE;
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
            }

            if(input) {
                string_obj* str = STRING_NEW("");
                if(!statment_tree_internal_commands_read_nextin(nextin, str))
                {
                    err_msg("interrupt");
                    string_delete(str);
                    return FALSE;
                }
                else {
                    int fd;
                    if(err) 
                        fd = nexterr;
                    else
                        fd = nextout;

                    if(!statment_tree_internal_commands_write_nextout(fd, string_c_str(str))) 
                    {
                        string_delete(str);
                        err_msg("signal intrrupt");
                        return FALSE;
                    }
                    string_delete(str);

                    int k;
                    for(k=1; k<vector_size(argv); k++) {
                        char* arg = string_c_str((string_obj*)
                                            vector_item(argv, k));
                        if(!statment_tree_internal_commands_write_nextout(fd,  arg))
                        {
                            err_msg("signal intrrupt");
                            return FALSE;
                        }

                        if(k<vector_size(argv)-1) {
                            if(!statment_tree_internal_commands_write_nextout(fd, " "))
                            {
                                err_msg("signal intrrupt");
                                return FALSE;
                            }
                        }
                    }
                    if(line_field) {
                        if(statment_tree_internal_commands_write_nextout(fd, "\n"))
                        {
                            err_msg("signal intrrupt");
                            return FALSE;
                        }
                    }

                    *rcode = 0;
                }
            }
            else {
                int fd;
                if(err) 
                    fd = nexterr;
                else
                    fd = nextout;

                int k;
                for(k=1; k<vector_size(argv); k++) {
                    char* arg
                       = string_c_str((string_obj*)vector_item(argv, k));
                    if(!statment_tree_internal_commands_write_nextout(fd, arg))
                    {
                        err_msg("signal intrrupt");
                        return FALSE;
                    }
                    if(k<vector_size(argv)-1) {
                        if(!statment_tree_internal_commands_write_nextout(fd, " ")) 
                        {
                            err_msg("signal intrrupt");
                            return FALSE;
                        }
                    }
                }

                if(line_field) {
                    if(!statment_tree_internal_commands_write_nextout(nextout, "\n"))
                    {
                        err_msg("signal intrrupt");
                        return FALSE;
                    }
                }

                *rcode = 0;
            }
            }
            break;

        case kPuts: {
            int i;
            BOOL line_field = TRUE;
            BOOL err = FALSE;
            for(i=0; i<vector_size(argv); i++) {
                char* arg = string_c_str(vector_item(argv,i));
                if(strcmp(arg, "-nl") == 0) {
                    line_field = FALSE;
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
                else if(strcmp(arg, "-i") == 0) {
                    input = TRUE;
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
                else if(strcmp(arg, "-e") == 0) {
                    err = TRUE;
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
            }

            if(input) {
                string_obj* str = STRING_NEW("");
                if(!statment_tree_internal_commands_read_nextin(nextin, str))
                {
                    err_msg("interrupt");
                    string_delete(str);
                    return FALSE;
                }
                else {
                    int fd;
                    if(err) 
                        fd = nexterr;
                    else
                        fd = nextout;

                    if(!statment_tree_internal_commands_write_nextout(fd, string_c_str(str)))
                    {
                        err_msg("signal intrrupt");
                        string_delete(str);
                        return FALSE;
                    }
                    string_delete(str);

                    int k;
                    for(k=1; k<vector_size(argv); k++) {
                        char* arg = string_c_str((string_obj*)
                                            vector_item(argv, k));
                        if(!statment_tree_internal_commands_write_nextout(fd, arg))
                        {
                            err_msg("signal intrrupt");
                            return FALSE;
                        }

                        if(k<vector_size(argv)-1) {
                            if(!statment_tree_internal_commands_write_nextout(fd, " "))
                            {
                                err_msg("signal intrrupt");
                                return FALSE;
                            }
                        }
                    }
                    if(line_field) {
                        if(!statment_tree_internal_commands_write_nextout(fd, "\n"))
                        {
                            err_msg("signal intrrupt");
                            return FALSE;
                        }
                    }

                    *rcode = 0;
                }
            }
            else {
                int fd;
                if(err) 
                    fd = nexterr;
                else
                    fd = nextout;

                int k;
                for(k=1; k<vector_size(argv); k++) {
                    char* arg
                       = string_c_str((string_obj*)vector_item(argv, k));
                    if(!statment_tree_internal_commands_write_nextout(fd, arg))
                    {
                        err_msg("signal intrrupt");
                        return FALSE;
                    }

                    if(k<vector_size(argv)-1) {
                        if(!statment_tree_internal_commands_write_nextout(fd, " "))
                        {
                            err_msg("signal intrrupt");
                            return FALSE;
                        }
                    }
                }

                if(line_field) {
                    if(!statment_tree_internal_commands_write_nextout(nextout, "\n"))
                    {
                        err_msg("signal intrrupt");
                        return FALSE;
                    }
                }

                *rcode = 0;
            }
            }
            break;

        case kCompile: 
            if(vector_size(argv) == 2) {
                char* fname = string_c_str(vector_item(argv, 1));
                char out_fname[PATH_MAX];

                noextname(out_fname, fname);
                strcat(out_fname, ".kio");

                if(!kitutuki_compile(fname, out_fname)) {
                    return FALSE;
                }

                *rcode = 0;
            }
            break;

        case kRaise: 
            if(vector_size(argv) == 2) {
                char* msg = string_c_str(vector_item(argv, 1));

                err_msg(msg);
                *rcode = 0;

                return FALSE;
            }
            break;

        case kLoad: {
            enum eLineField lf_before = gScriptLineField;
            BOOL compiled = FALSE;

            int i;
            for(i=0; i<vector_size(argv); i++) {
                char* arg = string_c_str(vector_item(argv,i));
                if(strcmp(arg, "-kio") == 0) {
                    compiled = TRUE;
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
                else if(strcmp(arg, "-Lw") == 0) {
                    gScriptLineField = kCRLF;
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
                else if(strcmp(arg, "-Lu") == 0) {
                    gScriptLineField = kLF;
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
                else if(strcmp(arg, "-Lm") == 0) {
                    gScriptLineField = kCR;
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
            }

            if(compiled) {
                *rcode = kitutuki_load_obj(string_c_str(vector_item(argv, 1)), nextout, nextin, nexterr);
            }
            else {
                *rcode = kitutuki_load(string_c_str(vector_item(argv, 1))
                        , nextout, nextin, nexterr);
            }

            gScriptLineField = lf_before;

            if(*rcode < 0) {
                return FALSE;
            }
            }
            break;

        case kWhile: {
            sWhile* w = (sWhile*)command->mExtra;

            while(1) {
                *rcode = run(w->joukensiki, title, nextout
                                , nextin, nexterr
                                , FALSE, TRUE);

                if(*rcode < 0) {
                    return FALSE;
                }
                else if(*rcode == SIGINT) {
                    //*rcode = 0; // そのままだと2重whileの時だめ
                    break;
                }
                else if(*rcode != 0) {
                    break;
                }

                *rcode = run(w->contents, "while command", nextout
                            , nextin, nexterr, FALSE, TRUE);

                if(*rcode < 0) {
                    return FALSE;
                }
                else if(*rcode == SIGINT) {
                    //*rcode = 0;  // そのままだと2重whileの時breakで
                                 // 全部脱出してしまう
                    break;
                }
            }
            }
            break;

        case kBreak:
            *rcode = SIGINT;
            break;

        case kExit:
            if(vector_size(argv) == 2) {
                gKitutukiExit = atoi(string_c_str(vector_item(argv, 1)));
            }
            else {
                gKitutukiExit = 0;
            }
            break;

        case kIf: {
            sIf* if_ = (sIf*)command->mExtra;

            /// if, elif節 ///
            int k;
            for(k=0; k<vector_size(if_->joukensiki_list); k++) {
                sStatments* joukensiki 
                        = vector_item(if_->joukensiki_list, k);
                *rcode = run(joukensiki, title, nextout, nextin, nexterr
                                , FALSE, FALSE);

                if(*rcode == 0) {
                    sStatments* contents 
                            = vector_item(if_->contents_list, k);
                    *rcode = run(contents, title, nextout, nextin
                                , nexterr, FALSE, FALSE);
                    if(*rcode < 0) {
                        return FALSE;
                    }
                    goto if_end;
                }
                else if(*rcode < 0) {
                    return FALSE;
                }
            }

            /// else節 ///
            if(vector_size(if_->contents_list) 
                    > vector_size(if_->joukensiki_list)) 
            {
                sStatments* contents = vector_item(if_->contents_list, k);
                *rcode = run(contents, title, nextout, nextin
                            , nexterr, FALSE, FALSE);
                if(*rcode < 0) {
                    return FALSE;
                }
            }
if_end:
            null_fun();
            }
            break;

        case kChoise: {
            int i;
            BOOL line_field = FALSE;
            BOOL list_view = FALSE;
            BOOL result_num = FALSE;
            for(i=0; i<vector_size(argv); i++) {
                char* arg = string_c_str(vector_item(argv,i));
                if(strcmp(arg, "-l") == 0) {
                    line_field = TRUE;
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
                else if(strcmp(arg, "-v") == 0) {
                    list_view = TRUE;
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
                else if(strcmp(arg, "-n") == 0) {
                    result_num = TRUE;
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
                else if(strcmp(arg, "-i") == 0) {
                    input = TRUE;
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
            }

            if(input) {
                if(vector_size(argv) >= 2) {
                    char* msg = string_c_str(vector_item(argv, 1));

                    vector_obj* v = VECTOR_NEW(10);
                    while(1) {
                        string_obj* str = STRING_NEW("");
                        int result = statment_tree_internal_commands_read_nextin_oneline(nextin, str, gLineField);
                        if(result == 0) {
                            string_chomp(str);
                            vector_add(v, str);
                        }
                        else if(result == -1) {
                            err_msg("interrupt");
                            string_delete(str);
                            for(j=0; j<vector_size(v); j++) {
                               string_delete(vector_item(v, j));
                            }
                            vector_delete(v);
                            return FALSE;
                        }
                        else {
                            string_delete(str);
                            break;
                        }
                    }

                    if(vector_size(v) > 0) {
                        char** strs 
                            = MALLOC(sizeof(char*)*(vector_size(v)));

                        int j;
                        for(j=0; j<vector_size(v); j++) {
                           strs[j] = string_c_str(vector_item(v, j));
                        }

                        int ret = statment_tree_internal_commands_choise(msg, strs, vector_size(v), list_view);

                        if(result_num) {
                            char buf[128];
                            if(line_field) {
                                sprintf(buf, "%d\n", ret);
                            }
                            else {
                                sprintf(buf, "%d", ret);
                            }
                            if(!statment_tree_internal_commands_write_nextout(nextout, buf))
                            {
                                err_msg("signal intrrupt");
                                FREE(strs);
                                for(j=0; j<vector_size(v); j++) {
                                   string_delete(vector_item(v, j));
                                }
                                vector_delete(v);
                                return FALSE;
                            }
                        }
                        else {
                            if(!statment_tree_internal_commands_write_nextout(nextout, strs[ret]))
                            {
                                err_msg("signal intrrupt");
                                FREE(strs);
                                for(j=0; j<vector_size(v); j++) {
                                   string_delete(vector_item(v, j));
                                }
                                vector_delete(v);
                                return FALSE;
                            }
                            if(line_field) {
                                if(!statment_tree_internal_commands_write_nextout(nextout, "\n"))
                                {
                                    err_msg("signal intrrupt");
                                    FREE(strs);
                                    for(j=0; j<vector_size(v); j++) {
                                       string_delete(vector_item(v, j));
                                    }
                                    vector_delete(v);
                                    return FALSE;
                                }
                            }
                        }

                        *rcode = 0;

                        FREE(strs);
                    }

                    for(j=0; j<vector_size(v); j++) {
                       string_delete(vector_item(v, j));
                    }
                    vector_delete(v);
                }
            }
            else {
                if(vector_size(argv) > 3) {
                    char* msg = string_c_str(vector_item(argv, 1));

                    char** strs = MALLOC(sizeof(char*)*(vector_size(argv)-2));

                    int i;
                    for(i=2; i<vector_size(argv); i++) {
                        strs[i-2] = string_c_str(vector_item(argv, i));
                    }

                    int ret = statment_tree_internal_commands_choise(msg, strs, vector_size(argv)-2, list_view);

                    if(result_num) {
                        char buf[128];
                        if(line_field) {
                            sprintf(buf, "%d\n", ret);
                        }
                        else {
                            sprintf(buf, "%d", ret);
                        }
                        if(!statment_tree_internal_commands_write_nextout(nextout, buf))
                        {
                            err_msg("signal intrrupt");
                            FREE(strs);
                            return FALSE;
                        }
                    }
                    else {
                        if(!statment_tree_internal_commands_write_nextout(nextout, strs[ret]))
                        {
                            err_msg("signal intrrupt");
                            FREE(strs);
                            return FALSE;
                        }
                        if(line_field) {
                            if(!statment_tree_internal_commands_write_nextout(nextout, "\n"))
                            {
                                err_msg("signal intrrupt");
                                FREE(strs);
                                return FALSE;
                            }
                        }
                    }

                    *rcode = 0;

                    FREE(strs);
                }
            }
            }
            break;


        case kSplit: {
            if(vector_size(argv) <= 2) {
                char* regex;
                if(vector_size(argv) == 2) {
                    regex = string_c_str(vector_item(argv, 1));
                }
                if(vector_size(argv) == 1) {
                    regex = "\\s+";
                }

                string_obj* line = STRING_NEW("");

                if(!statment_tree_internal_commands_read_nextin(nextin, line)) {
                    err_msg("interrupt");
                    string_delete(line);
                    return FALSE;
                }
                else {
                    char* p = string_c_str(line);
                    
                    regex_t* reg = hash_item(gRegexs, regex);
                    int r;

                    char* regex2 = regex;

                    if(reg == NULL) {
                        OnigErrorInfo err_info;
                        r = onig_new(&reg, regex2
                                , regex2 + strlen((char*)regex2)
                                , ONIG_OPTION_DEFAULT
                                , ONIG_ENCODING_UTF8
                                //, ONIG_SYNTAX_RUBY
                                //, ONIG_SYNTAX_PERL
                                , ONIG_SYNTAX_DEFAULT
                                , &err_info);

                        if(r == ONIG_NORMAL) {
                            hash_put(gRegexs, regex, reg);
                        }
                        else {
                            onig_free(reg);
                        }
                    }
                    else {
                        r = ONIG_NORMAL;
                    }

                    if(r == ONIG_NORMAL) {
                        string_obj* new_line = STRING_NEW("");
                        while(*p) {
                            char* target = p;
                            OnigRegion* region = onig_region_new();
                            int r2 = onig_search(reg, target
                               , target + strlen(target)
                               , target, target + strlen(target)
                               , region, ONIG_OPTION_NONE);

                            if(region->num_regs > 0 && region->beg[0] >= 0) {
                                /// マッチした文字列の前 ///
                                char* tmp = MALLOC(region->beg[0] + 1);

                                memcpy(tmp, target, region->beg[0]);
                                tmp[region->beg[0]] = 0;

                                string_put(new_line, tmp);

                                /// 出力 ///
                                string_push_back(new_line, "\n");
                                if(!statment_tree_internal_commands_write_nextout(nextout,  string_c_str(new_line)))
                                {
                                    string_delete(new_line);
                                    string_delete(line);
                                    err_msg("singal intrrupt");
                                    return FALSE;
                                }

                                FREE(tmp);

                                /// マッチした文字列の後 ///
                                p = target + region->end[0];

                                /// 一番最後でマッチしていたら空行を出力
                                if(*p == 0) {
                                    string_put(new_line, "\n");
                                    if(!statment_tree_internal_commands_write_nextout(nextout,  string_c_str(new_line)))
                                    {
                                        string_delete(new_line);
                                        string_delete(line);
                                        err_msg("singal intrrupt");
                                        return FALSE;
                                    }
                                }
                            }
                            else {
                                string_put(new_line, p);

                                /// 出力 ///
                                string_push_back(new_line, "\n");
                                if(!statment_tree_internal_commands_write_nextout(nextout,  string_c_str(new_line)))
                                {
                                    string_delete(new_line);
                                    string_delete(line);
                                    err_msg("singal intrrupt");
                                    return FALSE;
                                }

                                p = p + strlen(p);
                            }

                            onig_region_free(region, 1);
                        }
                        string_delete(new_line);
                    }
                    else {
                        err_msg("split: invalid regex");
                        string_delete(line);

                        return FALSE;
                    }

                    *rcode = 0;
                }

                string_delete(line);
            }
            }
            break;

        case kAdd: {
            BOOL flg_n = FALSE;
            int n;

            int l;
            for(l=0; l<vector_size(argv); l++) {
                char* arg = string_c_str(vector_item(argv, l));

                if(strcmp(arg, "-n") == 0 && l+1 < vector_size(argv)) {
                    flg_n = TRUE;
                    n = atoi(string_c_str(vector_item(argv, l+1)));
                    string_delete(vector_item(argv, l));
                    vector_erase(argv, l);
                    string_delete(vector_item(argv, l));
                    vector_erase(argv, l);
                    l--;
                    continue;
                }
            }

            if(flg_n) {
                string_obj* str = STRING_NEW("");

                if(!statment_tree_internal_commands_read_nextin(nextin, str)) 
                {
                    err_msg("interrupt");
                    string_delete(str);
                    return FALSE;
                }
                else
                {
                    char* str2 = string_c_str(vector_item(argv, 1));

                    if(n < 0) {
                        n = strlen(string_c_str(str)) + n + 1;
                    }

                    if(n < 0 || n > strlen(string_c_str(str))) {
                        err_msg("add: invalid -n range");
                        string_delete(str);
                        return FALSE;
                    }

                    string_insert(str, n, str2);

                    if(!statment_tree_internal_commands_write_nextout(nextout,  string_c_str(str)))
                    {
                        string_delete(str);
                        err_msg("singal intrrupt");
                        return FALSE;
                    }
                    *rcode = 0;
                    string_delete(str);
                }
            }
            else {
                string_obj* str = STRING_NEW("");

                if(!statment_tree_internal_commands_read_nextin(nextin, str)) 
                {
                    err_msg("interrupt");
                    string_delete(str);
                    return FALSE;
                }
                else {
                    char* str2 = string_c_str(vector_item(argv, 1));

                    string_push_back(str, str2);

                    if(!statment_tree_internal_commands_write_nextout(nextout, string_c_str(str)))
                    {
                        string_delete(str);
                        err_msg("signal intrrupt");
                        return FALSE;
                    }

                    *rcode = 0;
                    string_delete(str);
                }
            }
            }
            break;

        case kDel: {
            BOOL flg_n = FALSE;
            int n;

            int l;
            for(l=0; l<vector_size(argv); l++) {
                char* arg = string_c_str(vector_item(argv, l));

                if(strcmp(arg, "-n") == 0 && l+1 < vector_size(argv)) {
                    flg_n = TRUE;
                    n = atoi(string_c_str(vector_item(argv, l+1)));
                    string_delete(vector_item(argv, l));
                    vector_erase(argv, l);
                    string_delete(vector_item(argv, l));
                    vector_erase(argv, l);
                    l--;
                    continue;
                }
            }

            if(flg_n) {
                string_obj* str = STRING_NEW("");

                if(!statment_tree_internal_commands_read_nextin(nextin, str)) 
                {
                    err_msg("interrupt");
                    string_delete(str);
                    return FALSE;
                }
                else {
                    int num;
                    if(vector_size(argv) >= 2) {
                        char* arg1 = string_c_str(vector_item(argv, 1));
                        num = atoi(arg1);
                    }
                    else {
                        num = 1;
                    }

                    if(n < 0) {
                        n = strlen(string_c_str(str)) + n + 1;
                    }

                    if(n < 0 || n > strlen(string_c_str(str))) {
                        err_msg("del: invalid -n range");
                        string_delete(str);
                        return FALSE;
                    }

                    string_erase(str, n, num);

                    if(!statment_tree_internal_commands_write_nextout(nextout, string_c_str(str)))
                    {
                        err_msg("signal intrrupt");
                        string_delete(str);
                        return FALSE;
                    }
                    *rcode = 0;
                    string_delete(str);
                }
            }
            else {
                string_obj* str = STRING_NEW("");

                if(!statment_tree_internal_commands_read_nextin(nextin, str)) 
                {
                    err_msg("interrupt");
                    string_delete(str);
                    return FALSE;
                }
                else {
                    int num;
                    if(vector_size(argv) >= 2) {
                        char* arg1 = string_c_str(vector_item(argv, 1));
                        num = atoi(arg1);
                    }
                    else {
                        num = 1;
                    }

                    string_trunc(str, string_length(str)-num);

                    if(!statment_tree_internal_commands_write_nextout(nextout, string_c_str(str)))
                    {
                        string_delete(str);
                        err_msg("signal intrrupt");
                        return FALSE;
                    }

                    *rcode = 0;
                    string_delete(str);
                }
            }
            }
            break;

        case kJoin: {
            int i;
            BOOL line_field = FALSE;
            for(i=0; i<vector_size(argv); i++) {
                char* arg = string_c_str(vector_item(argv, i));
                if(strcmp(arg, "-l") == 0) {
                    line_field = TRUE;
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
            }

            string_obj* str = STRING_NEW("");

            char* field;
            if(vector_size(argv) == 2) {
                field = string_c_str(vector_item(argv, 1));
            }
            else {
                field = " ";
            }

            if(!statment_tree_internal_commands_read_nextin(nextin, str))
            {
                err_msg("interrupt");
                string_delete(str);
                return FALSE;
            }
            else {
                char* p = string_c_str(str);
                string_obj* buf = STRING_NEW("");
                while(*p) {
                    if(is_line_field2(gLineField, p)) {
                        if(gLineField == kCRLF) {
                            p+=2;
                        }
                        else {
                            p++;
                        }
                        char* p3 = field;
                        while(*p3) {
                            string_push_back2(buf, *p3++);
                        }

                        if(!statment_tree_internal_commands_write_nextout(nextout, string_c_str(buf)))
                        {
                            err_msg("signal intrrupt");
                            string_delete(str);
                            string_delete(buf);
                            return FALSE;
                        }

                        string_put(buf, "");
                    }
                    else {
                        string_push_back2(buf, *p++);
                    }
                }

                if(strcmp(string_c_str(buf), "") != 0) 
                    if(!statment_tree_internal_commands_write_nextout(nextout, string_c_str(buf)))
                    {
                        err_msg("signal intrrupt");
                        string_delete(str);
                        string_delete(buf);
                        return FALSE;
                    }

                if(line_field) {
                    if(!statment_tree_internal_commands_write_nextout(nextout, "\n"))
                    {
                        err_msg("signal intrrupt");
                        string_delete(str);
                        string_delete(buf);
                        return FALSE;
                    }
                }

                string_delete(str);
                string_delete(buf);

                *rcode = 0;
            }
            }
            break;

        case kAryNew: {
            int i;
            BOOL all = FALSE;
            BOOL chomp = TRUE;
            BOOL print = FALSE;
            BOOL line_field = FALSE;
            for(i=0; i<vector_size(argv); i++) {
                char* arg = string_c_str(vector_item(argv, i));
                if(strcmp(arg, "-a") == 0) 
                {
                    all = TRUE;
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
                else if(strcmp(arg, "-nc") == 0) 
                {
                    chomp = FALSE;
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
                else if(strcmp(arg, "-i") == 0) 
                {
                    input = TRUE;
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
                else if(strcmp(arg, "-p") == 0) 
                {
                    print = TRUE;
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
                else if(strcmp(arg, "-l") == 0) 
                {
                    line_field = TRUE;
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
            }

            if(all) {
                if(vector_size(argv) != 2) {
                    err_msg("ary_new -a ary_name");

                    return FALSE;
                }

                string_obj* str = STRING_NEW("");
                if(!statment_tree_internal_commands_read_nextin(
                                    nextin, str))
                {
                    err_msg("interrupt");
                    string_delete(str);
                    return FALSE;
                }
                else {
                    char* key = string_c_str(vector_item(argv, 1));

                    vector_obj* v = hash_item(gArrays, key);
                    if(v) {
                        int i;
                        for(i=0; i<vector_size(v); i++) {
                            string_delete(vector_item(v, i));
                        }
                        vector_clear(v);
                    }
                    else {
                        v = VECTOR_NEW(5);
                        hash_put(gArrays, key, v);
                    }
                
                    vector_add(v, str);
                
                    update_ary_env(key);
                    *rcode = 0;

                    if(print) {
                        if(!statment_tree_internal_commands_write_nextout(nextout, string_c_str(str)))
                        {
                            string_delete(str);
                            err_msg("signal intrrupt");
                            return FALSE;
                        }
                        if(line_field) {
                            if(!statment_tree_internal_commands_write_nextout(nextout, "\n"))
                            {
                                err_msg("signal intrrupt");
                                string_delete(str);
                                return FALSE;
                            }
                        }
                    }
                }
            }
            else if(input) {
                if(vector_size(argv) != 2) {
                    err_msg("ary_new -i ary_name");

                    return FALSE;
                }

                char* key = string_c_str(vector_item(argv, 1));

                vector_obj* v = hash_item(gArrays, key);
                if(v) {
                    int i;
                    for(i=0; i<vector_size(v); i++) {
                        string_delete(vector_item(v, i));
                    }
                    vector_clear(v);
                }
                else {
                    v = VECTOR_NEW(5);
                    hash_put(gArrays, key, v);
                }

                while(1) {
                    string_obj* line = STRING_NEW("");
                    int ret = statment_tree_internal_commands_read_nextin_oneline(nextin, line, gLineField);
                    if(ret == -1) {
                        err_msg("interrupt");
                        string_delete(line);
                        return FALSE;
                    }
                    else if(ret == 1) {
                        string_delete(line);
                        break;
                    }
                    if(chomp) string_chomp(line);

                    vector_add(v, line);

                    if(print) {
                        if(!statment_tree_internal_commands_write_nextout(nextout, string_c_str(line)))
                        {
                            err_msg("signal intrrupt");
                            return FALSE;
                        }
                        if(line_field) {
                            if(!statment_tree_internal_commands_write_nextout(nextout, "\n"))
                            {
                                err_msg("signal intrrupt");
                                return FALSE;
                            }
                        }
                    }
                }
                
                update_ary_env(key);

                *rcode = 0;
            }
            else {
                if(vector_size(argv) < 2) {
                    err_msg("ary_new ary_name value1 value2...");

                    return FALSE;
                }

                char* key = string_c_str(vector_item(argv, 1));
                
                vector_obj* v = hash_item(gArrays, key);
                if(v) {
                    int i;
                    for(i=0; i<vector_size(v); i++) {
                        string_delete(vector_item(v, i));
                    }
                    vector_clear(v);
                }
                else {
                    v = VECTOR_NEW(5);
                    hash_put(gArrays, key, v);
                }
                
                for(i=2; i<vector_size(argv); i++) {
                    char* arg = string_c_str(vector_item(argv, i));
                    vector_add(v, STRING_NEW(arg));
                    
                    if(print) {
                        if(!statment_tree_internal_commands_write_nextout(nextout, arg))
                        {
                            err_msg("signal intrrupt");
                            return FALSE;
                        }
                        if(line_field) {
                            if(!statment_tree_internal_commands_write_nextout(nextout, "\n"))
                            {
                                err_msg("signal intrrupt");
                                return FALSE;
                            }
                        }
                    }
                }
                
                update_ary_env(key);
                
                *rcode = 0;
            }
            }
            break;

        case kAryAdd: {
            int i;
            BOOL all = FALSE;
            BOOL flg_n = FALSE;
            BOOL chomp = TRUE;
            BOOL print = FALSE;
            BOOL line_field = FALSE;
            int n;
            for(i=0; i<vector_size(argv); i++) {
                char* arg = string_c_str(vector_item(argv, i));
                if(strcmp(arg, "-n") == 0 
                        && i+1 < vector_size(argv)) 
                {
                    flg_n = TRUE;
                    n = atoi(string_c_str(vector_item(argv, i+1)));
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
                else if(strcmp(arg, "-a") == 0) 
                {
                    all = TRUE;
                    string_delete(vector_item(argv,i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
                else if(strcmp(arg, "-nc") == 0) 
                {
                    chomp = FALSE;
                    string_delete(vector_item(argv,i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
                else if(strcmp(arg, "-i") == 0) {
                    input = TRUE;
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
                else if(strcmp(arg, "-l") == 0) {
                    line_field = TRUE;
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
                else if(strcmp(arg, "-p") == 0) {
                    print = TRUE;
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
            }

            if(input) {
                /// 標準入力から、設定
                if(vector_size(argv) != 2) {
                    err_msg("ary_add: invalid option");
                    return FALSE;
                }

                char* key = string_c_str(vector_item(argv, 1));

                vector_obj* v = hash_item(gArrays, key);
                if(v == NULL) {
                    v = VECTOR_NEW(5);
                    hash_put(gArrays, key, v);
                }

                int i=0;
                while(1) {
                    string_obj* line = STRING_NEW("");
                    int ret = statment_tree_internal_commands_read_nextin_oneline(nextin, line, gLineField);

                    if(ret == -1) {
                        err_msg("intterrupt");
                        string_delete(line);
                        return FALSE;
                    }
                    else if(ret == 1) {      // EOF
                        string_delete(line);
                        break;
                    }

                    if(chomp) string_chomp(line);

                    if(flg_n) {
                        if(n < 0) {
                            n = n + vector_size(v) + 1;
                        }

                        if(n < 0) {
                            err_msg("ary_add: invalid -n range");
                            string_delete(line);
                            return FALSE;
                        }

                        if(n+i < vector_size(v)) {
                            vector_insert(v, n+i, line);
                        }
                        else {
                            vector_add(v, line);
                        }
                    }
                    else {
                        vector_add(v, line);
                    }

                    if(print) {
                        if(!statment_tree_internal_commands_write_nextout(nextout, string_c_str(line)))
                        {
                            err_msg("signal intrrupt");
                            return FALSE;
                        }

                        if(line_field) {
                            if(!statment_tree_internal_commands_write_nextout(nextout, "\n"))
                            {
                                err_msg("signal intrrupt");
                                return FALSE;
                            }
                        }
                    }
                    
                    i++;
                }
               
                update_ary_env(key);

                *rcode = 0;
            }
            else if(all) {
                if(vector_size(argv) == 2) {
                    char* p = string_c_str((string_obj*)vector_item(argv, 1));
                    string_obj* str = STRING_NEW("");
                    if(!statment_tree_internal_commands_read_nextin(
                                        nextin, str))
                    {
                        err_msg("interrupt");
                        string_delete(str);
                        return FALSE;
                    }
                    else {
                        vector_obj* v = hash_item(gArrays, p);

                        if(v == NULL) {
                            v = VECTOR_NEW(5);
                            hash_put(gArrays, p, v);
                        }

                        if(flg_n) {
                            if(n < 0) {
                                n = n + vector_size(v) + 1;
                            }

                            if(n < 0) {
                                err_msg("ary_add: invalid -n range");
                                string_delete(str);
                                return FALSE;
                            }

                            if(n < vector_size(v)) {
                                vector_insert(v, n, str);
                            }
                            else {
                                vector_add(v, str);
                            }
                        }
                        else {
                            vector_add(v, str);
                        }

                        *rcode = 0;

                        if(print) {
                            if(!statment_tree_internal_commands_write_nextout(nextout, string_c_str(str)))
                            {
                                err_msg("signal intrrupt");
                                return FALSE;
                            }

                            if(line_field) {
                                if(!statment_tree_internal_commands_write_nextout(nextout, "\n"))
                                {
                                    err_msg("singal inttrupt");
                                    return FALSE;
                                }
                            }
                        }
                    }

                    update_ary_env(p);
                }
                else {
                    err_msg("ary_add: ary_add -a must have one var_name");
                    return FALSE;
                }
            }
            else {
                if(vector_size(argv) < 3) {
                    err_msg("ary_add: invalid option");
                    return FALSE;
                }

                char* key = string_c_str(vector_item(argv, 1));

                vector_obj* v = hash_item(gArrays, key);
                if(v == NULL) {
                    v = VECTOR_NEW(5);
                    hash_put(gArrays, key, v);
                }
                
                if(flg_n) {
                    if(n < 0) {
                        n = n + vector_size(v) + 1;
                    }

                    if(n < 0) {
                        err_msg("ary_add: invalid -n range");
                        return FALSE;
                    }

                    int i;
                    for(i=2; i<vector_size(argv); i++) {
                        string_obj* line = STRING_NEW("");
                        char* str = string_c_str(vector_item(argv, i));
                        string_put(line, str);

                        if(n+i-2 < vector_size(v)) {
                            vector_insert(v, n+i-2, line);
                        }
                        else {
                            vector_add(v, line);
                        }

                        if(print) {
                            if(!statment_tree_internal_commands_write_nextout(nextout, string_c_str(line)))
                            {
                                err_msg("signal intrrupt");
                                return FALSE;
                            }

                            if(line_field) {
                                if(!statment_tree_internal_commands_write_nextout(nextout, "\n"))
                                {
                                    err_msg("signal intrrupt");
                                    return FALSE;
                                }
                            }
                        }
                    }
                }
                else {
                    int i;
                    for(i=2; i<vector_size(argv); i++) {
                        string_obj* line = STRING_NEW("");
                        char* str = string_c_str(vector_item(argv, i));
                        string_put(line, str);

                        vector_add(v, line);

                        if(print) {
                            if(!statment_tree_internal_commands_write_nextout(nextout, string_c_str(line)))
                            {
                                err_msg("signal intrrupt");
                                return FALSE;
                            }

                            if(line_field) {
                                if(!statment_tree_internal_commands_write_nextout(nextout, "\n"))
                                {
                                    err_msg("signal intrrupt");
                                    return FALSE;
                                }
                            }
                        }
                    }
                }
                
                update_ary_env(key);

                *rcode = 0;
            }
            }
            break;

        case kAryErase:
            if(vector_size(argv) == 3) {
                char* key = string_c_str(vector_item(argv, 1));
                char* number = string_c_str(vector_item(argv, 2));
                int n = atoi(number);

                vector_obj* v = hash_item(gArrays, key);
                if(v && n >= 0 && n < vector_size(v)) {
                    string_delete(vector_item(v, n));
                    vector_erase(v, n);
                    *rcode = 0;
                }
                update_ary_env(key);
            }
            break;

        case kAryClear:
            if(vector_size(argv) == 2) {
                char* key = string_c_str(vector_item(argv, 1));

                vector_obj* v = hash_item(gArrays, key);
                if(v) {
                    int i;
                    for(i=0; i<vector_size(v); i++) {
                        string_delete(vector_item(v, i));
                        vector_erase(v, i);
                    }
                    vector_delete(v);

                    hash_erase(gArrays, key);
                    clear_ary_env(key);

                    *rcode = 0;
                }
            }
            break;

        case kMatch: {
            BOOL quit = FALSE;
            BOOL line_field = TRUE;
            BOOL all = FALSE;
            BOOL line_num = FALSE;
            string_obj* field = STRING_NEW(" ");
            int i;
            for(i=0; i<vector_size(argv); i++) {
                char* arg = string_c_str(vector_item(argv, i));
                if(strcmp(arg, "-q") == 0)
                {
                    quit = TRUE;
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
                else if(strcmp(arg, "-a") == 0)
                {
                    all = TRUE;
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
                else if(strcmp(arg, "-nl") == 0)
                {
                    line_field = FALSE;
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
                else if(strcmp(arg, "-n") == 0)
                {
                    line_num = TRUE;
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
                else if(strcmp(arg, "-i") == 0)
                {
                    input = TRUE;
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
                else if(strcmp(arg, "-f") == 0 
                    && i+1 < vector_size(argv)) 
                {
                    string_put(field 
                         , string_c_str(vector_item(argv, i+1)));
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
            }

            if(input) {
                if(vector_size(argv) >= 2) {
                    char* regex = string_c_str(vector_item(argv, 1));
                    char* regex2 = regex;

                    int n = 1;
                    for(;;) {
                        string_obj* str = STRING_NEW("");
                        int ret = statment_tree_internal_commands_read_nextin_oneline(nextin, str, gLineField);
                        if(ret == -1) {
                            err_msg("interrupt");
                            string_delete(str);
                            string_delete(field);
                            return FALSE;
                        }
                        else if(ret == 1) 
                        {
                            string_delete(str);
                            break;
                        }
                        
                        if(!statment_tree_internal_commands_match(
                            string_c_str(str), regex2, rcode, nextout
                            , !quit, line_field, line_num, n, field))
                        {
                            string_delete(str);
                            string_delete(field);
                            return FALSE;
                        }

                        string_delete(str);

                        n++;
                    }
                }
            }
            else {
                string_obj* str;
                if(all && vector_size(argv) == 2) {
                    str = STRING_NEW("");
                    if(!statment_tree_internal_commands_read_nextin
                            (nextin, str))
                    {
                        err_msg("interrupt");
                        string_delete(str);
                        string_delete(field);
                        return FALSE;
                    }
                }
                else if(vector_size(argv) == 3) {
                    str = STRING_NEW(string_c_str(vector_item(argv, 2)));
                }
                else {
                    break;
                }

                char* str2 = string_c_str(str);

                char* regex = string_c_str(vector_item(argv, 1));
                char* regex2 = regex;

                if(!statment_tree_internal_commands_match(
                    str2, regex2, rcode, nextout, !quit, line_field
                    , line_num, 1, field))
                {
                    string_delete(str);
                    string_delete(field);
                    return FALSE;
                }

                string_delete(str);
            }

            string_delete(field);
            }
            break;

        case kSub: {
            BOOL global = FALSE;
            BOOL quiet = FALSE;
            BOOL line_field = FALSE;
            int i;
            for(i=0; i<vector_size(argv); i++) {
                char* arg = string_c_str(vector_item(argv, i));
                if(strcmp(arg, "-g") == 0) {
                    global = TRUE;
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
                else if(strcmp(arg, "-q") == 0) {
                    quiet = TRUE;
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
                else if(strcmp(arg, "-l") == 0) {
                    line_field = TRUE;
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
            }
            
            if(vector_size(argv) == 3) {
                int hennkann_kaisuu = 0;

                string_obj* out = STRING_NEW("");
                while(1) {
                    string_obj* str = STRING_NEW("");

                    int ret = statment_tree_internal_commands_read_nextin_oneline(nextin, str, gLineField);
                    
                    if(ret == 1) { // EOF
                        string_delete(str);
                        break;
                    }
                    else if(ret == -1) {
                        err_msg("interrupt");
                        string_delete(str);
                        string_delete(out);
                        return FALSE;
                    }

                    int i;

                    char* regex = string_c_str(vector_item(argv, 1));

                    char* regex2 = regex;

                    regex_t* reg = hash_item(gRegexs, regex);
                    int r;

                    if(reg == NULL) {
                        OnigErrorInfo err_info;
                        r = onig_new(&reg, regex2
                                , regex2 + strlen((char*)regex2)
                                , ONIG_OPTION_DEFAULT
                                , ONIG_ENCODING_UTF8
                                //, ONIG_SYNTAX_RUBY
                                //, ONIG_SYNTAX_PERL
                                , ONIG_SYNTAX_DEFAULT
                                , &err_info);

                        if(r == ONIG_NORMAL) {
                            hash_put(gRegexs, regex, reg);
                        }
                        else {
                            onig_free(reg);

                            err_msg("sub: invalid regex");
                            string_delete(out);
                            string_delete(str);
                            return FALSE;
                        }
                    }
                    else {
                        r = ONIG_NORMAL;
                    }

                    char* str2 = string_c_str(str);
                    if(r == ONIG_NORMAL) {
                        if(global) {
                            int point = 0;
                            string_obj* hennkan_mojiretu 
                                            = STRING_NEW("");
                            while(1) {
                                OnigRegion* region = onig_region_new();
                                OnigErrorInfo err_info;
                                int r2 = onig_search(reg, str2
                                   , str2 + strlen(str2)
                                   , str2 + point
                                   , str2 + strlen(str2)
                                   , region, ONIG_OPTION_NONE);

/*
                                int r2 = onig_match(reg, str2
                                   , str2 + strlen(str2)
                                   , str2 + point
                                   , region, ONIG_OPTION_NONE);
*/
                                if(r2 == ONIG_MISMATCH) {
                                    //point++;
                                    onig_region_free(region, 1);
                                    break;
                                }
                                else {
                                    /// グループ化したものを
                                    ///  $1 $2などに変換 ///
                                    if(point == 0) {
                                        char* p = string_c_str(
                                            vector_item(argv, 2));

                                        while(*p) {
                                            if(*p == '$' 
                                                && *(p+1) == '$')
                                            {
                                                p++;
                                                string_push_back2(
                                                    hennkan_mojiretu
                                                        , *p++);
                                            }
                                            else if(*p == '$' 
                                                && *(p+1) >= '1'
                                                && *(p+1) <= '9')
                                            {
                                                char tmp[BUFSIZ];
                                                int n = *(p+1) - '0';

                                                if(n < region->num_regs) 
                                                {
                                                    p+=2;
                                                    memcpy(tmp
                                                        , str2 
                                                        + region->beg[n]
                                                        , region->end[n]
                                                        -region->beg[n]);

                                                    tmp[region->end[n]
                                                      - region->beg[n]] 
                                                        = 0;

                                                    char* p3 = tmp;
                                                    while(*p3) {
                                                       string_push_back2(
                                                        hennkan_mojiretu
                                                         , *p3++);
                                                    }
                                                }
                                                else {
                                                    string_push_back2(
                                                        hennkan_mojiretu
                                                        , *p++);
                                                    string_push_back2(
                                                        hennkan_mojiretu
                                                        , *p++);
                                                }
                                            }
                                            else { 
                                                string_push_back2(
                                                    hennkan_mojiretu
                                                        , *p++);
                                            }
                                        }
                                    }

                                    char* tmp = MALLOC(
                                        region->beg[0] - point + 1);

                                    memcpy(tmp, str2 + point
                                        , region->beg[0] - point);
                                    tmp[region->beg[0] - point] = 0;
                                    string_push_back(out, tmp);

                                    string_push_back(out, 
                                        string_c_str(hennkan_mojiretu));
                                    if(region->beg[0] == region->end[0]) 
                                    {
                                       string_push_back2(out
                                            , str2[region->beg[0]]);
                                    }

                                    if(region->beg[0] == region->end[0])
                                        point = region->end[0] + 1;
                                    else
                                        point = region->end[0];

                                    hennkann_kaisuu++;

                                    FREE(tmp);
                                }

                                onig_region_free(region, 1);

                                if(str2 + point >= str2 + strlen(str2)) {
                                    break;
                                }
                            }

                            if(point < strlen(str2))
                                string_push_back(out, str2 + point);

                            string_delete(hennkan_mojiretu);
                        }
                        else {
                            OnigRegion* region = onig_region_new();

                            int r2 = onig_search(reg, str2
                               , str2 + strlen((char*)str2)
                               , str2, str2 + strlen((char*)str2)
                               , region, ONIG_OPTION_NONE);

                            if(r2 >= 0) {
                                /// グループ化したものを
                                /// $1 $2などに変換 ///
                                
                                char* p = string_c_str(
                                    vector_item(argv, 2));

                                string_obj* hennkan_mojiretu
                                    = STRING_NEW("");
                                while(*p) {
                                    if(*p == '$' && *(p+1) == '$') {
                                        p++;
                                        string_push_back2(
                                            hennkan_mojiretu, *p++);
                                    }
                                    else if(*p == '$' 
                                       && *(p+1) >= '1' && *(p+1) <= '9')
                                    {
                                        int n = *(p+1) - '0';
                                        if(n < region->num_regs) {
                                            p+=2;
                                            char* tmp = MALLOC(
                                          region->end[n]-region->beg[n]+1);

                                            memcpy(tmp
                                                , str2 + region->beg[n]
                                                , region->end[n]
                                                    -region->beg[n]);

                                            tmp[region->end[n]
                                                - region->beg[n]] = 0;
                                            char* p3 = tmp;
                                            while(*p3) {
                                                string_push_back2(
                                                    hennkan_mojiretu
                                                    , *p3++);
                                            }

                                            FREE(tmp);
                                        }
                                        else {
                                            string_push_back2(
                                                hennkan_mojiretu, *p++);
                                            string_push_back2(
                                                hennkan_mojiretu, *p++);
                                        }
                                    }
                                    else { 
                                        string_push_back2(
                                            hennkan_mojiretu, *p++);
                                    }
                                }

                                char* tmp = MALLOC(region->beg[0] + 1);

                                memcpy(tmp, str2, region->beg[0]);
                                tmp[region->beg[0]] = 0;
                                string_push_back(out, tmp);

                                string_push_back(out
                                    , string_c_str(hennkan_mojiretu));

                                string_push_back(out
                                    , str2 + region->end[0]);

                                string_delete(hennkan_mojiretu);
                                hennkann_kaisuu++;

                                FREE(tmp);
                            }
                            else {
                                string_push_back(out, str2);
                            }

                            onig_region_free(region, 1);
                        }
                    }

                    string_delete(str);
                }

                if(!quiet) {
                    if(!statment_tree_internal_commands_write_nextout(nextout, string_c_str(out)))
                    {
                        err_msg("signal intrrupt");
                        string_delete(out);
                        return FALSE;
                    }

                    if(line_field) {
                        if(!statment_tree_internal_commands_write_nextout(nextout, "\n"))
                        {
                            err_msg("signal intrrupt");
                            string_delete(out);
                            return FALSE;
                        }
                    }
                }


                char buf[128];
                sprintf(buf, "%d", hennkann_kaisuu);
                kitutuki_set_local_var("SUBCOUNT", buf);

                string_delete(out);

                *rcode = 0;
            }
            }
            break;

        case kScan: {
            int i;
            string_obj* field = STRING_NEW(" ");
            for(i=0; i<vector_size(argv); i++) {
                string_obj* item = vector_item(argv, i);
                if(strcmp(string_c_str(item), "-i") == 0) {
                    input = TRUE;
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
                else if(strcmp(string_c_str(item), "-f") == 0 
                    && i+1 < vector_size(argv)) 
                {
                    string_put(field
                        , string_c_str(vector_item(argv, i+1)));
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
            }
            string_obj* str;
            if(input && vector_size(argv) == 2) {
                str = STRING_NEW("");
                if(!statment_tree_internal_commands_read_nextin
                        (nextin, str))
                {
                    string_delete(str);
                    string_delete(field);
                    return FALSE;
                }
            }
            else if(vector_size(argv) == 3) {
                str = STRING_NEW(string_c_str(vector_item(argv, 2)));
            }
            else {
                break;
            }

            char* str2 = string_c_str(str);

            char* regex = string_c_str(vector_item(argv, 1));
            char* regex2 = regex;
            regex_t* reg = hash_item(gRegexs, regex);
            int r;

            if(reg == NULL) {
                OnigErrorInfo err_info;
                r = onig_new(&reg, regex2
                        , regex2 + strlen((char*)regex2)
                        , ONIG_OPTION_DEFAULT
                        , ONIG_ENCODING_UTF8
                        //, ONIG_SYNTAX_RUBY
                        //, ONIG_SYNTAX_PERL_NG
                        , ONIG_SYNTAX_DEFAULT
                        , &err_info);

                if(r == ONIG_NORMAL) {
                    hash_put(gRegexs, regex, reg);
                }
                else {
                    onig_free(reg);
                }
            }
            else {
                r = ONIG_NORMAL;
            }

            if(r == ONIG_NORMAL) {
                char* p = str2;
                while(1) {
                    OnigRegion* region = onig_region_new();

                    int r2 = onig_match(reg, str2
                       , str2 + strlen(str2)
                       , p
                       , region, ONIG_OPTION_NONE);

                    if(r2 >= 0) {
                        *rcode = 0;

                        if(region->num_regs == 1) {
                            /// マッチした文字列 ///
                            char* tmp = MALLOC(region->end[0]
                                            -region->beg[0] + 1);

                            memcpy(tmp, str2 + region->beg[0]
                             , region->end[0]-region->beg[0]);

                            tmp[region->end[0]
                                - region->beg[0]] = 0;

                            if(!statment_tree_internal_commands_write_nextout(nextout, tmp))
                            {
                                err_msg("signal intrrupt");
                                FREE(tmp);
                                string_delete(field);
                                string_delete(str);
                                onig_region_free(region, 1);
                                return FALSE;
                            }
                            if(!statment_tree_internal_commands_write_nextout(nextout, "\n"))
                            {
                                err_msg("signal intrrupt");
                                FREE(tmp);
                                string_delete(field);
                                string_delete(str);
                                onig_region_free(region, 1);
                                return FALSE;
                            }

                            FREE(tmp);
                        }
                        else {
                            /// グループ化 ///
                            int i;
//printf("(region->num_regs: %d)\n", region->num_regs);
                            for (i=1; i<region->num_regs; i++) {
                                char* tmp = MALLOC(
                                    region->end[i]-region->beg[i] + 1);

                                memcpy(tmp, str2 + region->beg[i]
                                 , region->end[i]-region->beg[i]);

                                tmp[region->end[i]
                                    - region->beg[i]] = 0;

                                if(!statment_tree_internal_commands_write_nextout(nextout, tmp))
                                {
                                    err_msg("signal intrrupt");
                                    FREE(tmp);
                                    string_delete(field);
                                    string_delete(str);
                                    onig_region_free(region, 1);
                                    return FALSE;
                                }

                                if(i==region->num_regs-1) {
                                    if(!statment_tree_internal_commands_write_nextout(nextout, "\n"))
                                    {
                                        err_msg("signal intrrupt");
                                        FREE(tmp);
                                        string_delete(field);
                                        string_delete(str);
                                        onig_region_free(region, 1);
                                        return FALSE;
                                    }
                                }
                                else {
                                    if(!statment_tree_internal_commands_write_nextout(nextout, string_c_str(field)))
                                    {
                                        err_msg("signal intrrupt");
                                        FREE(tmp);
                                        string_delete(field);
                                        string_delete(str);
                                        onig_region_free(region, 1);
                                        return FALSE;
                                    }
                                }
//printf("(tmp: %s)\n", tmp);

                                FREE(tmp);
                            }
                        }

                        p = str2 + region->end[0];
                    }
                    else {
//printf("(検索失敗)\n");
                        p = p + 1;
                        /*
                        /// 検索失敗したら抜ける ///
                        onig_region_free(region, 1);
                        break;
                        */
                    }

                    onig_region_free(region, 1);

                    /// 最後まで行ったら抜ける ///
                    if(p >= str2 + strlen(str2)) {
//printf("(最後まで行った)\n");
                        break;
                    }
                }
            }
            else {
                err_msg("scan: invalid regex");

                string_delete(str);

                string_delete(field);
                return FALSE;
            } 
            string_delete(field);
            string_delete(str);
            }
            break;

        case kRead: {
            BOOL all = FALSE;
            int i;
            for(i=0; i<vector_size(argv); i++) {
                char* opt = string_c_str(vector_item(argv, i));
                if(strcmp(opt, "-a") == 0) {
                    all = TRUE;
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
            }

            if(vector_size(argv) == 2) {
                char* fname = string_c_str(vector_item(argv, 1));

                if(strcmp(fname, "STDOUT")==0 
                        || strcmp(fname, "STDERR")==0)
                {
                    err_msg("read: can't read from stdout or stderr");
                    return FALSE;
                }

                if(all) {
                    int fd;
                    if(strcmp(fname, "STDIN") == 0) {
                        fd = nextin;
                    }
                    else {
                        fd = (int)hash_item(gFileHandles, fname);
                        if(fd == 0) {   
                            fd = open(fname, O_RDONLY);

                            if(fd < 0) {
                                char buf[64];
                                sprintf(buf, "read: can't open file %s", fname);
                                err_msg(buf);
                                return FALSE;
                            }
                            
                            hash_put(gFileHandles, fname, (void*)fd);
                        }
                    }
                    
                    string_obj* str = STRING_NEW("");
                    if(!statment_tree_internal_commands_read_nextin(fd, str))
                    {
                        err_msg("interrupt");
                        string_delete(str);
                        if(fd != nextin) xclose(fd);
                        return FALSE;
                    }
                    else {
                        if(!statment_tree_internal_commands_write_nextout(nextout, string_c_str(str)))
                        {
                            err_msg("signal intrrupt");
                            string_delete(str);
                            if(fd != nextin) xclose(fd);
                            return FALSE;
                        }
                        *rcode = 0;

                        if(fd != nextin) xclose(fd);
                    }

                    string_delete(str);
                }
                else {
                    int fd;
                    if(strcmp(fname, "STDIN") == 0) {
                        fd = nextin;
                    }
                    else {
                        fd = (int)hash_item(gFileHandles, fname);
                        if(fd == 0) {   
                            fd = open(fname, O_RDONLY);

                            if(fd < 0) {
                                char buf[64];
                                sprintf(buf, "read: can't open file %s", fname);
                                err_msg(buf);
                                return FALSE;
                            }
                            
                            hash_put(gFileHandles, fname, (void*)fd);
                        }
                    }
                    
                    string_obj* str = STRING_NEW("");
                    int result 
                        = statment_tree_internal_commands_read_nextin_oneline(fd, str, gLineField);
//printf("kRead middle tcgetpgrp %d\n", tcgetpgrp(0));

                    if(result == 0) { 
                        if(!statment_tree_internal_commands_write_nextout(nextout, string_c_str(str)))
                        {
                            err_msg("signal intrrupt");
                            string_delete(str);

                            return FALSE;
                        }
                        *rcode = 0;
                    }
                    else if(result == -1) {
                        err_msg("interrupt");
                        string_delete(str);
                        if(fd != nextin) {
                            xclose(fd);
                        }

                        return FALSE;
                    }
                    else {
                        if(fd != nextin) {
                            xclose(fd);
                        }
                    }

                    string_delete(str);
                }
            }
            }
//printf("kRead end tcgetpgrp %d\n", tcgetpgrp(0));
            break;

        case kClose: {
            BOOL quiet = FALSE;
            int i;
            for(i=0; i<vector_size(argv); i++) {
                char* opt = string_c_str(vector_item(argv, i));
                if(strcmp(opt, "-q") == 0) {
                    quiet = TRUE;
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
            }
            if(vector_size(argv) == 2) {
                char* fname = string_c_str(vector_item(argv, 1));

                int fd = (int)hash_item(gFileHandles, fname);
                if(fd > 2) {
                    hash_erase(gFileHandles, fname);
                    xclose(fd);

                    *rcode = 0;

                    if(!quiet) {
                        string_obj* str = STRING_NEW("");
                        string_put(str, fname);
                        string_push_back(str, " is closed\n");

                        if(!statment_tree_internal_commands_write_nextout(nextout, string_c_str(str)))
                        {
                            err_msg("signal intrrupt");
                            string_delete(str);
                            return FALSE;
                        }

                        string_delete(str);
                    }
                }
                else if(!quiet) {
                    string_obj* str = STRING_NEW("");
                    string_put(str, fname);
                    string_push_back(str, " is not opend\n");

                    if(!statment_tree_internal_commands_write_nextout(nextout, string_c_str(str)))
                    {
                        err_msg("signal intrrupt");
                        string_delete(str);
                        return FALSE;
                    }

                    string_delete(str);
                }
            }
            }
            break;

        case kCd:
            if(vector_size(argv) == 1) {
                char* path = getenv("HOME");
                char cwd[PATH_MAX];
                char path2[PATH_MAX];

                mygetcwd(cwd);
                if(correct_path(cwd, path, path2)) {
                    struct stat stat_;
                    if(stat(path2, &stat_) == 0) {
                        if(S_ISDIR(stat_.st_mode)) {
                            setenv("PWD", path2, 1);
                            if(chdir(path2) >= 0) {
                                *rcode = 0;
                            }
                        }
                    }
                }
            }
            else if(vector_size(argv) == 2) {
                char* path = string_c_str(vector_item(argv, 1));
                char cwd[PATH_MAX];
                char path2[PATH_MAX];

                mygetcwd(cwd);
                if(correct_path(cwd, path, path2)) {
                    struct stat stat_;
                    if(stat(path2, &stat_) == 0) {
                        if(S_ISDIR(stat_.st_mode)) {
                            setenv("PWD", path2, 1);
                            if(chdir(path2) >= 0) {
                                *rcode = 0;
                            }
                        }
                    }
                }
            }
            break;

        case kSelector: {
            static int scrolltop = 0;
            static int cursor = 0;
            enum eKanjiCode code = gKanjiCode;
            int line_field = FALSE;
            BOOL multiple = FALSE;
            BOOL option_r_exist = FALSE;

            int i;
            for(i=0; i<vector_size(argv); i++) {
                char* arg = string_c_str(vector_item(argv,i));
                if(strcmp(arg, "-l") == 0) {
                    line_field = TRUE;
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
                else if(strcmp(arg, "-r") == 0) {
                    option_r_exist = TRUE;
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
                else if(strcmp(arg, "-c") == 0 
                    && i+1 < vector_size(argv)) 
                {
                    cursor = atoi(string_c_str(vector_item(argv, i+1)));
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
                else if(strcmp(arg, "-t") == 0 
                    && i+1 < vector_size(argv)) 
                {
                    scrolltop = atoi(string_c_str(vector_item(argv, i+1)));
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
                else if(strcmp(arg, "-m") == 0) {
                    multiple = TRUE;
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
                else if(strcmp(arg, "-s") == 0) {
                    code = kSjis;
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
                else if(strcmp(arg, "-e") == 0) {
                    code = kEucjp;
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
                else if(strcmp(arg, "-w") == 0) {
                    code = kUtf8;
                    string_delete(vector_item(argv, i));
                    vector_erase(argv, i);
                    i--;
                    continue;
                }
            }

            if(!option_r_exist) {
                scrolltop = 0;
                cursor = 0;
            }

            string_obj* str = STRING_NEW("");
            if(!statment_tree_internal_commands_read_nextin(nextin,str))
            {
                err_msg("interrupt");
                string_delete(str);
                return FALSE;
            }
            string_obj* result = STRING_NEW("");
            statment_tree_internal_commands_selector(str,result
                    , &scrolltop, &cursor, code, multiple, gLineField);

            *rcode = 0;

            if(!statment_tree_internal_commands_write_nextout(nextout, string_c_str(result)))
            {
                err_msg("sinal intrrupt");

                string_delete(result);
                string_delete(str);

                return FALSE;
            }
            if(line_field) {
                if(!statment_tree_internal_commands_write_nextout(nextout, "\n"))
                {
                    err_msg("signal intrrupt");
                    string_delete(result);
                    string_delete(str);

                    return FALSE;
                }
            }

            string_delete(result);
            string_delete(str);
            }
            break;

        case kMax: {
            BOOL new_line = FALSE;

            int l;
            for(l=0; l<vector_size(argv); l++) {
                char* arg = string_c_str(vector_item(argv, l));

                if(strcmp(arg, "-l") == 0) {
                    new_line = TRUE;
                    string_delete(vector_item(argv, l));
                    vector_erase(argv, l);
                    l--;
                    continue;
                }
            }

            if(vector_size(argv) == 3) {
                int left = atoi(string_c_str(vector_item(argv, 1)));
                int right = atoi(string_c_str(vector_item(argv, 2)));

                if(left > right) {
                    if(!statment_tree_internal_commands_write_nextout(nextout, string_c_str(vector_item(argv, 1))))
                    {
                        err_msg("signal intrrupt");
                        return FALSE;
                    }
                    if(new_line) {
                        if(!statment_tree_internal_commands_write_nextout(nextout, "\n"))
                        {
                            err_msg("signal intrrupt");
                            return FALSE;
                        }
                    }
                }
                else {
                    if(!statment_tree_internal_commands_write_nextout(nextout, string_c_str(vector_item(argv, 2))))
                    {
                        err_msg("signal intrrupt");
                        return FALSE;
                    }
                    if(new_line) {
                        if(!statment_tree_internal_commands_write_nextout(nextout, "\n"))
                        {
                            err_msg("signal intrrupt");
                            return FALSE;
                        }
                    }
                }
            }
            }
            break;

        case kMin: {
            BOOL new_line = FALSE;

            int l;
            for(l=0; l<vector_size(argv); l++) {
                char* arg = string_c_str(vector_item(argv, l));

                if(strcmp(arg, "-l") == 0) {
                    new_line = TRUE;
                    string_delete(vector_item(argv, l));
                    vector_erase(argv, l);
                    l--;
                    continue;
                }
            }

            if(vector_size(argv) == 3) {
                int left = atoi(string_c_str(vector_item(argv, 1)));
                int right = atoi(string_c_str(vector_item(argv, 2)));

                if(left < right) {
                    if(!statment_tree_internal_commands_write_nextout(nextout, string_c_str(vector_item(argv, 1))))
                    {
                        err_msg("signal intrrupt");
                        return FALSE;
                    }
                    if(new_line) {
                        if(!statment_tree_internal_commands_write_nextout(nextout, "\n"))
                        {
                            err_msg("signal intrrupt");
                            return FALSE;
                        }
                    }
                }
                else {
                    if(!statment_tree_internal_commands_write_nextout(nextout, string_c_str(vector_item(argv, 2))))
                    {
                        err_msg("signal intrrupt");
                        return FALSE;
                    }
                    if(new_line) {
                        if(!statment_tree_internal_commands_write_nextout(nextout, "\n"))
                        {
                            err_msg("signal intrrupt");
                            return FALSE;
                        }
                    }
                }
            }
            }
            break;
        

        case kExtName: 
            if(vector_size(argv) == 2) {
                char path[PATH_MAX];
                extname(path, string_c_str(vector_item(argv, 1)));
                if(!statment_tree_internal_commands_write_nextout(nextout, path))
                {
                    err_msg("signal intrrupt");
                    return FALSE;
                }
            }
            break;

        case kParentName:
            if(vector_size(argv) == 2) {
                char path[PATH_MAX];
                parentname(path, string_c_str(vector_item(argv, 1)));

                if(!statment_tree_internal_commands_write_nextout(nextout, path))
                {
                    err_msg("signal intrrupt");
                    return FALSE;
                }
            }
            break;

        case kNoExtName:
            if(vector_size(argv) == 2) {
                char path[PATH_MAX];
                noextname(path, string_c_str(vector_item(argv, 1)));
                if(!statment_tree_internal_commands_write_nextout(nextout, path))
                {
                    err_msg("signal intrrupt");
                    return FALSE;
                }
            }
            break;

        case kPopd: {
            string_obj* pwd = vector_pop_back(gDirStack);

            if(pwd) {
                char* path = string_c_str(pwd);
                char cwd[PATH_MAX];
                char path2[PATH_MAX];

                mygetcwd(cwd);
                if(correct_path(cwd, path, path2)) {
                    struct stat stat_;
                    if(stat(path2, &stat_) == 0) {
                        if(S_ISDIR(stat_.st_mode)) {
                            setenv("PWD", path2, 1);
                            if(chdir(path2) >= 0) {
                                *rcode = 0;
                            }
                        }
                    }
                }
                string_delete(pwd);
            }
            }
            break;

        case kPushd:
            vector_add(gDirStack, STRING_NEW(getenv("PWD")));
            *rcode = 0;
            break;

        default: {
            fprintf(stderr, "unexpected err. illegal command(%s)\n", string_c_str(vector_item(argv, 0)));
            exit(1);
        }
    }

    return TRUE;
}

////////////////////////////////////////////////////////////////////////
/// statment_treeの子関数
////////////////////////////////////////////////////////////////////////
static void statment_tree_redirect(sCommand* command, int * nextin
    , int* nextout, int* nexterr, int pipein, int pipeout, int pipeerr
    , int globalin, int globalout)
{
    BOOL err_and_output = FALSE;
    int k;
    for(k=0; k<vector_size(command->mRedirects); k++) {
        sRedirect* redirect = (sRedirect*)vector_item(command->mRedirects, k);
        
        if(redirect->mType == kRedirectGPipeOut) {
            if(*nextout != 1 && *nextout != pipeout) xclose(*nextout);
            *nextout = globalout;
        }
        else if(redirect->mType == kRedirectGPipeIn) {
            if(*nextin != 0 && *nextin != pipein) xclose(*nextin);
            *nextin = globalin;
        }
        else if(redirect->mType == kRedirectErrAndOutput) {
            err_and_output = TRUE;
        }
        else {
            mode_t mode;
            switch(redirect->mType) {
                case kRedirectInput:
                    mode = O_RDONLY;
                    break;
                
                case kRedirectOverwrite:
                    mode = O_RDWR | O_CREAT | O_TRUNC;
                    break;
                    
                case kRedirectAppend:
                    mode = O_RDWR | O_CREAT | O_APPEND;
                    break;
            }
            
            int openfd = open(string_c_str(redirect->mFName), mode, 0666);
            if(openfd < 0) {
                fprintf(stderr, "open %s is err1\n"
                            , string_c_str(redirect->mFName));
                exit(1);
            }

            switch(redirect->mType) {
                case kRedirectInput:
                    if(*nextin != 0 && *nextin != pipein) xclose(*nextin);
                    *nextin = openfd;
                    break;
                
                case kRedirectOverwrite:
                    if(redirect->mFd == 1) {
                        if(*nextout != 1 && *nextout != pipeout) 
                            xclose(*nextout);
                        *nextout = openfd;
                    }
                    else {
                        if(*nexterr != 2 && *nexterr != pipeerr)
                            xclose(*nexterr);
                        
                        *nexterr = openfd;
                    }
                    break;
                    
                case kRedirectAppend:
                    if(redirect->mFd == 1) {
                        if(*nextout != 1 && *nextout != pipeout) 
                            xclose(*nextout);
                        *nextout = openfd;
                    }
                    else {
                        if(*nexterr != 2 && *nexterr != pipeerr)
                            xclose(*nexterr);
                        
                        *nexterr = openfd;
                    }
                    break;
            }
        }
    }
    
    if(err_and_output) {
        *nexterr = *nextout;
    }
}

static void statment_tree_exec_cprog(sCommand* command, int nextin, int nextout, int nexterr, int globalin, int globalout, vector_obj* argv)
{
    /// リダイレクト ///
    int err_and_output = FALSE;
    int k;
    for(k=0; k<vector_size(command->mRedirects); k++) {
        sRedirect* redirect = (sRedirect*)
                vector_item(command->mRedirects, k);

        if(redirect->mType == kRedirectGPipeOut) {
             if(nextout != 1) xclose(nextout);
             nextout = globalout;
        }
        else if(redirect->mType == kRedirectGPipeIn) {
            if(nextin != 0) xclose(nextin);
            nextin = globalin;
        }
        else if(redirect->mType == kRedirectErrAndOutput) {
            err_and_output = TRUE;
        }
        else {
            mode_t mode;
            switch(redirect->mType) {
                case kRedirectInput:
                    mode = O_RDONLY;
                    break;
                
                case kRedirectOverwrite:
                    mode = O_RDWR | O_CREAT | O_TRUNC;
                    break;
                    
                case kRedirectAppend:
                    mode = O_RDWR | O_CREAT | O_APPEND;
                    break;
            }

            int openfd = open(string_c_str(redirect->mFName), mode, 0666);
            if(openfd < 0) {
                fprintf(stderr, "open %s is err2\n"
                                , string_c_str(redirect->mFName));
                exit(1);
            }
            
            switch(redirect->mFd) {
            case 0:
                if(nextin != 0) xclose(nextin);
                nextin = openfd;
                break;
                
            case 1:
                if(nextout != 1) xclose(nextout);
                nextout = openfd;
                break;
                
            case 2:
                if(nexterr != 2) xclose(nexterr);
                nexterr = openfd;
                break;
            }
        }
    }

    /// パイプ ///
//printf("argv %s\n", string_c_str(vector_item(argv, 0)));
//printf("nextin %d\n", nextin);
//printf("nextout %d\n", nextout);
    if(err_and_output) {
        if(dup2(nextout, 2) < 0) {
            err_msg("dup2 1");
            exit(1);
        }
    }
    
    if(nextin != 0) {
        if(dup2(nextin, 0) < 0) {
            char buf[32];
            sprintf(buf, "dup2 3 nextin %d", nextin);
            err_msg(buf);
            exit(1);
        }
        xclose(nextin);
    }
    
    if(nextout != 1) {
        if(dup2(nextout, 1) < 0) {
            err_msg("dup2 4");
            char buf[128];
            sprintf(buf, "(%d)", nextout);
            err_msg(buf);
            exit(1);
        }
        
        xclose(nextout);
    }
    
    if(nexterr != 2) {
        if(dup2(nexterr, 2) < 0) {
            err_msg("dup2 5");
            exit(1);
        }
        
        xclose(nexterr);
    }

    /// 親が開いたパイプなど関係のないファイルディスクリプタを
    //閉じておく ///
    if(gPipeFDs[0] != -1) close(gPipeFDs[0]);
    //if(gPipeFDs[1] != -1) close(gPipeFDs[1]);  // これはいらないはず
    
    hash_it* it = hash_loop_begin(gFileHandles);
    while(it) {
        int fd = (int)hash_loop_item(it);
        if(fd != 0 && fd != 1 && fd != 2) (void)close(fd);
        it = hash_loop_next(it);
    }

    /// char** に変換 ///
    char** argv2 = (char**)malloc(sizeof(char*)*(vector_size(argv)+1));

    int i;
    for(i=0; i<vector_size(argv); i++) {
        string_obj* item = (string_obj*)vector_item(argv, i);
        argv2[i] = string_c_str(item);
    }
    argv2[i] = NULL;

    /// exec ///
    execvp(argv2[0], argv2);
    //fprintf(stderr, "exec('%s') error\n", argv2[0]);
    kill(getppid(), SIGUSR1);
    exit(1);
}

static BOOL statment_tree_global_pipe(sCommand* command, int* globalin, int* globalout)
{
    int k;
    for(k=0; k<vector_size(command->mRedirects); k++) {
        sRedirect* redirect = (sRedirect*)vector_item(command->mRedirects, k);

        if(redirect->mType == kRedirectGPipeOut) {
            if(gPipeFDs[0] != -1) {
                xclose(gPipeFDs[0]);
            }
            if(gPipeFDs[1] != -1) {
                xclose(gPipeFDs[1]);
            }

            if(pipe(gPipeFDs) < 0) {
                fprintf(stderr, "pipe() in global pipe\n");
                exit(1);
            }

            *globalout = gPipeFDs[1];
            gPipeFDs[1] = -1;
        }
        else if(redirect->mType == kRedirectGPipeIn) {
            if(gPipeFDs[0] == -1) {
                err_msg("a global pipe input doesn't exist");
                return FALSE;
            }
            else {
                *globalin = gPipeFDs[0];
                gPipeFDs[0] = -1;
            }
        }
    }
    
    return TRUE;
 }
/*
static BOOL statment_tree_global_pipe(sCommand* command, int* globalin, int* globalout)
{
    int k;
    for(k=0; k<vector_size(command->mRedirects); k++) {
        sRedirect* redirect = (sRedirect*)vector_item(command->mRedirects, k);

        if(redirect->mType == kRedirectGPipeOut) {
            if(gPipeFDs[0] != -1 && gPipeFDs[1] != -1) {
                *globalout = gPipeFDs[1];
            }
            else if(gPipeFDs[0] != -1 && gPipeFDs[1] == -1) {
                xclose(gPipeFDs[0]);
                gPipeFDs[0] = -1;

                if(pipe(gPipeFDs) < 0) {
                    fprintf(stderr, "pipe\n");
                    exit(1);
                }
                *globalout = gPipeFDs[1];
            }
            else if(gPipeFDs[0] == -1 && gPipeFDs[1] == -1) {
                if(pipe(gPipeFDs) < 0) {
                    fprintf(stderr, "pipe\n");
                    exit(1);
                }
                *globalout = gPipeFDs[1];
            }
        }
        else if(redirect->mType == kRedirectGPipeIn) {
            if(gPipeFDs[0] == -1) {
                err_msg("global pipe input doesn't exist");
                return FALSE;
            }
            else {
                if(gPipeFDs[1] != -1) {
                    xclose(gPipeFDs[1]);
                    gPipeFDs[1] = -1;
                }

                *globalin = gPipeFDs[0];
                gPipeFDs[0] = -1;
            }
        }
        else if(redirect->mType == kRedirectGPipeCopyIn) {
            if(gPipeFDs[0] == -1) {
                err_msg("global pipe input doesn't exist");
                return FALSE;
            }
            else {
                if(gPipeFDs[1] != -1) {
                    xclose(gPipeFDs[1]);
                    gPipeFDs[1] = -1;
                }

                int pipe_fd1[2];
                if(pipe(pipe_fd1) < 0) {
                    fprintf(stderr, "pipe\n");
                    exit(1);
                }
                int pipe_fd2[2];
                if(pipe(pipe_fd2) < 0) {
                    fprintf(stderr, "pipe\n");
                    exit(1);
                }
                
                string_obj* str = STRING_NEW("");
                while(1) {
                    char buf[BUFSIZ];
                    int r;
                    r = read(gPipeFDs[0], buf, BUFSIZ);

                    if(r < 0) {
                        fprintf(stderr, "read\n");
                        exit(1);
                    }
                    if(r == 0) {
                        break;
                    }
                    buf[r] = 0;

                    string_push_back(str, buf);
                }
                xclose(gPipeFDs[0]);
                
                char* p = string_c_str(str);
                
                if(write(pipe_fd1[1], p, strlen(p)) < 0)
                {
                    perror("write");
                    exit(1);
                }
                xclose(pipe_fd1[1]);
                
                if(write(pipe_fd2[1], p, strlen(p)) < 0) {
                    perror("write");
                    exit(1);
                }
                xclose(pipe_fd2[1]);
                
                *globalin = pipe_fd1[0];
                
                gPipeFDs[0] = pipe_fd2[0];

                string_delete(str);
            }
        }
    }
    
    return TRUE;
 }
*/

/*
    pipein 最初のプログラムの入力
    pipeout 最後のプログラムの出力
    pipeerr 全体のプログラムのエラー出力
*/

static BOOL statment_tree_external_command(sStatment* statment, vector_obj* argv, int nextin, int nextout, int nexterr, int pipefds[2], pid_t* pgroup, int pipeout, sCommand* command, int globalin, int globalout, vector_obj* cprogs)
{
    char* argv0 = string_c_str(vector_item(argv, 0));

    /// fork ///
//printf("fork nextin %d nextout %d\n", nextin, nextout);
//printf("2.pipefds[0] %d pipefds[1] %d\n", pipefds[0], pipefds[1]);
    pid_t pid = fork();
    if(pid < 0) {
        err_msg("fork");
        exit(1);
    }

    /// 子プロセスの処理 ///
    if(pid == 0) {
        /// 親が開いた関係のないパイプは閉じておく ///
        if(pipefds[0] != -1) { close(pipefds[0]); }
        if(pipefds[1] != -1) { close(pipefds[1]); }

        /// 一つ目のプロセスは-1。
        if(*pgroup == -1) {
            *pgroup = getpid();
        }

        if(gAppType != kATOptC) {
            if(setpgid(0, *pgroup) < 0) {
                /// 失敗したら現在のプロセスがプロセスリーダー
                *pgroup = getpid();
                if(setpgid(0, *pgroup) < 0) {
                    perror("setpgid child");
                    exit(1);
                }

                /// そのことを親に通知
                int snd_pid = *pgroup;
                if(msgsnd(gMsgId, &snd_pid, sizeof(int), 0) < 0) 
                {
                    perror("msgsnd");
                    exit(1);
                }
            }

            if((gAppType == kATCursesApp 
                    || gAppType == kATConsoleApp)
                && !statment->mBackground 
                && pipeout == STDOUT_FILENO) 
            {
                if(tcsetpgrp(0, *pgroup) < 0) {
                    // 失敗したら現在のプロセスを
                    // プロセスグループリーダーにする
                    *pgroup = getpid();
                    if(setpgid(0, *pgroup) < 0) {
                        perror("setpgid child 2");
                        exit(1);
                    }
                    if(tcsetpgrp(0, *pgroup) < 0) {
                        perror("tcsetpgrp(child)");
                        exit(1);
                    }

                    /// そのことを親に通知
                    int snd_pid = *pgroup;
                    if(msgsnd(gMsgId, &snd_pid, sizeof(int), 0) < 0) 
                    {
                        perror("msgsnd");
                        exit(1);
                    }
                }
            }
        }

        kitutuki_restore_signal_default();
        sigchld_block(0);

        statment_tree_exec_cprog(command, nextin, nextout
                                , nexterr
                                , globalin, globalout, argv);
    }
    /// 親プロセスの処理 ///
    else {
        vector_add(cprogs
            , sChildProgram_new(pid, string_c_str(statment->mTitle)));

        if(*pgroup == -1) *pgroup = pid;

#if defined(__CYGWIN7__)
        if((gAppType == kATCursesApp 
                || gAppType == kATConsoleApp)
            && !statment->mBackground 
            && pipeout == STDOUT_FILENO) 
        {
            if(tcsetpgrp(0, *pgroup) < 0) {
                perror("tcsetpgrp(parent)");
                exit(1);
            }
        }
#endif

        /// 子供からのプロセスグループリーダの変更の通知があれば
        /// を受け取る
        int pid2;
        if(msgrcv(gMsgId, &pid2, sizeof(int), 0, IPC_NOWAIT|MSG_NOERROR) 
            == sizeof(int)) 
        {
            *pgroup = (pid_t)pid2;
            pid = pid2;
        }

#if defined(__LINUX__) || defined(__FREEBSD__)
        if(gAppType != kATOptC) (void)setpgid(pid, *pgroup);
        if((gAppType == kATCursesApp 
                || gAppType == kATConsoleApp)
            && !statment->mBackground 
            && pipeout == STDOUT_FILENO) 
        {
            (void)tcsetpgrp(0, *pgroup);
        }
#endif
    }
}

static BOOL statment_tree(sStatment* statment, BOOL* last_command_is_inner, int pipeout, int pipein, int pipeerr, char* title, int* rcode, pid_t* pgroup, vector_obj* cprogs, BOOL finaly_exec, BOOL in_loop) 
{
//printf("statment_tree: pipein %d piptout %d\n", pipein, pipeout);

    int nextin = pipein;
    int nextout = 1;
    int nexterr;
    
    int globalin;           // グローバルパイプ
    int globalout;
    int pipefds[2] = { -1, -1 };

    int j;

    for(j=0; j<vector_size(statment->mCommands); j++) {
        sCommand* command = vector_item(statment->mCommands, j);

        if(statment->mBackground && !gJobControl) {
            err_msg("parse: can't make process background using -c");
            return FALSE;
        }
        
        // argvを引数の削除とかできるようにコピーしておく
        vector_obj* argv = VECTOR_NEW(10);
        int ii;
        for(ii=0; ii<vector_size(command->mArgs); ii++) {
#if defined(HAVE_ICONV_H) && !defined(__CYGWIN7__)
            if(gTerminalKanjiCode != gKanjiCode) {
                char* arg = string_c_str(vector_item(command->mArgs, ii));
                size_t output_size = strlen(arg) * MB_CUR_MAX +1;
                char* output = MALLOC(output_size);

                if(kanji_convert(arg, output, output_size
                    , gTerminalKanjiCode, gKanjiCode) < 0)
                {
                    vector_add(argv, STRING_NEW(
                        string_c_str(vector_item(command->mArgs, ii))));
                }
                else {
                    vector_add(argv, STRING_NEW(output));
                }

                FREE(output);
            }
            else {
                vector_add(argv, STRING_NEW(
                string_c_str(vector_item(command->mArgs, ii))));
            }
#else
            vector_add(argv, STRING_NEW(
                string_c_str(vector_item(command->mArgs, ii))));
#endif
        }

        strcpy(gFName, string_c_str(statment->mFName));
        gLine = statment->mLine;

        BOOL first_program = j == 0;
        BOOL last_program = j == (vector_size(statment->mCommands)-1);

        if(!last_program) {
//printf("!last_program\n");
            if(pipe(pipefds) < 0) {
                err_msg("pipe");
                exit(1);
            }
//printf("pipefds[0] %d pipefds[1] %d\n", pipefds[0], pipefds[1]);
            nextout = pipefds[1];
            pipefds[1] = -1;
        }
        else {
//printf("last_program\n");
//printf("pipeout %d nextout %d\n", nextout, pipeout);
            nextout = pipeout;

            pipefds[0] = -1;
            pipefds[1] = -1;
        }
//printf("--- argv[0] %s\n", string_c_str(vector_item(argv, 0)));
//printf("nextin %d nextout %d\n", nextin, nextout);
        
        nexterr = pipeerr;
        globalin = -1;
        globalout = -1;
        
        if(!statment_tree_global_pipe(command, &globalin, &globalout)) 
        {
            for(ii=0; ii<vector_size(argv); ii++) {
                string_delete(vector_item(argv, ii));
            }
            vector_delete(argv);
            return FALSE;
        }

        char* arg0 = string_c_str(vector_item(argv, 0));
        sFunction* fun = hash_item(gFuncs, arg0);
//printf("1.pipefds[0] %d pipefds[1] %d\n", pipefds[0], pipefds[1]);
        sInnerCommand* inner_fun = hash_item(gInnerCommands, arg0);

        /// ローカル変数代入 ///
        char* equalp = strstr(arg0, "=");
        if(equalp) {
            char* var_name = MALLOC(equalp - arg0 + 1);
            string_obj* value = STRING_NEW(equalp + 1);

            memcpy(var_name, arg0, equalp - arg0);
            var_name[equalp - arg0] = 0;

            hash_obj* stack = vector_item(gStackFrame, vector_size(gStackFrame)-1);

            string_obj* var = hash_item(stack, var_name);
            
            if(var) {
                string_put(var, string_c_str(value));
                string_delete(value);
            }
            else {
                hash_put(stack, var_name, value);
            }

            *rcode = 0;

            FREE(var_name);
        }

        /// ユーザーコマンド ///
        else if(fun) {
            *last_command_is_inner = TRUE;

            statment_tree_redirect(command, &nextin, &nextout, &nexterr
                                    , pipein, pipeout, pipeerr
                                    , globalin, globalout);


            /// 引数名が指定されていなければスタックフレームを増やす ///
            vector_obj* fun_argv_before;
            vector_obj* fun_argv;
            if(strcmp(string_c_str(fun->arg_name), "") == 0) {
                /// 前の引数を保存 ///
                fun_argv_before = hash_item(gArrays, "ARGV");

                /// 引数を渡す ///
                int k;
                fun_argv = VECTOR_NEW(5);
                for(k=1; k<vector_size(argv); k++) {
                    char* value = string_c_str(vector_item(argv, k));
                    vector_add(fun_argv, STRING_NEW(value));
                }
                
                hash_put(gArrays, "ARGV", fun_argv);
                update_ary_env("ARGV");

                vector_add(gStackFrame, HASH_NEW(10));
            }
            /// 引数名が指定されていなければスタックフレームは増やさない ///
            else {
                /// 前の引数を保存 ///
                fun_argv_before = hash_item(gArrays
                                        , string_c_str(fun->arg_name));

                /// 引数を渡す ///
                int k;
                fun_argv = VECTOR_NEW(5);
                for(k=1; k<vector_size(argv); k++) {
                    char* value = string_c_str(vector_item(argv, k));
                    vector_add(fun_argv, STRING_NEW(value));
                }
                
                hash_put(gArrays, string_c_str(fun->arg_name), fun_argv);
                update_ary_env(string_c_str(fun->arg_name));
            }

            /// 実行 ///
            *rcode = run(fun->statments, title
                    , nextout, nextin, nexterr, finaly_exec, in_loop);

            if(strcmp(string_c_str(fun->arg_name), "") == 0) {
                /// スタックを消す ///
                hash_obj* stack = vector_pop_back(gStackFrame);

                hash_it* it = hash_loop_begin(stack);
                while(it != NULL) {
                    string_delete(hash_loop_item(it));
                    it = hash_loop_next(it);
                }
                hash_delete(stack);

                /// 引数を消す ///
                int k;
                for(k=0; k<vector_size(fun_argv); k++) {
                    string_delete(vector_item(fun_argv, k));
                }
                vector_delete(fun_argv);

                hash_erase(gArrays, "ARGV");

                /// 前の引数を元に戻す ///
                if(fun_argv_before) {
                    hash_put(gArrays, "ARGV", fun_argv_before);
                    update_ary_env("ARGV");
                }
            }
            else {
                /// 引数を消す ///
                int k;
                for(k=0; k<vector_size(fun_argv); k++) {
                    string_delete(vector_item(fun_argv, k));
                }
                vector_delete(fun_argv);

                hash_erase(gArrays, string_c_str(fun->arg_name));

                /// 前の引数を元に戻す ///
                if(fun_argv_before) {
                    hash_put(gArrays, string_c_str(fun->arg_name)
                                , fun_argv_before);
                    update_ary_env(string_c_str(fun->arg_name));
                }
            }

            /// 終了コード ///
            char buf2[256];
            sprintf(buf2, "%d", *rcode);
            kitutuki_set_local_var("RCODE", buf2);

            if(*rcode < 0) {
                for(ii=0; ii<vector_size(argv); ii++) {
                    string_delete(vector_item(argv, ii));
                }
                vector_delete(argv);
                return FALSE;
            }
        }

        /// 外部登録の内部コマンド ///
        else if(inner_fun) {
            *last_command_is_inner = TRUE;
            
            statment_tree_redirect(command, &nextin, &nextout, &nexterr
                                    , pipein, pipeout, pipeerr
                                    , globalin, globalout);

            *rcode = 1; // 初期値は1。コマンドが成功したら0を入れる
/*
#if defined(__LINUX__)
            if((gAppType == kATCursesApp 
                    || gAppType == kATConsoleApp)
                && !statment->mBackground )
            {
if(tcgetpgrp(0) < 0) {
    switch(errno) {
        case EBADF:
            puts("EBADF");
            break;

        case EINVAL:
            puts("EINVAL");
            break;

        case ENOTTY:
            puts("ENOTTY");
            break;

        case EPERM:
            puts("EPERM");
            break;

        default:
            puts("default");
            break;
    }
}

                if(tcsetpgrp(0, getpid()) < 0) {
                    switch(errno) {
                        case EBADF:
                            puts("EBADF");
                            break;

                        case EINVAL:
                            puts("EINVAL");
                            break;

                        case ENOTTY:
                            puts("ENOTTY");
                            break;

                        case EPERM:
                            puts("EPERM");
                            break;

                        default:
                            puts("default");
                            break;
                    }
                    perror("tcsetpgrp inner command");
                    exit(1);
                }
            }
#endif
*/

            BOOL gpipe_in = FALSE;
            int l;
            for(l=0; l<vector_size(command->mRedirects); l++) {
                sRedirect* r = vector_item(command->mRedirects, l);
                if(r->mType == kRedirectGPipeIn) {
                    gpipe_in = TRUE;
                }
            }

            BOOL r = inner_fun->mFun(rcode, argv
                        , nextout, nextin, nexterr
                        , title, !first_program || gpipe_in);

            char buf[256];
            sprintf(buf, "%d", *rcode);
            kitutuki_set_local_var("RCODE", buf);
            if(!r) {
                for(ii=0; ii<vector_size(argv); ii++) {
                    string_delete(vector_item(argv, ii));
                }
                vector_delete(argv);
                return FALSE;
            }
        }

        /// 内部コマンド ///
        else if(command->mKind != kCommand) {
//printf("command->mKind %d %s\n", command->mKind, gStatmentKindStrs[command->mKind]);
            *last_command_is_inner = TRUE;
            
            statment_tree_redirect(command, &nextin, &nextout, &nexterr
                                    , pipein, pipeout, pipeerr
                                    , globalin, globalout);

            *rcode = 1; // 初期値は1。コマンドが成功したら0を入れる
/*
#if defined(__LINUX__)
            if((gAppType == kATCursesApp 
                    || gAppType == kATConsoleApp)
                && !statment->mBackground )
            {
if(tcgetpgrp(0) < 0) {
    switch(errno) {
        case EBADF:
            puts("EBADF");
            break;

        case EINVAL:
            puts("EINVAL");
            break;

        case ENOTTY:
            puts("ENOTTY");
            break;

        case EPERM:
            puts("EPERM");
            break;

        default:
            puts("default");
            break;
    }
}

                if(tcsetpgrp(0, getpid()) < 0) {
                    switch(errno) {
                        case EBADF:
                            puts("EBADF");
                            break;

                        case EINVAL:
                            puts("EINVAL");
                            break;

                        case ENOTTY:
                            puts("ENOTTY");
                            break;

                        case EPERM:
                            puts("EPERM");
                            break;

                        default:
                            puts("default");
                            break;
                    }
                    perror("tcsetpgrp inner command");
                    exit(1);
                }
            }
#endif
*/
            BOOL gpipe_in = FALSE;
            int l;
            for(l=0; l<vector_size(command->mRedirects); l++) {
                sRedirect* r = vector_item(command->mRedirects, l);
                if(r->mType == kRedirectGPipeIn) {
                    gpipe_in = TRUE;
                }
            }
            BOOL r = statment_tree_internal_commands(command, rcode
                    , argv, nextout, nextin, nexterr, j, cprogs
                    , title, !first_program || gpipe_in);

            char buf[256];
            sprintf(buf, "%d", *rcode);
            kitutuki_set_local_var("RCODE",  buf);
            if(!r) {
                for(ii=0; ii<vector_size(argv); ii++) {
                    string_delete(vector_item(argv, ii));
                }
                vector_delete(argv);
                return FALSE;
            }
        }

        /// 外部コマンド ///
        else {
            *last_command_is_inner = FALSE;

            statment_tree_external_command(statment, argv, nextin, nextout, nexterr, pipefds, pgroup, pipeout, command, globalin, globalout, cprogs);
        }

        if(nextin != 0 && nextin != pipein && nextin != globalin) {
            xclose(nextin);
        }
        if(nextout != 1 && nextout != pipeout && nextout != globalout) {
            xclose(nextout);
        }
        if(nexterr != 2 && nexterr != pipeerr && nexterr != nextout) {
            xclose(nexterr);
        }
        if(globalin != -1) {
            xclose(globalin);
        }
        if(globalout != -1) {
            xclose(globalout);
        }
        
        nextin = pipefds[0];
//printf("nextin %d\n", nextin);

        for(ii=0; ii<vector_size(argv); ii++) {
            string_delete(vector_item(argv, ii));
        }
        vector_delete(argv);
    }

    return TRUE;
}

void run_wait_cprogs(vector_obj* cprogs, sStatment* statment, BOOL last_command_is_inner, int* rcode, int pgroup, BOOL err_exist)
{
    if(vector_size(cprogs) > 0) {       // プロセスがあった
        /// - バックグラウンド - ///
        if(statment->mBackground) {
            sJob* job = sJob_new();

            job->mPGroup = pgroup;
            sChildProgram* cprog = vector_item(cprogs, 0);
            string_put(job->mName, string_c_str(cprog->mTitle));

            int k;
            for(k=0; k<vector_size(cprogs); k++) {
                sChildProgram* cprog = vector_item(cprogs, k);
                vector_add(job->mPIDs, (void*)cprog->mPid);
                sChildProgram_delete(cprog);
            }

            vector_add(gJobs, job);
        }

        /// - フォアグラウンド - ///
        else {
OTHERCPROG_START:
null_fun();
            int l;
            for(l=0; l<vector_size(gOtherCProgs); l++) {
                sOtherCProg* other_cprog = vector_item(gOtherCProgs, l);
                
                int k;
                for(k=0; k<vector_size(cprogs); k++) {
                    sChildProgram* cprog = vector_item(cprogs, k);
                    
                    if(cprog->mPid == other_cprog->mPid) {
                        int status = other_cprog->mStatus;
                        
                        if(WIFSIGNALED(status)) {
                            *rcode = WTERMSIG(status);

                            char buf[256];
                            sprintf(buf, "%d", *rcode);
                            kitutuki_set_local_var("RCODE", buf);
                        }
                        /// 終了した場合 ///
                        else if(WIFEXITED(status)) {
                            *rcode = WEXITSTATUS(status);

                            char buf[256];
                            sprintf(buf, "%d", *rcode);
                            kitutuki_set_local_var("RCODE", buf);
                        }
                        /// シグナルの配送により停止した場合 ///
                        else if(WIFSTOPPED(status)) {
                            int sig = WSTOPSIG(status);

                            if(sig == SIGSTOP) {
                                sJob* job = sJob_new();


                                job->mPGroup = other_cprog->mPGroup;
                                string_put(job->mName,
                                        string_c_str(cprog->mTitle));
                                vector_add(job->mPIDs, (void*)cprog->mPid);
                                vector_add(gJobs, job);
                            }
                        }

                        sChildProgram_delete(cprog);
                        sOtherCProg_delete(other_cprog);
                        
                        vector_erase(gOtherCProgs, l);
                        vector_erase(cprogs, k);
                        
                        goto OTHERCPROG_START;
                    }
                }
            }

            /// kitutukiがSIGINTを食らっていたら
            /// 子プロセス全部にSIGINTを送っておく
            /// (前面にkitutukiがあったら子プロセスにはSIGINT
            //  が送られないため)
            if(err_exist) {
                int k;
                for(k=0; k<vector_size(cprogs); k++) {
                    sChildProgram* cprog = vector_item(cprogs, k);

                    kill(cprog->mPid, SIGINT);
                }
            }

            sigchld_block(0);       // ブロック解除
WAIT_START:
            while(vector_size(cprogs) > 0) {
                /// 子からのpgroupの変更の通知を受ける ///
                int pid2;
                if(msgrcv(gMsgId, (int*)&pid2, sizeof(int), 0, IPC_NOWAIT) 
                        == sizeof(int)) 
                {
                    pgroup = (pid_t)pid2;
                }

                pid_t pid;
                int status = 0;
                pid = waitpid(-1, &status, WUNTRACED);

                if(pid < 0) {
                    switch(errno) {
                        case EINTR:
                            goto WAIT_START;

                        case ECHILD:
                            goto WAIT_END;

                        case EINVAL: 
                            goto WAIT_END;

                        default:
                            goto WAIT_END;
                    }
                }
                else if(pid == 0) {
                    goto WAIT_START;
                }
                
                int k;
                sChildProgram* prog = NULL;
                for(k=0; k<vector_size(cprogs); k++) {
                    sChildProgram* p = 
                            (sChildProgram*) vector_item(cprogs, k);
                    
                    if(p->mPid == pid) {
                        prog = p;
                        break;
                    }
                }
                
                if(prog == NULL) {
                    vector_add(gOtherCProgs, sOtherCprog_new(pid, status
                                                , pgroup));
                }
                
                /// シグナルの配送により終了した場合 ///
                else if(WIFSIGNALED(status)) {
                    sChildProgram_delete(prog);
                    vector_erase(cprogs, k);

                    if(last_command_is_inner == FALSE) {
                        *rcode = WTERMSIG(status);

                        char buf[256];
                        sprintf(buf, "%d", *rcode);
                        kitutuki_set_local_var("RCODE", buf);
                    }
                }
                /// 終了した場合 ///
                else if(WIFEXITED(status)) {
                    sChildProgram_delete(prog);
                    vector_erase(cprogs, k);

                    if(last_command_is_inner == FALSE
                        && *rcode != SIGINT) 
                    {
                        *rcode = WEXITSTATUS(status);

                        char buf[256];
                        sprintf(buf, "%d", *rcode);
                        kitutuki_set_local_var("RCODE", buf);
                    }
                }
                /// シグナルの配送により停止した場合 ///
                else if(WIFSTOPPED(status)) {
                    sJob* job = sJob_new();

                    job->mPGroup = pgroup;
                    string_put(job->mName, string_c_str(prog->mTitle));
                    vector_add(job->mPIDs, (void*)prog->mPid);
                    vector_add(gJobs, job);
                
                    sChildProgram_delete(prog);
                    vector_erase(cprogs, k);

                    if(gAppType == kATCursesApp 
                        || gAppType == kATConsoleApp) 
                    {
                        job->mTty = MALLOC(sizeof(struct termios));
                        tcgetattr(STDIN_FILENO, job->mTty);
                    }
                }
            }

WAIT_END:

            if(gAppType == kATCursesApp || gAppType == kATConsoleApp) {
                //mreset_tty();
                /// kitutukiを端末の前面に出す ///
                if(tcsetpgrp(0, getpid()) < 0) {
                    err_msg("tcsetpgrp(kitutuki)");
                    exit(1);
                }
            }
        }
    }
}

/// pipeout: プログラムの出力先
/// pipein: プログラムの入力先
/// finaly_exec: パイプの最後でプログラムを実行したときforkせずにexecするかどうか
// -1が返ったらエラー処理する必要がある
static int run(sStatments* statments, char* title, int pipeout, int pipein, int pipeerr, BOOL finaly_exec, BOOL in_loop)
{
    int rcode = 0;

    int i;
    for(i=0; i<vector_size(statments->mStatments); i++) {
        sStatment* statment = vector_item(statments->mStatments, i);
        gLine = statment->mLine;
        if(statment->mNotEvaled) {
            string_obj* cmdline = STRING_NEW("");
            string_obj* cmdline2 = STRING_NEW("");

            gLine = statment->mLine;
            strcpy(gFName, string_c_str(statment->mFName));
            //sprintf(gFName, "%s %d: ", string_c_str(statment->mFName), gLine);

            int r;
            if(expand_macro) {
                int r = expand_macro(string_c_str(statment->mNotEvaled)
                        , cmdline);
            }
            else {
                r = 1;
                string_put(cmdline, string_c_str(statment->mNotEvaled));
            }

            if(r) {
                if(expand_env(string_c_str(cmdline), cmdline2)) {
                    sStatments* estatments = STATMENTSLIST_NEW();

                    //gLine = 1;
                    vector_obj* defined_funcs = VECTOR_NEW(20);
                    if(parse(string_c_str(cmdline2), gFName, estatments, defined_funcs)) 
                    {
                        parse_update_func(defined_funcs);
                        vector_delete(defined_funcs);

                        rcode = run(estatments, title, pipeout, pipein
                    , pipeerr, FALSE, in_loop);

                        if(rcode < 0) {
                            sStatments_delete(estatments);

                            string_delete(cmdline);
                            string_delete(cmdline2);
                            return -1;
                        }
                    }
                    else {
                        int i;
                        for(i=0; i<vector_size(defined_funcs); i++) {
                            sFunction_delete(vector_item(defined_funcs, i));
                        }
                        vector_delete(defined_funcs);

                        sStatments_delete(estatments);

                        string_delete(cmdline);
                        string_delete(cmdline2);
                        return -1;
                    }

                    sStatments_delete(estatments);
                }
                else {
                    string_delete(cmdline);
                    string_delete(cmdline2);
                    return -1;
                }
            }
            else {
                string_delete(cmdline);
                string_delete(cmdline2);
                return -1;
            }

            string_delete(cmdline);
            string_delete(cmdline2);
        }
        else {
            sigchld_block(1);
            list_clear(gSigChldRecieved);

            vector_obj* cprogs = VECTOR_NEW(10);

            pid_t pgroup = -1;
            BOOL last_command_is_inner;

            int r = statment_tree(statment, &last_command_is_inner
                          , pipeout, pipein, pipeerr
                          , title, &rcode, &pgroup, cprogs
                          , finaly_exec 
                                && i==vector_size(statments->mStatments)-1
                          , in_loop);
            /// プロセス回収 ///
            run_wait_cprogs(cprogs, statment, last_command_is_inner
                            , &rcode, pgroup, r == FALSE);

            vector_delete(cprogs);

            /// ランタイムエラー ///
            if(r == FALSE) {
                return -1;
            }
        }

        /// 行末の処理 ///
        if(gSigUser) {
            gSigUser = FALSE;
            err_msg("command not found\n");
            return -1;
        }
        else if(in_loop && (rcode == SIGINT || gSaphireSigInt)
            || !in_loop && gSaphireSigInt) 
        {
            gSaphireSigInt = FALSE;
            return SIGINT;
        }
        else if(statment->mTerminated == kTOrOr && rcode == 0) {
            while(i<vector_size(statments->mStatments) 
                    && statment->mTerminated != kTNormal) 
            {
                i++;
                statment = vector_item(statments->mStatments, i);
            }
        }
        else if(statment->mTerminated == kTAndAnd && rcode != 0) {
            while(i<vector_size(statments->mStatments) 
                    && statment->mTerminated != kTNormal) 
            {
                i++;
                statment = vector_item(statments->mStatments, i);
            }
        }
    }

    /// 空文の場合 ///
    if(vector_size(statments->mStatments) == 0) {
        /// CTRL-Cが押された ///
        if(gSaphireSigInt) {
            gSaphireSigInt = FALSE;
            return SIGINT;
        }
        /*
        else {
            err_msg("command is empty");
            return -1;           // 空文は偽
        }
        */
    }

    return rcode;
}

//////////////////////////////////////////////////////////////////////
// ファイル読み込み
//////////////////////////////////////////////////////////////////////
// -1ならエラーメッセージがgErrMsgに入っているので出力してください

int kitutuki_load(char* fname, int pipeout, int pipein, int pipeerr)
{
    string_obj* str = STRING_NEW("");

    int f = open(fname, O_RDONLY);
    if(f < 0) {
        string_delete(str);
        return -1;
    }

    char buf[BUFSIZ];
    while(1) {
        int size = read(f, buf, BUFSIZ-1);
        if(size == 0) {
            break;
        }
        if(size < 0) {
            string_delete(str);
            close(f);
            return -1;
        }

        buf[size] = 0;

        string_push_back(str, buf);
    }
    close(f);

    char fname2[FNAME_MAX];
    strcpy(fname2, gFName);
    int line = gLine;

    int rcode = kitutuki_shell(string_c_str(str), fname, pipeout, pipein, pipeerr);

    strcpy(gFName, fname2);
    gLine = line;

    string_delete(str);

    return rcode;
}


//////////////////////////////////////////////////////////////////////
// シェル実行
//////////////////////////////////////////////////////////////////////

// -1ならエラーメッセージがgErrMsgに入っているので出力してください

int kitutuki_shell(char* command, char* title, int pipeout, int pipein, int pipeerr)
{
    gSaphireSigInt = FALSE;
    sStatments* statments = STATMENTSLIST_NEW();

    if(title == NULL) 
        strcpy(gFName, command);
    else 
        strcpy(gFName, title);

    gLine = 1;

    string_obj* str = STRING_NEW("");
    if(!read_expand_brace_expansion(command, str, " ")) {
        sStatments_delete(statments);
        string_delete(str);
        return -1;
    }

    vector_obj* defined_funcs = VECTOR_NEW(20);
    if(!parse(string_c_str(str), gFName, statments, defined_funcs)) {
        int i;
        for(i=0; i<vector_size(defined_funcs); i++) {
            sFunction_delete(vector_item(defined_funcs, i));
        }
        vector_delete(defined_funcs);
        sStatments_delete(statments);
        string_delete(str);
        return -1;
    }
    else {
        parse_update_func(defined_funcs);
        vector_delete(defined_funcs);

        string_delete(str);

        /// マクロ展開のタイトルを得る ///
        string_obj* title2 = STRING_NEW("");
        if(title) {
            if(!expand_macro || !expand_macro(title, title2)) {
                string_put(title2, title);
            }
        }
        else {
            if(!expand_macro || !expand_macro(command, title2)) {
                string_put(title2, command);
            }
        }

        /// 実行 ///
        int rcode = run(statments, string_c_str(title2)
                    , pipeout, pipein, pipeerr, gAppType == kATOptC
                    , FALSE);

        /// 解放 ///
        string_delete(title2);
        sStatments_delete(statments);

        return rcode;
    }
}

// -1ならエラーメッセージがgErrMsgに入っているので出力してください

int kitutuki_shell3(string_obj* result, char* command, char* title)
{
    int pipefds[2];

    if(pipe(pipefds) < 0) {
        err_msg("pipe");
        exit(1);
    }

    int rcode = kitutuki_shell(command, title, pipefds[1], STDIN_FILENO, STDERR_FILENO);
    close(pipefds[1]);

    if(rcode < 0) {
        close(pipefds[0]);
        return -1;
    }

    char buf[BUFSIZ];
    while(2) {
        int size = read(pipefds[0], buf, BUFSIZ-1);
        if(size == 0) {
            break;
        }
        if(size < 0) {
            fprintf(stderr, "expand command err(read)");
            close(pipefds[0]);
            return -1;
        }

        buf[size] = 0;

        string_push_back(result, buf);
    }
    close(pipefds[0]);

    return rcode;
}

//////////////////////////////////////////////////////////////////////
// ジョブの数
//////////////////////////////////////////////////////////////////////
int kitutuki_job_num()
{
    return vector_size(gJobs);
}

//////////////////////////////////////////////////////////////////////
// ジョブのタイトルを返す
//////////////////////////////////////////////////////////////////////
char* kitutuki_job_title(int num)
{
    if(num>=0 && num < vector_size(gJobs)) {
        sJob* job = (sJob*)vector_item(gJobs, num);
        return string_c_str(job->mName);
    }
    else {
        return NULL;
    }
}

//////////////////////////////////////////////////////////////////////
// 全てのジョブをkillする
//////////////////////////////////////////////////////////////////////
void kitutuki_kill_all_jobs()
{
    int i;
    for(i=0; i<vector_size(gJobs); i++) {
        sJob* job = (sJob*)vector_item(gJobs, i);

        int j;
        for(j=0; j<vector_size(job->mPIDs); j++) {
            int pid = (int)vector_item(job->mPIDs, j);
            kill(pid, SIGKILL);
        }

        sJob_delete(job);
    }

    vector_clear(gJobs);
}

//////////////////////////////////////////////////////////////////////
// ジョブを消す
//////////////////////////////////////////////////////////////////////
void kitutuki_kill_job(int num)
{
    if(num>=0 && num < vector_size(gJobs)) {
        sJob* job = (sJob*)vector_item(gJobs, num);

        int i;
        for(i=0; i<vector_size(job->mPIDs); i++) {
            int pid = (int)vector_item(job->mPIDs, i);
            kill(pid, SIGKILL);
        }

        sJob_delete(job);

        vector_erase(gJobs, num);
    }
}



//////////////////////////////////////////////////////////////////////
// ジョブを前面に出す
//////////////////////////////////////////////////////////////////////
void (*kitutuki_job_done)(int job_num, char* job_title) = NULL;

static BOOL forground_job(int num)
{
    if(num>=0 && num < vector_size(gJobs)) {
        sJob* job = (sJob*)vector_item(gJobs, num);

        if(job->mTty) tcsetattr(STDIN_FILENO, TCSANOW, job->mTty);
        if(tcsetpgrp(0, job->mPGroup) < 0) {
            err_msg("tcsetpgrp failed. use ps fj to find out job process group and use cpg [job id] [pgroup] to change the job process group which kitutuki has.");
            return FALSE;
        }
        
        if(killpg(job->mPGroup, SIGCONT) < 0) {
            int i;
            for(i=0; i<vector_size(job->mPIDs); i++) {
                int pid = (int)vector_item(job->mPIDs, i);
                kill(pid, SIGCONT);
            }
        }

        /// wait ///
        int return_code = 0;

        while(vector_size(job->mPIDs) > 0) {
            pid_t pid = (pid_t)vector_item(job->mPIDs, 0);

            int status = 0;
            pid_t return_pid = waitpid(pid, &status, WUNTRACED);

            /// 終了した場合 ///
            if(WIFEXITED(status)) {
                vector_erase(job->mPIDs, 0);

                if(vector_size(job->mPIDs) == 0) {
                    if(kitutuki_job_done) 
                        kitutuki_job_done(num, string_c_str(job->mName));

                    sJob_delete(job);
                    vector_erase(gJobs, num);

                    /// kitutukiを前面に出す ///
                    //mreset_tty();
                    if(tcsetpgrp(0, getpid()) < 0) {
                        perror("tcsetpgrp");
                        exit(1);
                    }

                    return_code = WEXITSTATUS(status);
                    char buf[256];
                    sprintf(buf, "%d", return_code);
                    kitutuki_set_local_var("RCODE", buf);
                    break;
                }
            }
            /// シグナルの配送により終了した場合 ///
            else if(WIFSIGNALED(status)) {
                vector_erase(job->mPIDs, 0);

                if(vector_size(job->mPIDs) == 0) {
                    if(kitutuki_job_done) 
                        kitutuki_job_done(num, string_c_str(job->mName));

                    sJob_delete(job);
                    vector_erase(gJobs, num);

                    /// kitutukiを前面に出す ///
                    //mreset_tty();
                    if(tcsetpgrp(0, getpid()) < 0) {
                        perror("tcsetpgrp");
                        exit(1);
                    }

                    return_code = WTERMSIG(status);
                    char buf[256];
                    sprintf(buf, "%d", return_code);
                    kitutuki_set_local_var("RCODE", buf);
                    break;
                }
            }
            /// シグナルの配送によりフォワグランドジョブが停止した場合 ///
            else if(WIFSTOPPED(status)) {
                if(!job->mTty) {
                    job->mTty = MALLOC(sizeof(struct termios));
                    tcgetattr(STDIN_FILENO, job->mTty);
                }

                /// kitutuki を前面に出す ///
                //mreset_tty();
                if(tcsetpgrp(0, getpid()) < 0) {
                    perror("tcsetpgrp");
                    exit(1);
                }
                break;
            }
        }

        return TRUE;
    }

    err_msg("invalid job number");
    return FALSE;
}

//////////////////////////////////////////////////////////////////////
// ジョブにSIGCONTする
//////////////////////////////////////////////////////////////////////
static void background_job(int num)
{
    if(num>=0 && num < vector_size(gJobs)) {
        sJob* job = (sJob*)vector_item(gJobs, num);
        
        if(killpg(job->mPGroup, SIGCONT) < 0) {
            int pid = (int)vector_item(job->mPIDs, 0);
            kill(pid, SIGCONT);
        }
    }
}

//////////////////////////////////////////////////////////////////////
// バックグラウンドジョブが終了していれば後処理をする
//////////////////////////////////////////////////////////////////////
void kitutuki_wait_background_job()
{
    int i;
while_start:
    for(i=0; i<vector_size(gJobs); i++) {
        sJob* job = (sJob*)vector_item(gJobs, i);
        int j;
        for(j=0; j<vector_size(job->mPIDs); j++) {
            pid_t pid = (pid_t)vector_item(job->mPIDs, j);

            int status;
            pid_t return_pid = waitpid(pid, &status, WNOHANG);

            if(return_pid == pid || return_pid < 0 && errno == ECHILD) {
                vector_erase(job->mPIDs, j);

                if(vector_size(job->mPIDs) == 0) {
                    if(kitutuki_job_done) 
                        kitutuki_job_done(i, string_c_str(job->mName));

                    sJob_delete(job);
                    vector_erase(gJobs, i);

                    goto while_start;
                }
            }
        }
    }
}

/////////////////////////////////////////////////////////////////////
// 内部コマンドの登録
/////////////////////////////////////////////////////////////////////
void kitutuki_add_inner_command(char* name, fInnerCommand fun)
{
    hash_put(gInnerCommands, name, sInnerCommand_new(name, fun));
}

BOOL kitutuki_compile2(char* cmdline, char* fname, sStatments* statments)
{
    vector_obj* defined_funcs = VECTOR_NEW(20);
    if(!parse(cmdline, gFName, statments, defined_funcs)) {
        int i;
        for(i=0; i<vector_size(defined_funcs); i++) {
            sFunction_delete(vector_item(defined_funcs, i));
        }
        vector_delete(defined_funcs);
        return FALSE;
    }
    else {
        parse_update_func(defined_funcs);
        vector_delete(defined_funcs);

        return TRUE;
    }
}

int kitutuki_run(sStatments* statments, char* title, int pipeout, int pipein, int pipeerr)
{
    return run(statments, title, pipeout, pipein, pipeerr
            , gAppType == kATOptC, FALSE);
}

