/*
 *  Copyright (C) 2005 Kouji TAKAO <kouji@netlab.jp>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <string.h>
#include <glib/gi18n.h>

#include "helper.h"
#include "welcome.h"
#include "intergradation.h"
#include "authentication.h"
#include "window.h"
#include "confirm-quit.h"
#include "change-master-password.h"
#include "password-generator.h"
#include "preferences.h"
#include "view-factory.h"

typedef struct {
    gchar *template;
    GType (*get_type)(void);
} get_type_entry_t;

static get_type_entry_t get_types[] = {
    { "welcome",                gpass_welcome_get_type },
    { "intergradation",         gpass_gnome_intergradation_get_type },
    { "authentication",         gpass_authentication_get_type },
    { "window",                 gpass_window_get_type },
    { "confirm-quit",           gpass_confirm_quit_get_type },
    { "change-master-password", gpass_change_master_password_get_type },
    { "preferences",            gpass_preferences_get_type },
};

static GObjectClass *parent_class = NULL;

static void
gpass_view_factory_instance_init(GTypeInstance *instance, gpointer g_class)
{
    GPassViewFactory *self = GPASS_VIEW_FACTORY(instance);
    int i;
    
    self->views = g_hash_table_new(g_str_hash, g_str_equal);
    self->get_types = g_hash_table_new(g_str_hash, g_str_equal);
    for (i = 0; i < G_N_ELEMENTS(get_types); i++) {
        g_hash_table_insert(self->get_types,
                            get_types[i].template, &get_types[i]);
    }
}

static void
finalize_view(gpointer key, gpointer value, gpointer user_data)
{
    g_object_unref(G_OBJECT(value));
}

static void
gpass_view_factory_instance_finalize(GObject *object)
{
    GPassViewFactory *self = GPASS_VIEW_FACTORY(object);
    
    g_hash_table_destroy(self->get_types);
    g_hash_table_foreach(self->views, finalize_view, NULL);
    g_hash_table_destroy(self->views);
    G_OBJECT_CLASS(parent_class)->finalize(object);
}

static void
gpass_view_factory_class_init(gpointer g_class, gpointer g_class_data)
{
    GObjectClass *gobject_class = G_OBJECT_CLASS(g_class);

    parent_class = g_type_class_peek_parent(g_class);
    gobject_class->finalize = gpass_view_factory_instance_finalize;
}

GType
gpass_view_factory_get_type(void)
{
    static GType type = 0;
    if (type == 0) {
        static const GTypeInfo info = {
            sizeof(GPassViewFactoryClass),
            NULL,
            NULL,
            gpass_view_factory_class_init,
            NULL,
            NULL,
            sizeof(GPassViewFactory),
            0,
            gpass_view_factory_instance_init
        };
        type = g_type_register_static(G_TYPE_OBJECT, "GPassViewFactoryType",
                                      &info, 0);
    }
    return type;
}

GError *
gpass_view_factory_create_view(GPassViewFactory *self, const gchar *template,
                               gpointer model, GPassView **view)
{
    GPassView *result;
    get_type_entry_t *entry;
    GError *error = NULL;
    
    result = GPASS_VIEW(g_hash_table_lookup(self->views, template));
    if (result == NULL) {
        entry = g_hash_table_lookup(self->get_types, template);
        if (entry == NULL) {
            g_set_error(&error, 0, 0,_("No such template: %s\n"), template);
            return error;
        }
        result = GPASS_VIEW(g_object_new(entry->get_type(),
                                         "model", model,
                                         "template", template, NULL));
        g_hash_table_insert(self->views, entry->template, result);
    }
    else {
        g_object_set(G_OBJECT(result), "model", model, NULL);
    }
    *view = result;
    return NULL;
}

GError *
gpass_view_factory_destroy_view(GPassViewFactory *self, const gchar *template)
{
    GPassView *view;
    GError *error = NULL;
    
    view = GPASS_VIEW(g_hash_table_lookup(self->views, template));
    if (view == NULL) {
        g_set_error(&error, 0, 0,_("No such template: %s\n"), template);
    }
    else {
        g_object_unref(G_OBJECT(view));
        g_hash_table_remove(self->views, template);
    }
    return error;
}
