/* -*- Mode: C; tab-width: 8; indent-tabs-mode: nil; c-basic-offset: 8 -*- */
/* gdm-info-provider-firewire.c
 *
 * Copyright (C) 2007 David Zeuthen
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * as published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
 * 02111-1307, USA.
 */

#include <config.h>
#include <glib/gi18n.h>
#include "gdm-info-provider.h"

/***************************************** Firewire device *****************************************/

static gboolean 
get_provider_matches (GdmDevice *device)
{
        const char *subsys;
        subsys = gdm_device_get_property_string (device, "info.subsystem");
        if (subsys != NULL && g_ascii_strcasecmp (subsys, "ieee1394") == 0)
                return TRUE;
        return FALSE;
}

static char *
get_icon_name (GdmDevice *device)
{
        return NULL;
}

static char *
get_short_name (GdmDevice *device)
{
        return g_strdup (_("IEEE 1394 Device"));
}

static char *
get_long_name (GdmDevice *device)
{
        return g_strdup (_("IEEE 1394 Device"));
}

static char *
get_vendor (GdmDevice *device)
{
        return g_strdup (gdm_device_get_property_string (device, "ieee1394.vendor"));
}

static char *
get_product (GdmDevice *device)
{
        return g_strdup (gdm_device_get_property_string (device, "ieee1394.product"));
}

#define ADD_SUM(p, key, value)                                                       \
        do {                                                                         \
                if ((value) != NULL) {                                               \
                        p = g_slist_append (p, g_strdup (key));                      \
                        p = g_slist_append (p, (value));                             \
                }                                                                    \
        } while (FALSE)

static GSList *
get_summary (GdmDevice *device)
{
        int vid;
        int pid;
        int hw_ver;
        const char *vendor;
        const char *product;
        const char *device_file;
        guint64 guid;
        GSList *p = NULL;

        vid = gdm_device_get_property_int (device, "ieee1394.vendor_id");
        pid = gdm_device_get_property_int (device, "ieee1394.product_id");
        hw_ver = gdm_device_get_property_int (device, "ieee1394.hardware_version");
        vendor = gdm_device_get_property_string (device, "ieee1394.vendor");
        product = gdm_device_get_property_string (device, "ieee1394.product");
        device_file = gdm_device_get_property_string (device, "ieee1394.device");
        guid = gdm_device_get_property_uint64 (device, "ieee1394.guid");
        
        if (product != NULL)
                ADD_SUM (p, _("Model"), g_strdup (product));
        else if (pid != -1)
                ADD_SUM (p, _("Model"), g_strdup_printf (_("Unknown Model (id = 0x%04x)"), pid));
        
        if (vendor != NULL)
                ADD_SUM (p, _("Vendor"), g_strdup (vendor));
        else if (vid != -1)
                ADD_SUM (
                        p, _("Vendor"), g_strdup_printf (_("Unknown Vendor (id = 0x%04x)"), vid));
        
        if (hw_ver != -1)
                ADD_SUM (p, _("Revision"), g_strdup_printf (_("0x%06x"), hw_ver));
        
        ADD_SUM (p, _("GUID"), g_strdup_printf ("0x%0llx", guid));
        
        ADD_SUM (p, _("Device File"), g_strdup (device_file));
        
        ADD_SUM (p, _("Connection"), g_strdup (_("IEEE 1394 / Firewire / i.Link")));
        
        return p;
}

GdmInfoProviderIface gdm_info_provider_firewire =
{
        .get_provider_matches = get_provider_matches,
        .get_icon_name        = get_icon_name,
        .get_short_name       = get_short_name,
        .get_long_name        = get_long_name,
        .get_vendor           = get_vendor,
        .get_product          = get_product,
        .get_summary          = get_summary,
};

/***************************************** Firewire unit *****************************************/

static gboolean 
get_provider_matches_unit (GdmDevice *device)
{
        const char *subsys;
        subsys = gdm_device_get_property_string (device, "info.subsystem");
        if (subsys != NULL && g_ascii_strcasecmp (subsys, "ieee1394_unit") == 0)
                return TRUE;
        return FALSE;
}

static char *
get_icon_name_unit (GdmDevice *device)
{
        return NULL;
}

static char *
get_short_name_unit (GdmDevice *device)
{
        return g_strdup (_("IEEE 1394 Unit"));
}

static char *
get_long_name_unit (GdmDevice *device)
{
        return g_strdup (_("IEEE 1394 Unit"));
}

static char *
get_vendor_unit (GdmDevice *device)
{
        return NULL;
}

static char *
get_product_unit (GdmDevice *device)
{
        return NULL;
}

#define ADD_SUM(p, key, value)                                                       \
        do {                                                                         \
                if ((value) != NULL) {                                               \
                        p = g_slist_append (p, g_strdup (key));                      \
                        p = g_slist_append (p, (value));                             \
                }                                                                    \
        } while (FALSE)

static GSList *
get_summary_unit (GdmDevice *device)
{
        int specifier_id;
        GSList *p = NULL;

        specifier_id = gdm_device_get_property_int (device, "ieee1394_unit.specifier_id");
        
        ADD_SUM (p, _("Specifier ID"), g_strdup_printf ("0x%0x", specifier_id));
        return p;
}

GdmInfoProviderIface gdm_info_provider_firewire_unit =
{
        .get_provider_matches = get_provider_matches_unit,
        .get_icon_name        = get_icon_name_unit,
        .get_short_name       = get_short_name_unit,
        .get_long_name        = get_long_name_unit,
        .get_vendor           = get_vendor_unit,
        .get_product          = get_product_unit,
        .get_summary          = get_summary_unit,
};

