/**
 * @file meta-presence.c Meta-Presence API
 *
 * Copyright (C) 2004-2006 Christian Hammond.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program; if not, write to the Free
 * Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 * MA  02111-1307  USA
 */
#include "meta-presence.h"
#include "coco.h"
#include "filter.h"

static void
_galagod_dbus_presence_emit_idle_updated(GalagoPresence *presence,
										 gboolean idle,
										 dbus_uint32_t idle_start_time)
{
	DBusMessage *message;
	DBusMessageIter iter;

	g_return_if_fail(presence != NULL);

	message = dbus_message_new_signal(
		galago_object_get_dbus_path(GALAGO_OBJECT(presence)),
		GALAGO_DBUS_PRESENCE_INTERFACE,
		"IdleUpdated");

	dbus_message_iter_init_append(message, &iter);
	dbus_message_iter_append_basic(&iter, DBUS_TYPE_BOOLEAN, &idle);
	dbus_message_iter_append_basic(&iter, DBUS_TYPE_UINT32,  &idle_start_time);

	dbus_connection_send(galago_get_dbus_conn(), message, NULL);
	dbus_message_unref(message);
}

GalagodMetaPresence *
galagod_meta_presence_new(GalagodMetaAccount *meta_account)
{
	GalagodMetaPresence *meta_presence;
	GalagoAccount *main_account;
	char *obj_path;

	g_return_val_if_fail(meta_account != NULL, NULL);

	meta_presence = g_new0(GalagodMetaPresence, 1);
	main_account = galagod_meta_account_get_account(meta_account);
	meta_presence->presence = galago_account_create_presence(main_account);

	obj_path = galagod_presence_generate_path(meta_presence->presence);
	galago_object_set_dbus_path(GALAGO_OBJECT(meta_presence->presence),
								obj_path);
	g_free(obj_path);

	g_object_set_data(G_OBJECT(meta_presence->presence),
					  "meta-presence", meta_presence);

	g_signal_connect(G_OBJECT(meta_presence->presence), "idle_changed",
					 G_CALLBACK(_galagod_dbus_presence_emit_idle_updated),
					 NULL);

	galagod_presence_add_filter(meta_presence->presence);

	return meta_presence;
}

void
galagod_meta_presence_destroy(GalagodMetaPresence *meta_presence)
{
	g_return_if_fail(meta_presence != NULL);

	while (meta_presence->presences != NULL)
	{
		galagod_meta_presence_remove_presence(meta_presence,
			(GalagoPresence *)meta_presence->presences->data);
	}

	galagod_presence_remove_filter(meta_presence->presence);
	g_object_unref(meta_presence->presence);

	g_free(meta_presence);
}

GalagoPresence *
galagod_meta_presence_get_presence(const GalagodMetaPresence *meta_presence)
{
	g_return_val_if_fail(meta_presence != NULL, NULL);

	return meta_presence->presence;
}

GList *
galagod_meta_presence_get_presences(const GalagodMetaPresence *meta_presence)
{
	g_return_val_if_fail(meta_presence != NULL, NULL);

	return meta_presence->presences;
}

void
galagod_meta_presence_add_presence(GalagodMetaPresence *meta_presence,
								   GalagoPresence *presence)
{
	GalagoPresence *main_presence;
	const GList *l;

	g_return_if_fail(meta_presence != NULL);
	g_return_if_fail(presence      != NULL);

	meta_presence->presences = g_list_append(meta_presence->presences,
											 presence);

	g_object_set_data(G_OBJECT(presence), "meta-presence", meta_presence);

	main_presence = galagod_meta_presence_get_presence(meta_presence);

	galago_presence_set_idle(main_presence,
							 galago_presence_is_idle(presence),
							 galago_presence_get_idle_start_time(presence));

	galago_context_push(galagod_get_context());

	for (l = galago_presence_get_statuses(presence); l != NULL; l = l->next)
	{
		GalagoStatus *status = (GalagoStatus *)l->data;
		GalagoStatus *new_status;

		new_status = galago_status_duplicate(status);

		galago_presence_add_status(main_presence, new_status);
	}

	galago_context_pop();

	galagod_presence_add_filter(presence);
}

void
galagod_meta_presence_remove_presence(GalagodMetaPresence *meta_presence,
									  GalagoPresence *presence)
{
	g_return_if_fail(meta_presence != NULL);
	g_return_if_fail(presence      != NULL);

	galagod_presence_remove_filter(presence);

	g_object_set_data(G_OBJECT(presence), "meta-presence", NULL);

	meta_presence->presences = g_list_remove(meta_presence->presences,
											 presence);
}
