#include "config.h"

#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <errno.h>
#include <signal.h>
#include <string.h>
#include <stddef.h>
#include <ftw.h>
#include <cstdlib>
#include <limits.h>

#include "CompressedTable.hpp"
#include "CompressedMagic.hpp"
#include "Compress.hpp"
#include "TransformTable.hpp"

#include <iostream>
#include <vector>
#include <sstream>
#include <string>

using namespace std;

TransformTable		*g_TransformTable;
int			 g_BufferedMemorySize = 100 * 1024;
bool			 g_Debug = false;
CompressedMagic		 g_CompressedMagic;
CompressedTable		 g_CompressedTable;
volatile sig_atomic_t	 g_BreakFlag = 0;
bool			 g_RawOutput = true;
int			 g_BlockSize = 100 * 1024;

void catch_kill(int signum)
{
	g_BreakFlag = 1;
}

static void print_license(void)
{
	printf("%s version %s, Copyright (C) 2007  Milan Svoboda\n", PACKAGE_NAME, PACKAGE_VERSION);
	printf("%s comes with ABSOLUTELY NO WARRANTY;\n", PACKAGE_NAME);
	printf("This is free software, and you are welcome to redistribute it\n");
	printf("under certain conditions;\n\n");
}

static void print_help(void)
{
	print_license();

	printf("Usage: %s [OPTIONS] /input/file\n\n", "fusecompress_offline");

	printf("Input file may also be a directory name. Files in\n");
	printf("specified directory will be processed recursively.\n\n");

	printf("\t-h                   print this help\n");
	printf("\t-v                   print version\n");
	printf("\t-c lzo/bz2/gz/none   choose compression method,\n");
	printf("\t                      if not set decompress only\n");
	printf("\t-b size              block size in kilobytes\n");
	printf("\t-f ext1,ext2,ext3    force files with listed extensions\n");
	printf("\t                      to be always compressed\n");
}

bool prepare(const char *input , struct stat *input_st,
             const char *output, struct stat *output_st)
{
	int r;
	
	r = lstat(input, input_st);
	if (r == -1)
	{
		cerr << "File " << input << " cannot be opened!" << endl;
		cerr << " (" << strerror(errno) << ")" << endl;
		return false;
	}
	
	r = mknod(output, 0600, S_IFREG);
	if (r == -1)
	{
		cerr << "File " << output << " cannot be created!";
		cerr << " (" << strerror(errno) << ")" << endl;
		return false;
	}

	r = lstat(output, output_st);
	if (r == -1)
	{
		cerr << "File " << output << " cannot be opened!";
		cerr << " (" << strerror(errno) << ")" << endl;
		unlink(output);
		return false;
	}
	return true;
}

/**
 * input	- file name of the input file
 * output	- file name of the output file
 */
bool copy(const char *input, struct stat *st, const char *output)
{
	bool		 b = true;
	int		 r;
	int		 rr;
	int		 fd;
	char		*buf = NULL;
	struct stat	 sto;
	Compress	*c = NULL;
	Compress	*o = NULL;

	if (::prepare(input, st, output, &sto) == false)
	{
		return false;
	}

	if (g_RawOutput)
	{
		// Lie about file size to force Compress to
		// use FileRawNormal...
		//
		sto.st_size = 1;
	}

	try {
		buf = new char[g_BlockSize];

		c = new Compress(st);
		o = new Compress(&sto);
	}
	catch (const bad_alloc e)
	{
		delete[] buf;
		delete   c;
		delete   o;

		// Output filename was created, delete it now to
		// clean up.
		//
		unlink(output);
		
		cerr << "No free memory!" << endl;
		cerr << "Try to use smaller block size." << endl;
		return false;	
	}

	fd = c->open(input, O_RDONLY);
	if (fd == -1)
	{
		cerr << "File " << input << " cannot be opened!" << endl;
		cerr << " (" << strerror(errno) << ")" << endl;
		b = false;
		goto out1;
	}

	fd = o->open(output, O_WRONLY);
	if (fd == -1)
	{
		cerr << "File " << output << " cannot be opened!" << endl;
		cerr << " (" << strerror(errno) << ")" << endl;
		b = false;
		goto out2;
	}

	// Update struct stat of the compressed
	// file with real values of the uncompressed file.
	// 
	r = c->getattr(input, st);
	assert (r != -1);

	for (off_t i = 0; i < st->st_size; i += g_BlockSize)
	{
		if (g_BreakFlag)
		{
			cerr << "Interrupted when processing file " << input << endl;
			cerr << "File is left untouched" << endl;
			b = false;
			goto out;
		}
		r = c->read(buf, g_BlockSize, i);
		if (r == -1)
		{
			cerr << "Read failed! (offset: " << i << ", size: " << g_BlockSize << ")" << endl;
			b = false;
			goto out;
		}
		rr = o->write(buf, r, i);
		if (rr != r)
		{
			cerr << "Write failed! (offset: " << i << ", size: " << r << ")" << endl;
			b = false;
			goto out;
		}
	}
out:	o->release(input);
out2:	c->release(output);
	
out1:	delete c;
	delete o;

	delete[] buf;

	if (!b)
		unlink(output);

	return b;
}

inline bool test(const char *input)
{
	FileHeader	 fh;
	bool             fl;

	fl = fh.restore(input);

	if (g_RawOutput && !fl)
	{
		// Decompress files mode turned on and the
		// file is not compressed with FuseCompress.
		//
		return false;
	}

	if (fl)
	{
		// File is compressed with FuseCompress.
		//
		if (fh.type == g_TransformTable->getDefaultIndex())
		{
			// File is compressed with the same compression
			// method.
			//
			cout << "\tAlready compressed with required method!" << endl;
			return false;
		}
		return true;
	}

	// File is not compressed with FuseCompress.
	//
	if (g_CompressedTable.isForcedToCompress(input))
	{
		cout << "\tForced to compress file!" << endl;
		return true;
	}

	if (g_CompressedMagic.isNativelyCompressed(input))
	{
		cout <<"\tFile contains compressed data!" << endl;
		return false;
	}

	return true;
}

int compress(const char *input, const struct stat *st, int mode, struct FTW *n)
{
	if ((mode == FTW_F) && (S_ISREG(st->st_mode)))
	{
		const char	*ext;
		struct stat	 nst;
		stringstream	 tmp;

		cout << "Processing file " << input << endl;

		if (!test(input))
			return 0;
	
		// Create temp filename, make sure that
		// extension is preserved...
		//
		ext = strrchr(input, '/');
		ext = strrchr(ext, '.');

		tmp << input << ".__f_tmp_c__" << ext;

		string out(tmp.str());

		if (copy(input, &nst, out.c_str()) == false)
			return -1;

		if (rename(out.c_str(), input) == -1)
		{
			cerr << "Rename(" << out << ", ";
			cerr <<	input << ") failed with:";
			cerr << strerror(errno) << endl;
			return -1;
		}

		if (chmod(input, nst.st_mode) == -1)
		{
			cerr << "Chmod(" << input << ") failed with:";
			cerr << strerror(errno) << endl;
			return -1;
		}
	}
	return 0;
}

int main(int argc, char *argv[])
{
	int             next_option;
	char *		compressor = NULL;
	char  		input[PATH_MAX];
	char *          pinput = input;

	
	const char* const	 short_options = "b:hvc:xf:";

	do {
		next_option = getopt(argc, argv, short_options);
		switch (next_option)
		{
			case 'h':
				print_help();
				exit(EXIT_SUCCESS);

			case 'v':
				print_license();
				exit(EXIT_SUCCESS);

			case 'c':
				compressor = optarg;
				g_RawOutput = false;
				break;

			case 'b':
				g_BlockSize = atoi(optarg) * 1024;
				break;

			case 'f':
				g_CompressedTable.Add(optarg, ",");
				break;

			case -1:
				break;

			default:
				print_help();
				exit(EXIT_FAILURE);
		}
	} while (next_option != -1);

	argc -= optind;
	argv += optind;
	
	if (argc != 1)
	{
	    print_help();
	    exit(EXIT_FAILURE);
	}

	pinput = argv[0];

	if (pinput[0] != '/')
	{
		// Transform relative path to absolute path.
		//
		if (getcwd(input, sizeof(input)) == NULL)
		{
			cerr << "Cannot determine current working directory" << endl;
			exit(EXIT_FAILURE);
		}
		input[strlen(input)] = '/';
		strcpy(input + strlen(input), pinput);
	}
	else
		strcpy(input, pinput);

	g_TransformTable = new (std::nothrow) TransformTable();
	if (!g_TransformTable)
	{
		cerr << "No memory to allocate object of "
		     << "TransformTable class" << endl;
		abort();
	}

	// Set the default transformation as user wanted.
	// 
	if ((compressor) &&
	    (g_TransformTable->setDefault(compressor) == false))
	{
		cerr << "Compressor " << compressor << " not found!" << endl;
		exit(EXIT_FAILURE);
	}

	// Set signal handler to catch SIGINT (CTRL+C).
	//
	struct sigaction setup_kill;
	memset(&setup_kill, 0, sizeof(setup_kill));
	setup_kill.sa_handler = catch_kill;
	sigaction(SIGINT, &setup_kill, NULL);

	// Iterate over directory structure and execute compress
	// for every files there.
	//
	if (nftw(input, compress, 100, FTW_PHYS | FTW_CHDIR))
		exit(EXIT_FAILURE);

	exit(EXIT_SUCCESS);
}

