/************************************

cardinal - a proper cross platform twitter client
Copyright (C) 2009  Chris Fuenty <zimmy@zimmy.co.uk>

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 3 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License along
with this program; if not, write to the Free Software Foundation, Inc.,
51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.

************************************/

#include "infownd.h"

InfoWnd::InfoWnd(user_t tw, QString uname, QWidget *parent)
    :QDialog(parent),
    m_user(tw),
    m_uname(uname)
{
    if (objectName().isEmpty())
        setObjectName(QString::fromUtf8("UserInfoDialog"));
    resize(400, 233);

    m_bird = new QTwitLib();
    connect(m_bird, SIGNAL(OnResponseReceived(Returnables::Response*)), this, SLOT(birdResponseReceived(Returnables::Response*)));
    QSettings settings("Cardinal", "Cardinal");
    settings.beginGroup("Accounts");
    settings.beginGroup(uname);
    token_t token;
    token.token = settings.value("oauth_token").toByteArray();
    token.secret = settings.value("oauth_token_secret").toByteArray();
    settings.endGroup();
    settings.endGroup();
    m_bird->setToken(token);

    m_bird->GetFollowers(tw.id);
    m_bird->GetFriends(tw.id);

    m_profilePics = new ProfilePictureManager(this, true);
    connect(m_profilePics, SIGNAL(pictureRecieved(unsigned long long int, QByteArray)), this,
        SLOT(pictureRecieved(unsigned long long int, QByteArray)));
    m_profilePics->retrievePicture(tw.id, tw.imageURL);

    gridLayout = new QGridLayout(this);
    gridLayout->setObjectName(QString::fromUtf8("gridLayout"));
    tabWidget = new QTabWidget(this);
    tabWidget->setObjectName(QString::fromUtf8("tabWidget"));
    about = new QWidget();
    about->setObjectName(QString::fromUtf8("about"));
    gridLayout_2 = new QGridLayout(about);
    gridLayout_2->setObjectName(QString::fromUtf8("gridLayout_2"));
    labelPixmap = new QLabel(about);
    labelPixmap->setObjectName(QString::fromUtf8("labelPixmap"));
    QSizePolicy sizePolicy(QSizePolicy::Fixed, QSizePolicy::Fixed);
    sizePolicy.setHorizontalStretch(0);
    sizePolicy.setVerticalStretch(0);
    sizePolicy.setHeightForWidth(labelPixmap->sizePolicy().hasHeightForWidth());
    labelPixmap->setSizePolicy(sizePolicy);
    labelPixmap->setMinimumSize(QSize(48, 48));
    labelPixmap->setMaximumSize(QSize(48, 48));
    labelPixmap->setScaledContents(true);

    gridLayout_2->addWidget(labelPixmap, 0, 0, 3, 1);

    labelName = new QLabel(this);
    labelName->setObjectName(QString::fromUtf8("labelName"));
    QSizePolicy sizePolicy1(QSizePolicy::Minimum, QSizePolicy::Preferred);
    sizePolicy1.setHorizontalStretch(0);
    sizePolicy1.setVerticalStretch(0);
    sizePolicy1.setHeightForWidth(labelName->sizePolicy().hasHeightForWidth());
    labelName->setSizePolicy(sizePolicy1);
    QFont font;
    font.setPointSize(12);
    font.setBold(true);
    font.setWeight(75);
    labelName->setFont(font);

    gridLayout_2->addWidget(labelName, 0, 1, 1, 2);

    btnSendDM = new QPushButton(this);
    btnClose = new QPushButton(this);
    btnClose->setObjectName(QString::fromUtf8("btnClose"));
    QSizePolicy sizePolicy2(QSizePolicy::Preferred, QSizePolicy::Fixed);
    sizePolicy2.setHorizontalStretch(0);
    sizePolicy2.setVerticalStretch(0);
    sizePolicy2.setHeightForWidth(btnClose->sizePolicy().hasHeightForWidth());
    btnClose->setSizePolicy(sizePolicy2);

    gridLayout_2->addWidget(btnClose, 0, 3, 2, 1);

    labelScreenName = new QLabel(about);
    labelScreenName->setObjectName(QString::fromUtf8("labelScreenName"));
    sizePolicy1.setHeightForWidth(labelScreenName->sizePolicy().hasHeightForWidth());
    labelScreenName->setSizePolicy(sizePolicy1);

    gridLayout_2->addWidget(labelScreenName, 1, 1, 2, 2);

    btnFollow = new QPushButton(about);
    btnFollow->setObjectName(QString::fromUtf8("pushButton"));
    sizePolicy2.setHeightForWidth(btnFollow->sizePolicy().hasHeightForWidth());
    btnFollow->setSizePolicy(sizePolicy2);

    gridLayout_2->addWidget(btnFollow, 2, 3, 2, 1);

    labelLocation = new QLabel(about);
    labelLocation->setObjectName(QString::fromUtf8("labelLocation"));
    sizePolicy1.setHeightForWidth(labelLocation->sizePolicy().hasHeightForWidth());
    labelLocation->setSizePolicy(sizePolicy1);

    gridLayout_2->addWidget(labelLocation, 3, 1, 2, 2);

    verticalSpacer = new QSpacerItem(20, 49, QSizePolicy::Minimum, QSizePolicy::Expanding);

    gridLayout_2->addItem(verticalSpacer, 4, 0, 3, 1);

    btnBlock = new QPushButton(about);
    btnBlock->setObjectName(QString::fromUtf8("pushButton_2"));
    sizePolicy2.setHeightForWidth(btnBlock->sizePolicy().hasHeightForWidth());
    btnBlock->setSizePolicy(sizePolicy2);

    gridLayout_2->addWidget(btnBlock, 4, 3, 2, 1);

    labelBiography = new QLabel(about);
    labelBiography->setObjectName(QString::fromUtf8("labelBiography"));
    sizePolicy1.setHeightForWidth(labelBiography->sizePolicy().hasHeightForWidth());
    labelBiography->setSizePolicy(sizePolicy1);
    labelBiography->setWordWrap(true);
    gridLayout_2->addWidget(labelBiography, 5, 1, 2, 2);

    btnWebsite = new QPushButton(about);
    btnWebsite->setObjectName(QString::fromUtf8("pushButton_3"));
    sizePolicy2.setHeightForWidth(btnWebsite->sizePolicy().hasHeightForWidth());
    btnWebsite->setSizePolicy(sizePolicy2);

    gridLayout_2->addWidget(btnWebsite, 6, 3, 1, 1);

    verticalSpacer_2 = new QSpacerItem(20, 46, QSizePolicy::Minimum, QSizePolicy::Expanding);

    gridLayout_2->addItem(verticalSpacer_2, 7, 2, 1, 1);

    btnTwitter = new QPushButton(about);
    btnTwitter->setObjectName(QString::fromUtf8("pushButton_4"));
    sizePolicy2.setHeightForWidth(btnTwitter->sizePolicy().hasHeightForWidth());
    btnTwitter->setSizePolicy(sizePolicy2);

    gridLayout_2->addWidget(btnTwitter, 7, 3, 1, 1);
    gridLayout_2->addWidget(btnSendDM, 8, 3, 1, 1);

    verticalSpacer_3 = new QSpacerItem(20, 46, QSizePolicy::Minimum, QSizePolicy::Expanding);

    gridLayout_2->addItem(verticalSpacer_3, 8, 3, 1, 1);

    labelStats = new QLabel(about);
    labelStats->setObjectName(QString::fromUtf8("labelStats"));

    gridLayout_2->addWidget(labelStats, 9, 1, 1, 1);

    QList<user_t> users;

    tabWidget->addTab(about, QString());
    tabFollowers = new QWidget();
    tabFollowers->setObjectName(QString::fromUtf8("tabFollowers"));
    gridLayout_3 = new QGridLayout(tabFollowers);
    gridLayout_3->setObjectName(QString::fromUtf8("gridLayout_3"));
    followers = new UserView(users, tabFollowers);
    followers->setObjectName(QString::fromUtf8("followers"));
    followers->setInformativeText(tr("People who follow %1").arg(tw.screenName));
    gridLayout_3->addWidget(followers, 0, 0, 1, 1);

    tabWidget->addTab(tabFollowers, QString());
    tabFollowing = new QWidget();
    tabFollowing->setObjectName(QString::fromUtf8("tabFollowing"));
    gridLayout_4 = new QGridLayout(tabFollowing);
    gridLayout_4->setObjectName(QString::fromUtf8("gridLayout_4"));
    following = new UserView(users, tabFollowing);
    following->setObjectName(QString::fromUtf8("following"));
    following->setInformativeText(tr("People %1 are following").arg(tw.screenName));
    btnSendDM->setText(tr("Send Message"));
    btnSendDM->setToolTip(tr("Send a Direct Message to %1").arg(tw.screenName));

    connect(followers, SIGNAL(getUserInfo(unsigned long long int)), 
        this, SLOT(userInfo(unsigned long long int)));
    connect(following, SIGNAL(getUserInfo(unsigned long long int)), 
        this, SIGNAL(getUserInfo(unsigned long long int)));
    gridLayout_4->addWidget(following, 0, 0, 1, 1);

    tabWidget->addTab(tabFollowing, QString());
    tabWidget->setTabEnabled(1, false);
    tabWidget->setTabEnabled(2, false);
    
    gridLayout->addWidget(tabWidget, 0, 0, 1, 1);

    connect(btnFollow, SIGNAL(clicked()), this, SLOT(followTriggered()));
	connect(btnBlock, SIGNAL(clicked()), this, SLOT(blockTriggered()));
	connect(btnWebsite, SIGNAL(clicked()), this, SLOT(websiteTriggered()));
	connect(btnTwitter, SIGNAL(clicked()), this, SLOT(twitterTriggered()));
    connect(btnClose, SIGNAL(clicked()), this, SLOT(close()));
    connect(btnSendDM, SIGNAL(clicked()), this, SLOT(sendDM()));

    setWindowTitle(tr("Info On: ") + tw.screenName);

    labelName->setText(tw.name);
    labelScreenName->setText(tw.screenName);
    labelLocation->setText(tw.location);
    labelBiography->setText(tw.profile);
    labelStats->setText(QString::number(tw.followers) + tr(" followers, ") +  QString::number(tw.friends) + tr(" following, ") + QString::number(tw.tweets) + tr(" tweets"));
    if(tw.following == true) {
        btnFollow->setText(tr("Unfollow"));
    } else {
        btnFollow->setText(tr("Follow"));
    }
    btnBlock->setText(QApplication::translate("UserInfoDialog", "Block", 0, QApplication::UnicodeUTF8));
    btnWebsite->setText(QApplication::translate("UserInfoDialog", "Website", 0, QApplication::UnicodeUTF8));
    connect(btnWebsite, SIGNAL(pressed()), this, SLOT(openSite()));
    btnTwitter->setText(QApplication::translate("UserInfoDialog", "Twitter Page", 0, QApplication::UnicodeUTF8));
	QString tempPath = QDir::tempPath();

    tabWidget->setTabText(tabWidget->indexOf(about), tr("About %1").arg(tw.screenName));
    tabWidget->setTabText(tabWidget->indexOf(tabFollowers), tr("Followers"));
    tabWidget->setTabText(tabWidget->indexOf(tabFollowing), tr("Following"));
    btnClose->setText(tr("Close"));
    m_editor = new Editor(this);
    connect(m_editor, SIGNAL(submitDirectMsg(QString, QString, unsigned long long int)), this, SLOT(submitDirectMsg(QString, QString, unsigned long long int)));
}

InfoWnd::~InfoWnd()
{
    delete m_editor;
    delete following;
    delete followers;
    delete m_bird;
}

void InfoWnd::closeEvent(QCloseEvent *event)
{
    Q_UNUSED(event);
    close();
    delete this;
}

void InfoWnd::blockTriggered()
{
	emit blockUser(m_user.id, m_user.screenName);
}

void InfoWnd::followTriggered()
{
	if(m_user.following == true) {
		emit unfollow(m_user.id, m_user.screenName);
	} else {
		emit follow(m_user.id, m_user.screenName);
	}
}

void InfoWnd::sendDM()
{
    m_editor->clear();
    m_editor->setDirectMsgInformation(m_uname, m_user.id);
    m_editor->show();
}

void InfoWnd::twitterTriggered()
{
	QDesktopServices::openUrl(QUrl(tr("http://twitter.com/") + m_user.screenName));
}

void InfoWnd::websiteTriggered()
{
	QDesktopServices::openUrl(QUrl(m_user.url));
}

void InfoWnd::submitDirectMsg(QString account, QString text, unsigned long long int tid)
{
    m_bird->SendDirectMessage(tid, text);
    m_editor->hide();
}

void InfoWnd::pictureRecieved(unsigned long long int id, QByteArray pictureData)
{
    Q_UNUSED(id);
    QPixmap map;
    map.loadFromData(pictureData);
    map = map.scaled(48, 48);
    labelPixmap->setPixmap(map);
}

void InfoWnd::birdResponseReceived(Returnables::Response *res)
{
    switch(res->reqID)
    {
    case Returnables::FOLLOWERS:
        {
            QList<user_t> users;
            Returnables::Followers *follow = static_cast<Returnables::Followers *>(res);
            foreach(Returnables::BasicUserInfoElement *uie, follow->list)
            {
                user_t u;
                u.id = uie->user.id;
                u.imageURL = uie->user.profileImageUrl;
                u.location = uie->user.location;
                u.name = uie->user.name;
                u.screenName = uie->user.screenName;
                u.url = uie->user.url;
                u.followers = uie->user.followersCount;
                u.profile = uie->user.description;
                users.append(u);
            }
            followers->updateUserList(users);
            tabWidget->setTabEnabled(2, true);
            break;
        }
    case Returnables::FRIENDS:
        {
            QList<user_t> users;
            Returnables::Friends *friends = static_cast<Returnables::Friends *>(res);
            foreach(Returnables::BasicUserInfoElement *uie, friends->list)
            {
                user_t u;
                u.id = uie->user.id;
                u.imageURL = uie->user.profileImageUrl;
                u.location = uie->user.location;
                u.name = uie->user.name;
                u.screenName = uie->user.screenName;
                u.url = uie->user.url;
                u.followers = uie->user.followersCount;
                u.profile = uie->user.description;
                users.append(u);
            }
            tabWidget->setTabEnabled(1, true);
            following->updateUserList(users);
            break;
        }
    default:
        break;
    };
}
