/************************************

cardinal - a proper cross platform twitter client
Copyright (C) 2009-2010 Chris Fuenty <zimmy@zimmy.co.uk>

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 3 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License along
with this program; if not, write to the Free Software Foundation, Inc.,
51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.

************************************/

#include "identitymanager.h"

IdentityManager::IdentityManager(bool search)
    :m_search(search)
{
    firstRun = true;
    QSettings settings("Cardinal", "Cardinal");
    m_bird = new QTwitLib();
    connect(m_bird, SIGNAL(OnResponseReceived(Returnables::Response*)), this, SLOT(birdResponseReceived(Returnables::Response*)));
    connect(m_bird, SIGNAL(badLogin()), this, SLOT(badLogin()));
}

IdentityManager::~IdentityManager()
{
    delete m_bird;
    hpTimer->stop();
    delete hpTimer;
}

void IdentityManager::initConnection(QString user)
{
    uname = user;
    emit working(true);
    QSettings settings("Cardinal", "Cardinal");
    settings.beginGroup("Accounts");
    settings.beginGroup(user);
    int sec = settings.value("ptReload").toInt();
    token_t token;
    token.token = settings.value("oauth_token").toByteArray();
    token.secret = settings.value("oauth_token_secret").toByteArray();
    settings.endGroup();
    settings.endGroup();

    /* BUG FIX: Never allow this to be less than 1
        set it to 5 minutes if it is */
    if(sec < 1)
        sec = 5;

    int msec = sec * 60 * 1000;

    hpTimer = new QTimer(this);
    hpTimer->setInterval(msec);
    hpTimer->setSingleShot(false);
    connect(hpTimer, SIGNAL(timeout()), this, SLOT(hpTimeout()));
    hpTimer->start();

    m_bird->setToken(token);
    if(m_search == false)
        m_bird->VerifyCredentials();
}

bool IdentityManager::hasIdentities()
{
    return true;
}

void IdentityManager::submitTweet(QString txt, unsigned long long int id = 0)
{
    if(id == 0)
        m_bird->PostNewStatus(txt, 0, tr("Cardinal"));
    else
        m_bird->PostNewStatus(txt, id, tr("Cardinal"));
    emit hideEditor();
    m_bird->GetUsersTimeline(self.id);
}

void IdentityManager::submitDM(QString txt, unsigned long long int id = 0)
{
    m_bird->SendDirectMessage(id, txt);
    emit hideEditor();
    m_bird->GetSentDirectMessages();
}


void IdentityManager::birdResponseReceived(Returnables::Response *res)
{
    if(!res)
        return;

    switch(res->reqID)
    {
    case Returnables::VERIFY_CREDENTIALS:
    {
        Returnables::VerifyCredentials *self = static_cast<Returnables::VerifyCredentials *>(res);
        this->self.id = self->userExt->user.id;
        this->self.name = self->userExt->user.name;
        this->self.screenName = self->userExt->user.screenName;
        this->self.imageURL = self->userExt->user.profileImageUrl;
        this->self.location = self->userExt->user.location;
        this->self.url = self->userExt->user.url;
        this->self.followers = self->userExt->user.followersCount;
        m_online = true;
        emit isOnline();
        delete self;
        emit working(true);
        reload();
        break;
    }
    case Returnables::FRIENDS_TIMELINE:
    {
        if(m_search == true)
        {
            delete res;
            return;
        }

        Returnables::FriendsTimeline *fTimeline = static_cast<Returnables::FriendsTimeline *>(res);
        showFriendTimeline(fTimeline->list);
        delete fTimeline;
        break;
    }
    case Returnables::USER_TIMELINE:
    {
        if(m_search == true)
        {
            delete res;
            return;
        }

        Returnables::UserTimeline *fTimeline = static_cast<Returnables::UserTimeline *>(res);
        showFriendTimeline(fTimeline->list);
        delete fTimeline;
        break;
    }
    case Returnables::API_REQUESTS:
    {
        Returnables::ApiRequests *apis = static_cast<Returnables::ApiRequests *>(res);
        apiData.remainingHits = apis->remainingHits;
        apiData.limit = apis->hourlyLimit;
        apiData.reset = apis->resetTime;
        apiData.ts = apis->resetTimeSeconds;
        if(apiData.remainingHits > 149 && (apiData.ts - QDateTime::currentDateTime().toTime_t()) <= 1600)
            m_rtCheck.clear();

        emit updateAPIData(apiData);
        delete apis;
        break;
    }
    case Returnables::RECENT_MENTIONS:
    {
        Returnables::RecentMentions *rep = static_cast<Returnables::RecentMentions *>(res);
        showMentions(rep->list);
        delete rep;
        break;
    }
    case Returnables::SEARCH_STRING:
    {
        Returnables::SearchResult *result = static_cast<Returnables::SearchResult *>(res);
        showSearchResults(result->list, result->searchString);
        delete result;
        break;
    }
    case Returnables::USER_DETAILS:
    {
        Returnables::UserDetails *uie = static_cast<Returnables::UserDetails *>(res);
        /* currently, we will only need user details for showing info,
        we'll show the dialog here  --stitch */

        user_t u;
        u.id = uie->userExt->user.id;
        u.following = uie->userExt->details.following;
        u.imageURL = uie->userExt->user.profileImageUrl;
        u.location = uie->userExt->user.location;
        u.name = uie->userExt->user.name;
        u.screenName = uie->userExt->user.screenName;
        u.url = uie->userExt->user.url;
        u.tweets = uie->userExt->details.statusesCount;
        u.friends = uie->userExt->details.friendsCount;
        u.followers = uie->userExt->user.followersCount;
        u.profile = uie->userExt->user.description;
        //u.since = uie->details.createdAt;
        emit showInfoDialog(u);
        delete uie;
        break;
    }
    case Returnables::RECEIVED_DIRECT_MESSAGES:
    {
        if(m_search == true)
        {
            delete res;
            return;
        }

        Returnables::ReceivedDirectMessages *rd = static_cast<Returnables::ReceivedDirectMessages *>(res);
        newDirectMessages(rd->list);
        delete rd;
        break;
    }
    case Returnables::BLOCK_USER:
    {
        Returnables::BlockUser *block = static_cast<Returnables::BlockUser *>(res);
        QMessageBox msg(QMessageBox::Information, tr("Blocked User"), tr("You have blocked ") + block->user->user.screenName, QMessageBox::Ok);
        msg.exec();
        delete block;
        break;
    }
    case Returnables::FRIENDSHIP_EXISTS:
    {
        Returnables::FriendshipExist *fe = static_cast<Returnables::FriendshipExist *>(res);
        delete fe;
        break;
    }
    case Returnables::RETWEET:
    {
        Returnables::RetweetedStatus *re = static_cast<Returnables::RetweetedStatus *>(res);
        emit retweetedByMe(re->id);
        delete re;
        break;
    }
    case Returnables::RETWEETED_BY:
    {
        Returnables::RetweetsForStatus *re = static_cast<Returnables::RetweetsForStatus *>(res);
        showRetweetsForId(re);
        break;
    }
    case Returnables::SINGLE_STATUS:
    {
        if(m_search == true)
        {
            delete res;
            return;
        }

        Returnables::SingleStatus *single = static_cast<Returnables::SingleStatus *>(res);
        showSingleStatus(single);
        delete single;
        break;
    }
    case Returnables::REMOVE_STATUS:
    {
        Returnables::RemoveStatus *remove = static_cast<Returnables::RemoveStatus *>(res);
        emit statusRemoved(remove->status->status.id);
        delete remove;
        break;
    }
    default:
    {
        qWarning() << "A case of " << res->reqID << " was returned by the server; Cardinal does not know what to do. Ignoring.";
    }
    }

}
    
void IdentityManager::showRetweetsForId(Returnables::RetweetsForStatus *retweets)
{
    QList<user_t> users;
    foreach(Returnables::BasicUserInfoElement *uie, retweets->list)
    {
        user_t u;
        u.id = uie->user.id;
        u.imageURL = uie->user.profileImageUrl;
        u.name = uie->user.name;
        u.screenName = uie->user.screenName;
        u.url = uie->user.url;
        u.followers = uie->user.followersCount;
        u.profile = uie->user.description;
        users.append(u);
    }
    emit idsForRetweet(retweets->id, users);

    delete retweets;
}

void IdentityManager::showSingleStatus(Returnables::SingleStatus *status)
{
    tweet_t t;

    QStringList lst = status->status->status.createdAt.split(" ", QString::SkipEmptyParts);
    QString time2 = lst.at(0) + tr(" ") + lst.at(1) + tr(" ") + lst.at(2) + tr(" ") + lst.at(5) + tr(" ") + lst.at(3);
    QDateTime qtime = QDateTime::fromString(time2);
    qtime.setTimeSpec(Qt::UTC);
    t.time = qtime.toTime_t();
    t.id = status->status->status.id;
    t.tweet = status->status->status.text;
    t.uid = status->status->user.id;
    t.userName = status->status->user.name;
    t.screenName = status->status->user.screenName;
    t.reply = status->status->status.inReplyToStatusId;
    t.picUri = status->status->user.profileImageUrl;
    t.useragent = status->status->status.source;
    t.favorite = status->status->status.favorited;
    QString URL = status->status->user.url;
    emit working(false);
    emit addNewTweet(t, THREAD);

    delete status;
}

void IdentityManager::showFriendTimeline(QLinkedList<Returnables::StatusElement*> list)
{
    Returnables::StatusElement *element = NULL;
    while(!list.isEmpty())
    {
        element = list.takeLast();

        tweet_t t;

        QStringList lst = element->status.createdAt.split(" ", QString::SkipEmptyParts);
        QString time2 = lst.at(0) + tr(" ") + lst.at(1) + tr(" ") + lst.at(2) + tr(" ") + lst.at(5) + tr(" ") + lst.at(3);
        QDateTime qtime = QDateTime::fromString(time2);
        qtime.setTimeSpec(Qt::UTC);
        t.time = qtime.toTime_t();
        t.id = element->status.id;
        t.tweet = element->status.text;
        t.uid = element->user.id;
        t.userName = element->user.name;
        t.screenName = element->user.screenName;
        t.reply = element->status.inReplyToStatusId;
        t.picUri = element->user.profileImageUrl;
        t.useragent = element->status.source;
        t.favorite = element->status.favorited;
        QString URL = element->user.url;
        if(m_search == false && m_rtCheck.count() <= 10)
        {
           GetRetweets(t.id);
           m_rtCheck.append(t.id);
        }

        emit addNewTweet(t, TIMELINE_TWEET);
    }
    emit working(false);
    emit showNotification();

    foreach(Returnables::StatusElement *se, list)
    {
        delete se;
    }
}

void IdentityManager::showSearchResults(QLinkedList<Returnables::StatusElement*> list, QString searchStr)
{
    Returnables::StatusElement *element = NULL;
    while(!list.isEmpty())
    {
        element = list.takeLast();
        tweet_t t;
        QStringList lst = element->status.createdAt.split(" ", QString::SkipEmptyParts);
        QString time2 = lst.at(0) + tr(" ") + lst.at(1) + tr(" ") + lst.at(2) + tr(" ") + lst.at(5) + tr(" ") + lst.at(3);
        QDateTime qtime = QDateTime::fromString(time2);
        qtime.setTimeSpec(Qt::UTC);
        t.time = qtime.toTime_t();
        t.id = element->status.id;
        t.tweet = element->status.text;
        t.uid = element->user.id;
        t.userName = element->user.name;
        t.screenName = element->user.screenName;
        t.reply = element->status.inReplyToStatusId;
        t.picUri = element->user.profileImageUrl;
        t.useragent = element->status.source;
        t.categories = addCategoryBySearch(searchStr);
        t.favorite = element->status.favorited;
        QString URL = element->user.url;

        emit addNewTweet(t, SEARCH);
    }
    emit working(false);
    emit showNotification();
}

void IdentityManager::showMentions(QLinkedList<Returnables::StatusElement*> list)
{
    Returnables::StatusElement *element = NULL;
    while(!list.isEmpty())
    {
        element = list.takeLast();
        tweet_t t;
        QStringList lst = element->status.createdAt.split(" ", QString::SkipEmptyParts);
        QString time2 = lst.at(0) + tr(" ") + lst.at(1) + tr(" ") + lst.at(2) + tr(" ") + lst.at(5) + tr(" ") + lst.at(3);
        QDateTime qtime = QDateTime::fromString(time2);
        qtime.setTimeSpec(Qt::UTC);
        t.time = qtime.toTime_t();
        t.id = element->status.id;
        t.tweet = element->status.text;
        t.uid = element->user.id;
        t.userName = element->user.name;
        t.screenName = element->user.screenName;
        t.reply = element->status.inReplyToStatusId;
        t.picUri = element->user.profileImageUrl;
        t.useragent = element->status.source;
        t.favorite = element->status.favorited;
        QString URL = element->user.url;
        emit addNewTweet(t, MENTION);
    }
    emit working(false);
    emit showNotification();
}

void IdentityManager::newDirectMessages(QLinkedList<Returnables::DirectMessageElement *> list)
{
    Returnables::DirectMessageElement *element = NULL;
    while(!list.isEmpty())
    {
        element = list.takeLast();
        tweet_t t;
        QStringList lst = element->headerInfo.createdAt.split(" ", QString::SkipEmptyParts);
        QString time2 = lst.at(0) + tr(" ") + lst.at(1) + tr(" ") + lst.at(2) + tr(" ") + lst.at(5) + tr(" ") + lst.at(3);
        QDateTime qtime = QDateTime::fromString(time2);
        qtime.setTimeSpec(Qt::UTC);
        t.time = qtime.toTime_t();
        t.id = element->headerInfo.id;
        t.tweet = element->headerInfo.text;
        t.uid = element->headerInfo.senderId;
        t.userName = element->sender.name;
        t.screenName = element->sender.screenName;
        t.reply = 0;
        t.favorite = false;
        t.picUri = element->sender.profileImageUrl;
        t.categories.append(tr("Direct Messages"));

        QString URL = element->sender.url;
        emit addNewTweet(t, DIRECT_MESSAGE);
    }

    emit showNotification();
}

void IdentityManager::hpTimeout()
{
    firstRun = false;
    reload();
}

void IdentityManager::Logout()
{
    m_bird->Logout();
}

bool IdentityManager::isFirstRun()
{
    return firstRun;
}

void IdentityManager::reload()
{
    if(m_search == false)
    {
        emit working(true);
        m_bird->GetUsersTimeline(0);
        m_bird->GetFriendsTimeline();
        m_bird->GetReceivedDirectMessages();
        m_bird->GetRecentMentions();
        m_bird->RemainingApiRequests();
        runSearch();
    }
}

void IdentityManager::GetUserDetails(unsigned long long int id)
{
    m_bird->GetUserDetails(id);
}

user_t IdentityManager::identSelf()
{
    return self;
}

void IdentityManager::badLogin()
{
    if(m_search == false)
    {
        QMessageBox msg(QMessageBox::Question, tr("Cardinal - Bad Idenity"), tr("The Identity ") + this->uname + tr(" has an invalid username or password."), QMessageBox::Ok);
        msg.exec();
    }
}

void IdentityManager::getTweetById(unsigned long long int id)
{
    m_bird->GetSingleStatus(id);
}

void IdentityManager::runSearch()
{
    emit working(true);
    QSettings settings("Cardinal", "Cardinal");
    settings.beginGroup("Categories");
    QStringList availableCats = settings.childGroups();
    for (int i = 0; i < availableCats.count(); ++i)
    {
        settings.beginGroup(availableCats[i]);
        QStringList cats = settings.childKeys();
        for (int j = 0; j < cats.count(); ++j)
        {
            if(settings.value(cats[j]) == "Search")
            {
                m_bird->SearchByString(cats[j]);
                break;
            }
        }
        settings.endGroup();
    }
    settings.endGroup();
}

QStringList IdentityManager::addCategoryBySearch(QString search)
{
    QStringList categories;

    QSettings settings("Cardinal", "Cardinal");
    settings.beginGroup("Categories");
    QStringList availableCats = settings.childGroups();
    for (int i = 0; i < availableCats.count(); ++i)
    {
        settings.beginGroup(availableCats[i]);
        QStringList cats = settings.childKeys();
        for (int j = 0; j < cats.count(); ++j)
        {
            if(cats[j] == search && settings.value(cats[j]) == "Search")
            {
                categories.append(availableCats[i]);
                break;
            }
        }
        settings.endGroup();
    }
    settings.endGroup();
    return categories;
}

void IdentityManager::AddFriendship(unsigned long long int id)
{
    m_bird->AddFriendship(QString::number(id), true);
}

void IdentityManager::BlockUser(QString id)
{
    m_bird->BlockUser(id);
}

void IdentityManager::RemoveFriendship(unsigned long long int id)
{
    m_bird->RemoveFriendship(QString::number(id));
}

void IdentityManager::Retweet(unsigned long long int id)
{
    m_bird->Retweet(QString::number(id));
}

void IdentityManager::GetRetweets(unsigned long long int id)
{
    m_bird->RetrieveRetweets(QString::number(id));    
}

void IdentityManager::DeleteStatus(unsigned long long int id)
{
    m_bird->RemoveStatus(id);
}

void IdentityManager::MarkAsFavorite(unsigned long long int id, bool favorite)
{
    if(favorite == true)
        m_bird->AddFavorite(id);
    else
        m_bird->RemoveFavorite(id);
}

void IdentityManager::search(QString search)
{
    m_bird->SearchByString(search);
}
