/************************************

cardinal - a proper cross platform twitter client
Copyright (C) 2009-2010  Chris Fuenty <zimmy@zimmy.co.uk>

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 3 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License along
with this program; if not, write to the Free Software Foundation, Inc.,
51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.

************************************/

#include "AccountManager.h"

AccountEditor::AccountEditor(QString account = QString(), QWidget *parent = 0)
    :QDialog(parent),
    m_account(account)
{
    if (objectName().isEmpty())
        setObjectName(QString::fromUtf8("AccountEditorDialog"));
    resize(400, 175);
    gridLayout_6 = new QGridLayout(this);
    gridLayout_6->setObjectName(QString::fromUtf8("gridLayout_6"));
    tabWidget = new QTabWidget(this);
    tabWidget->setObjectName(QString::fromUtf8("tabWidget"));
    tab_General = new QWidget();
    tab_General->setObjectName(QString::fromUtf8("tab_General"));
    gridLayout = new QGridLayout(tab_General);
    gridLayout->setObjectName(QString::fromUtf8("gridLayout"));
    label_Username = new QLabel(tab_General);
    label_Username->setObjectName(QString::fromUtf8("label_Username"));

    gridLayout->addWidget(label_Username, 0, 0, 1, 1);

    leUserName = new QLineEdit(tab_General);
    leUserName->setObjectName(QString::fromUtf8("leUserName"));
    QSizePolicy sizePolicy(QSizePolicy::MinimumExpanding, QSizePolicy::Fixed);
    sizePolicy.setHorizontalStretch(0);
    sizePolicy.setVerticalStretch(0);
    sizePolicy.setHeightForWidth(leUserName->sizePolicy().hasHeightForWidth());
    leUserName->setSizePolicy(sizePolicy);

    gridLayout->addWidget(leUserName, 0, 1, 1, 1);

    label_Reload = new QLabel(tab_General);
    label_Reload->setObjectName(QString::fromUtf8("label_Reload"));

    gridLayout->addWidget(label_Reload, 1, 0, 1, 1);

    updateTime = new QSpinBox(tab_General);
    updateTime->setObjectName(QString::fromUtf8("updateTime"));
    updateTime->setMinimum(1);
    updateTime->setMaximum(60);
    updateTime->setValue(10);

    gridLayout->addWidget(updateTime, 1, 1, 1, 1);

    verticalSpacer = new QSpacerItem(20, 58, QSizePolicy::Minimum, QSizePolicy::Expanding);

    gridLayout->addItem(verticalSpacer, 2, 1, 1, 1);

    tabWidget->addTab(tab_General, QString());

    tab_Categories = new QWidget();
    tab_Categories->setObjectName(QString::fromUtf8("tab_Categories"));
    gridLayout_4 = new QGridLayout(tab_Categories);
    gridLayout_4->setObjectName(QString::fromUtf8("gridLayout_4"));
    categoryList = new QListWidget(tab_Categories);
    categoryList->setObjectName(QString::fromUtf8("categoryList"));

    gridLayout_4->addWidget(categoryList, 0, 0, 1, 2);

    addCategory = new QPushButton(tab_Categories);
    addCategory->setObjectName(QString::fromUtf8("addCategory"));

    gridLayout_4->addWidget(addCategory, 1, 0, 1, 1);

    removeCategory = new QPushButton(tab_Categories);
    removeCategory->setObjectName(QString::fromUtf8("removeCategory"));

    gridLayout_4->addWidget(removeCategory, 1, 1, 1, 1);

    tabWidget->addTab(tab_Categories, QString());
    tab_Ignores = new QWidget();
    tab_Ignores->setObjectName(QString::fromUtf8("tab_Ignores"));
    gridLayout_5 = new QGridLayout(tab_Ignores);
    gridLayout_5->setObjectName(QString::fromUtf8("gridLayout_5"));
    ignoreList = new QListWidget(tab_Ignores);
    ignoreList->setObjectName(QString::fromUtf8("ignoreList"));

    gridLayout_5->addWidget(ignoreList, 0, 0, 1, 2);

    addIgnore = new QPushButton(tab_Ignores);
    addIgnore->setObjectName(QString::fromUtf8("addIgnore"));

    gridLayout_5->addWidget(addIgnore, 1, 0, 1, 1);

    removeIgnore = new QPushButton(tab_Ignores);
    removeIgnore->setObjectName(QString::fromUtf8("removeIgnore"));

    gridLayout_5->addWidget(removeIgnore, 1, 1, 1, 1);

    tabWidget->addTab(tab_Ignores, QString());

    gridLayout_6->addWidget(tabWidget, 0, 0, 1, 1);

    buttonBox = new QDialogButtonBox(this);
    buttonBox->setObjectName(QString::fromUtf8("buttonBox"));
    buttonBox->setOrientation(Qt::Horizontal);
    buttonBox->setStandardButtons(QDialogButtonBox::Cancel|QDialogButtonBox::Ok);

    gridLayout_6->addWidget(buttonBox, 1, 0, 1, 1);

    tabWidget->setCurrentIndex(0);

    if(account != QString::null)
    {
        leUserName->setDisabled(true);
        setWindowTitle(tr("Editing Account: %1").arg(account));
    }
    else
    {
        setWindowTitle(tr("New Twitter Account"));
    }

    label_Username->setText(tr("User Name:"));
    label_Reload->setText(tr("Reload Time:"));
    updateTime->setSuffix(tr(" minutes"));
    tabWidget->setTabText(tabWidget->indexOf(tab_General), tr("General"));
    addCategory->setText(tr("Add Category"));
    removeCategory->setText(tr("Remove"));
    tabWidget->setTabText(tabWidget->indexOf(tab_Categories), tr("Categories"));
    addIgnore->setText(tr("Add Ignore Filter"));
    removeIgnore->setText(tr("Remove"));
    tabWidget->setTabText(tabWidget->indexOf(tab_Ignores), tr("Ignore Filter"));


    QSettings settings("Cardinal", "Cardinal");
    settings.beginGroup("Accounts");
    settings.beginGroup(m_account);
    leUserName->setText(settings.value("username").toString());
    updateTime->setValue(settings.value("ptReload").toInt());

    connect(addCategory, SIGNAL(clicked()), this, SLOT(addNewCategory()));
    connect(removeCategory, SIGNAL(clicked()), this, SLOT(removeCategoryA()));
    connect(categoryList, SIGNAL(itemSelectionChanged()), this, SLOT(catListItemSelectionChanged()));
    connect(categoryList, SIGNAL(itemDoubleClicked(QListWidgetItem*)), this, SLOT(categoryItemClicked(QListWidgetItem*)));
    connect(buttonBox, SIGNAL(accepted()), this, SLOT(accept()));
    connect(buttonBox, SIGNAL(rejected()), this, SLOT(reject()));
    connect(addIgnore, SIGNAL(clicked()), this, SLOT(addNewIgnoreFilter()));
    connect(removeIgnore, SIGNAL(clicked()), this, SLOT(removeIgnoreFilter()));
    settings.beginGroup("Categories");
    QStringList cats = settings.childGroups();
    for (int i = 0; i < cats.count(); ++i)
    {
        categoryList->addItem(cats[i]);
    }
    settings.endGroup();

    settings.endGroup();
    settings.endGroup();

    loadIgnoreFilters();
}

void AccountEditor::accept()
{
    m_account = leUserName->text();
    QSettings settings("Cardinal", "Cardinal");
    settings.beginGroup("Accounts");
    settings.beginGroup(m_account);
    settings.setValue("username", leUserName->text());
    settings.setValue("ptReload", updateTime->value());

    settings.endGroup();
    settings.endGroup();

    hide();

    emit accepted();
}

void AccountEditor::reject()
{
    hide();
}

void AccountEditor::addNewCategory()
{
   QString category = 
       QInputDialog::getText(this, tr("Add Category"), tr("Add a new category"), QLineEdit::Normal);

   if(category == QString(""))
       return;

    QSettings settings("Cardinal", "Cardinal");
    settings.beginGroup("Accounts");
    settings.beginGroup(m_account);
    settings.beginGroup("Categories");
    settings.beginGroup(category);
    settings.setValue("Name", category);
    settings.endGroup();
    settings.endGroup();
    settings.endGroup();
    settings.endGroup();

    categoryList->addItem(category);
}

void AccountEditor::removeCategoryA()
{
	QString categoryTxt = categoryList->item(categoryList->currentRow())->text();
	QMessageBox msg(QMessageBox::Question, tr("Remove Category?"), tr("Are you sure you want to remove the category: \"") + categoryTxt + tr("\"?"), QMessageBox::Yes | QMessageBox::No, this);
	int result = msg.exec();
	if(result == QMessageBox::Yes) {
		QSettings settings("Cardinal", "Cardinal");
                settings.beginGroup("Accounts");
                settings.beginGroup(m_account);
                settings.beginGroup("Categories");
		settings.remove(categoryTxt);
		settings.endGroup();
                settings.endGroup();
                settings.endGroup();
	}
	categoryList->setItemHidden(categoryList->item(categoryList->currentRow()), true);
}

void AccountEditor::categoryItemClicked(QListWidgetItem *item)
{
        editor = new CategoryEditor(this, m_account, item->text());
	if(editor->exec()) {}
	delete editor;
}

void AccountEditor::catListItemSelectionChanged()
{
        if(categoryList->selectedItems().count() == 0)
        {
		removeCategory->setEnabled(false);
	} else {
		removeCategory->setEnabled(true);
	}
}

void AccountEditor::loadIgnoreFilters()
{
    QSettings settings("Cardinal", "Cardinal");
    settings.beginGroup("Accounts");
    settings.beginGroup(m_account);
    QStringList ignores = settings.value("IgnoreList").toStringList();
    ignoreList->addItems(ignores);
    settings.endGroup();
    settings.endGroup();
}

void AccountEditor::addNewIgnoreFilter()
{
   QString filter =
       QInputDialog::getText(this, tr("Add Ignore Filter"), tr("Add a new ignore filter. Regular Expressions and wildcards are supported."), QLineEdit::Normal);

   if(filter == QString(""))
       return;

    QSettings settings("Cardinal", "Cardinal");
    settings.beginGroup("Accounts");
    settings.beginGroup(m_account);
    QStringList ignores = settings.value("IgnoreList").toStringList();
    ignores.append(filter);
    ignoreList->addItem(filter);
    settings.setValue("IgnoreList", ignores);
    settings.endGroup();
    settings.endGroup();
}

void AccountEditor::removeIgnoreFilter()
{
    int row = ignoreList->currentRow();
    QString filter = ignoreList->item(ignoreList->currentRow())->text();
    QMessageBox msg(QMessageBox::Question, tr("Remove Ignore Filter?"), tr("Are you sure you want to remove the ignore filter: \"%1\"?").arg(filter), QMessageBox::Yes | QMessageBox::No, this);
    int result = msg.exec();
    if(result == QMessageBox::Yes)
    {
        QSettings settings("Cardinal", "Cardinal");
        settings.beginGroup("Accounts");
        settings.beginGroup(m_account);
        QStringList ignores = settings.value("IgnoreList").toStringList();
        ignores.removeAt(row);
        categoryList->clear();
        settings.setValue("IgnoreList", ignores);
        settings.endGroup();
        settings.endGroup();
    }
}

void AccountEditor::ignoreListItemSelectionChanged()
{
    if(ignoreList->selectedItems().count() == 0)
        removeIgnore->setEnabled(false);
    else
        removeIgnore->setEnabled(true);
}
