/************************************
libcommie2 - an interface to the democratic peoples republic of oauth
Copyright (C) 2010 Chris Fuenty <zimmy@zimmy.co.uk>

Permission to use, copy, modify, and/or distribute this software for any
purpose with or without fee is hereby granted, provided that the above
copyright notice and this permission notice appear in all copies.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
************************************/
#ifndef OAUTH_H
#define OAUTH_H

#include <QObject>
#include <QtCore>
#include <QNetworkAccessManager>
#include <QNetworkRequest>
#include <QNetworkReply>
#include <QUrl>
#include <QDateTime>
#include <QByteArray>

#include <openssl/hmac.h>
#include <openssl/evp.h>

#ifdef WIN32
	#define EXPORT __declspec(dllexport)
#else
	#define EXPORT
#endif

#define OAUTH_VERSION                   "1.0"                       /* OAuth 1.0a */
#define OAUTH_METHOD                    "HMAC-SHA1"                 /* Signature Hash Method */
#define OAUTH_TOKEN                     "oauth_token"
#define OAUTH_TOKEN_SECRET              "oauth_secret"

#define HTTP_POST                       0x0002                      /* POST request */
#define HTTP_GET                        0x0004                      /* GET request */

#define ACTION_REQUEST                  0x0010
#define ACTION_ACCESS                   0x0020
#define ACTION_COMMAND                  0x0030
#define ACTION_NONE                     0x0040

#define HTTP_OK                         200
#define HTTP_UNAUTHORIZED               401
#define HTTP_FORBIDDEN                  403
#define HTTP_NOT_FOUND                  404

struct token_
{
    QByteArray token;
    QByteArray secret;
};

typedef struct token_ token_t;
typedef QMultiMap<QByteArray,QByteArray> ParameterMap;

class CommieLib : public QObject
{
    Q_OBJECT
public:
    EXPORT CommieLib(QByteArray consumerKey, QByteArray consumerSecret, QObject *parent);
    virtual ~CommieLib();
    EXPORT void obtainRequestToken(QUrl, int, ParameterMap);
    EXPORT void obtainAccessToken(QUrl, int, token_t, ParameterMap);
    EXPORT QByteArray obtainAuthorizationHeader(QUrl, int, token_t, ParameterMap);
    EXPORT ParameterMap obtainAuthorizationHeaderAsParams(QUrl, int, token_t, ParameterMap);
protected slots:
    void finished(QNetworkReply *reply);
signals:
    EXPORT void error(int error, QByteArray data);
    EXPORT void hasRequestToken(token_t token);
    EXPORT void hasAccessToken(token_t token);
private:
    ParameterMap constructSignatureAndBuildParams(QUrl, int, token_t, ParameterMap);
    void doPost(QUrl, token_t, ParameterMap);
    void doGet(QUrl, token_t, ParameterMap);
    QByteArray createNonce(time_t);
    QByteArray m_consumerKey;
    QByteArray m_consumerSecret;
    QNetworkAccessManager *m_http;
    QNetworkReply *m_reply;
    int m_error;
    bool m_busy;
    int m_action;
    token_t m_lastToken;
};

#endif
