/*
 * Java-Gnome Bindings Library
 *
 * Copyright 1998-2005 the Java-Gnome Team, all rights reserved.
 *
 * The Java-Gnome bindings library is free software distributed under
 * the terms of the GNU Library General Public License version 2.
 */
package org.freedesktop.cairo;

import org.gnu.glib.Handle;

public class ScaledFont extends CairoObject {

    /**
     * Create a new ScaledFont
     * 
     * @param fontFace
     *            The FontFace
     * @param matrix
     *            font space to user space transformation matrix for the font.
     *            In the simplest case of a N point font, this matrix is just a
     *            scale by N, but it can also be used to shear the font or
     *            stretch it unequally along the two axes.
     * @param ctm
     *            user to device transformation matrix with which the font will
     *            be used.
     */
    public ScaledFont(FontFace fontFace, Matrix matrix, Matrix ctm,
            FontOptions options) {
        super(cairo_scaled_font_create(fontFace.getHandle(),
                matrix.getHandle(), ctm.getHandle(), options.getHandle()));
    }

    protected void finalize() throws Throwable {
        cairo_scaled_font_destroy(getHandle());
        super.finalize();
    }

    ScaledFont(Handle hndl) {
        super(hndl);
    }

    public FontExtents getFontExtents() {
        Handle hndl = getNullHandle();
        cairo_scaled_font_extents(getHandle(), hndl);
        return new FontExtents(hndl);
    }

    /**
     * Gets the overall metrics for an array of glyphs. The X and Y offsets in
     * glyphs are taken from an origin of 0,0.
     * 
     * @param glyhps
     *            an array of glyph IDs with X and Y offsets
     * @return a TextExtent which contains the extents
     */
    public TextExtents getGlyphExtents(Glyph[] glyhps) {
        if (null == glyhps)
            return null;
        Handle[] g = new Handle[glyhps.length];
        for (int i = 0; i < glyhps.length; i++) {
            g[i] = glyhps[i].getHandle();
        }
        Handle hndl = getNullHandle();
        cairo_scaled_font_glyph_extents(getHandle(), g, hndl);
        return new TextExtents(hndl);
    }

    native static final private Handle cairo_scaled_font_create(Handle face,
            Handle matrix, Handle ctm, Handle options);

    native static final private void cairo_scaled_font_extents(Handle obj,
            Handle extents);

    native static final private void cairo_scaled_font_glyph_extents(
            Handle obj, Handle[] glyphs, Handle extents);

    // TODO: lifecycle
    native static final private void cairo_scaled_font_reference(Handle obj);

    native static final private void cairo_scaled_font_destroy(Handle obj);

}
