# DESCRIPTION: Perl ExtUtils: Type 'perl Makefile.PL' to create a Makefile for this package
#
# Copyright 2000-2010 by Wilson Snyder.  This program is free software;
# you can redistribute it and/or modify it under the terms of either the GNU
# Lesser General Public License Version 3 or the Perl Artistic License Version 2.0.

require 5.005;
use ExtUtils::MakeMaker;
use Carp;
use Config;

sub MY::postamble {
    my $out="";
    # Note OPTIMIZE is passed from upper makefile, so this code needed there too.
    if ($Config{osname} =~ /cygwin/i || $Config{archname} =~ /cygwin/i) {
	# Cygwin ExtUtils::MakeMaker ignores our LIBS declaration and says
	# "No library found for -lstdc++".  Force it.
	$out .= "LDLOADLIBS += -lstdc++\n";
	# Cygwin: High optimization causes g++ "out of memory"
	$out .= "OPTIMIZE += -O\n";
    }
    if ($Config{osname} =~ /darwin/i || $Config{archname} =~ /darwin/i) {
	# MakeMaker wants to create bundles on MacOSX rather than dylibs. We override DLEXT and LDDLFLAGS generated by MakeMaker in this case
	$out .= "DLEXT = dylib\n";
	$out .= sprintf("LDDLFLAGS = -dynamiclib -lstdc++ -L/System/Library/Perl/%vd/%s/CORE -lperl -lgcc_eh -L/usr/local/lib\n",$^V,$Config{archname});
    }
    $out .= "CCFLAGS += -Wall -Wno-unused -Werror\n" if $ENV{VERILATOR_AUTHOR_SITE};
    $out .= "CCFLAGS  += $ENV{VERILOGPERL_CCFLAGS}\n" if defined $ENV{VERILOGPERL_CCFLAGS};
    $out .= "OPTIMIZE += -Wno-unused\n" if $ENV{VERILATOR_AUTHOR_SITE};  # Makefile has another -Wall
    #$out .= "OPTIMIZE += -O0 -ggdb\n" if $ENV{VERILATOR_AUTHOR_SITE};  print "%Warning: -O0 --gdb on, also FLEX -d on\n";
    $out .= "OPTIMIZE += $ENV{VERILOGPERL_CCFLAGS}\n" if defined $ENV{VERILOGPERL_CCFLAGS};
    $out .= '
all:: README

README: Verilog-Perl.pod
	-$(RM_RF) $@
	pod2text --loose $< > $@

clean::
	-$(RM_RF) simv .vpassert test_dir *.tmp

dist: maintainer-copy distcheck README

## Maintainer use:
preexist:
	svnorcvs nexists $(DISTNAME)_$(VERSION_SYM)
	test -s README

tag:
	svnorcvs tag  $(DISTNAME)_$(VERSION_SYM)

maintainer-diff:
	svnorcvs diff $(DISTNAME)_$(VERSION_SYM)

maintainer-dist: preexist dist tag
	svnorcvs release $(DISTVNAME).tar.gz

maintainer-copy:

maintainer-clean:: distclean
	-$(RM_RF) README Makefile MANIFEST.bak $(MAKEFILE_OLD) */gen
';
    return $out;
}

my $fail;
local $! = undef;
my $have_gen = -d "Preproc/gen";
`flex --version`; if ($?) {
    if ($have_gen) { warn "\n-Note: 'flex' must be installed to build from sources\n"; }
    else { $fail=1; warn "\n%Error: 'flex' must be installed to build\n\n"; }
}
`bison --version`; if ($?) {
    if ($have_gen) { warn "\n-Note: 'bison' must be installed to build from sources\n"; }
    else { $fail=1; warn "\n%Error: 'bison' must be installed to build\n\n"; }
}
`g++ --version`; if ($?) { $fail=1; warn "\n%Error: 'gcc/g++' must be installed to build\n"; }
if ($fail) {
    if ($ENV{AUTOMATED_TESTING}) {
	exit(0);
    } else {
	die "%Error: Exiting due to above missing dependencies.\n";
    }
}

if (!-r "README" && !-r "Makefile") {
    warn "-Note: If building from 'git' sources (not from a CPAN tar file),\n"
	."-Note: ignore any 'files are missing' below for */gen/ and README:\n";
}
WriteMakefile(
	      DISTNAME  => 'Verilog-Perl',
	      NAME      => 'Verilog::Language',
	      AUTHOR	=> 'Wilson Snyder <wsnyder@wsnyder.org>',
	      ABSTRACT  => 'Verilog language utilities and parsing',
	      VERSION_FROM  => 'Language.pm',
	      NO_META	=> 1,
	      #OPTIMIZE	=> '-ggdb',
	      PREREQ_PM => {
		  'Pod::Usage' => 1.34,
		  'Data::Dumper' => 1,
		  'warnings' => 1,
	      },
	      BUILD_REQUIRES => {
		  'Digest::SHA1' => 0,
		  'Test' => 1,
		  'Test::More' => 0,
		  'Time::HiRes' => 1,
		  #'Test::Pod' => 1,  # Required only for author tests
		  #'Test::Perl::Critic' => 1,  # Required only for author tests
	      },
	      PMLIBDIRS => ['lib', 'Parser', 'Preproc', 'Netlist',],
	      EXE_FILES => [qw( vrename vpassert vppreproc vhier vsplitmodule )],
	      'clean'	=> {FILES => qw (test_dir signals.vrename .vpassert simv ),},
	      'dist'    => {COMPRESS => 'gzip -9f',
			    SUFFIX   => '.gz',
			    DIST_DEFAULT => 'README all tardist',
			},
	      );

my $mkv = `make --version`;
if ($? || $mkv !~ /GNU Make/i) {
    warn "-Important: Now type 'gmake MAKE=gmake' as this package requires GNU Make\n";
}
1;
