/** BEGIN COPYRIGHT BLOCK
 * Copyright (C) 2001 Sun Microsystems, Inc.  Used by permission.
 * Copyright (C) 2005 Red Hat, Inc.
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation version
 * 2.1 of the License.
 *                                                                                  
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *                                                                                  
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 * END COPYRIGHT BLOCK **/
#include <stdio.h>
#include <time.h>
#include <string.h>
#include <stdlib.h>
#include <ctype.h>
#include "version.h"
#include "admutil_pvt.h"
#include "libadminutil/distadm.h"

#ifdef XP_WIN32
#define strcasecmp stricmp
#define strncasecmp _strnicmp
#endif

#ifdef XP_UNIX
/*
 * Version information for the 'ident' and 'what commands
 *
 * NOTE: the first component of the concatenated rcsid string
 * must not end in a '$' to prevent rcs keyword substitution.
 */
static char rcsid[] = "$Header: AdminUtil " ADMSDK_PRODUCT  
        "  " ADMSDK_BUILDNUM " $";
static char sccsid[] = "@(#)AdminUtil " ADMSDK_PRODUCT  
		"  " ADMSDK_BUILDNUM;
#endif /* XP_UNIX */

 
/* Max size for a pathname */
#ifndef PATH_MAX
#define PATH_MAX 256
#endif

#ifndef FILE_PATHSEP
#define FILE_PATHSEP '/'
#endif

/* returns true if the given path is a valid file, false otherwise */
static int
is_file_ok(const char *path)
{
	PRFileInfo prinfo;
	int ret = 0;

	if (path && *path &&
		(PR_SUCCESS == PR_GetFileInfo(path, &prinfo)) &&
		prinfo.type == PR_FILE_FILE) {
		ret = 1;
	}

	return ret;
}

/* returns full path and file name if the file was found somewhere, false otherwise */
static char *
find_file_in_paths(
	const char *filename, /* the base filename to look for */
	const char *path /* path given by caller */
)
{
    char *retval = NULL;
    char *adminutilConfDir = getenv(ADMINUTIL_CONFDIR_ENV_VAR);

    /* try given path */
    if (path) {
        retval = PR_smprintf("%s/%s", path, filename);
    }
    if (!is_file_ok(retval) && adminutilConfDir) {
        PR_smprintf_free(retval);
        retval = PR_smprintf("%s/%s", adminutilConfDir, filename);
        if (!is_file_ok(retval)) {
            PR_smprintf_free(retval);
            retval = NULL;
        }
    }

    return retval;
}

/* Copy from libadmin.....  */
static unsigned char uuset[] = {
'A','B','C','D','E','F','G','H','I','J','K','L','M','N','O','P','Q','R','S','T',
'U','V','W','X','Y','Z','a','b','c','d','e','f','g','h','i','j','k','l','m','n',
'o','p','q','r','s','t','u','v','w','x','y','z','0','1','2','3','4','5','6','7',
'8','9','+','/' };
 
static int LDAP_CALL LDAP_CALLBACK
admin_ldap_rebind_proc (LDAP *ld, char **whop, char **passwdp,
                       int *authmethodp, int freeit, void *arg) {
  AdmldapHdnlPtr ptr = (AdmldapHdnlPtr)arg;

  if (freeit == 0) {
    *whop = ptr->sieDN;
    *passwdp = ptr->passwd;
    *authmethodp = LDAP_AUTH_SIMPLE;
  }

  return LDAP_SUCCESS;
}

PR_IMPLEMENT(int) 
admutil_uuencode(unsigned char *src, unsigned char *dst, int srclen)
{
   int  i, r;
   unsigned char *p;
 
/* To uuencode, we snip 8 bits from 3 bytes and store them as
   6 bits in 4 bytes.   6*4 == 8*3 (get it?) and 6 bits per byte
   yields nice clean bytes
 
   It goes like this:
        AAAAAAAA BBBBBBBB CCCCCCCC
        turns into the standard set of uuencode ascii chars indexed by numbers:
        00AAAAAA 00AABBBB 00BBBBCC 00CCCCCC
 
        Snip-n-shift, snip-n-shift, etc....
 */
 
   for (p=dst,i=0; i < srclen; i += 3) {
     /* Do 3 bytes of src */
     register char b0, b1, b2;
 
     b0 = src[0];
     if (i==srclen-1)
       b1 = b2 = '\0';
     else if (i==srclen-2) {
       b1 = src[1];
       b2 = '\0';
     }
     else {
       b1 = src[1];
       b2 = src[2];
     }
 
     *p++ = uuset[b0>>2];
     *p++ = uuset[(((b0 & 0x03) << 4) | ((b1 & 0xf0) >> 4))];
     *p++ = uuset[(((b1 & 0x0f) << 2) | ((b2 & 0xc0) >> 6))];
     *p++ = uuset[b2 & 0x3f];
     src += 3;
   }
   *p = 0;      /* terminate the string */
   r = (unsigned char *)p - (unsigned char *)dst;/* remember how many we did */
 
   /* Always do 4-for-3, but if not round threesome, have to go
          clean up the last extra bytes */
 
   for( ; i != srclen; i--)
                *--p = '=';
 
   return r;
}

PR_IMPLEMENT(char**)
admutil_strsdup(char** orig)
{
  int  cnt = 0, i = 0;
  char **dest, **tmpptr;
  
  if (!orig) return NULL;

  tmpptr = orig;
  while (*tmpptr++ != NULL) cnt++;

  dest = (char**)PR_Malloc((cnt+1)*sizeof(char*));

  for (i=0; i < cnt; i++) dest[i] = PL_strdup(orig[i]);

  dest[cnt] = NULL;

  return dest;
}

PR_IMPLEMENT(void)
admutil_strsFree(char** target)
{
  int  cnt = 0, i = 0;
  char **tmpptr;
  
  if (!target) return;

  tmpptr = target;
  while (*tmpptr++ != NULL) cnt++;

  for (i=0; i < cnt; i++) PR_Free(target[i]);

  PR_Free(target);

  return;
}


PR_IMPLEMENT(AttrNameList)
createAttrNameList(int entries)
{
  AttrNameList nl = (AttrNameList)PR_Malloc((entries+1)*sizeof(NameType));
  if (nl) memset(nl, '\0', (entries+1)*sizeof(NameType));
  return nl;
}

PR_IMPLEMENT(int)
addName(AttrNameList nl, int index, NameType name)
{
  nl[index] = PL_strdup(name);
  if (nl[index]) return ADMUTIL_OP_OK;
  else return ADMUTIL_OP_FAIL;
}

PR_IMPLEMENT(void)
deleteAttrNameList(AttrNameList nl)
{
  NameType name;
  AttrNameList nlptr = nl;
  if (nl) {
    while ((name = *nlptr++)) PR_Free(name);
    PR_Free(nl);
  }
}

PR_IMPLEMENT(void)
deleteValue(ValueType val)
{
    admutil_strsFree((char **)val);
}

PR_IMPLEMENT(AttributeList)
createAttributeList(int entries)
{
  AttributeList nvl = (AttributeList)PR_Malloc((entries+1)*sizeof(AttributePtr));
  if (nvl) memset(nvl, '\0', (entries+1)*sizeof(AttributePtr));
  return nvl;

}

PR_IMPLEMENT(int)
addAttribute(AttributeList nvl, int index, NameType name, ValueType val)
{
  AttributePtr nv = (AttributePtr)PR_Malloc(sizeof(Attribute));
  if (nv) {
    nv->attrName = PL_strdup(name);
    if (val) nv->attrVal = admutil_strsdup(val);
    else nv->attrVal = NULL;
    nvl[index] = nv;
    return ADMUTIL_OP_OK;
  }
  else return ADMUTIL_OP_FAIL;
}

PR_IMPLEMENT(int)
addSingleValueAttribute(AttributeList nvl, int index, NameType name, char* val) {
  AttributePtr nv = (AttributePtr)PR_Malloc(sizeof(Attribute));

  if (nv) {
    nv->attrName = PL_strdup(name);
    if (val) {
      nv->attrVal = (ValueType)PR_Malloc(2*sizeof(char*));
      nv->attrVal[0] = PL_strdup(val);
      nv->attrVal[1] = NULL;
    }
    else nv->attrVal = NULL;
    nvl[index] = nv;
    return ADMUTIL_OP_OK;
  }
  else return ADMUTIL_OP_FAIL;
}

PR_IMPLEMENT(void)
deleteAttributeList(AttributeList nvl) {
  AttributePtr attr;
  AttributeList nvlptr = nvl;
  if (nvl) {
    while ((attr = *nvlptr++)) {
      if (attr->attrName) PR_Free(attr->attrName);
      if (attr->attrVal) admutil_strsFree(attr->attrVal);
      PR_Free(attr);
    }
    PR_Free(nvl);
  }
}


/* Borrow from base/util.cpp  */
#define LF 10
#define CR 13

PR_IMPLEMENT(int)
admutil_getline(FILE *fstream, int maxlen, int lineno, char* buf)
{
  int i, x;
  memset(buf, '\0', maxlen);

  x = 0;
  while(1) {
    switch  (i = getc(fstream)) {
    case EOF:
      if (ferror(fstream)) {
        PR_snprintf(buf, maxlen, "I/O error reading file at line %d", lineno);
        return ADMUTIL_OP_FAIL;
      }
      buf[x] = '\0';
      return 1;
    case LF:
      if(x && (buf[x-1] == '\\')) {
        --x;
        continue;
      }
      buf[x] = '\0';
      return 0;
    case CR:
      continue;
    default:
      buf[x] = (char) i;
      if(++x == maxlen) {
        PR_snprintf(buf, maxlen, "line %d is too long", lineno);
        return -1;
      }
      break;
    }
  }
}

int
timecmp(char* ldapTime, time_t *cacheTime) {

  /* LDAP's timestamp example: 19970708020159Z  */
  int x, yr, mon, day, hr, min, sec;
  char zone;
  struct tm *cachetm;
  sscanf(ldapTime, "%4d%2d%2d%2d%2d%2d%c", &yr, &mon, &day, &hr, &min,
         &sec, &zone);

  if (zone == 'Z') cachetm = gmtime(cacheTime);
  else cachetm = localtime(cacheTime);
  
  if ( (x = yr - 1900 - cachetm->tm_year) || (x = mon - 1 - cachetm->tm_mon) ||
       (x = day - cachetm->tm_mday) || (x = hr - cachetm->tm_hour) ||
       (x = min - cachetm->tm_min) || (x = sec - cachetm->tm_sec) ) {
    return x;
  }
  else return 0;
}


char*
dn2AttrName(char* dn, char* rootDN)
{
  char**  dnList;
  char**  rootList;
  int    rootLen = 0, dnLen = 0, attrLen;
  char   buf[256];

  memset(buf, '\0', sizeof(buf));

  dnList = ldap_explode_dn(dn, 1);
  rootList = ldap_explode_dn(rootDN, 1);

  while (rootList[rootLen]) rootLen++;
  while (dnList[dnLen]) dnLen++;

  attrLen = dnLen - rootLen;

  while (attrLen > 0) {
    if (attrLen == 1)
      PR_snprintf(buf, sizeof(buf), "%s%s", buf, dnList[0]);
    else
      PR_snprintf(buf, sizeof(buf), "%s%s.", buf, dnList[attrLen-1]);
    attrLen--;
  }

  ldap_value_free(dnList);
  ldap_value_free(rootList);

  return PL_strdup(buf);
}

char*
attrName2dn(char* attrName, char* rootDN)
{
  char    result[512], buf[128];
  char    *rdn, *attrPtr, *resultPtr;

  if (!attrName) return NULL;
  if (*attrName == '\0') return PL_strdup(rootDN);
    
  PL_strncpyz(buf, attrName, sizeof(buf));
  memset(result , '\0', sizeof(result));

  resultPtr = result;
  attrPtr = buf;
  while (attrPtr) {
    attrPtr = strrchr(buf, '.');
    if (attrPtr) {
      *attrPtr = '\0';
      rdn = attrPtr+1;
    }
    else rdn = buf;
    if (result[0] == '\0')
      PR_snprintf(resultPtr, sizeof(result), "cn=%s", rdn);
    else
      PR_snprintf(resultPtr, sizeof(result), "%s, cn=%s", resultPtr, rdn);
  }
  PR_snprintf(resultPtr, sizeof(result), "%s, %s", resultPtr, rootDN);
  
  return PL_strdup(result);
}


/*
 * List Related stuff
 */
ListNodePtr
createListNode(char* name, void *val, int dflag) {

  ListNodePtr ptr;

  if (!name) return NULL;
  ptr = (ListNodePtr)PR_Malloc(sizeof(ListNode));
  if (!ptr) return NULL;
  ptr->name = PL_strdup(name);
  ptr->val = val;
  ptr->dflag = dflag;
  ptr->next = NULL;
  return ptr;
}

void
listDestroy(ListNodePtr list) {
  /* Clean up */
  ListNodePtr   node, nextptr;
  node = list;
  while (node) {
    nextptr = node->next;
    if (node->name) PR_Free (node->name);
    if (node->dflag) deleteValue ((ValueType)node->val);
    PR_Free (node);
    node= nextptr;
  }
}



ListNodePtr
listFindNode(ListNodePtr list, char* name) {
  /* Not implemented */
  ListNodePtr   node;

  node = list;
  while (node) {
    if (!strcasecmp(name, node->name)) return node;
    node = node->next;
  }
  return NULL;
}

void*
listGetValue(ListNodePtr list, char* name) {
  /* Not implemented */
  ListNodePtr   node;

  node = listFindNode(list, name);

  if (node) return node->val;
  return NULL;
}

int
listCount(ListNodePtr list) {
  /* Not implemented */
  ListNodePtr   node;
  int           cnt = 0;

  node = list;
  while (node) {
    cnt++;
    node = node->next;
  }
  return cnt;
}

ListNodePtr
listCat(ListNodePtr list1, ListNodePtr list2) {
  /* Not implemented */
  ListNodePtr   node;
  
  if (!list1) return list2;
  if (!list2) return list1;

  node = list1;
  while(node->next) node = node->next;
  node->next = list2;
  
  return list1;
}

/* Append to the end */
ListNodePtr
listAppend(ListNodePtr list, char* name, void* val, int dflag) {
  /* Not implemented */
  ListNodePtr   node, newNode;

  if (!list) return NULL;
  if (listFindNode(list, name)) return NULL;
  node = list;
  newNode = createListNode(name, val, dflag);
  if (!newNode) return NULL;

  while(node->next) node = node->next;
  node->next = newNode;
  
  return list;
}

/* Add to the front */
ListNodePtr
listAdd(ListNodePtr list, char* name, void* val, int dflag) {
  /* Not implemented */
  ListNodePtr   newNode;

  if (!list) return NULL;
  if (listFindNode(list, name)) return NULL;
  newNode = createListNode(name, val, dflag);
  if (!newNode) return NULL;
  newNode->next = list;

  return newNode;
}

ListNodePtr
listDelete(ListNodePtr list, char* name) {
  /* Not implemented */
  ListNodePtr   node, nextptr;

  if (!list) return NULL;

  if (!strcasecmp(list->name, name)) {
    nextptr = list->next;
    if (list->name) PR_Free(list->name);
    if (list->dflag) deleteValue ((ValueType)list->val);
    PR_Free(list);
    return nextptr;
  }

  node = list;
  while (node->next) {
    if (!strcasecmp(node->next->name, name)) {
      nextptr = node->next->next;
      if (node->next->name) PR_Free (node->next->name);
      if (node->next->dflag) deleteValue ((ValueType)node->next->val);
      PR_Free (node->next);
      node->next = nextptr;
      return list;
    }
    node = node->next;
  }
  return list;
}

/* 
 * Some utilities which use list
 */
AttributeList
nvlistConvert(ListNodePtr list) {
  AttributeList  resultList = NULL;
  int i = 0, cnt = listCount(list);
  ListNodePtr node = list;

  if (!list) return NULL;

  if (cnt > 0) {
    resultList = createAttributeList(cnt);
    while (node) {
      addAttribute(resultList, i++, node->name, (ValueType)(node->val));
      node = node->next;
    }
    return resultList;
  }
  return NULL;
}

void
nvlistDestroy(ListNodePtr list)
{
  ListNodePtr node = list;

  if (!list) return;

  while (node) {
    deleteValue(node->val);
    node = node->next;
  }

  listDestroy(list);
}

ValueType
valListConvert(ListNodePtr list)
{
  ValueType  resultList = NULL;
  int i = 0, cnt = listCount(list);
  ListNodePtr node = list;

  if (!list) return NULL;

  if (cnt > 0) {
    resultList = (ValueType)PR_Malloc((cnt+1)*sizeof(char*));
    while (node) {
      resultList[i++] = PL_strdup(node->name);
      node = node->next;
    }
    resultList[i] = NULL;
    return resultList;
  }
  return NULL;
}

ListNodePtr
createUpdateList(AttributeList nvl)
{
  AttributePtr    nv;
  AttributeList   nvlptr = nvl;
  char            namebuf[256];
  char           *attrName, *nodeName;
  ListNodePtr     resultList = NULL, nodePtr, attrPtr;

  while ((nv = *nvlptr++)) {
    PR_snprintf(namebuf, sizeof(namebuf), "%s", nv->attrName);
    attrName = strrchr(namebuf, '.');
    if (!attrName) {
      attrName = namebuf;
      nodeName = "";
    }
    else {
      *attrName = '\0';
      attrName++;
      nodeName = namebuf;
    }

    if (!resultList) {
      attrPtr = createListNode(attrName, admutil_strsdup(nv->attrVal), 1);
      resultList = createListNode(nodeName, (void*)attrPtr, 0);
    }
    else {
      nodePtr = (ListNodePtr)listFindNode(resultList, nodeName);
      if (!nodePtr) {
        attrPtr = createListNode(attrName, admutil_strsdup(nv->attrVal), 1);
        listAppend(resultList, nodeName, (void*)attrPtr, 0);
      }
      else {
        attrPtr = createListNode(attrName, admutil_strsdup(nv->attrVal), 1);
        listCat((ListNodePtr)(nodePtr->val), attrPtr);
      }
    }
  }
        
  return resultList;
}

void
destroyUpdateList(ListNodePtr updateList)
{
  ListNodePtr     nodePtr;

  nodePtr = updateList;

  while (nodePtr) {
    listDestroy((ListNodePtr)nodePtr->val);
    nodePtr = nodePtr->next;
  }

  listDestroy(updateList);
}

/*
 * Tree Related Stuff
 */
TreeNodePtr
createTreeNode(char* name, char* val)
{
  TreeNodePtr newNode;

  if (!name || *name == '\0') return NULL;
  if (!val) return NULL;
  newNode = (TreeNodePtr)PR_Malloc(sizeof(TreeNode));
  if (!newNode) return NULL;

  newNode->name = PL_strdup(name);
  newNode->val = createListNode(val, NULL, 0);
  newNode->left = NULL;
  newNode->right = NULL;

  return newNode;
}

int
treeCount(TreeNodePtr root)
{
  if (root) return (treeCount(root->left) + treeCount(root->right) + 1);
  else return 0;
}

TreeNodePtr
treeFindNode(TreeNodePtr node, char* name)
{
  int result;

  if (node) {
    if (!(result = strcasecmp(name, node->name))) return node;
    else if (result > 0) return treeFindNode(node->left, name);
    else return treeFindNode(node->right, name);
  }
  else return NULL;
}

ValueType
treeFindValue(TreeNodePtr root, char* name)
{
  TreeNodePtr  target;

  if ((target = treeFindNode(root, name))) return valListConvert(target->val);
  else return NULL;
}

char*
treeFindValueAt(TreeNodePtr root, char* name, int index)
{

  ValueType  vals;
  char*      val;

  vals = treeFindValue(root, name);

  if (vals) {
    val = PL_strdup(vals[0]);
    deleteValue(vals);
    return val;
  }
  else return NULL;
}

TreeNodePtr
treeAddNode(TreeNodePtr node, TreeNodePtr newNode) {

  int result;

  result = strcasecmp(newNode->name, node->name);
  if (!result) return NULL;
  else if (result > 0) {
    if (node->left) return treeAddNode(node->left, newNode);
    else {
      node->left = newNode;
      return newNode;
    }
  }
  else {  /* result < 0 */
    if (node->right) return treeAddNode(node->right, newNode);
    else {
      node->right = newNode;
      return newNode;
    }
  }
}

TreeNodePtr
treeAddValue(TreeNodePtr node, char* val) {

  if (!node) return NULL;
  if (node->val) listAppend(node->val, val, NULL, 0);
  else node->val = createListNode(val, NULL, 0);

  return node;
}

TreeNodePtr
treeAddNameValue(TreeNodePtr node, char* name, char* val) {

  TreeNodePtr targetNode;

  if (!name || *name == '\0') return NULL;
  targetNode = treeFindNode(node, name);

  if (targetNode) return treeAddValue(targetNode, val);
  else {
    targetNode = createTreeNode(name, val);
    if (targetNode) return treeAddNode(node, targetNode);
    return NULL;
  }
}

int
treeRemoveNode(TreeNodePtr node, char* name, int* removeFlag) {
  int result, flag = ADMUTIL_OP_OK, remove;
  TreeNodePtr tmpNodeLeft, tmpNodeRight;

  *removeFlag = 0;

  if (!node) return ADMUTIL_OP_FAIL;  /* Fail  */
  if (!name || *name == '\0') return ADMUTIL_OP_FAIL;  /* Fail  */

  result = strcasecmp(name, node->name);
  if (!result) { /* It is me to be deleted */
    PR_Free(node->name);
    listDestroy(node->val);
    
    /* Reorganize tree if necessary */
    tmpNodeLeft = node->left;
    tmpNodeRight = node->right;
    if (tmpNodeLeft && tmpNodeRight) {
      node->name = tmpNodeLeft->name;
      node->val = tmpNodeLeft->val;
      node->left = tmpNodeLeft->left;
      node->right = tmpNodeLeft->right;
      treeAddNode(node, tmpNodeRight);
      PR_Free(tmpNodeLeft);
    }
    else if (tmpNodeLeft) { /* Right is NULL */
      node->name = tmpNodeLeft->name;
      node->val = tmpNodeLeft->val;
      node->left = tmpNodeLeft->left;
      node->right = tmpNodeLeft->right;
      PR_Free(tmpNodeLeft);
    }
    else if (tmpNodeRight) { /* Left is NULL */
      node->name = tmpNodeRight->name;
      node->val = tmpNodeRight->val;
      node->left = tmpNodeRight->left;
      node->right = tmpNodeRight->right;
      PR_Free(tmpNodeRight);
    }
    else *removeFlag = 1; /* Both left and right are NULL */
    flag = ADMUTIL_OP_OK;  /* Should be success  */
  }
  else if (result > 0) {
    if (node->left) {
      flag = treeRemoveNode(node->left, name, &remove);
      if (remove) {
	PR_Free(node->left);
	node->left = NULL;
      }
    }
    else flag = ADMUTIL_OP_FAIL; /* Can't  find the node  */
  }
  else {
    if (node->right) {
      flag = treeRemoveNode(node->right, name, &remove);
      if (remove) {
	PR_Free(node->right);
	node->right = NULL;
      }
    }
    else flag = ADMUTIL_OP_FAIL;  /* Can't  find the node  */
  }
  return flag;
}

void
treeRemoveTree(TreeNodePtr tree)
{
  if (tree) {
    if (tree->left) { treeRemoveTree(tree->left); tree->left = NULL; };
    if (tree->right) { treeRemoveTree(tree->right); tree->right = NULL; }
    if (tree->name) { PR_Free(tree->name); tree->name = NULL; }
    if (tree->val) { listDestroy(tree->val); tree->val = NULL; }
    PR_Free(tree);
  }
}

ListNodePtr
treeBuildAttrList(char* nodeName, TreeNodePtr node)
{
  ListNodePtr resultList=NULL, leftPtr = NULL, nodePtr = NULL, rightPtr = NULL;
  char namebuf[256];

  if (!nodeName || nodeName[0] == '\0')
    PR_snprintf(namebuf, sizeof(namebuf), "%s", node->name);
  else
    PR_snprintf(namebuf, sizeof(namebuf), "%s.%s", nodeName, node->name);

  if (node->left) leftPtr = treeBuildAttrList(nodeName, node->left);
  nodePtr = createListNode(namebuf, valListConvert(node->val), 0);
  if (node->right) rightPtr = treeBuildAttrList(nodeName, node->right);

  if (leftPtr) resultList = leftPtr;
  if (nodePtr) resultList = listCat(resultList, nodePtr);
  if (rightPtr) resultList = listCat(resultList, rightPtr);

  return resultList;
}

void
treeExport(FILE *fstream, char* parentString, TreeNodePtr node)
{
  char *cptr, *sptr, valBuf[BUFSIZ];
  ListNodePtr listPtr;

  if (node->left) treeExport(fstream, parentString, node->left);
  if (!parentString || *parentString == '\0') 
    fprintf(fstream, "%s: ", node->name);
  else fprintf(fstream, "%s.%s: ", parentString, node->name);

  listPtr = node->val;
  while (listPtr) {
    PR_snprintf(valBuf, sizeof(valBuf), "%s", listPtr->name);
    sptr = valBuf;
    while ((cptr = strchr(sptr, '\n'))) {
      *cptr++ = '\0';
      fprintf(fstream, "%s\n ", sptr);
      sptr=cptr;
    }
    fprintf(fstream, "%s\n", sptr);
    listPtr = listPtr->next;
  }
  
  if (node->right) treeExport(fstream, parentString, node->right);
}


TreeNodePtr
treeImport(FILE *fstream, int* errorcode)
{
  int         status, lineno=1, valLen=0;
  char        linebuf[MAX_LEN], *name=NULL, *val=NULL;
  char        valBuf[BUFSIZ], *valptr = valBuf;
  int         valBuf_len = sizeof(valBuf);
  TreeNodePtr rootNode = NULL;

  if (!fstream) return NULL;
  if (!errorcode) return NULL;
  
  valBuf[0] = '\0';

  while(1) {
    switch(status = admutil_getline(fstream, sizeof(linebuf), lineno++, linebuf)) {
    case -1:
      /* Error on reading, SET ERRORCODE */
      *errorcode = ADMUTIL_SYSTEM_ERR;
      if (rootNode) treeRemoveTree(rootNode);
      return NULL;
      break;
    case 1:
      /* EOF, out of here */
      if (name) {
        if (rootNode) {
          treeAddNameValue(rootNode, name, valBuf);
        }
        else {
          rootNode = createTreeNode(name, valBuf);
        }
        PR_Free(name);
        name = NULL;
      }
      return rootNode;
      break;
    default:
      if (linebuf[0] == ' ') {
        /* This is continue of value  */
        /* Append value to value buffer  */
        val = &linebuf[1];
        valLen= PL_strlen(val);
        /* put back LF */
        *valptr++ = '\n';
        valBuf_len -= 1;
        PR_snprintf(valptr, valBuf_len, "%s", val);
        valBuf_len -= valLen;
        valptr = valptr + valLen;
      }
      else {
        if (name) {
          if (rootNode) {
            treeAddNameValue(rootNode, name, valBuf);
          }
          else {
            rootNode = createTreeNode(name, valBuf);
          }
          PR_Free(name);
          name = NULL;
          valptr=valBuf;
          valBuf[0] = '\0';
        }
        name = PL_strdup(strtok(linebuf, ":"));
        if (!name) break;
        val = linebuf+PL_strlen(name)+1;

        /* Skip over the whitesapce  */
        while (*val && isspace(*val)) val++;

        if (*val == '\0') {
          /* Empty value, Ingore it */
          break;
        }
        /* Put value to value buffer */
        valLen= PL_strlen(val);
        PR_snprintf(valptr, valBuf_len, "%s", val);
        valBuf_len -= valLen;
        valptr = valptr + valLen;
      }
    }
  }
}

/*
 * LDAPMod related utility functions
 */

LDAPMod*
createSingleValMod( char* namep, char* value, int mode)
{
  char*   valbuf[2];

  if (value) {
    valbuf[0] = value;
    valbuf[1] = NULL;

    return createMod(namep, valbuf, mode);
  }
  else return createMod(namep, NULL, mode);
}

LDAPMod*
createMod( char* namep, ValueType values, int mode)
{
  LDAPMod       *newMod;
  
  newMod = (LDAPMod*)PR_Malloc(sizeof(LDAPMod));
  if (!newMod) {
      return newMod;
  }
  newMod->mod_op = mode;
  newMod->mod_type = PL_strdup(namep);
  if (namep && !newMod->mod_type) {
      deleteMod(newMod);
      return NULL;
  }
  if (values) {
       newMod->mod_values = admutil_strsdup(values);
       if (!newMod->mod_values) {
           deleteMod(newMod);
           return NULL;
       }
  }
  else if (mode != LDAP_MOD_ADD) {
      newMod->mod_values = NULL;
  }
  else {
      /* For LDAP_MOD_ADD attribute value must be specified */
      newMod->mod_values = (char**)PR_Malloc(2*sizeof(char*));
      if (!newMod->mod_values) {
          deleteMod(newMod);
          return NULL;
      }
      newMod->mod_values[0] = PL_strdup("");
      if (!newMod->mod_values[0]) {
          deleteMod(newMod);
          return NULL;
      }
      newMod->mod_values[1] = NULL;
  }

  return newMod;
}

void
deleteMod(LDAPMod* mod)
{
  if (mod) {
    if (mod->mod_type) PR_Free ( mod->mod_type );
    if (mod->mod_values) admutil_strsFree (mod->mod_values);
    PR_Free( mod );
  }
}

void
deleteMods(LDAPMod** mods)
{
  LDAPMod* mod;
  LDAPMod** modsptr = mods;

  if (mods) {
    while ((mod = *modsptr++)) deleteMod(mod);
    PR_Free(mods);
  }

}

/* DT 12/13/97
 * admldapBuildInfo calls this function to unescape the URI and also normalize
 * the uri.  Normalizing the uri converts all "\" characters in the URI
 * and pathinfo portion to "/".  Does not touch "\" in query strings.
 */
PR_IMPLEMENT(void)
admUriUnescape(char *s)
{
    char *t, *u;

    if (!s) {
        return;
    }

    for(t = s, u = s; *t; ++t, ++u) {
        if((*t == '%') && t[1] && t[2]) {
            *u = ((t[1] >= 'A' ? ((t[1] & 0xdf) - 'A')+10 : (t[1] - '0'))*16) +
                  (t[2] >= 'A' ? ((t[2] & 0xdf) - 'A')+10 : (t[2] - '0'));
            t += 2;
        }
        else
            if(u != t)
                *u = *t;
    }
    *u = *t;
}

/*
 * Write the info back to its config file
 */
PR_IMPLEMENT(int)
admldapWriteInfoFile(AdmldapInfo info)
{
  AdmldapHdnlPtr admInfo = (AdmldapHdnlPtr)info;
  int errorcode = ADMUTIL_OP_OK;
  FILE *fileStream = NULL;

  if (admInfo && admInfo->configInfo && admInfo->configFilePath) {
	if((fileStream = fopen(admInfo->configFilePath, "w")) == NULL) {
	  /* Error open file  */
		errorcode = ADMUTIL_SYSTEM_ERR;
		goto done;
	}
	treeExport(fileStream, NULL, admInfo->configInfo);
  }

done:
  if (fileStream) {
	fclose(fileStream);
  }
  return errorcode;
}

/* This function is useful if you just want to read the adm.conf info
   without opening an ldap connection
*/
PR_IMPLEMENT(AdmldapInfo) 
admldapBuildInfoOnly(char* configRoot, int *errorcode)
{
  FILE           *fileStream;
  AdmldapHdnlPtr admInfo = NULL;
  TreeNodePtr    configInfo = NULL;
  char           *path = NULL;
  char           buf[MAX_LEN],
                 *name= NULL, *password=NULL;
  int            status;
  LDAPURLDesc    *ldapInfo;

  *errorcode = ADMUTIL_OP_OK;

  path = find_file_in_paths("adm.conf", configRoot);
  if (!path) {
    *errorcode = ADMUTIL_ENV_ERR;
	return NULL;
  }

  if((fileStream = fopen(path, "r")) == NULL) {
    /* Error open file  */
    *errorcode = ADMUTIL_SYSTEM_ERR;
    PR_smprintf_free(path);
    return NULL;
  }

  configInfo = treeImport(fileStream, errorcode);
  fclose(fileStream);

  if (!configInfo) { *errorcode = ADMUTIL_SYSTEM_ERR; return NULL; }

  admInfo = (AdmldapHdnlPtr)PR_Malloc(sizeof(AdmldapHdnl));
  if (!admInfo) { *errorcode = ADMUTIL_SYSTEM_ERR; return NULL; }
  memset(admInfo, '\0', sizeof(AdmldapHdnl));

  admInfo->configFilePath = path; /* hand off memory */
  path = NULL;
  if (!admInfo->configFilePath) {
      PR_Free(admInfo);
      *errorcode = ADMUTIL_SYSTEM_ERR;
      return NULL;
  }

  admInfo->configInfo = configInfo;

  if (!(admInfo->serverDirectoryURL = treeFindValueAt(admInfo->configInfo, "ldapurl", 0))) { /* admInfo owns malloced memory now */
    /* Error open file  */
    *errorcode = ADMUTIL_SYSTEM_ERR;
    destroyAdmldap((AdmldapInfo)admInfo);
    return NULL;
  }

  if (ldap_url_parse(admInfo->serverDirectoryURL, &ldapInfo)) {
    *errorcode = ADMUTIL_SYSTEM_ERR;
    return NULL;
  }

  admInfo->ldapInfo = ldapInfo;

  /* sieDN owns malloced memory returned by treeFindValueAt */
  admInfo->sieDN = treeFindValueAt(admInfo->configInfo, "sie", 0);

  /* Try to get local admin's name and password  */
  path = find_file_in_paths("admpw", configRoot);
  if (!path) {
    *errorcode = ADMUTIL_ENV_ERR;
    destroyAdmldap((AdmldapInfo)admInfo);
	return NULL;
  }

  if((fileStream = fopen(path, "r")) == NULL) {
    /* Error open file  */
    *errorcode = ADMUTIL_SYSTEM_ERR;
    PR_smprintf_free(path);
    destroyAdmldap((AdmldapInfo)admInfo);
    return NULL;
  }

  switch(status = admutil_getline(fileStream, sizeof(buf), 1, buf)) {
  case -1:
    /* Error on reading, SET ERRORCODE */
    *errorcode = ADMUTIL_SYSTEM_ERR;
    PR_smprintf_free(path);
    destroyAdmldap((AdmldapInfo)admInfo);
	fclose(fileStream);
    return NULL;
    break;
  case 1:
    /* EOF */
  default:
    password = strchr(buf, ':');
    *password++ = '\0';
    while (*password) {
      if (*password == ' ') password++;
      else break;
    }
    
    name = buf;
    if (*password) {
      *errorcode =  ADMUTIL_OP_OK;
      admInfo->admpwFilePath = path; /* hand off memory */
      path = NULL;
      admInfo->localAdminName = PL_strdup(name);
      admInfo->localAdminPassword = PL_strdup(password);
    }
    else {
      *errorcode =  ADMUTIL_OP_FAIL;
    }
  }
  fclose(fileStream);

  PR_smprintf_free(path);

  return (AdmldapInfo)admInfo;
}

PR_IMPLEMENT(AdmldapInfo) 
admldapBuildInfoCbk(char* configRoot, char *(*cbk)(), int *errorcode)
{
  char           *siePasswd = NULL;
  char           *siedn = NULL;
  char           *userdn = NULL;
  AdmldapHdnlPtr admInfo = NULL;
  int            ldapError = LDAP_SUCCESS;
  int            secureLDAP = 0;

  *errorcode = ADMUTIL_OP_OK;
  admInfo = (AdmldapHdnlPtr)admldapBuildInfoOnly(configRoot, errorcode);
  if (*errorcode != ADMUTIL_OP_OK) {
	  return (AdmldapInfo)admInfo;
  }

  /* returned value from ADM_Get... should NOT be freed */
  ADM_GetCurrentPassword(errorcode, &siePasswd); /* via PIPE */

  if (admldapGetSecurity((AdmldapInfo)admInfo)) {
    *errorcode = ADMUTIL_NO_SSL_SUPPORT;
    secureLDAP = 1;
  }

  /*  if userdn is initialized, override the siedn to make bind succeed */
  ADM_GetUserDNString(errorcode, &userdn);
  if (strcasecmp(userdn, ADM_NOT_INITIALIZED)) {
    siedn = admldapGetSIEDN(admInfo);
    admldapSetSIEDN(admInfo, userdn);
    admSetCachedSIEPWD(siePasswd);
  }

  if (!secureLDAP) {
    admInfo->ldapHndl = ldap_init(admInfo->ldapInfo->lud_host, admInfo->ldapInfo->lud_port);
  }

  /* authenticate to LDAP server*/

  /*
   * Attempt to authenticate to the directory.  This code will retry
   * attempts as long as there is a new password available to use.
   */
  {
    int configPassword; /* Indicates password is in config file */
    int retry;          /* Indicates that a previous password failed */

    configPassword = (siePasswd != NULL);
    retry = 0;

    /* Attempt to authenticate */
    for(;;retry = 1)
      {
        /* If the password is in the config file, then quit if it is wrong.
         * Otherwise attempt get a password from the user.
         */
        if (configPassword) {
          if (retry) break;
        } else {
          siePasswd = cbk(retry);
          if (siePasswd == NULL) {
            ldapError = LDAP_INVALID_CREDENTIALS;
            break;
          }
        }

        if (!secureLDAP) {
          ldapError = ldap_simple_bind_s(admInfo->ldapHndl, admInfo->sieDN, siePasswd);
          if (ldapError == LDAP_SUCCESS) break;

          /* Quit on errors other than password problems */
          if (ldapError != LDAP_INVALID_CREDENTIALS) break;
        }
      }
  }

  if ((ldapError != LDAP_SUCCESS ) && !(secureLDAP)){
#ifdef LDAP_DEBUG
      ldap_perror(admInfo->ldapHndl, "ldap_simple_bind_s");
#endif
      switch (ldapError) {
      case LDAP_INAPPROPRIATE_AUTH:
      case LDAP_INVALID_CREDENTIALS:
      case LDAP_INSUFFICIENT_ACCESS:
        /* authenticate failed: Should not continue */
        ldap_unbind(admInfo->ldapHndl);
        admInfo->ldapHndl = NULL;
        *errorcode = ADMUTIL_LDAP_ERR;
        break;;
      case LDAP_NO_SUCH_OBJECT:
      case LDAP_ALIAS_PROBLEM:
      case LDAP_INVALID_DN_SYNTAX:
        /* Not a good user DN */
        ldap_unbind(admInfo->ldapHndl);
        admInfo->ldapHndl = NULL;
        *errorcode = ADMUTIL_LDAP_ERR;
        break;
      default:
        ldap_unbind(admInfo->ldapHndl);
        admInfo->ldapHndl = NULL;
        *errorcode = ADMUTIL_LDAP_ERR;
      }
  }

  /* setup the referral */
  if (admInfo->ldapHndl)
  {
      ldap_set_rebind_proc(admInfo->ldapHndl, admin_ldap_rebind_proc,
                             (void *)admInfo);
  }

  if (siePasswd != NULL) {
      /* returned value from ADM_Get... should NOT be freed */
      admInfo->passwd=PL_strdup(siePasswd);
  }

  /* Reset the siedn if we changed it*/
  if (siedn) {
    admldapSetSIEDN(admInfo, siedn);
    PL_strfree(siedn);
  }

  return (AdmldapInfo)admInfo;
}

static char *
cachedPwdCbk(int retry)
{
  if (retry) return NULL;

  return admGetCachedSIEPWD();
}

PR_IMPLEMENT(AdmldapInfo) 
admldapBuildInfo(char* configRoot, int *errorcode)
{
  return admldapBuildInfoCbk(configRoot, cachedPwdCbk, errorcode);
}

PR_IMPLEMENT(void)
destroyAdmldap(AdmldapInfo info)
{
  AdmldapHdnlPtr   admInfo = (AdmldapHdnlPtr)info;
  if (admInfo) {
    treeRemoveTree(admInfo->configInfo);
    admInfo->configInfo = NULL;
    if (admInfo->configFilePath) { 
      PR_Free(admInfo->configFilePath); 
      admInfo->configFilePath=NULL; 
    }
    if (admInfo->serverDirectoryURL) {
      PR_Free(admInfo->serverDirectoryURL);
      admInfo->serverDirectoryURL = NULL; 
    }
    if (admInfo->admpwFilePath) {
      PR_Free(admInfo->admpwFilePath);
      admInfo->admpwFilePath = NULL;
    }
    if (admInfo->localAdminName) {
      PR_Free(admInfo->localAdminName);
      admInfo->localAdminName = NULL;
    }
    if (admInfo->localAdminPassword) {
      memset(admInfo->localAdminPassword, '\0', strlen(admInfo->localAdminPassword));
      PR_Free(admInfo->localAdminPassword);
      admInfo->localAdminPassword = NULL;
    }
    if (admInfo->sieDN) {
      PR_Free(admInfo->sieDN);
      admInfo->sieDN = NULL;
    }
    if (admInfo->userDN) {
      PR_Free(admInfo->userDN);
      admInfo->userDN = NULL;
    }
    if (admInfo->passwd) {
      memset(admInfo->passwd, '\0', strlen(admInfo->passwd));
      PR_Free(admInfo->passwd);
      admInfo->passwd = NULL;
    }
    if (admInfo->ldapHndl) {
      ldap_unbind(admInfo->ldapHndl);
      admInfo->ldapHndl = NULL;
    }
    
    if (admInfo->ldapInfo) {
      ldap_free_urldesc(admInfo->ldapInfo);
      admInfo->ldapInfo = NULL;
    }
    PR_Free(admInfo);
  }
}

PR_IMPLEMENT(char*)
admldapGetHost(AdmldapInfo info)
{
  AdmldapHdnlPtr admInfo = (AdmldapHdnlPtr)info;

  return PL_strdup(admInfo->ldapInfo->lud_host);
}

PR_IMPLEMENT(int)
admldapGetPort(AdmldapInfo info)
{
  AdmldapHdnlPtr admInfo = (AdmldapHdnlPtr)info;

  return admInfo->ldapInfo->lud_port;

}

PR_IMPLEMENT(int)
admldapGetSecurity(AdmldapInfo info)
{
  AdmldapHdnlPtr admInfo = (AdmldapHdnlPtr)info;
  int            ldapSecurity;

  if (admInfo->ldapInfo->lud_options & LDAP_URL_OPT_SECURE) ldapSecurity = 1;
  else ldapSecurity = 0;

  return ldapSecurity;

}

PR_IMPLEMENT(char*)
admldapGetBaseDN(AdmldapInfo info)
{
  AdmldapHdnlPtr admInfo = (AdmldapHdnlPtr)info;
  char          *ldapBaseDN;

  if (admInfo->ldapInfo->lud_dn) ldapBaseDN = PL_strdup(admInfo->ldapInfo->lud_dn);
  else ldapBaseDN = NULL;

  return ldapBaseDN;

}

PR_IMPLEMENT(char*)
admldapGetSecurityDir(AdmldapInfo info)
{
  AdmldapHdnlPtr admInfo = (AdmldapHdnlPtr)info;
  char          *securitydir;

  securitydir = treeFindValueAt(admInfo->configInfo, "securitydir", 0);
  if (!securitydir) return NULL;
  else return securitydir;
}

PR_IMPLEMENT(int)
admldapSetSecurityDir(AdmldapInfo info, const char *securityDir)
{
  AdmldapHdnlPtr admInfo = (AdmldapHdnlPtr)info;
  int removeFlag = 0;

  if (securityDir) {
	/* remove old one */
	treeRemoveNode(admInfo->configInfo, "securitydir", &removeFlag);
	treeAddNameValue(admInfo->configInfo, "securitydir", (char *)securityDir);
  }

  return ADMUTIL_OP_OK;
}

PR_IMPLEMENT(char*)
admldapGetSIEDN(AdmldapInfo info)
{
  AdmldapHdnlPtr admInfo = (AdmldapHdnlPtr)info;
  char          *ldapSIEDN = NULL;

  ldapSIEDN = treeFindValueAt(admInfo->configInfo, "sie", 0);
  if (!ldapSIEDN) return NULL;
  else return ldapSIEDN;
}

PR_IMPLEMENT(int)
admldapSetSIEDN(AdmldapInfo info, const char *sieDN)
{
  AdmldapHdnlPtr admInfo = (AdmldapHdnlPtr)info;
  int removeFlag = 0;

  if (sieDN) {
	/* remove old one */
	PL_strfree(admInfo->sieDN);
	treeRemoveNode(admInfo->configInfo, "sie", &removeFlag);
	/* add new one */
	admInfo->sieDN = PL_strdup(sieDN);
	treeAddNameValue(admInfo->configInfo, "sie", (char *)sieDN);
  }

  return ADMUTIL_OP_OK;
}

PR_IMPLEMENT(char*)
admldapGetSIEPWD(AdmldapInfo info)
{
  AdmldapHdnlPtr admInfo = (AdmldapHdnlPtr)info;
  if(admInfo->passwd)
    return PL_strdup(admInfo->passwd);
  else {
    admInfo->passwd = admGetCachedSIEPWD();
    if(admInfo->passwd)
      return PL_strdup(admInfo->passwd);
  }
  return NULL;
}

PR_IMPLEMENT(char*)
admldapGetISIEDN(AdmldapInfo info)
{
  AdmldapHdnlPtr admInfo = (AdmldapHdnlPtr)info;
  char          *ldapISIEDN = NULL;

  ldapISIEDN = treeFindValueAt(admInfo->configInfo, "isie", 0);
  if (!ldapISIEDN) return NULL;
  else  return ldapISIEDN;

}

PR_IMPLEMENT(int)
admldapSetISIEDN(AdmldapInfo info, const char *isieDN)
{
  AdmldapHdnlPtr admInfo = (AdmldapHdnlPtr)info;
  int removeFlag = 0;

  if (isieDN) {
	/* remove old one */
	treeRemoveNode(admInfo->configInfo, "isie", &removeFlag);
	treeAddNameValue(admInfo->configInfo, "isie", (char *)isieDN);
  }

  return ADMUTIL_OP_OK;
}

PR_IMPLEMENT(char *)
admldapGetDirectoryURL(AdmldapInfo info)
{
  AdmldapHdnlPtr admInfo = (AdmldapHdnlPtr)info;

  return PL_strdup(admInfo->serverDirectoryURL);
}

PR_IMPLEMENT(int)
admldapSetDirectoryURL(AdmldapInfo info, const char *ldapurl)
{
  AdmldapHdnlPtr admInfo = (AdmldapHdnlPtr)info;
  LDAPURLDesc    *ldapInfo;
  int errorcode = ADMUTIL_OP_OK;
  int removeFlag = 0;

  if (!ldapurl || ldap_url_parse(ldapurl, &ldapInfo)) {
	/* if the given url is not valid, don't do anything, just return an error */
    errorcode = ADMUTIL_SYSTEM_ERR;
    goto done;
  }

  /* The url is stored in 3 places in 3 different formats:
	 1 - the serverDirectoryURL string
	 2 - parsed in the ldapInfo structure
	 3 - the "ldapurl" key in the configinfo
  */
  /* first, free the old values */
  if (admInfo->ldapInfo) {
    ldap_free_urldesc(admInfo->ldapInfo);
  }
  PL_strfree(admInfo->serverDirectoryURL);
  treeRemoveNode(admInfo->configInfo, "ldapurl", &removeFlag);

  /* set the new values */
  admInfo->serverDirectoryURL = PL_strdup(ldapurl);
  admInfo->ldapInfo = ldapInfo;
  treeAddNameValue(admInfo->configInfo, "ldapurl", (char *)ldapurl);

done:
  return errorcode;
}

PR_IMPLEMENT(void)
admldapSetLDAPHndl(AdmldapInfo info, LDAP *ld)
{
  AdmldapHdnlPtr admInfo = (AdmldapHdnlPtr)info;

  if (admInfo->ldapHndl) ldap_unbind(admInfo->ldapHndl);
  admInfo->ldapHndl = ld;
}

LDAP *
admldapGetLDAPHndl(AdmldapInfo info)
{
  AdmldapHdnlPtr admInfo = (AdmldapHdnlPtr)info;

  return admInfo->ldapHndl;
}

PR_IMPLEMENT(char*)
admldapGetSysUser(AdmldapInfo info)
{
  AdmldapHdnlPtr admInfo = (AdmldapHdnlPtr)info;
  char          *sysuser = NULL;

  sysuser = treeFindValueAt(admInfo->configInfo, "sysuser", 0);
  if (!sysuser) return NULL;
  else  return sysuser;

}

PR_IMPLEMENT(char*)
admldapGetSysGroup(AdmldapInfo info)
{
  AdmldapHdnlPtr admInfo = (AdmldapHdnlPtr)info;
  char          *sysgroup = NULL;

  sysgroup = treeFindValueAt(admInfo->configInfo, "sysgroup", 0);
  if (!sysgroup) return NULL;
  else  return sysgroup;

}

PR_IMPLEMENT(char*)
admldapGetAdminDomain(AdmldapInfo info)
{
  AdmldapHdnlPtr admInfo = (AdmldapHdnlPtr)info;
  char          *admindomain = NULL;

  admindomain = treeFindValueAt(admInfo->configInfo, "AdminDomain", 0);
  if (!admindomain) return NULL;
  else  return admindomain;

}

PR_IMPLEMENT(char*)
admldapGetExpressRefreshRate(AdmldapInfo info)
{
  AdmldapHdnlPtr admInfo = (AdmldapHdnlPtr)info;
  char          *expressrefreshrate = NULL;

  expressrefreshrate = treeFindValueAt(admInfo->configInfo, "ExpressRefreshRate", 0);
  if (!expressrefreshrate) return NULL;
  else  return expressrefreshrate;

}

PR_IMPLEMENT(char*)
admldapGetExpressCGITimeout(AdmldapInfo info)
{
  AdmldapHdnlPtr admInfo = (AdmldapHdnlPtr)info;
  char          *expresscgitimeout = NULL;

  expresscgitimeout = treeFindValueAt(admInfo->configInfo, "ExpressCGITimeout", 0);
  if (!expresscgitimeout) return NULL;
  else  return expresscgitimeout;

}

PR_IMPLEMENT(char*)
admldapGetLdapStart(AdmldapInfo info)
{
  AdmldapHdnlPtr admInfo = (AdmldapHdnlPtr)info;
  char          *ldapstart = NULL;

  ldapstart = treeFindValueAt(admInfo->configInfo, "ldapStart", 0);
  if (!ldapstart) return NULL;
  else  return ldapstart;

}

PR_IMPLEMENT(char*)
admldapGetConfigFileName(AdmldapInfo info)
{
  AdmldapHdnlPtr admInfo = (AdmldapHdnlPtr)info;

  return PL_strdup(admInfo->configFilePath);
}

PR_IMPLEMENT(char*)
admldapGetAdmpwFilePath(AdmldapInfo info)
{
  AdmldapHdnlPtr admInfo = (AdmldapHdnlPtr)info;

  return PL_strdup(admInfo->admpwFilePath);
}

PR_IMPLEMENT(char*)
admldapGetLocalAdminName(AdmldapInfo info)
{
  AdmldapHdnlPtr admInfo = (AdmldapHdnlPtr)info;

  return PL_strdup(admInfo->localAdminName);
}

PR_IMPLEMENT(char *)
admldapGetUserDN(AdmldapInfo info, char *uid)
{
  AdmldapHdnlPtr admInfo = (AdmldapHdnlPtr)info;
  LDAP *ld = NULL;
  char *userDN = NULL;
  char *baseDN = NULL;
  char *uidFilter = NULL;
  int err;
  LDAPMessage *result = NULL;
		  
  if (NULL != admInfo->userDN) {
    userDN = admInfo->userDN;
    goto done;
  }
  if (NULL != uid && NULL != strchr(uid, '=')) {
    userDN = PL_strdup(uid);
	goto done;
  }
  ld = admldapGetLDAPHndl(info);
  if (NULL == ld) {
	goto done;
  }
  baseDN = admldapGetBaseDN(info);
  if (NULL == baseDN) {
	goto done;
  }
  uidFilter = PR_smprintf("(uid=%s)", uid?uid:admInfo->localAdminName);
  err = ldap_search_s(ld, baseDN, LDAP_SCOPE_SUBTREE, uidFilter,
				      NULL, 0, &result);
  if (err != LDAP_SUCCESS || ldap_count_entries(ld, result) == 0) {
    ldap_msgfree(result);
	goto done;
  } else {
    LDAPMessage *entry = ldap_first_entry(ld, result);
    userDN = ldap_get_dn(ld, entry);
    ldap_msgfree(result);
  }
done:
  PR_smprintf_free(uidFilter);
  if (baseDN) PR_Free(baseDN);
  if (userDN && (userDN != admInfo->userDN)) {
    PL_strfree(admInfo->userDN); /* free old one, if any */
    admInfo->userDN = userDN;
  } else {
    userDN = treeFindValueAt(admInfo->configInfo, "userdn", 0);
	if (userDN) {
      admInfo->userDN = userDN;
    } else {
      admInfo->userDN = NULL;
    }
  }
  return admInfo->userDN ? PL_strdup(admInfo->userDN) : NULL;
}

PR_IMPLEMENT(char*)
admGetLocalAdmin(char* configRoot, int *errorcode)
{
  FILE          *fileStream;
  char          *path = NULL, buf[MAX_LEN], *name = NULL;
  int           status;

  *errorcode = ADMUTIL_OP_OK;

  /* Try to get local admin's name and password  */
  path = find_file_in_paths("admpw", configRoot);
  if (!path) {
      *errorcode = ADMUTIL_ENV_ERR;
      return NULL;
  }

  if((fileStream = fopen(path, "r")) == NULL) {
    /* Error open file  */
    *errorcode = ADMUTIL_SYSTEM_ERR;
    PR_smprintf_free(path);
    return NULL;
  }
  PR_smprintf_free(path);
  path = NULL;

  switch(status = admutil_getline(fileStream, sizeof(buf), 1, buf)) {
  case -1:
    /* Error on reading, SET ERRORCODE */
    *errorcode = ADMUTIL_SYSTEM_ERR;
    fclose(fileStream);
    return NULL;
    break;
  case 1:
    /* EOF */
    /*    
    *errorcode =  ADMUTIL_OP_FAIL;
    return NULL;
    break;
    */
  default:
    fclose(fileStream);
    name = strtok(buf, ":");
    if (!name) {
      *errorcode =  ADMUTIL_OP_FAIL;
      return NULL;
    }
    else {
      *errorcode =  ADMUTIL_OP_OK;
      return PL_strdup(name);
    }
  }
}

static char *cachedSIEPWD = NULL;
 
/*
 * admSetCachedSIEPWD
 */
PR_IMPLEMENT(void)
admSetCachedSIEPWD(const char *pwd)
{
  if (cachedSIEPWD) {
    memset(cachedSIEPWD, '\0', strlen(cachedSIEPWD));
    PR_Free(cachedSIEPWD);
  }

  cachedSIEPWD = PL_strdup(pwd);
}

 
/*
 * admGetCachedSIEPWD 
 */
PR_IMPLEMENT(char *)
admGetCachedSIEPWD()
{
  char *result = NULL;

  if (cachedSIEPWD) result = PL_strdup(cachedSIEPWD);

  return result;
}
