/* 
 * Copyright (C) 2005  Network Applied Communication Laboratory Co., Ltd.
 *
 * This file is part of Rast.
 * See the file COPYING for redistribution information.
 *
 */

#include <stdlib.h>

#include "rast/config.h"
#include "rast/rast.h"
#include "rast/encoding.h"
#include "rast/filter.h"

static int initialized = 0;
static apr_pool_t *global_pool = NULL;

static int
rast_abort(int retcode)
{
    abort();
    return -1;  /* prevent compiler warnings */
}

static apr_abortfunc_t abort_func = rast_abort;

void
rast_set_abort_func(apr_abortfunc_t func)
{
    abort_func = func;
}

apr_pool_t *
rast_pool_create(apr_pool_t *parent)
{
    apr_pool_t *pool;

    if (apr_pool_create_ex(&pool, NULL, abort_func, NULL) != APR_SUCCESS) {
        abort_func(APR_ENOMEM);
    }
    return pool;
}

rast_error_t *
rast_initialize()
{
    const char *dir;
    rast_error_t *error;

    initialized++;
    if (initialized > 1) {
        return RAST_OK;
    }

    global_pool = rast_pool_create(NULL);

    dir = getenv("RAST_FILTER_MODULEDIR");
    if (dir == NULL) {
        dir = RAST_FILTER_MODULEDIR;
    }
    error = rast_load_filters(dir);
    if (error != RAST_OK) {
        return error;
    }

    dir = getenv("RAST_ENCODINGDIR");
    if (dir == NULL) {
        dir = RAST_ENCODINGDIR;
    }
    return rast_load_encoding_modules(dir);
}

void
rast_finalize()
{
    initialized--;
    if (initialized > 0) {
        return;
    }
    rast_unload_filters();
    rast_unload_encoding_modules();
    apr_pool_destroy(global_pool);
    global_pool = NULL;
}

apr_pool_t *
rast_get_global_pool()
{
    return global_pool;
}

rast_search_option_t *
rast_search_option_create(apr_pool_t *pool)
{
    rast_search_option_t *options;

    options = (rast_search_option_t *)
        apr_palloc(pool, sizeof(rast_search_option_t));
    options->start_no = 0;
    options->num_items = RAST_RESULT_ALL_ITEMS;
    options->need_summary = 0;
    options->summary_nchars = 100;
    options->sort_method = RAST_SORT_METHOD_SCORE;
    options->sort_order = RAST_SORT_ORDER_DEFAULT;
    options->sort_property = NULL;
    options->score_method = RAST_SCORE_METHOD_TFIDF;
    options->properties = NULL;
    options->num_properties = 0;
    options->all_num_docs = RAST_CALC_SCORE_STANDALONE;
    options->terms = NULL;
    options->num_terms = 0;
    return options;
}

/* vim: set filetype=c sw=4 expandtab : */
