//
// Gendarme.Rules.Smells.AvoidSwitchStatementsRule class
//
// Authors:
//	Néstor Salceda <nestor.salceda@gmail.com>
//
// 	(C) 2008 Néstor Salceda
//
// Permission is hereby granted, free of charge, to any person obtaining
// a copy of this software and associated documentation files (the
// "Software"), to deal in the Software without restriction, including
// without limitation the rights to use, copy, modify, merge, publish,
// distribute, sublicense, and/or sell copies of the Software, and to
// permit persons to whom the Software is furnished to do so, subject to
// the following conditions:
//
// The above copyright notice and this permission notice shall be
// included in all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
// EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
// MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
// NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE
// LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
// OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
// WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
//

using Mono.Cecil;
using Mono.Cecil.Cil;

using Gendarme.Framework;
using Gendarme.Framework.Rocks;

namespace Gendarme.Rules.Smells { 
	
	/// <summary>
	/// This rule avoids the Switch Statements smell.  This kind of smell
	/// could lead your code to duplication, because the same switch could
	/// be repeated in various places of your program.  Also, if you will
	/// need to do a little change, you should change every switch
	/// statement, the OO way is do it with polymorphism.
	/// </summary>
	/// <example>
	/// Bad example:
	/// <code>
	/// int balance = 0;
	/// foreach (Movie movie in movies) {
	/// 	switch (movie.GetTypeCode ()) {
	/// 	case MovieType.OldMovie:
	/// 		balance += movie.DaysRented * movie.Price / 2;
	/// 		break
	/// 	case MovieType.ChildMovie:
	/// 		//its an special bargain !!
	/// 		balance += movie.Price;
	/// 		break;
	/// 	case MovieType.NewMovie:
	/// 		balance += (movie.DaysRented + 1) * movie.Price;
	/// 		break:
	/// 	}
	/// }
	/// </code>
	/// </example>
	/// <example>
	/// Good example:
	/// <code>
	/// abstract class Movie {
	///	abstract int GetPrice ();
	/// }
	/// class OldMovie : Movie {
	///	public override int GetPrice () {
	///		return DaysRented * Price / 2;
	///	}
	/// }
	/// class ChildMovie : Movie {
	///	public override int GetPrice () {
	///		return movie.Price;
	///	}
	/// }
	/// class NewMovie : Movie {
	///	public override int GetPrice () {
	///		return (DaysRented + 1) * Price;
	///	}
	/// }
	///
	/// int balance = 0;
	/// foreach (Movie movie in movies)  
	/// 	balance += movie.GetPrice ()
	/// </code>
	/// </example>

	[Problem ("The problem with switch statements is the duplication.  You may find the same switch in several places.")]
	[Solution ("You should consider polymorphism.")]
	public class AvoidSwitchStatementsRule : Rule, IMethodRule {
		public RuleResult CheckMethod (MethodDefinition method) 
		{
			if (!method.HasBody)
				return RuleResult.DoesNotApply;
			
			//Perhaps you are checking autogenerated code from a
			//yield statement
			if (method.DeclaringType.IsGeneratedCode () && method.DeclaringType.Name.Contains ("Iterator"))
				return RuleResult.DoesNotApply;
				
			foreach (Instruction instruction in method.Body.Instructions) {
				if (instruction.OpCode == OpCodes.Switch) {
					Runner.Report (method, instruction, Severity.Low, Confidence.Total);
					return RuleResult.Failure;
				}
				
				//Sometimes the compiler generates a table
				//driven comparison, there is the code for
				//handling too.
				if (instruction.OpCode == OpCodes.Ldsfld) {
					FieldReference field = (FieldReference) instruction.Operand;
					if (field.Name.Contains ("switch") && field.IsGeneratedCode ()) {
						Runner.Report (method, instruction, Severity.Low, Confidence.Total);
						return RuleResult.Failure;
					}
				}
			}

			return Runner.CurrentRuleResult;
		}
	}
}
