/* 
**  certmap.c.c -- Apache LDAP authorization module
**
**  Read the files README and mod_authz_ldap.html for instructions on
**  configuring the module. Details of the license can be found in the
**  HTML documentation.
**
**  (c) 2000 Dr. Andreas Mueller
**
**  $Id: certmap.c,v 1.7 2004/03/30 23:35:50 afm Exp $
*/ 
#include "mod_authz_ldap.h"
#ifdef EAPI
#include <openssl/pem.h>
#include <openssl/x509v3.h>
extern char	*ssl_var_lookup(apr_pool_t *, server_rec *, conn_rec *,
			request_rec *, char *);
#ifndef AUTHZ_LDAP_HAVE_SSL
#define AUTHZ_LDAP_HAVE_SSL
#endif /* AUTHZ_LDAP_HAVE_SSL */
#endif /* EAPI */

/*************************************************************************
** Certificate Retrieval						**
*************************************************************************/

static char	*authz_ldap_get_issuer_dn(request_rec *r) {
#ifdef AUTHZ_LDAP_HAVE_SSL
	char			*issuer_dn;
	authz_ldap_config_rec	*sec;

	sec = ap_get_module_config(r->per_dir_config, &authz_ldap_module);
	issuer_dn = ssl_var_lookup(r->pool, r->server, r->connection, r,
		"SSL_CLIENT_I_DN");
	AUTHZ_DEBUG3("[%d] client issuer DN: %s", (int)getpid(),
		(issuer_dn) ? issuer_dn : "(null)");
	return issuer_dn;
#else /* AUTHZ_LDAP_HAVE_SSL */
	return NULL;
#endif /* AUTHZ_LDAP_HAVE_SSL */
}

static char	*authz_ldap_get_certificate(request_rec *r) {
#ifdef AUTHZ_LDAP_HAVE_SSL
	BIO		*bio_mem;
	int		l, j;
	unsigned char	*v;
	char		*xp, *cp, *p;
	authz_ldap_config_rec	*sec;
	X509		*xs;

	sec = ap_get_module_config(r->per_dir_config, &authz_ldap_module);

	/* retrieve the certificate in PEM form				*/
	xp = ssl_var_lookup(r->pool, r->server, r->connection, r,
		"SSL_CLIENT_CERT");
	if (NULL == xp) { 
		AUTHZ_DEBUG2("[%d] not certificate found", (int)getpid());
		return NULL;
	}

	/* turn the PEM data into a X.509 structure			*/
	bio_mem = BIO_new(BIO_s_mem());
	BIO_write(bio_mem, xp, strlen(xp) + 1);
	xs = PEM_read_bio_X509(bio_mem, NULL, NULL, NULL);
	BIO_free(bio_mem);
	if (NULL == xs) {
		AUTHZ_DEBUG3("[%d] could not read certificate from PEM: %s",
			(int)getpid(), xp);
		return NULL;
	}

	/* write the data from the certificate into DER form		*/
	bio_mem = BIO_new(BIO_s_mem());
	i2d_X509_bio(bio_mem, xs);
	l = BIO_ctrl_pending(bio_mem);
	v = ap_palloc(r->pool, l);
	BIO_read(bio_mem, v, l);
	BIO_free(bio_mem);

	/* quote the whole stuff so that it can safely be used in a 	*/
	/* filter							*/
	cp = ap_palloc(r->pool, 3 * l + 1);
	p = cp;
	for (j = 0; j < l; j++) {
		ap_snprintf(p, 4, "\\%02x", (unsigned char)v[j]);
		p += 3;
	}

	/* return the quoted certificate				*/
	return cp;
#else /* AUTHZ_LDAP_HAVE_SSL */
	return NULL;
#endif /* AUTHZ_LDAP_HAVE_SSL */
}

static char	*authz_ldap_get_subject_dn(request_rec *r) {
#ifdef AUTHZ_LDAP_HAVE_SSL
	char	*subject_dn;
	authz_ldap_config_rec	*sec;

	sec = ap_get_module_config(r->per_dir_config, &authz_ldap_module);
	subject_dn = ssl_var_lookup(r->pool, r->server, r->connection, r,
		"SSL_CLIENT_S_DN");
	AUTHZ_DEBUG3("[%d] client subject DN: %s", (int)getpid(),
		(subject_dn) ? subject_dn : "(null)");
	return subject_dn;
#else /* AUTHZ_LDAP_HAVE_SSL */
	return NULL;
#endif /* AUTHZ_LDAP_HAVE_SSL */
}

static char	*authz_ldap_get_certserial(request_rec *r) {
#ifdef AUTHZ_LDAP_HAVE_SSL
	char	*cp;
	authz_ldap_config_rec	*sec;

	sec = ap_get_module_config(r->per_dir_config, &authz_ldap_module);

	/* get serial number from X509 client certificate		*/
	cp = ssl_var_lookup(r->pool, r->server, r->connection, r,
		"SSL_CLIENT_M_SERIAL");
	AUTHZ_DEBUG3("[%d] client serial: %s", (int)getpid(),
		(cp) ? cp : "(null)");
	return cp;
#else /* AUTHZ_LDAP_HAVE_SSL */
	return NULL;
#endif /* AUTHZ_LDAP_HAVE_SSL */
}

/*************************************************************************
** Certificate Mapping							**
*************************************************************************/
 
/* the functions authz_ldap_get_ms_x509_alt_security_identity and
   authz_ldap_get_ms_user_principal_name were contributed by Julio Sanchez,
   here is how he describes the rationale behind it in an email message

	We are currently considering whether to keep on using mod_authz_ldap or
	look for some other arrangement and found that having it be able to map
	certificates using the information that Microsoft stores in AD would be
	a great plus.

	AD seems to use two methods.  First-rate certificates (generated by
	their own CAs, though they give some info on how to support those
	generated by other CAs) include a UPN (User Principal Name, closely
	related to Kerberos principals) as one value of the subjectAltName in
	the certificate and matching directory entries contain that value in the
	corresponding attribute.  This is the primary mapping method used by
	Windows 2000 server.

	However, it also allows (for some limited uses) to map semi-manually
	arbitrary certificates to directory entries.  In this case, the
	concatenation of issuer and subject names is placed, in a particular
	format, as one value of the altSecurityIdentities attribute.

	The attached patch, that has received limited testeing, implements
	mapping against AD.  This way, it is possible to use Apache in a
	corporate environment where the directory is build around Microsoft
	products.

 */
char	*authz_ldap_get_ms_x500_alt_security_identity(request_rec *r) {
#ifdef AUTHZ_LDAP_HAVE_SSL
	BIO		*bio_mem;
	int		l;
	char		*v;
	char		*xp;
	char		*p, *q;
	int		at_boundary;
	authz_ldap_config_rec	*sec;
	X509		*xs;
	X509_NAME	*name;

	sec = ap_get_module_config(r->per_dir_config, &authz_ldap_module);

	/* retrieve the certificate in PEM form				*/
	xp = ssl_var_lookup(r->pool, r->server, r->connection, r,
		"SSL_CLIENT_CERT");
	if (NULL == xp) { 
		AUTHZ_DEBUG2("[%d] not certificate found", (int)getpid());
		return NULL;
	}

	/* turn the PEM data into a X.509 structure			*/
	bio_mem = BIO_new(BIO_s_mem());
	BIO_write(bio_mem, xp, strlen(xp) + 1);
	xs = PEM_read_bio_X509(bio_mem, NULL, NULL, NULL);
	BIO_free(bio_mem);
	if (NULL == xs) {
		AUTHZ_DEBUG3("[%d] could not read certificate from PEM: %s",
			(int)getpid(), xp);
		return NULL;
	}

	/* build the data						*/
	bio_mem = BIO_new(BIO_s_mem());
	BIO_puts(bio_mem, "X509:<I>");
	name = X509_get_issuer_name(xs);
	/* This is essentially the format in RFC2253, except it is big
	   endian, i.e. country or dc come first and cn is last.  There
	   is a small gotcha that OpenSSL cannot do, we fix it later */
	X509_NAME_print_ex(bio_mem, name, 0,
			   ASN1_STRFLGS_RFC2253 |
			   XN_FLAG_SEP_COMMA_PLUS |
			   XN_FLAG_FN_SN |
			   XN_FLAG_DUMP_UNKNOWN_FIELDS);
	BIO_puts(bio_mem, "<S>");
	name = X509_get_subject_name(xs);
	X509_NAME_print_ex(bio_mem, name, 0,
			   ASN1_STRFLGS_RFC2253 |
			   XN_FLAG_SEP_COMMA_PLUS |
			   XN_FLAG_FN_SN |
			   XN_FLAG_DUMP_UNKNOWN_FIELDS);

	X509_free(xs);

	l = BIO_ctrl_pending(bio_mem);
	v = ap_palloc(r->pool, l);
	BIO_read(bio_mem, v, l);
	BIO_free(bio_mem);

	/* Now, Active directory uses E for emailAddress, so we have to
	   fix it ourselves */
	at_boundary = 1;
	for ( p=q=v; *p; p++,q++ ) {
		*q = *p;
		if ( at_boundary &&
		     !strncasecmp(p, "emailAddress=", strlen("emailAddress=") ) ) {
			*p = 'E';
			/* Leave p pointing at the last 's' before the
			   equals sign */
			p += strlen("emailAddress")-1;
		}
		if ( *p == ',' )
			at_boundary = 1;
		else
			at_boundary = 0;
	}
	*q = *p;
	AUTHZ_DEBUG3("[%d] client MS X500 altSecurityIdentity: %s", (int)getpid(),
		(v) ? v : "(null)");
	return v;
#else /* AUTHZ_LDAP_HAVE_SSL */
	return NULL;
#endif /* AUTHZ_LDAP_HAVE_SSL */
}

/* OID used as othername to carry the userPrincipalName according to MS */
#define USER_PRINCIPAL_NAME_OID		"1.3.6.1.4.1.311.20.2.3"

char	*authz_ldap_get_ms_user_principal_name(request_rec *r) {
#ifdef AUTHZ_LDAP_HAVE_SSL
	BIO		*bio_mem;
	int		l;
	char		*v;
	char		*xp;
	authz_ldap_config_rec	*sec;
	X509		*xs;
	int		extension_index;
	ASN1_OBJECT	*upnoid;

	sec = ap_get_module_config(r->per_dir_config, &authz_ldap_module);

	/* retrieve the certificate in PEM form				*/
	xp = ssl_var_lookup(r->pool, r->server, r->connection, r,
		"SSL_CLIENT_CERT");
	if (NULL == xp) { 
		AUTHZ_DEBUG2("[%d] not certificate found", (int)getpid());
		return NULL;
	}

	/* turn the PEM data into a X.509 structure			*/
	bio_mem = BIO_new(BIO_s_mem());
	BIO_write(bio_mem, xp, strlen(xp) + 1);
	xs = PEM_read_bio_X509(bio_mem, NULL, NULL, NULL);
	BIO_free(bio_mem);
	if (NULL == xs) {
		AUTHZ_DEBUG3("[%d] could not read certificate from PEM: %s",
			(int)getpid(), xp);
		return NULL;
	}

	v = NULL;
	upnoid = OBJ_txt2obj(USER_PRINCIPAL_NAME_OID,0);

	for ( extension_index=X509_get_ext_by_NID(xs, NID_subject_alt_name, -1);
	      extension_index>0;
	      extension_index=X509_get_ext_by_NID(xs, NID_subject_alt_name,
						  extension_index) ) {
		X509_EXTENSION	*extension;
		STACK_OF(GENERAL_NAME)	*subjectaltnames;
		GENERAL_NAME	*subjectaltname;

		extension = X509_get_ext(xs, extension_index);
		if ( !extension ) {
			AUTHZ_DEBUG3("[%d] error getting subjectaltname with index %d",
				     (int)getpid(), extension_index);
			X509_free(xs);
			return NULL;
		}
		subjectaltnames = X509V3_EXT_d2i(extension);
		if ( !subjectaltnames ) {
			AUTHZ_DEBUG3("[%d] error parsing subjectaltname with index %d",
				     (int)getpid(), extension_index);
			X509_free(xs);
			return NULL;
		}
	  
		/* X509_EXTENSION_free(extension); */

		/* subjectaltnames is a stack so it could, in theory
		 * contain more than one name, we don't think it
		 * happens in this case
		 */
		subjectaltname = sk_GENERAL_NAME_value(subjectaltnames, 0);
	  
		if ( subjectaltname->type != GEN_OTHERNAME ) {
			AUTHZ_DEBUG3("[%d] seen subjectaltname of different type %d",
				     (int)getpid(), subjectaltname->type);
			continue;
		} else if ( OBJ_cmp(subjectaltname->d.otherName->type_id,
				    upnoid) ) {
			AUTHZ_DEBUG3("[%d] seen othername of different type %s",
				     (int)getpid(), "TBD");
			continue;
		} else if ( subjectaltname->d.otherName->value->type !=
			    V_ASN1_UTF8STRING) {
			AUTHZ_DEBUG3("[%d] seen userPrincipalName with different string type %d",
				     (int)getpid(),
				     subjectaltname->d.otherName->value->type);
			continue;
		} else {
	  		/* copy the value out */
			l = subjectaltname->d.otherName->value->value.utf8string->length;
			v = ap_palloc(r->pool, l+1);
			memcpy(v,
			       subjectaltname->d.otherName->value->value.utf8string->data,
			       l);
			v[l] = '\0';
			AUTHZ_DEBUG3("[%d] found userPrincipalName: %s",
				     (int)getpid(), v);
			break;
		}
        }

	X509_free(xs);

	return v;
#else /* AUTHZ_LDAP_HAVE_SSL */
	return NULL;
#endif /* AUTHZ_LDAP_HAVE_SSL */
}

/* utility function to escape parentheses in filter expressions */
static void	authz_ldap_escape_parentheses(char *org, char *new) {
	char	*p = org, *q = new;
	while ((*p) && ((q - new) < MAX_STRING_LEN))
		if ((*p == '(') || (*p == ')')) {
			*q++ = '\\';
			*q++ = *p++;
		}
	*q = '\0';
}

/* utility function: map a certificate to a user name			*/
/* returns 1 if mapping was successful, 0 otherwise			*/
int	authz_ldap_map_user(request_rec *r) {
	authz_ldap_config_rec	*sec;
	char			dn[MAX_STRING_LEN];
	char			*attr, *issuer, *subject, *serial,
				*certificate;
	const char		*userdn;
	char			*base;
	int			scope;
	int			nentries;
	LDAPMessage		*result, *e;
	char			filter[MAX_STRING_LEN];
	char			*attrs[3];
	BerElement		*ber;
	char			**vals;
	int			attr_index;
	char			newname[MAX_STRING_LEN];
	char			escaped_issuer[MAX_STRING_LEN];
	char			escaped_subject[MAX_STRING_LEN];

	sec = ap_get_module_config(r->per_dir_config, &authz_ldap_module);

	/* if the connection notes table contains a value for 		*/
	/* authz_ldap::userdn, the mapping for the certificate was	*/
	/* successful, and we don't need to map again.			*/
	userdn = authz_ldap_get_userdn(r);
	if ((userdn != NULL) && (strlen(userdn) > 0)) {
		AUTHZ_DEBUG3("[%d] user mapping known: %s", (int)getpid(),
			userdn);
		return 1;
	}

	/* find issuer and subject distinguished name (short form)	*/
	issuer = authz_ldap_get_issuer_dn(r);
	if (NULL == issuer) {
		AUTHZ_DEBUG2("[%d] issuer not found in certificate",
			(int)getpid());
		return 0;
	}
	authz_ldap_escape_parentheses(issuer, escaped_issuer);

	subject = authz_ldap_get_subject_dn(r);
	if (NULL == subject) {
		AUTHZ_DEBUG2("[%d] subject not found in certificate",
			(int)getpid());
		return 0;
	}
	authz_ldap_escape_parentheses(subject, escaped_subject);
	authz_ldap_set_subject(r, subject);

	serial = authz_ldap_get_certserial(r);
	if (NULL == serial) {
		AUTHZ_DEBUG2("[%d] serial not found in certificate",
			(int)getpid());
		return 0;
	}
	authz_ldap_set_serial(r, serial);

	certificate = authz_ldap_get_certificate(r);
	if (NULL == certificate) {
		AUTHZ_DEBUG2("[%d] certificate not found in certificate",
			(int)getpid());
		return 0;
	}

	/* there are essentially four mapping functions, here is how	*/
	/* the right one gets selected:					*/
	if (AUTHMAPMETHOD_NONE == sec->mapmethod) {
		AUTHZ_DEBUG2("[%d] map method none called, should not happen",
			(int)getpid());
		return 0;
	}

	/* build a filter for the certificate map			*/
	if (AUTHMAPMETHOD_CERTIFICATE == sec->mapmethod) {
		ap_snprintf(filter, MAX_STRING_LEN,
			"(&(userCertificate=%s)"
			"(objectClass=strongAuthenticationUser))", certificate);
		base = sec->userbase;
		scope = sec->userscope;
		AUTHZ_DEBUG2("[%d] preparing certificate search",
			(int)getpid());
	}
	if (AUTHMAPMETHOD_ISSUERSERIAL == sec->mapmethod) {
		ap_snprintf(filter, MAX_STRING_LEN,
			"(&(issuerDN=%s)(serialNumber=%s)"
			"(objectClass=authzLDAPmap))", escaped_issuer, serial);
		base = sec->mapbase;
		scope = sec->mapscope;
		AUTHZ_DEBUG2("[%d] preparing issuer/serial search",
			(int)getpid());
	}
	if (AUTHMAPMETHOD_ISSUERSUBJECT == sec->mapmethod) {
		ap_snprintf(filter, MAX_STRING_LEN,
			"(&(issuerDN=%s)(subjectDN=%s)"
			"(objectClass=authzLDAPmap))",
			escaped_issuer, escaped_subject);
		base = sec->mapbase;
		scope = sec->mapscope;
		AUTHZ_DEBUG2("[%d] preparing issuer/subject search",
			(int)getpid());
	}
	if (AUTHMAPMETHOD_AD == sec->mapmethod) {
		char * ms_upn;
		char * ms_asi;
		AUTHZ_DEBUG2("[%d] embarking on active directory search",
			(int)getpid());
		if (NULL != (ms_upn = authz_ldap_get_ms_user_principal_name(r))) {
			ap_snprintf(filter, MAX_STRING_LEN,
				    "(userPrincipalName=%s)", ms_upn);
		} else if (NULL != (ms_asi =
			    authz_ldap_get_ms_x500_alt_security_identity(r))) {
			ap_snprintf(filter, MAX_STRING_LEN,
				    "(altSecurityIdentities=%s)", ms_asi);
		} else {
			AUTHZ_DEBUG2("[%d] mapping for AD failed: ",
				     (int)getpid());
			ap_snprintf(filter, MAX_STRING_LEN,
				    "(nosuchattribute=nosuchvalue)");
		}
		base = sec->mapbase;
		scope = sec->mapscope;
	}
	AUTHZ_DEBUG4("[%d] authz_ldap_search for certificate mapping: "
		"base = %s, filter = %s", (int)getpid(), sec->mapbase, filter);

	/* we have to retrieve the dn from the directory, which is 	*/
	/* either the DN of the object returned or the value of the	*/
	/* owner attribute						*/
	attr_index = 0;
	if (AUTHMAPMETHOD_NEEDSOWNER(sec->mapmethod)) {
		attrs[attr_index] = "owner";
		attr_index++;
		AUTHZ_DEBUG2("[%d] authz_ldap need owner", (int)getpid());
	}
	if (sec->mapusertoattr) {
		attrs[attr_index] = sec->mapusertoattr;
		attr_index++;
		AUTHZ_DEBUG3("[%d] authz_ldap need attribute: %s",
			(int)getpid(), sec->mapusertoattr);
	}
	if ( attr_index == 0 ) {
		attrs[attr_index] = LDAP_NO_ATTRS;
		attr_index++;
  	}
	attrs[attr_index] = NULL;
	/*attrs[1] = NULL; this seems to be wrong */
	if (authz_ldap_search(r, base, scope,
		filter, attrs, 0, &result) != LDAP_SUCCESS) {
		if (sec->loglevel >= APLOG_ERR)
		ap_log_rerror(APLOG_MARK, APLOG_NOERRNO|APLOG_ERR,
			APLOG_STATUS r,
			"[%d] certificate owner for [ %s | %s ] not found",
			(int)getpid(), issuer, subject);
		return 0;
	}

	/* make sure the number of entries is exactly 1			*/
	nentries = ldap_count_entries(sec->ldap, result);
	if (nentries == 0) {
		ldap_msgfree(result);
		if (sec->loglevel >= APLOG_ERR)
		ap_log_rerror(APLOG_MARK, APLOG_NOERRNO|APLOG_ERR,
			APLOG_STATUS r,
			"[%d] filter: %s base: %s, no such user", (int)getpid(),
			filter, base);
		return 0;
	}
	if (nentries > 1) {
		ldap_msgfree(result);
		if (sec->loglevel >= APLOG_ERR)
		ap_log_rerror(APLOG_MARK, APLOG_NOERRNO|APLOG_ERR,
			APLOG_STATUS r,
			"[%d] filter: %s base: %s, not unique", (int)getpid(),
			filter, base);
		return 0;
	}

	/* get the attribute value for the owner attribute		*/
	dn[0] = '\0';
	newname[0] = '\0';
	e = ldap_first_entry(sec->ldap, result);
	if (e != NULL) {
		if (AUTHMAPMETHOD_RETURNSUSER(sec->mapmethod)) {
			/* retrieve the DN from the user		*/
			char	*d;
			d = ldap_get_dn(sec->ldap, e);
			apr_cpystrn(dn, d, MAX_STRING_LEN);
			ldap_memfree(d);
			authz_ldap_set_userdn(r, dn);
			AUTHZ_DEBUG3("[%d] userdn set to %s", (int)getpid(),
				dn);

			/* if mapping to an attribute is requested	*/
			/* then there should be an attribute with that	*/
			/* value					*/
			if (sec->mapusertoattr) {
				vals = ldap_get_values(sec->ldap, e,
						       sec->mapusertoattr);
				if (vals != NULL) {
					apr_cpystrn(newname, vals[0],
						MAX_STRING_LEN);
					/* remember the mapped name	*/
					authz_ldap_set_mapped(r, newname);
					AUTHZ_DEBUG3("[%d] mapped set to %s",
						(int)getpid(), newname);
					ldap_value_free(vals);
				}
			}
			/* XXX We should really support this in other 	*/
			/* mappings 					*/
		} else {
			/* map method does not return a user, so we 	*/
			/* we have to go after the owner attribute	*/
			attr = ldap_first_attribute(sec->ldap, e, &ber);
			if (attr != NULL) {
				vals = ldap_get_values(sec->ldap, e, attr);
				if (vals != NULL) {
					apr_cpystrn(dn, vals[0], MAX_STRING_LEN);
					ldap_value_free(vals);
					authz_ldap_set_userdn(r, dn);
					AUTHZ_DEBUG4("[%d] userdn set to %s from [%s]",
						(int)getpid(), dn, attr);
					if (sec->mapusertoattr) {
						AUTHZ_DEBUG3("[%d] fetching map attribute [%s]",
							(int)getpid(), sec->mapusertoattr );

						vals = ldap_get_values(sec->ldap, e,
							sec->mapusertoattr);
						if (vals != NULL) {
							apr_cpystrn(newname, vals[0], MAX_STRING_LEN);
							authz_ldap_set_mapped(r, newname);
							AUTHZ_DEBUG4("[%d] mapped set to %s from [%s]",
								(int)getpid(), newname, sec->mapusertoattr);
							ldap_value_free(vals);
						}
					}
				}
				/* XXX not needed?
				if (ber != NULL)
					ber_free(ber, 0);
				*/
				ldap_memfree(attr);
			}
		}
	}
	ldap_msgfree(result);

	AUTHZ_DEBUG3("[%d] owner dn: %s", (int)getpid(),
		authz_ldap_get_userdn(r));

	/* set the the user to the distinguished name of the		*/
	/* certificate owner						*/
	AUTHZ_DEBUG5("[%d] user [ %s | %s ] mapped to '%s'", (int)getpid(),
		issuer, subject, newname ? newname : dn);

	/* that's it, if we get to this point, we can assume the 	*/
	/* mapping was a success					*/
	return 1;
}
