/* 
   Copyright (C) Andrew Tridgell 1998-2003,
   Con Kolivas 2006-2008
   
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
   
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/
/* lrzip compression - main program */
#include "rzip.h"
#include <sys/time.h>

static unsigned long hard_limit;

static void usage(void)
{
	printf("lrzip version %d.%d%d\n", LRZIP_MAJOR_VERSION, LRZIP_MINOR_VERSION, LRZIP_MINOR_SUBVERSION);
	printf("Copyright (C) Con Kolivas 2006-2008\n\n");
	printf("Based on rzip ");
	printf("Copyright (C) Andrew Tridgell 1998-2003\n");
	printf("usage: lrzip [options] <file...>\n");
	printf(" Options:\n");
	printf("     -w size       compression window in hundreds of MB\n");
	printf("                   default chosen by heuristic dependant on ram and chosen compression\n");
	printf("     -d            decompress\n");
	printf("     -o filename   specify the output file name and/or path\n");
	printf("     -O directory  specify the output directory when -o is not used\n");
	printf("     -S suffix     specify compressed suffix (default '.lrz')\n");
	printf("     -f            force overwrite of any existing files\n");
	printf("     -D            delete existing files\n");
	printf("     -q            don't show compression progress\n");
	printf("     -L level      set compression level (1-9, default 7)\n");
	printf("     -n            no backend compression - prepare for other compressor\n");
	printf("     -l            lzo compression (ultra fast)\n");
	printf("     -b            bzip2 compression\n");
	printf("     -g            gzip compression using zlib\n");
	printf("     -M            Maximum compression - (level 9 and all available ram)\n");
	printf("     -T value      LZMA compression threshold with LZO test. (1 (low) - 10 (high), default 2)\n");
	printf("     -N value      Set nice value to value (default 19)\n");
	printf("     -v[v]         Increase verbosity\n");
	printf("     -V            show version\n");
#if 0
	/* damn, this will be quite hard to do */
	printf("     -t          test compressed file integrity\n");
#endif
	printf("\nnote that lrzip cannot operate on stdin/stdout\n");
}

static void write_magic(int fd_in, int fd_out, struct rzip_control *control)
{
	struct stat st;
	char magic[24];
	uint32_t v;

	memset(magic, 0, sizeof(magic));
	strcpy(magic, "LRZIP");
	magic[4] = LRZIP_MAJOR_VERSION;
	magic[5] = LRZIP_MINOR_VERSION;

	if (fstat(fd_in, &st) != 0) {
		fatal("bad magic file descriptor!?\n");
	}

#if HAVE_LARGE_FILES
	v = htonl(st.st_size & 0xFFFFFFFF);
	memcpy(&magic[6], &v, 4);
	v = htonl(st.st_size >> 32);
	memcpy(&magic[10], &v, 4);
#else
	v = htonl(st.st_size);
	memcpy(&magic[6], &v, 4);
#endif

	/* save LZMA compression flags */
	if ( LZMA_COMPRESS(control->flags)) {
		magic[16] = (unsigned char) control->compression_level;
		magic[17] = (unsigned char) control->lc;
		magic[18] = (unsigned char) ((control->lp << 4) + control->pb);
	}

	if (write(fd_out, magic, sizeof(magic)) != sizeof(magic)) {
		fatal("Failed to write magic header\n");
	}
}

static void read_magic(int fd_in, off_t *expected_size, struct rzip_control *control)
{
	uint32_t v;
	char magic[24];

	if (read(fd_in, magic, sizeof(magic)) != sizeof(magic)) {
		fatal("Failed to read magic header\n");
	}

	*expected_size = 0;

	if (strncmp(magic, "LRZIP", 4) != 0) {
		fatal("Not an lrzip file\n");
	}

#if HAVE_LARGE_FILES
	memcpy(&v, &magic[6], 4);
	*expected_size = ntohl(v);
	memcpy(&v, &magic[10], 4);
	*expected_size |= ((off_t)ntohl(v)) << 32;
#else
	memcpy(&v, &magic[6], 4);
	*expected_size = ntohl(v);
#endif

	/* restore LZMA compression flags only if stored */
	if ( (int) magic[16] ) {
		control->compression_level = (unsigned short) magic[16];
		control->lc = (unsigned short) magic[17];
		control->pb = (unsigned short) magic[18] & 0xf;
		control->lp = (unsigned short) magic[18] >> 4;
	}
}


/* preserve ownership and permissions where possible */
static void preserve_perms(struct rzip_control *control,
			   int fd_in, int fd_out)
{
	struct stat st;

	if (fstat(fd_in, &st) != 0) {
		fatal("Failed to fstat input file\n");
	}
	if (fchmod(fd_out, (st.st_mode & 0777)) != 0) {
		fatal("Failed to set permissions on %s\n", control->outfile);
	}

	/* chown fail is not fatal */
	fchown(fd_out, st.st_uid, st.st_gid);
}	

	

/*
  decompress one file from the command line
*/
static void decompress_file(struct rzip_control *control)
{
	int fd_in, fd_out = -1, fd_hist = -1;
	off_t expected_size;
	char *tmp, *tmpoutfile, *infilecopy;

	/* make sure infile has an extension. If not, add it 
	  * because manipulations may be made to input filename, set local ptr
	*/
	if ((tmp = strrchr(control->infile,'.')) && strcmp(tmp,control->suffix)) {
		infilecopy = malloc(strlen(control->infile)+strlen(control->suffix)+1);
		if (infilecopy==NULL)
			fatal("failed to allocate memory for infile suffix\n");
		else {
			strcpy(infilecopy, control->infile);
			strcat(infilecopy, control->suffix);
		}
//		printf("Input filename is: %s\n", infilecopy);
	} else
		infilecopy = strdup(control->infile);

	/* regardless, infilecopy has the input filename */

	/* if output name already set, use it */
	if (control->outname)
		control->outfile = strdup(control->outname);
	else {
		/* default output name from infilecopy
		 * test if outdir specified. If so, strip path from filename of
		 * infilecopy, then remove suffix.
		*/
		if (control->outdir && (tmp=strrchr(infilecopy,'/')))
			tmpoutfile=strdup(tmp+1);
		else
			tmpoutfile = strdup(infilecopy);

		/* remove suffix to make outfile name */
		if ((tmp = strrchr(tmpoutfile,'.')) && !strcmp(tmp,control->suffix))
			*tmp='\0';

		control->outfile = malloc((control->outdir==NULL?0:strlen(control->outdir)) +
					  strlen(tmpoutfile) + 1);
		if (!control->outfile) 
			fatal("Failed to allocate outfile name\n");
	
		if (control->outdir) {	/* prepend control->outdir */
			strcpy(control->outfile, control->outdir);
			strcat(control->outfile, tmpoutfile);
		} else
			strcpy(control->outfile, tmpoutfile);
		free(tmpoutfile);
	}
	if (control->flags & FLAG_SHOW_PROGRESS) {
		printf("Output filename is: %s...Decompressing...", control->outfile);
		fflush(stdout);
	}

	fd_in = open(infilecopy,O_RDONLY);
	if (fd_in == -1) {
		fatal("Failed to open %s: %s\n", 
		      infilecopy, 
		      strerror(errno));
	}

	if ((control->flags & FLAG_TEST_ONLY) == 0) {
		if (control->flags & FLAG_FORCE_REPLACE) {
			fd_out = open(control->outfile,O_WRONLY|O_CREAT|O_TRUNC,0666);
		} else {
			fd_out = open(control->outfile,O_WRONLY|O_CREAT|O_EXCL,0666);
		}
		if (fd_out == -1) {
			fatal("Failed to create %s: %s\n", 
			      control->outfile, strerror(errno));
		}

		preserve_perms(control, fd_in, fd_out);
		
		fd_hist = open(control->outfile,O_RDONLY);
		if (fd_hist == -1) {
			fatal("Failed to open history file %s\n", 
			      control->outfile);
		}
	}

	
	read_magic(fd_in, &expected_size, control);
	runzip_fd(fd_in, fd_out, fd_hist, expected_size);

	/* if we get here, no fatal errors during decompression */
	printf("\rOutput filename is: %s: [OK] - %lld bytes                                 \n", control->outfile, expected_size);
	
	if ((control->flags & FLAG_TEST_ONLY) == 0) {
		if (close(fd_hist) != 0 ||
		    close(fd_out) != 0) {
			fatal("Failed to close files\n");
		}
	}

	close(fd_in);

	if ((control->flags & (FLAG_KEEP_FILES | FLAG_TEST_ONLY)) == 0) {
		if (unlink(control->infile) != 0) {
			fatal("Failed to unlink %s: %s\n", 
			      infilecopy, strerror(errno));
		}
	}

	free(control->outfile);
	free(infilecopy);
}

/*
  compress one file from the command line
*/
static void compress_file(struct rzip_control *control)
{
	int fd_in, fd_out;
	const char *tmp, *tmpinfile; 	/* we're just using this as a proxy for control->infile.
				 	 * Spares a compiler warning
			       		*/

	/* is extension at end of infile? */
	if ((tmp = strrchr(control->infile,'.')) && !strcmp(tmp,control->suffix)) {
		printf("%s: already has %s suffix. Skipping...\n", control->infile, control->suffix);
		return;
	}

	if (control->outname) {
		/* check if outname has control->suffix */
		if ((tmp=strrchr(control->outname,'.')) && strcmp(tmp,control->suffix)) {
			control->outfile = malloc(strlen(control->outname) +
						  strlen(control->suffix) + 1);
			if (!control->outfile) {
				fatal("Failed to allocate outfile name\n");
			}
			strcpy(control->outfile, control->outname);
			strcat(control->outfile, control->suffix);
			printf("Suffix added to %s.\nFull pathname is: %s\n", control->outname, control->outfile);
		} else	/* no, already has suffix */
			control->outfile = strdup(control->outname);
	} else {
		/* default output name from control->infile
		 * test if outdir specified. If so, strip path from filename of
		 * control->infile
		*/
		if (control->outdir && (tmp=strrchr(control->infile,'/')))
			tmpinfile=tmp+1;
		else
			tmpinfile=control->infile;

		control->outfile = malloc((control->outdir==NULL?0:strlen(control->outdir)) +
					  strlen(tmpinfile) + strlen(control->suffix) + 1);
		if (!control->outfile)
			fatal("Failed to allocate outfile name\n");

		if (control->outdir) {	/* prepend control->outdir */
			strcpy(control->outfile, control->outdir);
			strcat(control->outfile, tmpinfile);
		} else
			strcpy(control->outfile, tmpinfile);
		strcat(control->outfile, control->suffix);
		if ( control->flags & FLAG_SHOW_PROGRESS )
			printf("Output filename is: %s\n", control->outfile);
	}

	fd_in = open(control->infile,O_RDONLY);
	if (fd_in == -1) {
		fatal("Failed to open %s: %s\n", control->infile, strerror(errno));
	}
	
	if (control->flags & FLAG_FORCE_REPLACE) {
		fd_out = open(control->outfile,O_WRONLY|O_CREAT|O_TRUNC,0666);
	} else {
		fd_out = open(control->outfile,O_WRONLY|O_CREAT|O_EXCL,0666);
	}
	if (fd_out == -1) {
		fatal("Failed to create %s: %s\n", control->outfile, strerror(errno));
	}

	preserve_perms(control, fd_in, fd_out);

	write_magic(fd_in, fd_out, control);
	rzip_fd(fd_in, fd_out);

	if (close(fd_in) != 0 ||
	    close(fd_out) != 0) {
		fatal("Failed to close files\n");
	}

	if ((control->flags & FLAG_KEEP_FILES) == 0) {
		if (unlink(control->infile) != 0) {
			fatal("Failed to unlink %s: %s\n", control->infile, strerror(errno));
		}
	}

	free(control->outfile);
}

/*
 * Returns ram size on linux/darwin.
 */
#ifdef __APPLE__
static unsigned long get_ram(void)
{
	int mib[2];
	size_t len;
	int *p, ramsize;

	mib[0] = CTL_HW;
	mib[1] = HW_PHYSMEM;
	sysctl(mib, 2, NULL, &len, NULL, 0);
	p = malloc(len);
	sysctl(mib, 2, p, &len, NULL, 0);
	ramsize = *p / 1024; // bytes -> KB

	/* Darwin can't overcommit as much as linux so we return half the ram
	   size to fudge it to use smaller windows */
	ramsize /= 2;
	if (ramsize > hard_limit)
		ramsize = hard_limit;
	return ramsize;
}
#else
static unsigned long get_ram(void)
{
        long long ramsize;
	FILE *meminfo;
        char aux[256];

	if(!(meminfo = fopen("/proc/meminfo", "r")))
		fatal("fopen\n");

	while( !feof(meminfo) && !fscanf(meminfo, "MemTotal: %Lu kB", &ramsize) )
		fgets(aux,sizeof(aux),meminfo);
	if (fclose(meminfo) == -1)
		fatal("fclose");
	if (ramsize > hard_limit)
		ramsize = hard_limit;
	return (unsigned long)ramsize;
}
#endif

struct rzip_control control;
int main(int argc, char *argv[])
{
	extern int optind;
	int c, i, nice_val = 19;
	struct timeval start_time, end_time;
	int hours,minutes;
	double seconds,total_time; // for timers

	memset(&control, 0, sizeof(control));

	control.flags = FLAG_SHOW_PROGRESS | FLAG_KEEP_FILES;
	control.suffix = ".lrz";
	control.outdir = NULL;

	if (strstr(argv[0], "lrunzip")) {
		control.flags |= FLAG_DECOMPRESS;
	}

	/* Shafted by 32 bit limits we can only use 1/3 of 4GB */
	hard_limit =(1ULL << 32) / 3 / 1024;
	control.compression_level = 7;
	control.ramsize = get_ram() / 104858; /* hundreds of megabytes */
	control.window = 0;
	control.threshold = 0.95;	/* default lzo test compression threshold (level 2) with LZMA compression */
	/* for testing single CPU */
	#ifndef NOTHREAD
	control.threads = sysconf(_SC_NPROCESSORS_CONF);	/* get CPUs for LZMA */
	#else
	control.threads = 1;
	#endif

	/* generate crc table */
	CrcGenerateTable();

	while ((c = getopt(argc, argv, "L:hdS:tVvDfqo:w:nlbMO:T:N:g")) != -1) {
#if 0
		if (isdigit(c)) {
			control.compression_level = c - '0';
			continue;
		}
#endif
		switch (c) {
		case 'L':
			control.compression_level = atoi(optarg);
			if (control.compression_level < 1 || control.compression_level > 9)
				fatal("Invalid compression level (must be 1-9)\n");
			break;
		case 'w':
			control.window = atoi(optarg);
			break;
		case 'd':
			control.flags |= FLAG_DECOMPRESS;
			break;
		case 'S':
			control.suffix = optarg;
			break;
		case 'o':
			if (control.outdir)
				fatal("Cannot have -o and -O together\n");
			control.outname = optarg;
			break;
		case 't':
			fatal("integrity checking currently not implemented\n");
			control.flags |= FLAG_TEST_ONLY;
			break;
		case 'f':
			control.flags |= FLAG_FORCE_REPLACE;
			break;
		case 'D':
			control.flags &= ~FLAG_KEEP_FILES;
			break;
		case 'v':
			/* set verbosity flag */
			if ( !(control.flags & FLAG_VERBOSITY) && !(control.flags & FLAG_VERBOSITY_MAX) )
				control.flags |= FLAG_VERBOSITY;
			else if ( (control.flags & FLAG_VERBOSITY) ) {
				control.flags &= ~FLAG_VERBOSITY;
				control.flags |= FLAG_VERBOSITY_MAX;
			}
			break;
		case 'q':
			control.flags &= ~FLAG_SHOW_PROGRESS;
			break;
		case 'V':
			printf("lrzip version %d.%d%d\n",
				LRZIP_MAJOR_VERSION, LRZIP_MINOR_VERSION, LRZIP_MINOR_SUBVERSION);
			exit(0);
			break;
		case 'l':
			if (control.flags & FLAG_NOT_LZMA)
				fatal("Can only use one of -l, -b, -g or -n\n");
			control.flags |= FLAG_LZO_COMPRESS;
			break;
		case 'b':
			if (control.flags & FLAG_NOT_LZMA)
				fatal("Can only use one of -l, -b, -g or -n\n");
			control.flags |= FLAG_BZIP2_COMPRESS;
			break;
		case 'n':
			if (control.flags & FLAG_NOT_LZMA)
				fatal("Can only use one of -l, -b, -g or -n\n");
			control.flags |= FLAG_NO_COMPRESS;
			break;
		case 'M':
			control.compression_level = 9;
			control.window = control.ramsize / 10 * 9 ? : 1;
			break;
		case 'O':
			if (control.outname)	/* can't mix -o and -O */
				fatal("Cannot have options -o and -O together\n");
			if (strcmp(optarg+strlen(optarg)-1,"/")) {	/* need a trailing slash */
				control.outdir=malloc(strlen(optarg)+2);
				if (control.outdir == NULL)
					fatal("Failed to allocate for outdir\n");
				else {
					strcpy(control.outdir,optarg);
					strcat(control.outdir,"/");
				}
			} else		/* no, trailing slash already supplied */
				control.outdir = optarg;
			break;
		case 'T':
			/* invert argument, a threshold of 1 means that the compressed result can be
			 * 90%-100% of the sample size
			*/
			control.threshold = atoi(optarg);
			if (control.threshold < 1 || control.threshold > 10)
				fatal("Threshold value must be between 1 and 10\n");
			control.threshold = 1.05-control.threshold / 20;
			break;
		case 'N':
			nice_val = atoi(optarg);
			if (nice_val < -20 || nice_val > 19)
				fatal("Invalid nice value (must be -20..19)\n");
			break;
		case 'g':
			if (control.flags & FLAG_NOT_LZMA)
				fatal("Can only use one of -l, -b, -g or -n\n");
			control.flags |= FLAG_ZLIB_COMPRESS;
			break;
		default:
		case 'h':
			usage();
			return -1;
		}
	}

	argc -= optind;
	argv += optind;

	if (control.outname && argc > 1)
		fatal("Cannot specify output filename with more than 1 file\n");

	if (( (control.flags & FLAG_VERBOSITY) || (control.flags & FLAG_VERBOSITY_MAX) )
			&& !(control.flags & FLAG_SHOW_PROGRESS)) {
		printf("Cannot have -v and -q options. -v wins.\n");
		control.flags |= FLAG_SHOW_PROGRESS;
	}

	if (argc < 1) {
		usage();
		exit(1);
	}

	if (control.window > control.ramsize)
		printf("Compression window has been set to larger than ramsize, proceeding at your request. If you did not mean this, abort now.\n"); 

	/* The control window chosen is the largest that will not cause
	   massive swapping on the machine (60% of ram). Most of the pages
	   will be shared by lzma even though it uses just as much ram itself 
	   */
	if (!control.window)
		control.window = control.ramsize / 3 * 2 ? : 1;

	if (control.window > 14) {
		control.window = 14;
		printf("Limiting control window to 14 due to limitations in current version of lrzip.\n");
	}

	/* OK, if verbosity set, print summary of options selected */
	if ((control.flags & FLAG_VERBOSITY) || (control.flags & FLAG_VERBOSITY_MAX)) {
		printf("The following options are in effect for this %s.\n", 
			control.flags & FLAG_DECOMPRESS ? "DECOMPRESSION" : "COMPRESSION");
		if (LZMA_COMPRESS(control.flags))
			printf("Threading is %s. Number of CPUs detected: %d\n", control.threads > 1 ?"ENABLED":"DISABLED",
				control.threads);
		printf("Nice Value: %d\n", nice_val);
		if (control.flags & FLAG_SHOW_PROGRESS)
			printf("Show Progress\n");
		if (control.flags & FLAG_VERBOSITY)
			printf("Verbose\n");
		else if (control.flags & FLAG_VERBOSITY_MAX)
			printf("Max Verbosity\n");
		if (control.flags & FLAG_FORCE_REPLACE)
			printf("Overwrite Files\n");
		if (!(control.flags & FLAG_KEEP_FILES))
			printf("Remove input files on completion\n");
		if (control.outdir)
			printf("Output Directory Specified: %s\n", control.outdir);
		else if (control.outname)
			printf("Output Filename Specified: %s\n", control.outname);

		/* show compression options */
		if (!(control.flags & FLAG_DECOMPRESS)) {
			printf("Compression mode is: ");
			if (LZMA_COMPRESS(control.flags))
				printf("LZMA. LZO Test Compression Threshold: %.f\n",
						(control.threshold<1?11-control.threshold*10:1));
			else if (control.flags & FLAG_LZO_COMPRESS)
				printf("LZO\n");
			else if (control.flags & FLAG_BZIP2_COMPRESS)
				printf("BZIP2\n");
			else if (control.flags & FLAG_ZLIB_COMPRESS)
				printf("GZIP\n");
			else if (control.flags & FLAG_NO_COMPRESS)
				printf("RZIP\n");
			printf("Compression Window: %u = %uMB\n", control.window, control.window*104858);
			printf("Compression Level: %d\n", control.compression_level);
		}
		printf("\n");
	}

	if (setpriority(PRIO_PROCESS, 0, nice_val) == -1)
		fatal("Unable to set nice value\n");

	/* set default lzma settings */
	if (LZMA_COMPRESS(control.flags)) {
		control.lc = 3;
		control.lp = 0;
		control.pb = 2;
	}

	for (i=0;i<argc;i++) {
		control.infile = argv[i];
		gettimeofday(&start_time, NULL);

		if (control.flags & (FLAG_DECOMPRESS | FLAG_TEST_ONLY))
			decompress_file(&control);
		else
			compress_file(&control);

		/* compute total time */
		gettimeofday(&end_time, NULL);
		total_time = ( end_time.tv_sec + (double)end_time.tv_usec / 1000000 ) - 
				( start_time.tv_sec + (double)start_time.tv_usec / 1000000 );
		hours = (int)total_time / 3600;
		minutes = (int)(total_time - hours * 3600) / 60;
		seconds = total_time - hours * 60 - minutes * 60;
		if ( control.flags & FLAG_SHOW_PROGRESS )
			printf("Total time: %dh %dm %2.3fs\n", hours, minutes, seconds);
	}

	return 0;
}
