/*
 * lzma zlib simplified wrapper
 *
 * Copyright (c) 2005 Oleg I. Vdovikin <oleg@cs.msu.su>
 *
 * This library is free software; you can redistribute
 * it and/or modify it under the terms of the GNU Lesser
 * General Public License as published by the Free Software
 * Foundation; either version 2.1 of the License, or
 * (at your option) any later version.
 *
 * This library is distributed in the hope that it will be
 * useful, but WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
 * PURPOSE. See the GNU Lesser General Public License
 * for more details.
 *
 * You should have received a copy of the GNU Lesser General
 * Public License along with this library; if not, write to
 * the Free Software Foundation, Inc., 59 Temple Place,
 * Suite 330, Boston, MA 02111-1307 USA
 */

/* small location mods by Peter Hyman */

/*
 * default values for encoder/decoder used by wrapper
 */
#include "lzmalib.h"

#define ZLIB_LC 3
#define ZLIB_LP 0
#define ZLIB_PB 2

#ifdef WIN32
#include <initguid.h>
#else
#define INITGUID
#endif

#include "StdAfx.h"
#include "CPP/Common/MyWindows.h"
#include "CPP/7zip/Compress/LZMA/LZMADecoder.h"
#include "CPP/7zip/Compress/LZMA/LZMAEncoder.h"
#include "CPP/7zip/Common/StreamUtils.h"


#define STG_E_SEEKERROR                  ((HRESULT)0x80030019L)
#define STG_E_MEDIUMFULL                 ((HRESULT)0x80030070L)

class CInMemoryStream:
  public IInStream,
  public IStreamGetSize,
  public CMyUnknownImp
{
public:
  CInMemoryStream(const Bytef *data, UInt64 size) :
	  m_data(data), m_size(size), m_offset(0) {}

  virtual ~CInMemoryStream() {}

  MY_UNKNOWN_IMP2(IInStream, IStreamGetSize)

  STDMETHOD(Read)(void *data, UInt32 size, UInt32 *processedSize)
  {
	  if (size > m_size - m_offset)
		  size = m_size - m_offset;

	  if (size) {
		  memcpy(data, m_data + m_offset, size);
	  }

	  m_offset += size;

	  if (processedSize)
		  *processedSize = size;
	  
	  return S_OK;
  }

  STDMETHOD(ReadPart)(void *data, UInt32 size, UInt32 *processedSize)
  {
	return Read(data, size, processedSize);
  }

  STDMETHOD(Seek)(Int64 offset, UInt32 seekOrigin, UInt64 *newPosition)
  {
	  UInt64 _offset;

	  if (seekOrigin == STREAM_SEEK_SET) _offset = offset;
	  else if (seekOrigin == STREAM_SEEK_CUR) _offset = m_offset + offset;
	  else if (seekOrigin == STREAM_SEEK_END) _offset = m_size;
	  else return STG_E_INVALIDFUNCTION;

	  if (_offset > m_size)
		  return STG_E_SEEKERROR;

	  m_offset = _offset;

	  if (newPosition)
		  *newPosition = m_offset;

	  return S_OK;
  }

  STDMETHOD(GetSize)(UInt64 *size)
  {
	  *size = m_size;
	  return S_OK;
  }
protected:
	const Bytef *m_data;
	UInt64 m_size;
	UInt64 m_offset;
};

class COutMemoryStream:
  public IOutStream,
  public CMyUnknownImp
{
public:
  COutMemoryStream(Bytef *data, UInt64 maxsize) :
	  m_data(data), m_size(0), m_maxsize(maxsize), m_offset(0) {}
  virtual ~COutMemoryStream() {}

  MY_UNKNOWN_IMP1(IOutStream)

  STDMETHOD(Write)(const void *data, UInt32 size, UInt32 *processedSize)
  {
	  if (size != 0) {
		  if (size > m_maxsize - m_offset)
			  size = m_maxsize - m_offset;

		  if (size == 0)	// Lasse Collin
			  return E_FAIL;

		  memcpy(m_data + m_offset, data, size);

		  m_offset += size;
		  if (m_offset > m_size)
			m_size = m_offset;
	  }

	  if (processedSize)
		  *processedSize = size;

	  return S_OK;
  }

  STDMETHOD(WritePart)(const void *data, UInt32 size, UInt32 *processedSize)
  {
	  return Write(data, size, processedSize);
  }

  STDMETHOD(Seek)(Int64 offset, UInt32 seekOrigin, UInt64 *newPosition)
  {
	  UInt64 _offset;

	  if (seekOrigin == STREAM_SEEK_SET) _offset = offset;
	  else if (seekOrigin == STREAM_SEEK_CUR) _offset = m_offset + offset;
	  else if (seekOrigin == STREAM_SEEK_END) _offset = m_size;
	  else return STG_E_INVALIDFUNCTION;

	  if (_offset > m_maxsize)
		  return STG_E_SEEKERROR;

	  m_offset = _offset;

	  if (newPosition)
		  *newPosition = m_offset;

	  return S_OK;
  }

  STDMETHOD(SetSize)(Int64 newSize)
  {
	  if ((UInt64)newSize > m_maxsize)
		  return STG_E_MEDIUMFULL;

	  return S_OK;
  }
protected:
	Bytef *m_data;
	UInt64 m_size;
	UInt64 m_maxsize;
	UInt64 m_offset;
};

ZEXTERN int ZEXPORT lzma_compress OF((Bytef *dest,   uLongf *destLen,
                                  const Bytef *source, uLong sourceLen,
                                  int level, unsigned short lc, unsigned short lp,
				  unsigned short pb, unsigned threads))
{
	CInMemoryStream *inStreamSpec = new CInMemoryStream(source, sourceLen);
	CMyComPtr<ISequentialInStream> inStream = inStreamSpec;

	COutMemoryStream *outStreamSpec = new COutMemoryStream(dest, *destLen);
	CMyComPtr<ISequentialOutStream> outStream = outStreamSpec;

	NCompress::NLZMA::CEncoder *encoderSpec =
		new NCompress::NLZMA::CEncoder;
	CMyComPtr<ICompressCoder> encoder = encoderSpec;
	
	PROPID propIDs[] =
	{
		NCoderPropID::kDictionarySize,
		NCoderPropID::kPosStateBits,
		NCoderPropID::kLitContextBits,
		NCoderPropID::kLitPosBits,
		NCoderPropID::kAlgorithm,
		NCoderPropID::kNumFastBytes,
		NCoderPropID::kMatchFinder,
		NCoderPropID::kEndMarker,
		// for new API
		NCoderPropID::kNumThreads,
		NCoderPropID::kMatchFinderCycles,
	};
	const int kNumProps = sizeof(propIDs) / sizeof(propIDs[0]);

	PROPVARIANT properties[kNumProps];
	for (int p = 0; p < 6; p++)
		properties[p].vt = VT_UI4;
	properties[0].ulVal = UInt32(1 << (level + 16));// Dictionary (default 23 7+16)
	properties[1].ulVal = UInt32(pb?pb:ZLIB_PB);	// posStateBits (0 by default, 2 for 32 bit data)
	properties[2].ulVal = UInt32(lc?lc:ZLIB_LC);	// litContextBits, 3 for normal files. 0 for 32 bit data
	properties[3].ulVal = UInt32(lp?lp:ZLIB_LP); 	// litPostBits, 0 by default, 2 for 32 bit data
	properties[4].ulVal = UInt32(1);	// per new API example (algorithm) 0=fast, 1=max
	properties[5].ulVal = UInt32(128);	// numFastBytes

	properties[6].vt = VT_BSTR;
	properties[6].bstrVal = (BSTR)(const wchar_t *)L"BT4";

	if (level < 6) {
		properties[6].bstrVal = (BSTR)(const wchar_t *)L"BT3";
		if (level < 4) {
			properties[6].bstrVal = (BSTR)(const wchar_t *)L"BT2";
			if (level  == 1) {
				properties[6].bstrVal = (BSTR)(const wchar_t *)L"HC4";
				properties[4].ulVal = UInt32(0);	// Lasse Colin suggestion
			}
		}
	}

	properties[7].vt = VT_BOOL;
	properties[7].boolVal = VARIANT_TRUE;
	// new members
	properties[8].vt = VT_UI4;
	properties[8].ulVal = (UInt32)threads;		// pass argument in function

	// it must be last in property list
	properties[9].vt = VT_UI4;
	properties[9].ulVal = (UInt32) 16 + 128 / 2;	// Match Finder Cycles

	if (encoderSpec->SetCoderProperties(propIDs, properties, kNumProps) != S_OK)
		return Z_MEM_ERROR; // should not happen

	HRESULT result = encoder->Code(inStream, outStream, 0, 0, 0);
	if (result == E_OUTOFMEMORY)
	{
		return Z_MEM_ERROR;
	}
	else if (result != S_OK)
	{
		return Z_BUF_ERROR;	// should not happen
	}

	UInt64 fileSize;
	outStreamSpec->Seek(0, STREAM_SEEK_END, &fileSize);
	*destLen = fileSize;

	return Z_OK;
}

// need level passed to function otherwise decompress will bomb
ZEXTERN int ZEXPORT lzma_uncompress OF((Bytef *dest,   uLongf *destLen,
                                   const Bytef *source, uLong sourceLen, int level, 
				   unsigned short lc, unsigned short lp, unsigned short pb))
{
	CInMemoryStream *inStreamSpec = new CInMemoryStream(source, sourceLen);
	CMyComPtr<ISequentialInStream> inStream = inStreamSpec;

	COutMemoryStream *outStreamSpec = new COutMemoryStream(dest, *destLen);
	CMyComPtr<ISequentialOutStream> outStream = outStreamSpec;

	NCompress::NLZMA::CDecoder *decoderSpec =
		new NCompress::NLZMA::CDecoder;
	CMyComPtr<ICompressCoder> decoder = decoderSpec;
	
	lc=lc?lc:ZLIB_LC;
	lp=lp?lp:ZLIB_LP;
	pb=pb?pb:ZLIB_PB;

	if (decoderSpec->SetDecoderPropertiesRaw(lc, lp, pb,
		(1 << (level+16))) != S_OK) return Z_DATA_ERROR;

	UInt64 fileSize = *destLen;

	if (decoder->Code(inStream, outStream, 0, &fileSize, 0) != S_OK)
	{
		return Z_DATA_ERROR;
	}

	outStreamSpec->Seek(0, STREAM_SEEK_END, &fileSize);
	*destLen = fileSize;

	return Z_OK;
}
