/*
 *  Off-the-Record Messaging Toolkit
 *  Copyright (C) 2004-2008  Ian Goldberg, Chris Alexander, Nikita Borisov
 *                           <otr@cypherpunks.ca>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of version 2 of the GNU General Public License as
 *  published by the Free Software Foundation.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/* system headers */
#include <stdio.h>
#include <stdlib.h>

/* toolkit headers */
#include "parse.h"
#include "sesskeys.h"

static void usage(const char *progname)
{
    fprintf(stderr, "Usage: %s our_privkey their_pubkey\n"
"Calculate and display our public key, the session id, two AES keys,\n"
"and two MAC keys generated by the given DH private key and public key.\n",
	progname);
    exit(1);
}

int main(int argc, char **argv)
{
    unsigned char *argbuf;
    size_t argbuflen;
    gcry_mpi_t our_x, our_y, their_y;
    unsigned char *pubbuf;
    size_t publen;
    unsigned char sessionid[20], sendenc[16], rcvenc[16];
    unsigned char sendmac[20], rcvmac[20];
    int is_high;

    if (argc != 3) {
	usage(argv[0]);
    }

    argv_to_buf(&argbuf, &argbuflen, argv[1]);
    /* Private keys are only 320 bits long, so check for that to make
     * sure they didn't get the args the wrong way around */
    if (!argbuf || argbuflen > 40) usage(argv[0]);
    gcry_mpi_scan(&our_x, GCRYMPI_FMT_USG, argbuf, argbuflen, NULL);
    free(argbuf);
    argv_to_buf(&argbuf, &argbuflen, argv[2]);
    if (!argbuf) usage(argv[0]);
    gcry_mpi_scan(&their_y, GCRYMPI_FMT_USG, argbuf, argbuflen, NULL);
    free(argbuf);

    sesskeys_gen(sessionid, sendenc, rcvenc, &is_high, &our_y, our_x, their_y);
    sesskeys_make_mac(sendmac, sendenc);
    sesskeys_make_mac(rcvmac, rcvenc);

    /* Print our public key into a buffer */
    gcry_mpi_print(GCRYMPI_FMT_USG, NULL, 0, &publen, our_y);
    pubbuf = malloc(publen);
    if (!pubbuf) {
	fprintf(stderr, "Out of memory!\n");
	exit(1);
    }
    gcry_mpi_print(GCRYMPI_FMT_USG, pubbuf, publen, NULL, our_y);

    puts("");
    printf("We are the %s end of this key exchange.\n",
	    is_high ? "high" : "low");
    puts("");
    dump_data(stdout, "Our public key", pubbuf, publen);
    puts("");
    dump_data(stdout, "Session id", sessionid, 20);
    puts("");
    dump_data(stdout, "Sending   AES key", sendenc, 16);
    dump_data(stdout, "Sending   MAC key", sendmac, 20);
    dump_data(stdout, "Receiving AES key", rcvenc, 16);
    dump_data(stdout, "Receiving MAC key", rcvmac, 20);
    puts("");
    fflush(stdout);

    return 0;
}
