// K-3D
// Copyright (c) 1995-2006, Timothy M. Shead
//
// Contact: tshead@k-3d.com
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/** \file
		\author Tim Shead (tshead@k-3d.com)
		\author Romain Behar (romainbehar@yahoo.com)
*/

#include <gdkmm/cursor.h>
#include <gtkmm/widget.h>

#include "document_state.h"
#include "modifiers.h"
#include "icons.h"
#include "interactive.h"
#include "keyboard.h"
#include "transform_tool.h"
#include "tutorial_message.h"
#include "utility.h"
#include "viewport.h"

#include <k3dsdk/classes.h>
#include <k3dsdk/color.h>
#include <k3dsdk/fstream.h>
#include <k3dsdk/geometry.h>
#include <k3dsdk/gl.h>
#include <k3dsdk/i18n.h>
#include <k3dsdk/icamera.h>
#include <k3dsdk/imesh_sink.h>
#include <k3dsdk/iprojection.h>
#include <k3dsdk/itransform_sink.h>
#include <k3dsdk/mesh.h>
#include <k3dsdk/mesh_source.h>
#include <k3dsdk/property.h>
#include <k3dsdk/state_change_set.h>
#include <k3dsdk/transform.h>
#include <k3dsdk/xml.h>

#include <set>

namespace libk3dngui
{

namespace detail
{

double manipulators_scale(viewport::control& Viewport, const k3d::point3& Origin, const double Size)
{
	k3d::icamera* camera = Viewport.camera();
	return_val_if_fail(camera, 0);

	// Project unit axis on screen space
	const k3d::matrix4 screen_matrix = k3d::node_to_world_matrix(*Viewport.camera());
	const k3d::vector3 screen_parallel = screen_matrix * k3d::vector3(1, 0, 0);
	const k3d::point2 position = Viewport.project(Origin);
	const k3d::point2 x_axis = Viewport.project(Origin + screen_parallel);
	const double length = k3d::distance(position, x_axis);

	return_val_if_fail(length, 0);

	return Size / length;
}

bool is_front_facing(viewport::control& Viewport, const k3d::vector3& Normal, const k3d::point3& Origin, const k3d::matrix4& Orientation)
{
	return_val_if_fail(Viewport.gl_engine(), false);
	return_val_if_fail(Viewport.camera(), false);

	const k3d::matrix4 matrix = k3d::inverse(k3d::node_to_world_matrix(*Viewport.camera()));
	const k3d::point3 a = Origin + (Orientation * Normal);
	const k3d::point3 b = Origin + (Orientation * -Normal);
	return k3d::to_vector(matrix * a).length2() < k3d::to_vector(matrix * b).length2();
}

k3d::point3 get_selected_points(selection_mode_t SelectionMode, const k3d::mesh& Mesh, component_points_t& PointList)
{
	PointList.clear();
	k3d::point3 component_center(0, 0, 0);

	switch(SelectionMode)
	{
		case SELECT_POINTS:
		{
			// Get selected points
			unsigned long index = 0;
			for(k3d::mesh::points_t::const_iterator point = Mesh.points.begin(); point != Mesh.points.end(); ++point)
			{
				if((*point)->selection_weight)
				{
					component_point_t component_point;
					component_point.index = index;
					component_point.initial_position = (*point)->position;
					PointList.push_back(component_point);

					component_center += (*point)->position;
				}

				++index;
			}
		}
		break;
		case SELECT_LINES:
		{
			std::set<k3d::point*> points;

			// Get points belonging to selected lines
			for(k3d::mesh::polyhedra_t::const_iterator polyhedron = Mesh.polyhedra.begin(); polyhedron != Mesh.polyhedra.end(); ++polyhedron)
			{
				for(k3d::polyhedron::faces_t::const_iterator face = (*polyhedron)->faces.begin(); face != (*polyhedron)->faces.end(); ++face)
				{
					k3d::split_edge* edge = (*face)->first_edge;
					do
					{
						if(edge->selection_weight)
						{
							points.insert(edge->vertex);
							points.insert(edge->face_clockwise->vertex);
						}

						edge = edge->face_clockwise;
					}
					while(edge != (*face)->first_edge);
				}
			}

			unsigned long index = 0;
			for(k3d::mesh::points_t::const_iterator point = Mesh.points.begin(); point != Mesh.points.end(); ++point)
			{
				if(points.find(*point) != points.end())
				{
					component_point_t component_point;
					component_point.index = index;
					component_point.initial_position = (*point)->position;
					PointList.push_back(component_point);

					component_center += (*point)->position;
				}

				++index;
			}
		}
		break;
		case SELECT_FACES:
		{
			std::set<k3d::point*> points;

			// Get points belonging to selected faces
			for(k3d::mesh::polyhedra_t::const_iterator polyhedron = Mesh.polyhedra.begin(); polyhedron != Mesh.polyhedra.end(); ++polyhedron)
			{
				for(k3d::polyhedron::faces_t::const_iterator face = (*polyhedron)->faces.begin(); face != (*polyhedron)->faces.end(); ++face)
				{
					if(!(*face)->selection_weight)
						continue;

					k3d::split_edge* edge = (*face)->first_edge;
					do
					{
						points.insert(edge->vertex);

						edge = edge->face_clockwise;
					}
					while(edge != (*face)->first_edge);
				}
			}

			unsigned long index = 0;
			for(k3d::mesh::points_t::const_iterator point = Mesh.points.begin(); point != Mesh.points.end(); ++point)
			{
				if(points.find(*point) != points.end())
				{
					component_point_t component_point;
					component_point.index = index;
					component_point.initial_position = (*point)->position;
					PointList.push_back(component_point);

					component_center += (*point)->position;
				}

				++index;
			}
		}
		break;
		default:
			assert_not_reached();
	}

	// Compute average position
	const double point_number = static_cast<double>(PointList.size());
	if(point_number)
		component_center /= point_number;

	return component_center;
}

} // namespace detail

	k3d::matrix4 transform_tool::itarget::world_orientation()
	{
		if(LOCAL == current_system_type)
			return k3d::extract_rotation(k3d::node_to_world_matrix(*node));

		if(PARENT == current_system_type)
			return k3d::extract_rotation(k3d::parent_to_world_matrix(*node));

		return k3d::identity3D();
	}

	void transform_tool::itarget::set_transform_modifier(k3d::inode* Modifier)
	{
		modifier = Modifier;
		Modifier->deleted_signal().connect(sigc::mem_fun(*this, &itarget::reset_transform_modifier));
	}

	void transform_tool::itarget::reset_transform_modifier()
	{
		modifier = 0;
	}

	void transform_tool::itarget::set_coordinate_system_change_matrices()
	{
		switch(current_system_type)
		{
			case GLOBAL:
				m_system_matrix = k3d::inverse(k3d::node_to_world_matrix(*node));
				m_system_matrix_inverse = k3d::node_to_world_matrix(*node);
				// Zero translation components
				m_system_matrix[0][3] = m_system_matrix[1][3] = m_system_matrix[2][3] = 0;
				m_system_matrix_inverse[0][3] = m_system_matrix_inverse[1][3] = m_system_matrix_inverse[2][3] = 0;
				break;
			case LOCAL:
				m_system_matrix = k3d::identity3D();
				m_system_matrix_inverse = k3d::identity3D();
				break;
			case PARENT:
				m_system_matrix = k3d::inverse(k3d::node_to_world_matrix(*node)) * k3d::parent_to_world_matrix(*node);
				m_system_matrix_inverse = k3d::inverse(m_system_matrix);
				// Zero translation components
				m_system_matrix[0][3] = m_system_matrix[1][3] = m_system_matrix[2][3] = 0;
				m_system_matrix_inverse[0][3] = m_system_matrix_inverse[1][3] = m_system_matrix_inverse[2][3] = 0;
				break;
			default:
				assert_not_reached();
		}
	}

	// transform_target implementation
	transform_tool::transform_target::transform_target(k3d::inode* Node)
	{
		node = Node;

		// Sanity check
		return_if_fail(node);
	}

	k3d::point3 transform_tool::transform_target::world_position()
	{
		return k3d::world_position(*node);
	}

	unsigned long transform_tool::transform_target::target_number()
	{
		// There's always one object to move
		return 1;
	}

	void transform_tool::transform_target::reset()
	{
	}

	void transform_tool::transform_target::start_move()
	{
		if(create_transform_modifier(k3d::classes::FrozenTransformation(), "Move "))
			assert_warning(k3d::set_value(*modifier, "matrix", k3d::identity3D()));

		// Setup matrices
		m_original_matrix = boost::any_cast<k3d::matrix4>(k3d::get_value(*modifier, "matrix"));

		set_coordinate_system_change_matrices();
	}

	void transform_tool::transform_target::move(const k3d::vector3& Move)
	{
		if(!modifier)
			start_move();

		const k3d::matrix4 translation = k3d::translation3D(m_system_matrix * Move);
		assert_warning(k3d::set_value(*modifier, "matrix", m_original_matrix * translation));
	}

	void transform_tool::transform_target::start_rotation()
	{
		if(create_transform_modifier(k3d::classes::FrozenTransformation(), "Rotate "))
			assert_warning(k3d::set_value(*modifier, "matrix", k3d::identity3D()));

		// Setup matrices
		m_original_matrix = boost::any_cast<k3d::matrix4>(k3d::get_value(*modifier, "matrix"));

		set_coordinate_system_change_matrices();
	}

	void transform_tool::transform_target::rotate(const k3d::matrix4& RotationMatrix, const k3d::point3& WorldCenter)
	{
		if(!modifier)
			start_rotation();

		const k3d::matrix4 current_coordinate_system_rotation = m_system_matrix * RotationMatrix * m_system_matrix_inverse;
		assert_warning(k3d::set_value(*modifier, "matrix", m_original_matrix * current_coordinate_system_rotation));
	}

	void transform_tool::transform_target::start_scaling()
	{
		if(create_transform_modifier(k3d::classes::FrozenTransformation(), "Scale "))
			assert_warning(k3d::set_value(*modifier, "matrix", k3d::identity3D()));

		// Setup matrices
		m_original_matrix = boost::any_cast<k3d::matrix4>(k3d::get_value(*modifier, "matrix"));

		set_coordinate_system_change_matrices();
	}

	void transform_tool::transform_target::scale(const k3d::point3& Scaling, const k3d::point3& WorldCenter)
	{
		if(!modifier)
			start_scaling();

		const k3d::matrix4 current_coordinate_system_scaling = m_system_matrix * k3d::scaling3D(Scaling) * m_system_matrix_inverse;
		assert_warning(k3d::set_value(*modifier, "matrix", m_original_matrix * current_coordinate_system_scaling));
	}

	void transform_tool::transform_target::end_drag_motion()
	{
	}

	bool transform_tool::transform_target::create_transform_modifier(const k3d::uuid& Class, const std::string& Name)
	{
		if(modifier)
			return false;

		return_val_if_fail(node, false);

		// Check for an existing transform modifier
		k3d::inode* upstream_node = upstream_transform_modifier(*node);
		/** \todo check for same name too */
		if(upstream_node && (Class == upstream_node->factory().class_id()))
		{
			set_transform_modifier(upstream_node);
			return false;
		}

		const std::string modifier_name = Name + node->name();
		set_transform_modifier(insert_transform_modifier(*node, Class, modifier_name));

		return true;
	}

	// mesh_target implementation
	transform_tool::mesh_target::mesh_target(document_state& DocumentState, k3d::inode* Node, k3d::iproperty& MeshSourceProperty) :
		m_document_state(DocumentState),
		mesh_source_property(MeshSourceProperty),
		component_center(k3d::point3(0, 0, 0)),
		m_mesh_changed(true),
		m_drag_mutex(false)
	{
		node = Node;

		// Sanity checks
		return_if_fail(node);

		// Connect to mesh_source change_signal to be acknowledged of changes
		m_mesh_change_signal = mesh_source_property.property_changed_signal().connect(sigc::mem_fun(*this, &transform_tool::mesh_target::mesh_changed));

		reset_selection();
	}

	k3d::point3 transform_tool::mesh_target::world_position()
	{
		if(!m_drag_mutex && m_mesh_changed)
			reset_selection();

		return k3d::node_to_world_matrix(*node) * component_center;
	}

	unsigned long transform_tool::mesh_target::target_number()
	{
		return selected_points.size();
	}

	void transform_tool::mesh_target::reset_selection()
	{
		if(m_drag_mutex)
			return;

		k3d::mesh* mesh = boost::any_cast<k3d::mesh*>(mesh_source_property.property_value());
		return_if_fail(mesh);

		// Get selection and save initial position
		component_center = detail::get_selected_points(m_document_state.selection_mode().value(), *mesh, selected_points);

		m_mesh_changed = false;
	}

	void transform_tool::mesh_target::reset()
	{
		if(m_drag_mutex)
			return;

		// Reset position
		reset_selection();

		modifier = 0;
	}

	void transform_tool::mesh_target::init_transformation()
	{
		// Save initial positions
		k3d::mesh* mesh = boost::any_cast<k3d::mesh*>(mesh_source_property.property_value());
		return_if_fail(mesh);
		for(detail::component_points_t::iterator point = selected_points.begin(); point != selected_points.end(); ++point)
		{
			const unsigned long index = point->index;
			point->initial_position = mesh->points[index]->position;
			point->tweak_value = tweaks[index];
		}

		set_coordinate_system_change_matrices();
	}

	void transform_tool::mesh_target::start_move()
	{
		create_mesh_modifier("Move ");

		init_transformation();

		m_origin = component_center;
	}

	void transform_tool::mesh_target::move(const k3d::vector3& Move)
	{
		if(!modifier)
			start_move();

		m_drag_mutex = true;

		// Tweak points
		for(detail::component_points_t::const_iterator point = selected_points.begin(); point != selected_points.end(); ++point)
		{
			const k3d::point3 position = point->initial_position;
			const k3d::point3 new_position = m_system_matrix * Move + position;

			tweaks[point->index] = (new_position - position) + point->tweak_value;
		}

		// Update manipulators position
		component_center = m_system_matrix * Move + m_origin;

		update_mesh_modifier();
	}

	void transform_tool::mesh_target::start_rotation()
	{
		create_mesh_modifier("Rotate ");

		init_transformation();
	}

	void transform_tool::mesh_target::rotate(const k3d::matrix4& RotationMatrix, const k3d::point3& WorldCenter)
	{
		if(!modifier)
			start_rotation();

		m_drag_mutex = true;

		const k3d::matrix4 current_coordinate_system_rotation = m_system_matrix * RotationMatrix * m_system_matrix_inverse;
		//const k3d::point3 center = component_center;
		const k3d::point3 center = k3d::inverse(k3d::node_to_world_matrix(*node)) * WorldCenter;

		for(detail::component_points_t::const_iterator point = selected_points.begin(); point != selected_points.end(); ++point)
		{
			const k3d::point3 position = point->initial_position;
			const k3d::point3 new_position = current_coordinate_system_rotation * (position - center) + center;

			tweaks[point->index] = (new_position - position) + point->tweak_value;
		}

		update_mesh_modifier();
	}

	void transform_tool::mesh_target::start_scaling()
	{
		create_mesh_modifier("Scale ");

		init_transformation();
	}

	void transform_tool::mesh_target::scale(const k3d::point3& Scaling, const k3d::point3& WorldCenter)
	{
		if(!modifier)
			start_scaling();

		m_drag_mutex = true;

		const k3d::matrix4 scaling_3d = m_system_matrix * k3d::scaling3D(Scaling) * m_system_matrix_inverse;
		//const k3d::point3 center = component_center;
		const k3d::point3 center = k3d::inverse(k3d::node_to_world_matrix(*node)) * WorldCenter;

		for(detail::component_points_t::const_iterator point = selected_points.begin(); point != selected_points.end(); ++point)
		{
			const k3d::point3 position = point->initial_position;
			const k3d::point3 new_position = scaling_3d * (position - center) + center;

			tweaks[point->index] = (new_position - position) + point->tweak_value;
		}

		update_mesh_modifier();
	}

	void transform_tool::mesh_target::end_drag_motion()
	{
		m_drag_mutex = false;
	}

	void transform_tool::mesh_target::create_mesh_modifier(const std::string& Name)
	{
		//if(modifier)
		//	return;

		return_if_fail(node);

		// Get mesh to tweak
		k3d::mesh* mesh = boost::any_cast<k3d::mesh*>(mesh_source_property.property_value());
		return_if_fail(mesh);

		// Modify with TweakPoints
		const k3d::uuid tweak_points(0xed302b87, 0x49bf4fe6, 0x99064963, 0x17ec12d9);

		// Check for an existing mesh modifier
		k3d::inode* upstream_node = upstream_mesh_modifier(*node);
		if(upstream_node && (tweak_points == upstream_node->factory().class_id()))
		{
			set_transform_modifier(upstream_node);

			// Init tweaks
			tweaks = boost::any_cast<tweaks_t>(k3d::get_value(*modifier, "tweaks"));
			tweaks.resize(mesh->points.size(), k3d::point3(0, 0, 0));

			return;
		}

		// Create a new TweakPoints modifier
		const std::string modifier_name = Name + node->name() + " components";
		set_transform_modifier(insert_mesh_modifier(*node, tweak_points, modifier_name));

		// Initialize new TweakPoints modifier
		tweaks.clear();
		tweaks.resize(mesh->points.size(), k3d::point3(0, 0, 0));
		update_mesh_modifier();
	}

	void transform_tool::mesh_target::update_mesh_modifier()
	{
		assert_warning(k3d::set_value(*modifier, "tweaks", tweaks));
	}

// transform_tool implementation
void transform_tool::lbutton_down(viewport::control& Viewport, const k3d::point2& Coordinates, const k3d::key_modifiers& Modifiers)
{
	// Return if an action is in progress
	if(MOTION_CLICK_DRAG == m_current_motion)
		return;

	assert_warning(MOTION_NONE == m_current_motion);

	// Init action
	m_mouse_down_content = NOTHING;

	// Find what's under the mouse pointer
	k3d::selection::records picked_selectables;
	pick_selectables(picked_selectables, Viewport, Coordinates);

	// Shift modifier starts ADD action
	if(Modifiers.shift())
	{
		lmb_down_add();
		return;
	}

	// Control modifier starts SUBTRACT action
	if(Modifiers.control())
	{
		lmb_down_subtract();
		return;
	}

	// Manipulator selection starts move
	detail::manipulators_t manipulators;
	for(k3d::selection::records::iterator record = picked_selectables.begin(); record != picked_selectables.end(); ++record)
	{
		for(k3d::selection::record::tokens_t::const_iterator token = record->tokens.begin(); token != record->tokens.end(); ++token)
		{
			if(token->type == k3d::selection::USER1)
				manipulators.push_back(manipulator_name(token->id));
		}
	}

	const std::string manipulator = get_manipulator(manipulators);
	if(manipulator != "")
	{
		lmb_down_manipulator(manipulator);
		return;
	}

	// If a node was hit ...
	if(k3d::selection::get_node(m_mouse_down_selection))
	{
		if(m_document_state.is_selected(m_mouse_down_selection))
			lmb_down_selected();
		else
			lmb_down_deselected();

		return;
	}

	lmb_down_nothing();
}

// LMB down actions
void transform_tool::lmb_down_add()
{
	k3d::start_state_change_set(m_document, __PRETTY_FUNCTION__);
	m_tutorial_action = "lmb_down_add";

	m_mouse_down_content = SELECTION_ADD;
}

void transform_tool::lmb_down_subtract()
{
	k3d::start_state_change_set(m_document, __PRETTY_FUNCTION__);
	m_tutorial_action = "lmb_down_subtract";

	m_mouse_down_content = SELECTION_SUBTRACT;
}

void transform_tool::lmb_down_manipulator(const std::string& ManipulatorName)
{
	k3d::start_state_change_set(m_document, __PRETTY_FUNCTION__);
	m_tutorial_action = "lmb_down_manipulator_" + ManipulatorName;

	set_manipulator(ManipulatorName);
	set_motion(MOTION_DRAG);

	m_mouse_down_content = SELECTED_OBJECT;
}

void transform_tool::lmb_down_selected()
{
	k3d::start_state_change_set(m_document, __PRETTY_FUNCTION__);
	m_tutorial_action = "lmb_down_selected";

	m_mouse_down_content = SELECTED_OBJECT;
}

void transform_tool::lmb_down_deselected()
{
	k3d::start_state_change_set(m_document, __PRETTY_FUNCTION__);
	m_tutorial_action = "lmb_down_deselected";

	m_mouse_down_content = DESELECTED_OBJECT;

	// Deselect all
	m_document_state.deselect_all();
	// Select clicked object
	m_document_state.select(m_mouse_down_selection);
}

void transform_tool::lmb_down_nothing()
{
	k3d::start_state_change_set(m_document, __PRETTY_FUNCTION__);
	m_tutorial_action = "lmb_down_nothing";

	m_mouse_down_content = NOTHING;
}

void transform_tool::lbutton_click(const viewport::control& Viewport, const k3d::point2& Coordinates)
{
	if(MOTION_CLICK_DRAG == m_current_motion)
	{
		// Stop click-drag
		lmb_click_stop_motion();
		return;
	}

	switch(m_mouse_down_content)
	{
		case SELECTION_ADD:
			lmb_click_add();
		break;
		case SELECTION_SUBTRACT:
			lmb_click_subtract();
		break;
		case SELECTED_OBJECT:
			lmb_click_start_motion(Coordinates);
		break;
		case DESELECTED_OBJECT:
			lmb_click_replace();
		break;
		case NOTHING:
			lmb_click_deselect_all();
		break;
		default:
			assert_not_reached();
	}
}

// LMB click actions
void transform_tool::lmb_click_add()
{
	m_tutorial_action = "lmb_click_add";

	// Shift key modifier always adds to the selection
	if(k3d::selection::get_node(m_mouse_down_selection))
		m_document_state.select(m_mouse_down_selection);

	k3d::finish_state_change_set(m_document, "Selection add", __PRETTY_FUNCTION__);

	tool_selection::redraw_all();
}

void transform_tool::lmb_click_subtract()
{
	m_tutorial_action = "lmb_click_subtract";

	// Control key modifier always adds to the selection
	if(k3d::selection::get_node(m_mouse_down_selection))
		m_document_state.deselect(m_mouse_down_selection);

	k3d::finish_state_change_set(m_document, "Selection subtract", __PRETTY_FUNCTION__);

	tool_selection::redraw_all();
}

void transform_tool::lmb_click_replace()
{
	m_tutorial_action = "lmb_click_replace";

	// Replace selection
	m_document_state.deselect_all();
	if(k3d::selection::get_node(m_mouse_down_selection))
		m_document_state.select(m_mouse_down_selection);

	k3d::finish_state_change_set(m_document, "Selection replace", __PRETTY_FUNCTION__);

	tool_selection::redraw_all();
}

void transform_tool::lmb_click_start_motion(const k3d::point2& Coordinates)
{
	m_tutorial_action = "lmb_click_start_motion";

	disconnect_navigation_input_model();

	set_motion(MOTION_CLICK_DRAG);
	begin_mouse_move(Coordinates);

	tool_selection::redraw_all();
}

void transform_tool::lmb_click_stop_motion()
{
	m_tutorial_action = "lmb_click_stop_motion";

	const std::string label = complete_mouse_move();
	k3d::finish_state_change_set(m_document, label, __PRETTY_FUNCTION__);

	connect_navigation_input_model();

	tool_selection::redraw_all();

	end_drag_motion();
}

void transform_tool::lmb_click_deselect_all()
{
	m_tutorial_action = "lmb_click_deselect_all";

	// Deselect all
	m_document_state.deselect_all();

	k3d::finish_state_change_set(m_document, "Deselect all", __PRETTY_FUNCTION__);

	tool_selection::redraw_all();
}

void transform_tool::lbutton_start_drag(viewport::control& Viewport, const k3d::point2& Coordinates)
{
	disconnect_navigation_input_model();

	switch(m_mouse_down_content)
	{
		case SELECTED_OBJECT:
		case DESELECTED_OBJECT:
			lmb_start_drag_start_motion(Coordinates);
		break;
		case SELECTION_ADD:
		case SELECTION_SUBTRACT:
		case NOTHING:
			lmb_start_drag_box_select(Viewport, Coordinates);
		break;
		default:
			assert_not_reached();
	}
}

// LMB start drag actions
void transform_tool::lmb_start_drag_start_motion(const k3d::point2& Coordinates)
{
	m_tutorial_action = "lmb_start_drag_start_motion";

	m_off_screen_offset = k3d::point2(0, 0);

	set_motion(MOTION_DRAG);
	begin_mouse_move(Coordinates);
}

void transform_tool::lmb_start_drag_box_select(viewport::control& Viewport, const k3d::point2& Coordinates)
{
	m_tutorial_action = "lmb_start_drag_box_select";

	set_motion(MOTION_BOX_SELECT);
	m_box_selection = k3d::rectangle(Coordinates[0], Coordinates[0], Coordinates[1], Coordinates[1]);
	draw_rubber_band(Viewport);
}

void transform_tool::lmb_drag_box_select(viewport::control& Viewport, const k3d::point2& Coordinates)
{
	m_tutorial_action = "lmb_drag_box_select";

	on_box_select_motion(Viewport, Coordinates);
}

void transform_tool::lbutton_end_drag(viewport::control& Viewport, const k3d::point2& Coordinates)
{
	if(MOTION_DRAG == m_current_motion)
		lmb_end_drag_stop_motion();
	else if(MOTION_BOX_SELECT == m_current_motion)
		lmb_end_drag_box_select(Viewport, Coordinates);

	connect_navigation_input_model();
}

// LMB end drag actions
void transform_tool::lmb_end_drag_stop_motion()
{
	m_tutorial_action = "lmb_end_drag_stop_motion";

	const std::string label = complete_mouse_move();
	k3d::finish_state_change_set(m_document, label, __PRETTY_FUNCTION__);

	tool_selection::redraw_all();

	end_drag_motion();
}

void transform_tool::lmb_end_drag_box_select(viewport::control& Viewport, const k3d::point2& Coordinates)
{
	m_tutorial_action = "lmb_end_drag_box_select";

	draw_rubber_band(Viewport);

	on_box_select_objects(Viewport, Coordinates, m_box_selection);

	// Stop motion
	set_motion(MOTION_NONE);

	k3d::finish_state_change_set(m_document, "Box selection", __PRETTY_FUNCTION__);

	tool_selection::redraw_all();
}

void transform_tool::mbutton_click(viewport::control& Viewport, const k3d::point2& Coordinates, const k3d::key_modifiers& Modifiers)
{
	// Motion mode
	if(MOTION_NONE != m_current_motion)
	{
		mmb_click_next_constraint(Viewport, Coordinates);
		return;
	}

	// We aren't moving
	if(Modifiers.control())
	{
		mmb_click_switch_coordinate_system();
	}
	else if(Modifiers.shift())
	{
		mmb_click_manipulators_next_selection();
	}
	else
	{
		mmb_click_toggle_manipulators_visibility();
	}
}

void transform_tool::mmb_click_toggle_manipulators_visibility()
{
	m_tutorial_action = "mmb_click_toggle_manipulators_visibility";

	// Toggle the visible state of the manipulators
	m_visible_manipulators.set_value(!m_visible_manipulators.value());

	tool_selection::redraw_all();
}

void transform_tool::mmb_click_manipulators_next_selection()
{
	m_tutorial_action = "mmb_click_manipulators_next_selection";

	// Show manipulators on the next selection
	m_current_target = m_targets.size() ? (m_current_target + 1) % m_targets.size() : 0;

	tool_selection::redraw_all();
}

void transform_tool::mmb_click_switch_coordinate_system()
{
	m_tutorial_action = "mmb_click_switch_coordinate_system";

	// Switch coordinate system between global and local modes
	switch(m_coordinate_system.value())
	{
		case GLOBAL:
			set_coordinate_system(LOCAL);
			break;
		case LOCAL:
			set_coordinate_system(GLOBAL);
			break;
		default:
			break;
	}

	tool_selection::redraw_all();
}

void transform_tool::mmb_click_next_constraint(viewport::control& Viewport, const k3d::point2& Coordinates)
{
	m_tutorial_action = "mmb_click_next_constraint";

	// Set next constraint
	update_constraint(Viewport, Coordinates);

/*
	// Mouse warp
	const k3d::point2 coords = Viewport.project(world_position());

	int root_x = 0;
	int root_y = 0;
	Viewport.get_window()->get_origin(root_x, root_y);

	const k3d::point2 screen_coords = k3d::point2(coords[0] + root_x, coords[1] + root_y);

	// We temporarily disable motion, so warping the pointer doesn't cause unintended side effects
	const motion_t current_motion = m_current_motion;
	m_current_motion = MOTION_NONE;
	interactive::warp_pointer(screen_coords);
	handle_pending_events();

	// Acknowledge new mouse position
	m_current_constraint->begin_mouse_move(coords);

	m_current_motion = current_motion;
*/

	tool_selection::redraw_all();
}

void transform_tool::rbutton_click(const viewport::control& Viewport, const k3d::point2& Coordinates)
{
	if(MOTION_NONE == m_current_motion)
		rmb_click_selection_tool();
	else
		rmb_click_cancel_move();
}

// RMB click actions
void transform_tool::rmb_click_selection_tool()
{
	k3d::start_state_change_set(m_document, __PRETTY_FUNCTION__);
	m_tutorial_action = "rmb_click_selection_tool";

	m_document_state.set_active_tool(m_document_state.selection_tool());

	k3d::finish_state_change_set(m_document, "Selection tool", __PRETTY_FUNCTION__);

	tool_selection::redraw_all();
}

void transform_tool::rmb_click_cancel_move()
{
	m_tutorial_action = "rmb_click_cancel_move";

	cancel_mouse_move();

	tool_selection::redraw_all();
}

void transform_tool::cancel_mouse_move()
{
	// Stop motion
	set_motion(MOTION_NONE);

	// Undo changes
	k3d::cancel_state_change_set(m_document, __PRETTY_FUNCTION__);

	end_drag_motion();

	// Reset targets
	for(targets_t::iterator target = m_targets.begin(); target != m_targets.end(); ++target)
		(*target)->reset();

	reset();

	tool_selection::redraw_all();
}

std::string transform_tool::complete_mouse_move()
{
	set_motion(MOTION_NONE);

	return get_constraint_name();
}

void transform_tool::set_motion(const motion_t Motion)
{
	m_current_motion = Motion;
}

/// Returns current target's world position
k3d::point3 transform_tool::world_position()
{
	if(target_number())
	{
		if(SELECT_NODES == m_document_state.selection_mode().value())
		{
			m_current_target = m_current_target % m_targets.size();
			itarget* t = m_targets[m_current_target];
			return t->world_position();
		}
		else
		{
			k3d::point3 position(0, 0, 0);
			unsigned long count = 0;
			for(targets_t::iterator target = m_targets.begin(); target != m_targets.end(); ++target)
			{
				if((*target)->target_number())
				{
					position += (*target)->world_position();
					count++;
				}
			}

			position /= static_cast<double>(count);

			return position;
		}
	}

	return k3d::point3(0, 0, 0);
}

/// Returns current target's world orientation
k3d::matrix4 transform_tool::world_orientation()
{
	if(target_number())
	{
		m_current_target = m_current_target % m_targets.size();
		itarget* t = m_targets[m_current_target];

		return t->world_orientation();
	}

	return k3d::identity3D();
}

void transform_tool::update_manipulators_scale(viewport::control& Viewport, const k3d::point3& Origin)
{
	k3d::icamera* camera = Viewport.camera();
	return_if_fail(camera);

	// Project unit axis on screen space
	const k3d::matrix4 screen_matrix = k3d::node_to_world_matrix(*Viewport.camera());
	const k3d::vector3 screen_parallel = screen_matrix * k3d::vector3(1, 0, 0);
	const k3d::point2 position = Viewport.project(Origin);
	const k3d::point2 x_axis = Viewport.project(Origin + screen_parallel);
	const double length = k3d::distance(position, x_axis);

	return_if_fail(length);
	m_manipulators_scale = m_manipulators_size / length;
}

bool transform_tool::front_facing(viewport::control& Viewport, const k3d::vector3& Normal, const k3d::point3& Origin)
{
	return_val_if_fail(Viewport.gl_engine(), false);
	return_val_if_fail(Viewport.camera(), false);

	const k3d::matrix4 matrix = k3d::inverse(k3d::node_to_world_matrix(*Viewport.camera()));
	const k3d::matrix4 orientation = world_orientation();
	const k3d::point3 a = Origin + (orientation * Normal);
	const k3d::point3 b = Origin + (orientation * -Normal);
	return k3d::to_vector(matrix * a).length2() < k3d::to_vector(matrix * b).length2();
}

void transform_tool::off_screen_warp(viewport::control& Viewport, k3d::point2& NewCoordinates)
{
	// Get mouse coordinates
	int x, y;
	Gdk::ModifierType modifiers;
	Gdk::Display::get_default()->get_pointer(x, y, modifiers);
	k3d::point2 mouse(x, y);

	// Check for screen warp
	bool screen_warp = false;
	const k3d::point2 previous_offset = m_off_screen_offset;

	// Wrap the mouse if it goes off the top-or-bottom of the screen ...
	const int screen_height = Gdk::Display::get_default()->get_default_screen()->get_height();
	if(y == 0)
	{
		mouse = k3d::point2(mouse[0], screen_height - 2);
		screen_warp = true;

		m_off_screen_offset[1] -= screen_height;
	}
	else if(screen_height - 1 == y)
	{
		mouse = k3d::point2(mouse[0], 1);
		screen_warp = true;

		m_off_screen_offset[1] += screen_height;
	}

	// Wrap the mouse if it goes off the left-or-right of the screen ...
	const int screen_width = Gdk::Display::get_default()->get_default_screen()->get_width();
	if(x == 0)
	{
		mouse = k3d::point2(screen_width - 2, mouse[1]);
		screen_warp = true;

		m_off_screen_offset[0] -= screen_width;
	}
	else if(screen_width - 1 == x)
	{
		mouse = k3d::point2(1, mouse[1]);
		screen_warp = true;

		m_off_screen_offset[0] += screen_width;
	}

	if(!screen_warp)
	{
		// No warp
		NewCoordinates += m_off_screen_offset;
	}
	else
	{
		// Warp mouse pointer
		interactive::warp_pointer(mouse);

		// Set new position
		NewCoordinates += previous_offset;
	}
}

void transform_tool::clear_targets()
{
	for(targets_t::iterator target = m_targets.begin(); target != m_targets.end(); ++target)
	{
		delete *target;
	}

	m_targets.clear();
}

/// Retrieves the current document selection, refreshing the target list
void transform_tool::get_current_selection()
{
	// Convert the current document selection into the set of targets to be moved interactively
	clear_targets();

	const k3d::nodes_t nodes = m_document_state.selected_nodes();

	if(SELECT_NODES == m_document_state.selection_mode().value())
	{
		// Save transformable nodes as targets
		for(k3d::nodes_t::const_iterator node = nodes.begin(); node != nodes.end(); ++node)
		{
			if(!dynamic_cast<k3d::gl::idrawable*>(*node))
				continue;
			if(!dynamic_cast<k3d::itransform_sink*>(*node))
				continue;

			m_targets.push_back(new transform_target(*node));
			(*node)->deleted_signal().connect(sigc::mem_fun(*this, &transform_tool::target_list_changed));
		}
	}
	else
	{
		// Component mode : save mesh nodes as targets
		for(k3d::nodes_t::const_iterator node = nodes.begin(); node != nodes.end(); ++node)
		{
			if(!dynamic_cast<k3d::gl::idrawable*>(*node))
				continue;

			// Get node's mesh
			k3d::imesh_source* const mesh_source = dynamic_cast<k3d::imesh_source*>(*node);
			if(!mesh_source)
				continue;

			// Get non-transformed mesh output
			k3d::iproperty& property = mesh_source->mesh_source_output();
			m_targets.push_back(new mesh_target(m_document_state, *node, property));

			(*node)->deleted_signal().connect(sigc::mem_fun(*this, &transform_tool::target_list_changed));
		}
	}

	update_coordinate_system(0);
}

unsigned long transform_tool::target_number()
{
	unsigned long target_count = 0;
	for(targets_t::iterator target = m_targets.begin(); target != m_targets.end(); ++target)
		target_count += (*target)->target_number();

	return target_count;
}

void transform_tool::update_targets()
{
	// Update selection on new or deleted node(s)
	if(m_node_selection_changed)
	{
		get_current_selection();
		m_node_selection_changed = false;
	}
}

void transform_tool::start_move()
{
	for(targets_t::iterator target = m_targets.begin(); target != m_targets.end(); ++target)
		(*target)->start_move();
}

void transform_tool::move_targets(const k3d::vector3& Move)
{
	for(targets_t::iterator target = m_targets.begin(); target != m_targets.end(); ++target)
		(*target)->move(Move);

	tool_selection::redraw_all();
}

void transform_tool::start_rotation()
{
	for(targets_t::iterator target = m_targets.begin(); target != m_targets.end(); ++target)
		(*target)->start_rotation();
}

void transform_tool::rotate_targets(const k3d::matrix4& Rotation)
{
	if(!m_targets.size())
		return;

	// Compute average of all selected components
	k3d::point3 world_center(0, 0, 0);
	for(targets_t::iterator target = m_targets.begin(); target != m_targets.end(); ++target)
		world_center += (*target)->world_position();

	world_center /= static_cast<double>(m_targets.size());

	for(targets_t::iterator target = m_targets.begin(); target != m_targets.end(); ++target)
		(*target)->rotate(Rotation, world_center);

	tool_selection::redraw_all();
}

void transform_tool::start_scaling()
{
	for(targets_t::iterator target = m_targets.begin(); target != m_targets.end(); ++target)
		(*target)->start_scaling();
}

void transform_tool::scale_targets(const k3d::point3& Scaling)
{
	if(!m_targets.size())
		return;

	// Compute average of all selected components
	k3d::point3 world_center(0, 0, 0);
	for(targets_t::iterator target = m_targets.begin(); target != m_targets.end(); ++target)
		world_center += (*target)->world_position();

	world_center /= static_cast<double>(m_targets.size());

	for(targets_t::iterator target = m_targets.begin(); target != m_targets.end(); ++target)
		(*target)->scale(Scaling, world_center);

	tool_selection::redraw_all();
}

void transform_tool::end_drag_motion()
{
	for(targets_t::iterator target = m_targets.begin(); target != m_targets.end(); ++target)
		(*target)->end_drag_motion();
}

void transform_tool::set_coordinate_system(const coordinate_system_t CoordinateSystem)
{
	m_coordinate_system.set_value(CoordinateSystem);
}

/// Updates all targets : tells them what's the new coordinate system
void transform_tool::update_coordinate_system(k3d::iunknown*)
{
	switch(m_coordinate_system.value())
	{
		case GLOBAL:
			for(targets_t::iterator target = m_targets.begin(); target != m_targets.end(); ++target)
				(*target)->set_global();
			break;
		case LOCAL:
			for(targets_t::iterator target = m_targets.begin(); target != m_targets.end(); ++target)
				(*target)->set_local();
			break;
		case PARENT:
			for(targets_t::iterator target = m_targets.begin(); target != m_targets.end(); ++target)
				(*target)->set_parent();
			break;
		default:
			assert_not_reached();
	}

	tool_selection::redraw_all();
}

} // namespace libk3dngui

