#ifndef K3DSDK_IRENDER_FARM_H
#define K3DSDK_IRENDER_FARM_H

// K-3D
// Copyright (c) 1995-2004, Timothy M. Shead
//
// Contact: tshead@k-3d.com
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/** \file
		\brief Declares irender_farm, irender_job, and irender_frame, abstract interfaces encapsulating a system for rendering frames
		\author Tim Shead (tshead@k-3d.com)
*/

#include "iunknown.h"

#include <string>

namespace k3d
{

namespace filesystem { class path; }

/// Abstract interface encapsulating a single frame to be rendered
class irender_frame :
	public virtual iunknown
{
public:
	/// Returns a unique filepath that can be used as an input file for this frame
	virtual const filesystem::path add_input_file(const std::string& Name) = 0;
	/// Returns a unique filepath that can be used as an output file for this frame
	virtual const filesystem::path add_output_file(const std::string& Name) = 0;
	/// Sets-up a render operation for transforming input files to outputfiles (the render engine will be matched against choices in the user options file)
	virtual void add_render_operation(const std::string& Type, const std::string& Name, const filesystem::path& File, const bool VisibleRender) = 0;
	/// Sets-up a copy operation that will be executed once rendering is complete
	virtual void add_copy_operation(const filesystem::path& SourceFile, const filesystem::path& TargetFile) = 0;
	/// Sets-up a file view operation once copying is complete
	virtual void add_view_operation(const filesystem::path& FilePath) = 0;

protected:
	irender_frame() {}
	irender_frame(const irender_frame& RHS) {}
	irender_frame& operator=(const irender_frame& RHS) { return *this; }
	virtual ~irender_frame() {}
};

/// Abstract interface encapsulating zero-or-more frames to be rendered
class irender_job :
	public virtual iunknown
{
public:
	/// Adds a new "frame" to the job, to be rendered when the job is run
	virtual irender_frame& create_frame(const std::string FrameName) = 0;

protected:
	irender_job() {}
	irender_job(const irender_job& RHS) {}
	irender_job& operator=(const irender_job& RHS) { return *this; }
	virtual ~irender_job() {}
};

/// Abstract interface that encapsulates a "render farm" capable of rendering jobs
class irender_farm :
	public virtual iunknown
{
public:
	/// Creates a new job, in a "waiting" state
	virtual irender_job& create_job(const std::string JobName) = 0;
	/// Tells the render farm to begin executing the given job
	virtual void start_job(irender_job& Job) = 0;

protected:
	irender_farm() {}
	irender_farm(const irender_farm& RHS) {}
	irender_farm& operator=(const irender_farm& RHS) { return *this; }
	virtual ~irender_farm() {}
};

} // namespace k3d

#endif // K3DSDK_IRENDER_FARM_H


