/***************************************************************************
 *   Copyright (C) 2008 by Konstantinos Smanis                             *
 *   kon.smanis@gmail.com                                                  *
 *                                                                         *
 *   This file is part of KGRUBEditor.                                     *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin Street, Fifth Floor                                       *
 *   Boston, MA 02110-1301, USA.                                          *
 ***************************************************************************/

#ifndef GRUB_DEVICE_EDITOR_H
#define GRUB_DEVICE_EDITOR_H

//KDE
#include <khbox.h>
class KLineEdit;
class KMenu;
class KPushButton;

/**
 * @short An editor for GRUB devices/partitions.
 *
 * A KLineEdit accompanied by a KPushButton which contains
 * device/partition suggestions(=all the devices/partitions
 * detected which are shown in the menu).
 *
 * The button has a menu which depending on the current mode
 * can contain the following:
 * @li device suggestions
 * @li partition suggestions
 * @li both the above (submenus are used)
 *
 * By default nothing is shown (Modes::None)
 *
 * When the user selects a suggestion, the previous lineedit contents
 * get erased and a GRUB-compatible device name of the suggestion
 * fills the lineedit.
 */
class GRUBDeviceEditor : public KHBox
{
Q_OBJECT
Q_FLAGS( Modes )
public:
	/**
	* Modes of operation for the editor.
	* @li @param None - Show nothing.
	* @li @param Device - Show only devices (hard disks).
	* @li @param Partition - Show only partitions.
	* @li @param All - Show both devices and partitions.
	*/
	enum Mode
	{
		None = 0,
		Device = 1,
		Partition = 2,
		All = Device | Partition
	};
	Q_DECLARE_FLAGS( Modes, Mode )

	/**
	* Default constructor.
	*
	* @param parent QWidget parent. Self explanatory.
	*/
	GRUBDeviceEditor( QWidget *parent = 0 );

	/**
	* Returns the current mode.
	* @see Modes
	*/
	Modes mode() const;
	/**
	* Sets the mode of the device editor.
	* @see Modes
	*/
	void setMode( Modes mode );

	/**
	* @returns a pointer to the lineedit.
	*/
	KLineEdit* lineEdit() const;
	/**
	* @returns a pointer to the pushbutton.
	*/
	KPushButton* button() const;
	/**
	* @returns a pointer to the menu.
	*/
	KMenu* menu() const;
private:
	/**
	* Checks if a device already exists in the @param menu, so 
	* that no duplicates are entered.
	*
	* @param menu The menu where the check will occur.
	* @param device The device to check.
	*
	* @return whether the device already exists in the menu or not.
	*/
	bool deviceAlreadyExists( const KMenu *menu, const QString &device ) const;
private slots:
	/**
	* A suggestion was chosen from the menu list.
	*
	* Fills-in the KLineEdit with the appropriate root device/partition
	* (using standard GRUB device naming).
	*/
	void suggestionTriggered( QAction *action );
private:
	KLineEdit *m_edit;
	KPushButton *m_button;
	KMenu *m_menu;
	Modes m_mode;
};

Q_DECLARE_OPERATORS_FOR_FLAGS( GRUBDeviceEditor::Modes )

#endif
