/***************************************************************************
 *   Copyright (C) 2008 by Konstantinos Smanis                             *
 *   kon.smanis@gmail.com                                                  *
 *                                                                         *
 *   This file is part of KGRUBEditor.                                     *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin Street, Fifth Floor                                       *
 *   Boston, MA 02110-1301, USA.                                          *
 ***************************************************************************/

//Own
#include "password.h"

//KDE
#include <kmessagebox.h>
#include <kprocess.h>
#include <kstandarddirs.h>

//KGRUBEditor
#include "core/path.h"

PasswordAssistant::PasswordAssistant( GRUB::ComplexCommand::Password *password, QWidget *parent ) : KAssistantDialog( parent )
{
//SETUP UI
	setWindowIcon( KIcon( "dialog-password" ) );
	setCaption( i18nc( "@title:window", "Password Assistant" ) );
	setAttribute( Qt::WA_DeleteOnClose );
	showButton( KDialog::Help, false );
	setInitialSize( QSize( 600, 400 ) );

	QWidget *intro = new QWidget( this );
	ui_intro.setupUi( intro );
	addPage( intro, i18nc( "@title", "Introduction" ) );
	QWidget *step1 = new QWidget( this );
	ui_step1.setupUi( step1 );
	passwordPage = addPage( step1, i18nc( "@title", "Step 1/3 - Password" ) );
	QWidget *step2 = new QWidget( this );
	ui_step2.setupUi( step2 );
	addPage( step2, i18nc( "@title", "Step 2/3 - MD5 Encryption" ) );
	QWidget *step3 = new QWidget( this );
	ui_step3.setupUi( step3 );
	addPage( step3, i18nc( "@title", "Step 3/3 - New Configuration File" ) );
	QWidget *summary = new QWidget( this );
	ui_summary.setupUi( summary );
	summaryPage = addPage( summary, i18nc( "@title", "Summary" ) );
//ATTRIBUTE SETUP
	m_password = password;
//SHOW PASSWORD
	if ( !password->isEmpty() )
	{
		ui_step1.klineedit_password->setText( password->password() );
		ui_step2.checkBox_encrypt->setChecked( password->md5crypted() );
		ui_step3.gruburlrequester_config->setPath( password->configFile() );
	}
//MISC UI SETUP
	setPasswordPageValid( ui_step1.klineedit_password->text() );

	ui_step2.kpushbutton_encrypt->setIcon( KIcon( "document-encrypt" ) );
//SETUP CONNECTIONS
	setupConnections();
}
PasswordAssistant::~PasswordAssistant()
{
	m_password = 0;
}

void PasswordAssistant::setupConnections()
{
	connect( ui_step1.klineedit_password, SIGNAL( textChanged( const QString & ) ), SLOT( setPasswordPageValid( const QString & ) ) );
	connect( ui_step2.kpushbutton_encrypt, SIGNAL( pressed() ), SLOT( encryptPassword() ) );
	connect( this, SIGNAL( user2Clicked() ), SLOT( nextClicked() ) );
	connect( this, SIGNAL( user1Clicked() ), SLOT( finishClicked() ) );
}

void PasswordAssistant::setPasswordPageValid( const QString &text )
{
	setValid( passwordPage, !QString( text ).trimmed().isEmpty() );
}
void PasswordAssistant::encryptPassword()
{
	if ( ui_step1.klineedit_password->text().trimmed().isEmpty() )
	{
		KMessageBox::information( this, i18nc( "@info", "Empty password is not permitted." ) );
		return;
	}

	QString exe = KStandardDirs::findExe( encryptCommand );
	if ( exe.isEmpty() )
	{
		exe = KStandardDirs::findExe( encryptCommand, "/usr/sbin" );
		if ( exe.isEmpty() )
		{
			KMessageBox::sorry( this, i18nc( "@info", "<para>Unable to locate the <command>%1</command> executable.</para><para>Check that this executable actually exists, as well as your PATH enviroment variable.</para>", encryptCommand ) );
			return;
		}
	}

	KProcess md5;
	md5.setProgram( exe );
	md5.setOutputChannelMode( KProcess::MergedChannels );
	md5.start();
	md5.waitForStarted();
	md5.write( QByteArray().append( ui_step1.klineedit_password->text() + QString( "\n" ) ) );
	md5.write( QByteArray().append( ui_step1.klineedit_password->text() + QString( "\n" ) ) );
	md5.closeWriteChannel();
	md5.waitForFinished();

	switch( md5.exitCode() )
	{
		case -2:
			KMessageBox::error( this, i18nc( "@info", "<command>%1</command> could not be started and the operation was aborted.", encryptCommand ) );
			return;
		break;
		case -1:
			KMessageBox::error( this, i18nc( "@info", "<command>%1</command> crashed and the operation was aborted.", encryptCommand ) );
			return;
		break;
		case 0:
			KMessageBox::information( this, i18nc( "@info", "<para>The password you entered will now be encrypted into an inconceivable line.</para><para>When prompted by GRUB to enter the password, all you have to enter is the initial password, not this line.</para>" ), QString(), i18nc( "@option:check", "Don't remind me of this again." ) );
		break;
		default:
			KMessageBox::sorry( this, i18nc( "@info", "<command>%1</command> returned unknown exit code %2. An error could have occured.", encryptCommand, md5.exitCode() ) );
		break;
	}

	ui_step1.klineedit_password->setText( QString( md5.readAllStandardOutput().trimmed().split( '\n' ).last() ).trimmed() );
	ui_step2.checkBox_encrypt->setChecked( true );
}
void PasswordAssistant::nextClicked()
{
	if ( currentPage() == summaryPage )
	{
		ui_summary.label_password->setText( ui_step1.klineedit_password->text() );
		ui_summary.label_md5->setText( ui_step2.checkBox_encrypt->isChecked() ? i18nc( "@info:status", "Enabled" ) : i18nc( "@info:status", "Disabled" ) );
		ui_summary.label_config->setText( ui_step3.gruburlrequester_config->url().path().isEmpty() ? i18nc( "@info:status", "<none>" ) : ui_step3.gruburlrequester_config->url().path() );
	}
}
void PasswordAssistant::finishClicked()
{
	m_password->setPassword( ui_step1.klineedit_password->text() );
	m_password->setMd5crypted( ui_step2.checkBox_encrypt->isChecked() );
	m_password->setConfigFile( ui_step3.gruburlrequester_config->url().path() );
}

#include "password.moc"
