/***************************************************************************
 *   Copyright (C) 2008 by Konstantinos Smanis                             *
 *   kon.smanis@gmail.com                                                  *
 *                                                                         *
 *   This file is part of KGRUBEditor.                                     *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin Street, Fifth Floor                                       *
 *   Boston, MA 02110-1301, USA.                                          *
 ***************************************************************************/

//Own
#include "splashdelegate.h"

//Qt
#include <qdir.h>
#include <qfileinfo.h>
#include <qpainter.h>
#include <qpen.h>

//KDE
#include <kglobalsettings.h>
#include <klocale.h>
#include <threadweaver/JobCollection.h>
#include <threadweaver/ThreadWeaver.h>

//KGRUBEditor
#include "core/root.h"
#include "core/splashpreviewjob.h"
#include "settings.h"

SplashModel::SplashModel( QObject *parent ) : QAbstractListModel( parent )
{
}
QVariant SplashModel::data( const QModelIndex &index, int role ) const
{
	if ( !index.isValid() || index.row() >= m_splash.size() )
		return QVariant();

	QMap<QString, QPixmap>::const_iterator it = m_splash.constBegin();
	for ( int i = 0; i < index.row(); i++ )
		it++;

	switch ( role )
	{
		case Qt::DisplayRole:
			return it.key();
		case Qt::UserRole:
			return it.value();
		default:
			return QVariant();
	}
}
int SplashModel::indexOf( const QString &path ) const
{
	QMapIterator<QString, QPixmap> it( m_splash );
	int i = -1;

	while ( it.hasNext() )
	{
		i++;
		if ( it.next().key() == path )
			return i;
	}
	return -1;
}
int SplashModel::rowCount( const QModelIndex &parent ) const
{
	Q_UNUSED( parent )
	return m_splash.size();
}

void SplashModel::reload()
{
	if ( !ThreadWeaver::Weaver::instance()->isIdle() )
		return;

	kDebug() << "Creating splash previews";

	QPixmap pixmap( 64, 48 );
	pixmap.fill( Qt::black );
	m_splash[ i18n( "No Splash Image" ) ] = pixmap;

	ThreadWeaver::JobCollection *jobCollection = new ThreadWeaver::JobCollection( this );
	foreach( const QString &splashDir, Settings::splashDirs() )
	{
		Core::Root::makeDirReadableByUserGroup( splashDir, 0 );

		foreach( const QString &splashImage, QDir( splashDir ).entryList( QStringList() << "*.xpm.gz", QDir::Files ) )
		{
			SplashPreviewJob *splashPreviewJob = new SplashPreviewJob( splashDir + "/" + splashImage, this );
			connect( splashPreviewJob, SIGNAL( done( ThreadWeaver::Job * ) ), SLOT( appendSplashImagePreview( ThreadWeaver::Job * ) ) );
			jobCollection->addJob( splashPreviewJob );
		}
	}
	connect( jobCollection, SIGNAL( done( ThreadWeaver::Job * ) ), SLOT( splashImageThumbPreviewsDone( ThreadWeaver::Job * ) ) );
	ThreadWeaver::Weaver::instance()->enqueue( jobCollection );

	kDebug() << "Finished creating splash previews";
}
void SplashModel::appendSplashImagePreview( ThreadWeaver::Job *job )
{
	SplashPreviewJob *splashJob = dynamic_cast<SplashPreviewJob *>( job );
	if ( splashJob && !m_splash.contains( splashJob->absoluteFilePath() ) )
	{
		beginInsertRows( QModelIndex(), 0, 0 );
		m_splash[ splashJob->absoluteFilePath() ] = QPixmap::fromImage( splashJob->splashImageThumb() );
		endInsertRows();
	}
}
void SplashModel::removeSplashImagePreview( const QString &path )
{
	beginRemoveRows( QModelIndex(), 0, 0 );
	m_splash.remove( path );
	endRemoveRows();
}
void SplashModel::splashImageThumbPreviewsDone( ThreadWeaver::Job *job )
{
	Q_UNUSED( job )
	reset();
}

SplashDelegate::SplashDelegate( QObject *parent ) : QAbstractItemDelegate( parent )
{
}

void SplashDelegate::paint( QPainter *painter, const QStyleOptionViewItem &option, const QModelIndex &index ) const
{
	const QString filePath = index.model()->data( index, Qt::DisplayRole ).toString();
	const QString fileName = QFileInfo( filePath ).baseName();
	const QString fileParentDir = QFileInfo( filePath ).dir().path();
	bool fileExists = QFileInfo( filePath ).exists();
	QPixmap pixmap = index.model()->data( index, Qt::UserRole ).value<QPixmap>();

	// draw selection
	if ( option.state & QStyle::State_Selected )
	{
		QPen oldPen = painter->pen();
		QBrush oldBrush = painter->brush();

		painter->setPen( option.palette.color( QPalette::Highlight ) );
		painter->setBrush( option.palette.brush( QPalette::Highlight ) );
		painter->drawRect( option.rect.adjusted( 1, 1, -1, -1 ) );

		painter->setPen( oldPen );
		painter->setBrush( oldBrush );
	}
	// draw pixmap
	if ( !pixmap.isNull() )
		painter->drawPixmap( QRect( option.rect.x() + 2, option.rect.y() + 2, 64, 48 ), pixmap );
	// draw text
	painter->save();

	QFont fileNameFont = painter->font();
	fileNameFont.setWeight( QFont::Bold );
	painter->setFont( fileNameFont );

	int textRectX = option.rect.x() + 2 + 64 + 2;
	int textRectY = option.rect.y() + 2;
	QRect textRect( textRectX, textRectY, option.rect.width() - textRectX, 48 );

	QString text = fileName;
	if ( fileExists )
		text += '\n' + fileParentDir;

	QRect boundingRect = painter->boundingRect( textRect, Qt::AlignVCenter | Qt::TextWordWrap, text );
	painter->drawText( boundingRect, Qt::TextWordWrap, fileName );

	if ( fileExists )
	{
		QRect fileNameRect = painter->boundingRect( boundingRect, Qt::TextWordWrap, fileName );
		QRect filePathRect( fileNameRect.bottomLeft(), textRect.size() );
		painter->setFont( KGlobalSettings::smallestReadableFont() );
		painter->setPen( KGlobalSettings::inactiveTextColor() );
		painter->drawText( filePathRect, Qt::TextWordWrap, fileParentDir );
	}

	painter->restore();
}
QSize SplashDelegate::sizeHint( const QStyleOptionViewItem &option, const QModelIndex &index ) const
{
	Q_UNUSED( option )
	Q_UNUSED( index )
	return QSize( 100, 52 );
}

#include "splashdelegate.moc"
