/* -*- Mode: C; tab-width: 8; indent-tabs-mode: nil; c-basic-offset: 8 -*-
 *
 * Copyright (C) 2008 David Zeuthen <davidz@redhat.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 */

#ifdef HAVE_CONFIG_H
#  include "config.h"
#endif

#include <stdlib.h>
#include <string.h>

#include <glib.h>
#include <glib-object.h>
#include <dbus/dbus-glib.h>
#include <dbus/dbus-glib-lowlevel.h>

#include "devkit-client.h"
#include "devkit-device.h"
#include "devkit-client-marshal.h"
#include "devkit-client-glue.h"

/**
 * SECTION:devkit-client
 * @short_description: Accessing the DeviceKit daemon
 * @include: devkit-gobject/devkit-gobject.h
 *
 * This class provides an interface to the DeviceKit daemon for device
 * enumeration and changes. To get informed about changes, connect to
 * the #DevkitClient::device-event signal.
 **/

struct _DevkitClientPrivate
{
        DBusGConnection *bus;
        DBusGProxy *proxy;
        DBusGProxy *prop_proxy;
        char **subsystems;
        char *inhibit_cookie;
        char *daemon_version;
};

enum {
        PROP_0,
        PROP_SUBSYSTEMS,
        PROP_DAEMON_VERSION,
};

enum {
        DEVICE_EVENT_SIGNAL,
        LAST_SIGNAL,
};

static guint signals[LAST_SIGNAL] = { 0 };

G_DEFINE_TYPE (DevkitClient, devkit_client, G_TYPE_OBJECT)

DevkitDevice *
_devkit_device_new (const char   *subsystem,
                    const char   *native_path,
                    const char   *device_file,
                    const char  **device_file_symlinks,
                    GHashTable   *properties);

/* ---------------------------------------------------------------------------------------------------- */

static void
device_event_signal_handler (DBusGProxy   *proxy,
                             const gchar  *action,
                             const gchar  *subsystem,
                             const gchar  *native_path,
                             const gchar  *device_file,
                             const gchar **device_file_symlinks,
                             GHashTable   *properties,
                             gpointer      user_data)
{
        DevkitClient *client = DEVKIT_CLIENT (user_data);
        DevkitDevice *device;
        gint n;

        /* see devkit_client_connect() for why we match on subsystem here (instead of match rules) for now */
        if (client->priv->subsystems != NULL) {
                for (n = 0; client->priv->subsystems[n] != NULL; n++) {
                        if (strcmp (subsystem, client->priv->subsystems[n]) == 0)
                                break;
                }
                if (client->priv->subsystems[n] == NULL)
                        goto out;
        }

        device = _devkit_device_new (subsystem, native_path, device_file, device_file_symlinks, properties);
        g_signal_emit (client, signals[DEVICE_EVENT_SIGNAL], 0, action, device);
        g_object_unref (device);
out:
        ;
}

static void
devkit_client_finalize (GObject *object)
{
        DevkitClient *client = DEVKIT_CLIENT (object);

        if (client->priv->inhibit_cookie != NULL) {
                GError *error;

                error = NULL;
                if (!org_freedesktop_DeviceKit_uninhibit_shutdown (client->priv->proxy,
                                                                   client->priv->inhibit_cookie,
                                                                   &error)) {
                        g_warning ("Error unhibiting DeviceKit daemon from shutting down: %s", error->message);
                        g_error_free (error);
                }
                g_free (client->priv->inhibit_cookie);
        }

        g_strfreev (client->priv->subsystems);
        if (client->priv->proxy != NULL)
                g_object_unref (client->priv->proxy);
        if (client->priv->prop_proxy != NULL)
                g_object_unref (client->priv->prop_proxy);
        if (client->priv->bus != NULL)
                dbus_g_connection_unref (client->priv->bus);

        if (client->priv->daemon_version != NULL)
                g_free (client->priv->daemon_version);

        if (G_OBJECT_CLASS (devkit_client_parent_class)->finalize)
                (* G_OBJECT_CLASS (devkit_client_parent_class)->finalize) (object);
}

static void
devkit_client_set_property (GObject      *object,
                            guint         prop_id,
                            const GValue *value,
                            GParamSpec   *pspec)
{
        DevkitClient *client = DEVKIT_CLIENT (object);

        switch (prop_id) {
        case PROP_SUBSYSTEMS:
                if (client->priv->subsystems != NULL)
                        g_strfreev (client->priv->subsystems);
                client->priv->subsystems = g_strdupv (g_value_get_boxed (value));
                break;

        default:
                G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
                break;
        }
}

static void
devkit_client_get_property (GObject     *object,
                            guint        prop_id,
                            GValue      *value,
                            GParamSpec  *pspec)
{
        DevkitClient *client = DEVKIT_CLIENT (object);

        switch (prop_id) {
        case PROP_SUBSYSTEMS:
                g_value_set_boxed (value, client->priv->subsystems);
                break;

        case PROP_DAEMON_VERSION:
                g_value_set_string (value, devkit_client_get_daemon_version (client));
                break;

        default:
                G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
                break;
    }
}

/**
 * devkit_client_get_daemon_version:
 * @client: a #DevkitClient
 *
 * Gets the version of the running daemon.
 *
 * Returns: The version of the running daemon.
 **/
const gchar *
devkit_client_get_daemon_version (DevkitClient *client)
{
        GError *error;
        GHashTable *props;
        GValue *value;

        props = NULL;

        if (client->priv->daemon_version != NULL)
                goto out;

        error = NULL;
        if (!dbus_g_proxy_call (client->priv->prop_proxy,
                                "GetAll",
                                &error,
                                G_TYPE_STRING,
                                "org.freedesktop.DeviceKit",
                                G_TYPE_INVALID,
                                dbus_g_type_get_map ("GHashTable", G_TYPE_STRING, G_TYPE_VALUE),
                                &props,
                                G_TYPE_INVALID)) {
                g_warning ("Error invokving GetAll() to get properties: %s", error->message);
                g_error_free (error);
                goto out;
        }


        value = g_hash_table_lookup (props, "daemon-version");
        if (value == NULL) {
                g_warning ("No 'daemon-version' property");
                goto out;
        }

        client->priv->daemon_version = g_strdup (g_value_get_string (value));

 out:
        if (props != NULL)
                g_hash_table_unref (props);
        return client->priv->daemon_version;
}

static void
devkit_client_class_init (DevkitClientClass *klass)
{
        GObjectClass *gobject_class = (GObjectClass *) klass;

        gobject_class->set_property = devkit_client_set_property;
        gobject_class->get_property = devkit_client_get_property;
        gobject_class->finalize     = devkit_client_finalize;

        /**
         * DevkitClient:subsystems:
         *
         * The subsystems to listen for events on from the DeviceKit
         * daemon and generate #DevkitClient::device-event signals
         * from. If %NULL, events from all subsystems will be
         * reported.
         */
        g_object_class_install_property (gobject_class,
                                         PROP_SUBSYSTEMS,
                                         g_param_spec_boxed ("subsystems",
                                                             "The subsystems that we listen for changes on",
                                                             "The subsystems that we listen for changes on",
                                                             G_TYPE_STRV,
                                                             G_PARAM_CONSTRUCT_ONLY |
                                                             G_PARAM_READWRITE));

        /**
         * DevkitClient:daemon-version:
         *
         * The version of the running DeviceKit daemon.
         */
        g_object_class_install_property (gobject_class,
                                         PROP_DAEMON_VERSION,
                                         g_param_spec_string ("daemon-version",
                                                              "The version of the running daemon",
                                                              "The version of the running daemon",
                                                              NULL,
                                                              G_PARAM_READABLE));

        /**
         * DevkitClient::device-event:
         * @client: a #DevkitClient.
         * @action: an OS-specific string (e.g. "add", "remove", "change") describing the event
         * @device: details about the #DevkitDevice the event is happening on.
         *
         * Emitted when an event is happening on a device.
         **/
        signals[DEVICE_EVENT_SIGNAL] =
                g_signal_new ("device-event",
                              G_TYPE_FROM_CLASS (klass),
                              G_SIGNAL_RUN_LAST,
                              G_STRUCT_OFFSET (DevkitClientClass, device_event),
                              NULL, NULL,
                              devkit_client_marshal_VOID__STRING_OBJECT,
                              G_TYPE_NONE,
                              2,
                              G_TYPE_STRING,
                              DEVKIT_TYPE_DEVICE);


        g_type_class_add_private (klass, sizeof (DevkitClientPrivate));
}

static void
devkit_client_init (DevkitClient *client)
{
        client->priv = G_TYPE_INSTANCE_GET_PRIVATE (client,
                                                    DEVKIT_TYPE_CLIENT,
                                                    DevkitClientPrivate);
}

/**
 * devkit_client_new:
 * @subsystems: Either %NULL to listen on events on all subsystems or
 * a %NULL terminated string array of of subsystems to listen for
 * events on.
 *
 * Constructs a #DevkitClient object that listen for device events for
 * devices belonging to @subsystems and emits signals. You need to call
 * devkit_client_connect() before it can be used.
 *
 * Returns: A new #DevkitClient object.
 **/
DevkitClient *
devkit_client_new (const char **subsystems)
{
        return DEVKIT_CLIENT (g_object_new (DEVKIT_TYPE_CLIENT, "subsystems", subsystems, NULL));
}

/**
 * devkit_client_connect:
 * @client: A #DevkitClient object.
 * @error: Return location for error.
 *
 * Connects to the DeviceKit daemon and starts listening for
 * events. The DeviceKit daemon will be inhibited from shutting down
 * until @client is disposed.
 *
 * Returns: %TRUE if the connection was made, %FALSE if @error is set.
 **/
gboolean
devkit_client_connect (DevkitClient  *client,
                       GError       **error)
{
        gboolean ret;

        g_return_val_if_fail (DEVKIT_IS_CLIENT (client), FALSE);
        g_return_val_if_fail (client->priv->inhibit_cookie == NULL, FALSE);

        ret = FALSE;

        client->priv->bus = dbus_g_bus_get (DBUS_BUS_SYSTEM, error);
        if (client->priv->bus == NULL)
                goto out;

        dbus_g_object_register_marshaller (
                devkit_client_marshal_VOID__STRING_STRING_STRING_STRING_BOXED_BOXED,
                G_TYPE_NONE,
                G_TYPE_STRING,
                G_TYPE_STRING,
                G_TYPE_STRING,
                G_TYPE_STRING,
                G_TYPE_STRV,
                dbus_g_type_get_map ("GHashTable", G_TYPE_STRING, G_TYPE_STRING),
                G_TYPE_INVALID);


	client->priv->proxy = dbus_g_proxy_new_for_name (client->priv->bus,
                                                         "org.freedesktop.DeviceKit",
                                                         "/org/freedesktop/DeviceKit",
                                                         "org.freedesktop.DeviceKit");
        dbus_g_proxy_add_signal (client->priv->proxy,
                                 "DeviceEvent",
                                 G_TYPE_STRING,
                                 G_TYPE_STRING,
                                 G_TYPE_STRING,
                                 G_TYPE_STRING,
                                 G_TYPE_STRV,
                                 dbus_g_type_get_map ("GHashTable", G_TYPE_STRING, G_TYPE_STRING),
                                 G_TYPE_INVALID);

	client->priv->prop_proxy = dbus_g_proxy_new_for_name (client->priv->bus,
                                                              "org.freedesktop.DeviceKit",
                                                              "/org/freedesktop/DeviceKit",
                                                              "org.freedesktop.DBus.Properties");

        /* TODO: need to be able to specify match rules to upload to the bus daemon */
        dbus_g_proxy_connect_signal (client->priv->proxy, "DeviceEvent",
                                     G_CALLBACK (device_event_signal_handler), client, NULL);

        if (!org_freedesktop_DeviceKit_inhibit_shutdown (client->priv->proxy, &(client->priv->inhibit_cookie), error))
                goto out;

        ret = TRUE;

out:
        return ret;
}

#define STRUCT_TYPE (dbus_g_type_get_struct ("GValueArray",                                                    \
                                             G_TYPE_STRING,                                                    \
                                             G_TYPE_STRING,                                                    \
                                             G_TYPE_STRING,                                                    \
                                             G_TYPE_STRV,                                                      \
                                             dbus_g_type_get_map ("GHashTable", G_TYPE_STRING, G_TYPE_STRING), \
                                             G_TYPE_INVALID))

/**
 * devkit_client_enumerate_by_subsystem:
 * @client: A #DevkitClient object.
 * @subsystems: Either %NULL to enumerate all devices or a %NULL
 * terminated string array detailing what subsystems to get
 * information about.
 * @error: Return location for error.
 *
 * Retrieves an ordered list of the devices belonging to @subsystems from the DeviceKit daemon.
 *
 * Returns: %NULL if @error is set, otherwise a list of #DevkitDevice objects. The caller should free this by using g_object_unref() on each element in the list and then g_list_free() on the list.
 **/
GList *
devkit_client_enumerate_by_subsystem (DevkitClient  *client,
                                      const gchar  **subsystems,
                                      GError       **error)
{
        guint n;
        GPtrArray *devices;
        GList *ret;

        g_return_val_if_fail (DEVKIT_IS_CLIENT (client), NULL);
        g_return_val_if_fail (subsystems != NULL, NULL);

        ret = NULL;

        if (!org_freedesktop_DeviceKit_enumerate_by_subsystem (client->priv->proxy,
                                                               subsystems,
                                                               &devices,
                                                               error))
                goto out;

        for (n = 0; n < devices->len; n++) {
                GValue elem = {0};
                char *subsystem;
                char *native_path;
                char *device_file;
                char **device_file_symlinks;
                GHashTable *properties;
                DevkitDevice *device;

                g_value_init (&elem, STRUCT_TYPE);
                g_value_set_static_boxed (&elem, devices->pdata[n]);

                dbus_g_type_struct_get (&elem,
                                        0, &subsystem,
                                        1, &native_path,
                                        2, &device_file,
                                        3, &device_file_symlinks,
                                        4, &properties,
                                        G_MAXUINT);

                device = _devkit_device_new (subsystem,
                                             native_path,
                                             device_file,
                                             (const gchar **) device_file_symlinks,
                                             properties);
                ret = g_list_prepend (ret, device);

                g_free (subsystem);
                g_free (native_path);
                g_free (device_file);
                g_strfreev (device_file_symlinks);
                g_hash_table_unref (properties);
        }

out:
        ret = g_list_reverse (ret);
        return ret;
}


/**
 * devkit_client_enumerate_by_native_path:
 * @client: A #DevkitClient object.
 * @native_paths: A %NULL terminated string array of native paths.
 * @error: Return location for error.
 *
 * Retrieves an ordered list of the devices with information about the
 * devices given in @native_paths from the DeviceKit daemon.
 *
 * Returns: %NULL if @error is set, otherwise a list of #DevkitDevice objects. The caller should free this by using g_object_unref() on each element in the list and then g_list_free() on the list.
 **/
GList *
devkit_client_enumerate_by_native_path (DevkitClient  *client,
                                        const gchar  **native_paths,
                                        GError       **error)
{
        guint n;
        GPtrArray *devices;
        GList *ret;

        g_return_val_if_fail (DEVKIT_IS_CLIENT (client), NULL);
        g_return_val_if_fail (native_paths != NULL, NULL);

        ret = NULL;

        if (!org_freedesktop_DeviceKit_enumerate_by_native_path (client->priv->proxy,
                                                                 native_paths,
                                                                 &devices,
                                                                 error))
                goto out;

        for (n = 0; n < devices->len; n++) {
                GValue elem = {0};
                char *subsystem;
                char *native_path;
                char *device_file;
                char **device_file_symlinks;
                GHashTable *properties;
                DevkitDevice *device;

                g_value_init (&elem, STRUCT_TYPE);
                g_value_set_static_boxed (&elem, devices->pdata[n]);

                dbus_g_type_struct_get (&elem,
                                        0, &subsystem,
                                        1, &native_path,
                                        2, &device_file,
                                        3, &device_file_symlinks,
                                        4, &properties,
                                        G_MAXUINT);

                device = _devkit_device_new (subsystem,
                                             native_path,
                                             device_file,
                                             (const char **) device_file_symlinks,
                                             properties);
                ret = g_list_prepend (ret, device);

                g_free (subsystem);
                g_free (native_path);
                g_free (device_file);
                g_strfreev (device_file_symlinks);
                g_hash_table_unref (properties);
        }

out:
        ret = g_list_reverse (ret);
        return ret;
}
