
/*
 *  soldevicetypes.cpp
 *
 *  Copyright (C) 2009 David Hubner <hubnerd@ntlworld.com>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 *
 */

#include "soldevicetypes.h"

// ---- Processor

SolProcessorDevice::SolProcessorDevice(QTreeWidgetItem *parent, Solid::Device device) :
  SolDevice(parent, device) 
{ 
  deviceType = Solid::DeviceInterface::Processor;
  
  setDefaultDeviceText();
  setDefaultDeviceIcon();  
}

void SolProcessorDevice::setDefaultDeviceIcon() {
  setDeviceIcon(KIcon("cpu"));
}

void SolProcessorDevice::setDefaultDeviceText() {

  Solid::Processor *prodev = getInterface<Solid::Processor>(); 
  setText(0,"Processor " + (QString::number(prodev->number())));
  
}

QVListLayout *SolProcessorDevice::getInfoPanelLayout() {

  QList<QString> labels;
  Solid::Processor *prodev = getInterface<Solid::Processor>(); 
  
  deviceInfoLayout = new QVListLayout();
  
  QString extensions;
  switch(prodev->instructionSets()) {
    case Solid::Processor::IntelMmx:
      extensions.append(i18n("Intel MMX "));
    case Solid::Processor::IntelSse:
      extensions.append(i18n("Intel SSE "));
    case Solid::Processor::IntelSse2:
      extensions.append(i18n("Intel SSE2 "));
    case Solid::Processor::IntelSse3:
      extensions.append(i18n("Intel SSE3 "));
    case Solid::Processor::IntelSse4:
      extensions.append(i18n("Intel SSE4 "));
    case Solid::Processor::Amd3DNow:
      extensions.append(i18n("AMD 3DNow "));
    case Solid::Processor::AltiVec:
      extensions.append(i18n("ATI IVEC "));
      break;
    default:
      extensions.append(i18n("None"));
      break;
  }
  
  labels << i18n("Processor Number: ")
  << InfoPanel::friendlyString(QString::number(prodev->number())) 
  << i18n("Max Speed: ") 
  << InfoPanel::friendlyString(QString::number(prodev->maxSpeed()))
  << i18n("Supported Instruction Sets: ")
  << extensions;
  
  deviceInfoLayout->applyQListToLayout(labels);
  
  return deviceInfoLayout;
}

// ---- Storage

SolStorageDevice::SolStorageDevice(QTreeWidgetItem *parent, Solid::Device device) :
  SolDevice(parent, device)
{
    deviceType = Solid::DeviceInterface::StorageDrive;
    setDefaultDeviceText();
   
    createDeviceChildren<SolVolumeDevice>(this,device.udi(),Solid::DeviceInterface::StorageVolume);
}

void SolStorageDevice::setDefaultDeviceIcon() {
  setDeviceIcon(KIcon("drive-harddisk"));
}

void SolStorageDevice::setDefaultDeviceText() {
  
  Solid::StorageDrive *stodev = getInterface<Solid::StorageDrive>();
  setDefaultDeviceIcon();
  
  QString storageType;
  switch (stodev->driveType()) {
    case Solid::StorageDrive::HardDisk: 
      storageType = i18n("Harddisk");  break;
    case Solid::StorageDrive::CompactFlash:
      storageType = i18n("Compact Flash");  
      setDeviceIcon(KIcon("media-flash")); break;
    case Solid::StorageDrive::SmartMedia:
      storageType = i18n("Smart Media"); 
      setDeviceIcon(KIcon("media-flash-smart-media")); break;
    case Solid::StorageDrive::SdMmc:
      storageType = i18n("Sd Mmc");
      setDeviceIcon(KIcon("media-flash-sd-mmc")); break;
    case Solid::StorageDrive::CdromDrive:
      storageType = i18n("Optical");
      setDeviceIcon(KIcon("drive-optical")); break;
    case Solid::StorageDrive::MemoryStick:
      storageType = i18n("Memory Stick");
      setDeviceIcon(KIcon("media-flash-memory-stick")); break;
    case Solid::StorageDrive::Xd:
      storageType = i18n("Xd"); break;
    default:
      storageType = i18n("Unknown"); break;
  }
  
  QString deviceText = storageType + i18n(" Drive"); 
  setDeviceText(deviceText);
}

QVListLayout *SolStorageDevice::getInfoPanelLayout() {
  
  QList<QString> labels;
  Solid::StorageDrive *stodev = getInterface<Solid::StorageDrive>(); 
  
  deviceInfoLayout = new QVListLayout();
  
  QString bus;
  switch(stodev->bus()) {
    case Solid::StorageDrive::Ide:
      bus = i18n("IDE"); break;
    case Solid::StorageDrive::Usb:
      bus = i18n("USB"); break;
    case Solid::StorageDrive::Ieee1394:
      bus = i18n("IEEE1394"); break;
    case Solid::StorageDrive::Scsi:
      bus = i18n("SCSI"); break;
    case Solid::StorageDrive::Sata:
      bus = i18n("SATA"); break;
    case Solid::StorageDrive::Platform:
      bus = i18n("Platform"); break;
    default:
      bus = i18n("Unknown"); break;
  }
  
  labels << i18n("Bus: ")
  << bus
  << i18n("Hotpluggable?")
  << InfoPanel::convertTf(stodev->isHotpluggable())
  << i18n("Removable?") 
  << InfoPanel::convertTf(stodev->isRemovable());
  
  deviceInfoLayout->applyQListToLayout(labels);
  return deviceInfoLayout;
}

// ---- Network

SolNetworkDevice::SolNetworkDevice(QTreeWidgetItem *parent, Solid::Device device) :
  SolDevice(parent, device) 
{
  deviceType = Solid::DeviceInterface::NetworkInterface;
  
  setDefaultDeviceText(parent);
  setDefaultDeviceIcon();
}

void SolNetworkDevice::setDefaultDeviceText(QTreeWidgetItem *parent) {
  
  if(parent) {
    
      Solid::NetworkInterface *netdev = getInterface<Solid::NetworkInterface>(); 
      QString deviceText = netdev->ifaceName() + " (" + (netdev->isWireless() ? i18n("Wireless") : i18n("Wired")) + ") ";
  
      setDeviceText(deviceText);
  }
}

void SolNetworkDevice::setDefaultDeviceIcon() {
  
  Solid::NetworkInterface *netdev = getInterface<Solid::NetworkInterface>(); 
  
  if(netdev->isWireless() == true) {
    setDeviceIcon(KIcon("network-wireless"));
  } else {
    setDeviceIcon(KIcon("network-wired"));
  }
}

QVListLayout *SolNetworkDevice::getInfoPanelLayout() {
 
  QList<QString> labels;
  Solid::NetworkInterface *netdev = getInterface<Solid::NetworkInterface>(); 
  
  deviceInfoLayout = new QVListLayout();
  
  labels << i18n("Hardware Address: ")
  << InfoPanel::friendlyString(netdev->hwAddress())
  << i18n("Wireless?")
  << InfoPanel::convertTf(netdev->isWireless());
  
  deviceInfoLayout->applyQListToLayout(labels); 
  return deviceInfoLayout;
  
}

#include <QProgressBar>

// ---- Volume

SolVolumeDevice::SolVolumeDevice(QTreeWidgetItem *parent, Solid::Device device) :
  SolDevice(parent, device) 
{
  deviceType = Solid::DeviceInterface::StorageVolume;
  
  setDefaultDeviceIcon();
}

void SolVolumeDevice::setDefaultDeviceIcon() {
  setDeviceIcon(KIcon("partitionmanager"));
}

QVListLayout *SolVolumeDevice::getInfoPanelLayout() {

  QList<QString> labels;
  Solid::StorageVolume *voldev = getInterface<Solid::StorageVolume>(); 
  
  deviceInfoLayout = new QVListLayout();
  
  QString usage;
  switch(voldev->usage()) {
    case Solid::StorageVolume::Unused:
      usage = i18n("Unused"); break;
    case Solid::StorageVolume::FileSystem:
      usage = i18n("FileSystem"); break;
    case Solid::StorageVolume::PartitionTable:
      usage = i18n("Partition Table"); break;   
    case Solid::StorageVolume::Raid:
      usage = i18n("Raid"); break;
    case Solid::StorageVolume::Encrypted:
      usage = i18n("Encrypted"); break;
    default: 
      usage = i18n("Unknown"); break;
  }
  
  labels << i18n("FileSystem Type: ") 
  << InfoPanel::friendlyString(voldev->fsType())
  << i18n("Label: ")
  << InfoPanel::friendlyString(voldev->label(),i18n("Not Set"))
  << i18n("Volume Usage: ")
  << usage
  << i18n("UUID: ")
  << InfoPanel::friendlyString(voldev->uuid());

  deviceInfoLayout->applyQListToLayout(labels);
  return deviceInfoLayout;
}
  
// -- Audio

SolAudioDevice::SolAudioDevice(QTreeWidgetItem *parent, Solid::Device device) :
  SolDevice(parent, device) 
{
  deviceType = Solid::DeviceInterface::AudioInterface;
  
  setDefaultDeviceIcon();
}

void SolAudioDevice::setDefaultDeviceIcon() {
  setDeviceIcon(KIcon("audio-card"));
}

QVListLayout *SolAudioDevice::getInfoPanelLayout() {

  QList<QString> labels;
  Solid::AudioInterface *auddev = getInterface<Solid::AudioInterface>(); 
  
  deviceInfoLayout = new QVListLayout();

  QString AiType;
  switch(auddev->deviceType()) {
    case Solid::AudioInterface::AudioControl:
      AiType = i18n("Control"); break;
    case Solid::AudioInterface::AudioInput:
      AiType = i18n("Input"); break;
    case Solid::AudioInterface::AudioOutput:
      AiType = i18n("Output"); break;
    default:
      AiType = i18n("Unknown"); break;
  }
  
  QString ScType;
  switch(auddev->soundcardType()) {
    case Solid::AudioInterface::InternalSoundcard:
      ScType = i18n("Internal Soundcard"); break;
    case Solid::AudioInterface::UsbSoundcard:
      ScType = i18n("Usb Soundcard"); break;
    case Solid::AudioInterface::FirewireSoundcard:
      ScType = i18n("Firewire Soundcard"); break;
    case Solid::AudioInterface::Headset:
      ScType = i18n("Headset"); break;
    case Solid::AudioInterface::Modem:
      ScType = i18n("Modem"); break;
    default:
      ScType = i18n("Unknown"); break;
  }
  
  labels << i18n("Audio Interface Type: ")
  << AiType
  << i18n("SoundCard Type: ")
  << ScType;
  
  deviceInfoLayout->applyQListToLayout(labels);
  return deviceInfoLayout;
}

// Button

SolButtonDevice::SolButtonDevice(QTreeWidgetItem *parent, Solid::Device device) :
  SolDevice(parent, device) 
{
  deviceType = Solid::DeviceInterface::Button;
  
  setDefaultDeviceIcon();
}

void SolButtonDevice::setDefaultDeviceIcon() {
  setDeviceIcon(KIcon("insert-button"));
}

QVListLayout *SolButtonDevice::getInfoPanelLayout() {

  QList<QString> labels;
  Solid::Button *butdev = getInterface<Solid::Button>(); 
  
  deviceInfoLayout = new QVListLayout();
 
  QString type;
  switch(butdev->type()) {
    case Solid::Button::LidButton:
      type = i18n("Lid Button"); break;
    case Solid::Button::PowerButton:
      type = i18n("Power Button"); break;
    case Solid::Button::SleepButton:
      type = i18n("Sleep Button"); break;
    case Solid::Button::TabletButton:
      type = i18n("Tablet Button"); break;
    default:
      type = i18n("Unknown Button"); break;
  }
    
  labels << i18n("Button type: ")
  << type
  << i18n("Has State?")
  << InfoPanel::convertTf(butdev->hasState());
    
  deviceInfoLayout->applyQListToLayout(labels);
  return deviceInfoLayout;
}

// Media Player

SolMediaPlayerDevice::SolMediaPlayerDevice(QTreeWidgetItem *parent, Solid::Device device) :
  SolDevice(parent, device) 
{
  deviceType = Solid::DeviceInterface::PortableMediaPlayer;
  
  setDefaultDeviceIcon();
}

void SolMediaPlayerDevice::setDefaultDeviceIcon() {
  setDeviceIcon(KIcon("multimedia-player"));
}

QVListLayout *SolMediaPlayerDevice::getInfoPanelLayout() {

  QList<QString> labels;
  Solid::PortableMediaPlayer *mpdev = getInterface<Solid::PortableMediaPlayer>(); 
  
  labels << i18n("Supported Drivers: ")
  << mpdev->supportedDrivers()
  << i18n("Supported Protocols: ")
  << mpdev->supportedProtocols();

  deviceInfoLayout->applyQListToLayout(labels);
  return deviceInfoLayout;
}

// Camera

SolCameraDevice::SolCameraDevice(QTreeWidgetItem *parent, Solid::Device device) :
  SolDevice(parent, device) 
{
  deviceType = Solid::DeviceInterface::Camera;
  
  setDefaultDeviceIcon();
}

void SolCameraDevice::setDefaultDeviceIcon() {
  setDeviceIcon(KIcon("camera-photo"));
}

QVListLayout *SolCameraDevice::getInfoPanelLayout() {

  QList<QString> labels;
  Solid::Camera *camdev = getInterface<Solid::Camera>(); 

  labels << i18n("Supported Drivers: ")
  << camdev->supportedDrivers()
  << i18n("Supported Protocols: ")
  << camdev->supportedProtocols();
  
  deviceInfoLayout->applyQListToLayout(labels);
  return deviceInfoLayout;
}
  
// Battery

SolBatteryDevice::SolBatteryDevice(QTreeWidgetItem *parent, Solid::Device device) :
  SolDevice(parent, device) 
{
  deviceType = Solid::DeviceInterface::Battery;
  
  setDefaultDeviceIcon();
}

void SolBatteryDevice::setDefaultDeviceIcon() {
  setDeviceIcon(KIcon("battery"));
}

QVListLayout *SolBatteryDevice::getInfoPanelLayout() {
  
  QList<QString> labels;
  Solid::Battery *batdev = getInterface<Solid::Battery>(); 

  QString type;
  switch(batdev->type()) {
      case Solid::Battery::PdaBattery:
	type = "PDA"; break;
      case Solid::Battery::UpsBattery:
	type = "UPS"; break;
      case Solid::Battery::MouseBattery:
	type = "Mouse"; break;
      case Solid::Battery::PrimaryBattery:
	type = "Primary"; break;
      case Solid::Battery::KeyboardBattery:
	type = "Keyboard"; break;
      case Solid::Battery::KeyboardMouseBattery:
	type = "Keyboard + Mouse"; break;
      case Solid::Battery::CameraBattery:
	type = "Camera"; break;
      default:
	type = "Unknown";
  }
  
  QString state;
    switch(batdev->type()) {
      case Solid::Battery::Charging:
	state = "Charging"; break;
      case Solid::Battery::Discharging:
	state = "Discharging"; break;
      default:
	state = "No Charge"; 
    }
  
  labels << i18n("Battery Type: ")
  << type
  << i18n("Charge Status: ")
  << state;
  
  deviceInfoLayout->applyQListToLayout(labels);
  return deviceInfoLayout;
}
  
// Ac Adapter

SolAcAdapterDevice::SolAcAdapterDevice(QTreeWidgetItem *parent, Solid::Device device) :
  SolDevice(parent, device) 
{
  deviceType = Solid::DeviceInterface::AcAdapter;
}

QVListLayout *SolAcAdapterDevice::getInfoPanelLayout() {
  
  QList<QString> labels;
  Solid::AcAdapter *acdev = getInterface<Solid::AcAdapter>(); 
  
  labels << i18n("Is plugged in?")
  << InfoPanel::convertTf(acdev->isPlugged());
  
  deviceInfoLayout->applyQListToLayout(labels);
  return deviceInfoLayout;
}

// DVB

SolDvbDevice::SolDvbDevice(QTreeWidgetItem *parent, Solid::Device device) :
  SolDevice(parent, device) 
{
  deviceType = Solid::DeviceInterface::DvbInterface;
}

// Serial

SolSerialDevice::SolSerialDevice(QTreeWidgetItem *parent, Solid::Device device) :
  SolDevice(parent, device) 
{
  deviceType = Solid::DeviceInterface::SerialInterface;
}

QVListLayout *SolSerialDevice::getInfoPanelLayout() {
  
  QList<QString> labels;
  Solid::SerialInterface *serdev = getInterface<Solid::SerialInterface>(); 
  
  QString type;
  switch(serdev->serialType()) {
    case Solid::SerialInterface::Platform:
      type = "Platform"; break;
    case Solid::SerialInterface::Usb:
      type = "USB"; break;
    default:
      type = "Unknown"; 
  }
  
  labels << i18n("Serial Type: ")
  << type
  << i18n("Port: ")
  << QString::number(serdev->port());
  
  deviceInfoLayout->applyQListToLayout(labels);
  return deviceInfoLayout;
}

//  Smartcard

SolSmartCardDevice::SolSmartCardDevice(QTreeWidgetItem *parent, Solid::Device device) :
  SolDevice(parent, device) 
{
  deviceType = Solid::DeviceInterface::SmartCardReader;
}

QVListLayout *SolSmartCardDevice::getInfoPanelLayout() {
  
  QList<QString> labels;
  Solid::SmartCardReader *scdev = getInterface<Solid::SmartCardReader>(); 
  
  QString type;
  switch(scdev->readerType()) {
    case Solid::SmartCardReader::CardReader:
      type = "Card Reader"; break;
    case Solid::SmartCardReader::CryptoToken:
      type = "Crypto Token"; break;
    default:
      type = "Unknown"; 
  }
 
  labels << i18n("SmartCard Type: ")
  << type;
  
  deviceInfoLayout->applyQListToLayout(labels);
  return deviceInfoLayout;
}

// Video

SolVideoDevice::SolVideoDevice(QTreeWidgetItem *parent, Solid::Device device) :
  SolDevice(parent, device) 
{
  deviceType = Solid::DeviceInterface::Video;
  
  setDefaultDeviceIcon();
}

void SolVideoDevice::setDefaultDeviceIcon() {
  setDeviceIcon(KIcon("video-display"));
}

QVListLayout *SolVideoDevice::getInfoPanelLayout() {
  
  QList<QString> labels;
  Solid::Video *viddev = getInterface<Solid::Video>(); 
  
  labels << i18n("Supported Drivers: ")
  << viddev->supportedDrivers()
  << i18n("Supported Protocols: ")
  << viddev->supportedProtocols();
  
  deviceInfoLayout->applyQListToLayout(labels);
  return deviceInfoLayout;
}